/**
 * $Id$
 *
 * tf_validator.c
 *
 * 入力値、アクセス制御の検証を行うのに使用する関数、構造体を宣言、定義するファイル
 *
 */
#include "httpd.h"
#include "http_protocol.h"
#include "util_md5.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "apr_strings.h"
#include "apr_uri.h"
#include "apr_time.h"
#include "mod_dav.h"

#include "mod_dav_tf.h"
#include "util_common.h"
#include "util_ml.h"
#include "tf_sqlparser.h"
#include "tf_folder.h"
#include "tf_validator.h"
#include "liveprop.h"
#include "tf_cgi.h"
#include "tf_plugin.h"
#include "tf_autodelete.h"
#include "tf_rdbo.h"
#include "tf_rdbo_sql.h"
#include "tf_rdbo_dbms.h"
#include "tf_rdbo_group.h"
#include "tf_rdbo_user.h"
#include "util_ldap.h"
#include "tf_confirmreading.h"

#if APR_HAVE_LIMITS_H
#include <limits.h>
#endif

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define static values and macro
  ----------------------------------------------------------------------------*/
/*
 * path segment に使用される文字の禁則文字
 */
static const char not_allowed_set[] = {
	'\\', '/', ':', '*', '?', '"', '<', '>', '|'
};
#define NOT_ALLOWED_SET_LEN	sizeof(not_allowed_set)/sizeof(char)

/**
 * パスワード文字列に使用可能な最大数
 */
#define DIVY_MAX_PASSWORD_LEN	64

/*------------------------------------------------------------------------------
  Declare private prototype function
  ----------------------------------------------------------------------------*/
static int _validate_usr_accessdeny(request_rec *r, 
					const char *uri,
					divy_uri_spec *uri_spec);
static int _validate_group_access(request_rec *r,
					const char *userid,
					divy_uri_spec *uri_spec);
#ifdef DIVY_SUPPORT_PASSPOLICY
static dav_error * _validate_password_input(request_rec *r,
					const divy_rdbo_usr *usr_pr, const char *password,
					int force_plain);
#endif	/* DIVY_SUPPORT_PLUGIN */
static dav_error * _validate_creating_user_limit(request_rec *r);
static dav_error * _validate_control_user_limit(request_rec *r);
int _compar_id(const void *e1, const void *e2);
static dav_error * _validate_view_state(request_rec *r,
					int chk_mode,
					divy_resourcestate_iscreen *rstate_iscreen);
static dav_error * _validate_box_state(request_rec *r,
					int chk_mode,
					divy_resourcestate_iscreen *rstate_iscreen);
static char * _get_destination_uri(request_rec *r);
#ifdef DIVY_SUPPORT_PLUGIN
static dav_error * _validate_secured_folder(request_rec *r,
					const divy_uri_spec *src_u_spec, const divy_uri_spec *dst_u_spec);
#endif	/* DIVY_SUPPORT_PLUGIN */
static dav_error * _validate_pre_fsizelimit(request_rec *r, divy_infotype infotype, apr_int64_t filelen);
static dav_error * _validate_pre_userquota(request_rec *r, divy_infotype infotype, apr_int64_t filelen);
#ifdef DIVY_SUPPORT_EXTENDQUOTA
static dav_error * _validate_pre_sysquota(request_rec *r, divy_infotype infotype, apr_int64_t filelen);
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */
#ifdef DIVY_SUPPORT_GROUPQUOTA
static dav_error * _validate_pre_grpquota(request_rec *r, divy_infotype infotype, apr_int64_t filelen);
#endif /* DIVY_SUPPORT_GROUPQUOTA */
static dav_error * _validate_appointed_groupleader(request_rec *r, divy_infotype infotype,
													const dav_resource *resource, int is_src_resource);
static dav_error * _validate_group_via_groupleader(request_rec *r, divy_infotype infotype,
													divy_rdbo_grp *grp_pr, int is_src_resource,
													divy_uri_spec *src_u_spec, divy_uri_spec *dst_u_spec);
static int _validate_allow_ipaddr(request_rec *r, divy_array_t *addrs);
static int _build_string2ipsubnet_array(request_rec *r, const char *str, divy_array_t **addrs);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたu_spec のURIに対し発行されたメソッドが利用可能かどうかを検証して
 * 返却する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_minimum_access_guarantee(request_rec *r,
							divy_uri_spec *u_spec)
{
	dav_error *err        = NULL;
	apr_pool_t *p         = r->pool;
	int is_allowed        = 0;	/* 適切なアクセスかどうか */
	int mnum              = divy_get_method_number(r);
	int m_search          = ap_method_number_of("SEARCH");
	const char *userid    = divy_get_userid(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	dav_divy_dir_conf    *dconf = dav_divy_get_dir_config(r);
	const char *ua        = NULL;
	int ret;
	const char *uri, *avoid_wf_sslcerterr;
	divy_infotype infotype;
	int support_groupleader = divy_support_groupleader(r);
	const divy_rdbo_extstatus *extstatus = NULL;
	int support_access_control = divy_support_access_control(r);
	divy_array_t *ipaddrs = NULL;

	if (divy_support_extenduserstatus(r)) {
		extstatus = divy_get_extstatus(r);
	}
 
	if (u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"u_spec is EMPTY.");
		err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		goto return_state;
	}
	infotype = u_spec->infotype;
	uri      = u_spec->uri;

	/*
	 * サーバライセンスキーの検証
	 */
	if (sconf->svrlicensekey == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The server license key is missing.");
		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto return_state;
	}

	/* 扱えないuri であった場合 */
	if (u_spec->uritype & DIVY_URITYPE_UNKNOWN) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Invalid uri was specified. "
			"The server could not handle this uri.(uri = %s)", u_spec->uri);
		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto return_state;
	}

	/* SAML画面の場合は以下のチェックはしない */
	if (infotype == DIVY_INFOTYPE_saml) {
		err = NULL;
		goto return_state;
	}

	/* ログイン画面の場合は以下のチェックはしない */
	if (infotype == DIVY_INFOTYPE_login) {
		err = NULL;
		goto return_state;
	}
	
	/* 短縮URLの場合は以下のチェックはしない */
	if (infotype == DIVY_INFOTYPE_shorten) {
		err = NULL;
		goto return_state;
	}

	/* OPTIONSはチェックしない */
	if (divy_get_method_number(r) == M_OPTIONS) {
		err = NULL;
		goto return_state;
	}
	/*
	 * ユーザカウントは有効か?
	 *
	 * [ 無効と判断する条件 ]
	 *	* ユーザプロパティが存在しない -> ユーザIDが取得できているか
	 *	* 有効期限切れ
	 *	* 一時無効
	 */

	/* ユーザID が取れなかった */
	if (IS_EMPTY(userid)) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to get your userid. The server could not "
			"operation no more.(uri = %s)", uri);
		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto return_state;
	}

	/* 非アクティブユーザ / 有効期限切れであった */
	if (divy_support_extenduserstatus(r) && 
		(!divy_rdbo_is_active_user(divy_get_extstatus(r)) || divy_rdbo_is_expired_user(r))) {

		/* OPTIONS, SEARCH(userinformation) 以外はアクセス拒否 */
		if (!(mnum == M_OPTIONS || (mnum == m_search && infotype == DIVY_INFOTYPE_m_user))) {

			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The user was inactive or expired. (user = %s)", userid);
			err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			goto return_state;
		}
		/* OPTIONS, SEARCH は残りの条件を全てクリアすればアクセス可能
		 * だから、アクセスチェックは続行します。
		 * なお、SEARCHのBodyチェックはsearch プロバイダに任せます */
	}

	/* ログインユーザが管理者で管理者用のネットワークが指定されていた場合
	 * ネットワークが該当しなかったらエラーとします。*/
	if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN
			&& dconf->allowadminipaddrs != NULL
			&& _validate_allow_ipaddr(r, dconf->allowadminipaddrs) == 0)
	{
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The administrator is not permitted from this network.(user = %s) (access network()", userid);
		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto return_state;
	}

	/* 
	 * アクセス制御されているユーザならアドレスのチェックを行う
	 */
	if (support_access_control) {
		const char *allowhosts = divy_get_allowhosts(r);
		if (IS_FILLED(allowhosts)) {
			if (!_build_string2ipsubnet_array(r, allowhosts, &ipaddrs)) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"allow host ip Address build failed %s", allowhosts);
				err = dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				goto return_state;
			}

			/* 接続可能なアドレスか調べる */
			if (_validate_allow_ipaddr(r, ipaddrs) == 0) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"access forbidden. remote addr = %s "
						"(user = %s)", r->useragent_ip, userid);
				err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
				goto return_state;
			}
		}
		else {
			/*
			 * デフォルトルールを調べる
			 * デフォルトがDENYでもシステム権限のユーザは無視する
			 */
			if (dconf->defaultaccessrule == DIVY_DEFAULT_ACCESS_RULE_DENY && 
				!divy_rdbo_has_sysexec_privilege(extstatus) ) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"access forbidden.(user = %s)", userid);
				err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
				goto return_state;
			}
		}
	}

	/*
	 * アカウントに問題がなければTeamFileエージェントから
	 * アクセスされるURIは許可しておくこと
	 * (note)
	 * 	エージェントは、GETとSEARCHしかしないので、それ以外の
	 * 	メソッドは以下のif 文以外でチェックします。
	 */
	if ((mnum == m_search && infotype == DIVY_INFOTYPE_m_update) ||
	    (mnum == m_search && infotype == DIVY_INFOTYPE_m_msg) ||
	    (mnum == M_GET    && infotype == DIVY_INFOTYPE_m_update_e)) {
		err = NULL;
		goto return_state;
	}

	/*
	 * MS のWebfolder(正確にはProtocol Discovery) によるSSLクライアント認証問題を
	 * 回避するための対策
	 *
	 * [ 許可する条件 ]
	 *   * OPTIONS であること
	 *   * 特殊な環境変数で指示されていること
	 *
	 * SSLアクセラレータ環境に備えて、SSLかどうかのチェックはしていません.
	 */
	avoid_wf_sslcerterr = apr_table_get(r->subprocess_env, DIVY_APENV_AVOID_WF_SSLCERTERR);
	if (mnum == M_OPTIONS && IS_FILLED(avoid_wf_sslcerterr)) {
		/* 仕方がないので、このケースは許可します */
		err = NULL;
		goto return_state;
	}

	/*
	 * プライベートコレクション以下にアクセス出来るのは
	 * このコレクションのオーナだけ
	 *
	 * $root/$userid
	 * $root/$userid/xxxxx/xxxx
	 * ユーザごみ箱
	 */
	if (infotype == DIVY_INFOTYPE_user_e        ||
	    infotype == DIVY_INFOTYPE_user_trash    ||
	    infotype == DIVY_INFOTYPE_user_trash_e0 ||
	    infotype == DIVY_INFOTYPE_user_trash_ex ||
	    infotype == DIVY_INFOTYPE_user_e_regular) {
		apr_size_t usr_len;
		char *first_usr_uri = dav_divy_make_uri(p, dav_divy_get_root_uri(r), userid, NULL);

		usr_len = strlen(first_usr_uri);
		ret = strncmp(uri, first_usr_uri, usr_len);

		/* uri の先頭が $root/$userid では無かった場合 */
		if (ret != 0 ||
		    (ret == 0 && (uri[usr_len] != '\0' && uri[usr_len] != '/'))) {

			/* WebDAVFSはこのログを常に出すので通らないようにする */
			ua = dav_divy_get_user_agent(r);
			if (!ua || strncmp(ua, "WebDAVFS", 8) != 0) {
				ERRLOG2(p, APLOG_WARNING,
				       	DIVY_FST_CERR + DIVY_SST_DATA,
					"You could not access the "
					"private-folder of another user."
					"(uri = %s, userid = %s)"
					, uri, userid);
			}
			/*
			 * FORBIDDENのエラーではWebDAVRedirectorで再取得の連続となり
			 * リクエストが増えてしまいます。素直にNOTFOUNDとすること
			 * によりこの問題は解決します。
			 */
			err = dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
			goto return_state;
		}
	}

	/*
	 * グループコレクションにアクセスできるのは
	 * このグループに所属しているユーザだけ
	 *
	 * $root/Group Folder/[$groupid | $groupname]
	 * $root/Group Folder/[$groupid | $groupname]/xxxxx/yyyyy
	 * グループごみ箱
	 */
	if (infotype == DIVY_INFOTYPE_group_e        ||
	    infotype == DIVY_INFOTYPE_group_trash    ||
	    infotype == DIVY_INFOTYPE_group_trash_e0 ||
	    infotype == DIVY_INFOTYPE_group_trash_ex ||
	    infotype == DIVY_INFOTYPE_group_e_regular) {

		/* アクセス可能かどうかをチェック */
		ret = _validate_group_access(r, userid, u_spec);
		if (ret != OK) {
			/* そのグループに所属していないか、複数階層グループへの
			 * アクセスが禁止されたことでアクセス不能になった */
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"You could not access a group. (uri = %s, userid = %s)", uri, userid);
			err = dav_new_error(p, ret, 0, 0, "");
			goto return_state;
		}
	}

	/*
	 * データベース機能が使用可能かどうかの検証
	 * OFF の時には"データベース"フォルダにアクセスさせてはならない.
	 */
	if (sconf && sconf->use_db_opt == 0 &&
	    (infotype == DIVY_INFOTYPE_dblink			||
	     infotype == DIVY_INFOTYPE_dblink_e			||
	     infotype == DIVY_INFOTYPE_dbfolder			||
	     infotype == DIVY_INFOTYPE_dbfolder_e		||
	     infotype == DIVY_INFOTYPE_dbfolder_e_regular	||
	     infotype == DIVY_INFOTYPE_dbshfolder		||
	     infotype == DIVY_INFOTYPE_dbshfolder_e		||
	     infotype == DIVY_INFOTYPE_dbshfolder_e_regular)) {

		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The function of database-collaboration is not active."
			"(method = %s, uri = %s)", divy_get_method_name(r), uri);

		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto return_state;
	}

	/*
	 * 管理者しかアクセスできない特殊コレクションかどうか
	 */
	if (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN) {

		/* グループリーダには一部リソースへのアクセスを許可する */
		if (support_groupleader && divy_rdbo_is_groupleader(extstatus) &&
			(infotype == DIVY_INFOTYPE_m_management  ||
			 infotype == DIVY_INFOTYPE_m_sql_e       ||
			 infotype == DIVY_INFOTYPE_m_sql_rgrp    ||
			 infotype == DIVY_INFOTYPE_m_user_rgrp   ||
			 infotype == DIVY_INFOTYPE_m_group_e     ||
			 infotype == DIVY_INFOTYPE_m_group_ruser ||
			 infotype == DIVY_INFOTYPE_m_group_rsql  ||
			 infotype == DIVY_INFOTYPE_m_status      ||
			 infotype == DIVY_INFOTYPE_m_status_e)) {
			/* 許可するので何もしない */
		}
		else if (
			infotype == DIVY_INFOTYPE_m_management  ||
		    infotype == DIVY_INFOTYPE_m_sql_e       ||
		    infotype == DIVY_INFOTYPE_m_sql_rgrp    ||
		    infotype == DIVY_INFOTYPE_m_user_rgrp   ||
		    //infotype == DIVY_INFOTYPE_m_group       || // (2005.03.26) このチェックはSEARCHがやります
		    infotype == DIVY_INFOTYPE_m_group_e     ||
		    infotype == DIVY_INFOTYPE_m_group_ruser ||
		    infotype == DIVY_INFOTYPE_m_group_rsql  ||
		    infotype == DIVY_INFOTYPE_m_status      ||
		    infotype == DIVY_INFOTYPE_m_status_e    ||
		    infotype == DIVY_INFOTYPE_m_msg_e       ||
		    infotype == DIVY_INFOTYPE_m_dbms        ||
		    infotype == DIVY_INFOTYPE_m_dbms_e      ||
		    infotype == DIVY_INFOTYPE_m_execsql     ||
		    infotype == DIVY_INFOTYPE_m_execsql_e) {

			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"The server does not allow access to this "
					"management resource.(method = %s, uri = %s)", divy_get_method_name(r), uri);

			err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			goto return_state;
		}
	}

	/*
	 * 通常のリソース、コレクションの場合
	 */
	if (u_spec->uritype & DIVY_URITYPE_REGULAR) {
#ifdef DAV_SUPPORT_POST_HANDLING
		/* POSTリクエストがプライベートコレクション以下、グループコレクション以下
		 * 以外の通常リソース、コレクションに対し発行されたなのなら失敗 */
		if (mnum == M_POST &&
			!(infotype == DIVY_INFOTYPE_user_e_regular ||
			  infotype == DIVY_INFOTYPE_group_e_regular)) {

			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The server does not allow access to this "
				"resource by POST method.(uri = %s)", uri);
			err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			goto return_state;
		}

#endif	/* DAV_SUPPORT_POST_HANDLING */
		/* アクセスユーザが拒否リストにリストされていないか？*/
		if ((ret = _validate_usr_accessdeny(r, uri, u_spec)) != OK) {
			err = dav_new_error(p, ret, 0, 0, "");
			goto return_state;
		}

		err = NULL;
		goto return_state;
	}

	/*
	 * uri が処理場所を表していた場合
	 */
	if (u_spec->uritype & DIVY_URITYPE_ARBITER) {
		is_allowed = 0;

		/* Utility Folder,
		 * Updated Client, System Message,
		 * $root/.ticket の場合 */
		if (infotype == DIVY_INFOTYPE_utility  ||
		    infotype == DIVY_INFOTYPE_u_update ||
		    infotype == DIVY_INFOTYPE_u_msg    ||
		    infotype == DIVY_INFOTYPE_ticket) {
			/* GET, OPTIONS, POST だけを許可する */
			if (mnum == M_GET || mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				err = NULL;
				goto return_state;
			}
		}
		/* $root/.mobile */
		else if (infotype == DIVY_INFOTYPE_mobile) {
			/* GET, OPTIONS だけを許可する */
			if (mnum == M_GET || mnum == M_OPTIONS) {
				err = NULL;
				goto return_state;
			}
		}
		/* Autoindex スタートアップページの場合 */
		else if (infotype == DIVY_INFOTYPE_fileviewer) {
			/* GET, HEAD(GETと同じチェックでいい), POST だけを許可する */
			if (mnum == M_GET
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				err = NULL;
				goto return_state;
			}
		}
		/* $root/.management/USER の場合 */
		else if (infotype == DIVY_INFOTYPE_m_user) {
			if (mnum == m_search || mnum == M_OPTIONS
#ifdef DIVY_SUPPORT_PASSPOLICY
				|| mnum == M_PROPPATCH
				|| mnum == M_PROPFIND
#endif	/* DIVY_SUPPORT_PASSPOLICY */
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;
			}
		}
		/* 自動削除処理用URIの場合(詳細なチェックは自動削除処理に委譲) */
		else if (infotype == DIVY_INFOTYPE_autodelete) {
			if (mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;
			}
		}
		/* 開封通知の自動解除用URIの場合 (詳細なチェックは解除ロジックに委譲) */
		else if (infotype == DIVY_INFOTYPE_confirmreading) {
			if (mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;
			}
		}
		/* 仮想フォルダの要素を持っていた場合 */
		else if (u_spec->uritype & DIVY_URITYPE_VIRTUAL) {
			/* SEARCH, OPTIONS, PROPFIND だけを許可する */
			if (mnum == m_search || mnum == M_OPTIONS ||
			    mnum == M_PROPFIND) {
				is_allowed = 1;
			}
		}
		/* 処理場所だけの場合 */
		else {
			/* SEARCH, OPTIONS だけを許可する */
			if (mnum == m_search || mnum == M_OPTIONS) {
				is_allowed = 1;
			}
		}

		/* アクセスユーザが拒否リストに入っていないか? */
		if (is_allowed) {
			if ((ret = _validate_usr_accessdeny(r, uri, u_spec)) != OK) {
				err = dav_new_error(p, ret, 0, 0, "");	/* リストされていた */
				goto return_state;
			}
			err = NULL;
			goto return_state;
		}
		else {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_PROC,
				"The server does not accept \"%s\" method "
				"on this arbiter-uri resource.(uri = %s)",
				divy_get_method_name(r), uri);
			err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			goto return_state;
		}
	}

	/*
	 * uri が仮想フォルダを表していた場合
	 * (note)
	 * 	DIVY_URITYPE_ARBITER の判定を最初にやったのでここにくるのは
	 * 	DIVY_URITYPE_ARBITER を持たない仮想フォルダだけ。
	 */
	if (u_spec->uritype & DIVY_URITYPE_VIRTUAL) {

		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_PROC,
			"The server does not accept \"%s\" method "
			"on this virtual-uri resource.(uri = %s)", divy_get_method_name(r), uri);

		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto return_state;
	}

	/*
	 * uri が特殊コレクションを表していた場合
	 */
	is_allowed = 0;
	switch (infotype) {

		/* $root フォルダ */
		case DIVY_INFOTYPE_root:
			if (mnum == M_GET      || mnum == M_PROPPATCH ||
			    mnum == M_PROPFIND || mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;
			}
			break;
		/* グループフォルダ ($root/Group Folder) */
		case DIVY_INFOTYPE_group:
			if (mnum == M_GET      || mnum == M_PROPFIND ||
			    mnum == m_search   || mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;
			}
			break;
		/* グループコレクション     (メンバは含まれない) */
		case DIVY_INFOTYPE_group_e:
			if (mnum == M_GET      ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == m_search   || mnum == M_OPTIONS ||
			    mnum == M_COPY
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;
			}
			break;
		/* プライベートコレクション (メンバは含まれない) */
		case DIVY_INFOTYPE_user_e:
			if (mnum == M_GET      ||
			    mnum == M_PROPFIND ||
			    mnum == m_search   || mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;
			}
			break;
		/* グループごみ箱、ユーザごみ箱フォルダ自身 */
		case DIVY_INFOTYPE_group_trash:
		case DIVY_INFOTYPE_user_trash:

			/* ごみ箱URIにアクセス可能 && サポートされている &&
			 * 以下のメソッドの場合のみアクセスを許可します。
			 * (GET はAutoindexとHEAD許可用(by Webフォルダ)です) */
			if (divy_enable_trashfolder_access(r) &&
			    divy_support_trashfolder(r) &&
			    (mnum == M_PROPFIND || mnum == M_GET ||
			     mnum == M_OPTIONS  || mnum == m_search
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				 )) {
				is_allowed = 1;
			}
			break;
		/* ユーザごみ箱の直下 / さらにその下
		   グループごみ箱の直下 / さらにその下 */
		case DIVY_INFOTYPE_user_trash_e0:
		case DIVY_INFOTYPE_user_trash_ex:
		case DIVY_INFOTYPE_group_trash_e0:
		case DIVY_INFOTYPE_group_trash_ex:

			/* ごみ箱URIにアクセス可能 && サポートされている &&
			 * 以下のメソッドの場合のみアクセスを許可します。*/
			if (divy_enable_trashfolder_access(r) &&
			    divy_support_trashfolder(r) &&
			    (mnum == M_GET      || mnum == M_DELETE   ||
			     mnum == M_PROPFIND || mnum == M_MOVE     ||
			     mnum == M_LOCK     || mnum == M_UNLOCK   ||
			     mnum == m_search   || mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				 )) {
				is_allowed = 1;
			}
			break;
		/* linkdbsearch 結果フォルダ(メンバは含まれない)
		 * (note) POSTオペレーションはサポートされません */
		case DIVY_INFOTYPE_dbfolder_e:
			if (mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MKCOL    ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS) {
				is_allowed = 1;	/* 許可 */
			}
			break;
		/* 共有コレクションフォルダ(メンバは含まれない)
		 * (note) POSTオペレーションはサポートされません */
		case DIVY_INFOTYPE_dbshfolder_e:
			if (mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MKCOL    ||
			    mnum == M_MOVE     ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
		/* 更新クライアント */
		case DIVY_INFOTYPE_m_update_e:
			if (mnum == M_GET      || mnum == M_PUT       ||
			    mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MOVE     || 
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
		/* ユーザ */
		case DIVY_INFOTYPE_m_user_e:
			if (mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MKCOL    ||
			    mnum == M_MOVE     || mnum == M_COPY      ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				) {
				is_allowed = 1;

				/* 管理者とグループリーダでなければ以下のアクセスは禁止します.
				 * (note)
				 * ユーザに対しては一部を許可しなくてはならないので
				 * 個別にチェックする必要あり
				 * (PROPPATCH, LOCK, UNLOCK, SEARCH, OPTIONS, POST は許可しなくてはならない)
				 * other ユーザならグループリーダが削除できないという判定は別のところでやります.
				 */
				if (!divy_rdbo_has_administrative_privilege(r)) {
					if (mnum == M_DELETE   ||
						mnum == M_MKCOL    ||
						mnum == M_MOVE     || mnum == M_COPY) {
						is_allowed = 0;
					}
				}
				/* (note) グループリーダには更にいくつかアクセスが制限されているがそれは
				 * divy_validate_request_resourceやMKCOL, PROPPATCHのvalidator でチェックする */
			}
			break;
		/* グループ */
		case DIVY_INFOTYPE_m_group_e:
			if (mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MKCOL    ||
			    mnum == M_MOVE     || mnum == M_COPY      ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS) {
				is_allowed = 1;
				/* (note) グループリーダはいくつかアクセスが制限されているがそれは
				 * divy_validate_request_resourceやMKCOL, PROPPATCHのvalidator でチェックする */
			}
			break;
		/* SQL */
		case DIVY_INFOTYPE_m_sql_e:
			if (mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MKCOL    ||
			    mnum == M_MOVE     || mnum == M_COPY      ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS) {
				is_allowed = 1;
			}

			/* グループリーダの場合、更新処理は禁止する */
			if (support_groupleader && divy_rdbo_is_groupleader(extstatus)) {
				if (mnum == M_DELETE   || mnum == M_PROPPATCH ||
					mnum == M_MKCOL) {
					is_allowed = 0;
				}
			}
			break;
		/* DBMS 接続情報 */
		case DIVY_INFOTYPE_m_dbms_e:
		{
			if (mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MKCOL    ||
			    mnum == M_MOVE     || mnum == M_COPY      ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
		}
		/* 統計情報 */
		case DIVY_INFOTYPE_m_status_e:
			if (mnum != m_search || mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
		/* システムメッセージ */
		case DIVY_INFOTYPE_m_msg_e:
			if (mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MKCOL    ||
			                          mnum == M_COPY      ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			    mnum == m_search   || mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
		/* 送信メール */
		case DIVY_INFOTYPE_mail_e:
			if (dconf->mlserversend != DIVY_MLSERVERSEND_ON &&
			    (mnum == M_GET      || mnum == M_PUT       ||
			     mnum == M_DELETE   ||
			     mnum == M_PROPFIND ||
			     mnum == M_LOCK     || mnum == M_UNLOCK    ||
			     mnum == m_search   || mnum == M_OPTIONS)) {
				is_allowed = 1;
			}
			break;
		/* ユーザ・グループ・SQLリレーション */
		case DIVY_INFOTYPE_m_user_rgrp:
		case DIVY_INFOTYPE_m_group_ruser:
		case DIVY_INFOTYPE_m_group_rsql:
		case DIVY_INFOTYPE_m_sql_rgrp:
			if (mnum == M_DELETE   ||
			    mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			    mnum == M_MOVE     || mnum == M_COPY      ||
			    mnum == M_LOCK     || mnum == M_UNLOCK    ||
			                          mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
		/* Updated Client/$modulename */
		case DIVY_INFOTYPE_u_update_e:
			if (mnum == M_GET      || mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
#ifdef DAV_SUPPORT_POST_HANDLING
		/* システムCGI */
		case DIVY_INFOTYPE_cgibin_e:
			if (mnum == M_GET      || mnum == M_POST ||
			    mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
#endif	/* DAV_SUPPORT_POST_HANDLING */
#ifdef DIVY_SUPPORT_PLUGIN
		/* 個々のプラグイン */
		case DIVY_INFOTYPE_plugin_e:
			if (mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
#ifdef DAV_SUPPORT_POST_HANDLING
		/* プラグインCGI, プラグインアップロードファイル */
		case DIVY_INFOTYPE_plugin_cgibin_e:
		case DIVY_INFOTYPE_plugin_uploads_e:
			if (mnum == M_GET      || mnum == M_POST ||
			    mnum == M_OPTIONS) {
				is_allowed = 1;
			}
			break;
#endif	/* DAV_SUPPORT_POST_HANDLING */
#endif	/* DIVY_SUPPORT_PLUGIN */

		default:
			is_allowed = 0;
			break;
	}
		
	/* 禁止されたアクセスであった */
	if (!is_allowed) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_PROC,
			"The server does not accept \"%s\" method "
			"on special-uri resource.(uri = %s)", divy_get_method_name(r), uri);

		err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		goto return_state;
	}

	/*
	 * このユーザはuri にアクセスできるかどうかを検証
	 *
	 * (note) このチェックを最初にしない理由
	 * 	不正アクセスをしてくる人のためにDBを引くのは勿体無い。
	 * 	だから、全てのチェックが通った後でやることにしています。
	 */
	if ((ret = _validate_usr_accessdeny(r, uri, u_spec)) != OK) {
		err = dav_new_error(p, ret, 0, 0, "");
		goto return_state;
	}

return_state:
#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * この関数内でエラーが発生していた場合、パスワードポリシーエラーよりもプライオリティが高いはず。
	 * だから、ポリシーエラーはクリアしておくこと.(複数回呼び出し対応)
	 */
	if (err != NULL) {
		/* 特殊なヘッダ、環境変数の内容をリセット */
		divy_reset_resultstatus(r);
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	return err;
}

/**
 * 指定されたリソース resource がresource->uri に対しアクセス可能かどうかを検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_request_resource(request_rec *r,
							const dav_resource *resource)
{
	apr_pool_t *p             = r->pool;
	divy_uri_spec *u_spec     = NULL;
	divy_rdbo_resource *rdb_r = NULL;
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);
	int support_extstatus     = divy_support_extenduserstatus(r);
	int mnum                  = divy_get_method_number(r);
	int m_search              = ap_method_number_of("SEARCH");
	divy_infotype infotype, p_infotype;
	int ret;
	char *dst_uri, *request_uri;
	const divy_rdbo_extstatus *extstatus = NULL;
	int is_write_constraints  = 0;	/* 書き込み制約 */
	divy_uri_spec *dst_u_spec = NULL;
	int is_src_resource = 1, is_dst_resource = 0;
	int is_trusty = 1, deny = 0;
	dav_error *err;

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* 特殊なヘッダ、環境変数の内容をリセット */
	divy_reset_resultstatus(r);
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	if (resource == NULL || resource->info == NULL ||
		resource->info->rdb_r == NULL || resource->info->rdb_r->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"resource is EMPTY.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	rdb_r      = resource->info->rdb_r;
	u_spec     = rdb_r->u_spec;
	infotype   = u_spec->infotype;
	p_infotype = u_spec->p_infotype;

	if (support_extstatus) {
		extstatus = divy_get_extstatus(r);
		is_trusty = divy_rdbo_is_trusty_user(extstatus);
	}

	/*
	 * Destination のURIを取得する.
	 * (note)
	 *   本当は、src と dstが揃った場所(move_resourceやcopy_resource のハンドラの中)で
	 *   検証したいのだが、mod_dav のフレームワークに従うと、上記のハンドラでは
	 *   もはや手遅れ.  MOVEであればsrc が既に消えた状態になります.
	 *   validation で幾らエラーがでたところで、後の祭りという訳です.
	 *   仕方が無いのでmod_dav のフレームワークを逆手にとってここで検証します.
	 *   mod_dav が変わったらどうなるかわかりません.
	 */
	if (mnum == M_COPY || mnum == M_MOVE) {
		/* Destination uri の取得 */
		dst_uri = _get_destination_uri(r);
		(void) divy_parse_uri(p, dav_divy_get_root_uri(r), dst_uri, &dst_u_spec);

		/* メインリクエストのURIを使う */
		if (r->main != NULL) {
			request_uri = dav_divy_remove_endslash(p, r->main->uri);
		}
		else {
			request_uri = dav_divy_remove_endslash(p, r->uri);
		}
		ap_no2slash(request_uri);

		/* 検証しようとしているrdb_r->uri はsrc or dst ? */
		if (strcmp(rdb_r->uri, request_uri) == 0) {
			is_src_resource = 1;	/* src の検証だった */
			is_dst_resource = 0;
		}
		else if (IS_FILLED(dst_uri) && strcmp(rdb_r->uri, dst_uri) == 0) {
			is_src_resource = 0;
			is_dst_resource = 1;	/* dst の検証だった */
		}
	}

	/*
	 * ユーザアクセス権限のチェック (1)
	 * (システム実行権限の検証)
	 *
	 * [ 検証開始条件 ]
	 *   * ユーザ拡張ステータスがサポートされている
	 */
	if (support_extstatus) {
		/* 自動削除処理 or 開封通知の自動解除へのリクエストの場合
		 * システム実行権限を持たなければ許可しない */
		if (infotype == DIVY_INFOTYPE_autodelete && !divy_rdbo_has_sysexec_privilege(extstatus)) {
			ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server could not permit the autodelete-operation.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
		else if (infotype == DIVY_INFOTYPE_confirmreading && !divy_rdbo_has_sysexec_privilege(extstatus)) {
			ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server could not permit the operation of releasing confirmreading.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}

		/* アクセスユーザ自身がシステム実行権限を持つ場合
		 * 自動削除/開封通知の解除以外は許可してはならない(最優先) */
		if (divy_rdbo_has_sysexec_privilege(extstatus)) {
			if ((infotype == DIVY_INFOTYPE_autodelete && divy_autodel_enable(r, u_spec) < 0) ||
				(infotype == DIVY_INFOTYPE_confirmreading && divy_confirmreading_enable(r, u_spec) < 0)) {
				ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The server could not permit the access to this resource for system-exec user.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
			/* これ以降のチェックは不要
			 * (他のユーザ権限、ストリーム送信、パスワードポリシーとは無関係) */
			return NULL;
		}

		/* ユーザに対するアクセスの場合、対象ユーザがシステム実行権限を持っていれば
		 * アクセスを許可しない
		 * SEARCH, OPTIONS は許可すること. ただし、SEARCH(userinformationesarchのcontent)は
		 * 本来禁止しないとならないが、これはSEARCHプロバイダで実施してもらう) */
		if ((infotype == DIVY_INFOTYPE_m_user_e || infotype == DIVY_INFOTYPE_m_status_e) &&
			(mnum == M_DELETE   ||
			 mnum == M_PROPFIND || mnum == M_PROPPATCH ||
			 mnum == M_MKCOL    ||
			 mnum == M_MOVE     || mnum == M_COPY      ||
			 mnum == M_LOCK     || mnum == M_UNLOCK
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
			 ) && rdb_r->usr_pr != NULL && divy_rdbo_has_sysexec_privilege(rdb_r->usr_pr->extstatus)) {
			ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server could not permit the access to system-exec user.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/* グループ制約による書き込み制約の影響を受けるかどうか. */
	is_write_constraints = divy_rdbo_has_write_constraints_on_uri(r, rdb_r->uri, infotype);

	/*
	 * ユーザアクセス権限のチェック(2)
	 *
	 * [ 検証開始条件 ]
	 *   * ユーザ拡張ステータスがサポートされている
	 *   * rootフォルダ、プライベートコレクション以下、グループコレクション以下、
	 *     linkdbsearch結果フォルダ以下、共有コレクション以下、メールである
	 * (note)
	 *   プライベートコレクションは常に許可されているのでチェックされません。
	 */
	if (support_extstatus &&
		(infotype == DIVY_INFOTYPE_root                 ||
		 infotype == DIVY_INFOTYPE_group_e              ||
		 infotype == DIVY_INFOTYPE_group_trash          ||
		 infotype == DIVY_INFOTYPE_group_trash_e0       ||
		 infotype == DIVY_INFOTYPE_group_trash_ex       ||
		 infotype == DIVY_INFOTYPE_group_e_regular      ||
		 infotype == DIVY_INFOTYPE_dbfolder_e           ||
		 infotype == DIVY_INFOTYPE_dbfolder_e_regular   ||
		 infotype == DIVY_INFOTYPE_dbshfolder_e         ||
		 infotype == DIVY_INFOTYPE_dbshfolder_e_regular ||
		 infotype == DIVY_INFOTYPE_mail_e)) {

		deny = 0;
		/*
		 * Read-Write / Read-Only / Upload-Only ユーザのチェック
		 * (note)
		 * 	extstatus に複数の権限フラグが立っていたら readwrite, upload, read 
		 * 	の順に評価します。
		 */

		/*
		 * Read-Write の場合
		 */
		if (divy_rdbo_has_readwrite_privilege(extstatus) && !is_write_constraints) {
			/* 特にやることはありません */
		}
		/*
		 * Upload-Only の場合
		 */
		else if (divy_rdbo_has_upload_privilege(extstatus)) {

			/* [ 許可する操作 ]
			 *
			 *   OPTIONS, HEAD, GET(autoindex(コレクションへのGET)),
			 *   POST(autoindex(コレクションへのPOST)), PROPFIND,
			 *   SEARCH,  LOCK, UNLOCK, PUT(新規, メール送信用),
			 *   PROPPATCH(メール監視, 公開/非公開設定, サムネイル付加, 開封通知),
			 *   MKCOL(linkdbsearch結果フォルダ),
			 *   リソースファイルらしきファイルのすべての操作(divy_is_resourcefork(r)参照)
			 *   GETかPUTでかつ対象ファイルの作成者と現在のユーザIDが同じで,
			 *   作成日が今処理している時間よりconfのuldownloadallowtime時間以内 
			 *   プライベートコレクション以下のリソースに対する全ての操作
			 * (note)
			 * 	PROPPATCH(メール監視設定, 公開/非公開設定, サムネイル付加, 開封通知)
			 * 	のみを許可はチェック不能。ですのでPROPPATCH は許可しますので別にやって下さい。
			 */
			int isResourceFork = divy_is_resourcefork(r);
			if (!(mnum == M_OPTIONS     ||
				(mnum == M_GET && r->header_only)       ||
				(mnum == M_GET && resource->collection) ||
#ifdef DAV_SUPPORT_POST_HANDLING
				(mnum == M_POST && resource->collection)||
#endif	/* DAV_SUPPORT_POST_HANDLING */
				mnum == M_PROPFIND  || mnum == m_search ||
				mnum == M_LOCK      || mnum == M_UNLOCK ||
				(mnum == M_PUT && !resource->exists)    ||
				(mnum == M_PUT && infotype == DIVY_INFOTYPE_mail_e) ||
				mnum == M_PROPPATCH ||
				(isResourceFork) ||
				((mnum == M_GET||mnum == M_PUT) &&
				 (strcmp(rdb_r->creator_userid,divy_get_userid(r)) == 0) && 
				 rdb_r->creationdate + dconf->uldownloadallowtime > dav_divy_get_now_epoch()) ||
				(mnum == M_MKCOL && infotype == DIVY_INFOTYPE_dbfolder_e))) {

				ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server could not allow access this resource "
					"for upload-only user. (user = %s)", divy_get_userid(r));
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* さらに書き込み制約もされていた場合 */
			if (is_write_constraints) {
				/* ユーザのread-only と全く同じコードです.コピーコード. */
				/* [ 許可する操作 ]
				 *
				 *   OPTIONS, HEAD, GET, PROPFIND, SEARCH, PUT(メール送信用),
				 *   PROPPATCH(メール監視, 公開/非公開設定),
				 *   プライベートコレクション以下のリソースに対する全ての操作
				 * (note)
				 * 	PROPPATCH(メール監視設定, 公開/非公開設定, 開封通知) のみを許可は
				 * 	チェック不能。ですのでPROPPATCH は許可しますので別にやって下さい。
				 */
				if (!(mnum == M_OPTIONS    ||
					  mnum == M_GET      ||
					  mnum == M_PROPFIND || mnum == m_search ||
					  (mnum == M_PUT && infotype == DIVY_INFOTYPE_mail_e) ||
					  mnum == M_PROPPATCH
#ifdef DAV_SUPPORT_POST_HANDLING
					|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
					 )) {

					ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
							"The server could not allow access this resource "
							"for upload-only user and read-only group. (user = %s)", divy_get_userid(r));
					return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
				}
			}
		}
		/*
		 * Read-Only の場合
		 */
		else if (divy_rdbo_has_read_privilege(extstatus) || is_write_constraints) {
			/* [ 許可する操作 ]
			 *
			 *   OPTIONS, HEAD, GET, PROPFIND, SEARCH, PUT(メール送信用),
			 *   PROPPATCH(メール監視, 公開/非公開設定),
			 *   プライベートコレクション以下のリソースに対する全ての操作
			 * (note)
			 * 	PROPPATCH(メール監視設定, 公開/非公開設定, 開封通知) のみを許可は
			 * 	チェック不能。ですのでPROPPATCH は許可しますので別にやって下さい。
			 */
			if (!(mnum == M_OPTIONS    ||
				mnum == M_GET      ||
				mnum == M_PROPFIND || mnum == m_search ||
				(mnum == M_PUT && infotype == DIVY_INFOTYPE_mail_e) ||
				mnum == M_PROPPATCH
#ifdef DAV_SUPPORT_POST_HANDLING
				|| mnum == M_POST
#endif	/* DAV_SUPPORT_POST_HANDLING */
				)) {

				ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server could not allow access this resource "
					"for read-only user or read-only group. (user = %s)", divy_get_userid(r));
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
		}
		/* 何も設定されていなかった */
		else {
			/* 何もなければ何もさせない
			 * (このケースはデータ不正 or 意図的にこのようにしたケースのみ) */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The user did not have valid privileges. "
				"(user = %s)", divy_get_userid(r));
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/*
	 * 公開フォルダ/リソースへのアクセス検証 (1)
	 * 以下に該当する場合アクセス拒否
	 * [ 前提条件 ]
	 *   * 制限ユーザである
	 *   * グループコレクションよりも下の通常リソース/コレクションである
	 *   * 対象リソースは存在している
	 */
	if (support_extstatus &&
		!is_trusty && infotype == DIVY_INFOTYPE_group_e_regular && resource->exists) {

		deny = 0;

		/* 自身の状態プロパティは無効か ? */
		if (IS_INEFFECTIVE_STATE(rdb_r->rstate_pr)) {
			deny = 1;
		}

		/* 有効なView属性があった場合、
		 * グループコレクション直下のリソース/コレクションの検証 */
		if (!deny && p_infotype == DIVY_INFOTYPE_group_e) {

			/* 禁止されたオペレーションか?
			 * (2007/10/02 Tue)
			 *   PROPPATCH は許可することになりました. 開封通知や
			 *   サムネイルが付かないためです. ただし、公開非公開のパッチだけは
			 *   NGなので、個別のvalidator でチェックしてください. */
			if (mnum == M_MKCOL  || mnum == M_MOVE       ||
				//	mnum == M_DELETE || mnum == M_PROPPATCH  ||
					mnum == M_DELETE  ||
					(mnum == M_LOCK && resource->collection) ||
					(mnum == M_UNLOCK && resource->collection)) {
				deny = 2;
			}
			else if (mnum == M_COPY) {


				/* Destinationの親がグループフォルダだった場合 */
				if (dst_u_spec && dst_u_spec->p_infotype == DIVY_INFOTYPE_group_e) {
					deny = 3;	/* 拒否 */
				}
			}
		}

		/* 有効なView属性があった場合、
		 * 削除、コピー、移動、リネームできないリソースを含んでいないかどうか検証
		 * (note) 移動先、コピー先に有効でない状態があればやはり禁止 */
		if (!deny && (mnum == M_DELETE || mnum == M_COPY || mnum == M_MOVE)) {
			/* 有効でない状態プロパティが下にあるかどうか探す */
			ret = divy_rdbo_is_resourcestate_effective(r, rdb_r->uri,
					DIVY_RSTATE_TYPE_VIEW, 0/* 自身は無視 */);
			/* 有効でない状態プロパティを持っていた */
			if (ret == DIVY_STCODE_INEFFECTIVE_RSTATE_EXIST) {
				deny = 4;
			}
			else if (ret == DIVY_STCODE_EFFECTIVE_RSTATE_EXIST) {
				/* 問題なし。何もしない */
				deny = 0;
			}
			else {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get resourcestate.(uri = %s)", rdb_r->uri);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
		}

		if (deny) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Deny access to the resource. (reason = %d)", deny);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/*
	 * 公開フォルダ/リソースへのアクセス検証 (2)
	 * 以下に該当する場合アクセス拒否
	 * [ 前提条件 ]
	 *   * 制限ユーザである
	 *   * グループコレクションよりも下の通常リソース/コレクションである
	 *   * 対象リソースは存在しない
	 */
	if (support_extstatus &&
		!is_trusty && infotype == DIVY_INFOTYPE_group_e_regular && !resource->exists) {

		divy_rdbo_resourcestate *rstate = NULL;
		deny = 0;

		/* グループコレクション直下のリソース/コレクション */
		if (p_infotype == DIVY_INFOTYPE_group_e) {
			/* 禁止されたオペレーションか? (null LOCKも禁止) */
			if (mnum == M_PUT    ||
				mnum == M_MKCOL  || mnum == M_MOVE       ||
				mnum == M_DELETE || mnum == M_PROPPATCH  ||
				(mnum == M_LOCK && resource->collection) ||
				(mnum == M_UNLOCK && resource->collection)) {
				deny = 1;
			}
		}

		/* PUT, MKCOL の場合 */
		if (!deny && (mnum == M_PUT || mnum == M_MKCOL)) {
			/* 親リソースの状態・属性プロパティを調べる */
			if (divy_rdbo_get_parent_resourcestate_property(r, rdb_r->uri,
						DIVY_RSTATE_TYPE_VIEW, &rstate)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get parent resourcestate property.");
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}

			if (IS_INEFFECTIVE_STATE(rstate)) {
				deny = 2;
			}
		}
		/* MOVE, COPY の場合 */
		else if (!deny && (mnum == M_MOVE || mnum == M_COPY)) {

			/* src の検証だった */
			if (is_src_resource) {
				/* 存在しないsrc の検証には意味がない(どうせ404なので)
				 * という訳で何もしなくていい */
			}
			/* dst の検証だった */
			else if (is_dst_resource) {
				/* グループフォルダの直下であればコピー＆移動は禁止 */
				if (p_infotype == DIVY_INFOTYPE_group_e) {
					deny = 3;
				}
				/* グループフォルダの下の下の場合 */
				else {
					/* 親の状態に従うだけ */
					if (divy_rdbo_get_parent_resourcestate_property(r, rdb_r->uri,
								DIVY_RSTATE_TYPE_VIEW, &rstate)) {
						ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
								"Failed to get parent resourcestate property.");
						return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
					}

					if (IS_INEFFECTIVE_STATE(rstate)) {
						deny = 4;
					}
				}
			}
			/* 上記以外 */
			else {
				/* この関数はリクエストの基点となったリソースしか
				 * 扱えないって言ってるでしょ!!! */
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Could not validate this resource. (uri = %s)",
						rdb_r->uri);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
		}
		/* 残り全部 (OPTIONS, HEAD, GET, PROPFIND, SEARCH, DELETE, PROPPATCH,
		 * null LOCK, null UNLOCK) の場合 */
		else {
			/* 対象リソースが存在しないのだから拒否も何もない。
			 * 既にdeny状態であればそれもよし。何もしてはならない */
		}

		if (deny) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Deny access to the empty-resource. (reason = %d)", deny);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/*
	 * ストリームを読まずに最大Quotaリミットの検証
	 * 
	 * [ 検証開始条件 ]
	 *   * PUT メソッドであること
	 *   * 通常リソース or メールであること
	 *   * プラグイン専用アップロードファイルではない
	 *   * WebDAVメソッドゲートウェー経由のアップロードではない
	 * (note)
	 * 	ファイルストリームを操作する関数ではmod_dav のアーキテクチャ上
	 * 	ストリームを読み込んでしまうので、読み込まずに拒否するという
	 * 	外部仕様を実現することは不可能。
	 * 	仕方がないのでここで処理を行っています。
	 * 	原理原則を言えばストリームをハンドリングする関数上で処理すべきです。
	 */
	if (mnum == M_PUT && !resource->collection && (u_spec->uritype & DIVY_URITYPE_REGULAR)
#ifdef DIVY_SUPPORT_PLUGIN
			&& u_spec->infotype != DIVY_INFOTYPE_plugin_uploads_e
#endif	/* DIVY_SUPPORT_PLUGIN */
#ifdef DAV_SUPPORT_POST_HANDLING
			&& (r->main == NULL || r->main->method_number != M_POST)
#endif	/* DAV_SUPPORT_POST_HANDLING */
					) {

		const char *reject_softly;

		err = NULL;

		/* 強制切断可能なクライアントでのみ事前Quota確認を実施 */
		reject_softly = apr_table_get(r->subprocess_env, DIVY_APENV_REJECTQTOVERSOFTLY);
		if (!DIVY_APENV_IS_ON(reject_softly)) {

			/* maxfilesize 制限の検証 */
			err = _validate_pre_fsizelimit(r, infotype, rdb_r->getcontentlength);

#ifdef DIVY_SUPPORT_EXTENDQUOTA
			/* システムQuotaの事前判定 */
			if (err == NULL) {
				err = _validate_pre_sysquota(r, infotype, rdb_r->getcontentlength);
			}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

			/* ユーザQuota の事前確認 */
			if (err == NULL) {
				err = _validate_pre_userquota(r, infotype, rdb_r->getcontentlength);
			}

#ifdef DIVY_SUPPORT_GROUPQUOTA
			/* グループQuota の事前確認 */
			if (err == NULL) {
				err = _validate_pre_grpquota(r, infotype, rdb_r->getcontentlength);
			}
#endif /* DIVY_SUPPORT_GROUPQUOTA */
			/*
			 * Quota 制限を超えていたらソケットの強制Closeを実施する準備をする
			 * (note)
			 *	AP_CONN_CLOSE を設定してもいい根拠は、http_request.c の
			 *	ap_die 関数を参照。適切とは言いがたいのですが。
			 */
			if (err != NULL) {
				r->connection->keepalive = AP_CONN_CLOSE;
				return err;
			}
		}
	}

#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * パスワードの有効期限、猶予期間のチェック
	 * (例外)
	 *     $root の OPTIONS はパスワードポリシーチェックの例外とします.
	 *     .login に対してもユーザが未確定の為に例外とします。 
	 *     .st(短縮URL)に対してもここに来る必要はないので例外とする
	 *     .saml はパスワード管理はIdPで行うため例外とする
	 */
	if (!((infotype == DIVY_INFOTYPE_root && mnum == M_OPTIONS)
								|| (infotype == DIVY_INFOTYPE_login)
								|| (infotype == DIVY_INFOTYPE_shorten)
								|| (infotype == DIVY_INFOTYPE_saml))) {
		err = divy_validate_password_term(r, divy_get_userid(r),
					u_spec, DIVY_REQBODY_TYPE_UNKNOWN /* body はまだ不明 */);
		if (err != NULL) {
			return err;
		}
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* 任命済みグループリーダに関する検証 */
	err = _validate_appointed_groupleader(r, infotype, resource, is_src_resource);
	if (err != NULL) {
		return err;
	}

	/* グループリーダがグループにアクセスした場合の検証 */
	err = _validate_group_via_groupleader(r, infotype, rdb_r->grp_pr, is_src_resource, u_spec, dst_u_spec);
	if (err != NULL) {
		return err;
	}

	return NULL;
}

/**
 * URIのpath segment に含めてはならない文字が無いかどうか検証する。
 *
 * (note) path segment に含めてはならない文字
 * 	\/:,;*?"<>|	(いずれもWindowsのファイル名の制限)
 */
DIVY_DECLARE(int) divy_validate_pathsegment(apr_pool_t *p, int ignore, const char *str)
{
	int i, len = NOT_ALLOWED_SET_LEN;

	if (IS_EMPTY(str)) return 0;
	
	/* 禁則文字がstr の中にないかどうか調べる */
	for (i = 0; i < len; i++) {
		if (divy_strchr_c((char *)str, not_allowed_set[i]) != NULL) {
			if (ignore != 0 &&
				((not_allowed_set[i] == '\\' && (ignore & DIVY_IGNORE_BS)) ||
				 (not_allowed_set[i] == '/'  && (ignore & DIVY_IGNORE_SLASH)) ||
				 (not_allowed_set[i] == ':'  && (ignore & DIVY_IGNORE_COLON)) ||
				 (not_allowed_set[i] == '*'  && (ignore & DIVY_IGNORE_ASTERISK)) ||
				 (not_allowed_set[i] == '?'  && (ignore & DIVY_IGNORE_QUESTION)) ||
				 (not_allowed_set[i] == '"'  && (ignore & DIVY_IGNORE_DQUOTE)) ||
				 (not_allowed_set[i] == '<'  && (ignore & DIVY_IGNORE_LT)) ||
				 (not_allowed_set[i] == '>'  && (ignore & DIVY_IGNORE_GT)) ||
				 (not_allowed_set[i] == '|'  && (ignore & DIVY_IGNORE_BAR)))) {
				continue;	/* 無視する */
			}

			/* 禁則文字が見つかった場合 */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Found not allowed string in the path-segment."
				"(char = \"%c\", path segment = %s)",
				not_allowed_set[i], str);
			return 1;
		}
	}

	return 0;
}

/**
 * ユーザプロパティが正しいかどうか検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_user_property(request_rec *r,
					int chk_mode,
					const divy_user_iscreen *iscreen)
{
	apr_pool_t *p = r->pool;
	divy_uri_spec *u_spec;
	apr_int64_t maxst, maxres;
	int read, upload, readwrite;
	dav_error *err;
	const divy_rdbo_usr *src_usr_pr = NULL;
	int support_extstatus   = divy_support_extenduserstatus(r);
	int support_groupleader = divy_support_groupleader(r);
	int isAdmin = 0;	/* 管理者権限を持っていない */
	int isOwn   = 0;	/* 自分自身のプロパティではない */
	int support_grpconstraints = divy_support_grpconstraints(r);
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	int isChangePw = 0;	/* パスワード変更のためにユーザプロパティを変更するかどうか */
	int support_access_control = divy_support_access_control(r);

	/* userdiscovery プロパティがなかった場合 */
	if (iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userdiscovery\" element is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (iscreen->u_spec == NULL || IS_EMPTY(iscreen->u_spec->final_path_segment)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	u_spec = iscreen->u_spec;
	src_usr_pr = iscreen->src_usr_pr;

	/* userdiscovery を保持できるURIであること */
	if (u_spec->infotype != DIVY_INFOTYPE_m_user_e) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The server could not accept \"userdiscovery\" property "
				"on this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 管理者またはグループリーダ = 管理権限をもっているかどうか */
	if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN ||
		(support_groupleader && divy_rdbo_is_groupleader(own_extstatus))) {
		isAdmin = 1;
	}
	/* 自分自身のプロパティなの ? */
	if (strcmp(u_spec->final_path_segment, divy_get_userid(r)) == 0) {
		isOwn = 1;
	}

	/* 管理者権限を持たないユーザは他人のプロパティを何があろうとも触ってはならない */
	if (!isAdmin && !isOwn) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"You could not create or change other's user property."
				"Need to administrive privilege. (target user = %s)", u_spec->final_path_segment);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/*
	 * (note)
	 * 以下の判定では、管理者(他人 / 自身) or グループリーダ(他人 / 自身) or (一般 && 自分自身)
	 * しかないと考えていい
	 */

	/* ユーザID */
	if ((chk_mode & DIVY_VALIDATE_INSERT) && IS_EMPTY(iscreen->usrid)) {
		/* 登録時にユーザIDが無いのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userid\"(\"userdiscovery\" child) element is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	else if ((chk_mode & DIVY_VALIDATE_UPDATE) && IS_FILLED(iscreen->usrid)) {
		/* 更新時にユーザIDを指定されるのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userdiscovery\" element cannot have \"userid\" element for updating.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* ユーザIDの文字種/規則チェック */
	if (IS_FILLED(iscreen->usrid)) {

		/* 禁則チェック */
		if (divy_validate_pathsegment(p, 0, iscreen->usrid)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"userid\"(\"userdiscovery\" child) element has invalid char.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* 特殊フォルダの名称とバッティングしていないか？ */
		if (divy_validate_different_sfolder_name(p, iscreen->usrid)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Could not use the name of \"%s\" for userid",
				iscreen->usrid);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* リレーションプレフィックスの検証 */
		if (divy_validate_having_entity_prefix(p, iscreen->usrid)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"userid\"(\"userdiscovery\" child) "
				"element has relation prefix string.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* URI のユーザIDパート(final path segment) の比較 */
		if (strcmp(iscreen->usrid, u_spec->final_path_segment) != 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The content of \"name\"(\"userdiscovery\" child) "
				"element is different from the userid-part"
				"(final path segment) that the uri contained. "
				"(name = %s, final_path_segment = %s)",
				iscreen->usrid, u_spec->final_path_segment);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/* fullname がない */
	if (IS_EMPTY(iscreen->fullname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"name\"(\"userdiscovery\" child) element is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	/* (note) fullname を管理者以外が更新できないとしてしまうと、LDAP連携が崩壊します. 要注意 */

	/* password */
	if ((chk_mode & DIVY_VALIDATE_INSERT) && (IS_EMPTY(iscreen->password))) {
		/* MKCOLでパスワードが空の場合はエラー */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"password\"(\"userdiscovery\" child) element "
			"is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	else if ((chk_mode & DIVY_VALIDATE_UPDATE) && IS_EMPTY(iscreen->password)) {
		/* PROPPATCHでパスワードが空の場合はエラー */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"password\"(\"userdiscovery\" child) element is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
#ifdef DIVY_SUPPORT_PASSPOLICY
	/*
	 * パスワードポリシーに基づくパスワード文字列のチェック
	 * [ 条件 ]
	 *   * ユーザプロパティの更新処理であること
	 *   * 自分自身のパスワードを変更すること(ユーザの種別によらない)
	 *     --> 管理者画面のパスワード変更でも対象となります.
	 *  (note)
	 *    管理者が他人のパスワードを変更する行為は常にチェック対象外(外部仕様)
	 */
	if (chk_mode & DIVY_VALIDATE_UPDATE) {
		const char *u1 = divy_get_userid(r);
		const char *u2 = src_usr_pr->usrid;
		if (IS_FILLED(u1) && IS_FILLED(u2) && strcmp(u1, u2) == 0) {
			err = _validate_password_input(r, src_usr_pr,
					iscreen->password, 0/* plain かどうかは判断して */);
			if (err != NULL) {
				return err;
			}
		}
	}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* mailaddr の書式チェック */
	if (IS_FILLED(iscreen->mailaddr) && divy_ml_validate_mailaddr(p, iscreen->mailaddr) != ML_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"mailaddr\"(\"userdiscovery\" child) element "
			"has invalid format. (mailaddr = %s)",
			iscreen->mailaddr);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	/* (note) mailaddr を管理者以外が更新できないとしてしまうと、LDAP連携が崩壊します. 要注意 */

	/* registdtがある */
	if (IS_FILLED(iscreen->registdt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userdiscovery\" property cannot have \"creationdt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* updatedtがある */
	if (IS_FILLED(iscreen->updatedt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userdiscovery\" property cannot have \"updatedt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* accessdeny に指定された名称が全て正しくなかった ->
	 * 都合によりこの判定はパース時に実施することにしました */

	/* 管理者権限を持つ以外のユーザ or 管理権限を持つユーザ自身のaccessdeny に
	 * 触ってはならない(変更禁止) */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && (!isAdmin || (isAdmin && isOwn))) {
		int invalid = 0, i;
		if (iscreen->accessdeny != NULL && src_usr_pr->accessdeny != NULL) {
			for (i = 0; i < DIVY_FOLDER_ID_END; i++) {
				if (iscreen->accessdeny[i] != src_usr_pr->accessdeny[i]) {
					invalid = 1;
					break;
				}
			}
		}
		else if (iscreen->accessdeny != NULL && src_usr_pr->accessdeny == NULL) {
			/* iscreen のaccessdeny に含まれる値が全て0 ならばinvalidではない */
			for (i = 0; i < DIVY_FOLDER_ID_END; i++) {
				if (iscreen->accessdeny[i]) {
					invalid = 1;
					break;
				}
			}
		}
		else if (iscreen->accessdeny == NULL && src_usr_pr->accessdeny != NULL) {
			/* src のaccessdeny に含まれる値が全て0 ならばinvalidではない */
			for (i = 0; i < DIVY_FOLDER_ID_END; i++) {
				if (src_usr_pr->accessdeny[i]) {
					invalid = 1;
					break;
				}
			}
		}

		if (invalid) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not change user's accessdeny list.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/*
	 * Quota のチェック
	 */

	/* usedst が指定された(0 が指定されても無視) */
	if (IS_FILLED(iscreen->susedst) && apr_atoi64(iscreen->susedst) != APR_INT64_C(0)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userdiscovery\" property cannot have \"usedstrage.\" element "
			"(data = %s).", iscreen->susedst);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* usedres が指定された(0 が指定されても無視) */
	if (IS_FILLED(iscreen->susedres) && apr_atoi64(iscreen->susedres) != APR_INT64_C(0)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"userdiscovery\" property cannot have \"usedresource\" element "
			"(data = %s)", iscreen->susedres);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 最大Quotaサイズが数値かどうか */
	if (!dav_divy_isdigit_str(iscreen->smaxst)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The value of \"maxstorage\" (\"userdiscovery\" child) element must be digit. "
			"(data = %s)", iscreen->smaxst);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	maxst = apr_atoi64(iscreen->smaxst);

	/* 最大Quotaサイズがマイナス値 */
	if (maxst < APR_INT64_C(0)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"maxstorage\"(\"userdiscovery\" child) element has invalid value.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 管理者以外が最大Quotaサイズを更新するのはNG */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && !isAdmin && maxst != src_usr_pr->maxst) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"You could not change user's max-size userquota. Need to administrive privilege.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 最大Quotaファイル数が数値かどうか */
	if (!dav_divy_isdigit_str(iscreen->smaxres)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The value of \"maxresource\" (\"userdiscovery\" child) element must be digit. "
			"(data = %s)", iscreen->smaxres);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	maxres = apr_atoi64(iscreen->smaxres);

	/* 最大Quotaファイル数がマイナス値 */
	if (maxres < APR_INT64_C(0)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"maxresource\"(\"userdiscovery\" child) element has invalid value");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 管理者権限を持つユーザ以外が最大Quotaファイル数を更新するのはNG */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && !isAdmin && maxres != src_usr_pr->maxres) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"You could not change user's max-file userquota. Need to administrive privilege.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* ユーザの種類 (管理者権限) */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && iscreen->adminmode != src_usr_pr->adminmode) {
		/* 更新時に管理者権限(正確にはユーザの種類)が変更された場合 */

		/* 管理者は自分自身を降格することができない */
		if (isAdmin && isOwn) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not change own administrive privilege.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}

		/* 一般ユーザは管理者権限を変更することはできない */
		if (!isAdmin) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not change administrator privilege. Need to administrive privilege.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/*
	 * このプロパティ変更操作は、パスワード変更のために行われたものか?
	 * [ 条件 ]
	 *   * 更新操作である
	 *   * 自分自身のプロパティに対するアクセスである
	 *   * パスワード以外のプロパティ(設定のプロパティシートにある値) を変えていない
	 *   * パスワードが違っている
	 *
	 * なお、クライアントが古くても新しくてもパスワード変更にはuserdiscoveryの変更が
	 * 出てしまうケースがある. だから敢えて判定はしません.
	 */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) &&
		isOwn &&
		strcmp(iscreen->fullname, src_usr_pr->fullname) == 0 &&
		(IS_FILLED(iscreen->mailaddr) && IS_FILLED(src_usr_pr->mailaddr) && strcmp(iscreen->mailaddr, src_usr_pr->mailaddr) == 0) &&
		(IS_FILLED(iscreen->comment) && IS_FILLED(src_usr_pr->comment) && strcmp(iscreen->comment, src_usr_pr->comment) == 0) &&
		maxst == src_usr_pr->maxst &&
		maxres == src_usr_pr->maxres &&
		strcmp(iscreen->password, src_usr_pr->password) != 0) {

		/* (note) ユーザの種類や有効期限などの変えてはならない情報が変わっていたら
		 * それは別にチェックしていまうので、ここでは考えないことにします.
		 * パスワード変更の疑いが濃厚ということです */
		isChangePw = 1;	/* このフラグは後で使います */
	}

	/* ユーザ拡張ステータス機能が有効で
	 * この機能をサポートしたクライアントからのリクエストであるとき */
	if (support_extstatus && !iscreen->is_old_client) {

		/* ユーザ権限のチェック(排他ステータスが立っていないか) */
		read      = divy_rdbo_has_read_privilege(iscreen->extstatus);
		upload    = divy_rdbo_has_upload_privilege(iscreen->extstatus);
		readwrite = divy_rdbo_has_readwrite_privilege(iscreen->extstatus);
		if ((read && upload) || (read && readwrite) ||
			(upload && readwrite) || (read && upload && readwrite)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Could not specify \"read\" and \"upload\" and \"readwrite\" "
				"at the same time.");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* 制限ユーザのチェック */
		if (!divy_rdbo_is_trusty_user(iscreen->extstatus)) {

			/* 制限ユーザは管理者になれない */
			if (iscreen->adminmode == DIVY_ADMINMODE_ADMIN) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"Could not specify \"admin\" and \"limited\" "
					"at the same time.");

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* 制限ユーザにはView属性設定権限を与えてはならない */
			if (divy_rdbo_has_setview_privilege(iscreen->extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The limited user could not have \"set-view\" privilege.");

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* 制限ユーザにグループ制約の無視を許可してはならない
			 * (note) グループ制約をサポートしている場合(クライアントのバージョンは見ない) */
			if (support_grpconstraints &&
				divy_rdbo_has_user_groupconstraints_ignore(iscreen->extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The limited user could not have \"group-constraints-ignore\" privilege.");

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
		}

		/* 管理者のプロパティにグループ制約適用の有無を設定することはNG */
		if (support_grpconstraints &&
			divy_rdbo_has_user_groupconstraints_ignore(iscreen->extstatus) &&
			iscreen->adminmode == DIVY_ADMINMODE_ADMIN) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not set \"group-constraints-ignore\" for administrator.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}

		/* 一般ユーザによる更新の場合 */
		if ((chk_mode & DIVY_VALIDATE_UPDATE) && !isAdmin) {
			/* 管理者以外のユーザが値を更新するのはNG */
			int src_read, src_upload, src_readwrite;
			time_t expiration = 0L;

			src_read      = divy_rdbo_has_read_privilege(src_usr_pr->extstatus);
			src_upload    = divy_rdbo_has_upload_privilege(src_usr_pr->extstatus);
			src_readwrite = divy_rdbo_has_readwrite_privilege(src_usr_pr->extstatus);
			if (src_read != read || src_upload != src_upload || src_readwrite != readwrite) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change user's privilege. Need to administrive privilege.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* (自身の)View属性(公開権限)を変更はできない */
			if (divy_rdbo_has_setview_privilege(iscreen->extstatus) !=
				divy_rdbo_has_setview_privilege(src_usr_pr->extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change user's publisher privilege. Need to administrive privilege.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* (自身の)アクティブステータスを変更できない */
			if (divy_rdbo_is_active_user(iscreen->extstatus) !=
						divy_rdbo_is_active_user(src_usr_pr->extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change user's active status. Need to administrive privilege.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* (自身の)有効期限を変更できない */
			if (IS_FILLED(iscreen->sexpiration)) {
				expiration = dav_divy_iso8601totime_t(p, iscreen->sexpiration);
			}

			if (expiration != src_usr_pr->expiration) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change user's expiration status. Need to administrive privilege.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* (自身の)ユーザ種別を変更することはできない
			 * (note) adminmode の変更はチェック済み */
			if (divy_rdbo_is_trusty_user(iscreen->extstatus) !=
				divy_rdbo_is_trusty_user(src_usr_pr->extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change user type. Need to administrive privilege.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* (自身の)グループ制約適用の有無を変更することはできない */
			if (support_grpconstraints &&
				divy_rdbo_has_user_groupconstraints_ignore(iscreen->extstatus) !=
				divy_rdbo_has_user_groupconstraints_ignore(src_usr_pr->extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change \"group-constraints-ignore\". Need to administrive privilege.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
		}

		/* 管理権限を持つユーザによる更新の場合 */
		if ((chk_mode & DIVY_VALIDATE_UPDATE) && isAdmin) {
			time_t expiration = 0L;

			/* グループリーダ自身はそもそも自身のプロパティを変更できない
			 * パスワード変更は除外します. また、LDAP経由の変更でも無視します */
			if (isOwn &&
				support_groupleader && divy_rdbo_is_groupleader(own_extstatus) && !isChangePw) {

				/* LDAP経由を除外 */
				if (!(chk_mode & DIVY_VALIDATE_LDAP_UPDATE)) {

					ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
							"You could not change own user property."
							"Need to administrative privilege. (target user = %s)", divy_get_userid(r));
					return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
				}
			}

			/* 自身のView属性(公開権限)を変更できない */
			if (isOwn &&
				divy_rdbo_has_setview_privilege(iscreen->extstatus) !=
				divy_rdbo_has_setview_privilege(src_usr_pr->extstatus)) {

				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change own publisher privilege.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* 自身のアクティブステータスを変更できない */
			if (isOwn &&
				divy_rdbo_is_active_user(iscreen->extstatus) !=
			 	divy_rdbo_is_active_user(src_usr_pr->extstatus)) {

				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change own active status.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* 自身の有効期限を変更できない */
			if (IS_FILLED(iscreen->sexpiration)) {
				expiration = dav_divy_iso8601totime_t(p, iscreen->sexpiration);
			}

			if (isOwn && expiration != src_usr_pr->expiration) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change own expiration status.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* 自身のユーザ種別を変更できない
			 * (note) adminmode の変更はチェック済み */
			if (isOwn &&
				divy_rdbo_is_trusty_user(iscreen->extstatus) !=
				divy_rdbo_is_trusty_user(src_usr_pr->extstatus)) {

				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change own user type.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* 自身のグループ制約適用の有無を変更できない(勿論、他人のものは可能) */
			if (isOwn &&
				support_grpconstraints &&
				divy_rdbo_has_user_groupconstraints_ignore(iscreen->extstatus) !=
				divy_rdbo_has_user_groupconstraints_ignore(src_usr_pr->extstatus)) {

				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change own \"group-constraints-ignore\".");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
		}
	}

	/* グループ管理者機能が有効な場合 */
	if (support_groupleader) {

		/* 自身がグループリーダの場合 */
		if (divy_rdbo_is_groupleader(own_extstatus)) {

			/* 変更対象となるユーザがグループリーダや管理者だった場合には変更不可能
			 * ただし、パスワード変更のための更新, LDAPであれば特別に許可すべき */
			if ((chk_mode & DIVY_VALIDATE_UPDATE) && !(chk_mode & DIVY_VALIDATE_LDAP_UPDATE) &&
				(divy_rdbo_is_groupleader(src_usr_pr->extstatus) || src_usr_pr->adminmode == DIVY_ADMINMODE_ADMIN) &&
				!isChangePw) {

				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change the property of administrator or groupleaderuser. (ownid = %s)",
						divy_get_userid(r));
				/* (note) 意味は違うけどこのエラーコードを入れておくことにしました */
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_ADMINTYPE);	/* ヘッダに設定 */
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* Other ユーザはコピーできない (2007/10/10 Tue) */
			if ((chk_mode & DIVY_VALIDATE_COPY) && src_usr_pr->is_otheruser) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not copy other-user. (leader = %s, target = %s)",
						divy_get_userid(r), src_usr_pr->usrid);
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* 対象ユーザがグループリーダと管理者ユーザであれば
			 * グループリーダには作成 or 変更できない */
			if ((chk_mode & DIVY_VALIDATE_INSERT) &&
				(divy_rdbo_is_groupleader(iscreen->extstatus) || iscreen->adminmode == DIVY_ADMINMODE_ADMIN)) {

				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not set this user type. (ownid = %s)", divy_get_userid(r));
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_ADMINTYPE);	/* ヘッダに設定 */
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			/* (note) パスワード変更, LDAP経由は特別扱い */
			else if ((chk_mode & DIVY_VALIDATE_UPDATE) && !(chk_mode & DIVY_VALIDATE_LDAP_UPDATE) &&
					 (divy_rdbo_is_groupleader(iscreen->extstatus) || iscreen->adminmode == DIVY_ADMINMODE_ADMIN) &&
					 !isChangePw) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change user type of administrator or groupleader. (ownid = %s)",
						divy_get_userid(r));
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_ADMINTYPE);	/* ヘッダに設定 */
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* Other ユーザは編集できない
			 * (note) パスワード変更は特別扱い */
			if ((chk_mode & DIVY_VALIDATE_UPDATE) && !(chk_mode & DIVY_VALIDATE_LDAP_UPDATE) &&
				src_usr_pr->is_otheruser && !isChangePw) {

				ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not change other-user property. (leader = %s, target = %s)",
						divy_get_userid(r), src_usr_pr->usrid);
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_OTHERUSER);	/* ヘッダに設定 */
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
		}
		/* 自身が管理者の場合 */
		else {
			/* グループリーダを降格させる場合 */
			if ((chk_mode & DIVY_VALIDATE_UPDATE) &&
				divy_rdbo_is_groupleader(src_usr_pr->extstatus) &&
				(!divy_rdbo_is_groupleader(iscreen->extstatus) && iscreen->adminmode != DIVY_ADMINMODE_ADMIN)) {

				int has_user = 0;

				/* 管理下においているユーザを持っていればNG */
				if (divy_rdbo_has_owner_users(r, src_usr_pr->usrid, &has_user)) {

					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
							"Failed to get owner-user information.");
					return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				}

				if (has_user) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
							"Could not change usertype because you have owner-user.");

					divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NODEMOTE_GROUPLEADER);	/* ヘッダに設定 */
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
		}

		/*
		 * (2007.09.19 Wed) このチェックは省略します.
		 * そして、管理者・一般・制限ユーザは常に管理ユーザ数0、ユーザを管理下に置けないと
		 * 値を置き換えてしまいます. この方が親切でしょう.クライアントが意図した値には
		 * ならないけど. もともとこの設定は設定ミスでしか起きえませんし.
		 */
#if 0
		/* 対象ユーザがグループリーダではない場合(管理者、一般、制限ユーザ) */
		if (!divy_rdbo_is_groupleader(iscreen->extstatus)) {
			/* 管理ユーザ数が設定されていてはならない */
			if (IS_FILLED(iscreen->smaxusercreation) && atoi(iscreen->smaxusercreation) != 0) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
						"Could not specify \"maxusercreation\" element except groupleader.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* ユーザを管理下に置くことはできない */
			if (divy_rdbo_has_control_otheruser(iscreen->extstatus)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
						"Could not specify \"control-other-user\" element except groupleader.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
		}
#endif
	}

	/*
	 * アクセスコントール機能が有効な場合 
	 * IPアドレスの正当性のチェックのみを行います。
	 */
	if (support_access_control && IS_FILLED(iscreen->allowhosts)) {
		divy_array_t* addrs = NULL;
		if (_build_string2ipsubnet_array(r, iscreen->allowhosts, &addrs) == 0) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
						"Failed to allowhosts addr element. %s",
												iscreen->allowhosts);
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/*
	 * 登録数リミットのチェック
	 */
	if (chk_mode & DIVY_VALIDATE_INSERT) {
		/* ライセンス数との比較 */
		err = _validate_creating_user_limit(r);
		if (err) return err;

		/* グループリーダの最大作成数との比較 */
		if (support_groupleader) {
			err = _validate_control_user_limit(r);
			if (err) return err;
		}
	}

	return NULL;
}

/**
 * アップデートユーザがアクティブになる場合、サーバのライセンスに接触していない
 * か調査を行う。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_update_user_limit(request_rec* r)
{
	return _validate_creating_user_limit(r);
}

/**
 * グループプロパティが正しいかどうか検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_group_property(request_rec *r,
					int chk_mode,
					const divy_group_iscreen *iscreen)
{
	apr_pool_t *p            = r->pool;
	server_rec *s            = r->server;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	divy_uri_spec *u_spec;

	/* groupdiscovery プロパティがなかった場合 */
	if (iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupdiscovery\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	u_spec = iscreen->u_spec;

	/* groupdiscovery を保持できるURIであること */
	if (u_spec->infotype != DIVY_INFOTYPE_m_group_e) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server could not accept \"groupdiscovery\" property "
			"on this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* groupid が指定されていた */
	if (IS_FILLED(iscreen->grpid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupdiscovery\" property cannot have \"groupid\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* グループ名称 */
	if ((chk_mode & DIVY_VALIDATE_INSERT) && IS_EMPTY(iscreen->name)) {

		/* 登録時にグループ名称がないのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"name\"(\"groupdiscovery\" child) element is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	else if ((chk_mode & DIVY_VALIDATE_UPDATE) && IS_FILLED(iscreen->name)) {

		/* 更新時にグループ名称を指定されるのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupdiscovery\" element cannot have \"name\" element for updating.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* グループ名称の文字種/規則チェック */
	if (IS_FILLED(iscreen->name)) {

		/* 禁則チェック */
	    	if (divy_validate_pathsegment(p, 0, iscreen->name)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"name\"(\"groupdiscovery\" child) element has invalid char.");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* リレーションプレフィックスの検証 */
		if (divy_validate_having_entity_prefix(p, iscreen->name)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"name\"(\"groupdiscovery\" child) "
				"element has relation prefix string.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* URI のグループ名パート(final path segment) の比較 */
		if (strcmp(iscreen->name, u_spec->final_path_segment) != 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The content of \"name\"(\"groupdiscovery\" child) "
				"element is different from the name-part"
				"(final path segment) that the uri contained. "
				"(name = %s, final_path_segment = %s)",
				iscreen->name, u_spec->final_path_segment);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/* グループ階層の作成が制限されているとき、単階層のグループであること */
	if (dconf->syncgrpuri == DIVY_SYNCGRPURI_ON &&
	    divy_count_dirs(u_spec->other_part) > 1) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"At present, you could not create or update a group "
			"under another group. Please consider change of "
			"TeamFile setting.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}
	/* グループ階層は作成できるが、同名の表示名を持つグループの存在を許可しない場合
	   (PROPPATCH は同名のグループを編集するだけなので除外すること) */
	else if (dconf->syncgrpuri == DIVY_SYNCGRPURI_MIDDLE &&
							divy_get_method_number(r) != M_PROPPATCH) {
		divy_rdbo_grp *grp = NULL;
		if (divy_rdbo_get_group_property_by_name(r, u_spec->final_path_segment, &grp)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get group property by name.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 同名のグループが存在したらこれはNG */
		if (grp != NULL) {
			ERRLOG0(p, APLOG_NOTICE, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"We found a same group at another hierarchy."
				"This setting could not allow to make same group name.");
			return dav_new_error(p, HTTP_METHOD_NOT_ALLOWED, 0, 0, "");
		}
	}

	/* groupmailwatch のチェック */
	if (iscreen->mailwatch_pr != NULL) {
		int i, merged_trigger = 0;
		dav_divy_server_conf *sconf = dav_divy_get_server_config(s);

		/* メールオプションが無いのに、groupmailwatch が存在した場合 */
		if (sconf->use_mail_opt == 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The function of sending mail not supported.");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
		else {
			for (i = 0; i < METHODS; i++) {
				merged_trigger |= iscreen->mailwatch_pr->trigger_methods[i];
			}

			/* トリガメソッドが未指定 */
			if (merged_trigger == 0) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The content of \"triggermethod\" element "
					"is empty. Must contain a string-value.");

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* 正しい notification が指定されていること */
			if (iscreen->mailwatch_pr->notification == DIVY_NOTICE_UNKNOWN) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The child of \"notification\" element is missing.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
		}
	}

	/* creationdtが指定されていた */
	if (IS_FILLED(iscreen->registdt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupdiscovery\" property cannot have \"creationdt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* updatedtが指定されていた */
	if (IS_FILLED(iscreen->updatedt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"groupdiscovery\" property cannot have \"updatedt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	/* (note) constraints のチェックは特に不要 */

	/*
	 * グループリーダによる操作の制限 -> _validate_group_via_groupleader() に移動しました.
	 */

	return NULL;
}

/**
 * SQL プロパティが正しいかどうか検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_sql_property(request_rec *r,
					int chk_mode,
					const divy_sql_iscreen *iscreen)
{
	apr_pool_t *p                   = r->pool;
	divy_uri_spec *u_spec;
	const divy_rdbo_sql *sql_pr;
	divy_rdbo_sql *db_sql_pr; /* NULL なら後でひく */
	dav_divy_server_conf *sconf     = dav_divy_get_server_config(r->server);

	divy_sql_parser *parser         = NULL;
	divy_rdbo_sqldepend *dependlist = NULL, *dependlist_db     = NULL;
	divy_rdbo_sqldepend *first      = NULL, *dependlist_db_all = NULL, *d_list;
	divy_rdbo_sqlcontent *sqlc      = NULL;
	apr_hash_t *rsname_h            = NULL;
	divy_cset_t *usingsql_set       = NULL;
	int ret;
	divy_array_t *sqlcnt_array       = NULL;
	int do_inactive, do_change_type, do_delete_defval;
	divy_rdbo_clist *hidsubname_cl   = NULL, *first_hidsubname_cl = NULL;
	apr_hash_t *hidsubname_h         = NULL;

	/* sqldiscovery プロパティがなかった場合 */
	if (iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	u_spec    = iscreen->u_spec;
	sql_pr    = iscreen->sql_pr;
	db_sql_pr = (divy_rdbo_sql *) iscreen->db_sql_pr;

	/* sqldiscovery を保持できるURIであること */
	if (u_spec->infotype != DIVY_INFOTYPE_m_sql_e) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server could not accept \"sqldiscovery\" property "
			"on this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* SQLパーサの作成 */
	if (divy_sql_parser_create(p, &parser) != DIVY_SQLP_ST_OK){
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to create sql parser.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* SQLシーケンス番号 */
	if (IS_FILLED(sql_pr->sqlid)) {
		/* 登録・更新時にSQLシーケンス番号を持っていた */

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\" element cannot have \"sqlno\"");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* SQL表示名称 */
	if ((chk_mode & DIVY_VALIDATE_INSERT) && IS_EMPTY(sql_pr->labelname)) {

		/* 登録時にSQL表示名を持っていなかった */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"name\"(\"sqldiscovery\" child) "
			"element is missing or empty.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	else if ((chk_mode & DIVY_VALIDATE_UPDATE) && IS_FILLED(sql_pr->labelname)) {

		/* 更新時にSQL表示名を指定された */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\" element cannot have "
			"\"name\" for updating.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* SQL表示名の文字種/規則チェック */
	if (IS_FILLED(sql_pr->labelname)) {

		/* 禁則チェック */
		if (divy_validate_pathsegment(p, 0, sql_pr->labelname)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"name\" (\"sqldiscovery\" child) "
				"element has invalid char");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* リレーションプレフィックスの検証 */
		if (divy_validate_having_entity_prefix(p, sql_pr->labelname)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"name\"(\"sqldiscovery\" child) "
				"element has relation prefix string.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
		/* URI のSQL表示名パート(final path segment) の比較 */
		if (strcmp(sql_pr->labelname, u_spec->final_path_segment) != 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The content of \"name\"(\"sqldiscovery\" child) "
				"element is different from the labelname-part"
				"(final path segment) that the uri contained. "
				"(name = %s, final path segment = %s)",
				sql_pr->labelname, u_spec->final_path_segment);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/* 名前付きバインド変数構文以外ではSQL文が存在すること */
	if ((sql_pr->type != DIVY_SQL_TYPE_BIND) && IS_EMPTY(sql_pr->sql)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"statement\"(\"sqldiscovery\" child) "
			"element is missing or empty.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* DB機能が利用できない時には"通常SQL"を許可してはならない */
	if (sconf->use_db_opt != 1 && sql_pr->type == DIVY_SQL_TYPE_NORMAL) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Could not find valid db license. so,"
			"we could not permit to register a sql "
			"that has normalsql type.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 使用可能なDBMS一覧 */
	if (sql_pr->dbms_pr != NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\" property cannot have "
			"\"dbmslist\" element.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 使用DBMS名称 */
	if (IS_EMPTY(sql_pr->usedbmsname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"usedbmsname\"(\"sqldiscovery\" child) "
			"element is missing or empty.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 作成日付 */
	if (IS_FILLED(sql_pr->registdt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\" element cannot have "
			"\"creationdt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 更新日付 */
	if (IS_FILLED(sql_pr->updatedt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sqldiscovery\" element cannot have "
			"\"updatedt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* サブ名称 (RequiredSQL、名前付きバインド変数のみ) */
	ret = divy_sql_parser_validate_subname(parser, r, sql_pr->type,
					u_spec->other_part,
					(chk_mode & DIVY_VALIDATE_INSERT),
					sql_pr->subname);
	if (ret != DIVY_SQLP_ST_OK) {
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* cachemode の検証 */
	if (sql_pr->cachemode == DIVY_SQL_CACHE) {
		ret = divy_sql_parser_validate_cachemode(parser,
					sql_pr->type, sql_pr->subname,
					sql_pr->sql, sql_pr->cachemode);
		if (ret != DIVY_SQLP_ST_OK) {
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/*
	 * リポジトリ検索SQLの場合、SELECT 句に指定できるカラムが正しいこと
	 */
	if (sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) {
		apr_hash_t *invalid   = NULL;

		/* SELECT句の検証 */
		ret = divy_sql_parser_validate_selectcol(parser, sql_pr->sql, &invalid);
		if (ret != DIVY_SQLP_ST_OK) {

			if (ret == DIVY_SQLP_ST_NONSELECT) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The SELECT phrase did not exist in "
					"SQL statement. (sql = %s)", 
					sql_pr->sql);
			}
			else if (ret == DIVY_SQLP_ST_NONFROM){
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The FROM phrase did not exist in "
					"SQL statement. (sql = %s)", 
					sql_pr->sql);
			}
			else if (ret == DIVY_SQLP_ST_MUSTCOL_NF && invalid) {
				apr_hash_index_t *idx;
				char *msg = "", *str = NULL;
				dav_buffer sbuf = { 0 };
	
				dav_set_bufsize(p, &sbuf, 1024);
				sbuf.cur_len = 0;
				
				for (idx = apr_hash_first(p, invalid);
					idx; idx = apr_hash_next(idx)) {
					apr_hash_this(idx, (const void **) &str, NULL, NULL);

					dav_buffer_append(p, &sbuf, str);
					dav_buffer_append(p, &sbuf, ",");
				}
				msg = sbuf.buf;

				ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"Some required column did not exist in "
					"SELECT clause. Please check column name "
					"and correct this."
					"(sql = %s, missing column = %s)",
					sql_pr->sql, msg);
			}
			else {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_PROC,
					"An error occured while validate "
					"column(s) of SELECT");
			}

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/*
	 * WHERE句プロパティ(sql_pr->sqlcnt_pr) の検証(1)
	 * (検証内容)
	 * 	・名前付きバインド変数構文ではないこと
	 *	・表示名称の存在
	 *	・$$SSxxx、$$SMxxx、$$Bxxxが指定された時、この名称の存在
	 *	・$$SSxxx、$$SMxxx、$$Bxxxが指定された時、この名称の形式
	 *	・contype とsql_pr->typeの組合せに対するsqlmode の整合性
	 *	・sqlposition が存在すること
	 *
	 * (検証開始条件)
	 * 	・WHERE句プロパティが存在すること
	 */
	if (sql_pr->type == DIVY_SQL_TYPE_BIND && sql_pr->sqlcnt_pr) {

		/* 名前付きバインド変数構文にWHERE句プロパティがあった */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The namedbind sentence could not have "
			"the value of \"keyvalue\" element."
			"(bodywhere_id = %d, bodywhere_name = \"%s\")",
			sql_pr->sqlcnt_pr->id,
			REPLACE_NULL(sql_pr->sqlcnt_pr->name));

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* id 属性値の重複と歯抜けをチェックするための配列を生成 */
	if (sql_pr->sqlcnt_pr) {
		sqlcnt_array = divy_array_make(p, 10);
	}

	for (sqlc = sql_pr->sqlcnt_pr; sqlc; sqlc = sqlc->next) {
		/* DBカラム名 (バインド変数ならば必須) */

		/* 表示名称 */
		if (IS_EMPTY(sqlc->name)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"dispcolumnname\"(\"keyvalue\" child) "
				"element is missing or empty."
				"(keyvalue id = %d)", sqlc->id);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* RequiredSQL、名前付きバインド変数のサブ名称が存在するか */
		if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED &&
		    (sqlc->reqsql == NULL || sqlc->reqsql->rsvalue == NULL ||
		     IS_EMPTY(sqlc->reqsql->rsvalue->name))) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"reqsqlname\"(\"keyvalue\" child) "
				"element is missing or empty."
				"(keyvalue id = %d)", sqlc->id);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");

		}

		/* サブ名称の形式(RequiredSQL、名前付きバインド変数のみ) */
		if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED &&
		    !IS_REQSQL_NODE(sqlc->reqsql->rsvalue->name)) {
			/* 形式が正しくなかった */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The format of \"reqsqlname\"(\"keyvalue\" child) "
				"element is invalid. (data = %s, keyvalue id = %d)",
				sqlc->reqsql->rsvalue->name, sqlc->id);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* sqlmode とcontype の組合せチェック */
		if (sqlc->contype == DIVY_SQL_CONTYPE_BIND &&
		    (sqlc->reqsql && (
		     sqlc->reqsql->sqlmode == DIVY_SQL_MODE_SHOW ||
		     sqlc->reqsql->sqlmode == DIVY_SQL_MODE_HIDDEN ||
		     sqlc->reqsql->sqlmode == DIVY_SQL_MODE_SELECTED ||
		     sqlc->reqsql->sqlmode == DIVY_SQL_MODE_MULTISELECTED))) {

			/* 通常バインド変数には、sqlmodeを指定出来ません */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The normal-bind variable cannot have "
				"the \"sqlmode\" element."
				"(keyvalue id = %d, sqlmode = %d)",
				sqlc->id, sqlc->reqsql->sqlmode);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED &&
		    sqlc->reqsql->sqlmode == DIVY_SQL_MODE_UNKNOWN) {

			/* RequiredSQLには、sqlmode を指定しなければなりません */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"sqlmode\" element is missing. "
				"The RequiredSQL require the value of \"sqlmode\"."
				"(keyvalue id = %d, subname = %s)",
				sqlc->id, sqlc->reqsql->rsvalue->name);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* sqlmode とcontype & type の組合せチェック */
		/* 通常SQL、リポジトリSQL */
		if (sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
		    sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) {

			/* $$SM 以外のSQLはselectedとmultiselected を持てない */
			if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED &&
			    !IS_M_REQSQL_NODE(sqlc->reqsql->rsvalue->name) &&
			    (sqlc->reqsql->sqlmode == DIVY_SQL_MODE_SELECTED ||
			     sqlc->reqsql->sqlmode == DIVY_SQL_MODE_MULTISELECTED)) {

				ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The conbinations of the \"sqlmode\" that is "
					"specified in WHERE property and \"sqltype\" "
					"is wrong.(sqltype = %d, sqlmode = %d, "
					"subname = \"%s\")", sql_pr->type,
					sqlc->reqsql->sqlmode,
					sqlc->reqsql->rsvalue->name);

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");

			}
		}
		/* RequiredSQL */
		else if (sql_pr->type == DIVY_SQL_TYPE_REQUIRED) {

			/* $$SS, $$SM の場合、show 以外のモードを持てない */
			if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED &&
			    !IS_NBIND_NODE(sqlc->reqsql->rsvalue->name) &&
			    sqlc->reqsql->sqlmode != DIVY_SQL_MODE_SHOW) {

				ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The conbinations of the \"sqlmode\" that "
					"is specified in WHERE property and \"sqltype\" "
					"is wrong. When using in RequiredSQL, "
					"you must set the \"sqlmode\" to \"show\"."
					"(sqltype = %d, sqlmode = %d, namedbind = \"%s\")",
					sql_pr->type, sqlc->reqsql->sqlmode,
					sqlc->reqsql->rsvalue->name);

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			/* $$B の場合、show, hidden 以外は持てない */
			else if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED &&
				 IS_NBIND_NODE(sqlc->reqsql->rsvalue->name)) {

				if (sqlc->reqsql->sqlmode == DIVY_SQL_MODE_SHOW) {
					/* 問題なしなので何もしない */
				}
				else if (sqlc->reqsql->sqlmode == DIVY_SQL_MODE_HIDDEN) {

					/* システム予約された変数ならばOK */
					ret = divy_sql_parser_is_special_reservedsql(parser,
							sqlc->reqsql->rsvalue->name);

					/* 予約されていなかった場合,
					   問題がある可能性あり。リストを作る */
					if (ret != DIVY_SQLP_ST_RESERVED_SQL) {
						if (first_hidsubname_cl == NULL) {
							first_hidsubname_cl =
							hidsubname_cl = apr_pcalloc(p,
									sizeof(divy_rdbo_clist));
						}
						else {
							hidsubname_cl->next = apr_pcalloc(p,
								sizeof(divy_rdbo_clist));
							hidsubname_cl = hidsubname_cl->next;
						}
						hidsubname_cl->val  = sqlc->reqsql->rsvalue->name; 
						hidsubname_cl->next = NULL;
					}
				}
				else {
					/* 使用できない */
					ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The conbinations of the \"sqlmode\" that "
						"is specified in WHERE property and \"sqltype\" "
						"is wrong. When using in RequiredSQL, "
						"you must set the \"sqlmode\" to \"show\" "
						"or \"hidden\" with namedbind."
						"(sqltype = %d, sqlmode = %d, namedbind = \"%s\")",
						sql_pr->type, sqlc->reqsql->sqlmode,
						sqlc->reqsql->rsvalue->name);

					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
		}
		/* 名前付きバインド変数構文 */
		else {	/* そもそもWHERE句検証のループに入らないはずなので無視 */
		}

		/* sqlposition が存在すること(存在しなければ負の値) */
		if (sqlc->sqlposition < 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"sqlposition\" (\"keyvalue\" child) "
				"element is missing.");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* RequiredSQL,名前付きバインド変数の依存関係リストの作成 */
		if (sqlc->contype == DIVY_SQL_CONTYPE_REQUIRED) {
			/* subname のハッシュを作成 */
			if (rsname_h == NULL) rsname_h = apr_hash_make(p);
			apr_hash_set(rsname_h, sqlc->reqsql->rsvalue->name,
							APR_HASH_KEY_STRING, "");
			/* 循環参照チェックのために依存関係リストを作成 */
			if (first == NULL){
				first = dependlist =
					apr_pcalloc(p, sizeof(divy_rdbo_sqldepend));
			} else {
				dependlist->next =
					apr_pcalloc(p, sizeof(divy_rdbo_sqldepend));
				dependlist = dependlist->next;
			}
			
			if (sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
	    		    sql_pr->type == DIVY_SQL_TYPE_BIND) {
				/* sql_pr がRequiredSQL、名前付きバインド変数の場合 */
				dependlist->ptsubname = sql_pr->subname;
			}
			else {
				/* sql_pr が通常SQL、リポジトリSQLの場合(ダミーIDを入れる) */
				dependlist->ptsubname = DIVY_SQL_TOPPTVALUE;
			}
			dependlist->clsubname = sqlc->reqsql->rsvalue->name;
		}

		/* sqlcnt_pr を記録しておく */
		divy_array_add(sqlcnt_array, sqlc);
	}
	dependlist = first;	/* 先頭のポインタを代入 */

	/*
	 * WHERE 句プロパティに指定された名前付きバインド変数が
	 * hiddenモードで指定されていた場合の特別処理
	 */
	if (first_hidsubname_cl) {
		divy_rdbo_sql *hid_sql_pr = NULL;

		/* 指定されていた名前付き変数のデフォルト値を検索する */
		ret = divy_rdbo_get_sql_by_subname(r, first_hidsubname_cl,
						&hidsubname_h ,NULL);
		if (ret) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get sql by subname.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		for (hidsubname_cl = first_hidsubname_cl;
				hidsubname_h && hidsubname_cl;
				hidsubname_cl = hidsubname_cl->next) {

			/* デフォルト値の検証 */
			hid_sql_pr = apr_hash_get(hidsubname_h, hidsubname_cl->val,
							APR_HASH_KEY_STRING);
			if (hid_sql_pr) {
				ret = divy_sql_parser_is_nbind_setting(parser,
								hid_sql_pr->sql);
				/* デフォルト値が無かった場合 */
				if (ret == DIVY_SQLP_ST_NF_NBIND_VAL) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The conbinations of the \"sqlmode\" that "
						"is specified in WHERE property and \"sqltype\" "
						"is wrong. When using in RequiredSQL, "
						"you must set the \"sqlmode\" to \"show\" "
						"with value-less namedbind."
						"(sqltype = %d, namedbind = \"%s\")",
						sql_pr->type, hidsubname_cl->val);
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
				else if (ret == DIVY_SQLP_ST_FOUND_NBIND_VAL) {
					/* 問題なし */
				}
				else {
					/* エラー */
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to \"is_nbind_setting\".");
					return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				}
			}
		}
	}

	/*
	 * WHERE句プロパティ(sql_pr->sqlcnt_pr) の検証(2)
	 * (検証内容)
	 * 	・sql_pr->sqlcnt_pr->id の先頭インデックス番号
	 * 	・sql_pr->sqlcnt_pr->id の重複
	 * 	・sql_pr->sqlcnt_pr->id の歯抜け
	 *
	 * (検証開始条件)
	 * 	・WHERE句プロパティが存在すること
	 */
	if (sqlcnt_array && divy_array_getlength(sqlcnt_array) > 0) {
		int i, prev = 0, id;
		apr_int32_t length = divy_array_getlength(sqlcnt_array);
		divy_rdbo_sqlcontent *prev_sqlcnt = NULL;
		void **data = divy_array_getelements(sqlcnt_array);

		/* ソートの実施 */
		if (length > 1) {
			qsort(data, length, sizeof(void *), _compar_id);
		}

		/* 比較(id が負のケースはparseでチェック済み) */
		for (i = 0; i < length; i++) {
			sqlc = data[i];
			id   = sqlc->id;

			/* 先頭のインデックス番号は１でなければならない */
			if ((i == 0 && id != 1) ||
			    (prev_sqlcnt && id != 1 &&
			     prev_sqlcnt->contype != sqlc->contype)) {

				ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The first value of the \"id\" attribute "
					"must be \"1\".(id = %d, dispname = %s)",
					id, sqlc->name);

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			/* 重複している */
			else if (prev != 0 && prev_sqlcnt && prev == id &&
				 prev_sqlcnt->contype == sqlc->contype) {

				ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The value of \"id\" attribute "
					"must be single.(duplicate id = %d, dispname = %s)",
					id, sqlc->name);

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
			/* 歯抜け値がある */
			else if (prev != 0 && prev + 1 < id &&
				 prev_sqlcnt->contype == sqlc->contype) {

				ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The set of \"id\" attribute must be dense. "
					"There is a lost value."
					"(lost id = %d, current id = %d, dispname = %s)",
					prev + 1, id, sqlc->name);

				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* 今の値を記憶しておく */
			prev        = id;
			prev_sqlcnt = sqlc;
		}
	}

	/*
	 * SQL文がTeamFileの仕様に基づいて記述されているかどうかの検証
	 * (note)
	 * 	WHERE句プロパティの検証もSQLパーサに一部任せたので、
	 * 	"WHERE句プロパティ(sql_pr->sqlcnt_pr) の検証(2)"によりも
	 * 	後でチェックしなければならなくなりました。
	 * 	(さもなければ、正しいエラーメッセージが出ません。)
	 */
	ret = divy_sql_parser_validate_reqsql_withbody(parser, sql_pr->type,
						sql_pr->sql, sql_pr->sqlcnt_pr);
	if (ret != DIVY_SQLP_ST_OK) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The syntax of SQL is wrong. "
			"Please check sql statement.(sql = %s, retcd = %d)",
			sql_pr->sql, ret);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * WHERE句プロパティ(sql_pr->sqlcnt_pr) の検証(3)
	 * (検証内容)
	 * 	・RequiredSQL、名前付きバインド変数が DBに登録されている
	 * 	・RequiredSQL、名前付きバインド変数が アクティブである
	 *
	 * (検証開始条件)
	 * 	・WHERE句プロパティにRequiredSQL、名前付きバインド変数が指定
	 */
	if (rsname_h) {
		apr_hash_t *not_in_h = NULL;

		/* 存在しない or 非アクティブなSQLはあるか？ */
		ret = divy_rdbo_check_required(r, rsname_h, &not_in_h, 0, NULL);
		/* 予期しないエラーが発生 */
		if (ret == 1) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"An error occurred while checking "
				"existence of RequiredSQL.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		/* あった */
		else if (ret == DIVY_STCODE_REQUIRED_NF) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The RequiredSQL which is specified in sql "
				"statement is missing or not available."
				"(sql = %s)", sql_pr->sql);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");

		}
	}

	/*
	 * WHERE句プロパティ(sql_pr->sqlcnt_pr) の検証(4)
	 * (検証内容)
	 * 	・sql_pr を追加・更新することで、依存SQLの連鎖数が
	 * 	  最大値を超えないこと
	 *
	 * (検証開始条件)
	 * 	・WHERE句プロパティにRequiredSQL、名前付きバインド変数が指定
	 */
	if (rsname_h) {

		/* DB からrsname_h をトップノードとする依存関係一覧を取得 */
		ret = divy_rdbo_get_sqldepend_list(r, rsname_h,
						&dependlist_db, NULL);	
		if (ret != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"An error occurred while getting sqldepend of "
				"RequiredSQL.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		/* リストの連結 */
		if (dependlist_db) {
			if (dependlist) {
				for (d_list = dependlist_db; d_list->next;
						d_list = d_list->next);
				d_list->next = dependlist;
			}
		}
		else {
			dependlist_db = dependlist;
		}

		ret = divy_sql_parser_validate_maxchain(parser, dependlist_db);
		/* 連鎖数が制限値を超えた */
		if (ret == DIVY_SQLP_ST_OVER_MAX_CHAIN) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The chain limit is over."
				"Please correct relations. (sql = %s)",
				sql_pr->sql);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
		/* 予期しないエラー発生 */
		else if (ret != DIVY_SQLP_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to search sqldependency "
				"for checking the count of max sql chain. (sql = %s)",
				sql_pr->sql);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	/*
	 * WHERE句プロパティ(sql_pr->sqlcnt_pr) の検証(5)
	 * (検証内容)
	 * 	・sql_pr を追加・更新することで、依存関係がループしないこと
	 *
	 * (検証開始条件)
	 * 	・WHERE句プロパティにRequiredSQL、名前付きバインド変数が指定
	 * 	・sql_pr のSQLがRequiredSQLであること。
	 */
	if (rsname_h && sql_pr->type == DIVY_SQL_TYPE_REQUIRED) {

		/*
		 * 依存関係がループしているかどうか調べる
		 * (note) dependlist_db は 検証(4) で取得済みのはず
		 */
		ret = divy_sql_parser_validate_closedpath(parser, dependlist_db);
		/* 依存関係がループした */
		if (ret == DIVY_SQLP_ST_LOOP_DETECTED) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Closed Loop detected. "
				"Please correct relations. (sql = %s)",
				sql_pr->sql);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
		/* 予期しないエラー発生 */
		else if (ret != DIVY_SQLP_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to search sqldependency "
				"for checking closed-loop. (sql = %s)",
				sql_pr->sql);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	/*
	 * WHERE句プロパティ(sql_pr->sqlcnt_pr) の検証(6)
	 * (検証内容)
	 * 	・sql_pr のSQLを利用する他のSQLがないこと
	 *
	 * (検証開始条件)
	 * 	・更新処理であること。
	 * 	・sql_pr のSQLがRequiredSQL、名前付きバインド変数構文ならば、
	 * 	  アクティブフラグを非アクティブに変更しようとしている
	 * 	  or
	 * 	  sql_pr のSQLが通常SQL・リポジトリSQLならば、sqltype を
	 * 	  RequiredSQLまたは名前付きバインド変数から変更しようとしている
	 * 	  or
	 * 	  sql_pr が名前付きバインド変数構文で、
	 * 	  他のSQLからhidden指定されていて、
	 * 	  デフォルト値を消そうとしている
	 */
	do_inactive      = 0;
	do_change_type   = 0;
	do_delete_defval = 0;
	if (chk_mode & DIVY_VALIDATE_UPDATE) {

		/* db_sql_pr がNULLならDBをひく */
		if (db_sql_pr == NULL && divy_rdbo_get_sql_property(r,
					u_spec->other_part, &db_sql_pr)) {
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 非アクティブにしようとしていた */
		if (sql_pr->active == DIVY_SQL_INACTIVE &&
		    (sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
		     sql_pr->type == DIVY_SQL_TYPE_BIND)) {
			do_inactive = 1;
		}

		/* sqltype を変えようとしていた */
		if ((sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
		     sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) &&
		    (db_sql_pr->type == DIVY_SQL_TYPE_REQUIRED ||
		     db_sql_pr->type == DIVY_SQL_TYPE_BIND)) {
			do_change_type = 1;
		}

		/* (仮) デフォルト値を消そうとしていた */
		if (sql_pr->type == DIVY_SQL_TYPE_BIND && IS_EMPTY(sql_pr->sql)) {
			do_delete_defval = 1;	/* (仮) デフォルト値を消してしまう */
		}
	}

	if ((chk_mode & DIVY_VALIDATE_UPDATE) &&
	    (do_inactive || do_change_type || do_delete_defval)) {

		char *subname;
		if (sql_pr->type == DIVY_SQL_TYPE_NORMAL ||
		    sql_pr->type == DIVY_SQL_TYPE_REPOSITORY) {
			subname = db_sql_pr->subname;	/* 昔の値を利用 */
		}
		else {
			subname = sql_pr->subname;
		}

		/* DB から世の中に存在する全ての依存関係一覧を取得する */
		ret = divy_rdbo_get_sqldepend_list(r, NULL, &dependlist_db_all, NULL);	
		if (ret != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"An error occurred while getting sqldepend of "
				"RequiredSQL.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		/* リストの連結 */
		if (dependlist) {
			/* (note) dependlist はもう使用しないのでこの後ろに
			 * 	dependlist_db_all を繋げます。この方がループ回数が
			 * 	少なくて済むからです。 */
			for (d_list = dependlist; d_list->next; d_list = d_list->next);
			d_list->next = dependlist_db_all;
			dependlist_db_all = dependlist;	/* 入れ替える */
		}

		/* 利用するSQLがあるかどうか調べる */
		ret = divy_sql_parser_find_usingsql(parser, subname,
						dependlist_db_all, &usingsql_set);
		/* 他から利用されていた */
		if (ret == DIVY_SQLP_ST_USINGSQL_FOUND) {
			/* 非アクティブ化は失敗である */
			if (do_inactive) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not use \"inactive\" element, "
					"because this sql was made necessary from "
					"other SQL.(subname = %s)", subname);
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* sqltypeの変更は失敗である */
			if (do_change_type) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not change the value of \"sqltype\" "
					"element to \"normalsql\" or \"reposdbsql\", "
					"because this sql was made necessary from "
					"other SQL.(subname = %s)", subname);
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			/* デフォルト値の消去は失敗の可能性あり */
			if (do_delete_defval) {
				apr_hash_t *usingsql_namelist = NULL;

				/* hidden 利用されていなければ許可します */
				divy_rdbo_get_sql_labelname_by_idsubname_sqlcsubnamemode(r,
						divy_cset_tohash(usingsql_set),
						sql_pr->subname,
						DIVY_SQL_MODE_HIDDEN,
						&usingsql_namelist, NULL);
				if (usingsql_namelist != NULL) {
					/* hidden 利用されていたので駄目 */
					ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not make the value of \"statement\" "
						"element empty, "
						"because this sql was made necessary from "
						"other SQL.(subname = %s)", subname);
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
		}
		/* 予期しないエラー発生 */
		else if (ret != DIVY_SQLP_ST_OK) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to check the usingsql. "
				"(subname = %s)", subname);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * システムメッセージプロパティが正しいかどうか検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_sysmsg_property(request_rec *r,
					int chk_mode,
					const divy_sysmsg_iscreen *iscreen)
{
	apr_pool_t *p = r->pool;
	divy_uri_spec *u_spec;

	if (iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sysmsgdiscovery\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	u_spec  = iscreen->u_spec;

	/* sysmsgdiscovery を保持できるURIであること */
	if (u_spec->infotype != DIVY_INFOTYPE_m_msg_e) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server could not accept \"sysmsgdiscovery\" property "
			"on this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* メッセージID */
	if ((chk_mode & DIVY_VALIDATE_INSERT) && IS_EMPTY(iscreen->msgid)) {
		/* 登録時にメッセージIDがないのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"msgid\"(\"sysmsgdiscovery\" child) element is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	else if ((chk_mode & DIVY_VALIDATE_UPDATE) && IS_FILLED(iscreen->msgid)) {
		/* 更新時にメッセージIDがあるのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sysmsgdiscovery\" element cannot have \"msgid\" element for updating.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* メッセージIDの文字種/規則チェック */
	if (IS_FILLED(iscreen->msgid)) {

		/* 数値チェック */
		if (!dav_divy_isdigit_str(iscreen->msgid)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The value of \"msgid\"(\"sysmsgdiscovery\" "
				"child) element must be digit. (msgid = %s)",
				iscreen->msgid);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* URI のメッセージIDパート(final path segment) の比較 */
		if (strcmp(iscreen->msgid, u_spec->final_path_segment) != 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The content of \"msgid\"(\"sysmsgdiscovery\" child) "
				"element is different from the msgid-part"
				"(final path segment) that the uri contained. ");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}
	
	/* creationdtが指定されていた */
	if (IS_FILLED(iscreen->registdt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sysmsgdiscovery\" property cannot have "
			"\"creationdt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* updatedtが指定されていた */
	if (IS_FILLED(iscreen->updatedt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sysmsgdiscovery\" property cannot have "
			"\"updatedt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* LANGが文字以外（数字などはNG)だった */
	if (IS_FILLED(iscreen->lang))
	{
		if (dav_divy_isdigit_str(iscreen->lang)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The value of \"lang\"(\"sysmsgdiscovery\" "
			"child) element must not be digit. (msgid = %s)",
			iscreen->msgid);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* 2文字以上はNG */	
		if (strlen(iscreen->lang) > 2) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The \"sysmsgdiscovery\" property \"lang\" element missing.");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");

		}
	}

	return NULL;
}

/**
 * DBMS プロパティが正しいかどうか検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_dbms_property(request_rec *r,
					int chk_mode,
					const divy_dbms_iscreen *iscreen)
{
	apr_pool_t *p = r->pool;
	divy_uri_spec *u_spec;

	/* dbmsdiscovery プロパティがなかった場合 */
	if (iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsdiscovery\" element is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	u_spec = iscreen->u_spec;

	/* dbmsdiscovery を保持できるURIであること */
	if (u_spec->infotype != DIVY_INFOTYPE_m_dbms_e) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server could not accept \"dbmsdiscovery\" property "
			"on this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* dbmsid が指定されていた */
	if (IS_FILLED(iscreen->dbmsid)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsdiscovery\" property cannot have \"dbmsid\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* DBMS 識別名称 */
	if ((chk_mode & DIVY_VALIDATE_INSERT) && IS_EMPTY(iscreen->name)) {

		/* 登録時にDBMS識別名称がないのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"name\"(\"dbmsdiscovery\" child) element is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	else if ((chk_mode & DIVY_VALIDATE_UPDATE) && IS_FILLED(iscreen->name)) {

		/* 更新時にDBMS 識別名称を指定されるのはNG */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsdiscovery\" element cannot have \"name\" element for updating.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* DBMS 識別名称の文字種/規則チェック */
	if (IS_FILLED(iscreen->name)) {

		/* 禁則チェック */
		if (divy_validate_pathsegment(p, 0, iscreen->name)) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"name\"(\"dbmsdiscovery\" child) "
				"element has invalid char.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* リレーションプレフィックスの検証 */
		if (divy_validate_having_entity_prefix(p, iscreen->name)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"name\"(\"dbmsdiscovery\" child) "
				"element has relation prefix string.");
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* URI のDBMS名称パート(final path segment) の比較 */
		if (strcmp(iscreen->name, u_spec->final_path_segment) != 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The content of \"name\"(\"dbmsdiscovery\" child) "
				"element is different from the dbmsname-part"
				"(final path segment) that the uri contained. "
				"(name = %s, final_path_segment = %s)",
				iscreen->name, u_spec->final_path_segment);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/* DBMS 種類 */
	if (IS_EMPTY(iscreen->type)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"type\"(\"dbmsinfo\" child) element is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* DBMS ホスト名 */
	if (IS_EMPTY(iscreen->hostname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"hostname\"(\"dbmsinfo\" child) element is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* DBMS ポート番号 */
	if (IS_EMPTY(iscreen->sport) || !dav_divy_isdigit_str(iscreen->sport)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"hostport\"(\"dbmsinfo\" child) element is missing or empty. "
			"(port = %s)", REPLACE_NULL(iscreen->sport));

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* DBMS データベース名称 */
	if (IS_EMPTY(iscreen->dbname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbname\"(\"dbmsinfo\" child) element is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}


	/* DBMS 接続ユーザ名 */
	if (IS_EMPTY(iscreen->username)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"username\"(\"dbmsinfo\" child) element is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}


	/* DBMS 接続パスワード */
	if (IS_EMPTY(iscreen->password)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"password\"(\"dbmsinfo\" child) element is missing or empty.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 非アクティブであると指定されたとき */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && (iscreen->active == DIVY_DBMS_INACTIVE)) {
		apr_hash_t *usingdbms_set = NULL;

		/* このDBMSを利用しているSQL一覧を取得 */
		if (divy_rdbo_find_sql_usingdbms(r,
				u_spec->final_path_segment, &usingdbms_set)) {
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* Failed Dependency を返却する --> そういえば返却できませんでした */
		if (usingdbms_set && apr_hash_count(usingdbms_set) > 0) {
			/* (note) mod_davの制約のためFailed Dependency の原因を
			 * 返却することが出来ません。*/
			return dav_new_error(p, HTTP_FAILED_DEPENDENCY, 0, 0, "");
		}
	}

	/* creationdtが指定されていた */
	if (IS_FILLED(iscreen->registdt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsdiscovery\" property cannot have \"creationdt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* updatedtが指定されていた */
	if (IS_FILLED(iscreen->updatedt)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"dbmsdiscovery\" property cannot have \"updatedt\" element.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	return NULL;
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * systemquota プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_systemquota_property(request_rec *r,
					divy_sysquota_iscreen *sysquota_iscreen)
{
	const char *authkey;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	dav_divy_dir_conf    *dconf = dav_divy_get_dir_config(r);
	divy_error *derr = NULL;
	apr_pool_t *p    = r->pool;
	divy_rdbo_diskquota *cur_squota = NULL;
	int ret;
	apr_int64_t maxst, maxres;

	/* sysquota_iscreen がNULLではないこと */
	if (sysquota_iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"systemquota\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (sysquota_iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* $root に対するリクエストであること */
	if (sysquota_iscreen->u_spec->infotype != DIVY_INFOTYPE_root) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request-uri must be root-uri. (uri = %s)",
			sysquota_iscreen->u_spec->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 認証ヘッダの取得 */
	authkey = apr_table_get(r->headers_in, DIVY_HEADER_AUTHKEY);

	/* サーバライセンスキーとDIVY_HEADER_AUTHKEY の値が正確に一致していること */
	if (IS_EMPTY(authkey)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"%s\" request header is missing.", DIVY_HEADER_AUTHKEY);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}
	if (strcmp(authkey, sconf->svrlicensekey) != 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The value of \"%s\" request header is not right.", DIVY_HEADER_AUTHKEY);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 管理者ユーザであること */
	if (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"You need to have administrative authority "
			"for setting of system-quota.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* sysquota_iscreen->s_usedst に値が設定されていないこと */
	if (IS_FILLED(sysquota_iscreen->s_usedst)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"used-bytes\" element must be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sysquota_iscreen->s_maxst に値が設定されていて、数値であること. */
	if (IS_EMPTY(sysquota_iscreen->s_maxst)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"available-bytes\" element must not be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	if (!dav_divy_isdigit_str(sysquota_iscreen->s_maxst)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"available-bytes\" element must not be digit.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sysquota_iscreen->s_total_bytes に値が設定されていないこと */
	if (IS_FILLED(sysquota_iscreen->s_total_bytes)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"assigned-bytes\" element must be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sysquota_iscreen->s_avail_bytes に値が設定されていないこと */
	if (IS_FILLED(sysquota_iscreen->s_avail_bytes)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"free-bytes\" element must be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sysquota_iscreen->s_usedres に値が設定されていないこと */
	if (IS_FILLED(sysquota_iscreen->s_usedres)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"used-files\" element must be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sysquota_iscreen->s_maxres に値が設定されていて、数値であること. */
	if (IS_EMPTY(sysquota_iscreen->s_maxres)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"available-files\" element must not be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	if (!dav_divy_isdigit_str(sysquota_iscreen->s_maxres)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"available-files\" element must not be digit.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sysquota_iscreen->s_total_files に値が設定されていないこと */
	if (IS_FILLED(sysquota_iscreen->s_total_files)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"assigned-files\" element must be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* sysquota_iscreen->s_avail_files に値が設定されていないこと */
	if (IS_FILLED(sysquota_iscreen->s_avail_files)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"free-files\" element must be EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 現システムQuota情報の取得 */
	ret = divy_rdbo_get_systemquota_property(r, &cur_squota, NULL);
	if (ret) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get system quota information.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* システムQuotaエントリが存在しなかった(互換性に関する考慮) */
	if (cur_squota == NULL) {
		divy_rdbo_usr *sum_uquota = NULL;

		/* 全ユーザQuotaの総計を算出 */
		if (divy_rdbo_get_allocated_userquota(r, &sum_uquota)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get user quota information.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		cur_squota = apr_pcalloc(p, sizeof(divy_rdbo_diskquota));
		cur_squota->usedst  = sum_uquota->usedst;
		cur_squota->usedres = sum_uquota->usedres;
		cur_squota->maxst   = DIVY_UNLIMITED_SYSQUOTA_BYTES; /* 無制限 */
		cur_squota->maxres  = DIVY_UNLIMITED_SYSQUOTA_FILES; /* 無制限 */

		/* 物理Diskの情報を取得 */
		derr = divy_statfs(p, dconf->fsrootpath, &cur_squota->sfs);
		if (derr != NULL) {
			DUMP_ERRLOGS(p,derr);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
	}
	sysquota_iscreen->sfs = cur_squota->sfs;

	maxst  = apr_atoi64(sysquota_iscreen->s_maxst);
	maxres = apr_atoi64(sysquota_iscreen->s_maxres);

	/* ignore_assigned_quota が未設定ならばシステムの最大値を超えていないこと */
	if (!sysquota_iscreen->ignore_assigned_quota) {
		if (maxst != DIVY_UNLIMITED_SYSQUOTA_BYTES &&
		    maxst > cur_squota->sfs->total_bytes) {

			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The size-quota is bigger than the capacity of "
				"physical disk. (maxst = %"APR_INT64_T_FMT
				", disksize = %"APR_UINT64_T_FMT")",
				maxst, cur_squota->sfs->total_bytes);
			return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
		}

		if (maxres != DIVY_UNLIMITED_SYSQUOTA_FILES &&
		    maxres > cur_squota->sfs->total_files) {

			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The file-quota is bigger than the capacity of "
				"physical disk. (maxres = %"APR_INT64_T_FMT
				", diskfile = %"APR_UINT64_T_FMT")",
				maxres, cur_squota->sfs->total_files);
			return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
		}
	}

	/* 使用容量よりも下回らないこと(無制限Quotaは除外する) */
	if (maxst != DIVY_UNLIMITED_SYSQUOTA_BYTES && maxst < cur_squota->usedst) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The size-quota is smaller than the used capacity. "
			"(used capacity = %"APR_INT64_T_FMT
			", maxst = %"APR_UINT64_T_FMT")",
			cur_squota->usedst, maxst);
		return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}

	if (maxres != DIVY_UNLIMITED_SYSQUOTA_FILES && maxres < cur_squota->usedres) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The file-quota is smaller than the used capacity. "
			"(used capacity = %"APR_INT64_T_FMT
			", maxres = %"APR_UINT64_T_FMT")",
			cur_squota->usedres, maxres);
		return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

/**
 * mailwatch プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_mailwatch_property(request_rec *r,
					int chk_mode,
					const divy_mailwatch_iscreen *mwatch_iscreen)
{
	apr_pool_t *p    = r->pool;
	server_rec *s    = r->server;
	const char *uri;
	divy_uri_spec *u_spec;
	divy_rdbo_mailwatch *mwatch_pr;
	dav_divy_server_conf *sconf    = NULL;
	dav_divy_dir_conf    *dconf    = NULL;

	/* mailwatchプロパティがなかった場合 */
	if (mwatch_iscreen == NULL || mwatch_iscreen->mwatch_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"mailwatch\" property is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (mwatch_iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	u_spec    = mwatch_iscreen->u_spec;
	mwatch_pr = mwatch_iscreen->mwatch_pr;
	uri       = u_spec->uri;
 
	/* メール機能ライセンスがなかった場合 */
	sconf = dav_divy_get_server_config(s);
	if (sconf->use_mail_opt == 0) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_PROC,
			"The license of mail cannot be recognized.");

		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/* メール機能がOFFになっていた場合 */
	dconf = dav_divy_get_dir_config(r);
	if (IS_EMPTY(dconf->mail) || strcmp(dconf->mail, DIVY_ML_OFF) == 0) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_PROC,
			"Could not set mailwatch property. "
			"The function of Mail is OFF.");

		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/* 制限ユーザではないこと */
	if (divy_support_extenduserstatus(r) && !divy_rdbo_is_trusty_user(divy_get_extstatus(r))) {

		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Need to normal or admin privilege.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/*
	 * マーク可能コレクションであること。
	 * (note) 種類
	 *	$root/Group Folder/$groupid
	 *	$root/.dbfolder/$keys
	 *	$root/.dbshfolder/$name
	 *	$root/.management/GROUP/$name/.RU_XXXX
	 */
	if (u_spec->infotype != DIVY_INFOTYPE_group_e &&
	    u_spec->infotype != DIVY_INFOTYPE_dbfolder_e &&
	    u_spec->infotype != DIVY_INFOTYPE_dbshfolder_e &&
		u_spec->infotype != DIVY_INFOTYPE_m_group_ruser) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"This uri cannot have \'maiwatch\' property. "
			"(uri = %s)", uri);

		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/* オペレーションの判定 */
	if (chk_mode & DIVY_VALIDATE_DELETE) {
		/* DELETEの時には以下を実施する必要なし */
		return NULL;
	}

	/* 正しい notification が指定されていること */
	if (mwatch_pr->notification == DIVY_NOTICE_UNKNOWN) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The child of \"notification\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	for ( ; mwatch_pr; mwatch_pr = mwatch_pr->next){
		int i = 0, merged_trigger   = 0;
		
		/* トリガメソッドのチェック */
		for (i = 0; i < METHODS; i++) {
			merged_trigger |= mwatch_pr->trigger_methods[i];
		}

		/* トリガメソッドが未指定 */
		if (merged_trigger == 0) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The content of \"triggermethod\" element "
				"is empty. Must contain a string-value.");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * sharedcollection プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_sharedcollection_property(request_rec *r,
					int chk_mode,
					const divy_sharedcollection_iscreen *shcol_iscreen)
{
	apr_pool_t *p        = r->pool;
	char *shuri_top_part = NULL, *tmp;
	divy_rdbo_shlink *shlink_pr;
	const char *uri;

	/* sharedcollectionプロパティがなかった場合 */
	if (shcol_iscreen->shlink_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"sharedcollection\" property is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (shcol_iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	shlink_pr = shcol_iscreen->shlink_pr;
	uri       = shcol_iscreen->u_spec->uri;

	/* Read-Only, Upload-Onlyユーザではないこと */
	if (divy_support_extenduserstatus(r)) {
		const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);

		if (!divy_rdbo_has_readwrite_privilege(extstatus) &&
			(divy_rdbo_has_upload_privilege(extstatus) ||
			 divy_rdbo_has_read_privilege(extstatus))) {

			ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The upload-ony or read-only user could not change "
				"\"sharedcollection\" property.");
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/* リクエストURIはlinkdbsearch結果フォルダでない */
	if (shcol_iscreen->u_spec->infotype != DIVY_INFOTYPE_dbfolder_e) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server dose not allow this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* オペレーションの判定 */
	if (chk_mode & DIVY_VALIDATE_DELETE) {
		/* DELETEの時には以下を実施する必要なし */
		return NULL;
	}

	/* 共有コレクションのURIの上部を作成する */
	shuri_top_part = divy_build_dbshfolder_uri(p, dav_divy_get_root_uri(r), "/");

	for ( ; shlink_pr; shlink_pr = shlink_pr->next) {
		/* shname (共有コレクションの表示名) がない */
		if (IS_EMPTY(shlink_pr->displayname)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"shname\"(\"sharedcollection\" child) "
				"element is missing or empty. (uri = %s)", uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* shname に禁則文字が含まれている */
		if (divy_validate_pathsegment(p, 0, shlink_pr->displayname)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"shname\"(\"sharedccollection\" child) "
				"element has invalid char."
				"(shname = %s)", shlink_pr->displayname);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* linkuri (uri) がない */
		if (IS_EMPTY(shlink_pr->uri)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"linkuri\"(\"sharedcollection\" child) "
				"element is missing or empty.");

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		/* uri は共有コレクションではない */
		tmp = divy_strstr(shlink_pr->uri, shuri_top_part);
		if (tmp == NULL || tmp != shlink_pr->uri) {
			/* shlink_pr->uri の先頭がshuri_top_part ではない */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The content of \"linkuri\"(\"sharedcollection\" "
				"child) element does not have a valid "
				"shared collection folder uri."
				"(linkuri = %s)", shlink_pr->uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}


		/* uri のfinal path segment に禁則文字が含まれている */
		tmp = divy_strrchr(shlink_pr->uri, '/');
		if (divy_validate_pathsegment(p, 0, ++tmp)) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"linkuri\"(\"sharedccollection\" child) "
				"element has invalid char."
				"(linkuri = %s)", shlink_pr->uri);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * updatediscovery プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_updatediscovery_property(request_rec *r, 
					const divy_clmodule_iscreen *clmodule_iscreen)
{
	apr_pool_t *p = r->pool;
	const char *uri;
	divy_rdbo_clmodule *clmodule_pr;

	/* updatediscovery が無かった場合 */
	if (clmodule_iscreen == NULL || clmodule_iscreen->clmodule_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"updatediscovery\" property is missing.");

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (clmodule_iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	clmodule_pr = clmodule_iscreen->clmodule_pr;
	uri         = clmodule_iscreen->u_spec->uri;

	/* updatediscovery を保持できるURIであること */
	if (clmodule_iscreen->u_spec->infotype != DIVY_INFOTYPE_m_update_e) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The server could not accept \"updatediscovery\" property "
			"on this resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* name が指定されていた */
	if (IS_FILLED(clmodule_pr->name)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"updatediscovery\" property cannot have "
			"\"name\" for updating. (uri = %s)", uri);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* version 未指定 */
	if (IS_EMPTY(clmodule_pr->version)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"version\"(\"updatediscovery\" child) "
			"element is missing. (uri = %s)", uri);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* lineup 未指定 */
	if (IS_EMPTY(clmodule_pr->lineup)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"lineup\"(\"updatediscovery\" child) "
			"element is missing. (uri = %s)", uri);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* digest 未指定 */
	if (IS_EMPTY(clmodule_pr->digest)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"digest\"(\"updatediscovery\" child) "
			"element is missing. (uri = %s)", uri);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* creationdt が指定されていた */
	if (IS_FILLED(clmodule_pr->registdt)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"updatediscovery\" property cannot have "
			"\"creationdt\" element for updating. "
			"(uri = %s)", uri);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* updatedt が指定されていた */
	if (IS_FILLED(clmodule_pr->updatedt)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"updatediscovery\" property cannot have "
			"\"updatedt\" element for updating. "
			"(uri = %s)", uri);

		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	return NULL;
}

/**
 * resourcestate プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_resourcestate_property(request_rec *r,
					int chk_mode,
					divy_resourcestate_iscreen *rstate_iscreen)
{
	dav_error *err = NULL;
	apr_pool_t *p  = r->pool;
	divy_resourcestate_iscreen *iscreen;

	/* resourcestate プロパティがなかった場合 */
	if (rstate_iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"resourcestate\" property is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	
	/* 拡張ステータス機能は有効か? */
	if (!divy_support_extenduserstatus(r)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"resourcestate\" property did not supported.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 全てのtype に対し処理を検証処理を実施 */
	for (iscreen = rstate_iscreen; iscreen; iscreen = iscreen->next) {

		if (iscreen->u_spec == NULL) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"u_spec is NULL.(type = %d)", iscreen->type);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		if (IS_EMPTY(iscreen->uri)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"uri is NULL.(type = %d)", iscreen->type);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* type 毎に検証 */
		switch (iscreen->type) {

			/* View属性の検証 */
			case DIVY_RSTATE_TYPE_VIEW:
				err = _validate_view_state(r, chk_mode, iscreen);

				break;
			/* BOX属性の検証 */
			case DIVY_RSTATE_TYPE_BOX:
				err = _validate_box_state(r, chk_mode, iscreen);

				break;
			default:
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Unknown type.(type = %d)", iscreen->type);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* エラーを見つけたら直ちに報告したほうがいい */
		if (err) return err;
	}

	return NULL;
}

/**
 * thumbnail プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_thumbnail_property(request_rec *r,
					int chk_mode,
					divy_thumbnail_iscreen *thumbnail_iscreen)
{
	apr_pool_t *p  = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const divy_rdbo_extstatus *extstatus = NULL;

	/* thumbnail プロパティがなかった場合 */
	if (thumbnail_iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"thumbnail\" property is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* サムネイル機能は有効か? */
	if (dconf->thumbnail != DIVY_THUMBNAIL_ON) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The \"thumbnail\" property did not supported.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if ((chk_mode & DIVY_VALIDATE_UPDATE) &&
		 IS_EMPTY(thumbnail_iscreen->data)) {

		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The value of \"thumbnail\" is EMPTY.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

#ifdef DAV_SUPPORT_POST_HANDLING
	/*
	 * サムネイルを作ってもよい場所か?
	 */
	if (thumbnail_iscreen->u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
		tfs_hset_t *groupid_set = NULL;
		divy_rdbo_grp *grp_pr = NULL;

		/* リソースURIからグループプロパティを検索する */
		if (divy_rdbo_get_group_property_by_resuri(r, thumbnail_iscreen->u_spec->uri, &grp_pr)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get group property.(uri = %s)",
					thumbnail_iscreen->u_spec->uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 暗号化対象グループか? */
		if (divy_pi_cciphered_grpset(r, &groupid_set) == DIVY_PL_ST_OK &&
			groupid_set != NULL && grp_pr != NULL) {
			if (tfs_hset_contains(groupid_set, grp_pr->grpid)) {
				ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
						"Could not make thumbnail because this group was protected.");
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, NULL);
			}
		}

		/* read-only ユーザにはサムネイルの作成を許可しない(グループフォルダのみ対象) */
		if (divy_support_extenduserstatus(r)) {

			extstatus = divy_get_extstatus(r);
			if (divy_rdbo_has_read_privilege(extstatus)) {
				ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"The server could not allow creating thumbnail on this resource for read-only user"
						"(userid = %s)", divy_get_userid(r));
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, NULL);
			}
		}

		/* グループ制約による書き込み制約の影響を受ける場合にもサムネイルは許可しない */
		if (divy_support_grpconstraints(r) && 
				divy_rdbo_has_write_constraints_on_uri(r,
					thumbnail_iscreen->u_spec->uri, thumbnail_iscreen->u_spec->infotype)) {

			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The server could not allow creating thumbnail on this resource for read-only group"
					"(userid = %s)", divy_get_userid(r));
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, NULL);
		}
	}
#endif	/* DAV_SUPPORT_POST_HANDLING */

	return NULL;
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * passwordpolicy プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_passwordpolicy_property(request_rec *r,
					int chk_mode,
					divy_passwordpolicy_iscreen *passpolicy_iscreen)
{
	apr_pool_t *p = r->pool;
	int ival;

	/* passpolicy_iscreen がNULLではないこと */
	if (passpolicy_iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"passwordpolicy\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	if (passpolicy_iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* 管理者ユーザの管理フォルダ に対するリクエストであること */
	if (passpolicy_iscreen->u_spec->infotype != DIVY_INFOTYPE_m_user) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The request-uri must be management-user uri. (uri = %s)",
			passpolicy_iscreen->u_spec->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 管理者ユーザであること */
	if (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"You need to have administrative authority "
			"for setting of passwordpolicy.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	if ((chk_mode & DIVY_VALIDATE_UPDATE)) {
		apr_time_t comptime;
		apr_time_t now = apr_time_sec(apr_time_now());
		int isInt32    = (sizeof(time_t) == 4) ? 1 : 0;

		/* 最小パスワード長さは適切な数値(0 から64) であること */
		if (IS_FILLED(passpolicy_iscreen->sminlen)) {
			if (!dav_divy_isdigit_str(passpolicy_iscreen->sminlen)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"\"minimum-password-length\" must be digit.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			ival = atoi(passpolicy_iscreen->sminlen);
			if (ival > DIVY_MAX_PASSWORD_LEN) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"\"minimum-password-length\" is bigger than 64.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
		}

		/* パスワード更新周期(日)は適切な数値であること */
		if (IS_FILLED(passpolicy_iscreen->schange_pw_cycle)) {
			if (!dav_divy_isdigit_str(passpolicy_iscreen->schange_pw_cycle)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"\"change-pasword-cycle\" must be digit.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			if (isInt32) {
				/* 更新周期 + 現在時刻を秒数に直した時、
				 * time_t の範囲を超えないこと */
				comptime = (apr_time_t) atoi(passpolicy_iscreen->schange_pw_cycle);
				comptime = comptime * 24 * 3600 + now;
				if (comptime > INT_MAX) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"The \"change-pasword-cycle\" is too large, "
							"so, the server could not handle this value."
							"(val = %s)", passpolicy_iscreen->schange_pw_cycle);
					return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				}
			}
		}

		/* パスワード期限切れ後の猶予期間(日)は適切な数値であること */
		if (IS_FILLED(passpolicy_iscreen->sprobation)) {
			if (!dav_divy_isdigit_str(passpolicy_iscreen->sprobation)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"\"pasword-probation\" must be digit.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			if (isInt32) {
				/* 猶予期間 + 現在時刻を秒数に直した時、
				 * time_t の範囲を超えないこと */
				comptime = (apr_time_t) atoi(passpolicy_iscreen->sprobation);
				comptime = comptime * 24 * 3600 + now;
				if (comptime > INT_MAX) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"The \"pasword-probation\" is too large, "
							"so, the server could not handle this value."
							"(val = %s)", passpolicy_iscreen->sprobation);
					return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				}
			}
		}

		/* パスワード履歴保持件数も数値(0 から 12)であること */
		if (IS_FILLED(passpolicy_iscreen->shistory_pw_num)) {
			if (!dav_divy_isdigit_str(passpolicy_iscreen->shistory_pw_num)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"\"change-history-number\" must be digit.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			ival = atoi(passpolicy_iscreen->shistory_pw_num);
			if (ival > 12) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"\"change-history-number\" is bigger than 12.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}
		}

		/* 初回ログインパスワード変更日付のリミット日も数値であること */
		if (IS_FILLED(passpolicy_iscreen->sfirstlogin_limitday)) {
			if (!dav_divy_isdigit_str(passpolicy_iscreen->sfirstlogin_limitday)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"\"firstlogin-limitday\" must be digit.");
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
			}

			if (isInt32) {
				/* 猶予期間 + 現在時刻を秒数に直した時、
				 * time_t の範囲を超えないこと */
				comptime = (apr_time_t) atoi(passpolicy_iscreen->sfirstlogin_limitday);
				comptime = comptime * 24 * 3600 + now;
				if (comptime > INT_MAX) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"The \"firstlogin-limitday\" is too large, "
							"so, the server could not handle this value."
							"(val = %s)", passpolicy_iscreen->sfirstlogin_limitday);
					return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
				}
			}
		}
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * changepassword プロパティの解析結果をチェックする。
 * [ 有効性 ]
 *   パスワードポリシー機能が「サポート/未サポート」「有効/無効」に関わらず
 *   定義済みとなります.
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_changepassword_property(request_rec *r,
					int chk_mode,
					divy_changepassword_iscreen *changepassword_iscreen)
{
	apr_pool_t *p = r->pool;

	/* changepassword_iscreen がNULLではないこと */
	if (changepassword_iscreen == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The \"changepassword\" element is missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}
	if (changepassword_iscreen->u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"u_spec is NULL.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* ユーザ(in 管理者フォルダ) に対するリクエストであること */
	if (changepassword_iscreen->u_spec->infotype != DIVY_INFOTYPE_m_user_e) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The request-uri must be management-user uri. (uri = %s)",
				changepassword_iscreen->u_spec->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * パスワードポリシーに基づくパスワードのチェック
	 * (note)
	 *   ポリシーチェックは無条件に実施していい
	 */
	return _validate_password_input(r, changepassword_iscreen->src_usr_pr,
							changepassword_iscreen->password, 1/* plain */);
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * changeowner プロパティの解析結果をチェックする。
 * [ 有効性 ]
 *   グループリーダ機能能が「有効/無効」に関わらず定義済みとなります.
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_changeowner_property(request_rec *r,
					int chk_mode,
					divy_changeowner_iscreen *changeowner_iscreen)
{
	dav_error *err;
	apr_pool_t *p = r->pool;
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	const divy_rdbo_usr *usr_pr = changeowner_iscreen->src_usr_pr;
	const char *own_userid      = divy_get_userid(r);

	/*
	 * オーナの変更は、グループリーダと管理者のみが実施可能
	 */
	if (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN &&
		!divy_rdbo_is_groupleader(own_extstatus)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The change of ownerinfo is only executed by admin or groupleader."
				"(userid = %s)", own_userid);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/*
	 * グループリーダの場合、
	 */
	if (divy_rdbo_is_groupleader(own_extstatus)) {
	 	/*
		 * Otherユーザを管理下に置く権限があるのか?
		 */
		if (!divy_rdbo_has_control_otheruser(own_extstatus)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not change owner because you did not have control-other-user privilege."
					"(userid = %s)", own_userid);
			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCONTROL_OTHERUSER);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}

		/*
		 * 自分よりも上位または同等のユーザは引き込めない
		 */
		if (usr_pr != NULL &&
			(usr_pr->adminmode == DIVY_ADMINMODE_ADMIN || 
			 divy_rdbo_is_groupleader(usr_pr->extstatus))) {

			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not change owner of admin or groupleader."
					"(userid = %s)", own_userid);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	/*
	 * 最大作成数の制限を越えていないか?
	 */
	err = _validate_control_user_limit(r);
	if (err) return err;

	return NULL;
}

/**
 * changeleader プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_changeleader_property(request_rec *r,
					int chk_mode,
					divy_changeleader_iscreen *changeleader_iscreen)
{
	apr_pool_t *p = r->pool;
	const divy_rdbo_grp *grp_pr = changeleader_iscreen->src_grp_pr;
	const char *own_userid      = divy_get_userid(r);
	divy_rdbo_usr *usr_pr = NULL;
	divy_rdbo_rusr *rusr  = NULL;
	char *rusr_uri;

	/*
	 * 管理者のみが操作可能
	 */
	if (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"The change of groupleader is only executed by admin."
				"(userid = %s)", own_userid);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/*
	 * 任命/解任はトップグループに対してのみ可能
	 */
	if (divy_count_dirs(grp_pr->relativeuri) >= 2) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Could not set groupleader for sub-group.(userid = %s)", own_userid);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * 既にリーダが任命済みでは? -> 手動で解除しないとNG
	 * (note)
	 *   自分自身がリーダの場合は、やっても意味が無いが一応許しておく.
	 */
	if (changeleader_iscreen->action == DIVY_CHLEADER_ACTION_APPOINT) {
		if (IS_FILLED(grp_pr->ownerid) && strcmp(grp_pr->ownerid, changeleader_iscreen->ownerid) != 0) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not set groupleader because another user is already groupleader."
					"(userid = %s)", changeleader_iscreen->ownerid);

			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_ALREADY_EXISTS_LEADER);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/*
	 * 解任を指示されたが、そのグループのリーダではなかった
	 */
	if (changeleader_iscreen->action ==  DIVY_CHLEADER_ACTION_DISMISS) {
		if (IS_EMPTY(grp_pr->ownerid) || strcmp(grp_pr->ownerid, changeleader_iscreen->ownerid) != 0) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"Could not dismiss groupleader because this user is not groupleader for this group."
					"(userid = %s)", changeleader_iscreen->ownerid);

			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/*
	 * 任命/解任されるユーザはグループリーダの種類を持っていなければなりません
	 */
	if (divy_rdbo_get_user_property(r, changeleader_iscreen->ownerid, &usr_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get userinformation. (userid = %s)", changeleader_iscreen->ownerid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	/* 指定されたユーザが取得出来なかった */
	if (usr_pr == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"The specified user was missing. (userid = %s)", changeleader_iscreen->ownerid);
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}

	if (!divy_rdbo_is_groupleader(usr_pr->extstatus)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Could not change this user to groupleader because this user is not grouopleader type."
				"(userid = %s)", own_userid);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/*
	 * 対象グループに所属していなければグループリーダに任命されない
	 */
	/* ユーザリレーションURIを算出 */
	rusr_uri = divy_build_rusr_uri(p, changeleader_iscreen->u_spec->uri, changeleader_iscreen->ownerid);

	(void) divy_rdbo_get_rusr(r, rusr_uri, &rusr);
	if (rusr == NULL) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"Could not change this user to groupleader because this user did not belong to this group."
				"(userid = %s, group = %s)", own_userid, grp_pr->relativeuri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	changeleader_iscreen->usr_pr = usr_pr;	/* 記録しておく */

	return NULL;
}

/**
 * confirmreading プロパティの解析結果をチェックする。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_confirmreading_property(request_rec *r,
					int chk_mode,
					divy_confirmreading_iscreen *iscreen)
{
	apr_pool_t *p = r->pool;
	const divy_rdbo_resource *rdb_r = iscreen->rdb_r;
	divy_rdbo_confirmreading *cr_pr = NULL;

	/* 存在しないリソースに対するアクセスは禁止 */
	if (rdb_r == NULL) {
		return dav_new_error(p, HTTP_NOT_FOUND, 0, 0, "");
	}

	/* リソースに対してしか付かない (set) */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && rdb_r->resourcetype != DIVY_TYPE_RESOURCE) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"You could not set confirmreading because this resource is not file."
				"(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* グループフォルダ以下のリソースにしか付かない (set) */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) &&
		rdb_r->u_spec && rdb_r->u_spec->infotype != DIVY_INFOTYPE_group_e_regular) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"You could not set confirmreading to NONE-group resource."
				"(uri = %s)", rdb_r->uri);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* 既に開封通知がついていたらNG (set) */
	if (chk_mode & DIVY_VALIDATE_UPDATE) {
		/* 開封通知プロパティを取得 */
		if (divy_rdbo_get_confirmreading(r, iscreen->uri, iscreen->userid, &cr_pr, NULL)) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get confirmreading property. (uri = %s, userid = %s)",
					iscreen->uri, iscreen->userid);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 自分の開封通知が見つかった */
		if (cr_pr != NULL) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"This resource has own's confirmreading property. (uri = %s, userid = %s)",
					iscreen->uri, iscreen->userid);

			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_ALREADY_EXISTENCE);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	/* 開封通知が付いていないと解除できない,
	 * 他人の開封通知は解除できない (remove) */
	if (chk_mode & DIVY_VALIDATE_DELETE) {
		int isOwn = 0;

		/* 開封通知プロパティを取得 */
		if (divy_rdbo_get_confirmreading(r, iscreen->uri, NULL/*ユーザID無視*/, &cr_pr, NULL)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get confirmreading property. (uri = %s)", iscreen->uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* 開封通知が付いていなかった場合 */
		if (cr_pr == NULL) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"The confirmreading was missing. (uri = %s, userid = %s)",
					iscreen->uri, iscreen->userid);

			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_NO_EXISTENCE);
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}

		for (; cr_pr; cr_pr = cr_pr->next) {
			if (IS_FILLED(cr_pr->userid) && strcmp(cr_pr->userid, iscreen->userid) == 0) {
				isOwn = 1;
				break;
			}
		}

		/* 他人の開封通知は削除できない */
		if (!isOwn) {
			ERRLOG2(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"You could not remove other user's confirmreading. (uri = %s, userid = %s)",
					iscreen->uri, iscreen->userid);

			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_NO_REMOVE_OTHER);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	return NULL;
}


/**
 * 指定されたsrc が表すエンティティソースをdst のエンティティ
 * ディスティネーションとしてコピー可能かどうかを検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_copy_entity(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec)
{
	apr_pool_t *p = r->pool;
	int ret;
#ifdef DIVY_SUPPORT_PLUGIN
	dav_error *err;
#endif	/* DIVY_SUPPORT_PLUGIN */

	/*
	 * ソースとディスティネーションの種類の組み合わせを検証
	 */
	if ((src_u_spec->uritype == DIVY_URITYPE_REGULAR &&
	     dst_u_spec->uritype == DIVY_URITYPE_REGULAR) ||
	    (src_u_spec->infotype == DIVY_INFOTYPE_group_e &&
	     dst_u_spec->uritype  == DIVY_URITYPE_REGULAR)) {
		/* 通常リソース <-> 通常リリース or
		 * グループコレクション -> 通常フォルダ へはOK */

		/* 何もしない */
	}
	/* 通常ファイルではなくタイプが異なる時 */
	else if (src_u_spec->infotype != dst_u_spec->infotype) {

		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The destination uri must be same type of src uri."
			"(src = %s, dst = %s, method = %s)",
			src_u_spec->uri, dst_u_spec->uri, divy_get_method_name(r));
		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/*
	 * エンティティであるかどうかの検証
	 * (リレーションでなければ即ちエンティティ)
	 */
	if (IS_RELATION_RESOURCE(src_u_spec->infotype)) {

		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The copy of entity resource need to use entity uri."
			"(src = %s, dst = %s, method = %s)",
			src_u_spec->uri, dst_u_spec->uri, divy_get_method_name(r));
		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/*
	 * dst のfinal path segment に禁則文字が含まれていないかどうか
	 * (note)
	 * 	src やdst の残りのpath segment は既にDBに格納されており
	 * 	それは即ち、チェック済みであるという仮定の元、チェックを
	 * 	省略しています。
	 */
	if (dst_u_spec->uritype == DIVY_URITYPE_REGULAR) {
		ret = divy_validate_pathsegment(p, DIVY_IGNORE_COLON | DIVY_IGNORE_ASTERISK,
						dst_u_spec->final_path_segment);
	}
	else {
		ret = divy_validate_pathsegment(p, 0,
						dst_u_spec->final_path_segment);
	}
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The final path segment of distination uri "
			"has invalid char.(method = %s)", divy_get_method_name(r));
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* リレーションプレフィックスが付いていないかどうか調べる */
	if (divy_validate_having_entity_prefix(p, dst_u_spec->final_path_segment)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The final path segment of distination uri "
			"has relation prefix string.(method = %s)", divy_get_method_name(r));
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

#ifdef DIVY_SUPPORT_PLUGIN
	/* セキュリティで保護されたグループへのアクセスが許可されているかどうか */
	err = _validate_secured_folder(r, src_u_spec, dst_u_spec);
	if (err != NULL) {
		return err;
	}
#endif	/* DIVY_SUPPORT_PLUGIN */

	return NULL;
}

/**
 * 指定されたsrc が表すエンティティソースをdst のエンティティ
 * ディスティネーションとして移動可能かどうかを検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_move_entity(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec)
{
	apr_pool_t *p = r->pool;
	int ret;
#ifdef DIVY_SUPPORT_PLUGIN
	dav_error *err;
#endif	/* DIVY_SUPPORT_PLUGIN */

	/*
	 * ソースとディスティネーションの種類の組み合わせを検証
	 * (note)
	 * 	共有コレクションのMOVE は、クライアント側のメール監視処理が
	 * 	まともになるまで封印しておきます。
	 */
	/* 更新クライアントモジュール -> 通常リソース,
	 * 通常リソース, ユーザごみ箱, グループごみ箱 <->
	 * 通常リソース, ユーザごみ箱, グループごみ箱 */
	if ((src_u_spec->infotype == DIVY_INFOTYPE_m_update_e &&
	     dst_u_spec->uritype  == DIVY_URITYPE_REGULAR) ||
	    ((src_u_spec->uritype  == DIVY_URITYPE_REGULAR         ||
	      src_u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
	      src_u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
	      src_u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	      src_u_spec->infotype == DIVY_INFOTYPE_group_trash_ex) &&
	     (dst_u_spec->uritype  == DIVY_URITYPE_REGULAR         ||
	      dst_u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
	      dst_u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
	      dst_u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	      dst_u_spec->infotype == DIVY_INFOTYPE_group_trash_ex))) {
		/* 何もしない */
	}
	/* 上記以外でタイプが異なる時 */
	else if (src_u_spec->infotype != dst_u_spec->infotype) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Failed to move resource, becase the kind of src "
			"resource is different from kind of dst resource."
			"(src = %s, dst = %s)",
			src_u_spec->uri, dst_u_spec->uri);
		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/*
	 * エンティティであるかどうかの検証
	 * (リレーションでなければ即ちエンティティ)
	 */
	if (IS_RELATION_RESOURCE(src_u_spec->infotype)) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The MOVE of entity resource need to use entity uri."
			"(src = %s, dst = %s)",
			src_u_spec->uri, dst_u_spec->uri);
		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/*
	 * dst のfinal path segment に禁則文字が含まれていないかどうか
	 * (note)
	 * 	src やdst の残りのpath segment は既にDBに格納されており
	 * 	それは即ち、チェック済みであるという仮定の元、チェックを
	 * 	省略しています。
	 */
	if (dst_u_spec->uritype == DIVY_URITYPE_REGULAR) {
		ret = divy_validate_pathsegment(p, DIVY_IGNORE_COLON | DIVY_IGNORE_ASTERISK,
						dst_u_spec->final_path_segment);
	}
	else {
		ret = divy_validate_pathsegment(p, 0,
						dst_u_spec->final_path_segment);
	}
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The final path segment of distination uri "
			"has invalid char.(method = %s)", divy_get_method_name(r));
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	/* リレーションプレフィックスが付いていないかどうか調べる */
	if (divy_validate_having_entity_prefix(p, dst_u_spec->final_path_segment)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"The final path segment of distination uri "
			"has relation prefix string.(method = %s)", divy_get_method_name(r));
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

#ifdef DIVY_SUPPORT_PLUGIN
	/* セキュリティで保護されたグループへのアクセスが許可されているかどうか */
	err = _validate_secured_folder(r, src_u_spec, dst_u_spec);
	if (err != NULL) {
		return err;
	}
#endif	/* DIVY_SUPPORT_PLUGIN */

	return NULL;
}

/**
 * 指定されたsrc が表すリレーションソースをdst のリレーション
 * ディスティネーションとしてコピー可能かどうかを検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_copy_relation(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec)
{
	apr_pool_t *p = r->pool;
	char *src_uri_part, *dst_uri_part;

	/*
	 * src とdstが同じタイプかどうかの検証
	 */
	if (src_u_spec->infotype != dst_u_spec->infotype) {

		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The destination uri must be same type "
			"of src uri for \"%s\".(src = %s, dst = %s)",
			divy_get_method_name(r), src_u_spec->uri, dst_u_spec->uri);
		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/*
	 * リレーションであるかどうかの検証
	 */
	if (!IS_RELATION_RESOURCE(src_u_spec->infotype) ||
	    !IS_RELATION_RESOURCE(dst_u_spec->infotype)) {

		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"%s\" of relation resource need to "
			"use relation uri.(src = %s, dst = %s)",
			divy_get_method_name(r), src_u_spec->uri, dst_u_spec->uri);
		return dav_new_error(p, HTTP_CONFLICT, 0, 0, "");
	}

	/*
	 * リレーションのリネームにあたるオペレーションは禁止
	 * (note)
	 * ・final path segment が異なるオペレーションは禁止
	 * ・同親に対するオペレーションは禁止
	 */
	if (strcmp(src_u_spec->final_path_segment,
		   dst_u_spec->final_path_segment) != 0) {
		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The final path segment of dst resource "
			"is different from the final path segment of src resource. "
			"The rename operation of relation resource is forbidden."
			"(src = %s, dst = %s, method = %s)",
			src_u_spec->uri, dst_u_spec->uri, divy_get_method_name(r));
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	src_uri_part = apr_pstrndup(p, src_u_spec->other_part,
					strlen(src_u_spec->other_part) - 
					strlen(src_u_spec->final_path_segment));
	dst_uri_part = apr_pstrndup(p, dst_u_spec->other_part,
					strlen(dst_u_spec->other_part) -
					strlen(dst_u_spec->final_path_segment));

	if (strcmp(src_uri_part, dst_uri_part) == 0) {

		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The parent of dst resource "
			"is equal with the parent of src resource."
			"The rename operation of relation resource is forbidden."
			"(src = %s, dst = %s, method = %s)",
			src_u_spec->uri, dst_u_spec->uri, divy_get_method_name(r));
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	return NULL;
}

/**
 * 指定されたsrc が表すリレーションソースをdst のリレーション
 * ディスティネーションとして移動可能かどうかを検証する。
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_move_relation(request_rec *r,
					const divy_uri_spec *src_u_spec,
					const divy_uri_spec *dst_u_spec)
{
	/* copy の場合と同じチェックでよろしい */
	return divy_validate_copy_relation(r, src_u_spec, dst_u_spec);
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * userid のユーザのパスワード有効期限、猶予期間のチェック
 *
 */
DIVY_DECLARE(dav_error *) divy_validate_password_term(request_rec *r,
					const char *userid,
					divy_uri_spec *u_spec, divy_reqbody_type type)
{
	apr_pool_t *p = r->pool;
	apr_pool_t *cache_p = (r->main) ? r->main->pool : r->pool;
	divy_rdbo_passwordpolicy *passpolicy_pr = NULL;
	divy_rdbo_usr *usr_pr;
	const divy_rdbo_usr *cached_usr_pr;
	divy_rdbo_usr *tmp_usr_pr = NULL;
	time_t now, lastchange = 0L, pw_cycle, probation, compdt;
	divy_rdbo_passpolicystatus *passpolicy_status;
	int isSendmail = 0, ret = 1, mltype = DIVY_ML_NOTIFY_UNKNOWN, force = 0;
	dav_error *err = NULL;
	divy_ml_passwordexpired_data mldata = { 0 };
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	int issaml = 0;

	/* パスワードポリシー機能はサポートされているか? */
	if (!divy_support_passpolicy(r)) {
		return NULL;	/* 未サポートは何もしない */
	}

	if (IS_EMPTY(userid) || u_spec == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"userid or u_spec is EMPTY.");
		/* SAGA */
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* SAMLでログインしている場合はパスワードポリシーは適用しない
	 * 本関数の引数request_recはサブリクエストの場合がある為に記録している
	 * フラグがr->poolでは取れないことを想定し、r->mainのチェック
	 * を行っています。定義部を参照してください。
	 */
	issaml = divy_pcache_get_flag(cache_p, DIVY_PCACHE_FLG_USE_SAML);
	if (issaml == 1) {
		return NULL;	/* チェックしない */
	}

	/* パスワードポリシー例外規則の判定 */
	if (dconf->passpolicy_exs != NULL) {
		divy_passpolicy_exclusion *exs = dconf->passpolicy_exs;
		int exs_ip = 0, exs_uid = 0;

		/* IPアドレスルールをチェック */
		if (exs->ipaddrs != NULL) {
			apr_ipsubnet_t *ipsub;
			int i, len = divy_array_getlength(exs->ipaddrs);

			for (i = 0; i < len; i++) {
				ipsub = divy_array_get(exs->ipaddrs, i);
				if (ipsub != NULL && apr_ipsubnet_test(ipsub, r->useragent_addr)) {
					exs_ip = 1;	/* 例外規則にマッチした */
					break;
				}
			}
		}

		/* ユーザIDルールをチェック */
		if (exs->uid_reg != NULL) {
			const char *userid = divy_get_userid(r);
			if (!ap_regexec(exs->uid_reg, userid, 0, NULL, 0)) {
				exs_uid = 1;	/* 例外規則に合致 */
			}
		}

		/* AND -> 2つとも合致, OR -> どれか１つが合致 => 期間ルールのチェックはスキップ */
		if ((exs->cond == DIVY_PASSPOLICY_EXCLUSION_AND && (exs_ip && exs_uid)) ||
			(exs->cond == DIVY_PASSPOLICY_EXCLUSION_OR  && (exs_ip || exs_uid))) {
			return NULL;
		}
	}

	/* パスワードポリシープロパティを取得する
	 * (キャッシュ関数の使用は禁止. 同時アクセスで問題を起こすので) */
	if (divy_rdbo_get_passwordpolicy_property(r, DIVY_DEFAULT_POLICY_ID, &passpolicy_pr, NULL)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get passwordpolicy property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* パスワードポリシーが無効 or 期間ポリシーが無ければ終了 */
	if (passpolicy_pr == NULL || passpolicy_pr->status == 0 ||
		(passpolicy_pr->change_pw_cycle == 0 /*期限なし*/ &&
		 passpolicy_pr->firstlogin_limitday == -1 /*期限なし*/)) {
		return NULL;	/* 何もしない */
	}

	/* 対象ユーザのプロパティを再取得(競合条件があるので再取得の必要あり) */
	if (divy_rdbo_get_user_property(r, userid, &tmp_usr_pr)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get user property.(userid = %s)", userid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	usr_pr = tmp_usr_pr;	/* 競合を起こすのでdivy_get_usrprop(r) を使用してはならない */
	if (usr_pr == NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get user property.(userid = %s)", userid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	cached_usr_pr = divy_get_usrprop(r);	/* キャッシュ値を取得 */
	if (cached_usr_pr->lastaccess == 0L && usr_pr->lastaccess != 0L) {
		/* アクセス途中で最終更新日付が変化したということは、すなわち、Authのステージで
		 * 最終更新日付が変更された可能性が非常に濃厚. この場合、古いキャッシュ値を使って
		 * 計算しなければ長期出張者対応が動作しなくなる */
		usr_pr->lastaccess = cached_usr_pr->lastaccess;	/* 0L にする */
	}

	/* ポリシーステータスが存在しないユーザは
	 * ポリシーの運用時におけるON/OFFにより付いていない可能性あり.
	 * デフォルト値を入れておく.*/
	passpolicy_status = usr_pr->passpolicy_status;
	if (passpolicy_status == NULL) {
		time_t updatedt;

		/* デフォルト状態で作成 */
		passpolicy_status = divy_create_user_passwordpolicy(p, userid);

		updatedt = dav_divy_iso8601totime_t(p, usr_pr->updatedt);

		/* 状態の推測1(少なくとも1 回login していたか?) */
		if (usr_pr->lastaccess > 0L && usr_pr->lastaccess >= updatedt) {
			/* 初回ログイン日時をユーザの最終アクセス日付にする */
			passpolicy_status->firstlogin = usr_pr->lastaccess;

			/* パスワードの最終更新日付をユーザデータの最終更新日にする */
			passpolicy_status->lastchange = updatedt;
			/* メール送信日時は不明 */
		}
		force = 1;	/* DB に書き込む */
	}

	/* 今の日付を取得 */
	now = dav_divy_get_now_epoch();

	/*
	 * メールを送るべきかどうか.
	 * [ 条件 ]
	 *   * まだ1通も送っていない or  前回の送信日時から24時間以上経っている
	 *   * 送信者のメールアドレスがある
	 *   * 受信者のメールアドレスがある
	 */
	if (passpolicy_status->sendexpiredmail == 0L ||
		 (now - passpolicy_status->sendexpiredmail > 3600 * 24)) {
		dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
		const char *toaddr       = divy_get_mailaddr(r);

		if (IS_FILLED(toaddr) && IS_FILLED(dconf->mailadminaddr)) {
			isSendmail = 1;	/* メール送信の時間である */
		}
	}

	/* 初回ログイン時対応(即時変更) が必要で、
	 * 初回ログイン時にのパスワード変更操作が済んでいない場合 */
	if (passpolicy_pr->firstlogin_limitday == 0 &&
		passpolicy_status->firstlogin == 0L) {

		int mnum      = divy_get_method_number(r);
		int m_search  = ap_method_number_of("SEARCH");
		divy_infotype infotype = u_spec->infotype;

		/* ヘッダにセット */
		divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_CHANGENOW);

		mltype = DIVY_ML_NOTIFY_FLOGIN_CHANGENOW;
		/*
		 * [ 許可されるメソッドとURIパターン ]
		 *   * OPTIONS   (*)
		 *   * SEARCH     ($root/userinfo のalllist, $root/.management/USER/ のContent)
		 *   * PROPFIND   ($root/.management/USER/$userid)
		 *   * PROPPATCH  ($root/.management/USER/$userid)
		 *   * GET / POST ($root, $root/.cgi-bin/$name)
		 *
		 * 上記以外はForbidden 扱いとする
		 */
		if (!(mnum == M_OPTIONS ||
			  (mnum == m_search &&
			  	((infotype == DIVY_INFOTYPE_userinfo &&
				  (type == DIVY_REQBODY_TYPE_SEARCH_ALLLIST ||
				   type == DIVY_REQBODY_TYPE_UNKNOWN)) ||
				 (infotype == DIVY_INFOTYPE_m_user &&
				  (type == DIVY_REQBODY_TYPE_SEARCH_USERCONTENT ||
				   type == DIVY_REQBODY_TYPE_UNKNOWN)))
			  ) ||
			  (mnum == M_PROPFIND  && infotype == DIVY_INFOTYPE_m_user_e) ||
			  (mnum == M_PROPPATCH && infotype == DIVY_INFOTYPE_m_user_e) ||
			  ((mnum == M_GET || mnum == M_POST) &&
			     (infotype == DIVY_INFOTYPE_root
#ifdef DAV_SUPPORT_POST_HANDLING
				  || infotype == DIVY_INFOTYPE_cgibin_e
#endif	/* DAV_SUPPORT_POST_HANDLING */
				  )))) {
			apr_table_setn(r->headers_out, "Location",
					apr_psprintf(r->pool, "%s/"
									DIVY_SYSCGIFOLDER_NAME
									"/mainmenu",
									dav_divy_get_root_uri(r)));
			err = dav_new_error(p, HTTP_MOVED_TEMPORARILY, 0, 0, NULL);
		}
	}
	/* 初回ログイン時対応(期限付き) が必要で
	 * 初回ログイン時にのパスワード変更操作が済んでいない場合 */
	else if (passpolicy_pr->firstlogin_limitday > 0 &&
			 passpolicy_status->firstlogin == 0L) {

		time_t firstlogin_limitday =
			DIVY_PASSPOLICY_PROBATION_SEC(passpolicy_pr->firstlogin_limitday);

		/*
		 * 猶予期間のチェック
		 * [ 期間範囲 ]
		 *     * パスワード最終更新日時 > 0
		 *       パスワード最終更新日時 < now <= パスワード最終更新日時 + 猶予期間
		 *       (前提)
		 *          ユーザ作成時には、アカウント作成日時 == パスワード最終更新日時
		 *
		 *     * パスワード最終更新日時 == 0 (データ不正)
		 *       ユーザ作成日時 < now <= ユーザ作成日時 + 猶予期間
		 * (note)
		 *   パスワード最終更新日時よりもnow の方が必ず大きいので前者の条件チェックは不要
		 *   パスワード最終更新の後で管理者がパスワードポリシーのON/OFFを行なった場合には、
		 *   パスワードポリシーをONにした日付(開始日)を使用します (外部仕様)
		 */
		if (passpolicy_status->lastchange != 0L) {
			compdt = passpolicy_status->lastchange;
		}
		else {
			compdt = dav_divy_iso8601totime_t(p, usr_pr->registdt);
		}

		/* パスワード最終更新日付(or 作成日)よりもポリシー開始日の方が
		 * 大きければ置き換える */
		if (passpolicy_pr->startdt > 0L && passpolicy_pr->startdt > compdt) {
			compdt = passpolicy_pr->startdt;
		}

		if (now <= compdt + firstlogin_limitday) {
			/* ヘッダにセット */
			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_FIRST_PROBATION);

			mltype = DIVY_ML_NOTIFY_FLOGIN_PROBATION;
			mldata.flprobation_last = compdt + firstlogin_limitday;
		}
		/*
		 * 完全期限切れのチェック
		 * [ 期間範囲 ]
		 *   アカウント作成日時 + 猶予期間 < now
		 */
		else {
			/* 完全期限切れになるまでに1度もアクセスしていなければ
			 * 長期出張者ということで救済する.
			 * アクセスがあった場合には大人しく完全期限切れになっていただく.
			 */
			if (passpolicy_status->specialstart == 0L && usr_pr->lastaccess == 0L) {
				/* (note) lastaccess があれば救済はしない */
				passpolicy_status->specialstart  = now;	/* 今日を開始日とする */

				/* ヘッダにセット */
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_FIRST_PROBATION);
				mltype = DIVY_ML_NOTIFY_FLOGIN_PROBATION;
				mldata.flprobation_last =
					passpolicy_status->specialstart + firstlogin_limitday;
				force = 1;	/* DB は更新しておく */
			}
			/* 基点日付を持っていて、その日付がcompdt よりも大きければ
			 * 足掻いてみる(小さければ結果は明白) */
			else if (passpolicy_status->specialstart != 0L &&
					 passpolicy_status->specialstart > compdt) {

				if (passpolicy_status->specialstart > now) {
					isSendmail = 0;	/* あり得ないのだが一応 */
				}
				/* 猶予期間中の場合 */
				else if (passpolicy_status->specialstart <= now &&
						 now <= passpolicy_status->specialstart + firstlogin_limitday) {
					/* ヘッダにセット */
					divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_FIRST_PROBATION);
					mltype = DIVY_ML_NOTIFY_FLOGIN_PROBATION;
					mldata.flprobation_last =
						passpolicy_status->specialstart + firstlogin_limitday;
				}
				/* 完全期限切れ */
				else {
					/* ヘッダにセット */
					divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_FIRST_EXPIRED);
					mltype = DIVY_ML_NOTIFY_FLOGIN_EXPIRED;
					mldata.flprobation_last =
						passpolicy_status->specialstart + firstlogin_limitday;

					if (isSendmail) {
						/* メール送信と同時にログに出す */
						ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
								"This user's first password was expired and locked. "
								"Please check it. (userid = %s)", userid);
					}
					err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, NULL);	/* アクセス禁止 */
				}
			}
			/* 基点日付も効き目がなかったら、完全期限切れの処理を実施 */
			else {
				/* ヘッダにセット */
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_FIRST_EXPIRED);
				mltype = DIVY_ML_NOTIFY_FLOGIN_EXPIRED;
				if (passpolicy_status->specialstart == 0L ||
					passpolicy_status->specialstart <= compdt) {
					mldata.flprobation_last = compdt + firstlogin_limitday;
				}
				else {
					mldata.flprobation_last =
						passpolicy_status->specialstart + firstlogin_limitday;
				}

				if (isSendmail) {
					/* メール送信と同時にログに出す */
					ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
							"This user's first password was expired and locked. "
							"Please check it. (userid = %s)", userid);
				}
				err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, NULL);	/* アクセス禁止 */
			}
		}
	}
	/*
	 * 期限切れ対応が必要の場合(初回ログオンではない)
	 */
	else if (passpolicy_pr->change_pw_cycle > 0) {
		lastchange = passpolicy_status->lastchange;
		pw_cycle   = DIVY_PASSPOLICY_PASSPERIOD_SEC(passpolicy_pr->change_pw_cycle);
		probation  = DIVY_PASSPOLICY_PROBATION_SEC(passpolicy_pr->probation);

		/* パスワード最終更新日付(or 作成日)よりもポリシー開始日の方が
		 * 大きければ置き換える */
		if (passpolicy_pr->startdt > 0L && passpolicy_pr->startdt > lastchange) {
			compdt = passpolicy_pr->startdt;
		}
		else {
			compdt = lastchange;	/* 最終更新日を使う */
		}

		/*
		 * 猶予期間のチェック
		 * [ 期間の範囲 ]
		 *   パスワード最終更新日付 or ポリシー開始日 + 有効期限 < now <=
		 *   パスワード最終更新日付 or ポリシー開始日 + 有効期限 + 猶予期間
		 * (note)
		 *   lastaccess と共にチェックしているのはデータの下位互換性に留意したため
		 */
		if (compdt + pw_cycle < now && now <= compdt + pw_cycle + probation) {
			/* ヘッダにセット */
			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_PROBATION);
			mltype = DIVY_ML_NOTIFY_PASS_PROBATION;
			mldata.period         = compdt + pw_cycle;
			mldata.probation_last = mldata.period + probation;
		}
		/*
		 * 完全期限切れのチェック
		 *
		 * [ 期間の範囲 ]
		 *   パスワード最終更新日付 or ポリシー開始日 + 有効期限 + 猶予期間 < now
		 * (note)
		 *   lastaccess と共にチェックしているのはデータの下位互換性に留意したため
		 */
		else if (compdt + pw_cycle + probation < now) {
			/*
			 * 期限切れになってから今日までの間、1度もアクセスが
			 * ない場合に限り、今日を基点とした期限切れチェックを行う.
			 *
			 *                  この間にアクセスが無ければOK
			 *                 <--------------------------->
			 * +---------------+-----------+---------------+
			 * 変更日         期限切れ     完全期限切れ    now
			 * アクセスがあった場合には大人しく完全期限切れになっていただく.
			 * なお、パスワード更新を行うと最終アクセス日付も必ず変更になるので、
			 * 1度もアクセスがない時には、両者は同値となっているだろう.
			 */
			if (passpolicy_status->specialstart == 0L &&
				compdt + pw_cycle >= usr_pr->lastaccess) {
				passpolicy_status->specialstart = now;	/* 今日を開始点とする */

				isSendmail = 0;	/* メールは送信しない */
				force = 1;	/* DB は更新しておく */
			}
			/* 基点日付を持っていて、その日付がcompdt よりも大きければ
			 * 足掻いてみる(小さければ結果は明白) */
			else if (passpolicy_status->specialstart != 0L &&
					 passpolicy_status->specialstart > compdt) {

				/*  猶予期間に到達していない場合 */
				if (passpolicy_status->specialstart > now) {
					isSendmail = 0;	/* あり得ないのだが一応 */
				}
				/* 猶予期間中の場合(pw_cycle を作用させてない点に注意) */
				else if (passpolicy_status->specialstart <= now &&
						 now <= passpolicy_status->specialstart + probation) {
					/* ヘッダにセット */
					divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_PROBATION);
					mltype = DIVY_ML_NOTIFY_PASS_PROBATION;
					mldata.period         = passpolicy_status->specialstart;
					mldata.probation_last = mldata.period + probation;
				}
				/* 完全期限切れ */
				else {
					/* ヘッダにセット */
					divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_EXPIRED);
					mltype = DIVY_ML_NOTIFY_PASS_EXPIRED;
					mldata.period         = passpolicy_status->specialstart;
					mldata.probation_last = mldata.period + probation;

					if (isSendmail) {
						/* メール送信と同時にログに出す */
						ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
								"This user's password was expired and locked. "
								"Please check it. (userid = %s)", userid);
					}
					/* 完全に期限切れはFORBIDDENでアクセス拒否 */
					err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, NULL);	/* アクセス禁止 */
				}
			}
			/* 基点日付を使って処理しても結局駄目だったら完全期限切れ対応を実施 */
			else {
				/* ヘッダにセット */
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_EXPIRED);
				mltype = DIVY_ML_NOTIFY_PASS_EXPIRED;
				if (passpolicy_status->specialstart == 0 ||
					passpolicy_status->specialstart <= compdt) {
					mldata.period = compdt + pw_cycle;
				}
				else {
					mldata.period = passpolicy_status->specialstart + pw_cycle;
				}
				mldata.probation_last = mldata.period + probation;

				if (isSendmail) {
					/* メール送信と同時にログに出す */
					ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
							"This user's password was expired and locked. "
							"Please check it. (userid = %s)", userid);
				}
				err = dav_new_error(p, HTTP_FORBIDDEN, 0, 0, NULL);	/* アクセス禁止 */
			}
		}
		else {
			/* パスワード正常ケース */
			isSendmail = 0;	/* メールは不要 */
		}
	}
	else {
		/* 何の対応も必要ない場合 */
		isSendmail = 0;	/* メールは不要 */
	}

	if (isSendmail) {
		/* (note)
		 * 本当はDBに書く前にメールを送りたいのだが、それだと
		 * メールが複数送られてしまう. メール送信速度が遅いため
		 * DB 同時読み込みにより更新前のフラグを見てしまうためだ.
		 * 仕方が無いのでDB更新を先にします
		 */
		passpolicy_status->sendexpiredmail = now;	/* 今送る */
	}

	ret = 1;
	if (isSendmail || force) {
		/* DB にデータを書き込む */
		ret = divy_rdbo_update_user_passwordpolicy(r, passpolicy_status, NULL);
	}

	/* メールを送信する */
	if (ret == 0 && isSendmail && mltype != DIVY_ML_NOTIFY_UNKNOWN) {
		(void) divy_ml_notify_passwordexpired(r, mltype, usr_pr, &mldata);
	}

	return err;
}
#endif	/* DIVY_SUPPORT_PLUGIN */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * 定義値DIVY_HEADER_RESULTSTATUS が示すヘッダ文字列を生成して
 * ヘッダにセットする.
 *
 */
DIVY_DECLARE(void) divy_set_resultstatus(request_rec *r, divy_resultstatus_type type)
{
	const char *header = NULL;
	const char *old_val;

	switch (type) {
		/* パスワード変更が直ぐに必要 */
		case DIVY_RESULTSTATUS_TYPE_CHANGENOW:
			header = "password-changenow; firstlogin=1";
			break;
		/* 初回ログイン時の猶予期間 */
		case DIVY_RESULTSTATUS_TYPE_FIRST_PROBATION:
			header = "password-probation; firstlogin=1";
			break;
		/* 初回ログイン時の完全期限切れ */
		case DIVY_RESULTSTATUS_TYPE_FIRST_EXPIRED:
			header = "password-expired; firstlogin=1";
			break;
		/* 初回ログイン後の猶予期間 */
		case DIVY_RESULTSTATUS_TYPE_PROBATION:
			header = "password-probation; firstlogin=0";
			break;
		/* 初回ログイン後の完全期限切れ */
		case DIVY_RESULTSTATUS_TYPE_EXPIRED:
			header = "password-expired; firstlogin=0";
			break;
		/* 対象グループが非アクティブ */
		case DIVY_RESULTSTATUS_TYPE_INACTIVE_GROUP:
			header = "group-inactive";
			break;
		/* 子ユーザが居るのでリーダを削除できない */
		case DIVY_RESULTSTATUS_TYPE_NODELETE_GROUPLEADER:
			header = "user-no-delete-leader";
			break;
		/* 子ユーザが居るのでリーダを降格できない */
		case DIVY_RESULTSTATUS_TYPE_NODEMOTE_GROUPLEADER:
			header = "user-no-demote-leader";
			break;
		/* グループリーダは子ユーザの種類を上位に変更できない */
		case DIVY_RESULTSTATUS_TYPE_NOCHANGE_ADMINTYPE:
			header = "user-no-change-administrative-type";
			break;
		/* グループリーダはOtherユーザのプロパティを変更不可 */
		case DIVY_RESULTSTATUS_TYPE_NOCHANGE_OTHERUSER:
			header = "user-no-change-other-user";
			break;
		/* グループリーダはOtherユーザを削除できない */
		case DIVY_RESULTSTATUS_TYPE_NODELETE_OTHERUSER:
			header = "user-not-delete-other-user";
			break;
		/* グループリーダはトップグループを作成できない */
		case DIVY_RESULTSTATUS_TYPE_NOCREATE_TOPGROUP:
			header = "group-no-create-topgroup";
			break;
		/* グループリーダはトップグループを編集できない */
		case DIVY_RESULTSTATUS_TYPE_NOCHANGE_TOPGROUP:
			header = "group-no-change-topgroup";
			break;
		/* グループリーダはトップグループを削除できない */
		case DIVY_RESULTSTATUS_TYPE_NODELETE_TOPGROUP:
			header = "group-no-delete-topgroup";
			break;
		/* グループリーダは他リーダのトップグループ以下にサブグループを作成できない */
		case DIVY_RESULTSTATUS_TYPE_NOCREATE_SUBGROUP:
			header = "group-no-create-subgroup";
			break;
		/* グループリーダは他人のサブグループを編集できない */
		case DIVY_RESULTSTATUS_TYPE_NOCHANGE_SUBGROUP:
			header = "group-no-change-subgroup";
			break;
		/* グループリーダが最大作成数制限を越えて登録しようとした */
		case DIVY_RESULTSTATUS_TYPE_EXCEED_MAXCREATION:
			header = "user-exceed-max-creation-user";
			break;
		/* 対象グループには既にグループリーダが任命済みだった */
		case DIVY_RESULTSTATUS_TYPE_ALREADY_EXISTS_LEADER:
			header = "group-leader-already-exists";
			break;
		/* グループリーダは権限不足により、Otherユーザを引き込めない */
		case DIVY_RESULTSTATUS_TYPE_NOCONTROL_OTHERUSER:
			header = "user-no-control-other-user";
			break;
		/* 任命済みグループリーダの所属替え/削除はNG */
		case DIVY_RESULTSTATUS_TYPE_NOCHANGE_APPOINTED_GROUPLEADER:
			header = "user-no-change-appointed-groupleader";
			break;
		/* 他人の開封通知は解除できません */
		case DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_NO_REMOVE_OTHER:
			header = "confirmreading-no-remove-other";
			break;
		/* 開封通知が存在しないので解除できない */
		case DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_NO_EXISTENCE:
			header = "confirmreading-no-existence";
			break;
		/* 既に自身の開封通知が付いていたので設定できない  */
		case DIVY_RESULTSTATUS_TYPE_CONFIRMREADING_ALREADY_EXISTENCE:
			header = "confirmreading-already-existence";
			break;
		/* アップロードポリシー違反 */
		case DIVY_RESULTSTATUS_TYPE_UPLOADPOLICY_RULE_VIOLATION:
			header = "uploadpolicy-rule-violation";
			break;
		/* 二段階認証のコードを要求 */
		case DIVY_RESULTSTATUS_TYPE_2FA_AUTHENTICAITON_REQUIRED:
			header = "two-factor-authentication-required";
			break;
		/* BOXログイン失敗 */
		case DIVY_RESULTSTATUS_TYPE_BOX_AUTHENTICATION_FAILED:
			header = "box-authentication-failed";
			break;
		/* BOXの閲覧許可されたユーザではない */
		case DIVY_RESULTSTATUS_TYPE_BOX_INVALID_USER:
			header = "box-invalid-user";
			break;
		/* BOXの相手先メールアドレスはサーバポリシーにより禁止されています */
		case DIVY_RESULTSTATUS_TYPE_BOX_DISABLED_WILDCARD_TOMAILADDRESS:
			header = "box-disable-wildcard-tomailaddress";
			break;
		/* メールを送信しました */
		case DIVY_RESULTSTATUS_TYPE_SENDMAIL:
			header = "sendmail";
			break;
		/* (未定義) */
		default:
			return;
	}

	/* ヘッダに追加
	 * (note) r->headers_out だとErrorDocument でレスポンスが置き換えられると
	 * ヘッダが消えてしまう. 必ずr->err_headers_out でなければならない */
	apr_table_set(r->err_headers_out, DIVY_HEADER_RESULTSTATUS, header);

	/* 環境変数にも追加(CGI対応, エラーページ対応) */
	old_val = apr_table_get(r->subprocess_env, DIVY_CGIENV_RESULTSTATUS);
	if (IS_EMPTY(old_val)) {
		apr_table_set(r->subprocess_env, DIVY_CGIENV_RESULTSTATUS, header);
	}
	else {
		/* ステートが多重になってしまうときには、カンマで区切って渡す */
		char *new_val = apr_psprintf(r->pool, "%s,%s", old_val, header);
		apr_table_set(r->subprocess_env, DIVY_CGIENV_RESULTSTATUS, new_val);
	}

	/* ブラウザにエラーを表示するとき、上記と一緒にLocation名も必要になるので
	 * 一緒に入れておく */
	apr_table_set(r->subprocess_env, DIVY_CGIENV_LOCATION,
				ap_escape_uri(r->pool, dav_divy_get_root_uri(r)));
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * 定義値DIVY_HEADER_RESULTSTATUS が示すヘッダ文字列をリセットする.
 *
 */
DIVY_DECLARE(void) divy_reset_resultstatus(request_rec *r)
{
	/* ヘッダから除去 */
	apr_table_unset(r->err_headers_out, DIVY_HEADER_RESULTSTATUS);

	/* 環境変数からも除去 */
	apr_table_unset(r->subprocess_env, DIVY_CGIENV_RESULTSTATUS);

	/* Location 名は除去しなくていい */

	if (r->main != NULL) {
		apr_table_unset(r->main->err_headers_out, DIVY_HEADER_RESULTSTATUS);
		apr_table_unset(r->main->subprocess_env, DIVY_CGIENV_RESULTSTATUS);
	}
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * 強制削除がクライアントから指示されているかどうか.
 * (note)
 *     強制削除の指示は、HTTPリクエストヘッダ経由で行われます.
 *
 */
DIVY_DECLARE(int) divy_validate_is_forcedelete(request_rec *r)
{
	const char *sflag;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	/* HTTPリクエストヘッダを取得 */
	sflag = apr_table_get(r->headers_in, DIVY_HEADER_REQ_FORCEDELETE);

	/* 強制削除機能が有効で、適切なヘッダがあって、強制削除オプションが指定されていた */
	if (dconf->forcedelete == DIVY_FORCEDELETE_ON &&
		IS_FILLED(sflag) && strcmp(sflag, DIVY_HEADER_VAL_TRUE) == 0) {
		return 1;	/* 強制削除を指示された */
	}

	return 0;	/* 強制削除は指示されていない */
}

/**
 * リソースを調査します。アップロードポリシーで利用されます
 *
 * @param r         request_rec *
 * @param resource  dav_resource *
 *
 * @return int 0: 問題は見つからなかった
 *             1: 問題が見つかった
 *             2: 問題があったが静観せよ
 */
DIVY_DECLARE(int) divy_resource_inspection(request_rec *r, const dav_resource *resource)
{
	divy_rdbo_resource *rdb_r	= resource->info->rdb_r;
	divy_rdbo_grp *grp_pr		= NULL;
	char *rules_fnames			= NULL;
	char *displayname			= NULL;
	divy_rdbo_upload_policy *policy = NULL;
	divy_upload_policy_matchtype  matchtype = TF_UPLOAD_POLICY_MATCHTYPE_ALL;
	char *prefix				= NULL;
	char *suffix				= NULL;
	char *token, *token_ctx;
	char *pattern				= NULL;
	ap_regex_t preg;
	int prefix_pass = 0;
	int suffix_pass = 0;
	int char_pass = 0;
	int len_pass = 0;
	char *rule_length_str		= NULL;
	int fname_length = 0;
	char *rule_suffix_str		= NULL;
	int result = 0;
	char *parenturi, *pfolder;
	char *hash;
	const char *ua              = NULL;

	if (divy_rdbo_get_group_property_by_resuri(r, rdb_r->uri, &grp_pr)) {
		/* 致命的なエラー */
		ERRLOG1(r->pool, APLOG_CRIT, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get group information.(uri = %s)", rdb_r->uri);
		return 1;
	}

	/* ルールがないときはそのまま正常 */
	if (!grp_pr) return 0;
	if (!(grp_pr->policy)) return 0;

	policy = grp_pr->policy;
	/* 非アクティブならチェックしない*/
	if (grp_pr->policy->active == 0) return 0;	
	/* 日付がないならチェックしない */
	if (grp_pr->policy->updatedate == 0) return 0;

	/* マッチタイプを取得する */
	matchtype = policy->rules_matchtype;

	/*
	 * ファイル名チェック
	 */

	/* 調査するファイル名を.で分割する */
	displayname = apr_pstrdup(r->pool, rdb_r->displayname);
	prefix = apr_strtok(displayname, ".", &token_ctx);
	suffix = apr_strtok(NULL, ".", &token_ctx); /* ないこと(NULL)もある */

	/* ルールはない、もしくは複数あります。ない場合は次へスキップ */
	rules_fnames = apr_pstrdup(r->pool, policy->rules_fnames);
	if (IS_EMPTY(rules_fnames)) {
		prefix_pass = 1;
		goto checkout_filename;
	}
	while ((token = apr_strtok(rules_fnames, "/", &token_ctx)) != NULL) {
		if (token == NULL) break;
		rules_fnames = NULL;

		/* アスタリスクを正規表現へ変更する */
		token = (char *)dav_divy_replace_str(r->pool, token, "*", ".*", NULL);

		/* 置換処理：変数 ユーザID: ${userid} */
		if (divy_strstr(token, "${userid}") != NULL) {
			token = (char *)dav_divy_replace_str(r->pool, token, "${userid}", divy_get_userid(r), NULL);
		}
		/* 置換処理：変数 グループ名 */
		if (divy_strstr(token, "${groupname}") != NULL) {
			token = (char*)dav_divy_replace_str(r->pool, token, "${groupname}", grp_pr->name, NULL);
		}
		/* 置換処理：変数 親フォルダ名 */
		if (divy_strstr(token, "${parentfolder}") != NULL) {
			parenturi = divy_get_parenturi(r->pool, rdb_r->uri);
			pfolder = NULL;
			if (parenturi != NULL) {
				pfolder
					= dav_divy_extract_finalpath_segment(r->pool, parenturi);
			}
			if (pfolder != NULL) {
				token = (char*)dav_divy_replace_str(r->pool,
									token, "${parentfolder}", pfolder, NULL);
			}
		}

		if ((ap_regcomp(&preg, token, 0)) != 0) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to compile group policy filename rules "
					"matching pattern.");
			return 0;
		}

		/* ******  評価 ****** */
		if (ap_regexec(&preg, prefix, 0, NULL, 0) == 0) {
			/* マッチしたもう調べる必要はない */
			prefix_pass = 1;
			ap_regfree(&preg);

			/* ANYなら許可 */
			if (matchtype == TF_UPLOAD_POLICY_MATCHTYPE_ANY) return 0;

			goto checkout_filename;
		}

		ap_regfree(&preg);

	}

checkout_filename:

	/*
	 * 文字種別チェック
	 */
	if (policy->rules_chartype != TF_UPLOAD_POLICY_CHARTYPE_NONE) {
		switch(policy->rules_chartype) {
			case TF_UPLOAD_POLICY_CHARTYPE_NUM:
				pattern = "^[0-9]+$";
				break;

			case TF_UPLOAD_POLICY_CHARTYPE_ALP:
				pattern = "^[a-zA-Z]+$";
				break;

			default:
				pattern = "^[0-9a-zA-Z\\-]+$";
		}

		/* ******  評価 ****** */
		if ((ap_regcomp(&preg, pattern, 0) == 0)) {
			if (!ap_regexec(&preg, prefix, 0, NULL, 0)) {
				/* マッチした */
				char_pass = 1;
			}
		}
		ap_regfree(&preg);
		/* ANYなら許可 */
		if (char_pass == 1 && matchtype == TF_UPLOAD_POLICY_MATCHTYPE_ANY) return 0;
	}
	else {
		/* 設定なし */
		char_pass = 1;
	}


	/*
	 * 長さチェック
	 */
	if (IS_EMPTY(policy->rules_length)) {
		len_pass = 1;
	}
	else {
		rule_length_str = apr_pstrdup(r->pool, policy->rules_length);
		fname_length = strlen(prefix);

		/* アスタリスクがあるとこのルールはパス */
		if (divy_strchr(rule_length_str, '*') != NULL) {
			len_pass = 1;
			/* ANYなら許可 */
			if (matchtype == TF_UPLOAD_POLICY_MATCHTYPE_ANY) {
				return 0;
			}
			/* 調べる必要すらないのですぐ次へ進む*/
			goto checkout_length;
		}

		while ((token = apr_strtok(rule_length_str, ",", &token_ctx)) != NULL) {
			if (token == NULL) break;
			rule_length_str = NULL;

			if (dav_divy_isdigit_str(token)) {
				/* ******  評価 ****** */
				if (fname_length == atoi(token)) {
					/* マッチした */
					len_pass = 1;
					/* ANYなら許可 */
					if (matchtype == TF_UPLOAD_POLICY_MATCHTYPE_ANY) {
						return 0;
					}

					goto checkout_length;
				}
			}
			else {
				/* 数字ではなかった XX-XXの範囲指定の可能性 */

				/* - がない場合はスキップ */
				if (divy_strchr(token, '-') == NULL)  continue;

				char *strmin, *strmax;
				int min = 0, max = 256 * 3; /* デフォルト最小と最大 */
				char *basestr = apr_pstrdup(r->pool, token);	/* コピー */
				char *token_ctx2;

				/* 先頭が-であるなら0を連結します */
				if (*basestr == '-') {
					basestr = apr_psprintf(r->pool, "0%s", basestr);
				}
				strmin = apr_strtok(basestr, "-", &token_ctx2); 
				strmax = apr_strtok(NULL, "-", &token_ctx2); 

				if (dav_divy_isdigit_str(strmin)
									&& dav_divy_isdigit_str(strmax)) {
					min = atoi(strmin);
					max = atoi(strmax);
				}
				else {
					/* 展開したら数字ではなかった。エラー出力して次へ */
					ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
							"upload policy [rule: length] is digit."
							"min = %s / max = %s" , strmin, strmax);
					continue;
				}

				/* 最大と最小が逆ならその評価はしない */
				if (min > max) continue;

				/* ******  評価 ****** */
				if (min <= fname_length && fname_length <= max) {
					/* マッチした */
					len_pass = 1;
					/* ANYなら許可 */
					if (matchtype == TF_UPLOAD_POLICY_MATCHTYPE_ANY) {
						return 0;
					}

					goto checkout_length;
				}

			}
		}
	}

checkout_length:
	/*
	 * 拡張子チェック
	 */
	if (IS_EMPTY(policy->rules_suffix)) {
		suffix_pass = 1;
	}
	else {
		rule_suffix_str = apr_pstrdup(r->pool, policy->rules_suffix);
		ap_str_tolower(rule_suffix_str); /* ルールを小文字にする */
	}

	/* ルールがあるのに拡張子がないのはチェックしない */
	if (IS_FILLED(rule_suffix_str) && IS_EMPTY(suffix)) {
		suffix_pass = 0;	/* 意識的に0 */
		goto checkout_suffix;
	}

	while((token = apr_strtok(rule_suffix_str, ",", &token_ctx)) != NULL) {
		if (token == NULL) break;
		rule_suffix_str = NULL;

		ap_str_tolower(suffix);	/* 小文字 */

		/* アスタリスクを正規表現へ変更する */
		token = (char *)dav_divy_replace_str(r->pool, token, "*", ".*", NULL);

		/* ******  評価 ****** */
		if ((ap_regcomp(&preg, token,  0)) == 0) {
			if (ap_regexec(&preg, suffix, 0, NULL, 0) == AP_REG_NOMATCH) {
				ap_regfree(&preg);
				continue;
			}
			else {
				/* マッチした */
				suffix_pass = 1;
				ap_regfree(&preg);

				/* ANYなら許可 */
				if (matchtype == TF_UPLOAD_POLICY_MATCHTYPE_ANY) return 0;

				goto checkout_suffix;
			}

			ap_regfree(&preg);
		}

	}

checkout_suffix:

	/* 全体評価 */
	result = !(prefix_pass && char_pass && len_pass && suffix_pass);

	if (result) {

		/* 問題があった */

		divy_set_resultstatus(r,
				DIVY_RESULTSTATUS_TYPE_UPLOADPOLICY_RULE_VIOLATION);

#define RET2STRING(ret) ((ret == 1)?"PASS":"NG")
		char* errmsg = apr_psprintf(r->pool,
				"filename = %s / char = %s / length = %s / suffix = %s",
				RET2STRING(prefix_pass), RET2STRING(char_pass),
				RET2STRING(len_pass), RET2STRING(suffix_pass));

		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_PROC,
						"upload policy violation "
						"filename = %s [ result :  %s ]", rdb_r->displayname,
						errmsg);

#undef RET2STRING

		/*
		 * Microsoft Office Upload Centerが行うアップロードはユーザの意図しない
		 * 時にファイルをアップロードする。このエラーを通知せず静観エラー
		 * として処理をします。
		 */
		ua = dav_divy_get_user_agent(r);
		if (!ua || strncmp(ua, "Microsoft Office Upload Center", 30) == 0) {
			return 2;	/* 静観エラー */
		}

		/* アップロードにかかる時間をタイムアウトに設定します */
		hash = ap_md5(r->pool,
				(const unsigned char*)apr_psprintf(r->pool, "%s%s%s",
										errmsg, r->user, rdb_r->displayname));
		if (!divy_has_action_in_time(r, hash, rdb_r->displayname,
					apr_time_sec(apr_time_now() - r->request_time) + 5)) {
			/* メールを送る。 なにかあっても対処できない */
			(void)divy_ml_group_policy_inspection_result(r,
									rdb_r->displayname, grp_pr, prefix_pass,
									char_pass, len_pass, suffix_pass);
		}

	}

	return result;

}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * userid のユーザがuri のリソースにアクセス可能かどうか
 * ユーザアクセス拒否リストを見て判定する.
 *
 * @return int 
 * 	OK      : アクセスは許可
 * 	上記以外: アクセスは拒否
 */
static int _validate_usr_accessdeny(request_rec *r, 
					const char *uri,
					divy_uri_spec *uri_spec)
{
	apr_pool_t *p = r->pool;
	int i, infotype = uri_spec->infotype;
	int is_deny = 0;
	const int *accessdeny = NULL;
	int mnum = divy_get_method_number(r);
	const char *userid    = divy_get_userid(r);

	/*
	 * アクセス拒否リストを取得
	 */
	accessdeny = divy_get_accessdeny(r);

	/* 制限プロパティが存在しなければ OK */
	if (accessdeny == NULL) return OK;

	/* 制限プロパティが存在した場合、uri がそれにあたるかどうか調べる */
	is_deny = 0;	/* デフォルトは拒否しない */
	for (i = 0; i < DIVY_FOLDER_ID_END; i++) {

		/* 許可されていたフォルダは無視 */
		if (accessdeny[i] == 0) continue;

		/* 拒否されていたフォルダを調べる */
		switch (i) {
			/* ユーザフォルダのトップ */
			case DIVY_FOLDER_ID_user:
				/* ユーザフォルダ、ユーザごみ箱を拒否 */
				if (infotype == DIVY_INFOTYPE_user_e         ||
				    infotype == DIVY_INFOTYPE_user_trash     ||
				    infotype == DIVY_INFOTYPE_user_trash_e0  ||
				    infotype == DIVY_INFOTYPE_user_trash_ex  ||
				    infotype == DIVY_INFOTYPE_user_e_regular) {
					is_deny = 1;	/* 拒否する */
				}
				break;

			/* グループコレクションのトップ */
			case DIVY_FOLDER_ID_group:
				/* グループコレクションに対するアクセスを拒否 */
				if (infotype == DIVY_INFOTYPE_group          ||
				    infotype == DIVY_INFOTYPE_group_e        ||
				    infotype == DIVY_INFOTYPE_group_trash    ||
				    infotype == DIVY_INFOTYPE_group_trash_e0 ||
				    infotype == DIVY_INFOTYPE_group_trash_ex ||
				    infotype == DIVY_INFOTYPE_group_e_regular) {
					is_deny = 1;
				}
				break;

			/* データベースのトップ */
			case DIVY_FOLDER_ID_dblink:
				/* データベース検索と検索結果フォルダの閲覧を拒否
				 * (note)
				 * 	ここの拒否だけでは、SQL検索画面の表示
				 * 	は可能になってしまいます。
				 * 	uri のチェックだけではこれが限界です。*/
				if (infotype == DIVY_INFOTYPE_dblink     ||
				    infotype == DIVY_INFOTYPE_dblink_e   ||
				    infotype == DIVY_INFOTYPE_dbfolder   ||
				    infotype == DIVY_INFOTYPE_dbfolder_e ||
				    infotype == DIVY_INFOTYPE_dbfolder_e_regular ||
				    infotype == DIVY_INFOTYPE_dbshfolder    ||
				    infotype == DIVY_INFOTYPE_dbshfolder_e  ||
				    infotype == DIVY_INFOTYPE_dbshfolder_e_regular) {
					is_deny = 1;
				}
				break;

			/* リポジトリ検索のトップ */
			case DIVY_FOLDER_ID_reposdblink:
				/* リポジトリ検索と検索結果フォルダの閲覧を拒否
				 * (note)
				 * 	ここの拒否だけでは、SQL検索画面の表示
				 * 	は可能になってしまいます。
				 * 	uri のチェックだけではこれが限界です。*/
				if (infotype == DIVY_INFOTYPE_reposdblink   ||
				    infotype == DIVY_INFOTYPE_reposdblink_e) {
					is_deny = 1;
				}
				break;

			/* 管理者フォルダ */
			case DIVY_FOLDER_ID_management:
				if (infotype == DIVY_INFOTYPE_m_management) {
					is_deny = 1;
				}
				break;

			/* 更新クライアントのトップ */
			case DIVY_FOLDER_ID_m_update:
				if (infotype == DIVY_INFOTYPE_m_update ||
				    infotype == DIVY_INFOTYPE_m_update_e) {
					is_deny = 1;
				}
				break;

			/* SQLのトップ */
			case DIVY_FOLDER_ID_m_sql:
				/* SQLエンティティ、リレーション共に拒否
				 * (note)
				 * 	DIVY_INFOTYPE_m_sql のuriに対する制限を
				 * 	掛けると、managementlinkdbsearch、
				 * 	linkdbsearchが動作しなくなるので、掛けて
				 * 	いません。その代わり、SQLの一覧は取得できる
				 * 	可能性が出てきてしまいます。(編集は不可)
				 * 	uri レベルのチェックではこれが限界です */
				if (infotype == DIVY_INFOTYPE_m_sql_e    ||
				    infotype == DIVY_INFOTYPE_m_sql_rgrp ||
				    infotype == DIVY_INFOTYPE_m_group_rsql) {
					is_deny = 1;
				}
				break;

			/* グループのトップ */
			case DIVY_FOLDER_ID_m_group:
				/* グループエンティティの操作を拒否
				 * (リレーション操作は許可) */
				if (infotype == DIVY_INFOTYPE_m_group ||
				    infotype == DIVY_INFOTYPE_m_group_e) {
					is_deny = 1;
				}
				break;

			/* ユーザのトップ */
			case DIVY_FOLDER_ID_m_user:
				/* ユーザエンティティ、リレーション共に拒否 */

				/* (2005/01/18 Tue)
				 * ユーザフォルダ(in 管理者フォルダ), ユーザは
				 * パスワード変更で使用されるURIなので禁止
				 * 出来ませんでした.
				 * Body のチェックはプロバイダがやりますので
				 * ここでは単純に穴をあけるようにします */

				/* ユーザフォルダにはSEARCH,PROPFIND,OPTIONS を,
				 * ユーザには PROPFIND と PROPPATCH を許可します */
				if ((infotype == DIVY_INFOTYPE_m_user &&
					mnum != ap_method_number_of("SEARCH") &&
					mnum != M_PROPFIND &&
					mnum != M_OPTIONS
#ifdef DIVY_SUPPORT_PASSPOLICY
					&& mnum != M_PROPPATCH
					&& mnum != M_LOCK
					&& mnum != M_UNLOCK
#endif	/* DIVY_SUPPORT_PASSPOLICY */
					) ||
				    (infotype == DIVY_INFOTYPE_m_user_e &&
					mnum != M_PROPFIND &&
					mnum != M_PROPPATCH)  ||
				    infotype == DIVY_INFOTYPE_m_user_rgrp ||
				    infotype == DIVY_INFOTYPE_m_group_ruser) {
					is_deny = 1;
				}
				break;

			/* 統計情報のトップ */
			case DIVY_FOLDER_ID_m_status:
				if (infotype == DIVY_INFOTYPE_m_status ||
				    infotype == DIVY_INFOTYPE_m_status_e) {
					is_deny = 1;
				}
				break;
			
			/* システムメッセージのトップ */
			case DIVY_FOLDER_ID_m_msg:
				if (infotype == DIVY_INFOTYPE_m_msg ||
				    infotype == DIVY_INFOTYPE_m_msg_e) {
					is_deny = 1;
				}
				break;

			/* DBMSのトップ */
			case DIVY_FOLDER_ID_m_dbms:
				if (infotype == DIVY_INFOTYPE_m_dbms ||
				    infotype == DIVY_INFOTYPE_m_dbms_e) {
					is_deny = 1;
				}
				break;

			/* データベース(テスト用)のトップ */
			case DIVY_FOLDER_ID_m_execsql:
				if (infotype == DIVY_INFOTYPE_m_execsql ||
				    infotype == DIVY_INFOTYPE_m_execsql_e) {
					is_deny = 1;
				}
				break;
			/* 分類していないもの */
			default:
				break;
		}

		/* アクセス拒否リストに載っていたアクセスだった場合 */
		if (is_deny) {
			/* 直打による不正アクセスの可能性が高いのでユーザIDも出す */
			ERRLOG3(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_PROC,
				"The server does not accept \"%s\" method "
				"on this uri that is listed on access-deny folder."
				"(uri = %s, userid = %s)", divy_get_method_name(r), uri, userid);

			return HTTP_FORBIDDEN;
		}
	}
	return OK;
}

/**
 * 指定されたuserid のユーザがuri_spec が示すグループにアクセスできる
 * 権限をもっているかどうかの検証。
 *
 * @param r request_rec *
 * @param userid const char*
 * @param uri_spec divy_uri_spec *
 * @return int
 *	OK      : アクセスは許可
 *	上記以外: アクセスは拒否
 */
static int _validate_group_access(request_rec *r,
					const char *userid,
					divy_uri_spec *uri_spec)
{
	apr_pool_t *p       = r->pool;
	apr_pool_t *cache_p = (r->main) ? r->main->pool : r->pool;
	divy_cset_t *grpcol_set = NULL;
	char *grpcol_uri;

	/*
	 * このユーザがアクセスできるグループID一覧を取得する
	 */
	/* まずはキャッシュを調べる */
	grpcol_set = divy_pcache_get_data(cache_p, DIVY_PCACHE_DAT_USER_GRPINFO);
	if (grpcol_set == NULL) {
		/* なければDBを引く */
		if (divy_rdbo_get_grpcoluris_by_userid(r, userid, &grpcol_set, NULL)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get grpcol uri from userid.");
			return HTTP_INTERNAL_SERVER_ERROR;
		}

		/* グループコレクションURIを取得できなかった場合 */
		if (grpcol_set == NULL) {
			/* (note) NULLのままにしておくと何度もDBを
			 * 	検索してしまうのでそれを避けるために
			 * 	NULLではないが空のset を作っておく。
			 */
			grpcol_set = divy_cset_make(cache_p);
		}

		/* キャッシュにセットする */
		divy_pcache_set_data(cache_p, grpcol_set, DIVY_PCACHE_DAT_USER_GRPINFO);
	}

	/*
	 * userid のユーザがアクセスできるグループの一覧とURIが一致しているか
	 * どうか検証する
	 * (note)
	 * 	uri = $root/Group Folder/[$groupid | $grpname]/xxxxx/yyyyy/zzzzz
	 */
	if (uri_spec->infotype == DIVY_INFOTYPE_group_e) {
		grpcol_uri = (char *) uri_spec->uri;
	}
	else {
		char *segment = dav_divy_extract_firstpath_segment(p,
						&uri_spec->other_part[1]);

		/* [$groupid | $grpname]/xxxxx からグループコレクションURIを生成 */
		grpcol_uri = divy_build_group_uri(p, dav_divy_get_root_uri(r),
								segment);
	}

	if (divy_cset_contains(grpcol_set, grpcol_uri)) {
		return OK;	/* アクセス可能 */
	}
	else {
		/*
		 * FORBIDDENのエラーではWebDAVRedirectorで再取得の連続となり
		 * リクエストが増えてしまいます。素直にNOTFOUNDとすること
		 * によりこの問題は解決します。
		 */
		return HTTP_NOT_FOUND;	/* 存在しない */
	}
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * usr_pr が示すユーザの入力パスワードpassword をパスワードポリシーに基づき
 * チェックする.
 *
 * @param r request_rec *
 * @param usr_pr const divy_rdbo_usr * ユーザのプロパティ
 * @param password const char * 入力パスワード
 * @param force_plain int 入力パスワードを無条件にプレインパスワードとみなすかどうか
 * @return dav_error * NULL の時に正常. それ以外はポリシー違反あり.
 */
static dav_error * _validate_password_input(request_rec *r,
					const divy_rdbo_usr *usr_pr, const char *password,
					int force_plain)
{
	apr_pool_t *p   = r->pool;
	char *ch        = NULL;
	dav_error *err  = NULL;
	ap_regex_t preg    = { 0 };
	divy_rdbo_passwordpolicy *passpolicy_pr = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *userid       = divy_get_userid(r);
	const char *comp_password;
	int ret;
	int support_passpolicy  = divy_support_passpolicy(r);
	int len;

	/* データ不正 */
	if (IS_EMPTY(usr_pr->password)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"The old user password was EMPTY. (userid = %s)", userid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* パスワードポリシーがサポートされていれば、
	 * パスワードポリシーを取得する */
	if (support_passpolicy &&
		divy_get_cached_passwordpolicy(r, DIVY_DEFAULT_POLICY_ID, 0, &passpolicy_pr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get password-policy property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/*
	 * パスワード文字列の存在確認
	 * (note) このチェックはポリシーの有無に関わらず実施
	 */
	if (IS_EMPTY(password)) {
		/* パスワードポリシーが未サポート / 未設定 / 非アクティブ /
		 * 長さ制約が無いならば、従来通りのエラー扱い */
		if (!support_passpolicy || passpolicy_pr == NULL ||
			passpolicy_pr->status == 0 || passpolicy_pr->minlen == 0) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Password was EMPTY.(userid = %s)", userid);
		}
		else {
			apr_table_addn(r->headers_out, DIVY_HEADER_PASSERR,
					apr_psprintf(p, "password-too-short;minlen=%d",
						passpolicy_pr->minlen));
		}

		/* パスワードがNULLなので続行不能 */
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);	/* ログに出さない */
	}

	/* 最大文字数チェック
	 * (暗号化パスワード機能が有効な場合) */
	len = strlen(password);
	if (dconf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON &&
		len > DIVY_MAX_PASSWORD_LEN) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The password string is bigger than %d", DIVY_MAX_PASSWORD_LEN);
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL); /* ログに出さない */
	}

	/* パスワードポリシーが未サポート / 未設定 / 非アクティブならば終了 */
	if (!support_passpolicy ||
		passpolicy_pr == NULL || passpolicy_pr->status == 0) {
		return NULL;
	}

	/*
	 * 直前に入力されていたパスワードと同じかどうかチェック
	 * [ 実施条件 ]
	 *     このチェックはパスワードポリシー機能の有無に依らず実施します.
	 *
	 * (note) 暗号化パスワード機能との整合性
	 *     暗号化パスワード機能が有効であれば、password は暗号化されているものと
	 *     して扱う. DBに記録されていたパスワードと比較して、等しければ
	 *     「同一エラー」を、等しくなければ、再入力されたものとみなし
	 *     他のポリシーチェックを続行する.
	 */
	/* 暗号化パスワードが有効 */
	if (dconf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON) {
		/* 入力password を"プレイン"であるとみなす場合 */
		if (force_plain) {
			comp_password = ap_md5(p, (const unsigned char*)password);
			/* 同一パスワードチェック */
			if (strcmp(usr_pr->password, comp_password) == 0) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "same-password-found");
				if (err == NULL)
					err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);	/* ログに出さない */
				/* 続行 */
			}
		}
		/* 入力password が暗号化パスワードの場合 */
		else {
			/* (note) 暗号化パスワードが有効である場合、原理的に
			 * 入力されたpassword が暗号化されているのか、プレインなのかわからない.
			 * 仕方がないので、まずはプレインであるとみなして比較して、
			 * 次にMD5 にして比較して判断するしかない.
			 * さもないと暗号化パスワードが有効で、再入力されたケースを
			 * 救済できない. この問題は、そもそも同じパスワードが暗号化されて
			 * 戻ってくることにある. 仕方が無い.
			 */
			comp_password = password;
			if (strcmp(usr_pr->password, comp_password) == 0) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "same-password-found");
				/* 暗号化パスワードが有効 && 旧パスワードと同じであれば、
				 * 入力されてきたパスワードは暗号化パスワードとみなせる.
				 * ならば、これ以上のチェックは実施不能 */
				return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);	/* ログに出さない */
			}

			comp_password = ap_md5(p, (const unsigned char*)password);
			if (strcmp(usr_pr->password, comp_password) == 0) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "same-password-found");
				/* 暗号化パスワードが有効 && 暗号化してみたら旧パスワードと同じであれば、
				 * 入力されてきたパスワードはプレインパスワードだとみなせる.
				 * さらにこのケースでは暗号化後のパスワードが一致したので、
				 * 同一エラーとしてもよいはず. 続行可能であろう. */
			}
		}
	}
	/* 暗号化パスワードが無効 */
	else {
		comp_password = password;
		/* 同一パスワードチェック */
		if (strcmp(usr_pr->password, comp_password) == 0) {
			apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "same-password-found");
			if (err == NULL)
				err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);	/* ログに出さない */
			/* 続行 */
		}
	}

	/* 最小長さチェック */
	if (passpolicy_pr->minlen > 0) {
		if (strlen(password) < passpolicy_pr->minlen) {
			apr_table_addn(r->headers_out, DIVY_HEADER_PASSERR,
					apr_psprintf(p, "password-too-short;minlen=%d", passpolicy_pr->minlen));
			if (err == NULL)
				err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない*/
			/* 続行する(全てのチェックが終わるまで続けること. 外部仕様です) */
		}
	}

	/* ユーザID 文字列を含んでいないかどうかのチェック */
	if (passpolicy_pr->denyUseridString) {
		/* 小文字に変換してから比較する */
		char *pw = divy_tolower_str(p, password);
		char *uid = divy_tolower_str(p, userid);
		if (divy_strstr(pw, uid) != NULL) {
			apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "userid-character-found");
			if (err == NULL)
				err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
		}
	}

	/* 連続文字列を含んでいないかどうかのチェック */
	if (passpolicy_pr->denyCycleChr) {
		char *prev = NULL;
		for (ch = (char *)password; ch != NULL && *ch != '\0'; ch++) {
			if (prev != NULL && *prev == *ch) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "cycle-character-found");
				if (err == NULL)
					err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
				break;	/* 1つでも見つかったら抜ける */
			}
			prev = ch;
		}
	}

	/* アルファベット大文字はMUST */
	if (passpolicy_pr->isUpAlphabet) {
		if ((ret = ap_regcomp(&preg, "[A-Z]",  0)) == 0) {
			if (ap_regexec(&preg, password, 0, NULL, 0) == AP_REG_NOMATCH) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "upper-alphabet-notfound");
				if (err == NULL)
					err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
			}
			ap_regfree(&preg);
		}
	}

	/* アルファベット小文字はMUST */
	if (passpolicy_pr->isLoAlphabet) {
		if ((ret = ap_regcomp(&preg, "[a-z]",  0)) == 0) {
			if (ap_regexec(&preg, password, 0, NULL, 0) == AP_REG_NOMATCH) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "lower-alphabet-notfound");
				if (err == NULL)
					err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
			}
			ap_regfree(&preg);
		}
	}

	/* 数値はMUST */
	if (passpolicy_pr->isNumericNeeded) {
		if ((ret = ap_regcomp(&preg, "[0-9]",  0)) == 0) {
			if (ap_regexec(&preg, password, 0, NULL, 0) == AP_REG_NOMATCH) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "numeric-character-notfound");
				if (err == NULL)
					err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
			}
			ap_regfree(&preg);
		}
	}

	/* 記号文字はMUST */
	if (passpolicy_pr->isSymbolChrNeeded) {
		if ((ret = ap_regcomp(&preg,
				"[!\"#\\$%&'\\(\\)=\\~\\|`\\{\\+\\*\\}<>?^\\_^\\\\@\\[;:\\],\\. /-]",
				0)) == 0) {
			if (ap_regexec(&preg, password, 0, NULL, 0) == AP_REG_NOMATCH) {
				apr_table_add(r->headers_out, DIVY_HEADER_PASSERR, "symbol-character-notfound");
				if (err == NULL)
					err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
			}
			ap_regfree(&preg);
		}
	}

	/* 禁則文字を含んでいないかどうか */
	if (passpolicy_pr->ngword != NULL && passpolicy_pr->ngword->ngword_set != NULL) {
		divy_cset_t *ngword_set = passpolicy_pr->ngword->ngword_set;
		char buf[2] = { 0 };
		for (ch = (char *)password; ch != NULL && *ch != '\0'; ch++) {
			buf[0] = *ch; buf[1] = '\0';
			if (divy_cset_contains(ngword_set, buf)) {
				divy_sbuf *sbuf = NULL;
				divy_cset_index_t *idx = NULL;

				divy_sbuf_create(p, &sbuf, 10);
				for (idx = divy_cset_first(ngword_set);
									idx != NULL; idx = divy_cset_next(idx)) {
					divy_cset_this(idx, (const char **)&ch);
					divy_sbuf_append(sbuf, ch);
				}

				apr_table_addn(r->headers_out, DIVY_HEADER_PASSERR,
								apr_psprintf(p, "ngword-character-found;ngword=%s",
										divy_url_encode(p, divy_sbuf_tostring(sbuf))));
				if (err == NULL)
					err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
				break;
			}
		}
	}

	/* 履歴パスワードに一致しないかどうか */
	if (passpolicy_pr->history_pw_num > 0) {
		divy_rdbo_passhistory *ph = NULL;

		/* パスワード履歴を取得 */
		if (divy_rdbo_get_passwordhistory(r, userid, &ph, NULL)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get password-history.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		if (ph != NULL) {
			int i, cnt = 0;
			for (i = 0; i < DIVY_MAX_PASSHISTORY_NUM; i++) {
				if (IS_EMPTY(ph->passhistory[i])) {
					continue;
				}

				if (strcmp(comp_password, ph->passhistory[i]) == 0) {
					apr_table_add(r->headers_out, DIVY_HEADER_PASSERR,
									"previously-used-password-found");
					if (err == NULL)
						err = dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, NULL);/* ログに出さない */
					break;
				}
				/* 保持可能な履歴数に達したら、DBには記録されているかもしれないが
				 * 無視すること */
				if (++cnt == passpolicy_pr->history_pw_num) {
					break;
				}
			}
		}
	}

	return err;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * 現在登録されているユーザ数を調べて、ユーザ１人を追加しても
 * 契約人数を超えないかどうか検証する。
 * (note)
 *   システム実行権限を持つユーザはカウントしないこと
 *
 * @param r request_rec *
 * @return dav_error (NULL: 問題なし / not null: 問題あり)
 * 	HTTP_INTERNAL_SERVER_ERROR : ユーザ数のカウントに失敗した
 * 	HTTP_PAYMENT_REQUIRED      : 許容人数を超えた
 */
static dav_error * _validate_creating_user_limit(request_rec *r)
{

	int c_user_cnt    = 0;
	int contract_user = dav_divy_get_contract_user_count(r->server);
	int allow_user    = dav_divy_get_allow_user_count(r->server);
	apr_pool_t *p     = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	/*
	 * 適切なユーザ追加ライセンスがなかった場合
	 */
	if (contract_user == DIVY_INT32_C_UNSET) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"The license of creating user acount was not found."
				"Please check server-license key.");
		return dav_new_error(p, HTTP_PAYMENT_REQUIRED, 0, 0, "");
	}

	/*
	 * Location内で設定された最大ユーザ数制限が指定された場合
	 *
	 * [ 条件 ]
	 *   * 何らかの値が設定されている(ディレクティブが省略されていない)
	 *   * 無制限(=0)ではない有限の値が設定されている
	 */
	if (dconf->maxuser != DIVY_INT32_C_UNSET && dconf->maxuser > 0) {
		/* 現在登録されているユーザ数を調べる */
		if (divy_rdbo_count_usr(r, &c_user_cnt)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get current user count.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		c_user_cnt++;	/* これから登録する人をひとり加える */

		if (allow_user == DIVY_INT32_C_UNSET) {
			/* 判定しやすいように値を変える。この場合の"0"は
			 * 許容人数が0人であるということを表していることに注意 */
			allow_user = 0;
		}

		/* Location内の最大ユーザ数の設定値がライセンス数よりも多ければ警告を出して
		 * Location内の最大ユーザ数を無視する */
		if (contract_user > 0) {
			if (dconf->maxuser > contract_user + allow_user) {
				/* 警告を出す */
				ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
						"The number of maxuser is bigger than the license of creating user acount."
						"we use license's value.");
				/* contract_user + allow_user が比較対象 */

				/*
				 * 契約人数を超えているが、許容人数範囲内なら、
				 * 警告を出して、作成を許可する
				 */ 
				if (c_user_cnt > contract_user && c_user_cnt <= contract_user + allow_user) {

					ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
							"The number of contract user is over. But, the server "
							"allow you to be able to create some users.");
				}
				/* ライセンス数 + 許容人数を超えた時は、エラーとする */
				else if (c_user_cnt > contract_user + allow_user) {

					ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
							"The limit value of creating user is over. "
							"the server can not allow you to make a new user.");
					return dav_new_error(p, HTTP_PAYMENT_REQUIRED, 0, 0, "");
				}
			}
			/* contract_user + allow_user の方が大きい場合 */
			else {
				/* dconf->maxuser が比較対象 */
				if (c_user_cnt > dconf->maxuser) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
							"The limit value of creating user is over (TfMaxUser limit). "
							"the server can not allow you to make a new user.");
					return dav_new_error(p, HTTP_PAYMENT_REQUIRED, 0, 0, "");
				}
			}
		}
		/* ライセンス数が無制限の場合 -> dconf->maxuser が比較対象となる */
		else {
			/* dconf->maxuser が比較対象 */
			if (c_user_cnt > dconf->maxuser) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
						"The limit value of creating user is over (TfMaxUser limit). "
						"the server can not allow you to make a new user.");
				return dav_new_error(p, HTTP_PAYMENT_REQUIRED, 0, 0, "");
			}
		}
	}
	/* 最大ユーザ数が未設定 or 0 だった場合 */
	else {
		/* 無制限ライセンスを持っている場合 */
		if (contract_user == 0) {
			return NULL;	/* チェックする必要なし */
		}

		/* 現在登録されているユーザ数を調べる */
		if (divy_rdbo_count_usr(r, &c_user_cnt)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get current user count.");
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0,  0, "");
		}
		c_user_cnt++;

		if (allow_user == DIVY_INT32_C_UNSET) {
			allow_user = 0;
		}

		/*
		 * 契約人数を超えているが、許容人数範囲内なら、
		 * 警告を出して、作成を許可する
		 */ 
		if (c_user_cnt > contract_user && c_user_cnt <= contract_user + allow_user) {

			ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"The number of contract user is over. But, the server "
					"allow you to be able to create some users.");
		}
		/* 指定されて契約人数 + 許容人数を超えた時は、エラーとする */
		else if (c_user_cnt > contract_user + allow_user) {

			ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The limit value of creating user is over. "
					"the server can not allow you to make a new user.");
			return dav_new_error(p, HTTP_PAYMENT_REQUIRED, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * アクセスユーザ (= グループリーダ) が管理下に置いているユーザ数を調べて
 * ユーザ１人を追加しても最大値を超えないかどうか検証する。
 * (note)
 *   Other ユーザはカウントしません (外部仕様の要請)
 *
 * @param r request_rec *
 * @return dav_error (NULL: 問題なし / not null: 問題あり)
 * 	HTTP_INTERNAL_SERVER_ERROR : ユーザ数のカウントに失敗した
 * 	HTTP_PAYMENT_REQUIRED      : 許容人数を超えた
 */

static dav_error * _validate_control_user_limit(request_rec *r)
{
	apr_pool_t *p = r->pool;
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	const char *own_userid = divy_get_userid(r);
	apr_int32_t max        = divy_get_usermaxusercreation(r);
	int nuser = 0;

	/* グループリーダ機能が未サポートならば関係なし */
	if (!divy_support_groupleader(r)) {
		return NULL;
	}

	/*
	 * グループリーダ自身に課せられたユーザ作成リミットを評価する.
	 */
	if (divy_rdbo_is_groupleader(own_extstatus) && max > 0) {

		/* 管理下に置かれたユーザの総数を算出する */
		if (divy_rdbo_count_owner_users(r, own_userid, &nuser)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get the number of user under the groupleader. "
					"(leader = %s)", own_userid);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* これからユーザを登録したら(= nuser + 1)、max を越えるのか? */
		if (nuser + 1 > max) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The limit value of creating user is over for groupleader. "
					"the server can not allow you to make a new user. "
					"(leader = %s)", own_userid);

			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_EXCEED_MAXCREATION);	/* ヘッダに設定 */
			return dav_new_error(p, HTTP_PAYMENT_REQUIRED, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * qsort 関数のComparator.
 *
 * 通常バインド変数の昇順の後にRequiredSQLまたは名前付きバインド変数の
 * 昇順に並べる
 *
 * [ 比較ルール ]
 * ・同じcontype ならばid の大きさが大小を決める
 * ・contype == DIVY_SQL_CONTYPE_BIND と
 *   contype == DIVY_SQL_CONTYPE_REQUIRED ならば、
 *   DIVY_SQL_CONTYPE_BIND の方が常に大きい
 *
 * @param e1 const void * divy_rdbo_sqlcontent *
 * @param e2 const void * divy_rdbo_sqlcontent *
 * @return int 比較処理結果 (-1, 0, 1)
 */
int _compar_id(const void *e1, const void *e2)
{
	divy_rdbo_sqlcontent *sqlcnt1 = *(void **) e1;
	divy_rdbo_sqlcontent *sqlcnt2 = *(void **) e2;

	/* contype が同じ場合 */
	if (sqlcnt1->contype == sqlcnt2->contype) {
		if (sqlcnt1->id < sqlcnt2->id) {
			return -1;
		}
		else if (sqlcnt1->id == sqlcnt2->id) {
			return 0;
		}
		else {
			return 1;
		}
	}
	/* sqlcnt1 が通常バインド変数、sqlcnt2 がRequired系の場合 */
	else if (sqlcnt1->contype == DIVY_SQL_CONTYPE_BIND &&
	    	 sqlcnt2->contype == DIVY_SQL_CONTYPE_REQUIRED) {
		return -1;
	}
	/* sqlcnt1 がequired系、sqlcnt2 が通常バインド変数の場合 */
	else {
		return 1;
	}
}

/**
 * View属性(公開/非公開状態) の検証
 *
 * @param r request_rec *
 * @param chk_mode int
 * @param rstate_iscreen divy_resourcestate_iscreen * 検証する値
 */
static dav_error * _validate_view_state(request_rec *r,
					int chk_mode,
					divy_resourcestate_iscreen *rstate_iscreen)
{
	apr_pool_t *p = r->pool;
	divy_uri_spec *u_spec = rstate_iscreen->u_spec;
	divy_rdbo_resourcestate *rstate_pr = NULL;
	const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);

	/* View属性設定権限を持っているか? */
	if (!divy_rdbo_has_setview_privilege(extstatus)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"You could not have \"set-view\" privilege.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* 制限ユーザではないこと */
	if (!divy_rdbo_is_trusty_user(extstatus)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Need to normal or admin privilege.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* View属性を付加できるリソースか ? */
	if (u_spec->infotype != DIVY_INFOTYPE_group_e_regular) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"resourcestate\" property was only specifed in group-resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* set なのに公開属性がなければNG
	 * (note) 現状はset で非公開できない構造にしました */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && (rstate_iscreen->state.published == 0)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"published\" element was missing.");
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	rstate_iscreen->need_to_update = 1;	/* 原則的には更新処理が必要である */

	/*
	 * remove オペレーション
	 * (note)
	 * 	有効なView属性を持っていなくてもremove オペレーションは許可しています。
	 * 	寛容に扱うということです。ただ、DBへの書き込みなどは必要ないので
	 * 	スキップさせています。
	 */
	if ((chk_mode & DIVY_VALIDATE_DELETE)) {
		/* (note)
		 *   本来ならrstate->uri を使って現在の状態・属性プロパティを調べ
		 *   更新しなくてもいいケースであればスキップするのがまともな実装です。
		 *   残念ながら状態・属性プロパティを記録するテーブルは構造上データ
		 *   同時アクセスに耐えられず、そうであるにもかかわらず同時実行性能を
		 *   向上させる目的でテーブルロックも行っていないので、データ不正が
		 *   起き易い状態にあります。
		 *   そのため、ユーザからのアクセスがあれば不正になっている状態を
		 *   正すチャンスであると捕らえ、常に更新を行うようにしました。
		 *   そんなのは許容できないということであれば、親＆自分＆子リソースの
		 *   状態・属性プロパティをきちんとロックを掛けて調べて処理をスキップ
		 *   させて下さい。結果的に更新が必要だった場合にはコストが増大します。
		 */
		rstate_iscreen->state.published = 0;	/* 非公開にする */
		rstate_iscreen->need_to_update = 1;

		/* remove であればもうチェックは必要ありません */
		return NULL;
	}

	/* View属性(公開/非公開状態) をもっているか？ */
	if (divy_rdbo_get_resourcestate_property(r, rstate_iscreen->uri,
						DIVY_RSTATE_TYPE_VIEW, 1, &rstate_pr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get resourcestate property.");
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	rstate_iscreen->src_rstate_pr  = rstate_pr;

	/*
	 * set オペレーションの場合
	 *
	 * View属性の連続性, 既に保持していないかどうかを調べる
	 */
	/* 公開する場合 */
	if (rstate_iscreen->state.published) {

		/* グループフォルダ直下のリソース＆コレクションの場合 */
		if (u_spec->p_infotype == DIVY_INFOTYPE_group_e &&
		    u_spec->infotype   == DIVY_INFOTYPE_group_e_regular) {
			/* 公開されていた */
			if (IS_EFFECTIVE_STATE(rstate_pr)) {
				/* 本当はNGなのだがキャッシュ状況によって起き得るので寛容に対処 */
				rstate_iscreen->need_to_update = 0;
			}
			/* 公開属性なし / 非公開にされていた */
			else {
				/* 更新の必要あり。ここでの検証は終わり */
				rstate_iscreen->need_to_update = 1;
			}
		}
		/* グループフォルダ直下の下のリソース＆コレクションの場合 */
		else if (u_spec->p_infotype == DIVY_INFOTYPE_group_e_regular &&
			 u_spec->infotype   == DIVY_INFOTYPE_group_e_regular) {
			/* 自身が公開されていた */
			if (IS_EFFECTIVE_STATE(rstate_pr) && !rstate_pr->inherit) {
				/* 本当はデータ不正なのだけれど公開されていたのならまあいい。
				 * 但しデータ不正を正すためDBの更新は必要です*/
				rstate_iscreen->need_to_update = 1;
			}
			/* 自身が非公開 */
			else if (IS_INEFFECTIVE_STATE(rstate_pr) && rstate_pr && !rstate_pr->inherit) {
				/* 公開する必要あり */
				rstate_iscreen->need_to_update = 1;
			}
			/* 親が公開されていた */
			else if (IS_EFFECTIVE_STATE(rstate_pr) && rstate_pr->inherit) {
				/* 本当はNGだけどキャッシュ状況によっては起き得るので寛容に対処 */
				rstate_iscreen->need_to_update = 0;
			}
			/* 親が非公開 */
			else {
				/* 公開属性の連続性を維持できないのでこれはエラー 
				 * クライアントのキャッシュ状態によっては起き得ますが、原因がわからなくなるので
				 * ログには出力しておきます。*/
				ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The parent collection did not have effective published state. "
					"The published state must be continuous.(uri = %s)", rstate_iscreen->uri);
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
		}
		else {
			/* 対象外です。事前判定してあるので、特に何もしません。 */
		}
	}

	return NULL;
}

/**
 * BOX属性 の検証
 *
 * @param r request_rec *
 * @param chk_mode int
 * @param rstate_iscreen divy_resourcestate_iscreen * 検証する値
 */
static dav_error * _validate_box_state(request_rec *r,
					int chk_mode,
					divy_resourcestate_iscreen *rstate_iscreen)
{
	apr_pool_t *p = r->pool;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	divy_uri_spec *u_spec = rstate_iscreen->u_spec;
	divy_rdbo_resourcetype resourcetype = rstate_iscreen->resourcetype;
	const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);
	divy_rdbo_resourcestate *rstate_pr = NULL;
	divy_rdbo_extstatus *grp_extstatus = NULL;

	/* 制限ユーザではないこと */
	if (!divy_rdbo_is_trusty_user(extstatus)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"Need to normal or admin privilege.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* BOX属性を付加できるリソースで且つコレクションか ? */
	if (!(u_spec->infotype == DIVY_INFOTYPE_group_e_regular
							&& resourcetype == DIVY_TYPE_COLLECTION)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
			"The \"resourcestate\" property was only specifed in group-resource.");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* グループ属性を取得 */
	(void) divy_get_cached_availablegroupextstatus(r, r->uri, &grp_extstatus);

	/* BOX機能が有効になっているグループか? */
	if (!divy_rdbo_is_box_group(grp_extstatus)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The \"group\" disable box property");
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	/* setオペレーション */
	/* ボックス機能をOffにするのはsetオペレーションでは行えない(外部仕様) */
	if ((chk_mode & DIVY_VALIDATE_UPDATE) && (rstate_iscreen->state.box == 0)) {
		/* TODO ### ERRLOGをかきなさい */
		return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
	}

	rstate_iscreen->need_to_update = 1;	/* 更新処理が必要である */

	/* removeオペレーション */
	if ((chk_mode & DIVY_VALIDATE_DELETE)) {
		rstate_iscreen->state.box = 0;	/* 使用不可 */
		rstate_iscreen->need_to_update = 1;

		/* remove であればもうチェックは必要ありません */
		return NULL;
	}

	/* BOX属性 をもっているか？ */
	if (divy_rdbo_get_resourcestate_property(r, rstate_iscreen->uri,
						DIVY_RSTATE_TYPE_BOX, 1, &rstate_pr)) {
		/*
		 * BOX属性をもっていたリソースなら更新できるかを調べます
		 */
		if (rstate_pr->type == DIVY_RSTATE_TYPE_BOX) {
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}
	rstate_iscreen->src_rstate_pr  = rstate_pr;

	if (rstate_iscreen->state.box) {

		/* グループフォルダ直下のコレクションの場合 */
		if (u_spec->p_infotype == DIVY_INFOTYPE_group_e &&
		    u_spec->infotype   == DIVY_INFOTYPE_group_e_regular && 
			rstate_iscreen->resourcetype == DIVY_TYPE_COLLECTION) {
			
			/* 管理者でなくサーバポリシー(boxlimit)が設定されていれば
			 * チェックをする.管理者はポリシーの制限を受けません（外部仕様）
			 */
			if (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN && 
											conf->tfboxpolicylimit > 0) {
				time_t expiration = 0L;
				time_t creation = 0L;
				/* リミットがあるのに要求が無制限だった */
				if (IS_EMPTY(rstate_iscreen->optional.boxinfo->expiration)) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
							"box policy violation: unlimited expiration.");
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
				expiration = dav_divy_iso8601totime_t(p,
								rstate_iscreen->optional.boxinfo->expiration);
				creation = dav_divy_get_now_epoch();	/* 本日日付 */

				/* リミットの期間より長い有効期限だった */
				if (expiration - creation > conf->tfboxpolicylimit) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
							"box policy violation: over expiration.");
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}
			/* サーバポリシーによるメールアドレスのワイルドカード設定禁止
			 * 設定があれば一般ユーザは*を設定できないが、 * 管理者もしくは
			 * グループリーダーは可能です。
			 */
			if (divy_get_adminmode(r) != DIVY_ADMINMODE_ADMIN &&
					(divy_support_groupleader(r) &&
							divy_rdbo_is_groupleader(extstatus) == 0) &&
									conf->boxdisablewildcardmailaddr) {

				/* アスタリスクが含まれているとポリシー違反 */
				if (divy_strchr(rstate_iscreen->optional.boxinfo->tomailaddr,
																'*') != NULL) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"box policy violation: disable wildcard."
					"[user=%s] [to-mail=%s]", r->user,
					rstate_iscreen->optional.boxinfo->tomailaddr);

					/* result-stateヘッダを付与する */
					divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_BOX_DISABLED_WILDCARD_TOMAILADDRESS);
					return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
				}
			}

		}
		/* グループフォルダ直下の下のリソース＆コレクションの場合 */
		else if (u_spec->p_infotype == DIVY_INFOTYPE_group_e_regular &&
			 u_spec->infotype   == DIVY_INFOTYPE_group_e_regular) {

			/* 外部仕様上グループ直下のコレクションのみが設定を行えます
			 * その配下のリソースはすべて親に準ずることになります。
			 * 設定の解除などを行う場合には必ずグループ直下のコレクションを
			 * 起点にして行う必要があります。
			 */

			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to proppatch resoruce");
			/* グループの下の下へはエラーとする */
			return dav_new_error(p, HTTP_BAD_REQUEST, 0, 0, "");
		}
	}

	return NULL;
}

/**
 * Destination ヘッダから相対URIを切り出して返却する。
 * (例)
 *  http://aaa:81/divy/abc/ddd  --> /divy/abc/ddd
 *
 * @param r request_rec *
 * @return char * 切り出された相対URI
 */
static char * _get_destination_uri(request_rec *r)
{
	apr_status_t rv;
	apr_pool_t *p      = r->pool;
	char *dst_uri      = NULL;
	apr_uri_t dst_comp = { 0 };
	const char *dest   = apr_table_get(r->headers_in, "Destination");

	if (IS_EMPTY(dest)) {
		return NULL;
	}

	/* destination uri のパース */
	if ((rv = apr_uri_parse(p, dest, &dst_comp)) != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to parse destination uri.(code = %d)", rv);
		return NULL;
	}

	dav_divy_unescape_uri(p, dav_divy_remove_endslash(p, dst_comp.path), &dst_uri);
	ap_no2slash(dst_uri);

	return dst_uri;
}

#ifdef DIVY_SUPPORT_PLUGIN
/**
 * セキュリティで保護されたグループへのアクセスが許可されているかどうか.
 *
 * @return dav_error * NULL で許可されている
 */
static dav_error * _validate_secured_folder(request_rec *r,
					const divy_uri_spec *src_u_spec, const divy_uri_spec *dst_u_spec)
{
	apr_pool_t *p = r->pool;

	/* dst がグループコレクション以下の場合 */
	if (dst_u_spec->infotype == DIVY_INFOTYPE_group_trash ||
		dst_u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
		dst_u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
		dst_u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {

		tfs_hset_t *gset = NULL;
		int ret;
		/* 暗号化対象グループへの移動/コピーは禁止 */
		ret = divy_pi_cciphered_grpset(r, &gset);
		if (ret == DIVY_PL_ST_OK && gset != NULL) {
			divy_rdbo_grp *grp_pr = NULL;

			/* グループIDを取得する */
			if (divy_rdbo_get_group_property_by_resuri(r,
						dst_u_spec->uri, &grp_pr)) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get group property.");
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
			else if (grp_pr != NULL) {
				/* 暗号化対象になっていたら禁止対象の可能性あり */
				if (tfs_hset_contains(gset, grp_pr->grpid)) {
					if (src_u_spec->infotype == DIVY_INFOTYPE_group_trash ||
						src_u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
						src_u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
						src_u_spec->infotype == DIVY_INFOTYPE_group_e_regular) {
						divy_rdbo_grp *src_grp_pr = NULL;

						/* 同一グループ内の移動/コピーは許可 */
						if (strcmp(src_u_spec->other_part, dst_u_spec->other_part) == 0) {
							/* 許可する. 何もしない */
						}
						else {
							/* グループIDを取得する */
							if (divy_rdbo_get_group_property_by_resuri(r,
										src_u_spec->uri, &src_grp_pr)) {
								ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
										"Failed to get group property.");
								return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
							}
							else if (src_grp_pr != NULL) {
								/* src も暗号化されていれば許可 */
								if (tfs_hset_contains(gset, src_grp_pr->grpid)) {
									/* 何もしない */
								}
								/* src が暗号化されていない場合 */
								else {
									return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
								}
							}
						}
					}
					/* src がグループ以外であればNG */
					else {
						return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
					}
				}
			}
		}
		else if (ret == DIVY_PL_ST_ERR) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get groupid set.");
			/* 続ける(大目に見る) */
		}
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_PLUGIN */

/**
 * あるファイルサイズがリミットを越えていないかどうか事前確認する.
 *
 */
static dav_error * _validate_pre_fsizelimit(request_rec *r, divy_infotype infotype, apr_int64_t filelen)
{
	apr_pool_t *p = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	/* 値が0 / 未設定の場合はunlimited */
	if (dconf->maxfilesize == DIVY_INT64_C_UNSET || dconf->maxfilesize == APR_INT64_C(0)) {
		return NULL;	/* リミットなし */
	}

	/* メール送信のPUTの場合 */
	if (infotype == DIVY_INFOTYPE_mail_e) {

		/* メール送信可能な最大サイズを超えた時 */
		if (filelen > DIVY_MAX_MAILBODY_SIZE) {

			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"The maximum mail size restriction was exceeded. "
					"Please change a restriction value or warn a user."
					"(uri = %s, userid = %s, size = %"APR_INT64_T_FMT")",
					r->uri, divy_get_userid(r), filelen);
			return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
		}
	}
	/* 通常リソースのPUT */
	/* PUT可能な最大サイズを超えた時 */
	else if (dconf->maxfilesize < filelen) {

		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"The maximum file size restrictions which "
				"can be uploaded by one operation were exceeded. "
				"Please change a restriction value or warn a user."
				"(uri = %s, userid = %s, size = %"APR_INT64_T_FMT")",
				r->uri, divy_get_userid(r), filelen);
		return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}

	return NULL;
}

/**
 * ユーザQuota の事前確認.
 *
 */
static dav_error * _validate_pre_userquota(request_rec *r, divy_infotype infotype, apr_int64_t filelen)
{
	apr_pool_t *p = r->pool;
	int ret;
	divy_rdbo_usr uquota = { 0 };	/* ユーザQuota */

	/*
	 * [ 処理開始条件 ]
	 * 	* メールのPUTではないこと
	 * 	* 境界サイズを越えていること
	 */
	if (infotype == DIVY_INFOTYPE_mail_e || filelen <= DIVY_BOUNDARY_BYTE) {
		return NULL;	/* 判定しない */
	}

	/*
	 * ユーザQuotaの取得(ロックなし)
	 */
	uquota.usrid = (char *) divy_get_userid(r);

	ret = divy_rdbo_get_userquota(r, &uquota, 0, NULL);
	if (ret == DIVY_STCODE_USER_NOT_EXIST) {
		/* 対象ユーザがいなかった(これはありえないが) */
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}
	else if (ret != 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get user quota information. (userid = %s)", uquota.usrid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	/* Size Quota に抵触 */
	if (uquota.maxst != DIVY_UNLIMITED_USRQUOTA_BYTES &&
		uquota.maxst < uquota.usedst + filelen) {

		ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"There is not enough storage to write to this resource for user quota limit."
				"(userid = %s, %s)", uquota.usrid,
				apr_psprintf(p, "change size = %"APR_INT64_T_FMT
								", usedst = %"APR_INT64_T_FMT", maxst = %"APR_INT64_T_FMT,
								filelen, uquota.usedst, uquota.maxst));
		return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}
	/* ファイル数 Quota に抵触 */
	else if (uquota.maxres != DIVY_UNLIMITED_USRQUOTA_FILES &&
			 uquota.maxres < uquota.usedres + APR_INT64_C(1)) {

		ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"There is not enough storage to write to this resource for user quota limit."
				"(userid = %s, change count = 1, usedres = %"APR_INT64_T_FMT", "
				"maxres = %"APR_INT64_T_FMT")", uquota.usrid, uquota.usedst, uquota.maxst);
		return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
	}

	return NULL;
}

#ifdef DIVY_SUPPORT_EXTENDQUOTA
/**
 * システムQuotaの事前確認
 *
 * (2004/05/24 Mon takehara)
 * 		顧客の要望により、Quota の事前判定を行うことになりました。
 * 		ですが、ソケットを強制切断してエラーを返しても
 * 		それを受け取れないクライアントが存在するのは変わりありません。
 * 		Quotaエラーをそうであるとクライアントに通知できないのであれば、
 * 		たとえユーザを待たせないとしても、どの種類のエラーが
 * 		でたのか分からないので意味がありません。
 * 		だってまたエンドユーザは同じオペレーションを繰り返すからです。
 * 		何だか判らないエラーではね。という訳で疑問の残る対応です。
 * 		またこの修正によりパフォーマンスも悪くなります。
 */
static dav_error * _validate_pre_sysquota(request_rec *r, divy_infotype infotype, apr_int64_t filelen)
{
	apr_pool_t *p = r->pool;
	int ret;
	char *userid = (char *) divy_get_userid(r);
	divy_rdbo_diskquota squota = { 0 };	/* システムQuota */

	/*
	 * [ 処理開始条件 ]
	 * 	* メールのPUTではないこと
	 * 	* 境界サイズを越えていること
	 */
	if (infotype == DIVY_INFOTYPE_mail_e || filelen <= DIVY_BOUNDARY_BYTE) {
		return NULL;	/* 判定しない */
	}

	/*
	 * システムQuotaの取得
	 */
	squota.uri  = apr_pstrdup(p, dav_divy_get_root_uri(r));
	squota.type = DIVY_QTTYPE_SYSTEM;

	ret = divy_rdbo_get_systemquota(r, &squota, 0, NULL);
	if (ret == 0) {
		/* システムQuotaが存在した場合 */

		/* Size Quota に抵触 */
		if (squota.maxst != DIVY_UNLIMITED_SYSQUOTA_BYTES &&
			squota.maxst < squota.usedst + filelen) {

			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"There is not enough storage to write to this resource "
					"for system quota limit.(userid = %s, %s)", userid,
					apr_psprintf(p,
						"change size = %"APR_INT64_T_FMT
						", usedst = %"APR_INT64_T_FMT
						", maxst = %"APR_INT64_T_FMT,
						filelen, squota.usedst, squota.maxst));
			return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
		}
		/* ファイル数 Quota に抵触 */
		else if (squota.maxres != DIVY_UNLIMITED_SYSQUOTA_FILES &&
				squota.maxres < squota.usedres + APR_INT64_C(1)) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"There is not enough storage to write to this resource "
					"for system quota limit. (userid = %s, change count = 1, "
					"usedres = %"APR_INT64_T_FMT", maxres = %"APR_INT64_T_FMT")",
					userid, squota.usedst, squota.maxst);
			return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
		}
	}
	else if (ret == DIVY_STCODE_SYSQUOTA_NOT_EXIST) {
		/* エントリがなければ無制限Quota(互換性の考慮) */
		/* ここは何もしない */
	}
	else if (ret) {
		/* 予期せぬエラー発生 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get system-quota information. (location = %s)", squota.uri);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL;
}
#endif	/* DIVY_SUPPORT_EXTENDQUOTA */

#ifdef DIVY_SUPPORT_GROUPQUOTA
/**
 * グループQuotaの事前確認
 *
 */
static dav_error * _validate_pre_grpquota(request_rec *r, divy_infotype infotype, apr_int64_t filelen)
{
	apr_pool_t *p = r->pool;
	int ret;
	int isGroupFolder = 0;
	divy_rdbo_grp *grp_pr = NULL;

	if (!divy_support_groupquota(r)) return NULL;

	/* 場所のチェック */
	if (infotype == DIVY_INFOTYPE_group_e         || 
		infotype == DIVY_INFOTYPE_group_trash     ||
		infotype == DIVY_INFOTYPE_group_trash_e0  ||
		infotype == DIVY_INFOTYPE_group_trash_ex  || 
		infotype == DIVY_INFOTYPE_group_e_regular) 
	{
		isGroupFolder = 1;	/* グループフォルダです */
	}

	/*
	 * [ 処理開始条件 ]
	 * 	* メールのPUTではないこと
	 * 	* 境界サイズを越えていること
	 *  * グループの配下であること
	 *
	 */
	if (infotype == DIVY_INFOTYPE_mail_e || filelen <= DIVY_BOUNDARY_BYTE || 
		isGroupFolder == 0) {
		return NULL;	/* 判定しない */
	}

	/* グループプロパティを取得する */
	ret = divy_rdbo_get_group_property_by_resuri(r, r->uri, &grp_pr);

	if (ret == 0) {
		/* Size Quota に接触 */
		if (grp_pr->grp_q->maxst != DIVY_UNLIMITED_GRPQUOTA_BYTES &&
			grp_pr->grp_q->maxst < grp_pr->grp_q->usedst + filelen) {

			return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
		}
		/* ファイル数 Quota に接触 */
		else if (grp_pr->grp_q->maxres != DIVY_UNLIMITED_GRPQUOTA_FILES &&
				 grp_pr->grp_q->maxres < grp_pr->grp_q->maxres + APR_INT64_C(1)) {

			return dav_new_error(p, HTTP_INSUFFICIENT_STORAGE, 0, 0, "");
		}
		else if (ret == DIVY_STCODE_GRPQUOTA_NOT_EXIST) {
			/* エントリがなければ無制限Quota（互換性の考慮 */
		}
	}
	else if (ret) {
		/* 予期せぬエラー発生 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to get group-quota information. (grpid = %s)", grp_pr->grpid);
		return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
	}

	return NULL; /* 判定しない */
}
#endif /* DIVY_SUPPORT_GROUPQUOTA */

/**
 * 任命済みグループリーダに対するオペレーションの検証
 *
 */
static dav_error * _validate_appointed_groupleader(request_rec *r, divy_infotype infotype,
													const dav_resource *resource, int is_src_resource)
{
	apr_pool_t *p = r->pool;
	int mnum = divy_get_method_number(r);
	char *userid = NULL;
	divy_rdbo_grp *grp = NULL;

	/* 未サポートならば制限は無関係 */
	if (!divy_support_groupleader(r)) {
		return NULL;
	}

	/*
	 * 制限は任命済みグループリーダ(= ユーザリレーション, グループリレーション) に対する
	 * DELETE / MOVE(src) にのみかかる. それ以外は除外.
	 */
	if (!((infotype == DIVY_INFOTYPE_m_group_ruser || infotype == DIVY_INFOTYPE_m_user_rgrp) &&
		  ((mnum == M_MOVE && is_src_resource) || mnum == M_DELETE))) {
		return NULL;	/* 制限なし */
	}

	/*
	 * 競合オペレーション/ 存在しないリソースに対するアクセス等により
	 * 対象のグループリーダが既にグループに所属していなかったら何もできません.
	 */
	if (!resource->exists) {
		return NULL;
	}

	/*
	 * 任命済みグループリーダは移動 or 削除できない
	 */
			
	if (infotype == DIVY_INFOTYPE_m_group_ruser) {
		divy_rdbo_rusr *src_rusr = NULL;

		if (resource->info->list.rusr_pr != NULL) {
			src_rusr = resource->info->list.rusr_pr;
			if (src_rusr == NULL) {	/* リレーションがなかったということ */
				return NULL;
			}
		}

		if (divy_rdbo_get_hierarchy_group_property(r, src_rusr->grp_uri, 0, &grp, NULL)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get group property. (grp_uri = %s)", src_rusr->grp_uri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		userid = src_rusr->usrid;
	}
	else {
		divy_rdbo_rgrp *src_rgrp = NULL;

		if (resource->info->list.rgrp_pr != NULL) {
			src_rgrp = resource->info->list.rgrp_pr;
			if (src_rgrp == NULL) {	/* リレーションがなかったということ */
				return NULL;
			}
		}

		if (divy_rdbo_get_group_property_by_groupid(r, src_rgrp->grpid, &grp, NULL)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get group property. (grpid = %s)", src_rgrp->grpid);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}
		userid = src_rgrp->usrid;
	}

	/* 対象グループのオーナになっているユーザは移動/削除NG
	 * (note) サブグループは例外 */
	if (grp != NULL && grp->depth == 1 &&
		IS_FILLED(grp->ownerid) && strcmp(grp->ownerid, userid) == 0) {

		ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
				"You could not move or delete this user-relation or group-relation "
				"because this user is appointed as groupleader.(userid = %s)", userid);

		divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_APPOINTED_GROUPLEADER);
		return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
	}

	return NULL;
}

/**
 * アクセスユーザがグループリーダの場合、grp_pr が示すグループに操作を適用できるかどうか
 */
static dav_error * _validate_group_via_groupleader(request_rec *r, divy_infotype infotype,
													divy_rdbo_grp *grp_pr, int is_src_resource,
													divy_uri_spec *src_u_spec, divy_uri_spec *dst_u_spec)
{
	apr_pool_t *p = r->pool;
	int mnum = divy_get_method_number(r);
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	const char *own_userid = divy_get_userid(r);

	/* 未サポート / グループリーダでない / グループ以外であれば
	 * 制限は無関係 */
	if (!divy_support_groupleader(r) || !divy_rdbo_is_groupleader(own_extstatus) ||
		infotype != DIVY_INFOTYPE_m_group_e) {
		return NULL;
	}

	/* 以下の操作以外はここでは面倒を見ない
	 * (note) SEARCH はgrp_pr が常にNULLなので、チェックできません */
	if (!(mnum == M_DELETE   || mnum == M_PROPPATCH ||
		  mnum == M_MKCOL    || mnum == M_MOVE      ||
		  mnum == M_PROPFIND ||
		  mnum == M_LOCK     || mnum == M_UNLOCK)) {
		return NULL;
	}

	/* MKCOL 以外では、存在しないグループに対して検証できることはない
	 * (note) NULL リソースに対するPROPFIND もチェックしない(できない) */
	if (grp_pr == NULL && mnum != M_MKCOL) {
		return NULL;
	}

	/*
	 * MOVE (移動、リネーム) に対する制限 (src だけでチェック)
	 */
	if (mnum == M_MOVE && is_src_resource) {

		/* src が自身の管理下にないグループ -> NG */
		if (IS_EMPTY(grp_pr->ownerid) || strcmp(own_userid, grp_pr->ownerid) != 0) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not move this group of another user."
					"Need to administrive privilege.(group name = %s, userid = %s)",
					grp_pr->name, own_userid);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}

		/* src がトップグループ -> NG */
		if (grp_pr->depth == 1) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not move this top-group to another layler."
					"Need to administrive privilege.(group name = %s, userid = %s)",
					grp_pr->name, own_userid);
			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_TOPGROUP);	/* ヘッダ */
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
		/* src がサブグループ */
		else if (grp_pr->depth >= 2) {
			divy_rdbo_grp *top_grp_pr = NULL;

			/* トップグループの階層にしてしまうことは禁止 */
			if (divy_count_dirs(dst_u_spec->other_part) == 1) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not move this sub-group to top layler."
						"Need to administrive privilege.(group name = %s, userid = %s)",
						grp_pr->name, own_userid);
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}

			/* 自身が管理していないグループ以下への移動は禁止 (不正アクセス) */
			if (divy_rdbo_get_topgroup_property_by_grpuri(r, dst_u_spec->other_part, &top_grp_pr, NULL) ||
					top_grp_pr == NULL) {
				ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to get top-group property. (uri = %s)", dst_u_spec->other_part);
				return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
			}
			/* (note) トップグループへの移動は既に除外しているので dst はサブグループである */

			/* トップグループが管理者所有 or 他グループユーザ所有ならばNG (不正アクセス) */
			if (IS_EMPTY(top_grp_pr->ownerid) || strcmp(top_grp_pr->ownerid, own_userid) != 0) {
				ERRLOG2(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
						"You could not move this sub-group to layler of another user."
						"Need to administrive privilege.(group name = %s, userid = %s)",
						grp_pr->name, own_userid);
				return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
			}
		}
	}
	/*
	 * MKCOL に対する制限
	 */
	else if (mnum == M_MKCOL) {

		char *grpuri = dav_divy_remove_endslash(p, src_u_spec->other_part);
		divy_rdbo_grp *top_grp = NULL;

		/* トップグループのプロパティを取得 */
		if (divy_rdbo_get_topgroup_property_by_grpuri(r, grpuri, &top_grp, NULL)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get top-group information. (uri = %s)", grpuri);
			return dav_new_error(p, HTTP_INTERNAL_SERVER_ERROR, 0, 0, "");
		}

		/* トップグループの作成 -> NG */
		if (top_grp == NULL) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not create top-group by groupleader. (userid = %s)", own_userid);

			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCREATE_TOPGROUP);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}

		/* 他人がオーナとなっているトップグループの下にサブグループ作成 -> NG */
		if (IS_EMPTY(top_grp->ownerid) || strcmp(top_grp->ownerid, own_userid) != 0) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not create sub-group "
					"because top-group have been another groupleader.(userid = %s)", own_userid);
			divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCREATE_SUBGROUP);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}
	/*
	 * PROPFIND に対する制限
	 */
	else if (mnum == M_PROPFIND) {
		/* (note) SEARCH ではgrp_pr がNULLになるので、残念ながらここは通りませんでした */

		/* 自身の管理していないグループ -> NG (不正アクセス) */
		if (IS_EMPTY(grp_pr->ownerid) || strcmp(own_userid, grp_pr->ownerid) != 0) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not \"%s\" this group of another user."
					"Need to administrive privilege.(group name = %s, userid = %s)",
					divy_get_method_name(r), grp_pr->name, own_userid);
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}
	/*
	 * DELETE, PROPPATCH に対する制限
	 */
	else {
		/* 自身の管理していないグループ -> NG (不正アクセス) */
		if (IS_EMPTY(grp_pr->ownerid) || strcmp(own_userid, grp_pr->ownerid) != 0) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not \"%s\" this group of another user."
					"Need to administrive privilege.(group name = %s, userid = %s)",
					divy_get_method_name(r), grp_pr->name, own_userid);

			/* サブグループ -> NG */
			if (mnum == M_PROPPATCH && grp_pr->depth >= 2) {
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_SUBGROUP);
			}
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}

		/* トップグループ -> NG (自身の持ち物であっても駄目) */
		if (grp_pr->depth == 1) {
			ERRLOG3(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"You could not \"%s\" the top-group."
					"Need to administrive privilege.(group name = %s, userid = %s)",
					divy_get_method_name(r), grp_pr->name, own_userid);
			if (mnum == M_DELETE) {
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NODELETE_TOPGROUP);	/* ヘッダ */
			}
			else {
				divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_NOCHANGE_TOPGROUP);	/* ヘッダ */
			}
			return dav_new_error(p, HTTP_FORBIDDEN, 0, 0, "");
		}
	}

	return NULL;
}

/*
 * 指定されたIPアドレスの検証
 *
 * @return 1: 許可 / 0: 非許可
 */
static int _validate_allow_ipaddr(request_rec *r, divy_array_t* addrs)
{
	apr_ipsubnet_t *ipsub         = NULL;

	int i, len = divy_array_getlength(addrs);

	/* 一件も無い場合は許可 */
	if (len == 0) return 1;

	for (i = 0; i < len; i++) {
		ipsub = divy_array_get(addrs, i);
		if (ipsub != NULL) {
			if (apr_ipsubnet_test(ipsub, r->useragent_addr)) {
				/* 許可されている */
				return 1;
			}
		}
	}

	/*
	 * ここに来るのは許可アドレスが0個以上あったにも関わらず許可されないと
	 * 判断されたもの、それはつまりアドレスはlocalhost(127.0.0.1)ではないか？
	 */
	(void)apr_ipsubnet_create(&ipsub, "127.0.0.1", "255.255.255.255", r->pool);
	if (apr_ipsubnet_test(ipsub, r->useragent_addr)) {
		/* 無条件許可 */
		return 1;
	}

	/* ここにきたら許可されない */
	return 0;
}

/**
 * 文字列をipnetsubへ変換する
 *
 * @param r request_rec*	リクエスト構造体
 * @param str const char*	文字列（カンマ区切り可能）
 * @param addrs divy_array_t** 文字列より構築されたipsubnet構造体配列
 *
 * @return 1: 成功 / 0: 失敗
 */
static int _build_string2ipsubnet_array(request_rec *r, const char *str, divy_array_t **addrs) {

	char* ipaddrs, *token, *token_ctx, *s;
	apr_pool_t *p = r->pool;
	apr_ipsubnet_t *ipsub;
	apr_status_t rv;

	if (IS_EMPTY(str)) return 0; /* 空っぽは失敗 */

	/* コピー */
	ipaddrs = apr_pstrdup(p, str);
	while ((token = apr_strtok(ipaddrs, ",", &token_ctx)) != NULL) {
		if (token == NULL) break;
		ipaddrs = NULL;

		/* 前後のwhiteスペースを取り除く */
		token = (char*) dav_divy_trim_white(p, token);

		s = divy_strchr(token, '/');
		if (s != NULL) {
			*s++ = '\0';
		}

		/* 先頭がハイフンはスキップする */
		if (*token == '-') {
			continue;
		}

		ipsub = NULL;
		rv = apr_ipsubnet_create(&ipsub, token, s, p);
		if (ipsub == NULL) {
			return 0;
		}

		if (*addrs == NULL) {
			*addrs = divy_array_make(p, 3);
		}

		divy_array_add(*addrs, ipsub);
	}

	return 1;
}

