/**
 * $Id$
 *
 * プールにキャッシュされる値を管理するのに必要な手順やデータを定義ファイル
 *
 * このファイル導入の背景については、tf_valuecache.h を参照。
 */

#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "apr_strings.h"

#if APR_HAVE_STDARG_H
#include <stdarg.h>
#endif	/* APR_HAVE_STDARG_H */

#include "tfr.h"
#include "tf_valuecache.h"

/*------------------------------------------------------------------------------
  Fixed values and Define Macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define array
  ----------------------------------------------------------------------------*/
/**
 * divy_pcache_key と 文字列表現(const char *) との対応表
 * (note)
 * 	新しい要素の追加方法はdivy_pcache_key を参照。
 */
static const divy_pcache_key_relation _key_relation[] = {
	{ DIVY_PCACHE_DAT_GL_TABLEINFO,		"divy.dat.gl.tableinfo" 	},
	{ DIVY_PCACHE_DAT_REQ_LOCKREC,		"divy.dat.req.lockrec"		},
	{ DIVY_PCACHE_DAT_URI_USPEC,		"divy.dat.uri.uspec"		},
	{ DIVY_PCACHE_DAT_USER_GRPINFO,		"divy.dat.user.grpinfo"		},
	{ DIVY_PCACHE_FLG_KA_ACCESS,		"divy.flg.ka.access"		},
	{ DIVY_PCACHE_DAT_REQ_DBPROVIER,	"divy.dat.req.dbprovider"	},
	{ DIVY_PCACHE_FLG_GL_SEM,			"divy.flg.gl.sem"			},
	{ DIVY_PCACHE_DAT_GL_LDAPUTIL,		"divy.dat.gl.ldaputil"		},
	{ DIVY_PCACHE_DAT_REQ_LDAPCONFIG,	"divy.dat.req.ldapconfig"	},
	{ DIVY_PCACHE_DAT_REQ_LOGENV,		"divy.dat.req.logenv"		},
	{ DIVY_PCACHE_FLG_KA_SVRMAILSEND,	"divy.flg.ka.svrmailsend"	},
	{ DIVY_PCACHE_DAT_REQ_ALLOCATOR,	"divy.dat.req.allocator"	},
	{ DIVY_PCACHE_FLG_KA_LICENSEKEY,	"divy.flg.ka.licensekey"	},
#ifdef DIVY_SUPPORT_PLUGIN
	{ DIVY_PCACHE_DAT_GL_PLLOADER,		"divy.dat.gl.plloader" 		},
	{ DIVY_PCACHE_FLG_GL_PLLOADER,		"divy.flg.gl.plloader" 		},
	{ DIVY_PCACHE_DAT_GL_PLPOOL,		"divy.dat.gl.plpool" 		},
	{ DIVY_PCACHE_DAT_REQ_PLPOOL,		"divy.dat.req.plpool" 		},
#endif	/* DIVY_SUPPORT_PLUGIN */
	{ DIVY_PCACHE_DAT_GL_CMAP,			"divy.dat.gl.cmap" 			},
	{ DIVY_PCACHE_DAT_REQ_USERINFO,		"divy.dat.req.userinfo"		},
#ifdef DIVY_SUPPORT_PASSPOLICY
	{ DIVY_PCACHE_DAT_REQ_PASSPOLICY,	"divy.dat.req.passpolicy"	},
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	{ DIVY_PCACHE_FLG_USE_SAML,		"divy.flg.use.saml"				},
	{ DIVY_PCACHE_DAT_REQ_AVAILABLE_GRPINFO,	"divy.dat.req.availablegrpinfo"	},
	{ DIVY_PCACHE_DAT_REQ_SECUREDGRP,	"divy.dat.req.securedgrp"	},
	{ DIVY_PCACHE_DAT_KA_LDAPAUTH,	"divy.dat.ka.ldapauth"	},
	{ DIVY_PCACHE_FLG_GL_SHM,			"divy.flg.gl.shm"			},
	{ DIVY_PCACHE_DAT_SES_SID,			"divy.dat.ses.sid"          },
	{ DIVY_PCACHE_FLG_GL_JOBS,			"divy.flg.gl.jobs"			},
};

/*------------------------------------------------------------------------------
  Define structure and enum
  ----------------------------------------------------------------------------*/
typedef struct __divy_cache_vnode	divy_cache_vnode;

/**
 * キャッシュデータからなるtree のノードを表す構造体
 */
struct __divy_cache_vnode {

	/* キャッシュするデータ (Leafノード) */
	const void *data;

	/* divy_cache_vnode を保持するハッシュ (node) */
	apr_hash_t *node_h;
};

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static void * _vget_data(apr_pool_t *p, divy_pcache_key key1, va_list va);
static void _vset_data(apr_pool_t *p, const void *data,
					apr_status_t (*cleanup)(void *),
					divy_pcache_key key1, va_list va);
static const char * _keyid2str(divy_pcache_key key);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * プールp にkey でキャッシュされた値を取得する。
 *
 */
DIVY_DECLARE(void *) divy_pcache_get_data(apr_pool_t *p, divy_pcache_key key)
{
	return _vget_data(p, key, NULL);
}

/**
 * プールp に組合せキー(key1 + ...) でキャッシュされた値を取得する。
 *
 */
DIVY_DECLARE(void *) divy_pcache_vget_data(apr_pool_t *p,
						divy_pcache_key key1, ...)
{
	va_list va;
	void *data;

	va_start(va, key1);
	data = _vget_data(p, key1, va);
	va_end(va);

	return data;
}

/**
 * プールp にkey でdata をキャッシュする。(ポインタキャッシュ)
 *
 */
DIVY_DECLARE(void) divy_pcache_set_data(apr_pool_t *p,
					const void *data,
					divy_pcache_key key)
{
	_vset_data(p, data, NULL, key, NULL);
}

/**
 * プールp に組合せキー(key1 + ...) でdata をキャッシュする。
 * (ポインタキャッシュ)
 *
 */
DIVY_DECLARE(void) divy_pcache_vset_data(apr_pool_t *p,
					const void *data,
					divy_pcache_key key1, ...)
{
	va_list va;

	va_start(va, key1);
	_vset_data(p, data, NULL, key1, va);
	va_end(va);
}

/**
 * プールp にkey でdata をキャッシュする。(ポインタキャッシュ)
 * プールが破棄された時にdata を引数に渡してcleanup をコールする
 *
 */
DIVY_DECLARE(void) divy_pcache_fset_data(apr_pool_t *p,
					const void *data,
					apr_status_t (*cleanup)(void *),
					divy_pcache_key key)
{
	_vset_data(p, data, cleanup, key, NULL);
}

/**
 * プールp に組合せキー(key1 + ...) でdata をキャッシュする。
 * プールが破棄された時にdata を引数に渡してcleanup をコールする
 *
 */
DIVY_DECLARE(void) divy_pcache_vfset_data(apr_pool_t *p,
					const void *data,
					apr_status_t (*cleanup)(void *),
					divy_pcache_key key1, ...)
{
	va_list va;

	va_start(va, key1);
	_vset_data(p, data, cleanup, key1, va);
	va_end(va);
}

/*------------------------------------------------------------------------------
  Declare "flag cache" functions
  ----------------------------------------------------------------------------*/
/**
 * key でフラグキャッシュp からフラグ値を取得する。
 *
 */
DIVY_DECLARE(int) divy_pcache_get_flag(apr_pool_t *p, divy_pcache_key key)
{
	void *data;

	data = _vget_data(p, key, NULL);
	if (data == NULL) return 0;

	return (int) data;	/* void * で数値が入っているのでこのキャストが可能 */
}

/**
 * 組合せキー(key1 + ...) でフラグキャッシュp からフラグ値を取得する。
 *
 */
DIVY_DECLARE(int) divy_pcache_vget_flag(apr_pool_t *p,
					divy_pcache_key key1, ...)
{
	va_list va;
	void *data;

	va_start(va, key1);
	data = _vget_data(p, key1, va);
	va_end(va);

	if (data == NULL) return 0;

	return (int) data;	/* void * で数値が入っているのでこのキャストが可能 */
}

/**
 * key でフラグキャッシュpにflag値を記録する。
 *
 */
DIVY_DECLARE(void) divy_pcache_set_flag(apr_pool_t *p, int flag,
					divy_pcache_key key)
{
	_vset_data(p, (const void *)flag, NULL, key, NULL);
}

/**
 * 組合せキー(key1 + ...) でフラグキャッシュp にflag値を記録する。
 *
 */
DIVY_DECLARE(void) divy_pcache_vset_flag(apr_pool_t *p, int flag,
					divy_pcache_key key1, ...)
{
	va_list va;

	va_start(va, key1);
	_vset_data(p, (const void *)flag, NULL, key1, va);
	va_end(va);
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * プールp に組合せキー(key1 + va) でキャッシュされた値を取得する。
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param key1 divy_pcache_key 最初のkey値
 * @param va va_list 2番目以降のkey値
 * @return void * (key1 + va) でキャッシュされていた"何か"へのポインタ
 */
static void * _vget_data(apr_pool_t *p, divy_pcache_key key1, va_list va)
{
	const char *skey1;
	char *skey;
	divy_cache_vnode *vnode;

	/* key1 -> skey1 */
	skey1 = _keyid2str(key1);
	if (skey1 == NULL) return NULL;

	/* skey1 を使って値を検索 */
	(void) apr_pool_userdata_get((void **)&vnode, skey1, p);
	if (vnode == NULL) {
		/* 先頭に一致するデータがなければ以降の処理をやるだけ無駄 */
		return NULL;
	}

	/* 次の引数リストが存在しなかった場合 */
	if (va == NULL) {
		return (void *) vnode->data;	/* 以下は実施しない */
	}

	/* NULL が現れるまでループする */
	while ((skey = va_arg(va, char *)) != NULL) {
		/* skey で検索する */
		vnode = apr_hash_get(vnode->node_h, skey, APR_HASH_KEY_STRING);
		if (vnode == NULL) {
			/* これらキーの組み合わせには値がキャッシュされていない */
			break;
		}
	}

	/* 1個以上のキーがありそれらに関連付けられたvnode があった場合 */
	if (vnode != NULL) {
		return (void *) vnode->data;
	}

	return NULL;
}

/**
 * プールp に組合せキー(key1 + va) でdata をキャッシュする。
 * (ポインタキャッシュ)
 *
 * @param p apr_pool_t * キャッシュ値を保持するプールへのポインタ
 * @param data const void * keyでキャッシュする"何か"へのポインタ
 * @param cleanup クリーンアップ関数へのポインタ
 * @param key1 divy_pcache_key 最初のkey
 * @param ... const char * 2番目以降のkey値。必ずNULL終端にすること
 */
static void _vset_data(apr_pool_t *p, const void *data,
					apr_status_t (*cleanup)(void *),
					divy_pcache_key key1, va_list va)
{
	const char *skey1;
	char *skey;
	int found_newnode;
	divy_cache_vnode *vnode, *prev = NULL, *new_vnode;

	/* key1 -> skey1 */
	skey1 = _keyid2str(key1);
	if (skey1 == NULL) return;	/* キャッシュできない */

	/* skey1 を使って値をプールから取得 */
	(void) apr_pool_userdata_get((void **)&vnode, skey1, p);
	if (vnode == NULL) {
		/* 新しいノードを挿入 */
		vnode = apr_pcalloc(p, sizeof(divy_cache_vnode));
		(void) apr_pool_userdata_set(vnode, skey1, apr_pool_cleanup_null, p);
		/* 専用のcleanup ハンドラが提供されていたらそれを登録 */
		if (cleanup != NULL) {
			apr_pool_cleanup_register(p, data, cleanup, cleanup);
		}
	}

	/* 次の引数リストが存在しなかった場合 */
	if (va == NULL) {
		vnode->data = data;
		return;	/* 以下を実施する必要なし */
	}

	found_newnode = 0;
	/* NULL が現れるまでループする */
	while ((skey = va_arg(va, char *)) != NULL) {

		/* キーがあるのに下位ノードを保持するハッシュが存在しない
		 *   --> キーと値の関係を記録する下位ノードが今までなかった */
		if (!found_newnode && vnode != NULL && vnode->node_h == NULL) {
			found_newnode = 1;	/* 新しいノードだった */
		}

		/* キー値と対応するノード無かった
		 *  --> この先いくらキーが出てきてもノードは存在し得ない場合 */
		if (found_newnode) {
			/* 下位ノードを入れるハッシュを作り新しいノードを挿入 */
			vnode->node_h = apr_hash_make(p);

			new_vnode = apr_pcalloc(p, sizeof(divy_cache_vnode));
			apr_hash_set(vnode->node_h, apr_pstrdup(p, skey),
					APR_HASH_KEY_STRING, new_vnode);
			vnode = new_vnode;
		}
		/* 下位ノードを保存するハッシュが存在している
		 *  --> キーと対応するノードがあるかもしれない */
		else {
			prev = vnode;
			/* skey で検索する */
			vnode = apr_hash_get(vnode->node_h, skey, APR_HASH_KEY_STRING);
			if (vnode == NULL) {
				/* 新しいノードをprev に挿入 */
				new_vnode = apr_pcalloc(p, sizeof(divy_cache_vnode));
				apr_hash_set(prev->node_h, apr_pstrdup(p, skey),
						APR_HASH_KEY_STRING, new_vnode);
				vnode = new_vnode;	/* 対象ノードを移す */
				found_newnode = 1;	/* 新しいノードだった */
			}
		}
	}

	/* 値をノードの設定する */
	if (vnode != NULL) {
		vnode->data = data;
	}
}

/**
 * key を文字列表現に変換する。
 *
 * @param key divy_pcache_key
 * @return const char * 文字列表現。存在しないkeyであればNULLを返す
 */
static const char * _keyid2str(divy_pcache_key key)
{
	int i;

	/* key1 の整合性を検証 */
	if (key < 0 || key >= DIVY_PCACHE_END) {
		return NULL;	/* 存在しません */
	}

	/* index番号とkeyが一致していたらその値を返す */
	if (_key_relation[(int) key].key == key) {
		return _key_relation[(int) key].skey;
	}
	
	/* 一致していなければ_key_relation の中を捜す */
	for (i = 0; i < DIVY_PCACHE_END; i++) {
		if (_key_relation[i].key == key) {
			return _key_relation[i].skey;
		}
	}

	return NULL;	/* 見つからなかった(_key_relation が歯抜けになっていた) */
}


