/**
 * $Id$
 *
 * util.c
 *
 * ユーティリティ関数
 *
 * 2003/03/04 Tue takehara NEW
 */
#include "httpd.h"
#include "http_config.h"
#include "http_protocol.h"
#include "http_core.h"
#include "scoreboard.h"
#include "ap_mpm.h"

#include "apr.h"
#include "apr_strings.h"
#include "apr_lib.h"
#include "apr_time.h"
#include "apr_portable.h"
#include "apr_pools.h"
#include "apr_allocator.h"

#include "apreq2/apreq_cookie.h"

#if APR_HAVE_TIME_H
#include <time.h>
#endif	/* APR_HAVE_TIME_H */

#include "mod_dav_tf.h"
#include "util.h"
#include "util_common.h"
#include "tf_rdbo.h"
#include "liveprop.h"
#include "tf_folder.h"
#include "tf_valuecache.h"
#include "tf_linkedlist.h"
#include "tf_rdbo_util.h"
#include "tf_extmap.h"

/*--------------------------------------------------------------
  Define global value
  --------------------------------------------------------------*/
/* MPM の結果を記録するグローバル変数 */
static int _thread_limit;
static int _server_limit;
static int _is_async;
static int _threads_per_child;
static int _max_servers;

/*--------------------------------------------------------------
  Define fixed values and macro
  --------------------------------------------------------------*/
APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static divy_rdbo_usr * _get_cached_usr_property(request_rec *r);
static const char * _get_default_languagetag(request_rec *r);
static const char * _get_supported_languagetag(request_rec *r, const char *lang);

/*--------------------------------------------------------------
  Define public function
  --------------------------------------------------------------*/
DIVY_DECLARE(const char *) dav_divy_get_exclusivelock_string(void)
{
	static const char exclusivelock_str[] = CRLF
		"<"DAV_NS_PREFIX":lockentry>" CRLF
		"<"DAV_NS_PREFIX":lockscope>"
		"<"DAV_NS_PREFIX":exclusive/></"DAV_NS_PREFIX":lockscope>" CRLF
		"<"DAV_NS_PREFIX":locktype>"
		"<"DAV_NS_PREFIX":write/></"DAV_NS_PREFIX":locktype>" CRLF
		"</"DAV_NS_PREFIX":lockentry>" CRLF;
	
	return exclusivelock_str;
}

/**
 * ログインしているユーザの名前を取得する。
 */
DIVY_DECLARE(const char *) divy_get_fullname(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? usr_pr->fullname : NULL;
}

/**
 * ログインしているユーザのパスワードを取得する
 */
DIVY_DECLARE(const char *) divy_get_password(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? usr_pr->password : NULL;
}

/**
 * ログインしているユーザのメールアドレスを取得する
 */
DIVY_DECLARE(const char *) divy_get_mailaddr(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? usr_pr->mailaddr : NULL;
}

/**
 * ログインしているユーザの管理者権限を取得する。
 */
DIVY_DECLARE(divy_rdbo_adminmode) divy_get_adminmode(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? usr_pr->adminmode : DIVY_ADMINMODE_NORMAL;
}

/**
 * ログインしているユーザのIDを取得する。
 */
DIVY_DECLARE(const char *) divy_get_userid(request_rec *r)
{
	return (r->main != NULL) ? r->main->user : r->user;
}

/**
 * ログインしているユーザのResourceIDを取得する。
 */
DIVY_DECLARE(const char *) divy_get_resourceid(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? usr_pr->rsid : NULL;
}

/**
 * ログインしているユーザの物理ファイルのシーケンス番号を取得する。
 */
DIVY_DECLARE(const char *) divy_get_userseq(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? apr_psprintf(r->pool, "%d", usr_pr->usrseq) : NULL;
}

/**
 * ログインしているユーザの最終アクセス時間を取得する。
 */
DIVY_DECLARE(time_t) divy_get_lastaccess(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? usr_pr->lastaccess : (time_t) 0;
}

/**
 * ログインしているユーザが利用しているクライアントを表す文字列(User-Agent)を
 * 取得する。
 *
 */
DIVY_DECLARE(const char *) divy_get_lastaccessclient(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return  (usr_pr != NULL) ? usr_pr->lastaccesscl : NULL;
}

/**
 * ログインしているユーザのアクセス拒否フォルダの状態を取得する。
 *
 */
DIVY_DECLARE(const int *) divy_get_accessdeny(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->accessdeny : NULL;
}

/**
 * ログインしているユーザの有効期限を取得する。
 *
 */
DIVY_DECLARE(time_t) divy_get_expiration(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->expiration : 0;
}

/**
 * ログインしているユーザの権限を取得する。
 *
 */
DIVY_DECLARE(const divy_rdbo_extstatus *) divy_get_extstatus(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->extstatus : NULL;
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * ログインしているユーザのパスワード状態を取得する.
 *
 */
DIVY_DECLARE(const divy_rdbo_passpolicystatus *) divy_get_passpolicystatus(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->passpolicy_status : NULL;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * ログインしているコメントを取得する。
 *
 */
DIVY_DECLARE(const char *) divy_get_usercomment(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->comment : NULL;
}

/**
 * ログインしているユーザの最大サイズQuotaを取得する。
 *
 */
DIVY_DECLARE(apr_int64_t) divy_get_usermsquota(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->maxst : APR_INT64_C(0);
}

/**
 * ログインしているユーザの最大ファイル数Quotaを取得する。
 *
 */
DIVY_DECLARE(apr_int64_t) divy_get_usermfquota(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->maxres : APR_INT64_C(0);
}

/**
 * ログインしているユーザの使用サイズQuotaを取得する。
 *
 */
DIVY_DECLARE(apr_int64_t) divy_get_userusquota(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->usedst : APR_INT64_C(0);
}

/**
 * ログインしているユーザの使用ファイル数Quotaを取得する。
 *
 */
DIVY_DECLARE(apr_int64_t) divy_get_userufquota(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->usedres : APR_INT64_C(0);
}

/**
 * ログインしているユーザの全ユーザプロパティを取得する.
 *
 */
DIVY_DECLARE(const divy_rdbo_usr *) divy_get_usrprop(request_rec *r)
{
	return _get_cached_usr_property(r);
}

/**
 * ログインしているユーザのオーナIDを取得する.
 *
 */
DIVY_DECLARE(const char *) divy_get_userownerid(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->ownerid : NULL;
}

/**
 * ログインしているユーザが作成可能な(=管理可能な)最大ユーザ数を取得する.
 *
 */
DIVY_DECLARE(apr_int32_t) divy_get_usermaxusercreation(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->maxusercreation : 0;
}

/**
 * ログインしているユーザの許可されているアドレスを取得する
 *
 */
DIVY_DECLARE(const char *) divy_get_allowhosts(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->allowhosts : NULL;
}

/**
 * ログインしているユーザのログイン失敗回数を取得する
 *
 * @param r request_rec *
 * @return apr_int32_t ログイン失敗回数
 */
DIVY_DECLARE(apr_int32_t) divy_get_loginfailedcount(request_rec *r)
{
	divy_rdbo_usr *usr_pr = _get_cached_usr_property(r);

	return (usr_pr != NULL) ? usr_pr->loginfailedcount : 0 ;
}

DIVY_DECLARE(const char *) dav_divy_get_root_uri(request_rec *r)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	return conf->root_uri;
}

/**
 * 指定されたuriが示す文字列を、XMLのPCDATAとして指定可能な正しい形式に
 * なるよう文字列をエスケープして返却する。
 */
DIVY_DECLARE(const char *) dav_divy_escape_uri(apr_pool_t *p, const char *uri)
{
	const char *e_uri;

	/* uri をエスケープ(URIエンコード)する */
	e_uri = ap_escape_uri(p, uri);

	/* "&"が入っていないか？
	 * (note)
	 *    "&"はuriとしては適切であるがエレメントには含めては
	 *    ならないので特別扱いします。(D-IVYでは大抵入っている
	 *    ことはないでしょうが。) */
	if (divy_strchr_c((char *)e_uri, '&') == NULL)
		return e_uri;	/* なければ問題なし。*/

	return apr_xml_quote_string(p, e_uri, 0);
}

/**
 * 指定されたuriが示す文字列を URL デコードする。
 */
DIVY_DECLARE(int) dav_divy_unescape_uri(apr_pool_t *p, const char *uri,
		                                        char **decode)
{
	char *ch;

	/* "+" を半角スペースに置き換える */
	*decode = (char *) dav_divy_replace_str(p, uri, "+", " ", NULL);

	/* "%2F" を "/" に置き換える */
	*decode = (char *) dav_divy_replace_str(p, *decode, "%2F", "/", NULL);

	/* "%2f" を探す(大抵は見つからない) */
	ch = divy_strstr(*decode, "%2f");
	if (ch != NULL) {
		/* "%2f" を "/" に置き換える */
		*decode = (char *) dav_divy_replace_str(p, *decode, "%2f", "/", NULL);
	}

	/* URLデコード */
	if (ap_unescape_url(*decode) != 0) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to unescape string."
			"(orig = %s, decode = %s)", uri, *decode);
		return 1;
	}

	return 0;
}

/*
 * サーバライセンスキーを復号する
 * 復号化アルゴリズムについては、dav_divy_decipher_text を参照。
 */
DIVY_DECLARE(const char *) dav_divy_decipher_svrlicense(apr_pool_t *p, const char *ciphertext)
{
	return divy_decipher_text(p, ciphertext);
}

/**
 * サーバライセンスキーを暗号化する。
 * 暗号化アルゴリズムについては、dav_divy_encipher_text を参照。
 */
DIVY_DECLARE(const char *) dav_divy_encipher_svrlicense(apr_pool_t *p, const char *plaintext)
{
	return divy_encipher_text(p, plaintext);
}

/**
 * D-IVY サーバに登録できるユーザ数の上限値(契約上の上限値)を取得する。
 *
 */
DIVY_DECLARE(int) dav_divy_get_contract_user_count(server_rec *s)
{
	dav_divy_server_conf *conf = dav_divy_get_server_config(s);
	return conf->contract_user_cnt;
}

/**
 * dav_divy_get_max_user_count で取得されるユーザの最大値を超えて登録できる
 * ユーザ数の上限値。
 */
DIVY_DECLARE(int) dav_divy_get_allow_user_count(server_rec *s)
{
	dav_divy_server_conf *conf = dav_divy_get_server_config(s);
	return conf->allow_user_cnt;
}

/**
 * mod_dav#util.c の関数dav_xml_get_cdata の改良バージョン。
 * dav_xml_get_cdata において存在していた以下の動作を変更しています。
 *
 * (1) strip_white = 1 が動作しない件 (これはバグです！！)
 * (2) LFがCRを伴わずに現れた場合、CRLFに変換する
 */
DIVY_DECLARE(const char *) divy_xml_get_cdata(const apr_xml_elem *elem,
						apr_pool_t *pool, int strip_white)
{
	const char *tmp = NULL, *sp, *prev;
	char *buf = NULL, *first;
	apr_size_t len;

	if (elem == NULL) return NULL;	/* 何も出来ません */

	/*
	 * CDATA の中身を取得する
	 */
	tmp = dav_xml_get_cdata(elem, pool, strip_white);
	if (IS_EMPTY(tmp)) return tmp;

	/*
	 * strip_white = 1 の時のバグ対応
	 */
	if (strip_white) {
		tmp = dav_divy_trim_white(pool, tmp);
	}

	/*
	 * 単独で現れるLFをCFLFに置き換える
	 */
	len = strlen(tmp);
	first = buf = apr_pcalloc(pool, 2*len + 1);	/* 全てがLFだった場合の最大長 */
	prev = sp = tmp;
	while (1) {
		sp = divy_strchr_c((char *)sp, LF);

		if (sp == NULL) {
			/* 最後までをコピーする */
			if (*prev != '\0') {
				memcpy(buf, prev, &tmp[len] - prev);
			}
			buf += &tmp[len] - prev;
			*buf = '\0';
			break;
		}
		else if (sp == tmp || *(sp - 1) != CR) {
			/* LF の一つ前がCRでなければ置き換えを実施 */
			memcpy(buf, prev, sp - prev);
			buf += sp - prev;
			*buf     = CR;
			*(++buf) = LF;
			buf++;
		}
		else {
			memcpy(buf, prev, sp - prev + 1);
			buf += sp - prev + 1;
		}
		prev = ++sp;
	}

	
	return (const char *) first;
}

/**
 * 最終アクセス情報を更新の必要性を判断し、必要であれば更新する。
 *
 */
DIVY_DECLARE(int) dav_divy_update_lastaccess(request_rec *r)
{
	apr_pool_t *p = r->pool;
	int mnum = divy_get_method_number(r);
	int m_search = ap_method_number_of("SEARCH");
	int flag;
	time_t lastaccess = divy_get_lastaccess(r);

	/*
	 * 最終更新日付が既に更新されている場合には、
	 * 最終アクセス情報を更新する必要がないメソッドならば何もしない
	 */
	if (lastaccess > 0L &&
		(mnum == M_PROPFIND	|| mnum == M_OPTIONS || mnum == m_search)) {
		return 0;
	}

	/* pool からフラグを取得 */
	flag = divy_pcache_get_flag(p, DIVY_PCACHE_FLG_KA_ACCESS);
	if (flag) {
		return 0;	/* 更新の必要はない */
	}

	/*
	 * Autoindex, Agent経由のGET(つまり通常リソース以外のGET) では更新しない
	 * (例外) 初めてアクセスされたとき
	 */
	if (lastaccess > 0L && mnum == M_GET) {
		divy_uri_spec *u_spec = NULL;
		(void) divy_parse_uri(r->pool, dav_divy_get_root_uri(r), r->uri, &u_spec);

		/*
		 * リソース以外の更新日付を変更しない.
		 * (note) 最終更新情報が変更されたことにして、次回から更新ロジックを通さない
		 */
		if (!((u_spec->uritype == DIVY_URITYPE_REGULAR)        ||
		      u_spec->infotype == DIVY_INFOTYPE_user_trash_e0  ||
		      u_spec->infotype == DIVY_INFOTYPE_user_trash_ex  ||
		      u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
		      u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
			  u_spec->infotype == DIVY_INFOTYPE_m_update_e     ||
			  u_spec->infotype == DIVY_INFOTYPE_ticket)) {

			/* フラグをセットする */
			divy_pcache_set_flag(p, 1, DIVY_PCACHE_FLG_KA_ACCESS);
			return 0;
		}
	}

	/* 最終アクセス時刻/クライアントの更新 */
	if (divy_rdbo_set_LastAccess(r, NULL /* tf_ctx = NULL */) != 0) {
		return 1;
	}

	/* フラグをセットする */
	divy_pcache_set_flag(p, 1, DIVY_PCACHE_FLG_KA_ACCESS);

	return 0;
}

/**
 * data が示すURLパラメータデータ(POST, GETと共に提出されるデータ)を
 * 解析して、キー値と値からなるハッシュに焼きなおして返却する。
 *
 */ 
DIVY_DECLARE(int) dav_divy_parse_urlparam(apr_pool_t *p,
						const char *data,
						apr_hash_t **parsed_h)
{
	const char *val, *key;
	divy_linkedlist_t *ll = NULL, *first = NULL;

	*parsed_h = NULL;	/* 初期化 */

	if (IS_EMPTY(data)) return 0;

	while(*data != '\0' && (val = ap_getword(p, &data, '&')) != NULL) {
		key = ap_getword(p, &val, '=');
			
		ap_unescape_url((char *)key);

		/* "+" は半角スペースを意味するので、最初に置き換える */
		val = dav_divy_replace_str(p, val, "+", " ", NULL);
		ap_unescape_url((char *)val);

		if (*parsed_h == NULL) {
			*parsed_h = apr_hash_make(p);
		}

		/* key のリストを取得する */
		first = apr_hash_get(*parsed_h, key, APR_HASH_KEY_STRING);
		if (first == NULL) {
			ll = first = apr_palloc(p, sizeof(divy_linkedlist_t));
			first->next = NULL;

			/* 先頭をハッシュに格納 */
			apr_hash_set(*parsed_h, key, APR_HASH_KEY_STRING, first);
		} 
		/* キーがある場合 */
		else {
			/* 最後につなげる */
			for (ll = first; ll->next; ll = ll->next);
			ll->next = apr_palloc(p, sizeof(divy_linkedlist_t));
			ll = ll->next;
			ll->next = NULL;
		}

		/* value値のリスト作成 */
		ll->val = apr_pstrdup(p, val);
	}

	return 0;
}

/**
 * 指定されたuri の親リソースURIを算出して返却する。
 *
 */
DIVY_DECLARE(char *) divy_get_parenturi(apr_pool_t *p, const char *uri)
{
	char *parent_uri;
	apr_size_t len;

	/* 求めようもない */
	if (IS_EMPTY(uri)) return NULL;

	/* parent_uri の末尾からスラッシュを全て除去し、parent_uri の中に
	 * 含まれる2つ以上のスラッシュを1つのスラッシュに変更する */
	parent_uri = ap_make_dirstr_parent(p, dav_divy_remove_endslash(p, uri));
	if (IS_FILLED(parent_uri)) {
		ap_no2slash(parent_uri);

		len = strlen(parent_uri);
		/* 末尾に1つだけ残っているかもしれないスラッシュを除去する */
		if (len > 0 && parent_uri[len - 1] == '/') {
			parent_uri[len - 1] = '\0';
		}
	}

	return parent_uri;
}

/**
 * 指定されたuri の全ての親リソースURIを算出して返却する。
 *
 */
DIVY_DECLARE(divy_cset_t *) divy_get_parenturi_set(apr_pool_t *p, const char *uri)
{
	divy_cset_t *uri_set = NULL;
	char *ch, *s, *first;

	if (p == NULL || IS_EMPTY(uri)) return NULL;

	first = s = apr_pstrdup(p, uri);	/* src を保護する */

	/* URIを'/'で分解していく */
	while ((ch = divy_strrchr(s, '/')) != NULL) {

		/* 先頭に達してしまった */
		if (first == ch) break;
		if (uri_set == NULL) uri_set = divy_cset_make(p);

		*ch = '\0';	/* 終端を移動 */

		/* 先頭からch までの文字列をURIとして記録 */
		divy_cset_set(uri_set, apr_pstrdup(p, first));
	}

	return uri_set;
}

/**
 * 指定されたuri 自身および全親リソースURIを算出して返却する。
 *
 */
DIVY_DECLARE(divy_cset_t *) divy_get_uri_set(apr_pool_t *p, const char *uri)
{
	divy_cset_t *uri_set;

	if (p == NULL || IS_EMPTY(uri)) return NULL;

	/* 親URIの集合を算出 */
	uri_set = divy_get_parenturi_set(p, uri);
	if (uri_set == NULL) {
		uri_set = divy_cset_make(p);
	}

	/* 自身のuriを記録 */
	divy_cset_set(uri_set, apr_pstrdup(p, uri));

	return uri_set;
}

/**
 * User-Agent: ヘッダの内容を取得する。
 *
 */
DIVY_DECLARE(const char *) dav_divy_get_user_agent(request_rec *r)
{
	return apr_table_get(r->headers_in, "User-Agent");
}

/**
 * Accept-Language から言語パラメータを取得して返却する.
 *
 */
DIVY_DECLARE(divy_language_tag *) divy_get_accept_language_param(request_rec *r)
{
	apr_pool_t *p = r->pool;
	char *token, *qval, *lang, *key, *val;
	char *accept_lang, *token_cntx, *lang_cntx, *qval_cntx;
	double iqval = 0.0;
	divy_language_tag *first = NULL, *tag = NULL;

	accept_lang = apr_pstrdup(p, apr_table_get(r->headers_in, "Accept-Language"));
	if (IS_EMPTY(accept_lang)) return NULL;

	while ((token = apr_strtok(accept_lang, ",", &token_cntx)) != NULL) {
		if (token == NULL) break;
		accept_lang = NULL;

		/* 両端のwhitespace を飛ばす */
		token = (char *) dav_divy_trim_white(p, token);

		/* ";" でトークンを分ける */
		lang = apr_strtok(token, ";", &lang_cntx);
		if (IS_EMPTY(lang)) {
			/* qval がなければ優先度1 */
			iqval = 1.0;
		}
		else {
			qval = apr_strtok(NULL, ";", &lang_cntx);
			if (IS_EMPTY(qval)) {
				/* 本当はinvalid だが省略されたので最優先 */
				iqval = 1.0;
			}
			else {
				/* qval を "=" で数値に分離 */
				key = apr_strtok(qval, "=", &qval_cntx);
				if (IS_EMPTY(key) || *key != 'q') continue;

				val = apr_strtok(NULL, "=", &qval_cntx);
				if (IS_FILLED(val)) iqval = atof(val);
			}
		}

		/* リストに入れる */
		if (first == NULL) {
			first = tag = apr_pcalloc(r->pool, sizeof(divy_language_tag));
		}
		else {
			tag->next = apr_pcalloc(r->pool, sizeof(divy_language_tag));
			tag = tag->next;
		}
		tag->lang = dav_divy_trim_white(p, lang);	/* whitespace を飛ばす */
		tag->qval = iqval;
	}

	return first;
}

/**
 * Cookie: クッキーヘッダを取得する
 *
 * @param r request_rec *
 * @return const char * Cookie ヘッダの内容
 */
DIVY_DECLARE(const char*) dav_divy_get_cookie(request_rec *r)
{
	return apr_table_get(r->headers_in, "Cookie");
}

/**
 * Cookieに設定されている言語種別を取得する
 *
 * @param r request_rec *
 * @return const char *
 */
DIVY_DECLARE(const char*) dav_divy_get_cookie_lang(request_rec *r)
{
	const char* header = NULL;
	const char* lang   = NULL;
	apr_table_t* jar;

	header = dav_divy_get_cookie(r);

	if (IS_FILLED(header))
	{
		jar = apr_table_make(r->pool, APREQ_DEFAULT_NELTS);
		if (apreq_parse_cookie_header(r->pool, jar, header) != APR_SUCCESS) {
			return NULL;
		}

		lang = apr_table_get(jar, "lang");

	}

	return lang;
}

/**
 * サーバで使用する言語パラメータ名称を取得する.
 *
 */
DIVY_DECLARE(const char *) divy_get_language_param(request_rec *r)
{
	const char *lang = NULL;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	/* 強制言語を指定する言語タグが指定された場合 */
	if (IS_FILLED(sconf->force_lang)) {
		lang = _get_supported_languagetag(r, sconf->force_lang);
	}

	/* セッションをサポートしていてlangがNULLの場合 */
	if (divy_support_session(r) && lang == NULL) {
		lang = dav_divy_get_cookie_lang(r);
	}

	if (lang == NULL) {
		/* Accept-Language を取得 */
		divy_language_tag *tag = divy_get_accept_language_param(r);
		if (tag != NULL) {
			double max_qval = 0.0;
			const char *max_lang = NULL, *c;
			for (; tag != NULL; tag = tag->next) {
				c = _get_supported_languagetag(r, tag->lang);
				if (IS_FILLED(c)) {
					if (tag->qval > max_qval) {
						max_qval = tag->qval;
						max_lang = c;
					}
				}
			}
			if (IS_FILLED(max_lang)) {
				lang = max_lang;
			}
			else {
				lang = _get_default_languagetag(r);
			}
		}
		/* 強制言語も言語タグもデフォルト言語も無い場合は、デフォルト値を使用する */
		else {
			lang = _get_default_languagetag(r);
		}
	}

	/* 強制言語が無い or 不正だった場合 -> ネゴシエーション処理 */
	return lang;
}

/**
 * デフォルトの言語パラメータを取得する.
 *
 */
const char * divy_get_default_languagetag(request_rec *r)
{
	return _get_default_languagetag(r);
}

/**
 * request_rec から Content-Type の値を取得して返却する。
 *
 */
DIVY_DECLARE(const char *) divy_get_content_type(request_rec *r)
{
	const char *contenttype = r->content_type;
	char *ext = NULL, *name = NULL, *last = NULL;
	apr_hash_t *mime_map  = NULL;
	const divy_cmap_elem *map_e = NULL;

	/*
	 * 理由は不明ですが、r->content_type では殆どのケースにおいて値が
	 * "取得できません"。これは経験則です。多くのモジュールのコードでは、
	 * ここから取るようになっているため、あれば利用しますが、なければ
	 * ヘッダから直接取得します。この方法ですと、きちんと取得できます。
	 */
	if (IS_EMPTY(contenttype)) {
		contenttype = apr_table_get(r->headers_in, "Content-Type");
	}

	/*
	 * WindowsでIllrustratorをアップロードするとpostscriptファイル
	 * として登録をしてしまいます。
	 * これでは、ダウンロードで不具合を起こします。
	 * Chromeの場合はNGですが, FireFoxはOKとなるけどね
	 */
	if (IS_FILLED(contenttype)) {
		if (strncmp(contenttype, "application/postscript", 22) == 0) {
			name = dav_divy_extract_finalpath_segment(r->pool, r->uri);
			if (IS_EMPTY(name)) return NULL;

			ext = divy_strchr(name, '.');
			if (ext == NULL) return NULL;

			if (strcasecmp(ext, ".ai") == 0) {
				contenttype = "application/Illustrator"; /* 特別待遇 */
			}
		}

		if (strncmp(contenttype, "text/xml; charset=\"utf-8\"", 25) == 0) {
			name = dav_divy_extract_finalpath_segment(r->pool, r->uri);
			if (IS_EMPTY(name)) return NULL;

			last = divy_strrchr(name, '.');
			ext = (last != NULL) ? ++last : NULL;
			if (ext == NULL) return NULL;

			/* PDFはOfficeからのPDF Export機能 */
			if ( (strncmp(ext, "xls", 3) != 0) &&
				 (strncmp(ext, "xlsx", 4) != 0) &&
				 (strncmp(ext, "doc", 3) != 0) &&
				 (strncmp(ext, "docx", 4) != 0) &&
				 (strncmp(ext, "ppt", 3) != 0) &&
				 (strncmp(ext, "pptx", 4) != 0) &&
				 (strncmp(ext, "pdf", 3) != 0)) {

				return apr_pstrdup(r->pool, contenttype);
			}

			/* 拡張子、Content-Typeマップを読み込む */
			mime_map = divy_read_mimetype_map(r);

			/* 拡張子からマッピング情報を取得する */
			map_e = divy_get_extmapinfo(r->pool, mime_map, NULL, ext);
			if (map_e != NULL && IS_FILLED(map_e->ctype)) {
				return apr_pstrdup(r->pool, map_e->ctype);
			}
		}
	}

	return apr_pstrdup(r->pool, contenttype);
}

/**
 * request_rec から Content-Language の値を取得して返却する。
 *
 */
DIVY_DECLARE(const char *) divy_get_content_language(request_rec *r)
{
	apr_array_header_t *clang = r->content_languages;

	if (clang) {
		return apr_pstrdup(r->pool, clang->elts);
	}
	else {
		return NULL;
	}
}

/**
 * request_rec から Content-Length の値を取得して返却する。
 *
 */
DIVY_DECLARE(apr_int64_t) divy_get_content_length(request_rec *r)
{
	const char *slength = apr_table_get(r->headers_in, "Content-Length");

	if (slength) {
		return apr_atoi64(slength);
	}
	else {
		return 0;
	}
}

/**
 * request_rec から X-Expected-Entity-Length の値を取得して返却する。
 *
 */
DIVY_DECLARE(apr_int64_t) divy_get_x_expected_entity_length(request_rec *r)
{
	const char *slength = apr_table_get(r->headers_in, "X-Expected-Entity-Length");

	if (slength) {
		return apr_atoi64(slength);
	}
	else {
		return 0;
	}
}


/**
 * 指定されたuri を "http://server:port/uri" 形式のURLにして返却する。
 *
 */
DIVY_DECLARE(char *) divy_construct_url(apr_pool_t *p, const char *host,
					const char *uri, request_rec *r)
{
	const char *schema;
	char *url;
	unsigned int port = ap_get_server_port(r);

	/* host が設定されていなければデフォルト値を補う */
	if (IS_EMPTY(host)) {
		host = ap_get_server_name(r);
	}

	if (ap_is_default_port(port, r)) {
		/* デフォルトポート番号であれば、別ポート番号の可能性を探る */
		if (r->connection->local_addr == NULL ||
		    r->connection->local_addr->port == 0) {
			/* デフォルトポートを使用 */
			return apr_pstrcat(p, ap_http_scheme(r), "://", host, uri, NULL);
		}
		else {
			/* ポート番号の取得 */
			port = r->connection->local_addr->port;
		}
	}
	schema = ap_http_scheme(r);

	/* デフォルトポート番号はつけない */
	if (strcasecmp(schema, "https") == 0) {
		if (port == 443) {
			url = apr_psprintf(p, "https://%s%s", host, uri);
		}
		else {
			url = apr_psprintf(p, "https://%s:%u%s", host, port, uri);
		}
	}
	else {
		if (port == 80) {
			url = apr_psprintf(p, "http://%s%s", host, uri);
		}
		else {
			url = apr_psprintf(p, "http://%s:%u%s", host, port, uri);
		}
	}

	return url;
}

/**
 * 指定されたuri を指定された種類type に応じて適切なフルURL (schema://host:port/uri)を
 * 組み立てて返却する.
 * (note)
 *   この関数はリバースプロキシに対応する目的で導入されたdivy_construct_url の改良版です.
 *   今後は、こちらを利用して下さい.
 *
 */
DIVY_DECLARE(char *) divy_construct_url2(apr_pool_t *p, int urltype,
					const char *uri, request_rec *r, divy_buildurl_segment *url_segment)
{
	const char *sHost = NULL;
	char *sSchema = NULL;
	char *url;
	int port;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	/*
	 * まずはリクエストベースのホスト名、ポート番号、スキーマを取得しておく
	 */
	sHost = ap_get_server_name(r);

	(void) divy_get_request_conninfo(r, &sSchema, &port);
	if (IS_EMPTY(sSchema)) sSchema = "http";

	if (urltype == DIVY_URLTYPE_PUBLIC || urltype == DIVY_URLTYPE_PROTECTED) {
		divy_httpheader_via *via_list = divy_parse_via_header(r);

		/* Via ヘッダの解析 */
		if (dconf->rproxyuse == DIVY_RPROXYUSE_ON && IS_FILLED(dconf->rproxymatch) && via_list != NULL) {
			ap_regex_t *preg;

			/* Viaヘッダの末尾にある情報がリバースプロキシの候補を探す */
			for (; via_list->next != NULL; via_list = via_list->next);

			ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "regular expression: %s", dconf->rproxymatch);

			/* リバースプロキシとして認識しても良いかどうか判断する */
			preg = ap_pregcomp(p, dconf->rproxymatch, 0);
			if (preg == NULL) {
				ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
						"Failed to compile reverse-proxy matching pattern.");
				return "";
			}

			/* リバースプロキシとして認識した場合 */
			if (!ap_regexec(preg, via_list->sVia, 0, NULL, 0)) {
				int iPort;
				sHost = via_list->sHost;	/* ホスト名はそのまま使う */

				switch (dconf->rproxyschema) {
					case DIVY_RPROXY_SCHEMA_UNSET:
					case DIVY_RPROXY_SCHEMA_AUTO:
						/* 自動認識モードの場合には、スキーマ名とポート番号を推測する */

						iPort = (IS_FILLED(via_list->sPort)) ? atoi(via_list->sPort) : 80;	/* RFC2616より */
						if (dconf->rproxyhttpport == iPort) {
							sSchema = "http";
							port = iPort;
						}
						else if (dconf->rproxyhttpsport == iPort) {
							sSchema = "https";
							port = iPort;
						}
						else {
							/* 予期しないポート番号 */
							ERRLOG1(p, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
									"Invalid port number found in via header. (port: %d)", iPort);
							ap_pregfree(p, preg);
							return "";
						}
						ERRLOG2(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Matched schema: %s, Matched port: %u", sSchema, port);

						break;

					case DIVY_RPROXY_SCHEMA_HTTP:
						sSchema = "http";
						port = dconf->rproxyhttpport;
						break;

					case DIVY_RPROXY_SCHEMA_HTTPS:
						sSchema = "https";
						port = dconf->rproxyhttpsport;
						break;
				}
			}
			ap_pregfree(p, preg);
		}
		else {
			/* リバースプロキシ対応を行わない or Viaが存在しなかった場合
			 * -> リクエストベースの情報を使うので、ここでは何もしない */
		}

		/* 公開ホスト名、公開ポート番号、公開スキーマ名があればそれに置き換える */
		if (urltype == DIVY_URLTYPE_PUBLIC) {
			/* 公開ホスト名が見つかった */
			if (IS_FILLED(dconf->notifyservername)) {
				sHost = dconf->notifyservername;
			}

			/* 公開ポート番号が見つかった(auto, 未設定ならば除外) */
			if (dconf->notifyserverport != DIVY_NOTIFYSERVER_PORT_AUTO &&
				dconf->notifyserverport != DIVY_INT32_C_UNSET) {
				port = dconf->notifyserverport;
				ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Found TfNotityServerPort: %u", port);
			}

			/* 公開スキーマが見つかった */
			if (dconf->notifyserverschema == DIVY_NOTIFYSERVER_SCHEMA_HTTP) {
				sSchema = "http";
				ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Found TfNotifyServerSchema: %s", sSchema);
			}
			else if (dconf->notifyserverschema == DIVY_NOTIFYSERVER_SCHEMA_HTTPS) {
				sSchema = "https";
				ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Found TfNotifyServerSchema: %s", sSchema);
			}
			/* 上記以外のnotifyserverschema ならばauto */
		}
	}
	else if (urltype == DIVY_URLTYPE_INNER) {
		/* 特にすることはありません (もう算出済み) */
	}
	else {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Invalid urltype found (urltype: %d)", urltype);
		return "";
	}

	/*
	 * URL文字列の組み立て
	 */
	if (port == 80 || port == 443) {
		/* (note) デフォルトポート番号はつけない */
		url = apr_psprintf(p, "%s://%s%s", sSchema, sHost, uri);
	}
	else {
		url = apr_psprintf(p, "%s://%s:%u%s", sSchema, sHost, port, uri);
	}
	ERRLOG2(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Build URL(%d): %s", urltype, url);

	if (url_segment != NULL) {
		url_segment->schema = apr_pstrdup(p, sSchema);
		url_segment->host   = apr_pstrdup(p, sHost);
		url_segment->port   = port;
		url_segment->uri    = apr_pstrdup(p, uri);
	}

	return url;
}

/**
 * 専用のメモリアロケータを持つプール(r->pool のサブプール) を返却する。
 * 既に作成済みであればそれを返します。そうでなければ生成します。
 *
 */
DIVY_DECLARE(apr_pool_t *) divy_get_request_temporary_pool(request_rec *r)
{
	apr_status_t rv;
	apr_allocator_t *allocator = NULL;
	apr_pool_t *wp             = NULL;

	/* 既にキャッシュされているか? */
	wp = divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_REQ_ALLOCATOR);
	if (wp != NULL) {
		return wp;	/* キャッシュ済みのプールを返す */
	}

	/* 専用のアロケータ付きプールを生成する */
	rv = apr_allocator_create(&allocator);
	if (rv != APR_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to make new allocator.(code = %d)", rv);
		return NULL;	/* アロケータの取得に失敗した */
	}
	apr_allocator_max_free_set(allocator, APR_ALLOCATOR_MAX_FREE_UNLIMITED);

	/* r->pool のサブプールを生成する */
	rv = apr_pool_create_ex(&wp, r->pool, NULL, allocator);
	if (rv != APR_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to create subpool.(code = %d)", rv);
		return NULL;	/* サブプールの生成に失敗した */
	}

	/* アロケータのオーナをwp にする */
	apr_allocator_owner_set(allocator, wp);

	/* wp をr->pool にキャッシュする */
	divy_pcache_set_data(r->pool, wp, DIVY_PCACHE_DAT_REQ_ALLOCATOR);

	return wp;
}

/**
 * リクエストのメソッド番号を取得する。(プロトコルゲートウェイ対応)
 */
DIVY_DECLARE(int) divy_get_method_number(request_rec *r)
{
	/* プロトコルゲートウェイが実装されたら以下は変更します */
	return r->method_number;
}

/**
 * リクエストのメソッドの文字列を取得する. (プロトコルゲートウェイ対応)
 */
DIVY_DECLARE(char *) divy_get_method_name(request_rec *r)
{
	return apr_pstrdup(r->pool, r->method);
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * r->pool にキャッシュされたパスワードポリシー情報を取得する.
 * キャッシュされていなければDB から取得します.
 *
 */
DIVY_DECLARE(int) divy_get_cached_passwordpolicy(request_rec *r, int policyid,
		int use_cache, divy_rdbo_passwordpolicy **passpolicy_pr)
{
	*passpolicy_pr = NULL;

	/* パスワードポリシー機能がサポートされていなければNULL */
	if (!divy_support_passpolicy(r)) {
		return 0;
	}

	/* キャッシュされているか? */
	*passpolicy_pr = divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_REQ_PASSPOLICY);
	if (*passpolicy_pr == NULL && !use_cache) {
		/* パスワードポリシープロパティを取得する */
		if (divy_rdbo_get_passwordpolicy_property(r, policyid, passpolicy_pr, NULL)) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to get passwordpolicy property.");
			return 1;
		}
		/* キャッシュする */
		divy_pcache_set_data(r->pool, *passpolicy_pr, DIVY_PCACHE_DAT_REQ_PASSPOLICY);
	}

	return 0;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * Via リクエストヘッダをパースして返却する.
 *
 */
DIVY_DECLARE(divy_httpheader_via *) divy_parse_via_header(request_rec *r)
{
	apr_pool_t *p = r->pool;
	char *via_all, *via_header;
	char *token, *token_cntx, *via_cntx, *hostinfo_cntx, *hostinfo;
	char *sHttpVersion, *sHost, *sPort, *sComment, *sVia;
	divy_httpheader_via *via_list = NULL, *first = NULL;

	/* Via ヘッダを取得 */
	via_all = via_header = apr_pstrdup(p, apr_table_get(r->headers_in, "Via"));
	if (IS_EMPTY(via_all)) {
		return NULL;
	}
	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Via: %s", via_all);

	while ((token = apr_strtok(via_header, ",", &token_cntx)) != NULL) {
		if (token == NULL) break;
		via_header = NULL;

		/* 両端のwhitespace を飛ばす */
		token = (char *) dav_divy_trim_white(p, token);
		sVia = apr_pstrdup(p, token);	/* 書き換えられてしまうのでコピーしておく */
		ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Via(part): %s", sVia);

		/* 初期化 */
		sHttpVersion = NULL;
		sHost = NULL;
		sPort = NULL;
		sComment = NULL;

		/*
		 * HTTP version の取得
		 */

		sHttpVersion = apr_strtok(token, " ", &via_cntx);	/* " " でトークンを分ける */
		if (IS_EMPTY(sHttpVersion)) {
			/* HTTP version が存在しないのはRFC的に不正なViaヘッダである */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_SYNTAX,
					"The http-version string was missing in Via header.(Via: %s)", sVia);
			continue;
		}

		/*
		 * ホスト名とポート番号の取得
		 */
		hostinfo = apr_strtok(NULL, " ", &via_cntx);	/* ホスト情報を取得 */
		if (IS_FILLED(hostinfo)) {
			/* hostinfo を":"で分離 */
			sHost = apr_strtok(hostinfo, ":", &hostinfo_cntx);
			sPort = apr_strtok(NULL, ":", &hostinfo_cntx);	/* (note) ポート番号はないこともある */

			/* コメントの取得 (ホスト情報がなければコメントは無いし、識別できません) */
			sComment = apr_strtok(NULL, " ", &via_cntx);	/* (note) comment も無いこともある */
		}

		/* リストに入れる */
		if (first == NULL) {
			first = via_list = apr_pcalloc(r->pool, sizeof(divy_httpheader_via));
		}
		else {
			via_list->next = apr_pcalloc(r->pool, sizeof(divy_httpheader_via));
			via_list = via_list->next;
		}
		via_list->sHttpVersion = sHttpVersion;
		via_list->sHost = sHost;
		via_list->sPort = sPort;
		via_list->sComment = sComment;
		via_list->sVia = sVia;
		via_list->next = NULL;
		ERRLOG3(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "Via(host: %s, port: %s, comment: %s)", sHost, sPort, sComment);
	}

	return first;
}

/**
 * クライアントがリクエストしてきたポート番号とスキーマを調べて返却する.
 *
 */
DIVY_DECLARE(int) divy_get_request_conninfo(request_rec *r, char **schema, int *port)
{
	apr_pool_t *p = r->pool;
	unsigned int iPort;

	*schema = NULL;	/* 初期化 */
	*port   = -1;	/* 初期化 */

	/* スキーマの取得 */
	*schema = apr_pstrdup(p, ap_http_scheme(r));

	/* ポート番号の取得 */
	iPort = ap_get_server_port(r);
	if (ap_is_default_port(iPort, r)) {
		/* デフォルトポート番号であれば、別ポート番号の可能性を探る */
		if (r->connection->local_addr == NULL ||
				r->connection->local_addr->port == 0) {
			iPort = 80;	/* デフォルトポートを使用(内部用です) */
		}
		else {
			/* ポート番号の取得 */
			iPort = r->connection->local_addr->port;
		}
	}
	*port = iPort;
	return 0;
}

/**
 * r->pool にキャッシュされたアクセスユーザの所属グループ情報を取得する.
 * キャッシュは関数divy_rdbo_cache_userinfo() の中で行われます.
 *
 */
DIVY_DECLARE(int) divy_get_cached_availablegroup(request_rec *r, apr_hash_t **grp_h)
{
	apr_pool_t *cache_p = NULL;
	const char *userid = divy_get_userid(r);

	/* データはメインのrequest_rec に入っている */
	cache_p = (r->main != NULL) ? r->main->pool : r->pool;

	/* キャッシュ値の取得(無ければ所属グループなし) */
	*grp_h =  divy_pcache_vget_data(cache_p, DIVY_PCACHE_DAT_REQ_AVAILABLE_GRPINFO, userid, NULL);

	return 0;
}

/**
 * r->pool にキャッシュされたアクセスユーザの所属グループ情報の中から
 * uri のリソースが所属するグループのグループ拡張ステータスを取得する.
 * キャッシュは関数divy_rdbo_cache_userinfo() の中で行われます.
 *
 * @param r request_rec *
 * @param uri const char * 対象リソースのURI (グループコレクションに調整しなくてもいい)
 * @param grp_extstatus divy_rdbo_extstatus ** 取得したグループ拡張ステータス
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_get_cached_availablegroupextstatus(request_rec *r,
		const char *uri, divy_rdbo_extstatus **grp_extstatus)
{
	divy_rdbo_grp *grp_pr = NULL;

	*grp_extstatus = NULL;	/* 初期化 */

	divy_get_cached_availablegroup_by_uri(r, uri, &grp_pr);
	if (grp_pr != NULL) {
		*grp_extstatus = grp_pr->grp_extstatus;
	}
	return 0;
}

/**
 * r->pool にキャッシュされたアクセスユーザの所属グループ情報の中から
 * uri のリソースが所属するグループ情報を取得する.
 * キャッシュは関数divy_rdbo_cache_userinfo() の中で行われます.
 *
 */
DIVY_DECLARE(int) divy_get_cached_availablegroup_by_uri(request_rec *r,
									const char *uri, divy_rdbo_grp **grp_pr)
{
	apr_pool_t *p        = r->pool;
	apr_hash_t *grp_pr_h = NULL;
	char *grpcol_uri     = NULL;

	*grp_pr = NULL;	/* 初期化 */

	/* グループコレクションURIを切り出す */
	(void) divy_extract_groupcol_uri(p, dav_divy_get_root_uri(r), uri, &grpcol_uri);
	if (IS_EMPTY(grpcol_uri)) return 0;

	/* グループ情報のハッシュを取得する */
	(void) divy_get_cached_availablegroup(r, &grp_pr_h);
	if (grp_pr_h != NULL) {
		*grp_pr = apr_hash_get(grp_pr_h, grpcol_uri, APR_HASH_KEY_STRING);
	}

	return 0;
}

/**
 * 公開チケット用絶対アドレスURL文字列を生成する.
 *
 * 短縮URLをサポートしている場合
 * $schema://$root/.st/JnDmxDnes
 * 短縮URLをサポートしていない場合
 * $schema://$root/.ticket?u=xxx&WEBDAV_METHOD=PROPFIND
 *
 * @param r request_rec *
 * @param p apr_pool_t *
 * @param rdb_r djj/apivy_rdbo_resource * リソース情報
 * @return char * 生成された監視フォルダアクセスURL文字列
 * 	uri がNULLであれば空文字を返す
 */
DIVY_DECLARE(char *) divy_construct_oneclick_url(request_rec *r, apr_pool_t *p, divy_rdbo_resource *rdb_r)
{
	char *ticket, *uri2;
	char *rsid;

	if (rdb_r == NULL) return "";

	if (divy_support_shorten(r)) {
		/* 短縮URLをサポートしている */

		if (IS_EMPTY(rdb_r->rsid)) return "";

		/* rsidから短縮文字列を生成する */
		rsid = divy_get_rid2shorten(p, rdb_r->rsid, NULL);

		/* 短縮URIの文字列を作る */
		uri2 = apr_psprintf(p, "%s/%s",
				divy_build_shorten_uri(p, dav_divy_get_root_uri(r)), rsid);

		/* スキーマ名付きのURLを作る */
		return divy_construct_url2(p, DIVY_URLTYPE_PUBLIC, uri2, r, NULL);
	}
	else {
		/* 従来のチケット */
		if (IS_EMPTY(rdb_r->uri)) return "";

		/* URLからチケットを生成する */
		ticket = divy_create_ticket_str(p, rdb_r->uri);

		/* uri の生成(チケットとURLエンコードする) */
		uri2 = divy_build_ticket_url(p, dav_divy_get_root_uri(r), dav_divy_escape_uri(p, ticket));

		/* WebDAV メソッドGW用の識別子(飾り)をつける */
		uri2 = apr_psprintf(p, "%s&WEBDAV_METHOD=PROPFIND", uri2);

		/* スキーマ名付きのURLを作る */
		return divy_construct_url2(p, DIVY_URLTYPE_PUBLIC, uri2, r, NULL);
	}
}

/**
 * オーナownerid が作ったユーザは、アクセスユーザ(r->user) から見ると「Otherユーザ」かどうか.
 *
 */
DIVY_DECLARE(int) divy_is_otheruser(request_rec *r, const char *ownerid)
{
	const divy_rdbo_extstatus *extstatus = divy_get_extstatus(r);
	const char *own_userid = divy_get_userid(r);

	if (!divy_support_groupleader(r)) {
		return 0;	/* 未サポートならばOther ユーザにはならない */
	}

	/* 管理者の場合 */
	if (divy_get_adminmode(r) == DIVY_ADMINMODE_ADMIN) {
		/* オーナが居ない = 管理者所有のユーザ */
		if (IS_EMPTY(ownerid)) {
			return 0;	/* NOT Otherユーザ */
		}
		else {
			/* ownerid に値がある = 別のグループリーダが作ったユーザ */
			return 1;	/* Other ユーザ */
		}
	}
	/* グループリーダの場合 */
	else if (divy_rdbo_is_groupleader(extstatus)) {
		/* オーナが居ない = 管理者所有のユーザ */
		if (IS_EMPTY(ownerid)) {
			return 1;	/* Other ユーザ */
		}
		else {
			/* 自身が所有していたユーザ */
			if (strcmp(ownerid, own_userid) == 0) {
				return 0;	/* NOT Other ユーザ */
			}
			else {
				return 1;	/* Other ユーザ */
			}
		}
	}

	return 0;	/* NOT Other ユーザ */
}

/**
 * リクエストされているURLのFinalPathがリソースフォークファイルかを調べる
 * 厳密なリソースフォークの評価は行わない単純にURIとエージェントを確認する
 * だけです。
 * リソースフォークの条件
 * FinalPathセグメントの先頭二文字が == "._"
 * user agentがWebDAVFS/で始まる
 * 
 * @param r request_rec *
 * @return int リソースフォーク(1) / それ以外(0)
 *
 */
DIVY_DECLARE(int) divy_is_resourcefork(request_rec *r)
{
	char *fps = NULL;
	const char *ua = NULL;

	/* WebDAVFS(MacOS/XのFinder)でなければ終了 */
	ua = dav_divy_get_user_agent(r);
	if (!ua || strncmp(ua, "WebDAVFS", 8) != 0) {
		return 0; /* WebDAVFS以外は調べない */
	}

	fps = dav_divy_extract_finalpath_segment(r->pool, r->uri);
	if (!fps || strlen(fps) < 2) return 0;
	
	/* 先頭が._であればリソースフォークのファイルとみなす */
	if (*fps == '.' && *(fps+1) == '_') return 1;

	return 0; /* それ以外はすべて違う */
}

/**
 * ユーザエージェントがOffice系なのかを調べます
 *
 * @param r request_rec *
 * @return int Office(1) / それ以外(0) 
 */
DIVY_DECLARE(int) divy_is_Office(request_rec *r)
{
	const char *ua = NULL;

	ua = dav_divy_get_user_agent(r);
	if (!ua || strncmp(ua, "Microsoft Office", 16) == 0) {
		return 1;
	}

	return 0; /* Officeではない */

}

/**
 * ブラウザのパスワード変更表示をなくすか？
 *
 * @param request_rec *
 * @param int 表示しないかどうか (1: 表示しない / 0: 表示する)
 */
DIVY_DECLARE(int) divy_disable_change_password(request_rec *r)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char * header = NULL;
	const char* authtype = NULL;
	apr_table_t* jar;

	header = dav_divy_get_cookie(r);
	if (IS_FILLED(header)) {

		jar = apr_table_make(r->pool, APREQ_DEFAULT_NELTS);
		if (apreq_parse_cookie_header(r->pool, jar, header) != APR_SUCCESS) {
			/* TODO　エラーメッセージを */
			return 1;
		}

		authtype = apr_table_get(jar, "authtype");
	}

	if (dconf->disablechangepassword == DIVY_DISABLE_CHANGE_PASSWORD_ON) {
		return 1;
	}

	/* SAMLで認証を受けたら表示しない */
	if (IS_FILLED(authtype) && strcmp(authtype, "saml") == 0) {
		return 1;
	}

	return 0;

}

/**
 * 同じキーが経過時間にあったかどうかを調べます。存在しない場合は、記録します。
 *
 * @param r	request_rec *
 * @param key const char*
 * @param value const char*
 * @param timeout apr_uint32_t
 *
 * @return int (1: 存在した / 0: しなかった)
 */
DIVY_DECLARE(int) divy_has_action_in_time(request_rec *r, const char* key, const char* value, apr_uint32_t timeout)
{
	apr_status_t st = APR_SUCCESS;
	apr_pool_t *p = r->pool;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	char *prefix = apr_pstrdup(p, dconf->root_uri);
	char *retvalue = NULL;
	apr_size_t length = 0;
	apr_uint16_t flags = 0;
	int ret = 0;

	st = divy_memcache_get(p, dconf->memd, prefix, key,
											&retvalue, &length, &flags);

	if (st == APR_SUCCESS) {
		/* 同じアクションがあった */
		if (length > 0)  return 1;
	}

	ret = divy_memcache_set(p, dconf->memd, prefix, key,
											(char*)value, strlen(value), timeout, 0);
	if (ret) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"failed to set memcached data. Please checked to memcached process.");

	}

	/* 同じアクションは見つからなかった */
	return 0;
}

/**
 * カスタムのイメージパスを取得する
 *
 * @param request_rec *
 * @param char * fname	ファイル名
 *
 * @return char *		イメージのフルパス	
 */
DIVY_DECLARE(char *) divy_get_custom_image_path(request_rec *r, const char* fname)
{
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	apr_finfo_t finfo = { 0 };
	const char *location = dav_divy_get_root_uri(r);
	const char *hostname = apr_pstrdup(r->pool, r->server->server_hostname);
	const char *path = NULL;
	apr_status_t rv;

	if (IS_FILLED(fname) && sconf->tfcustomimagepath && location && hostname) {
		/* (note)
		 * 変数locationはスラッシュ付きの文字列の為にフォーマット内に``/``
		 * を入れていません。
		 */
		path = apr_psprintf(r->pool, "%s/%s%s/%s",
						sconf->tfcustomimagepath, hostname, location, fname);

		rv = apr_stat(&finfo, path, APR_FINFO_MTIME, r->pool);
		if (rv == APR_SUCCESS) {
			/* httpでアクセスするパスへ変更する */
			return apr_psprintf(r->pool, "/icons/teamfile/custom/%s%s/%s",
													hostname, location, fname);
		}
		else {
			ERRLOG2(r->pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
					"faild to custom top image (%s). result = %d", path, rv);
		}

	}

	return NULL;

}

/**
 * MPMのさまざまな情報を所得して設定する
 *
 * @param none
 * @return none
 */
DIVY_DECLARE(void) dav_divy_query_mpm_info(void)
{
	ap_mpm_query(AP_MPMQ_HARD_LIMIT_THREADS, &_thread_limit);
	ap_mpm_query(AP_MPMQ_HARD_LIMIT_DAEMONS, &_server_limit);
	ap_mpm_query(AP_MPMQ_IS_ASYNC, &_is_async);
	ap_mpm_query(AP_MPMQ_MAX_THREADS, &_threads_per_child);
	ap_mpm_query(AP_MPMQ_MAX_DAEMONS, &_max_servers);
}

/**
 * MPMの取得済みの情報を入手する
 *
 * @param mpmq divy_mpmq (see util/include/util_thread.h)
 * @return int 結果
 */
DIVY_DECLARE(int) dav_divy_get_mpm_info(divy_mpmq mpmq)
{
	switch(mpmq) {

		case DIVY_MPMQ_HARD_LIMIT_THREADS:
			return _thread_limit;

		case DIVY_MPMQ_HARD_LIMIT_DAEMONS:
			return _server_limit;

		case DIVY_MPMQ_IS_ASYNC:
			return _is_async;

		case DIVY_MPMQ_MAX_THREADS:
			return _threads_per_child;

		case DIVY_MPMQ_MAX_DAEMONS:
			return _max_servers;

		default:
			return DIVY_MPMQ_UNKNOWN;

	}

	return 0;
}
/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
static divy_rdbo_usr * _get_cached_usr_property(request_rec *r)
{
	apr_pool_t *cache_p = NULL;

	if (r == NULL) return NULL;

	/* データはメインのrequest_rec に入っている */
	cache_p = (r->main != NULL) ? r->main->pool : r->pool;

	return divy_pcache_get_data(cache_p, DIVY_PCACHE_DAT_REQ_USERINFO);
}

/**
 *
 */
static const char * _get_default_languagetag(request_rec *r)
{
	const char *lang = NULL;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	if (IS_FILLED(sconf->default_lang)) {
		lang = _get_supported_languagetag(r, sconf->default_lang);
		if (IS_EMPTY(lang)) {
			lang = DIVY_DEFAULT_LANGPARAM;
		}
	}
	else {
		lang = DIVY_DEFAULT_LANGPARAM;
	}

	return lang;
}

/**
 *
 */
static const char * _get_supported_languagetag(request_rec *r, const char *lang)
{
	const char *ret = NULL;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	if (IS_EMPTY(lang) || sconf->supported_languages == NULL ||
		divy_cset_count(sconf->supported_languages) == 0) return NULL;

	if (divy_cset_contains(sconf->supported_languages, lang)) {
		char *str;
		ret = apr_pstrdup(r->pool, lang);
		if ((str = divy_strchr((char *)ret, '-')) != NULL) {
			/* 国別コードは削って返す */
			*str = '\0';
		}
	}

	return ret;
}

