/**
 * $Id$
 *
 * Array (可変長配列コンテナ) の宣言・定義を行うヘッダファイル
 * (note)
 * 	この宣言は元々util_common.h にあったものです。
 * 	ぐちゃぐちゃになったので移しました。
 */
#ifndef INCLUDE_TF_ARRAY_H
#define INCLUDE_TF_ARRAY_H

#include "apr.h"
#include "apr_pools.h"

#include "tfr.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_array_t	divy_array_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 指定された初期要素数size 個の要素をもつポインタ保持用可変長配列を生成する。
 * (note)
 * 	size は初期サイズです。要素数が足りなくなれば自動的に追加されます。
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t divy_array_t が格納されるプール.以降この可変長配列が
 * 			破棄されるまでこのプールが使用されます。
 * @param size apr_int32_t 初期要素数
 * @return divy_array_t * 可変長配列へのポインタ
 */
DIVY_DECLARE(divy_array_t *) divy_array_make(apr_pool_t *p, apr_int32_t size);

/**
 * 可変長配列array の末尾にelement が示すオブジェクトを追加する。
 * (Reentrant)
 * 	各スレッドが別々のarrayをもつ限り、この関数はスレッドセーフです。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 * @param element void * 追加するオブジェクトへのポインタ
 */
DIVY_DECLARE(void) divy_array_add(divy_array_t *array, void *element);

/**
 * 可変長配列array の位置index のオブジェクトをelement で置き換える。
 * index が範囲外であれば何もしません。
 * (Reentrant)
 * 	各スレッドが別々のarrayをもつ限り、この関数はスレッドセーフです。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 * @param index apr_int32_t 追加位置
 * @param element void * 追加するオブジェクトへのポインタ
 */
DIVY_DECLARE(void) divy_array_set(divy_array_t *array,
					apr_int32_t index, void *element);

/**
 * 可変長配列array の位置index にあるオブジェクトを削除し、返却する。
 * (Reentrant)
 * 	各スレッドが別々のarrayをもつ限り、この関数はスレッドセーフです。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 * @param index apr_int32_t 削除位置
 * @return void * 削除位置に格納されていたオブジェクトへのポインタ
 */
DIVY_DECLARE(void *) divy_array_remove(divy_array_t *array, apr_int32_t index);

/**
 * 可変長配列array の総要素数を取得する。
 * (Reentrant)
 * 	各スレッドが別々のarrayをもつ限り、この関数はスレッドセーフです。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 * @return apr_int32_t 総要素数
 */
DIVY_DECLARE(apr_int32_t) divy_array_getlength(divy_array_t *array);

/**
 * 可変長配列array の中の位置index にある要素を取得する。
 * remove と異なり、index 位置の要素は除去されません。
 * (Reentrant)
 * 	各スレッドが別々のarrayをもつ限り、この関数はスレッドセーフです。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 * @param index apr_int32_t 取得位置(0から始まります)
 * @return void * indexに格納されていたオブジェクトへのポインタ
 */
DIVY_DECLARE(void *) divy_array_get(divy_array_t *array, apr_int32_t index);

/**
 * 可変長配列array の要素を全てクリアする。
 * (Reentrant)
 * 	各スレッドが別々のarrayをもつ限り、この関数はスレッドセーフです。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 * @param index apr_int32_t 取得位置(0から始まります)
 * @return void * indexに格納されていたオブジェクトへのポインタ
 */
DIVY_DECLARE(void) divy_array_clear(divy_array_t *array);

/**
 * 可変長配列array の要素全てを取り出す。
 * (note)
 * 	取り出した要素は、divy_array_t に対する追加、削除オペレーションが
 * 	行われると不定値となります。追加、削除オペレーションが実施される
 * 	度に、この関数を使って再取得しなければなりません。
 * (Reentrant)
 * 	各スレッドが別々のarrayをもつ限り、この関数はスレッドセーフです。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 * @return void ** 可変長配列先頭へのポインタ
 */
DIVY_DECLARE(void **) divy_array_getelements(divy_array_t *array);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_ARRAY_H */

