/**
 * $Id$
 *
 * Set (文字列集合コンテナ) の宣言・定義を行うヘッダファイル
 * (note)
 * 	この宣言は元々util_common.h にあったものです。
 * 	ぐちゃぐちゃになったので移しました。
 */
#ifndef INCLUDE_TF_CSET_H
#define INCLUDE_TF_CSET_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"

#include "tfr.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_cset_t		divy_cset_t;
typedef struct __divy_cset_index_t	divy_cset_index_t;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * divy_cset_t(文字列集合コンテナ) の生成を行う
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @return divy_cset_t * 生成したdivy_cset_t
 */
DIVY_DECLARE(divy_cset_t *) divy_cset_make(apr_pool_t *p);

/**
 * cst が示す文字列集合の中にval を追加する。
 * (Reentrant)
 * 	各スレッドが別々のcst を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param val const char * 文字列集合に追加する文字列
 */
DIVY_DECLARE(void) divy_cset_set(divy_cset_t *cst, const char *val);

/**
 * 指定されたval が文字列集合cst の中に含まれているかどうか。
 * (Reentrant)
 * 	各スレッドが別々のcst を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param cst divy_cset_t * 文字列集合へのポインタ
 * @param val const char * 調査対象の文字列
 * @return int 含まれているかどうか
 *	 	1: 含まれている
 * 		0: 含まれていない
 */
DIVY_DECLARE(int) divy_cset_contains(divy_cset_t *cst, const char  *val);

/**
 * 文字集合cst から指定されたvalが示す文字列を削除する。
 * (Reentrant)
 * 	各スレッドが別々のcst を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param cst divy_cset_t * 文字列集合へのポインタ
 * @param val const char * 削除対象の文字列
 */
DIVY_DECLARE(void) divy_cset_remove(divy_cset_t *cst, const char *val);

/**
 * 文字列集合cstに含まれる要素の数を返却する
 * (Reentrant)
 * 	各スレッドが別々のcst を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param cst divy_cset_t * 文字列集合を表す構造体へのポインタ
 * @return unsigned int 要素の数
 */
DIVY_DECLARE(unsigned int) divy_cset_count(divy_cset_t *cst);

/**
 * 文字列集合cset のiterator を取得する。
 * (Reentrant)
 * 	各スレッドが別々のcst を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param cst divy_cset_t *
 */
DIVY_DECLARE(divy_cset_index_t *) divy_cset_first(divy_cset_t *cst);

/**
 * iteratorであるci を１つ進める。
 * (Reentrant)
 * 	各スレッドが別々のci を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param ci divy_cset_index_t * 文字列集合のiterator
 * @return divy_cset_index_t * １つ進んだiterator
 */
DIVY_DECLARE(divy_cset_index_t *) divy_cset_next(divy_cset_index_t *ci);

/**
 * iterator が現在指し示す値を取得する。
 * (Reentrant)
 * 	各スレッドが別々のci を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param ci divy_cset_index_t * 文字列集合のiterator
 * @param val const char ** 取得した文字列
 */
DIVY_DECLARE(void) divy_cset_this(divy_cset_index_t *ci, const char **val);

/**
 * 文字列集合cst に含まれる要素を文字列からなる配列として返却する.
 * 配列の最後の要素は NULL になっています。
 * 返却される個々の文字列は、cst が持っている文字列とは別の領域に
 * コピーされています。(安全のため)
 * (Reentrant)
 * 	各スレッドが別々のcst を持つ場合に限りこの関数はスレッドセーフです。
 *
 * (例) 配列の中身
 * 	+----+---+---+----+
 * 	|aaaa|bbb|ccc|NULL|
 * 	+----+---+---+----+
 * 	char*
 *
 * @param cst divy_cset_t * 文字列集合を表す構造体へのポインタ
 * @return char ** 文字列へのポインタからなる配列
 */
DIVY_DECLARE(char **) divy_cset_toarray(divy_cset_t *cst);

/**
 * cstが表す文字列集合をハッシュのキー集合として返却する.
 * (キー値が文字集合のハッシュを返却する)
 * (Reentrant)
 * 	各スレッドが別々のcst を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param cst divy_cset_t * 文字列集合
 * @return apr_hash_t * ハッシュ
 */
DIVY_DECLARE(apr_hash_t *) divy_cset_tohash(divy_cset_t *cst);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_CSET_H */
