/**
 * $Id$
 *
 * tf_db.h
 *
 * DBプロバイダインターフェースの宣言・定義を行うヘッダファイル。
 *
 * このヘッダは、DBプロバイダにアクセスするために用意された共通
 * インターフェースを宣言・定義する役割とDBプロバイダが提供しなければならない
 * 関数のプロトタイプを定義する役割を持っています。
 * (note)
 * 	ハンドラ構造体のアクセスメソッド(関数)以外の関数は、TeamFile"モジュール"
 * 	からコールしないようにして下さい。それ以外はDBプロバイダ間で利用する
 * 	関数や構造体だからです。(util_db.h は例外です。)
 * 	TeamFileモジュールから使用するには、多くの場合util_db.h で宣言された
 * 	関数や構造体を使用した方がよいでしょう。そのためのユーティリティです。
 *
 * 	このヘッダにはTeamFileモジュールおよびApache本体固有のヘッダをinclude
 * 	してはなりません。(mod_davやhttpd.hなど)
 * 	includeしてよいものは、apr_xxx とTeamFile"モジュール"に依存しない共通
 * 	ヘッダ(tfr.hなど)だけです。
 *
 * 2002/12/23 Mon takehara NEW
 * 2004/02/08 Sun takehara Change role of this file.
 */
#ifndef INCLUDE_DIVY_DB_H
#define INCLUDE_DIVY_DB_H 

/* Apache header files */
#include "apr.h"
#include "apr_pools.h"
#include "apr_hooks.h"
#include "apr_time.h"
#include "apr_hash.h"

#include "tfr.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*--------------------------------------------------------------
  Define fixed values and define types.
  --------------------------------------------------------------*/
/* boolean */
#define DB_TRUE  1
#define DB_FALSE 0

/**
 * 独自フックハンドラからの戻り値
 */
#define DB_OK		0
#define DB_DECLINED	-1

/* 
 * 実行ステータスコードの値 --> __code変数を参照
 * (note)
 * 	この戻り値の成功可否をif ret < 0 のように値が負であるかどうかで
 * 	判定してはなりません。if ret != DB_SUCCESS が正しい判定方法です。
 */
#define DB_SUCCESS                  0	/* 成功した                      */
#define DB_ERROR                   -1	/* 何らかの理由により失敗した    */
#define DB_NO_DATA                 -2	/* 検索結果が0件だった           */
#define DB_NOT_ALLOWED_UPDATE_STMT -3	/* 更新系SQLがexecuteQueryで指定された */
#define DB_LACK_OF_POOL	           -4   /* プールに使用可能なコネクションが無くなった */
#define DB_UNSUPPORTED_DATA_TYPE   -5	/* サポートされていないデータ型のカラムが指定された */

/* アクセサ関数で使用する戻り値 */
#define DB_PFUNC_RET_BASE	100	/* バイアス値 */
#define DB_INVALID_PARAMS	(DB_PFUNC_RET_BASE+1)	/* パラメータ不正 */
#define DB_NOTFOUND_DBDS	(DB_PFUNC_RET_BASE+2)	/* DBデータソースが見つからなかった */
#define DB_FAILED_DBCONN	(DB_PFUNC_RET_BASE+3)	/* コネクションの取得に失敗した */
#define DB_LICENSE_VALID	(DB_PFUNC_RET_BASE+4)	/* 正しいDBライセンスを持っている */
#define DB_LICENSE_NOTHING	(DB_PFUNC_RET_BASE+5)	/* DBライセンスを持っていなかった */
#define DB_LICENSE_EXPIRE	(DB_PFUNC_RET_BASE+6)	/* DBライセンスが切れた */

/* execution return type */
typedef int RET_STAT;

/* Transaction Isolation Level */
enum __db_isolationlv {
	TRANSACTION_NONE=0,            /* Transaction not supported */
	TRANSACTION_READ_UNCOMMITTED,  /* Dirty read                */
	TRANSACTION_READ_COMMITTED,    /* Restraint Dirty read      */
	TRANSACTION_REPEATABLE_READ,   /* Restraint Repeatable read */
	TRANSACTION_SERIALIZABLE       /* Restraint Phantom read    */
};

typedef enum __db_isolationlv DbIsolationLv;

/*
 * SQL data types
 * (note)
 * 	The value of "DB_SQL_UNCLEAR" means a db provider does not support
 * 	to get the column type or find not clear type.
 */
enum __db_sqltype {
	DB_SQL_UNCLEAR=0, /* noclassification, not support, unclear type */
	DB_SQL_INT,       /* int type          */
	DB_SQL_LNG,       /* long type         */
	DB_SQL_BLG,       /* apr_int64_t       */
	DB_SQL_FLT,       /* double            */
	DB_SQL_DATE,      /* char*('YYYYMMDD') */
	DB_SQL_TIMESTAMP, /* time_t type of C  */
	DB_SQL_STR        /* char*(till msec)  */
};

typedef enum __db_sqltype DbSqlType;

/**
 * dbpool の状態を表す列挙型
 */
enum {
	DIVY_DBPOOL_UNSET = 0,	/* dbpool がセットされていない   */
	DIVY_DBPOOL_OFF,	/* dbpool がOFF                  */
	DIVY_DBPOOL_ON,		/* dbpool がON, 管理スレッドあり */
	DIVY_DBPOOL_NOTHREAD	/* dbpool はON, 管理スレッドなし */
};

/**
 * dbvhostshare の列挙型
 */
enum {
	DIVY_VHOSTSHARE_UNSET = 0,	/* dbvhostshare が未設定 */
	DIVY_VHOSTSHARE_OFF,		/* dbvhostshare が OFF   */
	DIVY_VHOSTSHARE_ON		/* dbvhostshare が ON    */
};

/*--------------------------------------------------------------
  Declare handler structure
  --------------------------------------------------------------*/
typedef struct __db_DataSource DbDataSource;
typedef struct __db_Connection DbConn;
typedef struct __db_PreparedStmt DbPreparedStmt;
typedef struct __db_ResultSet DbResultSet;

/* 
 * A context object depending on specific DBMS. 
 * Implematation of this object will be supplied by provider module.
 */
typedef struct __db_dbms_Context DbCntxt;

/*
 * A statement context object depending on specific DBMS.
 * Implematation of this object will be supplied by provider module.
 */
typedef struct __db_dbms_stmtCntxt DbStmtCntxt;

/*
 * A resultset context object depending on specific DBMS.
 * Implematation of this object will be supplied by provider module.
 */
typedef struct __db_dbms_rsetCntxt DbRsetCntxt;

/**
 * The context of connection manager.
 * Implematation of this object will be supplied "db connection manager".
 * Each db provider does not need to implement it.
 */
typedef struct __db_ConnPoolCntxt DbConnPoolCntxt;

/*--------------------------------------------------------------
  Declare structure
  --------------------------------------------------------------*/
/**
 * ある種類のDBプロバイダに関するライセンス情報を保持する構造体
 * (note)
 * 	接続先ごとではなく、種類ごとにライセンス情報は保持されます。
 * 	実体はtf_db.c にあります。不完全型ですのでポインタ宣言しかできません。
 */
typedef struct __db_ProviderLicense	DbProviderLicense;

/*--------------------------------------------------------------
  Define handler structure
  --------------------------------------------------------------*/

/**
 * DBMSを表すオブジェクト.
 * 多くの場合、このオブジェクトの１つが、ある特定のDBMSを識別します.
 * 接続先が異なれば同じ種類のDBMSであっても異なるDbDataSourceのインスタンス
 * を持つことが例外です.
 *
 * このオブジェクトはデータベースとの接続を表すDbConnのFactoryとしての
 * 役割を持っています.
 * 
 */
struct __db_DataSource {
	
	char *dbmstype;  /* DBMSの種類名      */
	char *hostname;  /* DB servername     */
	char *hostport;  /* DB serverPort     */
	char *dbname;    /* DB database name  */
	char *username;  /* DB username       */
	char *password;  /* DB password       */
	apr_int32_t dbpool;		/* DB コネクションプールを使用するかどうか */
	int dbvhostshare;		/* VirtualHost間でコネクションを共用するかどうか */
	apr_int32_t dbminspareconn;	/* コネクションを保持しておく最小数  */
	apr_int32_t dbmaxspareconn;	/* コネクションを保持しておける最大数*/
	
	/**
	 * DBMS接続情報の初期化が完了したかどうかを示すフラグ
	 */
#define DBENV_INIT_NOT_FINISHED 0	/* 完了していない                 */
#define DBENV_INIT_FINISHED     1	/* 完了している                   */
	int isFinishedInitEnv;
	
	/**
	 * DBMS 識別名称(接続先別に異なる名称)
	 * 異なるDBMSまたは、同じDBMSでも接続先が異なる場合には、
	 * 別の識別名称が与えられなければならない.
	 */
	char *dbmsname;
	
	/**
	 * 実行ステータスメッセージ
	 *
	 * 多くの場合、DBMS固有のエラーメッセージ文字列が格納されています。
	 * 上位モジュールは、ここに格納されたエラーメッセージをgetMsgアクセサ
	 * 関数を使って取得します。
	 *
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード
	 *
	 * DBMSに対してアクションを行い、その結果に基づいて決定されるステータス値.
	 * このコードは、DBMSを意識しないモジュールが次に取らなければならない
	 * 動作を決定する際の判断材料となります.
	 * "実行ステータスコードの値"で定義されたdefine 値を設定するように
	 * して下さい。
	 * 受け取り側は、必ず getCode 関数を使用して取得して下さい。
	 * (note) DB プロバイダ以外は触らないこと
	 */
	int __code;
	
	/**
	 * DBMS固有のコード文字列
	 * DBMSから返却されたDBMS固有のステータスコード.
	 * 主としてエラーコードが格納されます.
	 * 受け取り側は、必ず getNativeCode関数を使用してください.
	 *
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__nativecode;
	
	
	/**
	 * データベースとの接続をあらわすオブジェクトを取得する.
	 * (note)
	 * 	getDbConn の処理で失敗したエラー状態は、DbConnの
	 * 	__msg, __code, __nativecode に記録して下さい。
	 * 	つまり、DbConn のインスタンスは必ず作って渡す必要が
	 * 	あるということです。
	 *
	 * @param dbds DbDataSource* 自分自身へのポインタ
	 * @param  pool apr_pool * 変数割り当てに使用するプール(request_rec)
	 * @return DbConn* データベースとの接続を表す構造体へのポインタ
	 */
	DbConn * (*getDbConn)(DbDataSource *dbds, apr_pool_t *pool);
	
	/**
	 * 全てのコネクションを開放して再作成する.
	 * DBプロバイダがConnection Poolを実装している場合、そのPoolを
	 * リフレッシュするために使用されます.
	 * (note)
	 * 	Connection Pool の実装はDBプロバイダ側で実施しないことに
	 * 	なりました。このハンドラに対しては空実装をつけて下さい。
	 * @param dbds DbDataSource*
	 */
	void (*rebuildConn)(DbDataSource *dbds);
	
	/**
	 * 実行ステータスコードを取得する
	 *
	 * @param dbds const DbDataSource* 自分自身へのポインタ
	 * @return int 実行ステータスコード(define値を参照)
	 */
	int (*getCode)(const DbDataSource *dbds);
	
	/**
	 * DBMS固有のコード文字列を取得する
	 *
	 * @param dbds const DbDataSource*
	 * @return char* コードを表す文字列。整数であるとは限りません。
	 */
	char * (*getNativeCode)(const DbDataSource *dbds);
	
	/**
	 * 実行ステータスメッセージを取得する
	 *
	 * @param dbds const DbDataSource*
	 * @return char* メッセージを表す文字列。
	 */
	char * (*getMsg)(const DbDataSource *dbds);
};


/**
 * DBMSとの接続(コネクション)を表すオブジェクト.
 * このオブジェクト１つが、１つのDBMSコネクションを表します.
 *
 * (注意)
 * この構造体はDbDataSourceの関数 getDbConn から取得する必要があります。
 * mallocなどにより直接確保しないようにしてください。
 *
 */
struct __db_Connection {

	/**
	 * 特定のDBMSに依存したデータを保持する構造体へのポインタ
	 * ポインタの指し示す先を気にする必要はまったくない。
	 * (note) DB プロバイダ以外は触らないこと
	 */
	DbCntxt *dbcntxt;
	
	/**
	 * このコネクションを作成したFactory (DbDataSource)へのポインタ
	 */
	DbDataSource *dbds;
	
	/**
	 * トランザクションの分離レベルを保持する
	 */
	DbIsolationLv dbIsolationLv;
	
	/**
	 * 実行ステータスメッセージ (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	int __code;
	
	/**
	 * DBMS固有のコード文字列 (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__nativecode;

	/**
	 * DB コネクションプール用コンテキスト
	 * (note) DB プロバイダの管理を行う側で実装します。
	 *        各DBプロバイダ及びDbConn利用者は意識してはなりません。
	 */
	DbConnPoolCntxt *dbcpoolctx;
	
	/**
	 * DBとのコネクションを閉じ、使用していたリソースを開放する。
	 * この呼び出し以降、DbConnに対するあらゆる動作は不定です。
	 * (note)
	 * 	コネクションプールの実装は、DBプロバイダ側では
	 * 	実施しないことになりました。ですので、必ずリソースを
	 * 	解放してください。
	 *
	 * @param dbconn DbConn*
	 */
	void (*close)(DbConn *dbconn);
	
	/**
	 * コネクションが閉じられているかどうか。
	 *
	 * @param dbconn DbConn*
	 * @return int DB_TRUE: 閉じられている / DB_FALSE: 閉じられていない
	 * @deprecated DBMSによってはこのステータスを取得できないことがわかったので
	 *             このハンドラは廃止します。応答が戻ってくる保障はなく、
	 *             戻ってきたとしても正しいとは限りません。
	 */
	int (*isClosed)(DbConn *dbconn);
	
	/**
	 * SQLをプリコンパイルし、それを表すDbPreparedStmtを取得する。
	 * (note)
	 * 	prepareStatement の処理で失敗したエラー状態は、DbPreparedStmtの
	 * 	__msg, __code, __nativecode に記録して下さい。
	 * 	つまり、DbPreparedStmt のインスタンスは何があっても作って
	 * 	渡す必要があるということです。
	 *
	 * @param  dbconn DbConn*
	 * @param  sqltext const char*
	 * @param  pool apr_pool * 変数割り当てに使用するプール(request_rec)
	 * @return DbPreparedStmt* プリコンパイルされたSQLを表すオブジェクト
	 */
	DbPreparedStmt * (*prepareStatement)(DbConn *dbconn, 
	                                     const char *sqltext, apr_pool_t *pool);
	
	/**
	 * トランザクションを開始する。
	 * (note)
	 * 	SQL実行と同時に暗黙的にトランザクションが開始されるDBMSでは、
	 * 	このハンドラを空実装にして下さい。
	 *
	 * @param dbconn DbConn*
	 * @param timeout const unsigned int トランザクションタイムアウト(Sec)
	 */
	void (*startTrans)(DbConn *dbconn, const unsigned int timeout);
	
	/**
	 * startTrans によって開始されたトランザクションをコミットする。
	 *
	 * @param dbconn DbConn*
	 */
	void (*commit)(DbConn *dbconn);
	
	/**
	 * startTrans によって開始されたトランザクションをロールバックする。
	 *
	 * @param dbconn DbConn*
	 */
	void (*rollback)(DbConn *dbconn);
	
	/**
	 * このコネクションのトランザクションの分離レベルを設定する。
	 *
	 * @param dbconn DbConn*
	 * @param level const DbIsolationLv トランザクションの分離レベル
	 */
	void (*setTransIsolation)(DbConn *dbconn, const DbIsolationLv level);
	
	/**
	 * このコネクションに対してWHERE句なしで常に発行可能なSQL文
	 * (SELECT)を取得する。
	 * (note)
	 * 	このSQLは、コネクションがアクティブであることを確認するために
	 * 	DBコネクションマネージャが使用します。
	 * 	DBに負荷の掛からない、かつ必ず実行可能なSQLを返却して下さい。
	 * 	もし実行可能なSQL文がなければNULLを返却して下さい。
	 *
	 * @return const char * 実行可能なSQL文
	 * 			存在しなければNULLを返却
	 */
	const char * (*getExecutableSQL)(DbConn *dbconn);
	
	/**
	 * 実行ステータスコードを取得する
	 *
	 * @param dbconn const DbConn*
	 * @return int 実行ステータスコード(define値を参照)
	 */
	int (*getCode)(const DbConn *dbconn);
	
	/**
	 * DBMS固有のコード文字列を取得する
	 *
	 * @param dbconn const DbConn*
	 * @return char* コードを表す文字列。整数であるとは限りません。
	 */
	char * (*getNativeCode)(const DbConn *dbconn);
	
	/**
	 * 実行ステータスメッセージを取得する
	 *
	 * @param dbconn const DbConn*
	 * @return char* メッセージを表す文字列。
	 */
	char * (*getMsg)(const DbConn *dbconn);
};


/**
 * プリコンパイルされたSQLを表す
 * A DbPreparedStmt shows a pre-complied SQL.
 *
 */
struct __db_PreparedStmt {
	
	/**
	 * 特定のDBMSに依存したデータを保持する構造体へのポインタ(statement関連)
	 * ポインタの指し示す先を気にする必要はまったくない。
	 * (note) DB プロバイダ以外は触らないこと
	 */
	DbStmtCntxt *stmtcntxt;
	
	/**
	 * このDbPreparedStmtを作成したコネクションを表す
	 */
	DbConn *dbconn;
	
	/**
	 * 実行ステータスメッセージ (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	int __code;
	
	/**
	 * DBMS固有のコード文字列 (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__nativecode;
	
	/**
	 * 内部的な処理にのみ使用されるpoolへのポインタ
	 * (note) DB プロバイダ以外は触らないこと
	 */
	apr_pool_t *__pool;
	
	/**
	 * DbPreparedStmtを閉じ、DbPreparedStmtが使用していたリソースを開放する.
	 * この呼び出し以降、DbPreparedStmtに対するあらゆる動作は不定です。
	 * また、生成されたDbResultSetに対するアクセスも保証されていません。
	 * 理由は以下を参照。
	 * (note)
	 * 	DB プロバイダの実装によりますが、DbPreparedStmt から
	 * 	生成されたDbResultSet が存在する状態でこの関数を呼ぶと
	 * 	予期しない結果を生むことがあります。
	 * 	これは、DbPreparedStmt の中にDBから取得した結果セット
	 * 	(DbResultSet) を保持する領域を持っているためです。
	 * 	本来なら、このような実装は好ましくはありませんが、
	 * 	提供されているDBアクセスAPI自身にこのようなアーキテクチャを
	 * 	持つものが比較的多いようですので、この問題をそのまま
	 * 	引きずってしまっています。
	 *
	 * 	よって、呼び出し側では、DbResultSet が存在するのなら、
	 * 	DbPreparedStmt のcloseを実施しないようにして下さい。
	 *
	 * @param stmt DbPreparedStmt*
	 */
	void (*close)(DbPreparedStmt *stmt);
	
	/**
	 * DbPreparedStmtのSQL(select)クエリを実行し、それによって得られた
	 * 結果セットを返却する。
	 * なお、SELECT ... FOR UPDATE 構文を利用する場合には、DBMSの種類が
	 * 明示的に判る状況で無い限り、executeQueryForUpdate メソッドを利用して下さい。
	 *
	 * (note)
	 * 	更新系DML, DDL, DCLが指定された場合、DB_NOT_ALLOWED_UPDATE_STMT
	 * 	を __code に設定し、異常終了させて下さい。
	 * 	呼び出し側では、このコードが戻されたら、指定したSQLが
	 * 	SELECTではなかったと判断して、適切な処理を行ってください。
	 * (note)
	 * 	prepareStatement の処理で失敗したエラー状態は、DbResultSetの
	 * 	__msg, __code, __nativecode に記録して下さい。
	 * 	(つまり、DbResultSet のインスタンスは何があっても作って
	 * 	 渡す必要があるということです。)
	 * (note)
	 *	SQL文中にDBプロバイダがサポートしていないデータ型のカラムが指定
	 *	されると、DB_UNSUPPORTED_DATA_TYPE がステータスコードとして設定
	 *	されます。呼び出し側ではこれを適切に処理する必要があります。
	 *
	 * @param  stmt DbPreparedStmt*
	 * @param  pool apr_pool* 変数割り当てに使用するプール(request_rec)
	 * @return DbResultSet* クエリの結果セット
	 */
	DbResultSet * (*executeQuery)(DbPreparedStmt *stmt, apr_pool_t *pool);

	/**
	 * (2005/05/16 Mon takehara 新規追加)
	 * DbPreparedStmtのSQL(select ... for update)クエリを実行し、それによって得られた
	 * 結果セットを返却する。(カーソルなし)
	 *
	 * (note) executeQuery との違い
	 * 	executeQueryメソッドではSELECT ... FOR UPDATE 構文が利用できるかどうかは
	 * 	DBプロバイダの実装に依存します。
	 * 	一方、executeQueryForUpdate では必ずSELECT ... FOR UPDATE構文が動作しますが、
	 * 	カーソルが利用出来ない可能性があります。これもDBMS依存となります。
	 *
	 * (note) executeQuery との住み分け
	 *   * FOR UPDATE 構文を利用しないSELECT文 --> executeQuery
	 *   * FOR UPDATE 構文を利用するSELECT文   --> executeQueryForUpdate
	 * 
	 * 例外的に、DBMSの種類が明示的に特定され、カーソル付きのSELECT FOR UPDATE文を
	 * 使いたい場合に限りexecuteQuery を利用できるかもしれません。保証は呼び出し元で
	 * 行って下さい。
	 */
	DbResultSet * (*executeQueryForUpdate)(DbPreparedStmt *stmt, apr_pool_t *pool);

	/**
	 * DbPreparedStmtの更新系SQL(insert, update, delete)を実行する。
	 *
	 * @param stmt DbPreparedStmt*
	 * @param  pool apr_pool * 変数割り当てに使用するプール(request_rec)
	 * @return INSERT、UPDATE、DELETE文の場合は行数. 何も返さない文の場合は 0
	 */
	int (*executeUpdate)(DbPreparedStmt *stmt, apr_pool_t *pool);
	
	/**
	 * 指定されたパラメータ(バインド変数)にSQL INTEGER値として値をセットする。
	 *
	 * @param stmt DbPreparedStmt*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @param value const apr_int32_t 設定する値(32bit)
	 */
	void (*setInt)(DbPreparedStmt *stmt, const int index, const apr_int32_t value);
	
	/**
	 * 指定されたパラメータ(バインド変数)にSQL BIGINT 値として値をセットする。
	 * (note)
	 * 	SQL NUMERIC型のサポートは中止しました。変わりにこのハンドラは
	 * 	SQL BIGINT型をサポートすることになります。
	 *
	 * @param stmt DbPreparedStmt*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @param value const apr_int64_t 設定する値
	 */
	void (*setBigInt)(DbPreparedStmt *stmt, const int index, const apr_int64_t value);
	
	/**
	 * 指定されたパラメータ(バインド変数)にSQL DOUBLE値として値をセットする。
	 *
	 * @param stmt DbPreparedStmt*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @param value const double 設定する値
	 */
	void (*setDouble)(DbPreparedStmt *stmt, const int index, const double value);
	
	/**
	 * 指定されたパラメータ(バインド変数)にSQL VARCHAR または
	 * LONGVARCHAR 値として値をセットする。
	 *
	 * @param stmt DbPreparedStmt*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @param value const char* 文字列形式の値
	 */
	void (*setString)(DbPreparedStmt *stmt, const int index, const char *value);

	/**
	 * 実行ステータスコードを取得する
	 *
	 * @param stmt const DbPreparedStmt* 自分自身へのポインタ
	 * @return int 実行ステータスコード(define値を参照)
	 */
	int (*getCode)(const DbPreparedStmt *stmt);
	
	/**
	 * DBMS固有のコード文字列を取得する
	 *
	 * @param stmt const DbPreparedStmt* 自分自身へのポインタ
	 * @return char* コードを表す文字列。整数であるとは限りません。
	 */
	char * (*getNativeCode)(const DbPreparedStmt *stmt);
	
	/**
	 * 実行ステータスメッセージを取得する
	 *
	 * @param stmt const DbPreparedStmt* 自分自身へのポインタ
	 * @return cha * メッセージを表す文字列。
	 */
	char * (*getMsg)(const DbPreparedStmt *stmt);

	/**
	 * 指定された文字列の中のSQLワイルドカード文字列(%, _など)をエスケープ
	 * して、全文字列を再構成して返却する。
	 */
	char * (*escWildCard)(const DbPreparedStmt *stmt, const char *value);
};


/**
 * データベースの結果セットを表す.
 * SQLの select文を実行(executeQuery)することによって生成される.
 *
 */
struct __db_ResultSet {
	
	/**
	 * この ResultSetオブジェクトを作成したDbPreparedStmtオブジェクトを示す。
	 */
	DbPreparedStmt *stmt;
	
	/**
	 * 特定のDBMSに依存したデータを保持する構造体へのポインタ(resultset関連)
	 * ポインタの指し示す先を気にする必要はまったくない。
	 * (note) DB プロバイダ以外は触らないこと
	 */
	DbRsetCntxt *rsetcntxt;
	
	/**
	 * 実行ステータスメッセージ (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	int __code;
	
	/**
	 * DBMS固有のコード文字列 (DbDataSourceの__msgと同じ役割)
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char *__nativecode;
	
	/**
	 * 内部的な処理にのみ使用されるpoolへのポインタ
	 * (note) DB プロバイダ以外は触らないこと
	 */
	apr_pool_t *__pool;
	
	/**
	 * getxxx系メソッドが利用するpoolへのポインタ (2005/05/16 Mon takehara 新規追加)
	 * (note)
	 * 	このプールのライフサイクルはnext メソッドが1回コールされる間だけです。
	 * 	次のコールではクリアされる可能性があります。
	 * 	そうであるかどうかは呼び出し元の制御下にあるため、この条件と合致しない
	 * 	オブジェクトのメモリアロケータとしては利用しないで下さい。
	 * 	(例えば、DbRsetCntxt は必ず__pool から確保する必要があります)
	 */
	apr_pool_t *__scratchpool;

	/**
	 * このオブジェクト(検索した結果)の列数
	 * (note) DB プロバイダ以外は触らないこと
	 */
	int __columnCount;
	
	/**
	 * このオブジェクト(検索した結果)の列名を表す文字列へのポインタ配列
	 * インデックス番号によってはSQL型の配列と列のバイト数の配列との
	 * 対応を取ります.
	 * (note) DB プロバイダ以外は触らないこと
	 */
	char **__columnNames;
	
	/**
	 * このオブジェクト(検索した結果)のSQL型を表す数値(DbSqlType)の配列
	 * (note) DB プロバイダ以外は触らないこと
	 */
	DbSqlType *__columnTypes;
	
	/**
	 * このオブジェクト(検索した結果)の列のバイト数からなる配列
	 * (note) DB プロバイダ以外は触らないこと
	 */
	int *__columnSizes;
	
	/**
	 * DbResultSetを破棄する.
	 * この呼び出し以降、DbResultSetに対するあらゆる動作は不定です。
	 *
	 * @param rset DbResultSet*
	 */
	void (*close)(DbResultSet *rset);
	
	/**
	 * カーソルを現在の位置から１つ進める
	 * 初期状態では最初の行の前に位置付けられています.
	 *
	 * @param rset DbResultSet*
	 * @return 新しい現在の行が有効な場合は DB_TRUE
	 *         それ以上行が無い場合には DB_FALSE
	 */
	int (*next)(DbResultSet *rset);
	
	/**
	 * このオブジェクト(検索した結果)の列数を取得する.
	 * 
	 * @param rset DbResultSet*
	 * @return int 列数
	 */
	int (*getColumnCount)(DbResultSet *rset);
	
	/**
	 * 指定されたインデックス番号index に対応する列名を取得する.
	 *
	 * @param rset DbResultSet*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @return char * 列名
	 */
	char * (*getColumnName)(DbResultSet *rset, const int index);
	
	/**
	 * 指定された列のSQL型(DbSqlType)を取得する.
	 * (note)
	 * 	メタデータの取得をサポートしていないDBMSがあるため、
	 * 	この操作により正しく値を取得できないことがあります。
	 * 	残念ながら、この操作に依存したコードには注意が必要と
	 * 	なります。
	 *
	 * @param rset DbResultSet*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @return DbSqlType SQL型(列挙型)
	 * 	DB_SQL_UNCLEAR が返却されたときには、対象となるDBプロバイダが
	 * 	SQL型を取得するオペレーションをサポートしていなかったか、
	 * 	不明なデータ型だったと判断して下さい。
	 */
	DbSqlType (*getColumnType)(DbResultSet *rset, const int index);
	
	/**
	 * 指定された列のバイト数を取得する.
	 *
	 * (note)
	 *  DBMSによっては、指定された列のバイト数が返却されないことがあります。
	 *  (内部的に管理している管理領域のサイズが返却されることもあります。)
	 *  よって、呼び出し側では、使用するDBプロバイダが一意に特定出来ない
	 *  限りこの関数の戻り値による処理(メモリ領域の確保など)は、行うべき
	 *  ではありません。
	 *  getString関数等で取得される検索結果を示す文字列を別の領域にコピー
	 *  したい場合には、apr_pstrdup関数を使ってコピーするようにして下さい。
	 * 
	 * @param rset DbResultSet*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @return int バイト数
	 */
	int (*getColumnSize)(DbResultSet *rset, const int index);
	
	/**
	 * このオブジェクトの現在行にある指定された列の値をSQL INTEGER型
	 * (apr_int32_t)として取得する。
	 * (note)
	 * 	対応カラムのデータがNULLだった場合には、0を返却します。
	 *
	 * @param rset DbResultSet*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @return apr_int32_t
	 */
	apr_int32_t (*getInt)(DbResultSet *rset, const int index);
	
	/**
	 * このオブジェクトの現在行にある指定された列の値をSQL BIGINT型
	 * (apr_int64_t)として取得する。
	 * (note)
	 * 	対応カラムのデータがNULLだった場合には、0を返却します。
	 *
	 * @param rset DbResultSet*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @return apr_int64_t
	 */
	apr_int64_t (*getBigInt)(DbResultSet *rset, const int index);
	
	/**
	 * このオブジェクトの現在行にある指定された列の値をSQL FLOAT型、
	 * SQL DOUBLE型(double)として取得する。
	 * (note)
	 * 	対応カラムのデータがNULLだった場合には、0を返却します。
	 *
	 * @param rset DbResultSet*
	 * @param index const int パラメータの順番(初期値1とし+1ずつ増える)
	 * @return double
	 */
	double (*getDouble)(DbResultSet *rset, const int index);
	
	/**
	 * このオブジェクトの現在行にある指定された列の値をchar *型として取得する。
	 *
	 * @param rset DbResultSet*
	 * @param index int パラメータの順番(最初のパラメータを1とし+1ずつ増える)
	 * @return char*
	 */
	char * (*getString)(DbResultSet *rset, const int index);
	
	/**
	 * 実行ステータスコードを取得する
	 *
	 * @param rset const DbResultSet*
	 * @return int 実行ステータスコード(define値を参照)
	 */
	int (*getCode)(const DbResultSet *rset);
	
	/**
	 * DBMS固有のコード文字列を取得する
	 *
	 * @param rset const DbResultSet*
	 * @return char * コードを表す文字列。整数であるとは限りません。
	 */
	char * (*getNativeCode)(const DbResultSet *rset);
	
	/**
	 * 実行ステータスメッセージを取得する
	 *
	 * @param rset const DbResultSet*
	 * @return char * メッセージを表す文字列。
	 */
	char * (*getMsg)(const DbResultSet *rset);
};


/*--------------------------------------------------------------
  Declare Public Functions
  (note)
  	以下のPublic関数の実体はtf_db.c の中に持ちます。
	各DBプロバイダは、これら関数の実装(tf_db.c)を自身のモジュール
	にコンパイル＆リンクしてはなりません。
  --------------------------------------------------------------*/
/**
 * 指定されたproviderType に一致するDBプロバイダであった時、
 * DB プロバイダ構造体を作成 & 返却する関数
 * のためのHook関数宣言。(セミコロンは付けないで下さい。)
 * DB プロバイダの初期化(register_hookハンドラ) に利用します。
 *
 * (note) このHook関数宣言によって利用できるようになる関数
 *
 *  ・divy_hook_create_dbdatasource : フックハンドラとして関数を登録する
 *  ・divy_run_create_dbdatasource  : 上記関数で登録されたフック関数を呼び出す
 */
APR_DECLARE_EXTERNAL_HOOK(divy, DIVY, int, create_dbdatasource,
		(apr_pool_t *p, const char *providerType,  DbDataSource **dbds))

/**
 * 指定されたproviderType のDBデータソース(DbDataSource) を取得して返却する。
 * 取得に失敗した場合には、NULLを返却し、戻り値でステータスを返却します。
 * (note)
 * 	この関数を使って取得したDBデータソースにはDB接続に必要な接続パラメータが
 * 	一切格納されていません。これは呼び出し元が実施して下さい。その際、格納する
 * 	文字列の領域は、この関数に渡される引数p と同じプールから確保して下さい。
 * 	格納すべき値は以下の通り。これ以外は触らないこと。
 *	  * dbmsname  : (文字列) DBMS 識別名称(任意のユニーク値)
 *	  * hostname  : (文字列) DBMS 接続先ホスト名
 *	  * hostport  : (文字列) DBMS 接続先ポート番号 
 *	  * dbname    : (文字列) データベース名 / 接続記述子(Oracle)
 *	  * username  : (文字列) 接続先DBのユーザ名(スキーマ名)
 *	  * password  : (文字列) 接続先DBのユーザのパスワード
 *	  * isFinishedInitEnv : (数値) DBENV_INIT_FINISHED 初期化終了を表す値
 * (note)
 * 	この関数は、取得したDbDataSourceを内部的にキャッシュしたり、
 *	再利用したりしません。Apache モジュール上からDbDataSourceの取得を行う
 *	場合には、この関数ではなく、util_db.h で宣言された関数を使用して下さい。
 *
 * (取得に失敗するケース)
 * 	・適切なライセンスがない
 * 	・DBプロバイダライブラリが物理的に存在しない
 * 	
 * @param p apr_pool_t * DbDataSource用の領域を取得するプール
 * @param providerType const char * DBプロバイダの種類
 * @param licensekey const char * DBプロバイダライセンスキー文字列
 * @param dbds DbDataSource ** 取得したDBデータソースへのポインタ
 * 				なお、*dbds は最初にNULL初期化されます。
 * @return RET_STAT 処理ステータス
 * 	DB_SUCCESS         : 取得に成功
 * 	DB_ERROR           : 予期しない問題が発生したため取得に失敗した
 * 	DB_INVALID_PARAMS  : 引数がNULL値, 不適切な値だった
 * 	DB_NOTFOUND_DBDS   : DB データソースが見つからなかった
 * 	DB_LICENSE_NOTHING : DB プロバイダライセンスがなかった
 * 	DB_LICENSE_EXPIRE  : DB プロバイダライセンスが期限切れだった
 */
DIVY_DECLARE(RET_STAT) divy_db_lookup_dbdatasource(apr_pool_t *p,
						const char *providerType,
						const char *licensekey,
						DbDataSource **dbds);

/**
 * 指定されたdbds が示すDBデータソースからアクティブなDB コネクションを取得して
 * 返却する。
 * 取得に失敗した場合には、NULLを返却し、戻り値でステータスを返却します。
 *
 * この関数を使って取得したDBコネクション(DbConn) は、引数p が示すプールがクリア
 * されたり破棄された時、自動的にclose されます。但し、以下の条件に相当する場合には
 * 呼び出し元にてclose 処理を実施して下さい。
 * 	* 引数p のプールはグローバルなプールであり、クリアされるのはプログラムが
 * 	  終了する時である
 * 	* 何回もDbConn の取得を実施しなければならない
 *
 * 上記のガイドラインが守られない場合、不要なDBコネクションが残ったままとなり、
 * DBMSに悪影響を及ぼします。
 * (note)
 * 	この関数は、取得したDbConnを内部的にキャッシュしたり、再利用したりしません。
 * 	Apache モジュール上からDbDataSourceの取得を行う場合には、この関数ではなく
 * 	util_db.h で宣言された関数を必ず使用して下さい。
 * (note)
 *
 * @param p apr_pool_t * DBコネクションの実体を割り当てるプール
 * @param dbds DbDataSource * DBデータソース
 * 			なお、*dbconn は最初にNULL初期化されます。
 * @param dbconn DbConn ** 取得したDBコネクション
 * @return RET_STAT 処理ステータス
 * 	DB_SUCCESS        : 取得に成功
 * 	DB_ERROR          : 予期しない問題が発生したため取得に失敗した
 * 	DB_INVALID_PARAMS : 引数がNULL値, 不適切な値だった
 * 	DB_FAILED_DBCONN  : DBコネクションの取得に失敗した
 */
DIVY_DECLARE(RET_STAT) divy_db_getDbConn(apr_pool_t *p,
					DbDataSource *dbds, DbConn **dbconn);

/**
 * DBプロバイダライセンスdblicensekey をパースし、DBプロバイダの種類をキーとして
 * dblicense_h にライセンス構造体を格納する。
 *
 * @param p apr_pool_t * 作業用のプール(configと同じライフサイクルを持つこと)
 * @param dblicensekey const char * あるひとつのDBプロバイダライセンスキー文字列
 * @param dblicense_h apr_hash_t * DBプロバイダライセンス構造体を持つハッシュ
 * 				領域を割り当ててあること。
 * @return int 処理ステータス (1: ライセンスキーが改竄されていた / 0: 成功)
 */
DIVY_DECLARE(int) divy_db_parse_licensekey(apr_pool_t *p,
						const char *dblicensekey,
						apr_hash_t *dblicense_h);

/**
 * 指定されたlicense が使用可能な適切なDBプロバイダライセンスかどうかを検証する。
 *
 * (note) ライセンスが適切であると判断する条件
 * 	(1) DBプロバイダライセンスキーが登録されていること
 * 	(2) ライセンス期限内もしくは、無制限ライセンスであること
 *
 * @param p apr_pool_t *
 * @param license DbProviderLicense * DBプロバイダライセンス
 * @return int ライセンスがあるかどうか
 * 	DB_LICENSE_VALID   : 持っている
 * 	DB_LICENSE_NOTHING : 持っていない
 * 	DB_LICENSE_EXPIRE  : 持っていたが切れてしまった
 */
DIVY_DECLARE(RET_STAT) divy_db_validate_dblicense(apr_pool_t *p,
						DbProviderLicense *license);

/*
 * 以下はDBプロバイダが共通関数として利用するユーティリティです。
 * Publicスコープにありますが、DBプロバイダ以外は以下をコールしないように
 * して下さい。
 */

/**
 * 指定されたstrが示す文字列の中からバインド変数の識別文字'?'の位置を検索し
 * 見つかった位置のポインタを返却する。
 * 但し、'?'がシングルクオートと共に指定される、つまり単なる文字列の時には、
 * バインド変数と認識されません。
 * (例)
 *                          (p1)              (p2)
 * 	SELECT id, id2, 'abc?' FROM abc id3 = ?
 * 	p1, p2 はポインタを示す。
 * 	これがstrとして指定されたとき、返却されるポインタはp2のものになります。
 *
 * @param str const char * 検索対象の文字列
 * @param const char * バインド変数が見つかった文字列へのポインタ。('?'の位置)
 * 			見つからなかった場合にはNULLを返却
 */
DIVY_DECLARE(const char *) dav_divy_find_bindposition(const char *str);

/**
 * getxxx系メソッドが利用するpoolを設定する。(2005/05/16 Mon takehara 新規追加)
 *
 * @param rset DbResultSet *
 * @param scratchpool apr_pool_t * 利用するプールへのポインタ
 * 	scratchpool がNULLであれば何もしません。
 */
DIVY_DECLARE(void) divy_db_set_rset_outputpool(DbResultSet *rset, apr_pool_t *scratchpool);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_DIVY_DB_H */

