/**
 * $Id$
 *
 * tf_errlog.h
 *
 * エラーログ関連の構造体/関数を定義するヘッダファイル
 * ここで宣言＆定義されるものは、Apache本体に依存してはなりません。
 *
 * (note) このヘッダについて
 * 	Apacheに依存せずに、Apacheのログに出力するには？という問いには
 * 	本質的に解はない。Apache がそのような作りになっていないからである。
 * 	それでもApache非依存のライブラリからログ情報を何とか出力する必要が
 * 	あったため、制限つきながらここにあるような実装にしました。
 * 	ライブラリ関数は、ここにあるログを使ってログ出力を行ってください。
 *
 * 	このヘッダには２種類のログ方式が定義されています。
 *
 * 	(1) ライブラリロガー
 * 	----------------------------------
 * 		Apache のChild プロセス起動時にApacheログ関数へのポインタを
 * 		グローバル変数に記録しておき、そのポインタを使った以降の
 * 		ログ出力を行う方式。ログ出力には LERRLOG で始まる特殊な
 * 		マクロを使用します。
 * 		ライブラリロガーに渡す実体のログ関数へのポインタはdivy_lerrlog_t
 * 		が決めるプロトタイプを持っていればよいので、Apacheのロガー
 * 		だけでなく任意のロガーを実行時に入れ替えることが可能です。
 * 		最大の欠点は、ApacheのVirtualHost毎にログを切り替えられないことです。
 * 		server_rec が手に入らないので仕方ありません。
 *
 * 		このロガーは主としてライブラリ関数の中で使用されるでしょう。
 *
 * 	(2) エラーログ構造体 (divy_error)
 * 	----------------------------------
 * 		mod_dav が定義していたdav_error と同じ概念を表しています。
 * 		エラーログはその場で出力するのではなく、専用のエラーログ構造体の
 * 		インスタンスに記録しておき、Apacheとやりとり可能な場所にいったら
 * 		ログを出力するという方式です。
 * 		この方式はその場でログを出力できないため、受け取るモジュールは
 * 		大変になりますが、VirtualHostもきちんと解釈出来るようになります。
 * 		(受け取り側がきちんと処理できれば、ですが)
 *
 */
#ifndef INCLUDE_TF_ERRLOG_H
#define INCLUDE_TF_ERRLOG_H

#include "apr.h"
#include "apr_pools.h"

#include "tfr.h"
#include "tf_errno.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Define fixed value
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define macros
  ----------------------------------------------------------------------------*/
/**
 * ライブラリロガーマクロ
 *
 * このマクロは、server_rec や request_rec を参照できないライブラリやスレッドが
 * ロギングに利用するマクロです。ERRLOGx系とは異なりApache本体には依存していません。
 * このロガーは、Apacheモジュール以外で使用された場合、stderr にログを出力します。
 * Apacheモジュール上で動作するときには、Default サーバのerror_log ファイルに出力します。
 * (note)
 * 	・このロガーはデバッグレベルによるログの調整が出来ません
 * 	・DEBUG, INFO レベルのログは出力出来ません(重要ではないログは出さない)
 */
#ifdef HAVE_SYSLOG
#include <syslog.h>

#define LLOG_EMERG	LOG_EMERG
#define LLOG_ALERT	LOG_ALERT
#define LLOG_CRIT	LOG_CRIT
#define LLOG_ERR	LOG_ERR
#define LLOG_WARNING	LOG_WARNING
#define LLOG_NOTICE	LOG_NOTICE

#else

#define LLOG_EMERG	0
#define LLOG_ALERT	1
#define LLOG_CRIT	2
#define LLOG_ERR	3
#define LLOG_WARNING	4
#define LLOG_NOTICE	5

#endif	/* HAVE_SYSLOG */

#define LERRLOG0(level,stcd,f) \
{ \
	divy_lerrlog_t lerrlog = divy_get_lerrlogger(); \
	if (lerrlog != NULL) { \
		lerrlog(NULL, 0, -1, level, 0, NULL, "- - %s(%d): (%d) "f, \
				__func__, __LINE__, stcd); \
	} \
}

#define LERRLOG1(level,stcd,f,a1) \
{ \
	divy_lerrlog_t lerrlog = divy_get_lerrlogger(); \
	if (lerrlog != NULL) { \
		lerrlog(NULL, 0, -1, level, 0, NULL, "- - %s(%d): (%d) "f, \
				__func__, __LINE__, stcd, a1); \
	} \
}

#define LERRLOG2(level,stcd,f,a1,a2) \
{ \
	divy_lerrlog_t lerrlog = divy_get_lerrlogger(); \
	if (lerrlog != NULL) { \
		lerrlog(NULL, 0, -1, level, 0, NULL, "- - %s(%d): (%d) "f, \
				__func__, __LINE__, stcd, a1, a2); \
	} \
}

#define LERRLOG3(level,stcd,f,a1,a2,a3) \
{ \
	divy_lerrlog_t lerrlog = divy_get_lerrlogger(); \
	if (lerrlog != NULL) { \
		lerrlog(NULL, 0, -1, level, 0, NULL, "- - %s(%d): (%d) "f, \
				__func__, __LINE__, stcd, a1, a2, a3); \
	} \
}

/**
 * divy_new_error, divy_push_error 関数のdesc に文字列を入れる際に利用するマクロ
 */
#define DESC0(pool,f)		(char *)__func__, __LINE__, f
#define DESC1(pool,f,a1)	(char *)__func__, __LINE__, apr_psprintf(pool,f,a1)
#define DESC2(pool,f,a1,a2)	(char *)__func__, __LINE__, apr_psprintf(pool,f,a1,a2)
#define DESC3(pool,f,a1,a2,a3)	(char *)__func__, __LINE__, apr_psprintf(pool,f,a1,a2,a3)

/*------------------------------------------------------------------------------
  Define structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_error	divy_error;

/**
 * エラーログ情報を保持する構造体
 */
struct __divy_error {
	int level;		/* ログレベル (定義値: ログレベル) */
	int status;		/* ステータス (tf_errno.h で定義された値) */
	char *funcname;		/* 問題が発生した関数の名称 */
	int lineno;		/* 問題が発生した行番号     */
	const char *desc;	/* 詳細メッセージ */

	divy_error *prev;	/* 1つ古いエラーへのポインタ */
	divy_error *next;	/* 1つ新しいエラーへのポインタ */
};

/*------------------------------------------------------------------------------
  Declare type of functions
  ----------------------------------------------------------------------------*/
/**
 * ログ関数型の定義
 * (note)
 * 	このログ関数型を持つ関数であれば、一連のLERRLOGx マクロが使用する
 * 	ロガーとして使用することが可能です。
 * 	定義したら、関数divy_init_errlog_env を使ってロガーを設定してください。
 * 	但し、ロガーはApacheステージに依存することなく使用できなければ
 * 	なりません。
 */
typedef DIVY_DECLARE(void) (*divy_lerrlog_t)(const char *,int,int,int
			apr_status_t,apr_pool_t *, const char *, ...);

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/

/**
 * ライブラリロガー関数へのポインタを設定する
 * LERRLOGxマクロはここで設定された関数へのポインタを使って
 * ログ出力を行います。
 * (note)
 * 	複数回コールしても構いませんが、今ログを吐いている
 * 	ならば問題を引き起こします。
 * (Reentrant)
 * 	この関数はスレッドセーフではありません。シングルで
 * 	動いていることを確認してから使用してください。
 *
 * @param lerrlog divy_lerrlog_t エラーログ関数へのポインタ
 */
DIVY_DECLARE(void) divy_set_lerrlogger(divy_lerrlog_t lerrlog);

/**
 * ライブラリロガー関数へのポインタを取得する
 * (Reentrant)
 * 	この関数はスレッドセーフではないかもしれません。
 * 	divy_set_lerrlogger が正しくコールされる場合に
 * 	限り、マルチスレッド環境でも問題を起こしません。
 *
 * @return divy_lerrlog_t エラーログ関数へのポインタ
 */
DIVY_DECLARE(divy_lerrlog_t) divy_get_lerrlogger(void);

/**
 * divy_lerrlog_t 型のロガー(stderr 出力)
 *
 * このロガーはLERRLOGx マクロによるログ出力を標準エラーに出力します。
 * これを使いたければ、関数divy_init_errlog_envを使ってロガーを交換
 * して下さい。
 * 直接使用してはなりません !! (だから引数の説明は書かない)
 * (note)
 * 	引数p はNULLであることがあります。このロガーではプールを使った
 * 	コードを書いてはなりません。
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 */
DIVY_DECLARE(void) divy_output_err2stderr(const char *file, int line,
					int level, apr_status_t status,
					apr_pool_t *p, const char *fmt, ...)
					__attribute__((format(printf,6,7)));

/**
 * divy_lerrlog_t 型のロガー(stdout 出力)
 *
 * このロガーはLERRLOGx マクロによるログ出力を標準出力に出力します。
 * これを使いたければ、関数divy_init_errlog_envを使ってロガーを交換
 * して下さい。
 * 直接使用してはなりません !! (だから引数の説明は書かない)
 * (note)
 * 	引数p はNULLであることがあります。このロガーではプールを使った
 * 	コードを書いてはなりません。
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 */
DIVY_DECLARE(void) divy_output_err2stdout(const char *file, int line,
					int level, apr_status_t status,
					apr_pool_t *p, const char *fmt, ...)
					__attribute__((format(printf,6,7)));

/**
 * エラーログで使用するエラーレベル文字列を返却する
 * (note)
 * 	この関数はApacheCoreのコードをコピーしています。残念ながら
 * 	ライブラリ化を行う上で仕方の無いことでした。
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 *
 * @param lv int エラーログレベル (LLOG_xxx)
 * @return const char * エラーログレベルを表す文字列("error"など)
 */
DIVY_DECLARE(const char *) divy_get_errlevelstr(int lv);


/**
 * エラー情報の記録を行い、エラーログ構造体のインスタンスを生成して返却する。
 * desc メッセージの生成には、DESC0 からDESC3までのマクロを使用して下さい。
 * (ex)
 * 	divy_new_error(p, LLOG_ERR,
 * 			DIVY_FST_IERR + DIVY_SST_DATA, DESC1(p, "aa=%s",a));
 *
 * @param p apr_pool_t * 作業用プール
 * @param level int エラーログレベル
 * @param status int エラーステータス
 * @param funcname char * 関数名
 * @param lineno int 行番号
 * @param desc const char * エラーメッセージ
 * @return divy_error * エラー情報を保持する構造体
 */
DIVY_DECLARE(divy_error *) divy_new_error(apr_pool_t *p, int level, int status,
						char *funcname, int lineno,
						const char *desc);

/**
 * 指定されたエラー情報を記録し、それを指定されたerr と連結して返却する。
 * desc メッセージの生成には、DESC0 からDESC3までのマクロを使用することが
 * 出来ます。
 * (note)
 * 	err 内部ではエラー情報をスタック状に保持しています。
 *
 * @param p apr_pool_t * 作業用プール
 * @param level int エラーログレベル
 * @param status int エラーステータス
 * @param funcname char * 関数名
 * @param lineno int 行番号
 * @param desc const char * エラーメッセージ
 * @param prev divy_error * 前回のエラー
 * @return divy_error * エラー情報を保持する構造体
 */
DIVY_DECLARE(divy_error *) divy_push_error(apr_pool_t *p, int level, int status,
						char *funcname, int lineno,
						const char *desc, divy_error *prev);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_ERRLOG_H */

