/**
 * $Id$
 *
 * tf_misc.h
 *
 * ビジネスロジックフリーであり、かつ"動的に切り離し可能"なユーティリティ
 * 関数及び構造体を定義するヘッダファイル。
 * 取り込まれる機能間には関連性がありませんので、ある決まった一連の機能
 * を提供する必要が生じたなら、別のファイルを新たに作成してそこに実装
 * して下さい。
 * このヘッダで定義される機能はApache本体(ap系関数)には依存しませんので
 * ライブラリとして使用することが出来ます。
 * なお、このヘッダには対応するCファイル(tf_misc.c) がありますが、これには
 * 本当の機能を提供する実装はありません。単なるアクセサであったり、本機能
 * のヘルパであったりするだけです。何故なら、ここに実装してしまっては
 * 切り離せなくなってしまうからです。
 *
 * (note) "動的に切り離し可能"にしなければならない機能とは ?
 * 	主として動作環境や仕様に応じてコンパイル時ではなく実行時に
 * 	動的に切り離したい機能のことをさします。例えば、エンコーディング
 * 	スキームの変換が最初にその候補に選ばれました。何故なら、
 * 	ASCII圏の人たちには7bitコードが扱えれば十分でしょうし、第一
 * 	そのような動作環境にはnバイトコードを扱えるライブラリがあるとは
 * 	思えません。
 * 	"動的に切り離せること"を全ての機能に適用すべきではありません。
 * 	コードが分かりにく、複雑になるからです。これらの実現方式が
 * 	トリッキであることにも原因があります。あまりC言語標準とはいえません。
 */
#ifndef INCLUDE_TF_MISC_H
#define INCLUDE_TF_MISC_H

#include "apr.h"
#include "apr_hooks.h"
#include "tfr.h"

/*-----------------------------------------------------------------------------
  Define Fixed values and macro
  ---------------------------------------------------------------------------*/
/**
 * 独自フックハンドラからの戻り値
 */
#define MISC_OK		0
#define MISC_DECLINED	-1

/*-----------------------------------------------------------------------------
  Define function type
  ---------------------------------------------------------------------------*/
/**
 * エンコーディングスキームの変換を行う関数の型
 */
typedef int (*encode_str_t)(apr_pool_t *, const char *,
				const char *, char *, char **);

/*-----------------------------------------------------------------------------
  Declare public functions
  ---------------------------------------------------------------------------*/
/**
 * 指定されたfunc_t_name を定義するプロバイダであったとき、その機能を提供する
 * 関数を返却する構造体を作成 & 返却する。(セミコロンは付けないで下さい。)
 *
 * (note) このHook関数宣言によって利用できるようになる関数
 *
 *  ・divy_hook_get_miscfunction : フックハンドラとして関数を登録する
 *  ・divy_run_get_miscfunction  : 上記関数で登録されたフック関数を呼び出す
 */
APR_DECLARE_EXTERNAL_HOOK(divy, DIVY, int, get_miscfunction,
		(apr_pool_t *p, const char *func_t_name, void **func_p))

/**
 * fromfmt でエンコードされている文字列fromstr を、tofmtでエンコードされている
 * 文字列tostrに変換して返却する。
 *
 * @param p apr_pool_t * 返却文字列を割り当てるプールへのポインタ
 * @param fromfmt const char * 変換前のエンコード
 * @param tofmt const char *   変換後のエンコード
 * @param fromstr char * 変換前の文字列
 * @param tostr char **  変換後の文字列
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_misc_encode_str(apr_pool_t *p, const char *fromfmt,
				const char *tofmt, char *fromstr, char **tostr);



#endif	/* INCLUDE_TF_MISC_H */

