/**
 * $Id$
 *
 * メールプロバイダアクセス用ヘッダファイル
 * tf_ml.h
 *
 */
#ifndef INCLUDE_DIVY_ML_H
#define INCLUDE_DIVY_ML_H 

/* Apache header files */
#include "apr.h"
#include "apr_pools.h"
#include "apr_hooks.h"
#include "tfr.h"

/*--------------------------------------------------------------
  Define fixed values and define types.
  --------------------------------------------------------------*/

/* 
 * 実行ステータスコードの値 --> __code変数を参照
 */
#define ML_SUCCESS  0	/* 成功した                      */
#define ML_ERROR   -1	/* 何らかの理由により失敗した    */

/**
 * 独自フックハンドラからの戻り値
 */
#define ML_OK		0
#define ML_DECLINED	-1

/*
 * Content-Type のデフォルト値
 */
#define ML_DEFAULT_CONTENT_TYPE "text/plain; charset=\"iso-2022-jp\""

/*
 * MIME-Version: ヘッダの値
 */
#define ML_MIME_VERSION "1.0"

/*
 * SMTP Extensions: SMTP拡張
 */
#define _BIT(n)	(1 << (n))

#define DIVY_ML_EX_ENHANCEDSTATUSCODES	_BIT(0)		/* RFC 1893 RFC 2034 */
#define DIVY_ML_EX_PIPELINING		_BIT(1)		/* RFC 2920 */
#define DIVY_ML_EX_DSN			_BIT(2)		/* RFC 1891 */
#define DIVY_ML_EX_AUTH			_BIT(3)		/* RFC 2554 */
#define DIVY_ML_EX_STARTTLS		_BIT(4)		/* RFC 3207 */
#define DIVY_ML_EX_SIZE			_BIT(5)		/* RFC 1870 */
#define DIVY_ML_EX_CHUNKING		_BIT(6)		/* RFC 3030 */
#define	DIVY_ML_EX_BINARYMIME		_BIT(7)		/* RFC 3030 */
#define DIVY_ML_EX_8BITMIME		_BIT(8)		/* RFC 1652 */
#define DIVY_ML_EX_DELIVERBY		_BIT(9)		/* RFC 2852 */
#define DIVY_ML_EX_ETRN			_BIT(10)	/* RFC 1985 */
#define DIVY_ML_EX_XUSR			_BIT(11)	/* sendmail */
#define DIVY_ML_EX_IMMEDIATE_SSL	_BIT(12) /* SSLConnect後回し */

/*
 * SMTPのコマンド定義
 * 単純に名前をきめているだけです。
 */

#define CMD_QUIT _BIT(0)
#define CMD_HELP _BIT(1)
#define CMD_NOOP _BIT(2)
#define CMD_VRFY _BIT(3)
#define CMD_EXPN _BIT(4)
#define CMD_HELO _BIT(5)
#define CMD_MAIL _BIT(6)
#define CMD_RCPT _BIT(7)
#define CMD_DATA _BIT(8)
#define CMD_RSET _BIT(9)
#define CMD_SEND _BIT(10)	/* rfc2821で廃止 */
#define CMD_SOML _BIT(11)	/* rfc2821で廃止 */
#define CMD_SAML _BIT(12)	/* rfc2821で廃止 */
#define CMD_TURN _BIT(13)
#define CMD_EHLO _BIT(14)	/* 必須になりました */
#define CMD_AUTH _BIT(15)
#define CMD_STARTTLS _BIT(16)

/*
 * コマンドとメッセージの長さCRLF含む
 */
#define ML_CMDLEN 512
#define ML_MSGLEN 512

/*
 * ヘッダの一行のサイズ
 */
#define ML_HEADER_SIZE 76

/*
 * TeamFileメーラーX-Mailer
 */
#define ML_HEADER_X_MAILER	"TeamFile Mailer"

/*
 * デフォルトのプロトコルタイムアウト(sec)
 */
enum __ml_socket_timeout {
	TIMEOUT_GREETING_DEFAULT = (5 * 60), 
	TIMEOUT_ENVELOPE_DEFAULT = (5 * 60),
	TIMEOUT_DATA_DEFAULT	 = (2 * 60),
	TIMEOUT_TRANSFER_DEFAULT = (3 * 60),
	TIMEOUT_DATAQUIT_DEFAULT = (10* 60)
};

/*--------------------------------------------------------------
  Declare handler structure
  --------------------------------------------------------------*/
typedef struct __ml_DataSource MlDataSource;
typedef struct __ml_Session MlSession;
typedef struct __ml_Property MlProperty;
typedef struct __ml_message MlMessage;
typedef struct __ml_Address MlAddress;

/* 
 * MlDataSource の処理に必要で、かつ特定のメールプロバイダに依存する
 * 情報を保持するコンテキスト
 * 必要であれば、メールプロバイダ側で定義してください。
 */
typedef struct __ml_dsContext MldsCntxt;

/*
 * MlSession の処理に必要で、かつ特定のメールプロバイダに依存する
 * 情報を保持するコンテキスト。
 * 必要であれば、メールプロバイダ側で定義してください。
 */
typedef struct __ml_sesionCntxt MlSessionCntxt;

/*
 * MlMessage の処理に必要で、かつ特定のメールプロバイダに依存する
 * 情報を保持するコンテキスト。
 * 必要であれば、メールプロバイダ側で定義してください。
 */
typedef struct __ml_messageCntxt MlMessageCntxt;

/*
 * サーバより返却されるレスポンスデータを保持している構造体
 */
typedef struct __ml_response MlResponse;

/*--------------------------------------------------------------
  Define structures
  --------------------------------------------------------------*/

/**
 * サーバレスポンスを保持する。
 */
struct __ml_response {
	char	   *line;	/* サーバレスポンス１ラインすべて	*/
	MlResponse *next;	/* サーバレスポンスの次へのポインタ	*/
};


/**
 * 接続情報などのプロパティ情報を保持する。
 */
struct __ml_Property {
	char *hostname;	/* メールサーバの接続先ホスト名   */
	char *hostport;	/* メールサーバの接続先ポート番号 */
	char *adminaddr;/* 管理者ユーザのメールアドレス   */
	char *username;	/* (予約) */
	char *password;	/* (予約) */
	char *localhost;/* サーバのローカルホスト名称 */
	int	 interval;	/* 送信を行う間隔(秒指定) */
};

/**
 * メールメッセージを表す。
 * List 構造をとっているため、複数のメッセージも表現できる。
 * List の終端はNULLとする。
 *
 */
struct __ml_message {
	
	/* メールの件名(QまたはBエンコードされていない生の文字列) */
	char *subject;

	/*
	 * 特定のメールプロバイダに依存したデータを保持する構造体へのポインタ
	 * ポインタの指し示す先を気にする必要はまったくない。
	 */
	MlMessageCntxt *msgcntxt;
	
	/* 優先度 */
#define ML_PRIORITY_HIGH   1	/* 高い */
#define ML_PRIORITY_NORMAL 3	/* 普通 */
#define ML_PRIORITY_LOW    5	/* 低い */
	int priority;
	
	/* ヘッダ(アドレス,Subject) のエンコーディング形式 */
#define ML_ENCODING_NOP 0	/* エンコードしない  */
#define ML_ENCODING_B   1	/* Bエンコーディング */
#define ML_ENCODING_Q   2	/* Qエンコーディング */
	int header_encoding;
	
	/* MIME タイプ (Content-Type) */
#define ML_MIMETYPE_TEXT_PLAIN "text/plain"
#define ML_MIMETYPE_TEXT_HTML  "text/html"
#define ML_MIMETYPE_TEXT_XML   "text/xml"
	char *mime_type;
	
	/* charsetパラメータ(Content-Type) */
#define ML_CHARSET_ISO_2022_JP "iso-2022-jp"
#define ML_CHARSET_ISO_8859_1  "iso-8859-1"
#define ML_CHARSET_UTF_8       "utf-8"
#define ML_CHARSET_UTF8        "utf8"
#define ML_CHARSET_SHIFT_JIS   "shift_jis"
	char *charset;

	/* ヘッダ */
	apr_table_t *headers;

	/* From: に指定されたメールアドレス */
	MlAddress *from_addr;
	
	/* To: に指定されたメールアドレス */
	MlAddress *to_addr;
	
	/* Cc: に指定されたメールアドレス */
	MlAddress *cc_addr;
	
	/* Bcc: に指定されたメールアドレス */
	MlAddress *bcc_addr;
	
	/* Replay-To: に指定されたメールアドレス */
	MlAddress *replyto_addr;
	
	/* メールの本文(エンコードされていない生の文字列) */
	char *body;

	/* メールの本文のフッダー */
	char *bodyfodder;

	/* エンコードのリトライ */
	int encretry;

	/* 次の MlMessage へのポインタ(終端ならばNULL) */
	MlMessage *next;
};

/**
 * メールアドレスを表す。
 * List 構造をとっているため、複数のメッセージも表現できる。
 * List の終端はNULLとする。
 *
 */
struct __ml_Address {
	
	/* システム内でのユーザID */
	char *userid;
	/* メールアドレスの表示名を含まないメールアドレス */
	char *addr;
	/* メールアドレスの表示名(エンコードされていない) */
	char *displayname;
	
	/* 次の MlAddress へのポインタ(終端ならばNULL) */
	MlAddress *next;
};
 
/*--------------------------------------------------------------
  Define handler structure
  --------------------------------------------------------------*/

/**
 * メールサーバを１つのデータソースと見立てるために使用される構造体。
 *
 * このオブジェクトはメールサーバとのセッションを表すMlSessionの
 * Factoryとしての役割を持っています.
 * 複数のユーザ間で共有されますので、特定のユーザに関係する情報
 * (例えば、認証情報)を持つことは出来ません。それらは、MlSession に
 * 保持すべきです。
 * 
 */
struct __ml_DataSource {
	
	const char *type;	/* メールプロバイダの種類 */
	
	/*
	 * 特定のメールプロバイダに依存したデータを保持する構造体へのポインタ
	 * ポインタの指し示す先を気にする必要はまったくない。
	 */
	MldsCntxt *dscontxt;
	
	/**
	 * 実行ステータスメッセージ
	 * メールサーバから受け取ったメッセージまたは、メールプロバイダが
	 * 書き込んだメッセージを保持する。
	 * 参照には getMsg関数を必ず使用してください。
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード
	 * メールプロバイダに対してアクションを行い、その結果に基づいて
	 * 決定されるステータス値。
	 * このコードは、メールプロバイダの種類を意識しないモジュールが
	 * 次に取らなければならない動作を決定する際の判断材料となります。
	 * 受け取り側は、必ず getCode 関数を使用して取得して下さい。
	 */
	int __code;
	
	/**
	 * メールサーバから返却された固有のコード文字列
	 * 主としてエラーコードが格納されます.
	 * 受け取り側は、必ず getNativeCode関数を使用してください.
	 */
	char *__nativecode;
	
	/**
	 * あるメールサーバとのセッションを表すオブジェクトを取得する。
	 * (note)
	 *  SMTPプロトコルを扱う場合には、getMlSession でMTAとの接続を
	 *  行う必要はない。
	 *
	 * @param mlds const MlDataSource* 自分自身へのポインタ
	 * @param mlprop const MlProperty * 接続状態を保持する構造体
	 * @param  p apr_pool_t * 変数割り当てに使用するプール(request_rec)
	 * @return DbConn* データベースとの接続を表す構造体へのポインタ
	 */
	MlSession * (*getMlSession)(MlDataSource *mlds, const MlProperty *mlprop,
					apr_pool_t *p);
	
	/**
	 * 実行ステータスコードを取得する
	 *
	 * @param mlds const MlDataSource* 自分自身へのポインタ
	 * @return int 実行ステータスコード(define値を参照)
	 */
	int (*getCode)(const MlDataSource *mlds);
	
	/**
	 * メールプロバイダ固有のコード文字列を取得する
	 *
	 * @param mlds const MlDataSource* 自分自身へのポインタ
	 * @return char* コードを表す文字列。整数であるとは限りません。
	 */
	char * (*getNativeCode)(const MlDataSource *mlds);
	
	/**
	 * 実行ステータスメッセージを取得する
	 *
	 * @param mlds const MlDataSource* 自分自身へのポインタ
	 * @return char* メッセージを表す文字列。
	 */
	char * (*getMsg)(const MlDataSource *mlds);
};


/**
 * MiDataSource が表すメールサーバに対する１つのメールセッションを表す。
 *
 */
struct __ml_Session {
	
	/* メールサーバに対する接続情報 */
	MlProperty *mlprop;
	
	/**
	 * 特定のメールプロバイダに依存したデータを保持する構造体へのポインタ
	 * ポインタの指し示す先を気にする必要はまったくない。
	 */
	MlSessionCntxt *scntxt;
	
	/**
	 * このセッションを作成したFactory (MlDataSource)へのポインタ
	 */
	MlDataSource *mlds;
	
	/**
	 * 実行ステータスメッセージ (MlDataSourceの__msgと同じ役割)
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード (MlDataSourceの__msgと同じ役割)
	 */
	int __code;
	
	/**
	 * 固有のコード文字列 (MlDataSourceの__msgと同じ役割)
	 */
	char *__nativecode;
	
	/*
	 * セッション内で利用できるSMTP拡張
	 */
	apr_int64_t __extention;

	/**
	 * 指定されたmlmsg が示すメールメッセージを送信する。
	 *
	 * 送信開始と同時にメールサーバとの接続を開き、終了と同時に
	 * 接続を閉じます。
	 * 送信先は、mlsession が保持している。
	 *
	 * @param mlsession MlSession * 自分自身へのポインタ
	 * @param mlmsg MlMessage * 送信対象のメッセージ
	 * @param int 処理ステータス (ML_SUCCESS, ML_ERROR)
	 */
	int (*send)(MlSession *mlsession, MlMessage *mlmsg);
	
	/**
	 * 実行ステータスコードを取得する
	 *
	 * @param mlsession const MlSession* 自分自身へのポインタ
	 * @return int 実行ステータスコード(define値を参照)
	 */
	int (*getCode)(const MlSession *mlsession);
	
	/**
	 * メールプロバイダ固有のコード文字列を取得する
	 *
	 * @param mlsession const MlSession* 自分自身へのポインタ
	 * @return char* コードを表す文字列。整数であるとは限りません。
	 */
	char * (*getNativeCode)(const MlSession *mlsession);
	
	/**
	 * 実行ステータスメッセージを取得する
	 *
	 * @param mlsession const MlSession* 自分自身へのポインタ
	 * @return char* メッセージを表す文字列。
	 */
	char * (*getMsg)(const MlSession *mlsession);
};


/*--------------------------------------------------------------
  Declare Public Functions
  --------------------------------------------------------------*/
/**
 * 指定されたproviderType に一致するメールプロバイダであった時、
 * メールプロバイダ構造体を作成 & 返却する関数
 * のためのHook関数宣言。
 * メールプロバイダの初期化に利用します。
 *
 * (note) このHook関数宣言によって利用できるようになる関数
 *
 *  ・divy_hook_create_mldatasource : フックハンドラとして関数を登録する
 *  ・divy_run_create_mldatasource  : 上記関数で登録されたフック関数を呼び出す
 */
APR_DECLARE_EXTERNAL_HOOK(divy, DIVY, int, create_mldatasource,
		(apr_pool_t *p, const char *providerType,  MlDataSource **mlds))

#endif	/* INCLUDE_DIVY_ML_H */

