/**
 * $Id$
 *
 * tf_provider.h
 *
 * xxx プロバイダライブラリの管理を行う構造体・関数の定義
 */
#ifndef INCLUDE_TF_PROVIDER_H
#define INCLUDE_TF_PROVIDER_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_hooks.h"
#include "tfr.h"

/* ステータス */
#define DIVY_LIB_ST_OK	0	/* 正常 */
#define DIVY_LIB_ST_ERR 1	/* 異常 */

/* インターフェースバージョン */
#define DIVY_LIB_IF_VERSION 1

/*-----------------------------------------------------------------------------
  Define fiexd value
 -----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
  Declare structures and types
 -----------------------------------------------------------------------------*/
typedef struct divy_lib_provider_t divy_lib_provider_t;

/*-----------------------------------------------------------------------------
  Declare function and Define structure
 -----------------------------------------------------------------------------*/
/*
 * xxx プロバイダライブラリフック構造体
 * (役割)
 *	xxx プロバイダライブラリとTeamFileモジュール本体との間を繋ぐインターフェース。
 *	TeamFileモジュール本体にxxx プロバイダライブラリが依存してしまわないように
 *	するための役割も持っています。
 *	"xxx" というのは "db"や"ml"に読み替えてください。
 */
struct divy_lib_provider_t {
	/**
	 * インターフェースのバージョン番号
	 *
	 * このメンバは、各プロバイダライブラリとインターフェースとの
	 * バージョン間の不一致が起きないことを保証する目的で使用します。
	 * このバージョン番号は、インターフェースのメンバが変更になったときに
	 * インクリメントされます。インターフェース間のバージョン番号が厳密に
	 * 一致しなければ古いバージョン番号を持つプロバイダモジュールは
	 * 使用できません。
	 * (note)
	 * 	このメンバの位置は"絶対に"変更してはなりません。必ず構造体の
	 * 	第1メンバでなければなりません。
	 */
	apr_int32_t version;

	/**
	 * 次のプロバイダライブラリフックへのポインタ
	 */
	divy_lib_provider_t *next;

	/**
	 * プロバイダライブラリのタイプ
	 * (値の種類)
	 * 	"db"	: DB プロバイダライブラリ
	 * 	"ml"	: メール プロバイダライブラリ
	 * 	"misc"	: その他の他雑なプロバイダライブラリ
	 * (note)
	 * 	define 値を使って定義しないのは、DIVY_PROVIDER_HOOK_STRUCT マクロ
	 * 	の問題です。
	 */
	const char *type;

	/**
	 * プロバイダライブラリを識別名称
	 * 例えば、DBプロバイダライブラリでは、"pg" や "ora" がこれに当たります。
	 */
	const char *name;

	/**
	 * 初期化ハンドラ関数へのポインタ
	 *
	 * この関数は、Apacheが起動してから唯１度だけコールされる
	 * コールバック関数です。
	 * 特別な初期化処理は、DBプロバイダライブラリのような
	 * APRコールバックの登録が必要な場合、この関数へのポインタに
	 * コールバック関数を登録しておくことで実施できます。
	 * (note)
	 * 	pconf はChildプロセスのプールよりもグローバルな
	 * 	プールです。このプール領域はApacheがダウンするまで
	 * 	解放されないため慎重に使用する必要があります。
	 *
	 * @param pconf apr_pool_t * コンフィグプールへのポインタ
	 * @return int 処理ステータス
	 * 	DIVY_LIB_ST_OK  : 正常
	 * 	DIVY_LIB_ST_ERR : 失敗
	 */
	int (*init_func)(apr_pool_t *pconf);

	/**
	 * 終了ハンドラ関数へのポインタ
	 *
	 * この関数は、Apacheが終了するとき唯１度だけコールされる
	 * コールバック関数です。
	 *
	 * @return int 処理ステータス
	 * 	DIVY_LIB_ST_OK  : 正常
	 * 	DIVY_LIB_ST_ERR : 失敗
	 */
	int (*fini_func)(void);

	/**
	 * 初期化済みであるかどうかを示す
	 *
	 * 0: 未初期化 / 1: 初期化済み
	 */
	apr_int32_t is_init;

	/**
	 * 予約
	 * (note)
	 * 	以下のメンバは、xxx プロバイダ本体を再コンパイルせずに
	 * 	この構造体のメンバを動的に増やす目的で使用するダミーメンバです。
	 * 	構造体のサイズが変更されないよう領域を予約しているだけです。
	 * 	従って、任意のデータをここに入れてはなりません。
	 */
	void *data1;
	void *data2;
};

/* xxxプロバイダライブラリ構造体登録関数 */

/**
 * プロバイダライブラリマネージャの初期化
 * (呼び出しタイミング)
 *	Apacheモジュールと共に動作する場合、post config ステージで
 *	呼び出して下さい。
 * (note)
 * 	多重呼び出し、同期呼び出しは禁止です。呼び出し元にて保証して下さい。
 *
 * @param pconf apr_pool_t * コンフィグプール
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
DIVY_DECLARE(int) divy_init_providers_env(apr_pool_t *pconf);

/**
 * xxxプロバイダライブラリハンドラをシステムに登録する。
 *
 * @param provider divy_lib_provider_t * プロバイダライブラリ構造体へのポインタ
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
DIVY_DECLARE(int) divy_register_lib_provider(divy_lib_provider_t *provider);

/**
 * static にリンクされるプロバイダライブラリのハンドラ構造体を
 * 取り出して、ローディングする
 * (note)
 * 	static リンクされるプロバイダライブラリが存在しなければこの関数は
 * 	何もしません。
 *
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
DIVY_DECLARE(int) divy_register_preloaded_lib_provider(void);

/**
 * システムに登録されている全てのxxx プロバイダライブラリの初期化ハンドラを
 * コールバックする。
 * (note)
 * 	この関数コールの前にdivy_register_preloaded_lib_provider をコール
 * 	する必要があります。しなければstatic リンクされたxxx プロバイダ
 * 	ライブラリの初期化ハンドラはコールバックされません。
 * (note)
 * 	複数回コールされないよう呼び出し元で保証して下さい。
 *
 * @param pconf apr_pool_t * コンフィグプールへのポインタ
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
DIVY_DECLARE(int) divy_run_lib_init_func(apr_pool_t *pconf);

/**
 * システムに登録されている全てのxxx プロバイダライブラリの終了ハンドラを
 * コールバックする。
 * (note)
 * 	この関数は、通常divy_run_lib_init_func のコールで自動的に、
 * 	Apacheの終了時に呼び出されるよう調整されます。
 * 	多くの場合、この関数をこのtf_provider.c 以外から呼び出す
 * 	必要はありません。
 * (note)
 * 	外からコールするのなら、複数回コールされないよう呼び出し元で
 * 	保証して下さい。
 *
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
DIVY_DECLARE(int) divy_run_lib_fini_func(void);


/*
 * init_funcハンドラでコール可能な登録関数の宣言
 */
APR_DECLARE_EXTERNAL_HOOK(divy, DIVY, void, child_init, (apr_pool_t *pchild))
typedef apr_status_t (*divy_lib_child_exit_t)(void *);
DIVY_DECLARE(void) divy_hook_child_exit(divy_lib_child_exit_t child_exit,
					apr_pool_t *pchild, void *data); 
/**
 * 何もしない初期化ハンドラ
 *
 * @return int 処理ステータス(DIVY_LIB_ST_OK)
 */
int divy_lib_nop_init_func(apr_pool_t *pconf);

/**
 * 何もしない終了ハンドラ
 *
 * @return int 処理ステータス(DIVY_LIB_ST_OK)
 */
int divy_lib_nop_fini_func(void);

/*-----------------------------------------------------------------------------
  Declare values 
 -----------------------------------------------------------------------------*/
DIVY_DECLARE_DATA extern divy_lib_provider_t *divy_preloaded_lib_providers[];

/*-----------------------------------------------------------------------------
  Define macros
 -----------------------------------------------------------------------------*/
/**
 * xxxプロバイダライブラリ構造体登録マクロ
 */
/* 構造体の名前 */
#define DIVY_PROVIDER_HOOK_STRUCT_NAME(type,name)      \
	divy_lib_##type##_##name##_provider

/* 構造体の定義 */
#define DIVY_PROVIDER_HOOK_STRUCT(type,name,init_func,fini_func)    \
	divy_lib_provider_t DIVY_DECLARE_DATA \
	DIVY_PROVIDER_HOOK_STRUCT_NAME(type,name) = {   \
		DIVY_LIB_IF_VERSION,	\
		NULL,	\
		#type,	\
		#name,	\
		&init_func,   \
		&fini_func,   \
		0,	\
		NULL,	\
		NULL	\
	}

#endif	/* INCLUDE_TF_PROVIDER_H */

