/**
 * $Id$
 *
 * 可変長文字列バッファの宣言・定義を行うヘッダファイル
 * (note)
 * 	この宣言は元々util_common.h にあったものです。
 * 	ぐちゃぐちゃになったので移しました。
 */
#ifndef INCLUDE_TF_SBUF_H
#define INCLUDE_TF_SBUF_H

#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"

#include "tfr.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*------------------------------------------------------------------------------
  Declare incomplete type structure
  ----------------------------------------------------------------------------*/
typedef struct __divy_sbuf	divy_sbuf;

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 指定されたsize バイトの文字列を格納可能な文字列バッファsbuf を生成して
 * 返却する。
 * 確保される文字列バッファには文字列終端文字'\0' を格納する領域が自動的に
 * 追加されます。(つまり size + 1バイト分確保されるということ)
 *
 * バッファが足りなくなったときの増分(delta) は以下のように決定します。
 * (1) size/2 が デフォルト増加サイズ よりも小さいときには
 * 	デフォルト増加サイズを使用する
 * (2) size/2 が デフォルト増加サイズよりも大きい時には
 * 	size/2 が使用される
 * (note)
 * 	size が負の時には、デフォルト値(> 0) のバッファを確保します。
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t * 文字列バッファを割り当てるプール
 * @param sbuf divy_sbuf ** 文字列バッファ
 * @param size apr_size_t 文字列バッファのサイズ(byte)
 */
DIVY_DECLARE(void) divy_sbuf_create(apr_pool_t *p, divy_sbuf **sbuf,
							apr_size_t size);

/**
 * 指定されたstr が示す文字列の長さを初期値としての文字列バッファsbuf を
 * 初期化して返却する。
 * 確保される文字列バッファは strlen(str) + 1 バイトとなります。
 * つまり、str の文字列全てを保持できるだけのバッファを確保します。
 * なお、str がNULLの場合にはデフォルトサイズのバッファが確保されます。
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t * 文字列バッファを割り当てるプール
 * @param sbuf divy_sbuf ** 文字列バッファ
 * @param str const char * 最初にセットする文字列
 */
DIVY_DECLARE(void) divy_sbuf_createbystr(apr_pool_t *p, divy_sbuf **sbuf,
							const char *str);

/**
 * 文字列バッファsbuf に設定された文字列バッファの現在のサイズ(byte)を取得する。
 * (note)
 * 	格納されている文字列の長さではありません。バッファ全体の長さです。
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @return apr_size_t 文字列バッファの現在の長さ(byte)
 */
DIVY_DECLARE(apr_size_t) divy_sbuf_getcapacity(divy_sbuf *sbuf);

/**
 * 文字列バッファsbuf に追加された文字列の総長さ(byte)を取得する。
 * (note)
 * 	divy_sbuf_getcapacity から取得される値とは異なることがあります。
 * 	また、'\0'分はカウントされません。(strlenと同じです)
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @return apr_size_t 追加された文字列の総長さ(byte)
 */
DIVY_DECLARE(apr_size_t) divy_sbuf_getlength(divy_sbuf *sbuf);

/**
 * 文字列バッファsbuf に最後に追加された文字列の後ろににstr が示す文字列を
 * 追加する。
 * str がNULLの時には何もしません。
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @param str const char * 追加する文字列
 */
DIVY_DECLARE(void) divy_sbuf_append(divy_sbuf *sbuf, const char *str);

/**
 * 文字列バッファsbuf に最後に追加された文字列の後ろににstr が示す文字列を
 * len バイトだけ追加する。str がNULLやlen が0 の時には何もしません。
 * (note) divy_sbuf_append との違い
 * 	この関数は、文字列長さが呼び出し側で既に判明している場合(例えば
 * 	固定値を入れるなど)にdivy_sbuf_appendよりもstrlenを計算しない分だけ
 * 	高速になります。(僅かですが。)
 * 	それ以外の処理はdivy_sbuf_append と同じです。
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @param str const char * 追加する文字列
 */
DIVY_DECLARE(void) divy_sbuf_appendbyte(divy_sbuf *sbuf, apr_size_t len,
							const char *str);

/**
 * 文字列バッファsbuf のstart が示すインデックス位置に文字列str を追加する。
 * start位置にあった文字はstrlen(str) 分だけ後ろに移動されます。
 * (例)
 * 	sbuf: "01234abcd"の時
 * 	divy_sbuf_insert(sbuf, 3, "xxxxx") -> "012xxxxx34abcd"
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @param start apr_size_t インデックス位置
 * @param str const char * 追加する文字列
 * @return int 処理ステータス (0: 正常 / 1: 異常)
 * 	divy_sbuf_getlength() の結果よりもstartが大きいとき、または
 * 	startが負の値の時には 1 を返却して異常終了します。
 */
DIVY_DECLARE(int) divy_sbuf_insert(divy_sbuf *sbuf, apr_size_t start,
							const char *str);

/**
 * 文字列バッファsbuf のstart が示すインデックス位置からインデックス位置
 * (end - 1) までをstr が示す文字列で置換する
 * str が 置換長さよりも長い場合には切り取って挿入します。
 * また、str が 置換長さよりも短い場合には、strlen(str) までをコピーします。
 * end がバッファ長さよりも大きい場合にはバッファを自動拡張します。
 * (例)
 * 	sbuf: "01234abcd"の時
 * 	divy_sbuf_replace(sbuf, 2, 5, "xxxxx") -> "01xxxabcd"
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @param start apr_size_t 置き換え先頭インデックス位置
 * @param end apr_size_t   置き換え終端位置のバイト数
 * @param str const char * 置き換える文字列
 * @return int 処理ステータス (0: 正常 / 1: 異常)
 * 	divy_sbuf_getlength() の結果よりもstartが大きいとき、または
 * 	startが負の値の時には 1 を返却して異常終了します。
 */
DIVY_DECLARE(int) divy_sbuf_replace(divy_sbuf *sbuf, apr_size_t start,
					apr_size_t end, const char *str);

/**
 * 文字列バッファsbuf から格納されている全ての文字列を取得する。
 * (note)
 * 	取得される文字列は、新しい領域にコピーされません。
 * 	バッファが持っていたポインタを返却するだけです。
 * 	そのため、divy_sbuf_tostring で取得した文字列をコピーせずに
 * 	divy_sbuf_clear を実施すると、取得文字列は空になってしまいます。
 * 	divy_sbuf_clear との組み合わせでは注意が必要です。
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @return char * 取得した文字列
 */
DIVY_DECLARE(char *) divy_sbuf_tostring(divy_sbuf *sbuf);

/**
 * 文字列バッファsbuf に格納された文字列を全て消去する。
 * ただし、確保されたバッファ領域はクリアされません。
 * (note)
 * 	divy_sbuf_tostring 関数との組み合わせでは注意が必要です。
 * (Reentrant)
 * 	各スレッドが別々のsbuf を持つ場合に限りこの関数はスレッドセーフです。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 */
DIVY_DECLARE(void) divy_sbuf_clear(divy_sbuf *sbuf);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_SBUF_H */
