/**
 * $Id$
 *
 * tf_vscan.h
 *
 * ウィルス検索プロバイダインターフェースの宣言、定義を行うヘッダファイル。
 *
 * このヘッダには、
 * 	* ウィルス検索プロバイダにアクセスするための関数及び構造体の宣言
 * 	* ウィルス検索プロバイダが提供すべきサービスのインターフェース宣言
 *
 * があります。
 *
 * (note)
 * 	このヘッダにはTeamFileモジュールおよびApache本体固有のヘッダをinclude
 * 	してはなりません。(mod_davやhttpd.hなど)
 * 	includeしてよいものは、apr_xxx とTeamFile"モジュール"に依存しない共通
 * 	ヘッダ(tfr.hなど)だけです。
 *
 * 2004/10/04 Mon takehara NEW
 */
#ifndef INCLUDE_TF_VSCAN_H
#define INCLUDE_TF_VSCAN_H 

#include "apr.h"
#include "apr_pools.h"
#include "apr_hooks.h"
#include "apr_time.h"
#include "apr_strings.h"

#include "tfr.h"

#ifdef APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/*------------------------------------------------------------------------------
  Define fixed values ,define types and macro
  ----------------------------------------------------------------------------*/
/**
 * 独自フックハンドラからの戻り値
 */
#define VSC_OK		0
#define VSC_DECLINED	-1

/**
 * ウィルスパターンデータベースのリフレッシュ時間間隔[sec]
 *
 * ウィルス検索エンジンがパターンデータベースのリフレッシュをサポートしていれば
 * この時間間隔でリフレッシュされます。(正確には、リフレッシュを指示するだけ)
 */
#define DIVY_VSC_DEFAULT_REFRESH_INTERVAL	APR_TIME_C(1800)

/**
 * 関数 divy_vsc_getCode, divy_vsc_getNativeCode, divy_vsc_getMsg で使用する
 * 構造体の種類を表す列挙型
 */
enum {
	DIVY_VSC_TYPE_DATASOURCE = 0,
	DIVY_VSC_TYPE_SESSION,
};

/**
 * 実行メッセージ、ネイティブメッセージのサイズ(byte)
 */
#define DIVY_VSC_MSG_SIZE	512

/**
 * 実行ステータスコード＆メッセージ用データを作成するマクロ
 * (ウィルス検索プロバイダ専用)
 *
 * 各種ハンドラ構造体(VscDataSource, VscSession) を生成する時のみに
 * 使用して下さい。
 */
#define DIVY_VSC_NEW_STATUS(data,p) \
{ \
	if ((data) != NULL) { \
		(data)->__code = DIVY_VSC_ST_OK; \
		(data)->__msg  = apr_pcalloc(p, sizeof(char) * DIVY_VSC_MSG_SIZE); \
		(data)->__nativecode  = apr_pcalloc(p, sizeof(char) * DIVY_VSC_MSG_SIZE); \
	} \
}

/**
 * ステータスコード＆ステータスメッセージを初期化するマクロ
 * (ウィルス検索プロバイダ専用)
 *
 * 各種ハンドラ構造体のコール前に呼んでください。
 */
#define DIVY_VSC_INIT_STATUS(data) \
{ \
	if ((data) != NULL) { \
		(data)->__code = DIVY_VSC_ST_OK; \
		if ((data)->__msg != NULL) { \
			memset((data)->__msg, '\0', sizeof(char) * DIVY_VSC_MSG_SIZE); \
		} \
		if ((data)->__nativecode != NULL) { \
			memset((data)->__nativecode, '\0', sizeof(char) * DIVY_VSC_MSG_SIZE); \
		} \
	} \
}

/**
 * 実行ステータスコード(__code) に値を埋めるマクロ
 * (ウィルス検索プロバイダ専用)
 */
#define DIVY_VSC_SET_STATUS_CD(data,cd) \
	if ((data) != NULL) (data)->__code = cd

/**
 * ネイティブコード(__nativecode) に値を埋めるマクロ
 */
#define DIVY_VSC_SET_STATUS_NCD(data, ncd) \
	if ((data) != NULL && (data)->__nativecode != NULL && ncd != NULL) \
		apr_cpystrn((data)->__nativecode, ncd, DIVY_VSC_MSG_SIZE)

/**
 * 実行ステータスメッセージ(__msg) に値を埋めるマクロ
 */
#define DIVY_VSC_SET_STATUS_MSG(data, msg) \
	if ((data) != NULL && (data)->__msg != NULL && msg != NULL) \
		apr_cpystrn((data)->__msg, msg, DIVY_VSC_MSG_SIZE)


/*------------------------------------------------------------------------------
  Define structure/enum
  ----------------------------------------------------------------------------*/
/**
 * ステータスコード
 */
enum {
	DIVY_VSC_ST_OK = 0,		/* 成功 */
	DIVY_VSC_ST_INVALID_PARAMS,	/* 引数が不正だった */
	DIVY_VSC_ST_NOTFOUND_VSCDS,	/* プロバイダが存在しなかった */

	DIVY_VSC_ST_LOADERR = 1000,	/* パターンデータのロードに失敗した   */
	DIVY_VSC_ST_UNSUPPORTED_LOAD,	/* ロードオペレーションを未サポート   */
	DIVY_VSC_ST_NO_CONNECTED,	/* 検索エンジンに接続できなかった     */
	DIVY_VSC_ST_SUPPORTED_CACHE,	/* このセッションはキャッシュ可能     */
	DIVY_VSC_ST_UNSUPPORTED_CACHE,	/* このセッションはキャッシュ不可能   */
	DIVY_VSC_ST_SUPPORTED_STREAM,	/* ストリーム検索が利用出来る         */
	DIVY_VSC_ST_UNSUPPORTED_STREAM,	/* ストリーム検索は利用できない       */

	/*
	 * ウィルス検索時に発生するステータス
	 */
	DIVY_VSC_ST_INFECTED = 3000,	/* ウィルス感染を発見した             */
	DIVY_VSC_ST_REPAIRED,		/* ウィルス感染部分を修復できた       */
	DIVY_VSC_ST_DELETED,		/* ウィルス感染ファイルを削除した     */

	DIVY_VSC_ST_NO_REPAIRED,	/* 感染を検出したが修復出来なかった   */
	DIVY_VSC_ST_TOOLARGE_CONTENT,	/* 検索できないサイズのファイルだった */
	DIVY_VSC_ST_UNSUPPORTED_TYPE,	/* 検索できない種類のファイルだった   */
	DIVY_VSC_ST_CANNOT_OPEN,	/* 対象ファイルをopen出来なかった     */
	DIVY_VSC_ST_CANNOT_READ_FILEST,	/* 対象ファイルのステータス取得に失敗 */
	DIVY_VSC_ST_UNSUPPORTED_ACTION, /* サポートしていないaction だった    */

	/*
	 * その他
	 */
	DIVY_VSC_ST_ERR = 9999,		/* 予期しないエラー */
};

/**
 * ウィルス発見時にプロバイダが取るべきアクションの種類。
 * この種類はウィルス検索を依頼する側が指定します。
 * (note)
 * 	一般的なウィルス検索エンジンが持っている"駆除する"と
 * 	"リネームする"という検索オプションはサポートしていません。
 * 	我々にとって不都合だからです。
 */
enum __vsc_Action {
	DIVY_VSC_ACT_NOTICE = 0,	/* 通知するだけ */
	DIVY_VSC_ACT_REPAIRE,		/* 修復する     */
	DIVY_VSC_ACT_DELETE,		/* 該当ファイルを削除する */
};

typedef enum __vsc_Action	VscAction;

/*------------------------------------------------------------------------------
  Declare handler structure
  ----------------------------------------------------------------------------*/
typedef struct __vsc_DataSource		VscDataSource;
typedef struct __vsc_Session		VscSession;
typedef struct __vsc_Property		VscProperty;

/**
 * VscDataSource の処理に必要で、かつ特定のウィルス検索プロバイダに依存する情報
 * を保持するコンテキスト。
 * 主としてウィルスパターンデータの保管場所に使用します。
 * 必要であれば、ウィルス検索プロバイダ側で定義すること。
 * ウィルス検索プロバイダ以外はこれにアクセスしてはならない。
 */
typedef struct __vsc_dsContxt		VscDsCntxt;

/**
 * VscSession の処理に必要で、かつ特定のウィルス検索プロバイダに依存する情報
 * を保持するコンテキスト。
 * 必要であれば、ウィルス検索プロバイダ側で定義すること。
 * ウィルス検索プロバイダ以外はこれにアクセスしてはならない。
 * なお、ストリーム検索に必要なデータは VscStreamFile に保持すること。
 *
 * (note) VscSessionCntxt と VscStreamFile の役割分担
 * 	VscStreamFile はある1つのVscSessionCntxt の中で行われる高々１回の
 * 	検索オペレーションに使用されるコンテキストです。ですので、ストリーム
 * 	検索にのみ保持していればよいデータ(例えば、書き込まれたバッファなど)は
 * 	VscStreamFile に持つべきです。
 * 	1つのセッションに渡って保持する必要があるデータはVscSessionCntxt に
 * 	保持して下さい。
 */
typedef struct __vsc_sessionCntxt	VscSessionCntxt;

/**
 * ある1つのファイルストリームを表す情報(不完全型, ストリーム系API専用)
 * ストリーム系APIをサポートするウィルス検索プロバイダは、この構造体を
 * 定義し、ストリーム検索に必要な情報を保持するようにして下さい。
 * VscSessionCntxt との相違については、VscSessionCntxtを参照。
 */
typedef struct __vsc_StreamFile		VscStreamFile;

/*------------------------------------------------------------------------------
  Define handler structure
  ----------------------------------------------------------------------------*/
/**
 * [[ ハンドラ構造体の使い方 ]]
 *   ハンドラ構造体がどのような概念を抽象化しているかについては、各構造体の
 *   説明を参照して下さい。ここではハンドラ構造体の使い方について、利用する側と
 *   実装する側の２つの立場からそれぞれ説明します。
 *
 * [ ハンドラ構造体を利用する側(プロバイダを利用する側) ]
 *   * 対象
 *   -----------
 *   	ウィルス検索を実施するオブジェクト、関数。例えば、TeamFile本体や管理GUI
 *   	など。原理的には"ウィルス検索プロバイダ"ではない全てのコードから利用
 *   	することが可能。
 *   	なおこの対象者はVscSessionやVscDataSource の実装を見ることは出来ない。
 *
 *   * 利用方法
 *   -----------
 *   	(1) 所定のPublic関数(divy_db_lookup_vscdatasource) を通じてVscDataSource
 *   	    を取得します。VscDataSource がある種類のウィルス検索プロバイダを
 *   	    仮想化するオブジェクトです。なお、VscDataSource取得関数はutil_vscan.h
 *   	    にも定義されています。Apache モジュールからコールする場合にはそちらを
 *   	    利用すること。
 *   	(2) 必要であればVscDataSource にウィルスパターンデータベースを読み込み
 *   	    ます。(divy_vsc_loadVirusData)
 *   	(3) VscDataSourceを使ってVscSession を取得します。(divy_vsc_getVscSession)
 *   	    これはウィルス検索プロバイダとの接続状態を表す状態オブジェクトです。
 *   	    VscSessionがアクティブである限り、ウィルス検索プロバイダとの通信が
 *   	    可能な状態にあるということです。
 *   	(4) (3)で取得したVscSessionをストリームウィルス検索に利用できるかどうかは
 *   	    divy_vsc_canScanStream で判定します。
 *   	    また取得したセッションがキャッシュ可能かどうかはdivy_vsc_canCacheSession
 *   	    で判ります。
 *   	(5) ファイルパスベースのスキャンであればdivy_vsc_scanFile を、
 *   	    ストリームベースのスキャンであればdivy_vsc_startStreamScan から始まる
 *   	    一連のストリーム系検索APIを利用して下さい。
 *   	    なお、ウィルス検索プロバイダが取るべきアクションはDIVY_VSC_ACT_NOTICE
 *   	    (通知するだけ) が適切でしょう。
 *	(6) ステータスコードを判定し、適切に処理して下さい。
 *
 *
 * [ ハンドラ構造体を実装する側(プロバイダを実装する側) ]
 *   * 対象
 *   -----------
 *   	この役割を持つ既存のモジュールは存在しません。裏を返すと、TeamFile本体
 *   	やDBプロバイダなどはこの役割を持ってはならないということです。
 *   	この役割を担う実体の事を"ウィルス検索プロバイダ"と呼びます。
 *
 *   * 利用方法
 *   -----------
 *      (1) 1つのCファイルを用意し、このヘッダファイルをinclude した後、直に
 *          DIVY_VSC_PROVIDER_PRIVATE を定義して、各種構造体の実装を手に入れる。
 *   	(2) 各種ハンドラ構造体のメンバのうち、関数へのポインタとなっている部分を
 *   	    各プロバイダ毎に実装します。
 *   	    また、プロバイダに依存するコンテキストがあれば不完全型で宣言された
 *   	    コンテキスト構造体(VscDsCntxt,VscSessionCntxt)を定義してください。
 *   	(3) ストリーム系ウィルス検索APIをサポートするかどうか決定し、サポート
 *   	    するのなら、ファイルストリームコンテキスト(VscStreamFile) を実装
 *   	    して下さい。
 */

#ifdef DIVY_VSC_PROVIDER_PRIVATE
/**
 * ある１種類のウィルス検索エンジンを"データソース"にマッピングする構造体。
 *
 * この構造体のインスタンスは、ウィルス検索プロバイダ毎に１種類ずつ生成されます。
 * ウィルス検索機能を利用する側では、このインスタンスをある一連の手続きによって
 * 取得することで、この機能をキックするきっかけとなる関数をコールできるように
 * なります。例え単にファイルをウィルススキャンしたいだけでもこの手続きを踏む
 * 必要があります。
 *
 * [ プロバイダを実装する側への要請 ]
 *   * Cファイルの中で、DIVY_VSC_PROVIDER_PRIVATE をこのヘッダファイルincludeの
 *     直前にdefine し、実装を見えるようにして下さい。
 *   * 実装するハンドラ関数の呼び出しは全てリエントラントでなければなりません。
 *     ウィルス検索エンジンがそうでないのならば、そうなるよう保証して下さい。
 *
 * [ プロバイダを利用する側への要請 ]
 *   * この構造体の実装は見えないはず。用意されたPublic関数から機能を利用して
 *     下さい。
 * (note)
 * 	この構造体は、プロバイダフレームワークの要請で導入されたものです。
 * 	ですので、多くのウィルス検索エンジンにはDataSourceに対応する概念は
 * 	存在していなくても気にしないで下さい。
 */
struct __vsc_DataSource {

	/**
	 * 実行ステータスメッセージ
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * 参照には divy_vsc_getMsg 必ず使用してください。
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * 参照には divy_vsc_getCode を必ず使用して下さい。
	 */
	int __code;
	
	/**
	 * ウィルス検索エンジンから返却された固有のコード文字列
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * 参照には divy_vsc_getNativeCode を必ず使用して下さい。
	 */
	char *__nativecode;

	/**
	 * 作業用のプールへのポインタ
	 * (このデータソースとライフサイクルを共にするプール)
	 */
	apr_pool_t *__p;

	/**
	 * ウィルス検索プロバイダの種類を表す名称
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *   * ウィルス検索プロバイダ毎に一意でかつ、コンパイル時に決定できる
	 *     状態にしておいてください。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * これをどのように使用するかは呼び出し側が考えてください。
	 *     但し、絶対に上書きしてはなりません。また一部ポインタを
	 *     書き換えてもなりません。そうしてしまった場合の動作は不定です。
	 *     (多分CoreDumpします)
	 */
	const char *type;

	/**
	 * 特定のウィルス検索プロバイダに依存したデータを保持する構造体へのポインタ
	 * [ プロバイダを実装する側への要請 ]
	 *   * 必要であれば定義して下さい。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * ポインタの指し示す先を気にする必要はまったくありません。
	 */
	VscDsCntxt *vsccntxt;

	/**
	 * ウィルスパターンデータベースが前回ロードされた時間[マクロsec]
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *   * VscDataSource を新規生成した時には0に初期化して下さい。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * この変数を直接参照しないで下さい。
	 */
	apr_time_t loaded_time;

	/**
	 * ウィルスパターンデータベースのローディングをサポートしているかどうか。
	 *
	 * @param vscds VscDataSource* 自分自身へのポインタ
	 * @return int 処理ステータス
	 * 	DIVY_VSC_ST_OK               : サポートしている
	 * 	DIVY_VSC_ST_UNSUPPORTED_LOAD : サポートしていない
	 */
	int (*canLoadVirusData)(VscDataSource *vscds);

	/**
	 * ウィルス検索に必要なウィルスパターンデータベースをロードする。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *
	 *   * ロードされたデータはvscds が示すインスタンス中に保持すること。
	 *     (より正確にはコンテキスト構造体の中)
	 *   * 予めロードしておくという機構がないウィルス検索エンジンを使用する
	 *     のならば、DIVY_VSC_ST_UNSUPPORTED_LOAD を返すだけのハンドラを
	 *     実装すること。未実装(NULL) はNG。
	 *   * ロードされたデータはある1つのVscDataSource毎に保持して下さい。
	 *     このことは複数のVscDataSource間に渡って共有するような機構を
	 *     プロバイダ側で明示的に設けてはならない、言っているだけでウィルス
	 *     検索エンジンがそのようにしてしまっているのなら、それは構いません。
	 *     (どうしようもないので。)
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *
	 *   * このハンドラはdivy_vsc_loadVirusData 経由でコールすること。
	 *   * ロードオペレーションをサポートしていないウィルス検索プロバイダも
	 *     存在するので、呼び出し側ではそのことをステータスコードから適切に
	 *     判断して下さい。
	 *
	 * @param vscds VscDataSource* 自分自身へのポインタ
	 * @param vscprop const VscProperty * 接続情報を保持する構造体
	 * @return int 処理ステータス
	 * 	DIVY_VSC_ST_OK  : ロードに成功
	 * 	DIVY_VSC_ST_ERR : 予期しないエラーで失敗
	 *
	 * getCode() のステータス
	 * 	* DIVY_VSC_ST_OK -->
	 * 	  ** DIVY_VSC_ST_OK
	 * 	  ** DIVY_VSC_ST_UNSUPPORTED_LOAD : ロードは未サポート
	 * 	* DIVY_VSC_ST_ERR -->
	 * 	  ** DIVY_VSC_ST_LOADERR          : ロード操作に失敗
	 */
	int (*loadVirusData)(VscDataSource *vscds, const VscProperty *vscprop);

	/**
	 * あるウィルス検索エンジンとの接続を表すオブジェクトを取得する。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *
	 *   * ウィルス検索エンジンの特性から判断して"接続できた"とみなせる
	 *     状態にして下さい。例えばソケットを開くタイプのプロバイダなら
	 *     コネクトした状態にするということです。
	 *   * 何があっても戻り値のVscSession はアロケートして返却すること。
	 *     当然ステータスコードも設定する。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *
	 *   * このハンドラはdivy_vsc_getVscSession 経由でコールすること。
	 *   * 第３引数のプールp のライフサイクルに注意して下さい。
	 *     セッションの寿命はp のライフサイクルにも依存します。
	 *   * 第２引数vscprop のライフサイクルは第３引数と同じでなければ
	 *     なりません。
	 *   * 取得できたセッションをキャッシュしてもよいのもかどうかは
	 *     残念ながらプロバイダの実装に依存します。
	 *     VscSession のcanCacheSession ハンドラをコールして判断して下さい。
	 *
	 * @param vscds VscDataSource* 自分自身へのポインタ
	 * @param vscprop const VscProperty * 接続情報を保持する構造体
	 * @param  p apr_pool_t * 変数割り当てに使用するプール
	 * @return VscSession * ウィルス検索エンジンとの接続を表す構造体へのポインタ
	 */
	VscSession * (*getVscSession)(VscDataSource *vscds,
				const VscProperty *vscprop, apr_pool_t *p);

	/**
	 * このプロバイダはストリーム系スキャンをサポートしているか。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *
	 *   * ストリーム系検索APIをウィルス検索エンジンがサポートしていなかったら
	 *     サポート出来ないとして下さい。自前実装も可能ですが、その場合には
	 *     インメモリで全てのデータをキャッシュしないようにすること。
	 *   * サポートできない場合でもストリーム系APIの空実装は用意すること。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *
	 *   * このハンドラはdivy_vsc_canScanStream 経由でコールすること。
	 *   * ここでサポートしていないと言われたら、ストリーム系検索APIは
	 *     コールしないこと。動作は不定です。
	 *
	 * @param vscds VscDataSource* 自分自身へのポインタ
	 * @return 処理ステータス
	 * 	DIVY_VSC_ST_SUPPORTED_STREAM   : サポートしている
	 *	DIVY_VSC_ST_UNSUPPORTED_STREAM : サポートしていない
	 *	DIVY_VSC_ST_ERR                : 予期しないエラーが発生した
	 */
	int (*canScanStream)(VscDataSource *vscds);
};

#endif	/* DIVY_VSC_PROVIDER_PRIVATE */

#ifdef DIVY_VSC_PROVIDER_PRIVATE
/**
 * ウィルス検索エンジンとの"接続"状態を表す状態オブジェクト。
 * VscSession が取得出来ていれば接続が完了してウィルス検索が可能である状態
 * ということになります。
 *
 * [ プロバイダを実装する側への要請 ]
 *
 *   * Cファイルの中で、DIVY_VSC_PROVIDER_PRIVATE をこのヘッダファイルincludeの
 *     直前にdefine し、実装を見えるようにして下さい。
 *   * 実装するハンドラ関数の呼び出しは全てリエントラントでなければなりません。
 *     ウィルス検索エンジンがそうでないのならば、そうなるよう保証して下さい。
 *
 * [ プロバイダを利用する側への要請 ]
 *
 *   * この構造体の実装は見えないはず。用意されたPublic関数から機能を利用して
 *     下さい。
 *
 */
struct __vsc_Session {

	/**
	 * 実行ステータスメッセージ
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * 参照には divy_vsc_getMsg 必ず使用してください。
	 */
	char *__msg;
	
	/**
	 * 実行ステータスコード
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * 参照には divy_vsc_getCode を必ず使用して下さい。
	 */
	int __code;
	
	/**
	 * ウィルス検索エンジンから返却された固有のコード文字列
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * 参照には divy_vsc_getNativeCode を必ず使用して下さい。
	 */
	char *__nativecode;

	/**
	 * 作業用のプールへのポインタ
	 * (このセッションとライフサイクルを共にするプール)
	 */
	apr_pool_t *__p;

	/**
	 * 接続情報を表す構造体へのポインタ
	 */
	VscProperty *vscprop;

	/**
	 * 特定のプロバイダに依存したデータを保持する構造体へのポインタ
	 * [ プロバイダを実装する側への要請 ]
	 *   * 必要であれば定義して下さい。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * ポインタの指し示す先を気にする必要はまったくありません。
	 */
	 VscSessionCntxt *vscscntxt;
	
	/**
	 * このセッションはキャッシュ可能かどうか。
	 * "キャッシュ可能"であるとは、複数回のリクエストに渡って生成された
	 * vscsession を使いまわせるかどうかという意味です。これは
	 * ウィルス検索プロバイダによって異なります。
	 *
	 * [ プロバイダを実装する側への実装のヒント ]
	 *
	 *   * キャッシュの可能性は各セッション毎に保持していますが、多くの場合
	 *     ウィルス検索エンジンのコールインターフェースの性質によって
	 *     一意に決まるでしょう。ですので、常に固定値を返すようなコードでも
	 *     問題はありません。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *
	 *   * このハンドラはdivy_vsc_canCacheSession 経由でコールすること。
	 *   * キャッシュ不可能と報告されたのにキャッシュした場合の動作は
	 *     不定です。
	 *   * getVscSession の時に渡したプールのライフサイクルに注意。
	 *     プールの寿命が極めて短ければcanCacheSession が真であっても
	 *     実質利用できない可能性があるためです。
	 *
	 * @param vscsession VscSession* 自分自身へのポインタ
	 * @return int 処理ステータス
	 * 	DIVY_VSC_ST_SUPPORTED_CACHE   : このセッションはキャッシュ可能
	 * 	DIVY_VSC_ST_UNSUPPORTED_CACHE : このセッションはキャッシュ不可能
	 * 	DIVY_VSC_ST_ERR               : 予期しないエラーで失敗
	 */
	int (*canCacheSession)(VscSession *vscsession);

	/**
	 * このセッションを"閉じた"状態にする。
	 * セッションが閉じられた後、セッションに対する操作の結果は不定です。
	 * 
	 * [ プロバイダを実装する側への要請 ]
	 *   * closeSession ハンドラは複数回コールされる可能性があります。
	 *     それでも問題がないよう実装すること
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * ストリーム系APIを使ってスキャンしていた場合、このハンドラを
	 *     コールしても一部リソースが残ったままになります。必ず
	 *     endStreamScan 若しくはabortStreamScan させてください。
	 *
	 * @param vscsession VscSession* 自分自身へのポインタ
	 */
	void (*closeSession)(VscSession *vscsession);


	/*----------------------------------------------------------------------
	 * ファイル系ウィルス検索API
	 ---------------------------------------------------------------------*/
	/**
	 * 指定されたfilepath が示すパスの物理ファイルをスキャンし、ウィルス感染
	 * し、指定されたaction に基づいた動作を行う。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *   * action は呼び出し毎に異なるかもしれません。ですので、vscsession
	 *     にキャッシュしてはなりません。
	 *   * loadVirusData ハンドラがコールされていない可能性もありますので、
	 *     コールされていなければ自身でロードして処理すること。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * このハンドラはdivy_vsc_scanFile 経由でコールすること。
	 *   * 書き込み途中のファイルを渡してはなりません。
	 *   * 排他的にopen されたままのファイルを渡してはなりません。
	 *
	 * @param vscsession VscSession* 自分自身へのポインタ
	 * @param action VscAction 検索アクションの種類(enum 参照)
	 * @param filepath const char * 検索対象のファイルパス
	 * @return int 処理ステータス
	 * 	DIVY_VSC_ST_OK       : ウィルスに感染していなかった
	 * 	DIVY_VSC_ST_INFECTED : ウィルスに感染していた
	 * 	DIVY_VSC_ST_ERR      : 予期しないエラーで失敗
	 *
	 * getCode() のステータス
	 * 	* DIVY_VSC_ST_OK
	 * 	* DIVY_VSC_ST_INFECTED :
	 * 	  ** DIVY_VSC_ST_INFECTED : 感染していたことだけを通知
	 * 	  ** DIVY_VSC_ST_REPAIRED : 修復できた
	 * 	  ** DIVY_VSC_ST_DELETED  : ウィルス感染ファイルを削除できた
	 * 	* DIVY_VSC_ST_ERR
	 * 	  ** DIVY_VSC_ST_NO_REPAIRED        : 感染を検出したが修復出来なかった
	 * 	  ** DIVY_VSC_ST_TOOLARGE_CONTENT   : 検索できないサイズのファイルだった
	 * 	  ** DIVY_VSC_ST_UNSUPPORTED_TYPE   : 検索できない種類のファイルだった
	 * 	  ** DIVY_VSC_ST_CANNOT_OPEN        : 対象ファイルをOpen出来なかった
	 * 	  ** DIVY_VSC_ST_CANNOT_READ_FILEST : 対象ファイルのステータス取得に失敗
	 * 	  ** DIVY_VSC_ST_UNSUPPORTED_ACTION : サポートしていないaction だった
	 */
	int (*scanFile)(VscSession *vscsession, VscAction action,
							const char *filepath);

	/*----------------------------------------------------------------------
	 * ストリーム系ウィルス検索API
	 ---------------------------------------------------------------------*/
	/**
	 * ストリームウィルス検索の開始を宣言する。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *   * action は呼び出し毎に異なるかもしれません。ですので、vscsession
	 *     にキャッシュしてはなりません。
	 *   * loadVirusData ハンドラがコールされていない可能性もありますので、
	 *     コールされていなければ自身でロードして処理すること。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * このハンドラはdivy_vsc_startStreamScan 経由でコールすること。
	 *
	 * @param vscsession VscSession* 自分自身へのポインタ
	 * @param action VscAction 検索アクションの種類(enum 参照)
	 * @param sfile VscStreamFile ** ウィルス検索ストリームへのポインタ
	 * @return int 処理ステータス
	 *	DIVY_VSC_ST_OK      : ストリーム開始処理に成功
	 *	DIVY_VSC_ST_ERR     : ストリーム開始処理に失敗
	 */
	int (*startStreamScan)(VscSession *vscsession, VscAction action,
							VscStreamFile **sfile);

	/**
	 * ウィルス検索プロバイダに長さlenのデータbytes を渡す。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *   * ウィルス検索エンジンによっては(というか殆どのエンジンでは)全ての
	 *     データを受けきるまでウィルス検索を実施しません。そうであっても
	 *     呼び出し元には成功したとそのまま返却すれば十分です。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * このハンドラはdivy_vsc_writeStreamBytes 経由でコールすること。
	 *   * ウィルス感染が判明したら、endStreamScan で必ず終了させて下さい。
	 *
	 * @param vscsession VscSession* 自分自身へのポインタ
	 * @param sfile VscStreamFile * ウィルス検索ストリームへのポインタ
	 * @param bytes const char * 受け渡すデータ(\0 で終了している必要なし)
	 * @param len apr_size_t bytes の長さ
	 * @return int 処理ステータス (scanFile と同じ)
	 */
	int (*writeStreamBytes)(VscSession *vscsession, VscStreamFile *sfile,
				const char *bytes, apr_size_t len);

	/**
	 * 検索対象データを全てストリームに書き込んだことを通知し、最終的な
	 * ウィルス感染状況を取得する。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *   * このハンドラは複数回コールされる可能性があります。また
	 *     abortStreamScan と交互にコールされることもあるでしょう。一度でも
	 *     ファイルストリームの終了処理が済んでいたら複数回の呼び出しで
	 *     反応しないように実装して下さい。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * このハンドラはdivy_vsc_endStreamScan 経由でコールすること。
	 *   * 全てのデータをストリームに入れたら必ずこれをコールすること。
	 *     途中で止めるのならabortStreamScan して下さい。
	 *
	 * @param vscsession VscSession* 自分自身へのポインタ
	 * @param sfile VscStreamFile * ウィルス検索ストリームへのポインタ
	 * @return int 処理ステータス (scanFile と同じ)
	 */
	int (*endStreamScan)(VscSession *vscsession, VscStreamFile *sfile);

	/**
	 * ストリームウィルス検索を途中で止める。
	 *
	 * [ プロバイダを実装する側への要請 ]
	 *   * このハンドラは複数回コールされる可能性があります。また
	 *     endStreamScan と交互にコールされることもあるでしょう。一度でも
	 *     ファイルストリームのアボート処理が済んでいたら複数回の呼び出しで
	 *     反応しないように実装して下さい。
	 *
	 * [ プロバイダを利用する側への要請 ]
	 *   * このハンドラはdivy_vsc_abortStreamScan 経由でコールすること。
	 *   * startStreamScan ハンドラをコールしていて途中で検索を止める場合、
	 *     必ずこのハンドラをコールすること。コールしないとリソース解放漏れ
	 *     等の問題が発生するかもしれません。(ウィルス検索エンジンの実装
	 *     に依存します。)
	 *
	 * @param vscsession VscSession* 自分自身へのポインタ
	 * @param sfile VscStreamFile * ウィルス検索ストリームへのポインタ
	 */
	void (*abortStreamScan)(VscSession *vscsession, VscStreamFile *sfile);
};

#endif	/* DIVY_VSC_PROVIDER_PRIVATE */

/**
 * ウィルス検索プロバイダに渡すConfiguration情報を保持する構造体。
 *
 * [ 役割 ]
 * 	この構造体は、Apacheが持っているConfigurationをウィルス検索プロバイダに
 * 	受け渡しする目的で存在しています。ですので、この構造体は呼び出し側
 * 	つまりApache本体に近いモジュールで作成され、ウィルス検索プロバイダに
 * 	渡されることになります。
 */
struct __vsc_Property {

	/* ウィルス検索エンジンに渡す環境変数へのポインタ(任意) */
	const char * const *env;

	/* コマンド呼び出し型のウィルス検索エンジンならば、その実行パスと
	 * 引数からなる配列
	 *
	 * argv[0] が実行ファイルパス、argv[1]以降が引数 */
	const char * const *argv;

	/* ウィルス検索エンジンのホスト名(任意) */
	const char *hostname;

	/* ウィルス検索エンジンのサービスポート番号(任意) */
	int port;

	/* ウィルスパターンデータベースのリフレッシュ時間間隔[sec]
	 * 0 ならば毎回アップデートされます。*/
	int refreshinterval;
};

/*------------------------------------------------------------------------------
  Declare public functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたproviderType に一致するウィルス検索プロバイダであった時、
 * ウィルス検索プロバイダ構造体を作成 & 返却する関数
 * のためのHook関数宣言。
 * ウィルス検索プロバイダの初期化に利用します。
 *
 * (note) このHook関数宣言によって利用できるようになる関数
 *
 *  ・divy_hook_create_vscdatasource : フックハンドラとして関数を登録する
 *  ・divy_run_create_vscdatasource  : 上記関数で登録されたフック関数を呼び出す
 */
APR_DECLARE_EXTERNAL_HOOK(divy, DIVY, int, create_vscdatasource,
		(apr_pool_t *p, const char *providerType,  VscDataSource **vscds))

/**
 * 指定されたproviderType のウィルス検索プロバイダデータソース(VscDataSource) を
 * 取得して返却する。
 * 取得に失敗した場合には、NULLを返却し、戻り値でステータスを返却します。
 * (note)
 * 	この関数は、取得したVscDataSourceを内部的にキャッシュしたり、
 *	再利用したりしません。Apache モジュール上からVscDataSourceの取得を行う
 *	場合には、この関数ではなく、util_vscan.h で宣言された関数を使用して下さい。
 *
 * @param p apr_pool_t * VscDataSource用の領域を取得するプール
 * @param providerType const char * ウィルス検索プロバイダの種類
 * @param vscds VscDataSource ** 取得したウィルス検索プロバイダへのポインタ
 * 				なお、*vscds は最初にNULL初期化されます。
 * @return RET_STAT 処理ステータス
 * 	DIVY_VSC_ST_OK             : 取得に成功
 * 	DIVY_VSC_ST_INVALID_PARAMS : 引数がNULL値, 不適切な値だった
 * 	DIVY_VSC_ST_NOTFOUND_VSCDS : ウィルス検索プロバイダが存在しなかった
 * 	DIVY_VSC_ST_ERR            : 予期しない問題が発生したため取得に失敗した
 */
DIVY_DECLARE(int) divy_vsc_lookup_vscdatasource(apr_pool_t *p,
						const char *providerType,
						VscDataSource **vscds);

/**
 * getCode ハンドラのラッパー。
 *
 * @parm data void * VscDataSource/VscSession へのポインタ
 * @param type int data の種類
 * 		DIVY_VSC_TYPE_DATASOURCE : VscDataSource
 * 		DIVY_VSC_TYPE_SESSION    : VscSession
 * @return int getCode()の戻り値
 */
DIVY_DECLARE(int) divy_vsc_getCode(void *data, int type);

/**
 * getNativeCode ハンドラのラッパー。
 *
 * @parm data void * VscDataSource/VscSession へのポインタ
 * @param type int data の種類
 * 		DIVY_VSC_TYPE_DATASOURCE : VscDataSource
 * 		DIVY_VSC_TYPE_SESSION    : VscSession
 * @return char * getNativeCode()の戻り値
 */
DIVY_DECLARE(char *) divy_vsc_getNativeCode(void *data, int type);

/**
 * getMsg ハンドラのラッパー。
 *
 * @parm data void * VscDataSource/VscSession へのポインタ
 * @param type int data の種類
 * 		DIVY_VSC_TYPE_DATASOURCE : VscDataSource
 * 		DIVY_VSC_TYPE_SESSION    : VscSession
 * @return char * getMsg()の戻り値
 */
DIVY_DECLARE(char *) divy_vsc_getMsg(void *data, int type);


/**
 * canLoadVirusData ハンドラのラッパー (see canLoadVirusData)
 */
DIVY_DECLARE(int) divy_vsc_canLoadVirusData(VscDataSource *vscds);

/**
 * ウィルスパターンデータベースが前回ロードされた時間と比較し
 * リフレッシュ間隔を超えていれば、パターンデータベースを更新する。
 * loadVirusData ハンドラのラッパー + α (see loadVirusData)
 *
 * @param vscds VscDataSource *
 * @param vscprop const VscProperty *
 * @return int 処理ステータス
 */
DIVY_DECLARE(int) divy_vsc_loadVirusData(VscDataSource *vscds,
					const VscProperty *vscprop);

/**
 * getVscSession ハンドラのラッパー (see getVscSession)
 */
DIVY_DECLARE(VscSession *) divy_vsc_getVscSession(VscDataSource *vscds,
				const VscProperty *vscprop, apr_pool_t *p);

/**
 * canScanStream ハンドラのラッパー (see canScanStream)
 */
DIVY_DECLARE(int) divy_vsc_canScanStream(VscDataSource *vscds);

/**
 * canCacheSession ハンドラのラッパー (see canCacheSession)
 */
DIVY_DECLARE(int) divy_vsc_canCacheSession(VscSession *vscsession);

/**
 * closeSession ハンドラのラッパー (see closeSession)
 */
DIVY_DECLARE(void) divy_vsc_closeSession(VscSession *vscsession);

/**
 * scanFile ハンドラのラッパー (see scanFile)
 */
DIVY_DECLARE(int) divy_vsc_scanFile(VscSession *vscsession,
					VscAction action, const char *filepath);

/**
 * startStreamScan ハンドラのラッパー (see startStreamScan)
 */
DIVY_DECLARE(int) divy_vsc_startStreamScan(VscSession *vscsession,
					VscAction action, VscStreamFile **sfile);

/**
 * writeStreamBytes ハンドラのラッパー (see writeStreamBytes)
 */
DIVY_DECLARE(int) divy_vsc_writeStreamBytes(VscSession *vscsession,
					VscStreamFile *sfile,
					const char *bytes, apr_size_t len);

/**
 * endStreamScan ハンドラのラッパー (see endStreamScan)
 */
DIVY_DECLARE(int) divy_vsc_endStreamScan(VscSession *vscsession,
					VscStreamFile *sfile);

/**
 * abortStreamScan ハンドラのラッパー (see abortStreamScan)
 */
DIVY_DECLARE(void) divy_vsc_abortStreamScan(VscSession *vscsession,
					VscStreamFile *sfile);


#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_TF_VSCAN_H */

