/**
 * $Id$
 *
 * util_common.h
 *
 * ビジネスロジックフリーなユーティリティ関数の宣言・定義を行うヘッダファイル
 * これらのインターフェースは、全てのモジュール間で共通に利用できる関数を
 * 纏め上げる目的をもって導入されました。
 * (note) util.h との違い
 * 	util.h はもはや、util_business といってもよいほどビジネスロジックに
 * 	汚染されてしまいました。そのため、モジュール間で共通に使用したい関数を
 * 	定義することが出来なくなっています。なぜなら、あらゆるライブラリがuti.h
 * 	をincludeすることでリンクされてしまうからです。ライブラリ化を目指す
 * 	モジュールでは致命的な問題です。
 * 	仕方が無いのでutil_common.h というひねりのない名称のヘッダを用意し、
 * 	今度こそはビジネスロジックフリーな関数群だけを定義できるようにしました。
 * (note)
 * 	xxx_rdbo やxxx プロバイダ独自の共通関数、及びap系関数を使う関数は絶対に
 * 	入れてはなりません。
 * 	このヘッダファイルにinclude出来るのはapr_xxx とtfr.h だけです。
 *
 * 2004/02/08 Sun takehara NEW
 */
#ifndef INCLUDE_UTIL_COMMON_H
#define INCLUDE_UTIL_COMMON_H

#include "apr.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_time.h"
#include "apr_file_io.h"
#include "apr_thread_proc.h"

#include "tfr.h"
#include "tf_errno.h"
#include "tf_errlog.h"
#include "tf_cset.h"	/* 互換性のためこれが必要 */
#include "tf_sbuf.h"	/* 互換性のためこれが必要 */
#include "tf_array.h"	/* 互換性のためこれが必要 */

#if APR_HAVE_STDIO_H
#include <stdio.h>
#endif	/* APR_HAVE_STDIO_H */
#if APR_HAVE_STRING_H
#include <string.h>
#endif	/* APR_HAVE_STRING_H */
#if APR_HAVE_SYS_TIME_H
#include <sys/time.h>
#endif
#ifdef HAVE_TIME_H
#include <time.h>
#endif

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*--------------------------------------------------------------
  Define fixed value
  --------------------------------------------------------------*/
/* dav_divy_escape_xmlstr で使用される文字列の変換規則 */
#define DIVY_XML_T2T_CDATA	0x01	/* CDATA をつける       */
#define DIVY_XML_T2T_QUOTE	0x02	/* 実体参照に置き換える */

/* 関数divy_format_time_t で使用するスタイル */
#define DIVY_TIME_STYLE_ISO8601  1
#define DIVY_TIME_STYLE_RFC822   2
#define DIVY_TIME_STYLE_JAPANESE 3
#define DIVY_TIME_STYLE_RFC2822  4

/* 処理結果 */
#define DIVY_UTIL_COMMON_OK	0
#define DIVY_UTIL_COMMON_ERR	1
#define DIVY_UTIL_COMMON_EOF	2

/* 関数 divy_execute_cmd で返却されるステータスコード */
#define DIVY_UTIL_COMMON_EXCEPTCD_SET	3

/*--------------------------------------------------------------
  Define macros
  --------------------------------------------------------------*/
/**
 * NULL を"(null)"文字に変換するマクロ
 *
 * @param s char * / const char * 文字列
 */
#define REPLACE_NULL(s) ((s == NULL) ? "(null)" : s)

/*
 * 長さ0の文字列をNULLに変えるマクロ
 *
 * @param s char * / const char * 文字列へのポインタ
 */
#define REPLACE_EMPTYSTR(s) ((s != NULL && *(s) == '\0') ? NULL : s)

/*
 * 指定された文字列がNULLならば空文字に変換するマクロ
 *
 * @param s char * / const char * 文字列へのポインタ
 */
#define TO_EMPTY(s)	((s == NULL) ? "" : s)

/**
 * NULL または空文字であるかどうかを判定するマクロ
 *
 * @param s char * / const char * 文字列へのポインタ
 */
#define IS_EMPTY(s) (s == NULL || *(s) == '\0')

/**
 * 文字列s が充填されている(NULLではない && 空ではない)ことを判定するマクロ
 * (note)
 * 	!IS_EMPTY(s) と同じ意味になります。
 * 	コード上の判りやすさを狙って別名にしているだけです。
 *
 * @param s char * / const char * 文字列へのポインタ
 */
#define IS_FILLED(s) (s != NULL && *(s) != '\0')

/*--------------------------------------------------------------
  Define structure and enum
  --------------------------------------------------------------*/
/**
 * ファイルシステム情報を保持する構造体
 * (note)
 * 	あまり沢山の情報は持っていません。
 */
typedef struct __divy_statfs_t	divy_statfs_t;
struct __divy_statfs_t {
	const char *path;		/* 対象ディレクトリパス       */
	apr_size_t bsize;		/* ブロックサイズ(byte)       */
	apr_uint64_t total_bytes;	/* 総容量(byte)               */
	apr_uint64_t avail_bytes;	/* 利用可能な残りサイズ(byte) */
	apr_uint64_t total_files;	/* 総i-node数                 */
	apr_uint64_t avail_files;	/* 利用可能な残りi-node数     */
};

/*--------------------------------------------------------------
  Declare public function
  --------------------------------------------------------------*/
/**
 * 文字列 s 中で最初に文字 c が現れた位置へのポインタを返す。
 * (Reentrant)
 * 	動作プラットフォームに依存します。
 *
 * @param s char *
 * @param c int 文字
 * @return char * 位置へのポインタ
 */
DIVY_DECLARE(char *) divy_strchr(char *s, int c);

/**
 * 文字列 s 中で最初に文字 c が現れた位置へのポインタを返す。(const版)
 * (Reentrant)
 * 	動作プラットフォームに依存します。
 *
 * @param s const char *
 * @param c int 文字
 * @return const char * 最初に現れた位置へのポインタ
 */
DIVY_DECLARE(const char *) divy_strchr_c(const char *s, int c);

/**
 * 文字列 s 中で最後に文字 c が現れた位置へのポインタを返す。
 * (Reentrant)
 * 	動作プラットフォームに依存します。
 *
 * @param s char *
 * @param c int 文字
 * @return char * 最後に現れた位置へのポインタ
 */
DIVY_DECLARE(char *) divy_strrchr(char *s, int c);

/**
 * 文字列 s 中で最後に文字 c が現れた位置へのポインタを返す。(const版)
 * (Reentrant)
 * 	動作プラットフォームに依存します。
 *
 * @param s const char *
 * @param c int 文字
 * @return char * 最後に現れた位置へのポインタ
 */
DIVY_DECLARE(const char *) divy_strrchr_c(const char *s, int c);

/**
 * 部分文字列 str が文字列 s 中で最初に現れる位置を見つける。
 * (Reentrant)
 * 	動作プラットフォームに依存します。
 *
 * @param s char *
 * @param str const char * 部分文字列
 * @return char * 最後に現れた位置へのポインタ
 */
DIVY_DECLARE(char *) divy_strstr(char *s, const char *str);

/**
 * 部分文字列 str が文字列 s 中で最初に現れる位置を見つける。(const版)
 * (Reentrant)
 * 	動作プラットフォームに依存します。
 *
 * @param s const char *
 * @param str const char * 部分文字列
 * @return char * 最後に現れた位置へのポインタ
 */
DIVY_DECLARE(const char *) divy_strstr_c(const char *s, const char *str);

# define divy_strchr(s, c)	strchr(s, c)
# define divy_strchr_c(s, c)	strchr(s, c)
# define divy_strrchr(s, c)	strrchr(s, c)
# define divy_strrchr_c(s, c)	strrchr(s, c)
# define divy_strstr(s, str)	strstr(s, str)
# define divy_strstr_c(s, str)	strstr(s, str)

/**
 * 与えられた文字列の末尾から'/'を除去した文字列を生成して返却する。
 * (Reentrant)
 *
 * @param p apr_pool_t * プールへのポインタ
 * @param str const char * 操作対象の文字列
 * @return '/'を除去した文字列
 */
DIVY_DECLARE(char *) dav_divy_remove_endslash(apr_pool_t *p, const char *str);

/**
 * 与えられた文字列の末尾に'/'がなければ、'/'を付けて返却する。
 * (Reentrant)
 * 	str が処理中に変更されない場合に限り、この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param str const char * 操作対象文字列
 * @return char * '/'を末尾に付けた文字列
 */
DIVY_DECLARE(char *) dav_divy_append_endslash(apr_pool_t *p, const char *str);

/**
 * 与えられたuri からfinal path segmentを切り出して返却する。
 * なお、uri の末尾についた'/'は無視する。
 * (例) uri = "/divy/abcd/AAAA///" ---> "AAAA" を返却する 
 *
 * 指定するuri の文字列は、必ず '\0'で終わっていなければなりません。
 * (Reentrant)
 * 	uri が処理中に変更されない場合に限り、この関数はスレッドセーフです。
 *
 * @param p apr_pool_t * プールへのポインタ
 * @param uri char * 検索対象のuri
 * @return 切り取った文字列. 但し、uri が NULLだったり、'\0'だったときには、
 *         NULL を返却する。
 */
DIVY_DECLARE(char *) dav_divy_extract_finalpath_segment(apr_pool_t *p, const char *uri);

/**
 * 与えられたuri から先頭のパスセグメント(スラッシュを含まない)を
 * 切り出して返却する。
 * '/' が与えられた場合には空文字("")を返します。
 * (例)
 * 	uri = "/root/abcd/aiueo"   ---> "root"
 * 	uri = "///root/abcd/aiueo" ---> "root"
 * 	uri = "/"                  ---> ""
 * 	uri = "root"               ---> "root"
 *
 * (Reentrant)
 * 	uri が処理中に変更されない場合に限り、この関数はスレッドセーフです。
 *
 * @param p apr_pool_t * プールへのポインタ
 * @param uri const char * 検索対象のuri
 * @return char * 切り出した文字列。uri がNULLまたは'\0'ならNULLを、"/"ならば
 * 		空文字を返却します。
 */
DIVY_DECLARE(char *) dav_divy_extract_firstpath_segment(apr_pool_t *p, const char *uri);

/**
 * 指定されたuri_part を先頭からスラッシュ区切りで組み立てて返却する。
 * 既にスラッシュがuri_partの先頭や末尾に指定されていれば、２重に付けないようにします。
 * 終端には、必ずNULLを指定して下さい。
 * また、指定可能な可変長文字列の数は、 DIVY_MAX_VALIST_URI 個です。
 * これを超えたらエラーとなります。
 * (例)
 * 	dav_divy_make_full_uri(p, "/divy", "/abc", "ddd", NULL);
 * 	--> "/divy/abc/ddd" となります。
 *
 *
 * @param p apr_pool_t *
 * @param root * const char * uri のトップに位置する文字列
 * @param uri_part * const char * uri の部分文字列
 * @param ... char * uri の部分文字列
 * @return char * 組み立てられたURI文字列
 * 			引数の数がDIVY_MAX_VALIST_URIよりも大きい時にはNULL
 */
DIVY_DECLARE_NONSTD(char *) dav_divy_make_uri(apr_pool_t *p,
				const char *root, const char *uri_part, ...);

/**
 * 指定されたpath の階層数(path segmentの個数)を数えてその数を返却する。
 * (note)
 * 	この関数は、Apache本体が定義している関数 ap_count_dirs と同じです。
 * 	ライブラリ化に伴いコピーしたものです。Apache本体のコードが
 * 	変わってしまったらSyncして下さい。
 *
 * @param path const char *
 * @return int 個数. path がNULLであれば0を返却します。
 */
DIVY_DECLARE(int) divy_count_dirs(const char *path);

/**
 * 指定されたprefix が示す文字列をstr の先頭から取り除き
 * 残りの文字列部分を返却する.
 * (例)
 * 	str = "/divy/abcd/aaaaabb", prefix = "/divy"
 * 	--> "/abcd/aaaaabb" が返却される
 * (Reentrant)
 * 	prefix, str が処理中に変更されない場合に限り、この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param str const char * ソース文字列
 * @param prefix const char * 切り取る文字列
 * @return char * prefix がNULLまたは空文字またはstr にprefix と
 * 		一致する文字列がなかった場合には、str がそのまま返却されます。
 */
DIVY_DECLARE(char *) dav_divy_truncate_prefix(apr_pool_t *p,
					const char *str, const char *prefix);

/*
 * 指定されたsrc 文字列内で、最後にdst文字列が現れる位置までの文字列を切り出して
 * 返却する。
 * (例) src = "/abcd/efg/0123/iii/0123/a" dst = "0123" 
 *      結果=  /abcd/efg/0123/iii/
 * (Reentrant)
 * 	str, dst が処理中に変更されない場合に限り、この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param src const char * 検索対象文字列
 * @param dst const char * 検索文字列
 * @return char * 見つかった位置までの文字列。見つからなければ、NULLを返す
 */
DIVY_DECLARE(char *) dav_divy_find_prestr(apr_pool_t *p, const char *src, const char *dst);

/**
 * src に含まれる from の文字列を to の文字列に置き換えて返却する。
 * surround にNULL または空文字以外が指定された場合、その文字列によって
 * 囲まれた文字列のみを置換対象とします。
 * なお、surround の対応関係が取れていないと、置換しません。
 * また、from の文字列がsrc に含まれていなくても何もしません。src を
 * そのまま返却します。
 *
 * (例) 文字列「abcdebg'aaa?dd?d'ee?e'zzz?」
 *      from = "?", to = "@@", surround = "'" の時
 *   -->    = 「abcdebg'aaa@@dd@@d'ee?e'zzz?」
 *
 *	「ee?e'zzz?」の部分の「?」が置換されなかったのは
 *	surround文字列「'」 の対応が取れていなかったためです。
 * (Reentrant)
 * 	引数のポインタが示すデータが処理中に変更されない場合に限り
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param src const char * 置換対象の文字列
 * @param from const char * 置き換える文字列
 * @param to   const char * 置き換え後の文字列
 * @param surround const char * 囲み文字。NULLの時には、src 全てが置き換え対象
 *                              なお、閉じられていない場合には、変換しません。
 * @return const char * 置き換え後の文字列。
 */
DIVY_DECLARE(const char *) dav_divy_replace_str(apr_pool_t *p, const char *src,
			const char *from, const char *to, const char *surround);

/**
 * 指定された文字列str をXMLのエレメントとして挿入する際に、問題が
 * 起きないようエスケープする。
 * (Reentrant)
 * 	strが示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 * 
 * @param p apr_pool_t *
 * @param str char * 変換対象の文字列
 * @param style int 変換規則。OR をとって複数指定することも可能です。
 * 		DIVY_XML_T2T_CDATA : CDATA セクションをstrの前後に入れます
 * 		DIVY_XML_T2T_QUOTE : '<', '>', '&', '"' を実体参照に置き換える
 * @return const char * 変換後の文字列
 */
DIVY_DECLARE(const char *) dav_divy_escape_xmlstr(apr_pool_t *p, char *str, int style);

/**
 * 指定された data が数値データのみで構成されているかどうか検証する。
 * (Reentrant)
 * 	data が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 *
 * @param data const char * チェック対象の文字列
 * @return int 処理結果
 * 		0: 数値ではない値が1つ以上混じっていた
 * 		1: 全て数値だった
 */
DIVY_DECLARE(int) dav_divy_isdigit_str(const char *data);

/**
 * 与えられた文字列の前後からスペースを除去した文字列を生成して返却する。
 * (Reentrant)
 * 	s が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t * プールへのポインタ
 * @param s const char * 操作対象の文字列
 * @return スペースを除去した文字列
 */
DIVY_DECLARE(const char *) dav_divy_trim_white(apr_pool_t *pool, const char *s);

/**
 * 指定された str が示す文字列を先頭からlen が示す"文字数"分だけ切り出して
 * 返却する。
 * 切り出し処理は、str に含まれる文字の文字境界で正しく切り出すよう
 * 実施されます。
 *
 * (note)
 * 	str に指定する文字列は、現在のところ、UTF-8でエンコードされていなければ
 * 	なりません。
 * 	異なるエンコーディングスキームでは正しく切り取ることが出来ません。
 * (Reentrant)
 * 	str が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param str const char * ソースとなる文字列
 * @param len int 切り出す文字数(バイト数ではありません)
 * @return const char * 切り出した文字列
 * 			切り出し処理に失敗したらNULLを返却します。
 */
DIVY_DECLARE(const char *) dav_divy_truncate_str(apr_pool_t *p,
						const char *str, int len);

/**
 * 指定された str が示す文字列を先頭からbytelen が示すバイト数分だけ切り出して
 * 返却する。
 * 切り出し処理は、str に含まれる文字の文字境界で正しく切り出すよう
 * 実施されるため、bytelenよりも１から３バイト短くなることがあります。
 *
 * (note)
 * 	str に指定する文字列は、現在のところ、UTF-8でエンコードされていなければ
 * 	なりません。
 * 	異なるエンコーディングスキームでは正しく切り取ることが出来ません。
 * (Reentrant)
 * 	str が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param str const char * ソースとなる文字列
 * @param bytelen int 切り出すバイトする(文字数ではありません)
 * @return const char * 切り出した文字列
 * 			切り出し処理に失敗したらNULLを返却します。
 */
DIVY_DECLARE(const char *) dav_divy_truncate_str_perbyte(apr_pool_t *p,
						const char *str, int bytelen);

/**
 * 指定された str が示す文字列の文字数を数えて返却する
 * (Reentrant)
 * 	str が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 *
 * (note)
 * 	str に指定する文字列は、現在のところ、UTF-8でエンコードされていなければ
 * 	なりません。
 * 	異なるエンコーディングスキームでは正しくカウントできません。
 *
 * @param p apr_pool_t *
 * @param str const char * ソースとなる文字列
 * @return int 文字数。カウントに失敗した場合は-1を返却。
 */
DIVY_DECLARE(int) dav_divy_count_utf8str(apr_pool_t *p, const char *str);

/**
 * 与えられた文字列strを16進数の文字列として返却する。
 * (例) ABCDあい -> 41424344E38182E38184
 * (Reentrant)
 * 	str が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 * (note)
 * 	途中にヌル終端を含むバイナリデータを変換する場合には、
 * 	dav_divy_btohexを使用して下さい。
 *
 * @param p apr_pool_t *
 * @param str const char * 変換対象文字列
 * @return char * 16進表示になった文字列
 */
DIVY_DECLARE(char *) dav_divy_atohex(apr_pool_t *p, const char *str);

/**
 * 与えられたバイナリデータdat を16進数の文字列として返却する。
 * バイナリデータですので、ヌル終端である必要はありません。
 * 逆にNULL終端であってもlen の範囲内であれば16進数表現に変換して
 * しまいますので、注意して下さい。
 *
 * (例) ABCDあい -> 41424344E38182E38184
 * (Reentrant)
 * 	dat が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 * (note)
 * 	少なくともlen の長さだけdat の領域が確保されていなければ
 * 	問題を起こすことがあります。呼び出し元にて保証して下さい。
 *
 * @param p apr_pool_t *
 * @param len apr_size_t dat の長さ
 * @param dat unsigned const char *
 * @return char * 16進表示になった文字列
 */
DIVY_DECLARE(char *) dav_divy_btohex(apr_pool_t *p, apr_size_t len,
						unsigned const char *dat);

/**
 * 与えられた16進数の文字列表現を文字列に変換する。
 * 関数dav_divy_atohexの逆変換。
 * (Reentrant)
 * 	str が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 *
 * @param apr_pool_t *
 * @param const char * str 16進数の文字列
 * @return char * 変換された文字列。
 * 		dav_divy_atohex から得られた文字列にならかの細工を行って
 * 		いる場合、正しく変換できないこともあります。
 * 		(0-9a-f以外の文字が指定された場合)
 * 		その場合、NULLを返却します。
 */
DIVY_DECLARE(char *) dav_divy_hextoa(apr_pool_t *p, const char *str);

/**
 * ISO8601形式にフォーマットされた文字列iso8601strをtime_t 型に変換する。
 * (Reentrant)
 * 	iso8601str が示すデータが処理中に変更されない場合に限り、
 * 	この関数はスレッドセーフです。
 *
 * (note) ISO8601形式
 * 	YYYY-MM-DD"T"HH:MI:SS"Z"
 *
 * @param p arp_pool_t *
 * @param iso8601str const char * 変換対象の文字列
 * @return time_t 変換された値
 * 		なお、フォーマットが不正であった場合には0を返します。
 */
DIVY_DECLARE(time_t) dav_divy_iso8601totime_t(apr_pool_t *p, const char *iso8601str);

/**
 * 指定された時間(time_t)を指定スタイルに変換して、バッファに詰めて
 * 返却する。なお、time_t は、GMTベースであるとして扱います。
 *
 * @param p apr_pool_t * datestr が確保されるプール
 * @param src_time const time_t 変換元の時間
 * @param style    int 変換スタイル
 * 		DIVY_TIME_STYLE_ISO8601  : ISO8601形式
 * 		DIVY_TIME_STYLE_RFC822   : RFC822形式
 * 		DIVY_TIME_STYLE_JAPANESE : 日本で標準的に使用されている日付形式
 * 		DIVY_TIME_STYLE_RFC2822  : RFC2822形式
 *
 * @param datestr char ** 日付形式の文字列(失敗していればNULL)
 * @return int 処理ステータス (0 : 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_format_time_t(apr_pool_t *p, const time_t src_time,
					int style, char **datestr);

/**
 * 現在時刻のエポックタイム(time_t)を取得する。
 * (Reentrant)
 * 	この関数はスレッドセーフです。
 *
 * @return time_t
 */
DIVY_DECLARE(time_t) dav_divy_get_now_epoch(void);

/**
 * 指定された平文plaintextを暗号化する。
 * (Reentrant)
 * 	plaintext が処理中に変更されない限り、この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param plaintext const char * 平文
 * @return const char * 暗号化した文字列
 */
DIVY_DECLARE(const char *) divy_encipher_text(apr_pool_t *p,
							const char *plaintext);

/**
 * 指定された暗号文ciphertextを復号する。
 * なお、ciphertext は、関数dav_divy_encipher_text を使って暗号化されたものでなければ
 * なりません。(そうではないものは復号できません。)
 * (Reentrant)
 * 	ciphertext が処理中に変更されない限り、この関数はスレッドセーフです。
 *
 * @param p apr_pool_t *
 * @param ciphertext const char * 暗号文
 * @return const char * 復号した文字列
 */
DIVY_DECLARE(const char *) divy_decipher_text(apr_pool_t *p,
							const char *ciphertext);

/**
 * 与えられたsizeが表すバイト数を操作者が見やすい形式にする。
 * [ 変換ルール ] (TeamFile クライアントに準拠)
 * 	* size < 1024                     : 何もしない (マイナスも含む)
 * 	* size > 1024 && size < 1024*1024 : KB表示した時の小数点第１位を切り捨て
 * 	                                    3桁ごとにカンマ表示
 *
 * @param p apr_pool_t *
 * @param size apr_int64_t 変換サイズ
 * @param roundbyte int 1KB未満を丸める
 * @return const char * KBまたはMB付の文字列
 */
DIVY_DECLARE(const char *) divy_byte2displaysize(apr_pool_t *p,
                                                 apr_int64_t size, int roundbyte);

/**
 * リクエストURIをユニークにするためのシーケンスURIパラメータを取得する。
 *
 * @param p apr_pool_t * シーケンスURIパラメータを割り当てるプール
 */
DIVY_DECLARE(const char *) divy_get_urlseq(apr_pool_t *p);

/**
 * f が示すファイルから1行(改行コードまで)を読み込んで文字列として返却する。
 * 読み込んだ文字列から改行コードは除去します。
 * なお、最大読み込みバイト数は DIVY_MAX_READLINE_BYTES で定義された値です。
 * 有効な文字が無ければ*linebuf はNULLになります。
 * (note)
 * 	DIVY_UTIL_COMMON_ERR の場合、*linebuf として戻される値は、エラーが
 * 	発生する直前までの値となります。
 *
 * @param p apr_pool_t * 作業用プール
 * @param f apr_file_t * 読み込むファイル
 * @param linebuf char **
 * @return int 処理ステータ
 * 		DIVY_UTIL_COMMON_OK  : 成功
 * 		DIVY_UTIL_COMMON_ERR : 失敗
 * 		DIVY_UTIL_COMMON_EOF : EOF
 */
DIVY_DECLARE(int) divy_file_readline(apr_pool_t *p, apr_file_t *f, char **linebuf);

/**
 * path が示すディレクトリのファイルシステム情報を取得して返却する。
 *
 * @param p apr_pool_t * 作業用プール
 * @param path const char * 対象ディレクトリのパス
 * @param sfs divy_statfs_t ** 取得したファイルシステム情報.
 *                            取得に失敗したらNULL
 * @return divy_error * エラー情報。エラーがなければNULL
 */
DIVY_DECLARE(divy_error *) divy_statfs(apr_pool_t *p,
					const char *path, divy_statfs_t **sfs);

/**
 * 指定されたcommand が示すコマンドを引数argv を付けて実行する。
 * なお、comannd がstdout, stderr に出力した結果は無視されます。
 *
 * (note) シェルコマンドの実行について
 * 	シェルスクリプトをこの関数を使って実行する場合には、cmdtype に
 * 	APR_SHELLCMD を指定して下さい。
 *
 * (note) exceptcd_set について
 * 	基本的には、起動されるコマンドからのステータスコードが 0 以外であったなら
 * 	エラーである方針を採っている。
 * 	だが、別の幾つかのステータスコードも"正常"であると扱って欲しい場合も
 * 	あるだろう。exceptcd_set は、このような用途に対応するため、正常であると
 * 	みなして欲しいステータスコード一覧を渡すための変数である。
 * 	(例)
 * 		int exceptcd_set[] = {3, 5, 0}
 * 		--> ステータスコード 3, 5, 0 が正常である。
 * 	なお、exceptcd_set の終端は必ず0 で終わっていなければなりません。さもないと
 * 	落ちてしまいます。これは仕様です。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param command const char *       実行するコマンドのフルパス(ex, /bin/ls)
 * @param argv const char * const [] 実行するコマンドとその引数。
 *                                   最後は必ずNULL終端(execvと同じ)
 *                                   (ex, argv[0]="/bin/ls", argv[1]="/tmp", argv[2]=NULL)
 * @param env const char * const *   起動される新しいプロセスの環境変数テーブル。
 *                                   NULL ならば無視される。
 * @param cmdtype apr_cmdtype_e      command の種類(see apr_cmdtype_e)
 * @param curdir  const char *       cd するディレクトリのパス。不要ならばNULLにする
 * @param exitcode int *             起動されたプロセスから戻されるステータスコード
 * @param exceptcd_set int *         エラー扱いしない起動プロセスのステータスコード。
 *                                   不要であればNULLにする。終端は必ず0。
 * @return int 終了ステータスコード
 *	 	DIVY_UTIL_COMMON_OK          : 正常
 * 		DIVY_UTIL_COMMON_ERR         : 失敗
 * 		DIVY_UTIL_COMMON_EXCEPTCD_SET: exceptcd_setに含まれていたステータスだった
 */
DIVY_DECLARE(int) divy_execute_cmd(apr_pool_t *p,
				const char *command, const char * const argv[],
				const char * const *env, apr_cmdtype_e cmdtype,
				const char *curdir, int *exitcode, int *exceptcd_set);

/**
 * コマンドライン文字列 cmdline をパースしてchar * の配列を生成して返却する。
 * 終端はNULLです。
 *
 * (例) /bin/ls -alF /tmp/abc	をパースした場合
 *  +----------------------------------------+
 *  | /bin/ls\0 | -alF\0 | /tmp/abc\0 | NULL |
 *  +----------------------------------------+
 *  半角スペースは見易くするためにいれたもの。実際のデータには入っていません。
 *
 * @param p apr_pool_t * 戻り値が確保されるプール
 * @param cmdline const char * コマンドライン文字列
 * @return char ** パースしたコマンドライン
 * 			cmdline がNULLまたは空であればNULLを返却します。
 */
DIVY_DECLARE(char **) divy_parse_commandline(apr_pool_t *p, const char *cmdline);

/**
 * n を３桁毎に","で区切った文字列に変換する。
 *
 * @param p apr_pool_t * 作業用プール
 * @param n apr_int64_t 変換する数値
 * @return const char * フォーマット変換された文字列
 */
DIVY_DECLARE(const char *) divy_i2displaynum(apr_pool_t *p, apr_int64_t n);

/**
 * 指定されたネットワークインターフェース名ifname に割り当てられた
 * IPV4のIPアドレスを取得して返却する。
 * 対象のネットワークインターフェースにIPアドレスが未設定であれば、
 * "0.0.0.0" が返却されます。
 *
 * @param p apr_pool_t *
 * @param ifname const char * ネットワークインターフェース名
 * @return 取得したIPアドレス
 */
DIVY_DECLARE(char *) divy_get_netifaddr(apr_pool_t *p, const char *ifname);

/**
 * 文字列str の中に含まれる大文字に出来る文字列を大文字に変換する.
 *
 * @param p apr_pool_t *
 * @param str const char * 変換対象の文字列
 * @return char * 変換した文字列
 */
DIVY_DECLARE(char *) divy_toupper_str(apr_pool_t *p, const char *str);

/**
 * 文字列str の中に含まれる小文字に出来る文字列を小文字に変換する.
 *
 * @param p apr_pool_t *
 * @param str const char * 変換対象の文字列
 * @return char * 変換した文字列
 */
DIVY_DECLARE(char *) divy_tolower_str(apr_pool_t *p, const char *str);

/**
 * 文字列str をURLエンコードする.
 *
 * [ エンコードのルール ]
 *   * 1バイトの英数字(a-zA-Z0-9)はそのまま
 *   * 記号(*-.@_) もそのまま
 *   * 半角スペースは"+" にする
 *   * 上記以外は全て%xx 形式にする
 *
 * @param p apr_pool_t *
 * @param str const char * 変換対象の文字列
 * @return char * 変換した文字列
 */
DIVY_DECLARE(char *) divy_url_encode(apr_pool_t *p, const char *str);

/**
 * ランダム文字列を作成する
 * @param r request_rec
 * @return セッションID char *
 */
DIVY_DECLARE(char*) make_random_string(apr_pool_t *pool);

#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_UTIL_COMMON_H */
