/**
 * $Id$
 *
 * util_thread.h
 *
 * スレッド専用ユーティリティ関数用ヘッダファイル
 *
 * (お願い)
 *	このファイルには、スレッド処理固有の関数を入れてください。
 *	例えば、pthreadをaprを使用せずに生で呼び出すようなコードです。
 *	目的は、プラットフォーム間で異なるスレッド関数の呼び出しを
 *	ここで吸収することにあります。aprになっていれば、その限り
 *	ではありませんが、他と共有できそうなものであればここに記述
 *	してください。
 *
 * 2003/12/24 Wed takehara NEW
 */
#ifndef INCLUDE_UTIL_THREAD_H
#define INCLUDE_UTIL_THREAD_H

#include "apr.h"
#include "apr_thread_proc.h"
#include "apr_thread_mutex.h"
#include "apr_thread_cond.h"

#include "tfr.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/*--------------------------------------------------------------
  Define fixed value
  --------------------------------------------------------------*/


/*--------------------------------------------------------------
  Define macros
  --------------------------------------------------------------*/
/**
 * スレッドクリーンアップハンドラの登録・解除マクロ
 * (note)
 * 	本来なら関数化すべきですが、pthreadが用意している登録・解除用
 * 	関数(実はマクロ)に細工がしてあるため事実上不可能となっています。
 * 	なので、こちらもマクロで対処します。
 * (note)
 * 	pthread の実装を利用する場合、登録と解除用のマクロは同じ関数
 * 	ブロックの中でコールしなければなりません。別の関数や別のブロック
 * 	(ifやwhileなど)では、コンパイルエラーになり実行すらできません。
 */

/*
 * クリーンアップハンドラの登録
 *
 * @param handler void (*)(void *) クリーンアップハンドラ関数へのポインタ
 * @param arg void * クリーンアップハンドラ関数に渡される引数へのポインタ
 */
#define DAV_DIVY_THREAD_CLEANUP_PUSH(handler,arg) pthread_cleanup_push(handler,arg)

/*
 * クリーンアップハンドラの解除
 *
 * @param execute int 解除した後、クリーンアップハンドラを実行するかどうか
 *                    0: 実行しない / 非0: 実行する
 */
#define DAV_DIVY_THREAD_CLEANUP_POP(execute) pthread_cleanup_pop(execute)

/*--------------------------------------------------------------
  Define enum
  --------------------------------------------------------------*/
/**
 * MPMの種類を表す列挙型
 */
enum __divy_mpm_type {
	DIVY_MPM_TYPE_PROCESS = 0,	/* プロセスタイプのMPMである */
	DIVY_MPM_TYPE_THREAD,		/* スレッドタイプのMPMである */
};
typedef enum __divy_mpm_type	divy_mpm_type;

/*--------------------------------------------------------------
  Declare public function
  --------------------------------------------------------------*/

/**
 * 指定されたsignum のシグナルだけをアンブロックする。
 * (note)
 * 	consider_mpm が0に指定されていた場合には、ロードされているMPMの種類に
 * 	依らず、強制的にアンブロック処理を実施します。
 * 	1であれば、MPMの種類を考慮してアンブロック処理を実施します。
 * 	(アンブロックしないという選択をすることもあります。)
 * (note)
 * 	SIGKILL, SIGSTOPや同期シグナルをアングロックすることは出来ません。
 *
 * @param signum int シグナル番号 (SIGTERM, SIGUSR1など)
 * @param consider_mpm int MPMによる影響を考慮するかどうか(1: 考慮する / 0: 考慮しない)
 */
DIVY_DECLARE(void) dav_divy_unblock_signal(int signum, int consider_mpm);

/**
 * 指定されたthread_id が示すスレッドを取り消す(cancel)
 * (note)
 * 	APRのスレッドライブラリにはスレッドの取り消し(cancel)機能が
 * 	存在しないため、独自に実装しました。
 * 	そのためプラットフォームによっては使用できない可能性があります。
 * 	(POSIX スレッドAPIが実装されていないプラットフォームのこと)
 *
 * @param thread_id apr_thread_t * スレッド識別子
 * @return int 処理ステータス (非0の場合失敗)
 */
DIVY_DECLARE(int) dav_divy_thread_cancel(apr_thread_t *thread_id);

/**
 * MPM の種類(プロセス型であるかスレッド型であるか)を設定する
 * この結果をスレッド動作の決定に使用します。
 * (note)
 * 	この関数は1プロセスに付き1度だけコールされるよう呼び出し元にて
 * 	保証して下さい。
 *
 * @param mpm_type divy_mpm_type MPMの種類を設定する
 */
DIVY_DECLARE(void) dav_divy_set_mpm_type(divy_mpm_type mpm_type);


#ifdef __cplusplus
}
#endif

#endif	/* INCLUDE_UTIL_THREAD_H */

