/**
 * $Id$
 *
 * Array (可変長配列コンテナ) の宣言・定義を行うファイル
 *
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */

#include "tf_array.h"
#include "util_common.h"

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
#define DIVY_ARRAY_DEFAULT_NALLOC	10
#define DIVY_ARRAY_DEFAULT_DELTA	20

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * [ 不完全型の定義 ]
 *  任意のポインタ型の変数を格納する可変長配列構造体
 *  実体を保持しない点に注意。実体を持たせたいのなら、apr_array_header_t を
 *  利用してください。その場合にはRISCマシンにおけるアライメントバウンダリの
 *  問題に注意する必要があります。
 */
struct __divy_array_t {
	apr_int32_t nalloc;	/* 可変長配列として割り当てられている総要素数 */
	apr_int32_t nelts;	/* 可変長配列に格納されているの要素の数 */
	apr_int32_t delta;	/* 次回拡張する際に使用される配列の長さ */
	apr_pool_t *p;		/* 作業用プール */

	/* 可変長配列へのポインタ(void *) をlength 個持つ配列 */
	void **elements;
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _realloc_array(divy_array_t *array);

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 指定された初期要素数size 個の要素をもつポインタ保持用可変長配列を生成する。
 *
 */
DIVY_DECLARE(divy_array_t *) divy_array_make(apr_pool_t *p, apr_int32_t size)
{
	divy_array_t *array = apr_pcalloc(p, sizeof(divy_array_t));
	array->p = p;

	if (size >= 0) {
		array->nalloc = size;
		/* delta の算出 */
		array->delta = array->nalloc * 2;
	}
	else {
		array->nalloc = DIVY_ARRAY_DEFAULT_NALLOC;
		array->delta  = DIVY_ARRAY_DEFAULT_DELTA;
	}
	/* 可変長配列の枠を生成 */
	array->elements = apr_pcalloc(p, sizeof(void *) * array->nalloc);
	array->nelts    = 0;

	return array;
}

/**
 * 可変長配列array の末尾にelement が示すオブジェクトを追加する。
 *
 */
DIVY_DECLARE(void) divy_array_add(divy_array_t *array, void *element)
{
	if (array == NULL) {
		return;
	}

	if (array->nalloc < array->nelts + 1) {
		_realloc_array(array);
	}

	array->elements[array->nelts] = element;
	array->nelts++;
}

/**
 * 可変長配列array の位置index のオブジェクトをelement で置き換える。
 * index が範囲外であれば何もしません。
 *
 */
DIVY_DECLARE(void) divy_array_set(divy_array_t *array,
					apr_int32_t index, void *element)
{
	if (array == NULL || index < 0 || index > array->nelts) {
		return;
	}

	array->elements[index] = element;
}

/**
 * 可変長配列array の位置index にあるオブジェクトを削除し、返却する。
 *
 */
DIVY_DECLARE(void *) divy_array_remove(divy_array_t *array, apr_int32_t index)
{
	void *element;
	apr_int32_t i;

	if (array == NULL || index < 0 || index > array->nelts) {
		return NULL;
	}

	element = array->elements[index];	/* 要素の取り出し */
	/* index 要素を除去する */
	for (i = index; i < array->nelts - 1; i++) {
		array->elements[i] = array->elements[i+1];
	}
	array->elements[array->nelts - 1] = NULL;
	array->nelts--;	/* 要素を1つ減らす */

	return element;
}

/**
 * 可変長配列array の総要素数を取得する。
 *
 */
DIVY_DECLARE(apr_int32_t) divy_array_getlength(divy_array_t *array)
{
	if (array == NULL) return 0;

	return array->nelts;
}

/**
 * 可変長配列array の中の位置index にある要素を取得する。
 * remove と異なり、index 位置の要素は除去されません。
 *
 */
DIVY_DECLARE(void *) divy_array_get(divy_array_t *array, apr_int32_t index)
{
	if (array == NULL || index < 0 || index > array->nelts) {
		return NULL;
	}

	return array->elements[index];
}

/**
 * 可変長配列array の要素を全てクリアする。
 *
 */
DIVY_DECLARE(void) divy_array_clear(divy_array_t *array)
{
	if (array == NULL) return;

	array->nalloc = 0;
	array->nelts  = 0;
	array->elements = NULL;
}

/**
 * 可変長配列array の要素全てを取り出す。
 *
 */
DIVY_DECLARE(void **) divy_array_getelements(divy_array_t *array)
{
	return array->elements;
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * 可変長配列array を拡張する。
 *
 * @param array divy_array_t * 可変長配列へのポインタ
 */
static void _realloc_array(divy_array_t *array)
{
	void **data;
	apr_int32_t new_size;

	if (array == NULL) return;

	/* 連続領域を確保するため、新たに確保しなおす */
	new_size = array->nalloc + array->delta;
	data = apr_pcalloc(array->p, sizeof(void *) * new_size);
	memcpy(data, array->elements, sizeof(void *) * array->nalloc);

	array->nalloc   = new_size;
	array->elements = data;
}


