/**
 * $Id$
 *
 * Set (文字列集合コンテナ) の定義を行うファイル
 */
#include "apr.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"

#include "tfr.h"
#include "tf_cset.h"

/*------------------------------------------------------------------------------
  Define structures
  ----------------------------------------------------------------------------*/
/**
 * [ 不完全型の定義 ]
 * 文字列からなる値の集合を保持する構造体 (ユーティリティ構造体)
 * 同値のデータはただ１つだけ保持できます。
 */
struct __divy_cset_t {
	/* 集合を実現するハッシュ */
	apr_hash_t *_set;
	apr_pool_t *p;
};

/**
 * [ 不完全型の定義 ]
 * divy_cset_t のiterator (ユーティリティ構造体)
 */
struct __divy_cset_index_t {
	/* 現状、iterator の実装はハッシュ関数に頼っている */
	apr_hash_index_t *hi;

	divy_cset_t *cst;
	apr_pool_t *p;
};

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * divy_cset_t の生成を行う
 */
DIVY_DECLARE(divy_cset_t *) divy_cset_make(apr_pool_t *p)
{
	divy_cset_t *cst = apr_pcalloc(p, sizeof(divy_cset_t));
	cst->_set = apr_hash_make(p);
	cst->p = p;

	return cst;
}

/**
 * cst が示す文字列集合の中にval を追加する。
 */
DIVY_DECLARE(void) divy_cset_set(divy_cset_t *cst, const char *val)
{
	if (cst == NULL) return;

	/* 値が埋まっていることを示す"x"を詰めておく */
	apr_hash_set(cst->_set, val, APR_HASH_KEY_STRING, "x");
}

/**
 * 指定されたval が文字列集合cst の中に含まれているかどうか。
 *
 */
DIVY_DECLARE(int) divy_cset_contains(divy_cset_t *cst, const char  *val)
{
	const char *str = NULL;
	if (cst == NULL) return 0;	/* ないとしか言いようがない */

	str = apr_hash_get(cst->_set, val, APR_HASH_KEY_STRING);
	if (str != NULL && *str == 'x') {
		return 1;	/* 集合に含まれていた */
	}

	return 0;
}

/**
 * 文字集合cst から指定されたvalが示す文字列を削除する。
 *
 */
DIVY_DECLARE(void) divy_cset_remove(divy_cset_t *cst, const char *val)
{
	if (cst == NULL || val == NULL) return;	/* 何も出来ません */

	/* ハッシュの値をNULL にする --> 要素から消えたことになる */
	apr_hash_set(cst->_set, val, APR_HASH_KEY_STRING, NULL);
}

/**
 * 文字列集合に含まれる要素の数を返却する
 *
 * @param cst divy_cset_t * 文字列集合を表す構造体へのポインタ
 * @return unsigned int 要素の数
 */
DIVY_DECLARE(unsigned int) divy_cset_count(divy_cset_t *cst)
{
	if (cst == NULL) return 0;	/* ないとしか言いようがない */

	return apr_hash_count(cst->_set);
}

/**
 * 文字列集合cset のiterator を取得する。
 *
 */
DIVY_DECLARE(divy_cset_index_t *) divy_cset_first(divy_cset_t *cst)
{
	divy_cset_index_t *ci;
	apr_hash_index_t *hi;

	if (cst == NULL) return NULL;

	hi = apr_hash_first(cst->p, cst->_set);
	if (hi == NULL) {
		/* set の中に要素が入っていなかった時 */
		return NULL;	/* ciも生成しない */
	}
		
	ci = apr_pcalloc(cst->p, sizeof(divy_cset_index_t));

	ci->hi  = hi;
	ci->cst = cst;
	ci->p   = cst->p;

	return ci;
}

/**
 * iteratorであるci を１つ進める。
 *
 */
DIVY_DECLARE(divy_cset_index_t *) divy_cset_next(divy_cset_index_t *ci)
{
	apr_hash_index_t *hi;

	if (ci == NULL) return NULL;

	hi = apr_hash_next(ci->hi);
	if (hi) {
		ci->hi = hi;
		return ci;	/* 元のまま */
	}
	else {
		/* もうiteration する値が無くなった場合 */
		return NULL;
	}
}

/**
 * iterator が現在指し示す値を取得する。
 *
 */
DIVY_DECLARE(void) divy_cset_this(divy_cset_index_t *ci, const char **val)
{
	*val = NULL;
	if (ci == NULL || ci->hi == NULL) return;

	apr_hash_this(ci->hi, (const void **) val, NULL, NULL); 
}

/**
 * 文字列集合cst に含まれる要素を文字列からなる配列として返却する
 *
 */
DIVY_DECLARE(char **) divy_cset_toarray(divy_cset_t *cst)
{
	char **array = NULL, **first = NULL;
	unsigned int count = 0;
	const char *str;
	apr_pool_t *p         = cst->p;
	divy_cset_index_t *ci = NULL;

	if (cst == NULL) return NULL;

	/* 要素のカウント */
	count = divy_cset_count(cst);
	if (count == 0) return NULL;

	first = array = apr_pcalloc(p, sizeof(char *) * (count + 1));

	for (ci = divy_cset_first(cst); ci; ci = divy_cset_next(ci)) {

		/* 文字列集合から１つの文字列を取得する */
		divy_cset_this(ci, &str);
		if (str) {
			*array = apr_pstrdup(p, str);	/* コピーする */
			array++;
		}
	}
	*array = NULL;	/* sentinel */

	return first;
}

/**
 * cstが表す文字列集合をハッシュのキー集合として返却する.
 * (キー値が文字集合のハッシュを返却する)
 *
 */
DIVY_DECLARE(apr_hash_t *) divy_cset_tohash(divy_cset_t *cst)
{
	if (cst == NULL) return NULL;

	/* 内部実装のハッシュをそのまま返却する */
	return cst->_set;
}


