/**
 * $Id$
 *
 * tf_db.c
 *
 * DBプロバイダインターフェースの定義を行う。
 * (note)
 * 	ここでの実装は、DBプロバイダおよび、TeamFileモジュールとDBプロバイダ間の
 * 	インターフェースレイヤ(util_db.c) が使用します。それ以外から使用するのは
 * 	多くの場合誤りです。
 * 	このファイルがinclude可能なヘッダファイルは、apr_xxxとTeamFileモジュール
 * 	に依存しない共通ヘッダだけです。ですので、Apacheログを出すことすらでき
 * 	ません。(http_log.h をinclude出来ないので)
 *
 * 2003/01/06 Mon takehara NEW
 * 2004/02/08 Sun takehara Change role of this file.
 */
#ifdef HAVE_CONFIG_H
#ifndef INCLUDE_UTIL_CONFIG_H
#define INCLUDE_UTIL_CONFIG_H
#include "config.h"
#endif	/* INCLUDE_UTIL_CONFIG_H */
#endif	/* HAVE_CONFIG_H */

/* Apache header files */
#include "apr.h"
#include "apr_pools.h"
#include "apr_hooks.h"
#include "apr_strings.h"
#include "apr_hash.h"

#include "tf_db.h"
#include "util_common.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */

/*------------------------------------------------------------------------------
  Define static values and macro
  ----------------------------------------------------------------------------*/
/*
 * DBプロバイダライセンスキーに含まれるライセンス情報を取得するためのキー値の定義
 */
#define DIVY_DBLKEY_TYPE     "tp"	/* DBプロバイダの種類 */
#define DIVY_DBLKEY_STARTDT  "st"	/* ライセンス発効日(エポックタイム)   */
#define DIVY_DBLKEY_TRIALDU  "du"	/* 試用期間(日) */

/*------------------------------------------------------------------------------
  Define Hook structure and function
  ----------------------------------------------------------------------------*/
/**
 * Hook構造体の定義と生成
 */
APR_HOOK_STRUCT(
	APR_HOOK_LINK(create_dbdatasource)
	)

/**
 * 指定されたproviderType に一致するDBプロバイダであった時、
 * DB プロバイダ構造体を作成 & 返却する関数
 * のためのHookハンドラ.
 * DB プロバイダの初期化(register_hookハンドラ) に利用します。
 * (note)
 * 	providerType と一致しないDBプロバイダであった場合、DB_DECLINED を返却し
 * 	DbDataSourceへのポインタを返却しないこと。
 * 	逆に一致するDBプロバイダであった場合、DB_OK を返却し、DbDataSourceへの
 * 	ポインタを返却すること。
 */
APR_IMPLEMENT_EXTERNAL_HOOK_RUN_FIRST(divy, DIVY, int, create_dbdatasource,
		(apr_pool_t *p, const char *providerType, DbDataSource **dbds),
		(p, providerType, dbds), DB_DECLINED)

/*------------------------------------------------------------------------------
  Define structure
  ----------------------------------------------------------------------------*/
/*
 * [ 不完全型の実装 ]
 * ある種類のDBプロバイダに関するライセンス情報を保持する構造体
 * (note)
 * 	接続先ごとではなく、種類ごとにライセンス情報は保持されます。
 */
struct __db_ProviderLicense {

	char *dbmstype;		/* DB type (DBMSの種類名) */
	time_t starttime;	/* ライセンス発行日時のエポックタイム */
#define DIVY_DB_LICENSE_FOREVER 0	/* 無制限であることを示す */
	time_t duration;	/* ライセンス発効日から使用可能な期間(秒に変換) */
};

/*------------------------------------------------------------------------------
  Declare Privete Functions
  ----------------------------------------------------------------------------*/
static apr_status_t _cleanup_dbconn(void *data);

/*------------------------------------------------------------------------------
  Define Public Functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたproviderType のDBデータソース(DbDataSource) を取得して返却する。
 * 取得に失敗した場合には、NULLを返却し、戻り値でステータスを返却します。
 *
 */
DIVY_DECLARE(RET_STAT) divy_db_lookup_dbdatasource(apr_pool_t *p,
						const char *providerType,
						const char *licensekey,
						DbDataSource **dbds)
{
	int ret;
	RET_STAT retst;
	DbProviderLicense *license = NULL;
	apr_hash_t *dblicense_h;

	if (p == NULL || IS_EMPTY(providerType)) {
		return DB_INVALID_PARAMS;
	}

	*dbds = NULL;	/* 初期化 */

	/* DB プロバイダのライセンスを検証する */
	if (IS_EMPTY(licensekey)) {
		return DB_LICENSE_NOTHING;	/* ライセンスなし */
	}
	dblicense_h = apr_hash_make(p);
	ret = divy_db_parse_licensekey(p, licensekey, dblicense_h); 
	if (ret) {
		return DB_LICENSE_NOTHING;	/* ライセンスエラー */
	}

	license = apr_hash_get(dblicense_h, providerType, APR_HASH_KEY_STRING);
	retst = divy_db_validate_dblicense(p, license);
	if (retst != DB_LICENSE_VALID) {
		return retst;	/* ライセンスエラー */
	}

	/* 全DBプロバイダに問い合わせる */
	ret = divy_run_create_dbdatasource(p, providerType, dbds);
	if (ret == DB_DECLINED) {
		return DB_NOTFOUND_DBDS;
	}

	/* 既に分かっている値だけを格納する */
	(*dbds)->dbmstype       = apr_pstrdup(p, providerType);
	(*dbds)->dbpool         = 0;	/* DBプールなし */
	(*dbds)->dbvhostshare   = 0;	/* VirtualHost間でコネクションは共有しない */
	(*dbds)->dbminspareconn = 0;	/* 最小プール数 */
	(*dbds)->dbmaxspareconn = 0;	/* 最大プール数 */

	return DB_SUCCESS;
}

/**
 * 指定されたdbds が示すDBデータソースからアクティブなDB コネクションを取得して
 * 返却する。
 *
 */
DIVY_DECLARE(RET_STAT) divy_db_getDbConn(apr_pool_t *p,
					DbDataSource *dbds, DbConn **dbconn)
{
	if (dbds == NULL) {
		return DB_INVALID_PARAMS;
	}

	*dbconn = NULL;		/* 初期化 */

	/* DbConn の取得 */
	*dbconn = dbds->getDbConn(dbds, p);
	if ((*dbconn)->getCode(*dbconn) != DB_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn. Reason: %s",
			REPLACE_NULL((*dbconn)->getMsg(*dbconn)));
		if (*dbconn != NULL) (*dbconn)->close(*dbconn);	/* closeが必要 */

		return DB_FAILED_DBCONN;
	}

	/* クリーンアップハンドラへの登録 */
	apr_pool_cleanup_register(p, *dbconn, _cleanup_dbconn, apr_pool_cleanup_null);

	return DB_SUCCESS;
}

/**
 * DBプロバイダライセンスdblicensekey をパースし、DBプロバイダの種類をキーとして
 * dblicense_h にライセンス構造体を格納する。
 *
 */
DIVY_DECLARE(int) divy_db_parse_licensekey(apr_pool_t *p,
						const char *dblicensekey,
						apr_hash_t *dblicense_h)
{
	char *plaintext;
	char *token, *key, *val;
	char *tok_cntx, *key_cntx;
	DbProviderLicense *license = NULL;

	if (dblicense_h == NULL || IS_EMPTY(dblicensekey)) {
		return 0;
	}

	/* 
	 * 取得したライセンスキーを復号化＆解析して、値を取り出す
	 */
	plaintext = (char *) divy_decipher_text(p, dblicensekey);
	if (plaintext == NULL) {
		/* ライセンスキーが正しいフォーマットではない / 改竄されていた */
		return 1;
	}

	license = apr_pcalloc(p, sizeof(DbProviderLicense));
	license->starttime = -1;	/* 初期値 */
	license->duration  = -1;

	/*
	 * ";" でトークンを分ける
	 */
	while ((token = apr_strtok(plaintext, ";", &tok_cntx)) != NULL) {
		plaintext = NULL;
		/*
		 * "=" でトークンを分ける
		 */
		key = apr_strtok(token, "=", &key_cntx);
		if (IS_EMPTY(key)) return 1;

		val = apr_strtok(NULL, "=", &key_cntx);
		if (IS_EMPTY(val)) return 1;

		if (strcmp(key, DIVY_DBLKEY_TYPE) == 0) {
			license->dbmstype = apr_pstrdup(p, val);
		}
		else if (strcmp(key, DIVY_DBLKEY_STARTDT) == 0) {
			if (!dav_divy_isdigit_str(val)) {
				return 1;
			}
			license->starttime = (time_t) atol(val);
		}
		else if (strcmp(key, DIVY_DBLKEY_TRIALDU) == 0) {
			if (!dav_divy_isdigit_str(val)) {
				return 1;
			}
			license->duration = atol(val);
			if (license->duration > 0) {
				/* 日単位 -> 秒単位 */
				license->duration = license->duration * 60 * 60 * 24;
			}
		}
	}

	/* DBプロバイダライセンスをハッシュに入れる */
	if (IS_FILLED(license->dbmstype) &&
		license->starttime > 0 && license->duration >= 0) {
		apr_hash_set(dblicense_h, license->dbmstype,
					APR_HASH_KEY_STRING, license);
	}

	return 0;
}

/**
 * 指定されたlicense が使用可能な適切なDBプロバイダライセンスかどうかを検証する。
 *
 * (note) ライセンスが適切であると判断する条件
 * 	(1) DBプロバイダライセンスキーが登録されていること
 * 	(2) ライセンス期限内もしくは、無制限ライセンスであること
 *
 */
DIVY_DECLARE(RET_STAT) divy_db_validate_dblicense(apr_pool_t *p,
						DbProviderLicense *license)
{
	if (license == NULL) {
		return DB_LICENSE_NOTHING;
	}

	/* ライセンスは有効期限内か? */
	if (license->duration == DIVY_DB_LICENSE_FOREVER) {
		return DB_LICENSE_VALID;	/* 無制限ライセンス!! */
	}
	else if (dav_divy_get_now_epoch() >
				license->starttime + license->duration) {
		/* ライセンス切れでした */
		return DB_LICENSE_EXPIRE;
	}

	return DB_LICENSE_VALID;
}

/**
 * 指定されたstrが示す文字列の中からバインド変数の識別文字'?'の位置を検索し
 * 見つかった位置のポインタを返却する。
 * 但し、'?'がシングルクオートと共に指定される、つまり単なる文字列の時には、
 * バインド変数と認識されません。
 *
 */
DIVY_DECLARE(const char *) dav_divy_find_bindposition(const char *str)
{
	const char *sp;
	int inside_quoto = 0;	/* シングルクオートの中を調べていることを示す */

	if (IS_EMPTY(str)) return NULL;

	sp = str;
	while (*sp != '\0') {
		if (*sp == '\'') {
			if (inside_quoto == 0) {
				inside_quoto = 1;
			}
			else {
				inside_quoto = 0;
			}
		}
		else if (*sp == '?') {
			if (inside_quoto == 0) {
				break;
			}
		}

		sp++;	/* 1つ進める */
	}

	/* 見つからなかった場合 */
	if (*sp == '\0') {
		return NULL;
	}

	return sp;	/* 見つかった位置を返す */
}

/**
 * getxxx系メソッドが利用するpoolを設定する。(2005/05/16 Mon takehara 新規追加)
 *
 */
DIVY_DECLARE(void) divy_db_set_rset_outputpool(DbResultSet *rset, apr_pool_t *scratchpool)
{
	if (scratchpool != NULL) {
		rset->__scratchpool = scratchpool;
	}
}

/*------------------------------------------------------------------------------
  Define Privete Functions
  ----------------------------------------------------------------------------*/
static apr_status_t _cleanup_dbconn(void *data)
{
	DbConn *dbconn = data;

	if (dbconn == NULL) {
		return APR_SUCCESS;	/* それでも成功と返す必要あり */
	}

	/* DBコネクションを閉じる */
	dbconn->close(dbconn);
	if (dbconn->getCode(dbconn) != DB_SUCCESS) {
		/* 失敗したらログだけ出しておく */
		LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to close DbConn. "
			"(providerName = %s) Reason: %s",
			dbconn->dbds->dbmsname, dbconn->getMsg(dbconn));
	}

	return APR_SUCCESS;
}

