/**
 * $Id$
 *
 * tf_errlog.c
 *
 * エラーログ関連の構造体/関数の定義
 * Apache本体に依存してはなりません。
 */
#ifdef HAVE_CONFIG_H
#ifndef INCLUDE_UTIL_CONFIG_H
#define INCLUDE_UTIL_CONFIG_H
#include "config.h"
#endif	/* INCLUDE_UTIL_CONFIG_H */
#endif	/* HAVE_CONFIG_H */


#include "apr.h"
#include "apr_pools.h"
#include "apr_time.h"
#include "tf_errlog.h"
#include "util_common.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */
#if APR_HAVE_STDARG_H
#include <stdarg.h>
#endif	/* APR_HAVE_STDARG_H */

/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structures
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define static value
  ----------------------------------------------------------------------------*/
/**
 * ライブラリロガーが利用するロガー実体へのポインタ
 */
static divy_lerrlog_t _lerrlog = NULL;

/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/
/**
 * ログ関数へのポインタを設定する
 *
 */
DIVY_DECLARE(void) divy_set_lerrlogger(divy_lerrlog_t lerrlog)
{
	_lerrlog = lerrlog;
}

/**
 * ライブラリロガー関数へのポインタを取得する
 *
 */
DIVY_DECLARE(divy_lerrlog_t) divy_get_lerrlogger()
{
	return _lerrlog;
}

/**
 * divy_lerrlog_t 型のロガー(stderr 出力)
 *
 */
DIVY_DECLARE(void) divy_output_err2stderr(const char *file, int line,
					int level, apr_status_t status,
					apr_pool_t *p, const char *fmt, ...)
{
	const char *slevel;
	char sctime[APR_CTIME_LEN];
	va_list args;

	/* エラーログレベルの取得 */
	slevel = divy_get_errlevelstr(level);
	if (IS_FILLED(slevel) && IS_FILLED(fmt)) {

		/* 現在の時刻を取得する */
		apr_ctime(sctime, apr_time_now());

		/* ログを出力する */
		va_start(args, fmt);
		fprintf(stderr, "[%s] [%s] ", sctime, slevel);
		vfprintf(stderr, fmt, args);
		fprintf(stderr, "\n");
		va_end(args);
	}
}

/**
 * divy_lerrlog_t 型のロガー(stdout 出力)
 *
 */
DIVY_DECLARE(void) divy_output_err2stdout(const char *file, int line,
					int level, apr_status_t status,
					apr_pool_t *p, const char *fmt, ...)
{
	const char *slevel;
	char sctime[APR_CTIME_LEN];
	va_list args;

	/* エラーログレベルの取得 */
	slevel = divy_get_errlevelstr(level);
	if (IS_FILLED(slevel) && IS_FILLED(fmt)) {

		/* 現在の時刻を取得する */
		apr_ctime(sctime, apr_time_now());

		/* ログを出力する */
		va_start(args, fmt);
		fprintf(stdout, "[%s] [%s] ", sctime, slevel);
		vfprintf(stdout, fmt, args);
		fprintf(stdout, "\n");
		va_end(args);
	}
}

/**
 * エラーログで使用するエラーレベル文字列を返却する
 *
 */
DIVY_DECLARE(const char *) divy_get_errlevelstr(int lv)
{
	if (lv == LLOG_EMERG)
		return "emerg";
	else if (lv == LLOG_ALERT)
		return "alert";
	else if (lv == LLOG_CRIT)
		return "crit";
	else if (lv == LLOG_ERR)
		return "error";
	else if (lv == LLOG_WARNING)
		return "warn";
	else if (lv == LLOG_NOTICE)
		return "notice";
	else
		return NULL;
}

/**
 * エラー情報の記録を行い、エラーログ構造体のインスタンスを生成して返却する。
 *
 */
DIVY_DECLARE(divy_error *) divy_new_error(apr_pool_t *p, int level, int status,
						char *funcname, int lineno,
						const char *desc)
{
	divy_error *err = apr_palloc(p, sizeof(divy_error));

	err->level    = level;
	err->status   = status;
	err->funcname = funcname;
	err->lineno   = lineno;
	err->desc     = TO_EMPTY(desc);
	err->prev     = NULL;
	err->next     = NULL;

	return err;
}

/**
 * 指定されたエラー情報を記録し、それを指定されたerr と連結して返却する。
 *
 */
DIVY_DECLARE(divy_error *) divy_push_error(apr_pool_t *p, int level, int status,
						char *funcname, int lineno,
						const char *desc, divy_error *prev)
{
	divy_error *err = divy_new_error(p, level, status, funcname, lineno, desc);

	/* リストを連結する */
	err->prev = prev;
	prev->next= err;

	return err;
}


