/**
 * $Id$
 *
 * tf_provider.c
 *
 * xxx プロバイダライブラリの管理を行う構造体・関数の実装
 */
#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "apr_hooks.h"
#include "util_common.h"
#include "tf_provider.h"
#if APR_HAVE_STDIO_H
#include <stdio.h>
#endif
#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif
#if APR_HAVE_UNISTD_H
#include <unistd.h>
#endif

/*------------------------------------------------------------------------------
  Declare values
  ----------------------------------------------------------------------------*/
extern divy_lib_provider_t *divy_preloaded_lib_providers[];

/*------------------------------------------------------------------------------
  Define Hook structure and function
  ----------------------------------------------------------------------------*/
APR_HOOK_STRUCT(
	APR_HOOK_LINK(child_init)
	)

APR_IMPLEMENT_EXTERNAL_HOOK_VOID(divy, DIVY, child_init,
				(apr_pool_t *pchild),(pchild))

/*------------------------------------------------------------------------------
  Define static values
  ----------------------------------------------------------------------------*/
/**
 * プロバイダライブライハンドラ構造体へのポインタを保持する
 * リストの先頭へのポインタ
 */
static divy_lib_provider_t *divy_lib_provider_top = NULL;

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static apr_status_t _pconf_cleanup(void *data);

/*------------------------------------------------------------------------------
  Define public functions
  ----------------------------------------------------------------------------*/
/**
 * プロバイダライブラリマネージャの初期化
 *
 */
DIVY_DECLARE(int) divy_init_providers_env(apr_pool_t *pconf)
{
	/*
	 * APR グローバルプールの状態をチェック(for library)
	 * (note)
	 * 	APRが定義するフックハンドラを利用するために
	 * 	まずはAPR自身が初期化されていることを確認します。
	 * 	この関数がApacheモジュールとして動作する場合には
	 * 	必ず初期化済みになっています。
	 */
	if (apr_hook_global_pool == NULL) {
		apr_hook_global_pool = pconf;	/* プールの設定 */
	}

	return DIVY_LIB_ST_OK;
}

DIVY_DECLARE(void) divy_hook_child_exit(divy_lib_child_exit_t child_exit,
					apr_pool_t *pchild, void *data)
{
	apr_pool_cleanup_register(pchild, data, child_exit,
					apr_pool_cleanup_null);
}

/**
 * xxxプロバイダライブラリハンドラをシステムに登録する。
 *
 */
DIVY_DECLARE(int) divy_register_lib_provider(divy_lib_provider_t *provider)
{
	/* ロガーがアクティブかどうかを判定する */
	if (divy_get_lerrlogger() == NULL) {
		/*
		 * stderr のロガーを使う
		 * (note)
		 * 	この関数が呼び出されるステージではApacheのロガーが
		 * 	アクティブでない可能性が高い。だがエラーメッセージは
		 * 	出さなければならないので、stderr に出力するようにした。
		 * 	TeamFile のモジュールと共に利用する場合には後で
		 * 	置き換えてくれるのであまり気にしない。
		 */
		divy_set_lerrlogger(divy_output_err2stderr);
	}

	if (provider == NULL) {
		LERRLOG0(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"provider is null.");
		return DIVY_LIB_ST_ERR;
	}

	/* 多重初期化防止(1) */
	if (provider->next != NULL) {
		return DIVY_LIB_ST_OK;
	}

	/* インターフェースとのバージョン互換をチェック */
	if (provider->version != DIVY_LIB_IF_VERSION) {
		LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The version of lib provider is diffrent from "
			"interface version. Please check."
			"(provider = %d, interface = %d)",
			provider->version, DIVY_LIB_IF_VERSION);
		return DIVY_LIB_ST_ERR;
	}

	/* 多重初期化防止(2) */
	if (provider->is_init) {
		return DIVY_LIB_ST_OK;	/* 何もしない */
	}

	/* リストに繋げる */
	if (divy_lib_provider_top == NULL) {
		divy_lib_provider_top = provider;
		divy_lib_provider_top->next = NULL;
	}
	else {
		divy_lib_provider_t *pr;

		/* 同じタイプ、名称のプロバイダは登録させない */
		for (pr = divy_lib_provider_top; pr->next != NULL; pr = pr->next) {
			/* 多重初期化防止(3) */
			if (strcmp(pr->type, provider->type) == 0 &&
			    strcmp(pr->name, provider->name) == 0) {
				return DIVY_LIB_ST_OK;
			}
		}
		pr->next = provider;
		provider->next = NULL;
	}
#if 0
	LERRLOG2(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG, "@%s,%s\n",
			provider->type, provider->name);
#endif

	return DIVY_LIB_ST_OK;
}

/**
 * static にリンクされるプロバイダライブラリのハンドラ構造体を
 * 取り出して、ローディングする
 *
 */
DIVY_DECLARE(int) divy_register_preloaded_lib_provider()
{
	int ret;
	divy_lib_provider_t **lib_p;

	/* プリロード済みのxxx プロバイダライブラリを全て取り出す */
	for (lib_p = divy_preloaded_lib_providers; *lib_p != NULL; lib_p++) {

		/* プリロード済みのプロバイダライブラリを登録 */
		ret = divy_register_lib_provider(*lib_p);
		if (ret != DIVY_LIB_ST_OK) {
			LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to registe preloaded provider."
				"(type = %s, name = %s)",
				(*lib_p)->type, (*lib_p)->name);
			return DIVY_LIB_ST_ERR;
		}
	}

	return DIVY_LIB_ST_OK;
}

/**
 * システムに登録されている全てのxxx プロバイダライブラリの初期化ハンドラを
 * コールバックする。
 *
 */
DIVY_DECLARE(int) divy_run_lib_init_func(apr_pool_t *pconf)
{
	divy_lib_provider_t *provider;
	int ret;

	if (pconf == NULL) {
		LERRLOG0(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"pconf is NULL.");
		return DIVY_LIB_ST_ERR;
	}

	/* ##### FIXME mutex ロックが必要? */
	for (provider = divy_lib_provider_top;
			provider != NULL; provider = provider->next) {
		/* 多重初期化防止 */
		if (provider->is_init) continue;

		/* 初期化ハンドラのコールバック */
		ret = provider->init_func(pconf);
		if (ret != DIVY_LIB_ST_OK) {
			LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to execute init_func."
				"(type = %s, name = %s)",
				provider->type, provider->name);
			/* 続ける */
		}
		else {
			/* 初期化完了であるとマークする */
			provider->is_init = 1;
		}
	}

	/*
	 * divy_run_lib_fini_func をpconf のdestroyでコールされるよう
	 * pconf に登録する
	 */
	apr_pool_cleanup_register(pconf, NULL, _pconf_cleanup, apr_pool_cleanup_null);

	return DIVY_LIB_ST_OK;
}

/**
 * システムに登録されている全てのxxx プロバイダライブラリの終了ハンドラを
 * コールバックする。
 *
 */
DIVY_DECLARE(int) divy_run_lib_fini_func(void)
{
	int ret;

	/* ##### FIXME mutex ロックが必要か ? */

	/* プロバイダライブラリをリストから外しながらループする */
	for (; divy_lib_provider_top != NULL;
			divy_lib_provider_top = divy_lib_provider_top->next) {

		/* 終了ハンドラをコールバック */
		ret = divy_lib_provider_top->fini_func();
		if (ret != DIVY_LIB_ST_OK) {
			LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to execute fini_func."
				"(type = %s, name = %s)",
				divy_lib_provider_top->type,
				divy_lib_provider_top->name);
			/* 続ける */
		}
		/* 初期化されていないことにする */
		divy_lib_provider_top->is_init = 0;
	}

	return DIVY_LIB_ST_OK;	/* 何があっても成功とする */
}

/**
 * 何もしない初期化ハンドラ
 */
int divy_lib_nop_init_func(apr_pool_t *pconf)
{
	return DIVY_LIB_ST_OK;
}

/**
 * 何もしない終了ハンドラ
 */
int divy_lib_nop_fini_func()
{
	return DIVY_LIB_ST_OK;
}


/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
/**
 * pconf のクリーンアップハンドラ
 */
static apr_status_t _pconf_cleanup(void *data)
{
	/* 終了ハンドラのコールバック */
	(void) divy_run_lib_fini_func();

	return APR_SUCCESS;	/* 成功と返さないとApacheが困ってしまう */
}


