/**
 * $Id$
 *
 * 可変長文字列バッファの宣言・定義を行うファイル
 *
 */
#include "apr.h"
#include "apr_pools.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */

#include "tf_sbuf.h"
#include "util_common.h"
/*------------------------------------------------------------------------------
  Define fixed values and macro
  ----------------------------------------------------------------------------*/
#define DIVY_SBUF_DEFAULT_CAPACITY	256	/* デフォルトバッファサイズ(byte) */
#define DIVY_SBUF_DEFAULT_DELTA		128	/* デフォルトバッファ増加量(byte) */
#define DIVY_SBUF_EXTEND_DELTA_CNT	5	/* delta拡張までのカウント数      */

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/**
 * [ 不完全型の定義 ]
 * 可変長文字列バッファを実現する構造体の定義
 */
struct __divy_sbuf {
	apr_size_t capacity;	/* 割り当てられた文字列バッファ の長さ(byte) */
	apr_size_t length;	/* 格納された文字列の長さ */
	apr_size_t delta;	/* 文字列バッファの拡張サイズ(byte) */
	apr_int32_t extendcnt;	/* 文字列バッファが拡張された回数 */

	/* 追加された文字列を持つバッファ */
	char *buf;
	apr_pool_t *p;		/* 作業用のプール */
};

/*------------------------------------------------------------------------------
  Declare private function
  ----------------------------------------------------------------------------*/
static void _realloc_sbuf(divy_sbuf *sbuf, apr_size_t len);

/*------------------------------------------------------------------------------
  Declare public function
  ----------------------------------------------------------------------------*/
/**
 * 指定されたsize バイトの文字列バッファsbuf を生成して返却する。
 *
 */
DIVY_DECLARE(void) divy_sbuf_create(apr_pool_t *p, divy_sbuf **sbuf,
							apr_size_t size)
{
	*sbuf = apr_palloc(p, sizeof(divy_sbuf));
	(*sbuf)->p = p;

	if (size >= 0) {
		(*sbuf)->capacity = size;
		(*sbuf)->delta    = size / 2;
		if ((*sbuf)->delta < DIVY_SBUF_DEFAULT_CAPACITY) {
			(*sbuf)->delta = DIVY_SBUF_DEFAULT_DELTA;
		}
	}
	else {
		(*sbuf)->capacity = DIVY_SBUF_DEFAULT_CAPACITY;	/* デフォルト値 */
		(*sbuf)->delta    = DIVY_SBUF_DEFAULT_DELTA;
	}
	(*sbuf)->length    = 0;
	(*sbuf)->extendcnt = 0;

	/* 文字列バッファは'\0'の1バイト分だけ余計に確保しておく */
	(*sbuf)->buf = apr_palloc(p, sizeof(char) * ((*sbuf)->capacity + 1));
	*(*sbuf)->buf = '\0';	/* 長さ0のバッファを作成する */
}

/**
 * 指定されたstr が示す文字列の長さを初期値としての文字列バッファsbuf を
 * 初期化して返却する。
 *
 */
DIVY_DECLARE(void) divy_sbuf_createbystr(apr_pool_t *p, divy_sbuf **sbuf,
							const char *str)
{
	apr_size_t size = DIVY_SBUF_DEFAULT_CAPACITY;

	if (str != NULL) {
		size = strlen(str);
	}

	/* size 分のバッファを確保する */
	divy_sbuf_create(p, sbuf, size);

	/* str のコピー */
	if (str != NULL) {
		memcpy((*sbuf)->buf, str, size);
		(*sbuf)->buf[size] = '\0';	/* 終端文字を入れる */
		(*sbuf)->length = size;
	}
	else {
		(*sbuf)->buf[0] = '\0';
		(*sbuf)->length = 0;
	}
}

/**
 * 文字列バッファsbuf に設定された文字列バッファの現在のサイズ(byte)を取得する。
 *
 */
DIVY_DECLARE(apr_size_t) divy_sbuf_getcapacity(divy_sbuf *sbuf)
{
	return (sbuf != NULL) ? sbuf->capacity : 0;
}

/**
 * 文字列バッファsbuf に追加された文字列の総長さ(byte)を取得する。
 *
 */
DIVY_DECLARE(apr_size_t) divy_sbuf_getlength(divy_sbuf *sbuf)
{
	return (sbuf != NULL) ? sbuf->length : 0;
}

/**
 * 文字列バッファsbuf に最後に追加された文字列の後ろににstr が示す文字列を
 * 追加する。
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @param str const char * 追加する文字列
 */
DIVY_DECLARE(void) divy_sbuf_append(divy_sbuf *sbuf, const char *str)
{
	if (str == NULL || sbuf == NULL) return;

	/* 文字列 str をsbuf に追加する */
	divy_sbuf_appendbyte(sbuf, strlen(str), str);
}

/**
 * 文字列バッファsbuf に最後に追加された文字列の後ろににstr が示す文字列を
 * len バイトだけ追加する。
 *
 */
DIVY_DECLARE(void) divy_sbuf_appendbyte(divy_sbuf *sbuf, apr_size_t len,
							const char *str)
{
	if (len == 0 || str == NULL || sbuf == NULL) return;

	/* バッファ領域の拡張が必要なら拡張させる(+1 多く取る) */
	_realloc_sbuf(sbuf, len + 1);

	/* バッファにコピーする */
	memcpy(sbuf->buf + sbuf->length, str, len);
	sbuf->length += len;
	sbuf->buf[sbuf->length] = '\0';
}

/**
 * 文字列バッファsbuf のstart が示すインデックス位置に文字列str を追加する。
 * start位置にあった文字はstrlen(str) 分だけ後ろに移動されます。
 *
 */
DIVY_DECLARE(int) divy_sbuf_insert(divy_sbuf *sbuf, apr_size_t start,
							const char *str)
{
	apr_size_t len;
	char *pos;

	if (sbuf == NULL || start > sbuf->length) {
		return 1;
	}

	if (str == NULL) return 0;	/* 何もしなくていいだろう */
	len = strlen(str);

	/* バッファ領域の拡張が必要なら拡張させる */
	_realloc_sbuf(sbuf, len);

	pos = sbuf->buf + start;	/* 追加位置 */
	memmove(pos + len, pos, (sbuf->length - start));
	memcpy(pos, str, len);

	sbuf->length += len;
	sbuf->buf[sbuf->length] = '\0';

	return 0;
}

/**
 * 文字列バッファsbuf のstart が示すインデックス位置からインデックス位置
 * (end - 1) までをstr が示す文字列で置換する
 * str が 置換長さよりも長い場合には切り取って挿入します。
 * また、str が 置換長さよりも短い場合には、strlen(str) までをコピーします。
 * end がバッファ長さよりも大きい場合にはバッファを自動拡張します。
 *
 */
DIVY_DECLARE(int) divy_sbuf_replace(divy_sbuf *sbuf, apr_size_t start,
					apr_size_t end, const char *str)
{
	apr_size_t len, cpy_len, realoc_len = 0;

	if (sbuf == NULL || start > sbuf->length) {
		return 1;
	}

	cpy_len = end - start;
	if (str == NULL || cpy_len <= 0) return 0;	/* 何もしなくていいだろう */
	len = strlen(str);
	if (cpy_len > len) {
		cpy_len = len;	/* 最大コピーサイズをlen にする */
	}

	/* バッファ領域の拡張が必要なら拡張させる */
	if (cpy_len > sbuf->length - start) {
		realoc_len = cpy_len - (sbuf->length - start);
		_realloc_sbuf(sbuf, realoc_len);
	}

	/* start 位置から文字列をコピーする */
	memcpy(sbuf->buf + start, str, cpy_len);
	if (realoc_len > 0) {
		sbuf->length += realoc_len;     /* 増加分だけ加算する */
	}
	sbuf->buf[sbuf->length] = '\0';

	return 0;
}

/**
 * 文字列バッファsbuf から格納されている全ての文字列を取得する。
 *
 */
DIVY_DECLARE(char *) divy_sbuf_tostring(divy_sbuf *sbuf)
{
	return (sbuf != NULL) ? sbuf->buf : NULL;
}

/**
 * 文字列バッファsbuf に格納された文字列を全て消去する。
 * ただし、確保されたバッファ領域はクリアされません。
 *
 */
DIVY_DECLARE(void) divy_sbuf_clear(divy_sbuf *sbuf)
{
	if (sbuf == NULL) return;

	sbuf->length = 0;	/* 0バイトしか記録されていないものとする */
	sbuf->buf[0] = '\0';
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * 文字列バッファsbuf にlen バイト分の文字列領域を確保できるかどうか検証し、
 * 出来なければ割り当てられるようsbuf を拡張する。
 * (note) バッファ拡張のルール
 * 	(1) 現バッファの総空き容量をlen が上回っていれば len + deltaだけ拡張する
 * 	(2) バッファの拡張がDIVY_SBUF_EXTEND_DELTA_CNT 回以上発生していたら
 * 	    delta を現在の値の2倍に増やして、バッファの拡張を行う
 *
 * @param sbuf divy_sbuf * 文字列バッファへのポインタ
 * @param len apr_size_t 割り当てたいバイト長
 */
static void _realloc_sbuf(divy_sbuf *sbuf, apr_size_t len)
{
	if (len <= 0) return;	/* sbuf のチェックは済んでいるものとします */

	/* 文字列バッファをオーバしていないか? */
	if (sbuf->length + len > sbuf->capacity) {
		char *newbuf;

		/* 文字列バッファ拡張回数のチェック */
		if (sbuf->extendcnt > 0 &&
			sbuf->extendcnt % DIVY_SBUF_EXTEND_DELTA_CNT == 0) {
			/* 拡張が多いのは恐らくdeltaが小さいため。倍に増やしてみる */
			sbuf->delta = sbuf->delta * 2;
		}

		/* 連続領域を確保するため、新たに確保しなおす */
		sbuf->capacity += len + sbuf->delta;
		newbuf = apr_palloc(sbuf->p, sizeof(char) * sbuf->capacity);
		memcpy(newbuf, sbuf->buf, sbuf->length);	/* 旧バッファをコピー */
		newbuf[sbuf->length] = '\0';
		sbuf->buf = newbuf;	/* バッファの新旧交代 */

		sbuf->extendcnt++;	/* 拡張カウントをインクリメント */
	}
}


