/**
 * $Id$
 *
 * tf_vscan.c
 *
 * ウィルス検索プロバイダインターフェースの宣言、定義を行うファイル。
 *
 * (note)
 * 	このヘッダにはTeamFileモジュールおよびApache本体固有のヘッダをinclude
 * 	してはなりません。(mod_davやhttpd.hなど)
 * 	includeしてよいものは、apr_xxx とTeamFile"モジュール"に依存しない共通
 * 	ヘッダ(tfr.hなど)だけです。
 *
 * 2004/10/04 Mon takehara NEW
 */
#ifdef HAVE_CONFIG_H
#ifndef INCLUDE_UTIL_CONFIG_H
#define INCLUDE_UTIL_CONFIG_H
#include "config.h"
#endif	/* INCLUDE_UTIL_CONFIG_H */
#endif	/* HAVE_CONFIG_H */

/* Apache header files */
#include "apr.h"
#include "apr_pools.h"
#include "apr_hooks.h"
#include "apr_strings.h"
#include "apr_hash.h"

#include "util_common.h"

#define DIVY_VSC_PROVIDER_PRIVATE
#include "tf_vscan.h"

/*------------------------------------------------------------------------------
  Define static values and macro
  ----------------------------------------------------------------------------*/
/**
 * func が利用できるかどうか。
 * (1: 利用可能 / 0: 利用できない)
 */
#define IS_AVAILABLE_FUNCTION(ctx,func) ((ctx) != NULL && (ctx)->func != NULL)

/*------------------------------------------------------------------------------
  Define Hook structure and function
  ----------------------------------------------------------------------------*/
/**
 * Hook構造体の定義と生成
 */
APR_HOOK_STRUCT(
	APR_HOOK_LINK(create_vscdatasource)
	)

/**
 * 指定されたproviderType に一致するウィルス検索プロバイダであった時、
 * ウィルス検索プロバイダ構造体を作成 & 返却する関数
 * のためのHook関数宣言。
 * ウィルス検索プロバイダの初期化に利用します。
 * (note)
 * 	providerType と一致しないウィルス検索プロバイダであった場合、
 * 	VSC_DECLINED を返却し、VscDataSourceへのポインタを返却しないこと。
 * 	逆に一致するプロバイダであった場合、VSC_OK を返却し、VscDataSourceへの
 * 	ポインタを返却すること。
 */
APR_IMPLEMENT_EXTERNAL_HOOK_RUN_FIRST(divy, DIVY, int, create_vscdatasource,
		(apr_pool_t *p, const char *providerType, VscDataSource **vscds),
		(p, providerType, vscds), VSC_DECLINED)

/*------------------------------------------------------------------------------
  Define structure
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare Privete Functions
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define Public Functions
  ----------------------------------------------------------------------------*/
/**
 * 指定されたproviderType のウィルス検索プロバイダデータソース(VscDataSource) を
 * 取得して返却する。
 * 取得に失敗した場合には、NULLを返却し、戻り値でステータスを返却します。
 *
 */
DIVY_DECLARE(int) divy_vsc_lookup_vscdatasource(apr_pool_t *p,
						const char *providerType,
						VscDataSource **vscds)
{
	int ret;

	if (p == NULL || IS_EMPTY(providerType)) {
		return DIVY_VSC_ST_INVALID_PARAMS;
	}

	*vscds = NULL;	/* 初期化 */

	/* 全ウィルス検索プロバイダに問い合わせる */
	ret = divy_run_create_vscdatasource(p, providerType, vscds);
	if (ret == VSC_DECLINED) {
		/* プロバイダが存在しなかった */
		return DIVY_VSC_ST_NOTFOUND_VSCDS;
	}

	return DIVY_VSC_ST_OK;
}

/**
 * getCode ハンドラのラッパー。
 *
 */
DIVY_DECLARE(int) divy_vsc_getCode(void *data, int type)
{
	if (type == DIVY_VSC_TYPE_DATASOURCE) {
		VscDataSource *vscds = (VscDataSource *)data;
		return vscds->__code;
	}
	else {
		VscSession *vscsession = (VscSession *)data;
		return vscsession->__code;
	}
}

/**
 * getNativeCode ハンドラのラッパー。
 *
 */
DIVY_DECLARE(char *) divy_vsc_getNativeCode(void *data, int type)
{
	if (type == DIVY_VSC_TYPE_DATASOURCE) {
		VscDataSource *vscds = (VscDataSource *)data;
		return vscds->__nativecode;
	}
	else {
		VscSession *vscsession = (VscSession *)data;
		return vscsession->__nativecode;
	}
}

/**
 * getMsg ハンドラのラッパー。
 *
 */
DIVY_DECLARE(char *) divy_vsc_getMsg(void *data, int type)
{
	if (type == DIVY_VSC_TYPE_DATASOURCE) {
		VscDataSource *vscds = (VscDataSource *)data;
		return vscds->__msg;
	}
	else {
		VscSession *vscsession = (VscSession *)data;
		return vscsession->__msg;
	}
}

/**
 * canLoadVirusData ハンドラのラッパー (see canLoadVirusData)
 *
 */
DIVY_DECLARE(int) divy_vsc_canLoadVirusData(VscDataSource *vscds)
{
	if (!IS_AVAILABLE_FUNCTION(vscds, canLoadVirusData))
		return DIVY_VSC_ST_UNSUPPORTED_LOAD;	/* サポートしていない */

	return vscds->canLoadVirusData(vscds);
}

/**
 * ウィルスパターンデータベースが前回ロードされた時間と比較し
 * リフレッシュ間隔を超えていれば、パターンデータベースを更新する。
 * loadVirusData ハンドラのラッパー + α (see loadVirusData)
 */
DIVY_DECLARE(int) divy_vsc_loadVirusData(VscDataSource *vscds,
					const VscProperty *vscprop)
{
	apr_time_t now, old = APR_TIME_C(0);
	int ret;

	if (!IS_AVAILABLE_FUNCTION(vscds, loadVirusData))
		return DIVY_VSC_ST_OK;	/* 正常扱いする */

	/* ローディングをサポートしているのか？ */
	if ((ret = divy_vsc_canLoadVirusData(vscds)) != DIVY_VSC_ST_OK) {
		return ret;
	}

	/* 現在時刻を取得 */
	now = apr_time_now();
	if (vscds->loaded_time == APR_TIME_C(0)) {
		old = APR_TIME_C(0);
		vscds->loaded_time = now;	/* 現在時刻をセット */
	}
	/* リフレッシュ間隔を経過 */
	else if (now <= vscds->loaded_time +
			apr_time_from_sec(vscprop->refreshinterval)) {
		old = vscds->loaded_time;
		vscds->loaded_time = now;
	}
	else {
		/* ロードする必要なし。--> ロードできたことにする */
		return DIVY_VSC_ST_OK;
	}

	/*
	 * パターンデータベースの更新
	 */
	ret = vscds->loadVirusData(vscds, vscprop);
	if (ret == DIVY_VSC_ST_OK) {
		/* ロードオペレーションが未サポートの場合 */
		int ret2 = divy_vsc_getCode(vscds, DIVY_VSC_TYPE_DATASOURCE);
		if (ret2 == DIVY_VSC_ST_UNSUPPORTED_LOAD) {
			vscds->loaded_time = APR_TIME_C(0);	/* ロードされていない */
		}
	}
	else if (ret != DIVY_VSC_ST_OK) {
		vscds->loaded_time = old;	/* ロードされたのは昔 */
	}

	return ret;
}

/**
 * getVscSession ハンドラのラッパー (see getVscSession)
 */
DIVY_DECLARE(VscSession *) divy_vsc_getVscSession(VscDataSource *vscds,
				const VscProperty *vscprop, apr_pool_t *p)
{
	if (!IS_AVAILABLE_FUNCTION(vscds, getVscSession))
		return NULL;	/* 取得不能 */

	return vscds->getVscSession(vscds, vscprop, p);
}

/**
 * canScanStream ハンドラのラッパー (see canScanStream)
 */
DIVY_DECLARE(int) divy_vsc_canScanStream(VscDataSource *vscds)
{
	if (!IS_AVAILABLE_FUNCTION(vscds, canScanStream))
		return DIVY_VSC_ST_UNSUPPORTED_STREAM;	/* 未サポート */

	return vscds->canScanStream(vscds);
}

/**
 * canCacheSession ハンドラのラッパー (see canCacheSession)
 */
DIVY_DECLARE(int) divy_vsc_canCacheSession(VscSession *vscsession)
{
	if (!IS_AVAILABLE_FUNCTION(vscsession, canCacheSession))
		return DIVY_VSC_ST_UNSUPPORTED_CACHE;	/* キャッシュできない */

	return vscsession->canCacheSession(vscsession);
}

/**
 * closeSession ハンドラのラッパー (see closeSession)
 */
DIVY_DECLARE(void) divy_vsc_closeSession(VscSession *vscsession)
{
	if (!IS_AVAILABLE_FUNCTION(vscsession, closeSession))
		return;

	return vscsession->closeSession(vscsession);
}

/**
 * scanFile ハンドラのラッパー (see scanFile)
 */
DIVY_DECLARE(int) divy_vsc_scanFile(VscSession *vscsession,
					VscAction action, const char *filepath)
{
	if (!IS_AVAILABLE_FUNCTION(vscsession, scanFile))
		return DIVY_VSC_ST_ERR;	/* エラーとします */

	return vscsession->scanFile(vscsession, action, filepath);
}

/**
 * startStreamScan ハンドラのラッパー (see startStreamScan)
 */
DIVY_DECLARE(int) divy_vsc_startStreamScan(VscSession *vscsession,
					VscAction action, VscStreamFile **sfile)
{
	if (!IS_AVAILABLE_FUNCTION(vscsession, startStreamScan))
		return DIVY_VSC_ST_OK;	/* 正常扱いする */

	return vscsession->startStreamScan(vscsession, action, sfile);
}

/**
 * writeStreamBytes ハンドラのラッパー (see writeStreamBytes)
 */
DIVY_DECLARE(int) divy_vsc_writeStreamBytes(VscSession *vscsession,
					VscStreamFile *sfile,
					const char *bytes, apr_size_t len)
{
	if (!IS_AVAILABLE_FUNCTION(vscsession, writeStreamBytes))
		return DIVY_VSC_ST_OK;	/* 正常扱いする */

	return vscsession->writeStreamBytes(vscsession, sfile, bytes, len);
}

/**
 * endStreamScan ハンドラのラッパー (see endStreamScan)
 */
DIVY_DECLARE(int) divy_vsc_endStreamScan(VscSession *vscsession,
					VscStreamFile *sfile)
{
	if (!IS_AVAILABLE_FUNCTION(vscsession, endStreamScan))
		return DIVY_VSC_ST_ERR;	/* エラーとします */

	return vscsession->endStreamScan(vscsession, sfile);
}

/**
 * abortStreamScan ハンドラのラッパー (see abortStreamScan)
 */
DIVY_DECLARE(void) divy_vsc_abortStreamScan(VscSession *vscsession,
					VscStreamFile *sfile)

{
	if (!IS_AVAILABLE_FUNCTION(vscsession, abortStreamScan))
		return;

	return vscsession->abortStreamScan(vscsession, sfile);
}


