/**
 * $Id$
 *
 * util_common.h
 *
 * ビジネスロジックフリーなユーティリティ関数の宣言・定義
 *
 * 2004/02/08 Sun takehara NEW
 */
#ifdef HAVE_CONFIG_H
#ifndef INCLUDE_UTIL_CONFIG_H
#define INCLUDE_UTIL_CONFIG_H
#include "config.h"
#endif	/* INCLUDE_UTIL_CONFIG_H */
#endif	/* HAVE_CONFIG_H */

#include "apr.h"
#include "apr_hash.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"
#include "apr_xml.h"
#include "apr_lib.h"
#include "apr_portable.h"
#include "apr_md5.h"
#include "apr_base64.h"
#include "apr_uuid.h"
#include "apr_file_io.h"
#include "apr_thread_proc.h"

#if APR_HAVE_STDLIB_H
#include <stdlib.h>
#endif	/* APR_HAVE_STDLIB_H */
#if APR_HAVE_STRING_H
#include <string.h>
#endif	/* APR_HAVE_STRING_H */
#if APR_HAVE_UNISTD_H
#include <unistd.h>
#endif	/* APR_HAVE_UNISTD_H */
#if APR_HAVE_STDARG_H
#include <stdarg.h>
#endif	/* APR_HAVE_STDARG_H */
#if APR_HAVE_ERRNO_H
#include <errno.h>
#endif	/* APR_HAVE_ERRNO_H */
#if HAVE_SYS_PARAM_H
# include <sys/param.h>
#endif	/* HAVE_SYS_PARAM_H */
#if HAVE_SYS_MOUNT_H
# include <sys/mount.h>
#endif	/* HAVE_SYS_MOUNT_H */
#if HAVE_SYS_STATVFS_H
#include <sys/statvfs.h>
#endif	/* HAVE_SYS_STATVFS_H */

#if APR_HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif	/* APR_HAVE_SYS_TYPES_H */
#if APR_HAVE_SYS_SOCKET_H
#include <sys/socket.h>
#endif	/* APR_HAVE_SYS_SOCKET_H */
#if APR_HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif	/* APR_HAVE_SYS_IOCTL_H */
#if APR_HAVE_NETINET_IN_H
#include <netinet/in.h>
#endif	/* APR_HAVE_NETINET_IN_H */
#if HAVE_NET_IF_H
#include <net/if.h>
#endif	/* HAVE_NET_IF_H */
#if HAVE_ARPA_INET_H
#include <arpa/inet.h>
#endif	/* HAVE_ARPA_INET_H */

#include "util_common.h"
#include "tf_sbuf.h"
#include "tf_array.h"

/*--------------------------------------------------------------
  Define fixed values and macro
  --------------------------------------------------------------*/
/* 関数 dav_divy_make_full_uri で使用される指定可能可変長文字列の最大数 */ 
#define DIVY_MAX_VALIST_URI	100

/**
 * UTF-8 文字列の解析に利用するマクロ
 */
#define IS_1B_UTF8(ch)	((ch >> 7) == 0x00)	/* １バイトチェック(0xxxxxxx) */
#define IS_2B_UTF8(ch)	((ch >> 5) == 0x06)	/* ２バイトチェック(110xxxxx) */
#define IS_3B_UTF8(ch)	((ch >> 4) == 0x0e)	/* ３バイトチェック(1110xxxx) */	
#define IS_4B_UTF8(ch)	((ch >> 3) == 0x1e)	/* ４バイトチェック(11110xxx) */

/**
 * divy_byte2displaysize 用
 */
#define DIVY_KB_SIZE	APR_INT64_C(1024)
#define DIVY_MB_SIZE	APR_INT64_C(1048576)

/**
 * divy_file_readline が読み込める最大バイト数
 */
#define DIVY_MAX_READLINE_BYTES		2147483647

/*--------------------------------------------------------------
  Define structures
  --------------------------------------------------------------*/

/*--------------------------------------------------------------
  Define static values
  --------------------------------------------------------------*/
/**
 * 関数 divy_url_encode で使用するURLエンコード対象にならない文字一覧
 */
static const unsigned char _unenc_str[] =
	"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789*-.@_";
/**
 * 関数 divy_url_encode で使用する%xx変換用の配列
 */
static const char _ctable[] = "0123456789abcdef";

/*--------------------------------------------------------------
  Declare private function
  --------------------------------------------------------------*/
static void _execute_child_log(apr_pool_t *p, apr_status_t rv, const char *desc);

/*--------------------------------------------------------------
  Define public function
  --------------------------------------------------------------*/
/**
 * 与えられた文字列の末尾から'/'を除去した文字列を生成して返却する。
 *
 */
DIVY_DECLARE(char *) dav_divy_remove_endslash(apr_pool_t *p, const char *str)
{
	char *ret;
	int len = 0;

	if (str == NULL) return NULL;

	ret = apr_pstrdup(p, str);
	len = strlen(ret);

	/* 末尾のスラッシュは複数個あるかもしれないのでループして探す */
	while (len > 1 && str[len - 1] == '/') {
		/* スラッシュが見つかったら、そこを終端にしてしまう */
		ret[len - 1] = '\0';
		len = strlen(ret);
	}
	return ret;
}

/**
 * 与えられた文字列の末尾に'/'がなければ、'/'を付けて返却する。
 *
 */
DIVY_DECLARE(char *) dav_divy_append_endslash(apr_pool_t *p, const char *str)
{
	char *ret;
	int  len = 0;

	if (str == NULL) return NULL;

	len = strlen(str);
	if (len > 1 && str[len - 1] == '/') {
		ret = apr_pstrdup(p, str);
	}
	else {
		ret = apr_pstrcat(p, str, "/", NULL);
	}

	return ret;
}

/**
 * 与えられたuri からfinal path segmentを切り出して返却する。
 * なお、uri の末尾についた'/'は無視する。
 *
 */
DIVY_DECLARE(char *) dav_divy_extract_finalpath_segment(apr_pool_t *p, const char *uri)
{
	char *tmp = NULL;
	char *ps  = NULL;

	if (uri == NULL) return NULL;
	
	/* uri の末尾からスラッシュを取り除く */
	tmp = dav_divy_remove_endslash(p, uri);

	/* 末尾からスラッシュが見つかるまでを切り出す */
	ps = divy_strrchr(tmp, '/');
	if (ps == NULL || ps == uri) {
		return NULL; /* スラッシュがなかった、スラッシュだけだった */
	}

	return apr_pstrdup(p, ++ps);
}

/**
 * 与えられたuri から先頭のパスセグメント(スラッシュを含まない)を
 * 切り出して返却する。
 * '/' が与えられた場合には空文字("")を返します。
 *
 */
DIVY_DECLARE(char *) dav_divy_extract_firstpath_segment(apr_pool_t *p, const char *uri)
{
	char *ps, *first;
	int find_first = 0;

	if (IS_EMPTY(uri)) return NULL;

	/* (note) 空文字の可能性は排除したので1文字以上あるはず */
	if (uri[0] == '/' && uri[1] == '\0') {
		return "";
	}

	/* uri をコピーする(触ってしまうので) */
	first = ps = apr_pstrdup(p, uri);

	/* 先頭のpath segment を切り出す */
	while (*ps) {
		if (find_first == 0 && *ps != '/') {
			find_first = 1;	/* 先頭要素が見つかった */
			first = ps;	/* 先頭を入れ替える */
		}
		else if (find_first == 1 && *ps == '/') {
			*ps = '\0';	/* ここを終点にしてしまう */
			break;
		}
		else {
			ps++;	/* 次へ */
		}
	}

	return first;
}

/**
 * 指定されたuri_part を先頭からスラッシュ区切りで組み立てて返却する。
 * 既にスラッシュがuri_partの先頭や末尾に指定されていれば、２重に付けないようにします。
 * 終端には、必ずNULLを指定して下さい。
 * また、指定可能な可変長文字列の数は、 DIVY_MAX_VALIST_URI 個です。
 * これを超えたらエラーとなります。
 *
 */
DIVY_DECLARE_NONSTD(char *) dav_divy_make_uri(apr_pool_t *p,
				const char *root, const char *uri_part, ...)
{
	char *cp, *ubuf_str, *d, *s;
	divy_sbuf *ubuf = NULL;
	int cnt = 0;
	va_list args;

	/* バッファの作成 */
	divy_sbuf_create(p, &ubuf, 1024);

	/* root, uri_part をセットする */
	if (IS_FILLED(root)) {
		divy_sbuf_append(ubuf, root);
	}

	if (uri_part != NULL) {		/* (note) IS_FILLED では駄目!! */
		va_start(args, uri_part);

		if (IS_FILLED(uri_part)) {
			/* ２重スラッシュになるが気にしない。後で除去するので */
			divy_sbuf_appendbyte(ubuf, 1, "/");
			divy_sbuf_append(ubuf, uri_part);
		}

		/* NULL が現れるまでループする */
		while ((cp = va_arg(args, char *)) != NULL) {
			if (++cnt > DIVY_MAX_VALIST_URI) {
				LERRLOG0(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"the number of args is over.");
				va_end(args);	/* 領域の開放 */
				return NULL;	/* エラーとする */
			}

			if (IS_FILLED(cp)) {
				/* ２重スラッシュになるが気にしない。後で除去するので */
				divy_sbuf_appendbyte(ubuf, 1, "/");
				divy_sbuf_append(ubuf, cp);
			}
		}
	}

	/* va_start で初期化したデータを解放 */
	va_end(args);

	/* 2重スラッシュを取り除く */
	ubuf_str = divy_sbuf_tostring(ubuf);
	s = d = ubuf_str;
	while (*s) {
		if ((*d++ = *s) == '/') {
			do {
				++s;
			} while (*s == '/');
		}
		else {
			++s;
		}
	}
	*d = '\0';

	/* 組み立てたURIを返却する */
	return ubuf_str;
}

/**
 * 指定されたpath の階層数(path segmentの個数)を数えてその数を返却する。
 *
 */
DIVY_DECLARE(int) divy_count_dirs(const char *path)
{
	int x, n;

	if (IS_EMPTY(path)) return 0;

	for (x = 0, n = 0; path[x]; x++) {
		if (path[x] == '/')
			n++;
	}

	return n;
}

/**
 * 指定されたprefix が示す文字列をstr の先頭から取り除き
 * 残りの文字列部分を返却する.
 * (例)
 * 	str = "/divy/abcd/aaaaabb", prefix = "/divy"
 * 	--> "/abcd/aaaaabb" が返却される
 */
DIVY_DECLARE(char *) dav_divy_truncate_prefix(apr_pool_t *p,
					const char *str, const char *prefix)
{
	apr_size_t p_len;

	if (IS_EMPTY(str) || IS_EMPTY(prefix)) {
		return (char *) str;
	}
	p_len = strlen(prefix);

	/* 先頭の文字列が一致しなかった */
	if (p_len > strlen(str) || strncmp(str, prefix, p_len) != 0) {
		return (char *) str;
	}

	/* p_len 以降の文字列をコピーする */
	return apr_pstrdup(p, &str[p_len]);
}

/*
 * 指定されたsrc 文字列内で、最後にdst文字列が現れる位置までの文字列を切り出して
 * 返却する。
 * (例) src = "/abcd/efg/0123/iii/0123/a" dst = "0123" 
 *      結果=  /abcd/efg/0123/iii/
 */
DIVY_DECLARE(char *) dav_divy_find_prestr(apr_pool_t *p, const char *src, const char *dst)
{
	char *cpy_src;
	char *prev_sp = NULL, *sp = NULL;

	if (IS_EMPTY(src) || dst == NULL) return NULL;

	cpy_src = apr_pstrdup(p, src);
	sp      = cpy_src;
	while (sp) {
		sp = divy_strstr_c(sp, dst);	/* dst の文字を探す */
		if (sp == NULL) break;	/* 見つからなくなったらやめる */
		prev_sp = sp;
		sp++;	/* 次のポインタ以降を検索対象にする */
	}

	/* 見つからなかった */
	if (prev_sp == NULL) return NULL;

	/* 終端を\0 で終了させる */
	cpy_src[prev_sp - cpy_src] = '\0';

	return cpy_src;
}

/**
 * src に含まれる from の文字列を to の文字列に置き換えて返却する。
 * surround にNULL または空文字以外が指定された場合、その文字列によって
 * 囲まれた文字列のみを置換対象とします。
 * なお、surround の対応関係が取れていないと、置換しません。
 *
 */
DIVY_DECLARE(const char *) dav_divy_replace_str(apr_pool_t *p, const char *src,
			const char *from, const char *to, const char *surround)
{
	const char *result;
	int found_surround = 0, len_surround, found_first = 0;
	const char *sp, *prev_sp, *replaced;
	char *part;

	if (IS_EMPTY(src) || IS_EMPTY(from) || IS_EMPTY(to)) return src;

	len_surround = surround ? strlen(surround) : 0;

	/*
	 * surround が指定されていなかった場合
	 */
	if (IS_EMPTY(surround)) {
		char *first = NULL, *buf = NULL;
		int len_result = 0, find_from_cnt = 0;
		int len      = strlen(src);
		int len_from = strlen(from);
		int len_to   = strlen(to);
		/*
		 * 結果文字列を格納するバッファを作成する
		 */
		/* from が src の中に何回現れるかカウントする */
		sp = src;
		while ((sp = divy_strstr_c(sp, from)) != NULL) {
			find_from_cnt++;
			sp += len_from;	/* from の長さ分進める */
			if (*sp == '\0') break;
		}

		/* 1つも見つからなかったら何もしない */
		if (find_from_cnt == 0) {
			return src;	/* src をそのまま返却 */
		}

		if (len_from == len_to) {
			/* src 文字列をコピーする */
			first = buf = apr_pstrdup(p, src);
			while ((buf = divy_strstr_c(buf, from)) != NULL) {
				memcpy(buf, to, len_to);
				buf += len_from;	/* コピーした分を飛ばす */
			}
		}
		else {

			/* 結果文字列長の計算 */
			len_result = len +  find_from_cnt * (len_to - len_from);

			/* バッファの確保 */
			first = buf = apr_pcalloc(p, sizeof(char)*(len_result + 1));

			/* src に含まれる from を to に置き換えて返却する */
			sp      = src;
			prev_sp = sp;
			while (1) {
				sp = divy_strstr_c(sp, from);
				if (sp)	{
					/* from を to に置き換えて、その値をbuf に追加 */
					memcpy(buf, prev_sp, sp - prev_sp);
					buf += sp - prev_sp;
					memcpy(buf, to, len_to);
					buf += len_to;
				}
				else {
					/* src の最後までをコピーする */
					if (*prev_sp != '\0') {
						memcpy(buf, prev_sp, &src[len] - prev_sp);
					}
					first[len_result] = '\0';
					break;
				}
				sp += len_from;
				prev_sp = sp;
			}
		}
		return first;	/* buf の先頭を返す */
	}

	/*
	 * surround が指定されていた場合
	 * surround の中身を切り出す
	 */
	sp      = src;
	prev_sp = sp;
	result  = "";
	while (sp) {
		/* surround 文字列を探す */
		sp = divy_strstr_c(sp, surround);
		if (sp) {
			if (found_first) {
				/* prev_sp から surround 文字の直前までをコピー */
				part = apr_pstrndup(p, prev_sp, sp - prev_sp);

				/* from -> to を行う (再帰呼び出し) */
				replaced = dav_divy_replace_str(p, part,
						from, to, NULL);
				result = apr_pstrcat(p, result,
						replaced, surround, NULL); 
				found_first = 0;
			}
			else {
				/* prev_sp から surround 文字終端位置までコピー */
				part = apr_pstrndup(p, prev_sp,
						sp - prev_sp + len_surround);
				result = apr_pstrcat(p, result, part, NULL);	
				found_first = 1;
			}
		}
		else {
			if (!found_surround) {
				/* 1個もsurround が無かった */
				/* from -> to を行う (再帰呼び出し) */
				result = dav_divy_replace_str(p, src,
						from, to, NULL);
			}
			else {
				/* prev_sp から src 最後までをコピー */
				part = apr_pstrdup(p, prev_sp);
				result = apr_pstrcat(p, result, part, NULL);	
			}
			break;
		}
		sp += len_surround;	/* 次の検索開始位置に移動する */
		prev_sp = sp;		/* この位置を記録しておく */
		found_surround++;
	}

	return result;
}

/**
 * 指定された文字列str をXMLのエレメントとして挿入する際に、問題が
 * 起きないようエスケープする。
 * 
 *	DIVY_XML_T2T_CDATA : CDATA セクションをstrの前後に入れます
 * 	DIVY_XML_T2T_QUOTE : '<', '>', '&', '"', ''' を実体参照に置き換える
 */
DIVY_DECLARE(const char *) dav_divy_escape_xmlstr(apr_pool_t *p, char *str, int style)
{
	const char *tmp_str;
	char *x;
	int i, j;

	/* str はNULL or 空文字か? */
	if (IS_EMPTY(str)) return "";

	/* 実体参照への変換 */
	if (style & DIVY_XML_T2T_QUOTE) {
		tmp_str = apr_xml_quote_string(p, str, 1);
		for (i = 0, j = 0; tmp_str[i] != '\0'; i++) {
			if (tmp_str[i] == '\'') 
				j += 5;
		}
		if (j > 0) {
			x = apr_pcalloc(p, i + j + 1);
			for (i = 0, j = 0; tmp_str[i] != '\0'; i++) {
				if (tmp_str[i] == '\'') {
					memcpy(&x[j], "&#39;", 5);
					j += 5;
				}
				else {
					memcpy(&x[j], &tmp_str[i], 1);
					j += 1;
				}
			}
			tmp_str = x;
		}
	}
	else {
		tmp_str = str;
	}

	/* CDATA をつける */
	if (style & DIVY_XML_T2T_CDATA) {
		return apr_psprintf(p, "<![CDATA[%s]]>", tmp_str);
	}
	else {
		return tmp_str;
	}
}

/**
 * 指定された data が数値データのみで構成されているかどうか検証する。
 *
 * @param data const char * チェック対象の文字列
 * @return int 処理結果
 * 		0: 数値ではない値が1つ以上混じっていた
 * 		1: 全て数値だった
 */
DIVY_DECLARE(int) dav_divy_isdigit_str(const char *data)
{
	if (IS_EMPTY(data)) return 0;

	while (*data != '\0') {
		/* char が数値かどうか */
		if (!apr_isdigit(*data)) {
			return 0;	/* 数値ではなかった */
		}
		data++;
	}
	return 1;	/* 数値だった */
}

/**
 * 指定された文字列前後の空白を除去する
 *
 */
DIVY_DECLARE(const char *) dav_divy_trim_white(apr_pool_t *pool, const char *s)
{
	char *str;
	apr_size_t len;

	if (IS_EMPTY(s)) return s;

	str = apr_pstrdup(pool, s);

	/* 
	 * 取得文字列のポインタを空白でなくなる位置まで移動
	 */
 	while (apr_isspace(*str))
		++str;

	len = strlen(str);
	/* 
	 * 取得文字列最後から最初までループし終端または空白でなくなる位置を取得
	 * 取得した位置に終端文字を設定
	 */
	while (len-- > 0 && (str[len] == '\0' || apr_isspace(str[len])))
		continue;
       	
	str[len + 1] = '\0';

	return str;
}

/**
 * 指定された str が示す文字列を先頭からlen が示す"文字数"分だけ切り出して
 * 返却する。
 * 切り出し処理は、str に含まれる文字の文字境界で正しく切り出すよう
 * 実施されます。
 *
 * (note)
 * 	str に指定する文字列は、現在のところ、UTF-8でエンコードされていなければ
 * 	なりません。
 * 	異なるエンコーディングスキームでは正しく切り取ることが出来ません。
 * (note)
 * [ UTF-8文字列切り出しのアルゴリズム ]
 *
 * 	UTF-8によってエンコードされた文字列には、次の性質があります。
 * 	文字列を1バイト境界で分けた時、
 *	・0xxxxxxx  --> １バイトで表現できるUTF-8文字の先頭
 *	・110xxxxx  --> ２バイトで表現できるUTF-8文字の先頭
 *	・1110xxxx  --> ３バイトで表現できるUTF-8文字の先頭
 *	・11110xxx  --> ４バイトで表現できるUTF-8文字の先頭
 *	・10xxxxxx  --> ２から４バイトで表現できるUTF-8文字の途中
 *
 *	(1) str に含まれる文字をUTF-8キャラクタストリームを損なわないよう
 *	    数え上げる
 *	(2) (1) がlenと等しくなったら、先頭からそこまでの文字列を返却する
 */
DIVY_DECLARE(const char *) dav_divy_truncate_str(apr_pool_t *p,
						const char *str, int len)
{
	char *retstr = NULL, *first = NULL;
	int cnt = 0, bytelen = 0;
	apr_size_t slen;
	unsigned char ch;

	if (IS_EMPTY(str) || len <= 0) return str;

	/* Bit検査において、最大4byteだけstrをオーバーランする可能性が
	 * あるので、4byte長くして\0にしておく */
	slen = strlen(str);
	retstr = apr_pcalloc(p, sizeof(char) * (slen + 1 + 4));
	memcpy(retstr, str, slen); 

	first = retstr;
	/* (1) str に含まれる"文字数"を数える */
	while (*retstr != '\0') {
		ch = (unsigned char) (*retstr);

		if (IS_1B_UTF8(ch)) {
			cnt++;
			bytelen += 1;
			retstr  += 1;	/* 1byteスキップ */
		}
		else if (IS_2B_UTF8(ch)) {
			cnt++;
			bytelen += 2;
			retstr  += 2;	/* 2byteスキップ */
		}
		else if (IS_3B_UTF8(ch)) {
			cnt++;
			bytelen += 3;
			retstr  += 3;	/* 3byteスキップ */
		}
		else if (IS_4B_UTF8(ch)) {
			cnt++;
			bytelen += 4;
			retstr  += 4;	/* 4byteスキップ */
		}
		/* 10xxxxxx の場合 */
		else {
			/* ここにくるのはキャラクタシーケンスが
			 * 壊れていたときだけ。これも救います。 */
			bytelen += 1;
			retstr  += 1;
		}

		/* (2) 先頭からlen間でのバイトを切り取って返却する */
		if (len == cnt) {
			first[bytelen] = '\0';
			break;
		}
	}

	return first;
}

/**
 * 指定された str が示す文字列を先頭からbytelen が示すバイト数分だけ切り出して
 * 返却する。なお、文字列終端\0 はbytelenには含まれません。
 * (つまり、strlen(戻り値)とした場合、bytelen と等しくなるということです。)
 *
 * 切り出し処理は、str に含まれる文字の文字境界で正しく切り出すよう
 * 実施されるため、bytelenよりも１から３バイト短くなることがあります。
 *
 * (note)
 * 	str に指定する文字列は、現在のところ、UTF-8でエンコードされていなければ
 * 	なりません。
 * 	異なるエンコーディングスキームでは正しく切り取ることが出来ません。
 *
 * (note)
 * [ UTF-8文字列切り出しのアルゴリズム ]
 * 	UTF-8 のキャラクタシーケンスの性質を利用します。
 *
 *	(1) bytelen でstr を切り取る
 *	(2) 切り取った文字列の\0 の除いた末尾から最大４バイトをスキャンして、
 *	    １つのUTF-8文字シーケンスが切れてしまっていないかどうかを検査します。
 *	    (2)-1 : 切れていなければ、bytelenで切り取っただけの文字列を返却
 *	    (2)-2 : 切れていれば、不完全になったバイトをbytelenで切り出された
 *	            文字列の末尾から除去する。
 */
DIVY_DECLARE(const char *) dav_divy_truncate_str_perbyte(apr_pool_t *p,
						const char *str, int bytelen)
{
	char *retstr = NULL;
	int clen = 4;		/* チェックバイトの長さ */
	int i, skipch = 0, trunclen = 0;
	unsigned char ch;

	if (IS_EMPTY(str) || bytelen <= 0) return str;

	/* strの長さよりもbytelenの方が長いか？ */
	if (bytelen >= strlen(str))	return str;	/* 切り取る必要なし */

	/* (1) bytelen でstrから文字を切り出す */
	retstr = apr_pstrmemdup(p, str, bytelen);

	/* チェックに使用するチェックバイト数を算出する */
	if (bytelen < 4) clen = bytelen;

	/* (2) 末尾のバイトからclen分だけ検査する */
	for (i = 1; i <= clen; i++) {
		ch = (unsigned char) retstr[bytelen - i];
		if (IS_1B_UTF8(ch)) {
			if (skipch) {
				/* 恐らくキャラクタシーケンスが壊れている.
				 * 修正しておくことにする */
				trunclen = i;
			}
			break;
		}
		else if (IS_2B_UTF8(ch)) {
			if (skipch != 1) trunclen = i;
			break;
		}
		else if (IS_3B_UTF8(ch)) {
			if (skipch != 2) trunclen = i; 
			break;
		}
		else if (IS_4B_UTF8(ch)) {
			if (skipch != 3) trunclen = i;
			break;
		}
		/* 10xxxxxx の場合 */
		else {
			++skipch;
		}
	}

	if (trunclen) {
		retstr[bytelen - trunclen] = '\0';
	}

	return retstr;
}

/**
 * 指定された str が示す文字列の文字数を数えて返却する
 *
 * (note)
 * 	str に指定する文字列は、現在のところ、UTF-8でエンコードされていなければ
 * 	なりません。
 * 	異なるエンコーディングスキームでは正しくカウントできません。
 *
 * @param p apr_pool_t *
 * @param str const char * ソースとなる文字列
 * @return int 文字数。
 */
DIVY_DECLARE(int) dav_divy_count_utf8str(apr_pool_t *p, const char *str)
{
	int cnt = 0;
	char *_str;
	unsigned char ch;
	apr_size_t slen;

	if (IS_EMPTY(str)) return 0;

	/* Bit検査において、最大4byteだけstrをオーバーランする可能性が
	 * あるので、4byte長くして\0にしておく */
	slen = strlen(str);
	_str = apr_pcalloc(p, sizeof(char) * (slen + 1 + 4));
	memcpy(_str, str, slen); 

	while (*_str != '\0') {
		ch = (unsigned char) (*_str);

		if (IS_1B_UTF8(ch)) {
			cnt++;
			_str  += 1;	/* 1byteスキップ */
		}
		else if (IS_2B_UTF8(ch)) {
			cnt++;
			_str  += 2;	/* 2byteスキップ */
		}
		else if (IS_3B_UTF8(ch)) {
			cnt++;
			_str  += 3;	/* 3byteスキップ */
		}
		else if (IS_4B_UTF8(ch)) {
			cnt++;
			_str  += 4;	/* 4byteスキップ */
		}
		/* UTF-8 キャラクタシーケンスの途中が現れた場合 */
		else {
			/* ここにくるのはキャラクタシーケンスが
			 * 壊れていたときだけ。取り敢えずポインタを進めて
			 * 様子を見る。カウントアップはしない */
			++_str;
		}
	}

	return cnt;
}

/**
 * 与えられた文字列strを16進数の文字列として返却する。
 * (例) ABCDあい -> 41424344e38182e38184
 *
 */
DIVY_DECLARE(char *) dav_divy_atohex(apr_pool_t *p, const char *str)
{
	if (IS_EMPTY(str)) return "";

	return dav_divy_btohex(p, strlen(str), (unsigned const char *)str);
}

/**
 * 与えられたバイナリデータdat を16進数の文字列として返却する。
 * バイナリデータですので、ヌル終端である必要はありません。
 * 逆にNULL終端であってもlen の範囲内であれば16進数表現に変換して
 * しまいますので、注意して下さい。
 *
 */
DIVY_DECLARE(char *) dav_divy_btohex(apr_pool_t *p, apr_size_t len,
						unsigned const char *dat)
{
	static const char *hex = "0123456789abcdef";
	int i;
	char *hex_str = NULL, *first = NULL;
	unsigned char ch;

	first = hex_str = apr_pcalloc(p, sizeof(char) * (len * 2 + 1));

	for (i = 0; i < len; i++) {
		ch = (unsigned char) dat[i];
		*hex_str++ = hex[ch >> 4];
		*hex_str++ = hex[ch & 0xF];
	}
	*hex_str++ = '\0';

	return first;
}

/**
 * 与えられた16進数の文字列表現を文字列に変換する。
 * 関数dav_divy_atohexの逆変換。
 *
 */
DIVY_DECLARE(char *) dav_divy_hextoa(apr_pool_t *p, const char *str)
{
	char *retstr, *first;
	apr_size_t len;
	unsigned char ch, c;
	int i, even = 0;

	if (IS_EMPTY(str)) return "";

	/* 返却文字列は、strよりも小さいはず。なので、
	 * 同じ長さ分だけ確保しておけば問題ない */
	len = strlen(str);
	first = retstr = apr_pcalloc(p, sizeof(char)*(len + 1));
	for (i = 0; i < len; i++) {
		ch = (unsigned char) str[i];
		if (ch == '0') {
			c = 0x00;
		}
		else if (ch == '1') {
			c = 0x01;
		}
		else if (ch == '2') {
			c = 0x02;
		}
		else if (ch == '3') {
			c = 0x03;
		}
		else if (ch == '4') {
			c = 0x04;
		}
		else if (ch == '5') {
			c = 0x05;
		}
		else if (ch == '6') {
			c = 0x06;
		}
		else if (ch == '7') {
			c = 0x07;
		}
		else if (ch == '8') {
			c = 0x08;
		}
		else if (ch == '9') {
			c = 0x09;
		}
		else if (ch == 'A' || ch == 'a') {
			c = 0x0a;
		}
		else if (ch == 'B' || ch == 'b') {
			c = 0x0b;
		}
		else if (ch == 'C' || ch == 'c') {
			c = 0x0c;
		}
		else if (ch == 'D' || ch == 'd') {
			c = 0x0d;
		}
		else if (ch == 'E' || ch == 'e') {
			c = 0x0e;
		}
		else if (ch == 'F' || ch == 'f') {
			c = 0x0f;
		}
		else {
			/* 不正な文字が指定された */
			return NULL;
		}

		if (even == 0) {
			/* 上位ビットに値を挿入 */
			*retstr = (c << 4) & 0xf0;
			even = 1;
		}
		else {
			/* 下位ビットに値を挿入 */
			*retstr |= c;	
			retstr++;
			even = 0;
		}
	}
	*retstr = '\0';
	
	return first;
}

/**
 * ISO8601形式にフォーマットされた文字列iso8601strをtime_t 型に変換する。
 *
 * (note) ISO8601形式
 * 	YYYY-MM-DD"T"HH:MI:SS"Z"
 *   +++0123456789 0 12345678 9+++
 *
 * 	(例) 2003-03-19T15:40:56Z
 */
DIVY_DECLARE(time_t) dav_divy_iso8601totime_t(apr_pool_t *p, const char *iso8601str)
{
	apr_os_exp_time_t *tms = NULL;
	apr_time_exp_t apr_expt = { 0 };
	apr_time_t apr_time;
	apr_status_t rv;

	char *str;
	apr_size_t len, pos, tzpos;

	if (IS_EMPTY(iso8601str)) {
		return (time_t) 0;
	}

	pos = len = strlen(iso8601str);
	/* フォーマットの確認 */
	if (len < 20 ||
	    iso8601str[4]  != '-' || iso8601str[7]  != '-' ||
	    iso8601str[10] != 'T' || iso8601str[13] != ':' ||
	    iso8601str[16] != ':') {

		return (time_t) 0;
	}

	tzpos = 0;
	if (iso8601str[19] == 'Z') {
		tzpos = 19;
	}
	else if (iso8601str[19] == '.') {

		while(pos != 19) {
			if (iso8601str[pos] == 'Z') {
				tzpos = pos;
				break;
			}
			pos--;
		}

	}

	if (tzpos == 0) {
		return (time_t) 0;
	}

	/* iso8601str を変更してしまうのでコピーしておく */
	str = apr_pstrdup(p, iso8601str);
	tms = apr_pcalloc(p, sizeof(apr_os_exp_time_t));

	/* 区切り文字を全て文字列終端にしてしまう */
	str[4]='\0';
	str[7]='\0';
	str[10]='\0';
	str[13]='\0';
	str[16]='\0';
	str[19]='\0';
	str[tzpos]='\0';

	tms->tm_year = atoi(str) - 1900;
	tms->tm_mon  = atoi(str+=5) - 1;
	tms->tm_mday = atoi(str+=3);
	tms->tm_hour = atoi(str+=3);
	tms->tm_min  = atoi(str+=3);
	tms->tm_sec  = atoi(str+=3);

	/* apr_os_exp_time_t -> apr_time_exp_t */
	apr_os_exp_time_put(&apr_expt, &tms, p);
	/* GMT では以下は必要なし */
	apr_expt.tm_gmtoff = 0;
	/* 戻りがtime_tであるならu_secの値は無意味 */
	apr_expt.tm_usec = 0;

	/* apr_time_exp_t --> apr_time */
	rv = apr_time_exp_get(&apr_time, &apr_expt);
	if (rv != APR_SUCCESS) {
		return (time_t) 0;
	}

	/* apr_time -> time_t */
	return (time_t) apr_time_sec(apr_time);
}

/**
 * 指定された時間(time_t)を指定スタイルに変換して、バッファに詰めて
 * 返却する。なお、time_t は、GMTベースであるとして扱います。
 *
 */
DIVY_DECLARE(int) divy_format_time_t(apr_pool_t *p, const time_t src_time,
					int style, char **datestr)
{
	*datestr = NULL;	/* 初期化 */

	/* ISO8601形式 */
	if (style == DIVY_TIME_STYLE_ISO8601) {
		struct tm *tms;

		/* time_t型 -> tm 構造体 */
		tms = gmtime(&src_time);
		if (tms == NULL) {
			LERRLOG0(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to convert src_time to tm structure.");
			return 1;
		}

		/* 文字列の作成 */
		*datestr = apr_psprintf(p, "%.4d-%.2d-%.2dT%.2d:%.2d:%.2dZ",
				tms->tm_year+1900, tms->tm_mon+1, tms->tm_mday,
				tms->tm_hour, tms->tm_min, tms->tm_sec);
	}
	/* RFC822形式 */
	else if (style == DIVY_TIME_STYLE_RFC822) {
		apr_time_t apr_time;

		
		/* time_t -> apr_time_tに変換 */
		if (apr_time_ansi_put(&apr_time, src_time) != APR_SUCCESS) {
			return 1;
		}

		/* バッファの作成 */
		*datestr = apr_palloc(p, sizeof(char) * APR_RFC822_DATE_LEN);

		/* apr_time_t -> RFC822形式 */
		if(apr_rfc822_date(*datestr, apr_time) != APR_SUCCESS) {
			return 1;
		}
	}
	/* 日本でよく(?) 使われている日付形式 */
	else if (style == DIVY_TIME_STYLE_JAPANESE) {
		apr_time_exp_t exp_t = { 0 };
		apr_size_t timelen   = 0;

		*datestr = apr_pcalloc(p, sizeof(char) * 30);

		/* time_t -> apr_time_exp_t */
		apr_time_exp_lt(&exp_t, apr_time_from_sec(src_time));
		apr_strftime(*datestr, &timelen, 30,
					"%Y/%m/%d %H:%M:%S %Z", &exp_t);
		(*datestr)[timelen] = '\0';
	}
	/* RFC2822 形式 */
	else if (style == DIVY_TIME_STYLE_RFC2822) {
		apr_time_exp_t exp_t = { 0 };
		apr_size_t timelen   = 0;

		*datestr = apr_pcalloc(p, sizeof(char) * (APR_RFC822_DATE_LEN + 10));

		/* time_t -> apr_time_exp_t */
		apr_time_exp_lt(&exp_t, apr_time_from_sec(src_time));
		apr_strftime(*datestr, &timelen, (APR_RFC822_DATE_LEN + 10),
					"%a, %d %b %Y %H:%M:%S %z (%Z)", &exp_t);
		(*datestr)[timelen] = '\0';
	}
	else {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Undefined style param: %d" , style);
		return 1;
	}

	return 0;
}

/**
 * 現在時刻のエポックタイム(time_t)を取得する。
 *
 * @return time_t
 */
DIVY_DECLARE(time_t) dav_divy_get_now_epoch(void)
{
	return (time_t) apr_time_sec(apr_time_now());
}

/**
 * 指定された平文plaintextを暗号化する。
 *
 * [ 暗号化のアルゴリズム ]
 *
 * (1) タイムスタンプと平文からMD5ダイジェストメッセージを作成
 * (2) (1) で作成したMD5と(タイムスタンプ + 平文)の XOR を取る
 * (3) (2) で作成したXOR文字列とMD5を合成する
 * (4) (3) で作成した文字列をBase64でエンコードする
 *
 * MD5 は平文が改変されていないかどうかをチェックするため、XOR は平文を
 * 直感的に推測しにくくするため、Base64エンコードは、バイナリデータを
 * ASCIIに変換するために導入しています。
 * 暗号文から平文が生成できる種類の暗号化です。 
 */
DIVY_DECLARE(const char *) divy_encipher_text(apr_pool_t *p,
							const char *plaintext)
{
	unsigned char hash[APR_MD5_DIGESTSIZE];
	apr_status_t status;
	time_t now;
	char *tmp_str;
	char *tm_ori_str, *xor_str, *xor_md5_str, *base64_str;
	int i, len, xor_md5_str_len, base64_len;

	if (IS_EMPTY(plaintext)) return NULL;

	/* タイムスタンプの作成 */
	time(&now);

	/* timestamp + plaintext */
	tm_ori_str = apr_pstrcat(p, apr_psprintf(p, "%010ld", now), 
					plaintext, NULL);

	len = strlen(tm_ori_str);
	/*
	 * (1) MD5(timestamp + plaintext) -> MD5hash
	 */ 
	status = apr_md5(hash, tm_ori_str, len);
	if (status != APR_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to execute apr_md5. code = %d", status);
		return NULL;
	}

	/*
	 * (2) (timestamp + plaintext) ^ MD5hash -> XORstr
	 */ 
	xor_str = apr_pcalloc(p, len);
	for (i = 0; i < len; i++) {
		xor_str[i] = (unsigned char) tm_ori_str[i] ^ hash[i % APR_MD5_DIGESTSIZE];
	}

	/* 
	 * (3) XORstr + MD5hash
	 */
	xor_md5_str_len = len + APR_MD5_DIGESTSIZE + 1;
	xor_md5_str = apr_pcalloc(p, sizeof(char) * xor_md5_str_len);

	memcpy(xor_md5_str, xor_str, len);
	tmp_str = &xor_md5_str[len];
	memcpy(tmp_str, hash, APR_MD5_DIGESTSIZE);
	xor_md5_str[xor_md5_str_len - 1] = '\0';

	/*
	 * (4) Base64.encode(XORstr + MD5hash) -> Base64 String
	 */ 
	base64_str = apr_pcalloc(p, 
			sizeof(char) * (apr_base64_encode_len(xor_md5_str_len) + 1));
	base64_len = apr_base64_encode(base64_str, xor_md5_str, xor_md5_str_len);
	base64_str[base64_len] = '\0';

	return base64_str;

}

/**
 * 指定された暗号文ciphertextを復号する。
 * なお、ciphertext は、関数dav_divy_encipher_text を使って暗号化されたものでなければ
 * なりません。(そうではないものは復号できません。)
 *
 * [ 復号化のアルゴリズム ]
 *
 * (1) 受け取った暗号文をBase64でデコードする
 * (2) デコードされた文字列を、XORを取った文字列とMD5ダイジェストメッセージの
 *     ２つの部分に分離する
 * (3) XORを取った文字列とMD5ダイジェストメッセージとでXORを取る
 * (4) (3) で得られた文字列の内、先頭から10バイト目以降が復号化された文字列
 *
 * [ 改変のチェック ]
 *
 *  不正書き換えを防止する目的で、以下の２つの確認を行っている
 *  ・サーバライセンスキーの長さが正しいかどうか
 *  ・復号化したサーバライセンスキーとチェックサム値(MD5)が正しいかどうか
 *
 *  [ 暗号の強度 ]
 *
 *  暗号化された文字列は、暗号文作成間隔が1[s] 以上はなれているとき、必ず
 *  異なる値になる。MD5メッセージのランダム性と同程度になるはずであり、
 *  暗号文からは、推測がしにくい。
 *  この暗号化はアルゴリズムが公開されると必ず解けてしまう種類のものです。
 *  暗号化・復号化のアルゴリズムは公開しないようお願いします。
 */
DIVY_DECLARE(const char *) divy_decipher_text(apr_pool_t *p,
							const char *ciphertext)
{
	char *tm_ori_str, *debase64_str, *xor_str, *md5_str;
	int i, len, xor_str_len;
	apr_status_t status;
	unsigned char hash[APR_MD5_DIGESTSIZE + 1];

	if (IS_EMPTY(ciphertext)) return NULL;

	/*
	 * (1) Base64.decode(ciphertext) -> XORstr + MD5hash
	 */ 
	debase64_str = apr_pcalloc(p, 
			sizeof(char) * (apr_base64_decode_len(ciphertext) + 1));
	len = apr_base64_decode(debase64_str, ciphertext);
	debase64_str[len] = '\0';

	/* 不正改変されていないか？ (チェック1) */
	if (len < APR_MD5_DIGESTSIZE) {
		LERRLOG0(LLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"We found invalid key format. Probably key "
			"is broken. Please check the key string");
		return NULL;
	}

	/*
	 * (2) XORstr + MD5hash -> XORstr , MD5hash
	 */
	xor_str_len = len - (APR_MD5_DIGESTSIZE + 1);
	xor_str = apr_pstrmemdup(p, debase64_str, xor_str_len);
	md5_str = &debase64_str[xor_str_len];

	/* 
	 * (3) XORstr ^ MD5hash -> timestamp + plaintext
	 */
	tm_ori_str = apr_pcalloc(p, sizeof(char) * (xor_str_len + 1));
	for (i = 0; i < xor_str_len; i++) {
		tm_ori_str[i] = (unsigned char) xor_str[i] 
				^ (unsigned char) md5_str[i % APR_MD5_DIGESTSIZE];
	}
	tm_ori_str[xor_str_len] = '\0';

	/* 不正改変されていないか？(チェック2) */
	status = apr_md5(hash, tm_ori_str, xor_str_len);
	if (status != APR_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to execute apr_md5.(code = %d)", status);
		return NULL;
	}

	hash[APR_MD5_DIGESTSIZE] = '\0';

	if (memcmp(md5_str, hash, APR_MD5_DIGESTSIZE) != 0) {
		/* キーが壊れていたか、改変されていたとき */
		LERRLOG0(LLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"We found invalid key format.(check sum error)"
			"Probably key is broken.");
		return NULL;
	}

	/*
	 * (4) timestamp + plaintext -> plaintext
	 */
	return apr_pstrdup(p, &tm_ori_str[10]);	/* タイムスタンプを切捨てる */
}

/**
 * 与えられたsizeが表すバイト数を操作者が見やすい形式(KB, MB表示) にする。
 *
 */

DIVY_DECLARE(const char *) divy_byte2displaysize(apr_pool_t *p,
                                                 apr_int64_t size, int roundbyte)
{
	const char *ret;

	/* KB 以下の場合 */
	if (size < DIVY_KB_SIZE) {
		if (roundbyte) {
			if (size == APR_INT64_C(0)) {
				ret = apr_pstrdup(p, "0 KB");
			}
			else {
				ret = apr_pstrdup(p, "1 KB");
			}
		}
		else {
			ret = apr_psprintf(p, "%"APR_INT64_T_FMT, size);
		}
	}
	/* KB 以上 MB 未満の場合 */
	else if (size < DIVY_MB_SIZE) {
		ret = apr_psprintf(p, "%"APR_INT64_T_FMT" KB", size / DIVY_KB_SIZE);
	}
	/* MB 以上の場合 */
	else {
		char *kbsize, *retstr;
		apr_size_t len, retlen;
		int i;

		kbsize = apr_psprintf(p, "%"APR_INT64_T_FMT, size / DIVY_KB_SIZE);
		len = strlen(kbsize);
		retlen = len + ((len % 3) ? (len/3) : (len/3 -1)); /* len + ","の数 */
		ret = retstr =
			apr_palloc(p, sizeof(char) * (retlen + 3 + 1)); /* " KB" + \0 */

		for (i = 0; i < len; i++) {
			*retstr++ = kbsize[i];
			if (i != len - 1 && (len - i)%3 == 1) {
				*retstr++ = ',';
			}
		}
		*retstr++ = ' ';
		*retstr++ = 'K';
		*retstr++ = 'B';
		*retstr   = '\0';
	}

	return ret;
}

/**
 * リクエストURIをユニークにするためのシーケンスURIパラメータを取得する。
 *
 */
DIVY_DECLARE(const char *) divy_get_urlseq(apr_pool_t *p)
{
	return apr_psprintf(p, "seq=%"APR_TIME_T_FMT, apr_time_now());
}

/**
 * f が示すファイルから1行(改行コードまで)を読み込んで文字列として返却する。
 * 読み込んだ文字列から改行コードは除去します。
 *
 */
DIVY_DECLARE(int) divy_file_readline(apr_pool_t *p, apr_file_t *f, char **linebuf)
{
	apr_status_t rv;
	divy_sbuf *sbuf = NULL;
	char cp;
	apr_size_t rbytes = 0;
	int ret = DIVY_UTIL_COMMON_OK;
	int found_cr = 0;	/* CF を読んだかどうか */

	if (f == NULL) return DIVY_UTIL_COMMON_ERR;	/* 何も出来ない */

	/* 初期化 */
	*linebuf = NULL;

	while (rbytes <= DIVY_MAX_READLINE_BYTES) {
		/* 常に1バイトずつ読む(戻り読み防止) */
		if ((rv = apr_file_getc(&cp, f)) != APR_SUCCESS) {
			if (rv == APR_EOF){	/* ファイル終了 */
				ret = DIVY_UTIL_COMMON_EOF;	/* EOF発生 */
				break;
			}
			ret = DIVY_UTIL_COMMON_ERR;	/* エラー発生 */
			break;
		}

		/* 直前の文字がCR で今がLF ならばここまで読んで終わり */
		if (found_cr && cp == '\n') {
			break;
		}
		/* 直前の文字がCR だったが今回はLF では無かった */
		else if (found_cr && cp != '\n') {
			/* ストリームに書き戻して終了 */
			(void) apr_file_ungetc(cp, f);
			break;
		}
		/* LF を検出した */
		else if (cp == '\n') {
			break;	/* 終了 */
		}
		/* CR を検出した */
		else if (cp == '\r') {
			found_cr = 1;	/* CR 検出 */
		}
		else {
			if (sbuf == NULL) {
				/* (note) lazy initialize は必須 */
				divy_sbuf_create(p, &sbuf, 2048);
			}
			divy_sbuf_appendbyte(sbuf, 1, &cp);
		}
		rbytes += 1;	/* 読み込みバイト数を記録 */
	}

	/* 固定長文字列に変換 */
	if (sbuf != NULL) {
		*linebuf = divy_sbuf_tostring(sbuf);
	}

	return ret;
}

/**
 * path のファイルシステム情報を取得して返却する。
 *
 */
DIVY_DECLARE(divy_error *) divy_statfs(apr_pool_t *p,
					const char *path, divy_statfs_t **sfs)
{
	int ret;
#ifdef HAVE_SYS_STATVFS_H
	struct statvfs _sfs = { 0 };
#else
	struct statfs _sfs  = { 0 };
#endif	/* HAVE_SYS_STATVFS_H */

	if (IS_EMPTY(path)) {
		return divy_new_error(p, LLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					DESC0(p,"path is EMPTY."));
	}

	*sfs = NULL;	/* 初期化 */

	/* ファイルシステム情報の取得 */
#ifdef HAVE_SYS_STATVFS_H
	ret = statvfs(path, &_sfs);
#else
	ret = statfs(path, &_sfs);
#endif	/* HAVE_SYS_STATVFS_H */
	if (ret == -1) {
		return divy_new_error(p, LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			DESC1(p,"Failed to get filesystem information. (code = %d)", errno));
	}

	/* 取得した情報を構造体に焼き直す */
	*sfs = apr_pcalloc(p, sizeof(divy_statfs_t));

	(*sfs)->path        = apr_pstrdup(p, path);
#ifdef HAVE_SYS_STATVFS_H
	(*sfs)->bsize       = _sfs.f_frsize;
	(*sfs)->total_bytes = (apr_uint64_t) _sfs.f_blocks * _sfs.f_frsize;
	(*sfs)->avail_bytes = (apr_uint64_t) _sfs.f_bavail * _sfs.f_frsize;
	(*sfs)->total_files = (apr_uint64_t) _sfs.f_files;
	(*sfs)->avail_files = (apr_uint64_t) _sfs.f_ffree;
#else
	(*sfs)->bsize       = _sfs.f_bsize;
	(*sfs)->total_bytes = (apr_uint64_t) _sfs.f_blocks * _sfs.f_bsize;
	(*sfs)->avail_bytes = (apr_uint64_t) _sfs.f_bavail * _sfs.f_bsize;
	(*sfs)->total_files = (apr_uint64_t) _sfs.f_files;
	(*sfs)->avail_files = (apr_uint64_t) _sfs.f_ffree;
#endif	/* HAVE_SYS_STATVFS_H */

	return NULL;
}

/**
 * 指定されたcommand が示すコマンドを引数argv を付けて実行する。
 * なお、comannd がstdout, stderr に出力した結果は無視されます。
 */
DIVY_DECLARE(int) divy_execute_cmd(apr_pool_t *p,
				const char *command, const char * const argv[],
				const char * const *env, apr_cmdtype_e cmdtype,
				const char *curdir, int *exitcode, int *exceptcd_set)
{
	apr_status_t rv;
	apr_exit_why_e exitwhy;

	apr_procattr_t *procattr = NULL;
	apr_proc_t proc          = { 0 };
	apr_pool_t *child_p      = NULL;
	int i, ret = 0;

	/* 初期化 */
	*exitcode = 0;

	/* 作業用のサブプールを作成する */
	if ((rv = apr_pool_create(&child_p, p)) != APR_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to create sub-pool for child process."
			"(code = %d)", rv);
		ret = 1;
		goto cleanup_child_env;
	}

	/* process の環境を作成 */
	if ((rv = apr_procattr_create(&procattr, child_p)) != APR_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to create process attribute. (code = %d)", rv);
		ret = 1;
		goto cleanup_child_env;
	}

	/* 作成するChildのパイプに親プロセスのstdin, stdout, stderr をどう繋ぐか */
	if ((rv = apr_procattr_io_set(procattr, APR_PARENT_BLOCK, APR_PARENT_BLOCK,
						APR_PARENT_BLOCK)) != APR_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to set child io. (code = %d)", rv);
		ret = 1;
		goto cleanup_child_env;
	}

	/* ディレクトリ設定 */
	if (IS_FILLED(curdir)) {
		if ((rv = apr_procattr_dir_set(procattr, curdir)) != APR_SUCCESS) {
			LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to change directory. "
				"(code = %d, dir = %s)", rv, curdir);
			ret = 1;
			goto cleanup_child_env;
		}
	}

	/* コマンドタイプ設定 */
	if ((rv = apr_procattr_cmdtype_set(procattr, cmdtype)) != APR_SUCCESS) {
		LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to set cmdtype.(code = %d, cmdtype = %d)",
			rv, cmdtype);
		ret = 1;
		goto cleanup_child_env;
	}

	/* 子プロセスのエラーを出力するように設定 */
	if ((rv = apr_procattr_child_errfn_set(procattr,
					_execute_child_log)) != APR_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to set error function.(code = %d)", rv);
		ret = 1;
		goto cleanup_child_env;
	}

	/* process 実行 */
	if ((rv = apr_proc_create(&proc, command, argv,
					env, procattr, child_p)) != APR_SUCCESS) {
		LERRLOG1(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to execute command.(code = %d)", rv);
		ret = 1;
		goto cleanup_child_env;
	}

	/* process の終了待ち */
	rv = apr_proc_wait(&proc, exitcode, &exitwhy, APR_WAIT);
	if (rv != APR_CHILD_DONE){
		LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to wait for child process. "
			"(code = %d, exitwhy = %d)", rv, exitwhy);
		ret = 1;
		goto cleanup_child_env;
	} 
	else if (*exitcode) {
		divy_sbuf *sbuf = NULL;
		/* このコードが exceptcd_set に含まれていなければエラーログを出す */
		if (exceptcd_set != NULL) {
			int *p_code;
			for (p_code = exceptcd_set; *p_code != 0; p_code++) {
				if (*exitcode == *p_code) {
					/* 除外リストに含まれていたエラーだった */
					ret = DIVY_UTIL_COMMON_EXCEPTCD_SET;
					goto cleanup_child_env;
				}
			}
		}
		divy_sbuf_create(child_p, &sbuf, 128);

		for (i = 1; argv[i] != NULL; i++){
			divy_sbuf_append(sbuf, argv[i]);
			if (argv[i+1] != NULL) {
				divy_sbuf_appendbyte(sbuf, 1, " ");
			}
		}

		LERRLOG3(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to execute child process. "
			"(code = %d, cmd = %s, argv = %s)",
			*exitcode, argv[0], divy_sbuf_tostring(sbuf));
		ret = 1;
		goto cleanup_child_env;
	}

cleanup_child_env:
	/* Childプロセス専用のプールを破棄する */
	if (child_p != NULL) {
		apr_pool_destroy(child_p);
		child_p = NULL;
	} 

	return ret;
}

/**
 * コマンドライン文字列 cmdline をパースしてchar * の配列を生成して返却する。
 * 終端はNULLです。
 *
 */
DIVY_DECLARE(char **) divy_parse_commandline(apr_pool_t *p, const char *cmdline)
{
	char *token, *token_cntx, *cmd;
	divy_array_t *array = NULL;

	if (IS_EMPTY(cmdline)) return NULL;

	/* 可変長配列の作成 */
	array = divy_array_make(p, 10);

	/* cmdline をコピーして使う(必須) */
	cmd = apr_pstrdup(p, cmdline);

	/* cmdline を半角スペースでトークンナイズする */
	while ((token = apr_strtok(cmd, " ", &token_cntx)) != NULL) {
		if (token == NULL) break;
		cmd = NULL;

		/* 配列に入れる */
		divy_array_add(array, token);
	}
	divy_array_add(array, NULL);	/* sentinel */

	/* 可変長配列から固定長配列へ変換 */

	return (char **) divy_array_getelements(array);
}

/**
 * n を３桁毎に","で区切った文字列に変換する。
 *
 */
DIVY_DECLARE(const char *) divy_i2displaynum(apr_pool_t *p, apr_int64_t n)
{
	char *ret, *str, *first;
	apr_size_t len, retlen, i;

	str = apr_psprintf(p, "%"APR_INT64_T_FMT, n);

	/* 1000未満(3桁よりも小さいとき) */
	if (n < APR_INT64_C(1000)) {
		return str;
	}

	len = strlen(str);
	retlen = len + ((len % 3) ? (len/3) : (len/3 -1)); /* len + ","の数 */
	first = ret = apr_pcalloc(p, sizeof(char) * (retlen + 1));
	for (i = 0; i < len; i++) {
		*ret++ = str[i];
		if (i != len - 1 && (len - i)%3 == 1) {
			*ret++ = ',';
		}
	}
	*ret = '\0';

	return first;
}

/**
 * 指定されたネットワークインターフェース名ifname に割り当てられた
 * IPV4のIPアドレスを取得して返却する。
 * 対象のネットワークインターフェースにIPアドレスが未設定であれば、
 * "0.0.0.0" が返却されます。
 *
 */
DIVY_DECLARE(char *) divy_get_netifaddr(apr_pool_t *p, const char *ifname)
{
	char *ip_str = NULL;
	int fd, ret;
	struct ifreq ifr;

	if (IS_EMPTY(ifname)) return apr_pstrdup(p, "0.0.0.0");

	/* データグラムソケットを開いてみてその時のIPアドレスを取得する */
	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) == -1) {
		return apr_pstrdup(p, "0.0.0.0");
	}

	ifr.ifr_addr.sa_family = AF_INET;
	strncpy(ifr.ifr_name, ifname, IFNAMSIZ-1);

	if ((ret = ioctl(fd, SIOCGIFADDR, &ifr)) != 0) {
		return apr_pstrdup(p, "0.0.0.0");
	}

	ip_str = apr_pstrdup(p, inet_ntoa(((struct sockaddr_in *)&ifr.ifr_addr)->sin_addr));

	close(fd);

	return ip_str;
}

/**
 * 文字列str の中に含まれる大文字に出来る文字列を大文字に変換する.
 *
 */
DIVY_DECLARE(char *) divy_toupper_str(apr_pool_t *p, const char *str)
{
	char *s, *first;
	const char *c;

	if (IS_EMPTY(str)) return NULL;

	first = s = apr_pcalloc(p, strlen(str) + 1);
	for (c = str; c != NULL && *c != '\0'; c++) {
		*s = apr_toupper(*c);
		s++;
	}
	*s = '\0';
	return first;
}

/**
 * 文字列str の中に含まれる小文字に出来る文字列を小文字に変換する.
 *
 */
DIVY_DECLARE(char *) divy_tolower_str(apr_pool_t *p, const char *str)
{
	char *s, *first;
	const char *c;

	if (IS_EMPTY(str)) return NULL;

	first = s = apr_pcalloc(p, strlen(str) + 1);
	for (c = str; c != NULL && *c != '\0'; c++) {
		*s = apr_tolower(*c);
		s++;
	}
	*s = '\0';
	return first;
}

/**
 * 文字列str をURLエンコードする.
 *
 */
DIVY_DECLARE(char *) divy_url_encode(apr_pool_t *p, const char *str)
{
	unsigned char *ret, *first;
	unsigned char *s = (unsigned char *)str;
	unsigned char c;
	int i, found, len = sizeof(_unenc_str);

	if (IS_EMPTY(str)) {
		return NULL;
	}
	/* 全てURLエンコードされたとすると 3 * strlen(str) + 1 になる */
	first = ret = apr_palloc(p, 3 * strlen(str) + 1);
	while ((c = *s)) {
		found = 0;
		for (i = 0; i < len; i++) {
			if (c == _unenc_str[i]) {
				found = 1;
				*ret++ = c;
				break;
			}
		}

		if (!found) {
			*ret++ = '%';
			*ret++ = _ctable[c >> 4];
			*ret++ = _ctable[c & 0xf];
		}
		++s;
	}
	*ret = '\0';

	return (char *)first;
}

/**
 * ランダム文字列を作成する
 * @param r request_rec
 * @return セッションID char *
 */
DIVY_DECLARE(char*) make_random_string(apr_pool_t *pool)
{
	apr_uuid_t uuid;
	char buff[APR_UUID_FORMATTED_LENGTH + 1];
	unsigned char md5[APR_MD5_DIGESTSIZE+1] = {0};
	const char hex[16] = "0123456789abcdef";
	unsigned char ch;
	char *digest, *first;
	int i = 0;
	apr_status_t st;

	apr_uuid_get(&uuid);
	apr_uuid_format(buff, &uuid);

	st = apr_md5(md5, buff, strlen(buff));
	if (st != APR_SUCCESS) return NULL; 

	digest = first = apr_pcalloc(pool, APR_MD5_DIGESTSIZE * 2 + 1);

    for(i=0; i < APR_MD5_DIGESTSIZE; i++) {
        ch = (unsigned char)md5[i];
        *digest++ = hex[ch >> 4];
        *digest++ = hex[ch & 0xF];
    }
    *digest++ = '\0';

	return first;
}


/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * divy_execute_cmd が使用するエラーログロガー
 * apr_proc_create 関数がコールバックで呼び出します。
 */
static void _execute_child_log(apr_pool_t *p, apr_status_t rv, const char *desc)
{
	LERRLOG2(LLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
		"Child process return code. (code = %d, desc = %s)", 
		rv, desc);
	return;
}

