/**
 * $Id$
 *
 * util_auth.c
 *
 * Authユーティリティ関数
 *
 *
 */

#include "httpd.h"
#include "http_core.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"
#include "apr_portable.h"
#include "apr_tables.h"
#include "apr_uuid.h"
#include "apr_errno.h"

#include "apreq2/apreq.h"
#include "apreq2/apreq_cookie.h"

#include "util_auth.h"
#include "util_md5.h"
#include "util_ml.h"
#include "util.h"

#include "tf_rdbo_user.h"
#include "tf_folder.h"

/*------------------------------------------------------------------------------
  Define Fixed values and macro
  ----------------------------------------------------------------------------*/

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define static values
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define structures
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Define Global values
  ----------------------------------------------------------------------------*/

/*------------------------------------------------------------------------------
  Declare private prototype function
  ----------------------------------------------------------------------------*/
static const char* divy_util_auth_get_cookie_value(request_rec* r, const char* name);

static divy_rdbo_session* divy_util_auth_get_memcache_session(request_rec *r, const char* key, int withupdatetime);

static time_t divy_util_auth_get_2faperiod_epoch(request_rec *r);
/*------------------------------------------------------------------------------
  Define public function
  ----------------------------------------------------------------------------*/

/**
 * Cookieを取得してログインを正常に行っているか調べる
 *
 * @param r request_rec*
 * @param password char**
 * @param sessionid char**
 * @return 1: 正常 / 0:失敗
 */
DIVY_DECLARE(int)
divy_util_auth_parse_cookie(request_rec* r, char** password, char** sessionid)
{
	const char* header;
	char *sess_cookie = NULL;
	apr_table_t* jar;
	divy_auth_session session = {0};
	apreq_cookie_t *cookie;

	TRACE(r->pool);

	/* Cookieヘッダを調べる */
	header = dav_divy_get_cookie(r);
	if (IS_EMPTY(header)) {
		/* Cookieヘッダがない場合 */
		/* 何もLOGを出さない */
		return DIVY_AUTH_SESSION_ERROR;
	}
	else {
		/* Cookie パース */
		jar = apr_table_make(r->pool, APREQ_DEFAULT_NELTS);
		if (apreq_parse_cookie_header(r->pool, jar, header) != APR_SUCCESS) {
			/* パース失敗 */
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
					"Failed to parse cookie. cookie value = %s", header);
			return DIVY_AUTH_SESSION_ERROR;
		}

		/* クッキーの内容を取得する */
		sess_cookie = (char *)apr_table_get(jar, TF_COOKIE_SES_NAME);

	}

	/* 既に(tf-session)ブラウザセッションがあるか調べる */
	if (IS_FILLED(sess_cookie)) {
		/* すでにあるはずだから第二引数のuidはsessionテーブルかmemcacheより
 		 * 取得する
 		 */
		session.sid = apr_pstrdup(r->pool, sess_cookie);
		if (divy_util_auth_get_memcache_userinfo(r, &session)) {
			/*
			 * ブラウザセッションが存在していたがレコードにない場合
			 * いったんすべてのCookieを削除するためにエラーとする
			 * セッションを送り付けられてきたのにサーバー側で見つからなかった
			 * 場合はタイムアウトされたと思ってもよい
			 */
			ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_INFO + DIVY_SST_DATA,
					"session data not found (sid=%s)", session.sid); 
			divy_set_delete_session_cookie(r);
			return DIVY_AUTH_SESSION_TIMEOUT;
		}
		else {
			/* 成功 */
			/* この時点ではまだ正しいかわからないので設定するのは早い */
			//divy_pcache_set_data(r->pool, sess_cookie, DIVY_PCACHE_DAT_SES_SID);

			/* cookie を作成 */
			cookie = apreq_cookie_make(r->pool, TF_COOKIE_SES_NAME,
				strlen(TF_COOKIE_SES_NAME), sess_cookie, strlen(sess_cookie));
			cookie->path = (char*)dav_divy_get_root_uri(r);
			apr_table_setn(r->err_headers_out, "Set-Cookie",
									apreq_cookie_as_string(cookie, r->pool));
			r->user   = apr_pstrdup(r->pool, session.uid);
			*password = apr_pstrdup(r->pool, session.password);
			*sessionid = apr_pstrdup(r->pool, session.sid);

			return DIVY_AUTH_SESSION_OK;

		}
	}

	return DIVY_AUTH_SESSION_ERROR; /* 失敗 */

}

/**
 * 認証を行わない(スルーする)URIかを調査する
 * @param r request_rec*
 * @return int 1: スルー（認証しないでいい) / 0: 認証が必要
 */
DIVY_DECLARE(int) dav_divy_auth_through_uri(request_rec *r)
{
	divy_uri_spec *u_spec = NULL;

	(void) divy_parse_uri(r->pool, dav_divy_get_root_uri(r), r->uri, &u_spec);

	if (u_spec->infotype == DIVY_INFOTYPE_saml && divy_support_saml(r)) {
		return 1; /* SAMLがサポートで且つSAMLのURLならスルー */
	}

	if (u_spec->infotype == DIVY_INFOTYPE_login) {
		//if (divy_get_method_number(r) == M_GET)  
			return 1; /* GETはそのまま表示すればいいのでスルー */
	}

	if (u_spec->infotype == DIVY_INFOTYPE_shorten) {

		/* 短縮URLでもパラメータがないものは認証をさせます */
		if (IS_EMPTY(r->args)) 
			return 0;

		return 1; /* スルーする(認証不要) */
	}

	return 0; /* 認証必須 */
}

/**
 * リクエストのクライアントがブラウザかどうかを調べる
 *  @param r request_rec*
 *  @return 1: ブラウザ / 0: それ以外
 */
DIVY_DECLARE(int)
divy_util_auth_request_is_browser(request_rec *r)
{
	dav_divy_dir_conf   *dconf = dav_divy_get_dir_config(r);
	const char *accept = NULL, *origin = NULL;
	const char *ua = NULL;
	divy_uri_spec *u_spec = NULL;
	const char * header = NULL;
	char *authtype = NULL;
	apr_table_t* jar;
	const char *ch;

	/* sessionを利用しない場合はURLを調べて短縮URLならブラウザとする */
	if (divy_support_session(r) == 0) {
		/* URLをパースする */
		(void) divy_parse_uri(r->pool, dav_divy_get_root_uri(r),
							  						r->uri, &u_spec);
		if (u_spec->infotype != DIVY_INFOTYPE_shorten) {
			return 0;
		}
	}

	/* MozillaもしくはOperaでなければブラウザではないと返します */
	ua = dav_divy_get_user_agent(r);
	if (!ua || (strncmp(ua, "Mozilla", 7) != 0
					&& strncmp(ua, "Opera", 5) != 0)) {
		return 0;
	}

	/*
	 * ブラウザで且つCookieが含まれていた場合authtypeがあるかもしれない
	 * このauthtype=samlの場合はSAMLでの認証を行った証拠です。
	 * キャッシュとして記録をします。
	 */
	header = dav_divy_get_cookie(r);
	if (IS_FILLED(header)) {
		jar = apr_table_make(r->pool, APREQ_DEFAULT_NELTS);
		if (apreq_parse_cookie_header(r->pool, jar, header) != APR_SUCCESS) {
			/* TODO　エラーメッセージを */
			return 1;
		}

		authtype = (char*)apr_table_get(jar, "authtype");
		if (IS_FILLED(authtype) && (strcmp(authtype, "saml") == 0)) {
			divy_pcache_set_flag(r->pool, 1, DIVY_PCACHE_FLG_USE_SAML);
		}

	}

	/* AcceptヘッダーかOriginヘッダーがあるのがブラウザとする */
	/* 大体がAcceptヘッダーが付いているがGoogleDocsのScriptが
	 * 付いていない
	 * Originが付いているということはCORSに対応させるともいえる
	 */
	accept = apr_table_get(r->headers_in, "Accept");
	origin = apr_table_get(r->headers_in, "Origin");

	/* 環境変数の強制WebDAVクラアントかの有無を取得 */
	ch = apr_table_get(r->subprocess_env, DIVY_APENV_FORCE_WEBDAV_USER_AGENT);

	/* 
	 * 次の条件にすべて合致してブラウザとする
	 * Acceptヘッダがある もしくは Originヘッダがある
	 * 強制WebDAVクライアントになっていない
	 */
	return ((IS_FILLED(accept) || IS_FILLED(origin)) && !DIVY_APENV_IS_ON(ch));
}

/**
 *  許可された場所からのパスワードなしログインなのかを調べる
 *
 *  @param r request_rec*
 *	@param sent_pw const char*	送信されたパスワード
 *  @return 1: 許可されている / 0: 許可されない
 */
DIVY_DECLARE(int)
divy_util_auth_allow_ignore_login(request_rec *r, const char* sent_pw)
{
	dav_divy_dir_conf   *dconf = dav_divy_get_dir_config(r);
	apr_ipsubnet_t      *ipsub = NULL;

	int i, len = divy_array_getlength(dconf->authignoreipaddrs);

	for (i = 0; i < len; i++) {
		ipsub = divy_array_get(dconf->authignoreipaddrs, i);
		if (ipsub != NULL) {
			if (apr_ipsubnet_test(ipsub, r->useragent_addr)) {
				/* マスターパスワードがあればそれと比較 */
				if (dconf->masterpassword && sent_pw) {
					if (strcmp(dconf->masterpassword, sent_pw)) {
						/* マスターパスワードと不一致 */
						return 0;
					}
				}
				/* 許可されている */
				return 1;
			}
		}
	}

	/* ここにきたら許可されない */
	return 0;
}

/**
 * login画面へ移動する
 *
 * @param r request_rec*
 * @param sid const char*	セッションID
 * @param reason int 遷移理由(エラーコード)
 * @return int (HTTP_STATUS)
 */
DIVY_DECLARE(int) divy_redirect_login(request_rec *r, const char *sid, int reason)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	apr_pool_t *pool = r->pool;
	divy_uri_spec *u_spec = NULL;
	const char* header = NULL;
	char *p, *s, *next;
	const char* root_uri = dav_divy_get_root_uri(r);

	/* ブラウザではない場合はすぐAuthエラーを戻すだけでいい */
	if (divy_util_auth_request_is_browser(r) == 0) {
		if (reason == HTTP_UNAUTHORIZED) {
			divy_note_auth_failure(r);
		}
		return reason;
	}

	/* URLをパースする */
	(void) divy_parse_uri(r->pool, root_uri, r->uri, &u_spec);

	/*
	 * 短縮URLの場合で、且つパラメータがある場合だけリダイレクトは無視できます
	 */
	if (u_spec->infotype == DIVY_INFOTYPE_shorten && IS_FILLED(r->args)) {
		return OK;
	}
	else {
		s = apr_pstrdup(r->pool, r->unparsed_uri);
		p = divy_strstr(s, "&WEBDAV_METHOD=");
		if (p != NULL) *p = '\0';

        next = apr_psprintf(pool, "%s?next=%s", divy_build_login_uri(pool, root_uri), s);
	}

	if (reason != HTTP_OK) {
		next = apr_psprintf(pool, "%s&st=%d", next, reason);
	}

	if (IS_FILLED(sid)) {
		ERRLOG2(pool, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"delete session uid=%s, sid=%s", r->user, sid);
		if (dconf->usememcache == DIVY_TFMEMCACHE_ON) {
			/* SAMLが有効になっていない場合だけ削除する */
			if (divy_support_saml(r) == 0) {
				(void)divy_memcache_delete(pool, dconf->memd, root_uri, sid, 0);
			}
			else {
				next = apr_psprintf(pool, "%s&token=%s", next, sid);
			}
		}
		else {
			(void)divy_rdbo_delete_session(r, NULL, NULL, sid);
		}
	}

	/* Locationの設定 */
	apr_table_setn(r->headers_out, "Location", next);

	/* Cookieヘッダを取得する */
	header = apr_table_get(r->headers_in, "Cookie");
	if (IS_FILLED(header)) {
		/* Cookieを削除するヘッダを追加する */
		(void)divy_set_delete_session_cookie(r);
	}
	return HTTP_MOVED_TEMPORARILY;
}

/**
 * Basic認証のヘッダーを作成する
 */
DIVY_DECLARE(void) divy_note_auth_failure(request_rec *r)
{
	TRACE(r->pool);

	char *authtype = "Basic";	/* default BASIC */
	char *authparam = NULL;

	const char *auth_line = apr_table_get(r->headers_in,
						(PROXYREQ_PROXY == r->proxyreq)
						   ? "Proxy-Authorizeation"
						   : "Authorization");

	if (auth_line) {
		authtype = ap_getword(r->pool, &auth_line, ' ');
	}

	if (strcasecmp(authtype, "Bearer") == 0) {
		authparam = apr_psprintf(r->pool, "Bearer realm=\"\" authorization_uri=%s, resource_uri=%s, trusted_issuers=\"https://login.teamfile.com/*/\" token_type=\"app_asserted_user_v1 service_asserted_app_v1\"",  "https://testdrive.teamfile.com/api/oauth2/authorize", "https://testdrive.teamfile.com");
		apr_table_addn(r->err_headers_out, "X-FEServer", "testdrive.teamfile.com");
		// digestも通用しない(2023/1218チェック済み)
		//authparam = apr_psprintf(r->pool, "%s realm=\"%s\" nonce=\"RMH1usDrAwA=6dc290ea3304de42a7347e0a94089ff5912ce0de\",algorithm=MD5, qop=\"auth\"", "Digest", ap_auth_type(r));
	}
	else {
		authparam = apr_psprintf(r->pool, "%s realm=\"%s\"", authtype, ap_auth_type(r));
	}

	apr_table_setn(r->err_headers_out,
						(PROXYREQ_PROXY == r->proxyreq) ? "Proxy-Authenticate"
							:  "WWW-Authenticate", authparam);
}

/**
 * セッションクッキーを削除するヘッダを作る
 * @param r request_rec*
 */
DIVY_DECLARE(void) divy_set_delete_session_cookie(request_rec *r)
{
	apr_pool_t *pool = r->pool;
	apreq_cookie_t *cookie;
	char *value = NULL;

	value = apr_pstrdup(pool, "deleted");
	cookie = apreq_cookie_make(pool, TF_COOKIE_SES_NAME,
					strlen(TF_COOKIE_SES_NAME), value, strlen(value));
	cookie->path = (char*)dav_divy_get_root_uri(r);

	/* 過去日付をいれてCookieを削除させる */
	apreq_cookie_expires(cookie, "-3y");
	apr_table_add(r->err_headers_out, "Set-Cookie",
							apreq_cookie_as_string(cookie, pool));

	cookie = apreq_cookie_make(pool, "authtype", 8, value, strlen(value));
	cookie->path = (char*)dav_divy_get_root_uri(r);
	apreq_cookie_expires(cookie, "-3y");
	apr_table_add(r->err_headers_out, "Set-Cookie",
							apreq_cookie_as_string(cookie, pool));

}

/**
 * BOXのパスワードが正しいかを調べます
 */
DIVY_DECLARE(int)
divy_util_box_authenticate_user(request_rec *r, const char* key, int update)
{
	const char* value = NULL;
	divy_rdbo_session *session = NULL;

	value = divy_util_auth_get_cookie_value(r, TF_COOKIE_SES_NAME);
	if (IS_EMPTY(value)) return 1; 	/* エラー */

	if (divy_enable_memcache(r)) {
		session = divy_util_auth_get_memcache_session(r, value, update);
	}
	else {
		session = divy_rdbo_get_session(r, value);
	}
	if (session == NULL) {
		/* 値があったがセッションがなかったらセッションを削除する */
		divy_set_delete_session_cookie(r);
		return 1;	/* エラー */
	}

	/* BOXのパスワードと同じか調べる */
	if (IS_EMPTY(session->password) || (IS_FILLED(session->password) && (strcmp(session->password, key) != 0))) {
		/* パスワード不一致 cookie削除 / セッション削除 */
		divy_set_delete_session_cookie(r);
		divy_util_auth_remove_session(r, value);
		return 1; /* エラー */
	}

	return 0;	/* 正常 */
}

/**
 * BOXを展開できるユーザかを調べます
 */
DIVY_DECLARE(int)
divy_util_box_validate_user(request_rec *r,
								const char* id, const char* boxvalidmailaddr) {

	divy_auth_session session = {0};
	char *mailaddr = NULL;
	char *addrs = NULL;
	char *token = NULL;
	char *ctx = NULL;
	const char* sid;
	int match = 0; /* 不一致 */
	ap_regex_t preg;

	if (IS_EMPTY(boxvalidmailaddr)) {
		return 0;	/* 設定されていないので評価必要なし正常扱い */
	}

	TRACE(r->pool);

	/*
	 * SIDからセッションに保持しているメールアドレスを取得する
	 */

	/* クッキーからSIDを取得する */
	sid = divy_util_auth_get_cookie_value(r, TF_COOKIE_SES_NAME);
	if (IS_EMPTY(sid)) {
		return -1; /* 未設定 */
	}

	session.sid = apr_pstrdup(r->pool, sid);
	if (divy_util_auth_get_memcache_userinfo(r, &session)) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"sid not found [%s]", sid);
		return -1; /* 未設定 */
	}

	mailaddr = session.validmail;

	/*
	 * BOXのメールアドレスは複数人許可することができる
	 * カンマ区切りを調べていく
	 */
	if (IS_FILLED(mailaddr)) {
		addrs = apr_pstrdup(r->pool, boxvalidmailaddr);

		/* アスタリスクを正規表現に変換する */
		addrs = (char *)dav_divy_replace_str(r->pool, addrs, "*", ".*", NULL);

		/* メールアドレスは大文字小文字を意識しないので小文字に変換する */
		ap_str_tolower(mailaddr);
		ap_str_tolower(addrs);
		
		while ((token = apr_strtok(addrs, ",", &ctx)) != NULL) {
			addrs = NULL;
			if (token == NULL) break;

			token = (char *) dav_divy_trim_white(r->pool, token);
			if (strcmp(mailaddr, token) == 0) {
				match = 1; /* 一致した */
				break;
			}
			if ((ap_regcomp(&preg, token, 0)) == 0) {
				if (ap_regexec(&preg, mailaddr, 0, NULL, 0) == 0) {
					match = 1; /* 一致した */
					ap_regfree(&preg);
					break;
				}
				ap_regfree(&preg);
			}
		}
	}
	else {
		/* メールアドレスが空白の場合は未設定扱い */
		return -1;
	}

	if (!match) {
		/* 不一致 cookie削除 / セッション削除 */
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"invalid mail address [%s]", mailaddr);
		divy_set_delete_session_cookie(r);
		divy_util_auth_remove_session(r, sid);
	}

	return !match;
}

DIVY_DECLARE(int)
divy_util_box_set_cookie(request_rec *r, divy_auth_session *session)
{
	apr_pool_t *pool = r->pool;
	apreq_cookie_t *cookie;

	TRACE(pool);

	/*
	 * P3Pヘッダを追加します
	 *
	 * これを追加しないとIEでCookieを保持してくれなくなります。
	 * 細かい意味は開発者の BugTrack-ServerFunction/273 を参考にすること
	 */
	apr_table_setn(r->err_headers_out, "P3P",
					"CP=\"CAO DSP LAW CURa ADMa DEVa TAIa PSAa PSDa IVAa "
					"IVDa OUR BUS IND UNI COM NAV INT\"");

	/* セッション情報をmemcacheに格納する */
	if (divy_util_auth_set_memcache_session(r, session, 0)) {
		/* 失敗 */
		return 1;
	}

	cookie = apreq_cookie_make(pool, TF_COOKIE_SES_NAME,
			strlen(TF_COOKIE_SES_NAME), session->sid, strlen(session->sid));
	cookie->path = (char*)dav_divy_get_root_uri(r);
	apr_table_setn(r->err_headers_out, "Set-Cookie",
							apreq_cookie_as_string(cookie, pool));

	return 0;
}

/**
 * セッション上のパスワードを変更する
 *
 * @param r 			request_rec*
 * @param uid			const char*	 ユーザーID
 * @param newpassword	const char*	 パスワード
 * @return int 0 : 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_util_auth_update_session_password(request_rec* r, const char* uid, const char* newpassword)
{

	apr_status_t st;
	divy_rdbo_session *session = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	apr_pool_t* p = r->pool;
	char* newvalue = NULL;
	char* prefix = apr_pstrdup(p, dconf->root_uri);

	/*
	 * tf-sessionを利用しているリクエストの場合でも、パスワード
	 * 変更は、CGIでのアップデートとなってしまいます。
	 * その場合、発生元のブラウザからのCookieはCGIにきません。
	 * （CGIからはBasic認証を行う）
	 * その為に、Basic認証でありながらブラウザからのtf-sessionの値を
	 * x-tf-sessionヘッダにしてTeamFileにアクセスするようにしました。
	 */
	const char* sid = apr_table_get(r->headers_in, "x-tf-session");
	
	TRACE(r->pool);

	if (IS_EMPTY(uid) || IS_EMPTY(sid) || IS_EMPTY(newpassword)) 
		return 1;

	session = divy_util_auth_get_memcache_session(r, sid, 0 /* no update */);
	if (session == NULL) return 0; /* 見つからなかったらそのまま終了 */
	
	newvalue = apr_psprintf(p, "uid=%s\tpassword=%s", session->userid, newpassword);

	if (IS_FILLED(session->mailaddr)) {
		newvalue = apr_psprintf(p, "%s\tmailaddr=%s", newvalue, session->mailaddr);
	}
	if (IS_FILLED(session->uniqueid)) {
		newvalue = apr_psprintf(p, "%s\tuniqueid=%s", newvalue, session->uniqueid);
	}

	/* 暗号化する */
	newvalue = (char*)divy_encipher_text(p, newvalue);

	st = divy_memcache_set(p, dconf->memd, prefix, session->sid, newvalue, 
								strlen(newvalue), dconf->sessiontimeout, 0);

	if (st == APR_SUCCESS) return 0;

	return 1;	/* エラー */

}

/**
 * セッションからユーザ情報を取得する
 *
 * @param r				request_rec *		リクエスト構造体
 * @param session		divy_auth_session* セッション
 * @return 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int)
divy_util_auth_get_memcache_userinfo(request_rec* r, divy_auth_session *session)
{

	divy_rdbo_session *rdbo_sess = NULL;
	if (session == NULL) return 1;

	TRACE(r->pool);

	rdbo_sess = divy_util_auth_get_memcache_session(r,
											session->sid, 0 /* no update */);

	TRACE(r->pool);

	if (rdbo_sess != NULL) {
		session->uid      = apr_pstrdup(r->pool, rdbo_sess->userid);
		session->password = apr_pstrdup(r->pool, rdbo_sess->password);
		session->validmail= apr_pstrdup(r->pool, rdbo_sess->mailaddr);
		session->authcode = apr_pstrdup(r->pool, rdbo_sess->authcode);
		session->uniqueid = apr_pstrdup(r->pool, rdbo_sess->uniqueid);
		return 0;
	}

	return 1; 	/* 失敗 */

}

/**
 * セッションにユーザー情報を追加する
 */
DIVY_DECLARE(int) 
divy_util_auth_set_memcache_session(request_rec *r, divy_auth_session *session, int tmp)
{

	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	apr_pool_t *p = r->pool;
	apr_status_t st;
	char* value = NULL;
	char *prefix = apr_pstrdup(p, dconf->root_uri);
	int timeout = (tmp) ? 5 : dconf->sessiontimeout;

	TRACE(p);

	/* IDは必須 */
	if (IS_EMPTY(session->uid))
		return 1;

	/* ID */
	value = apr_psprintf(p, "uid=%s", session->uid);

	/* パスワード */
	value = apr_psprintf(p, "%s\tpassword=%s", value,
			IS_EMPTY(session->password) ? "" : session->password);

	/* メールがあれば追加 */
	if (IS_FILLED(session->validmail)) {
		value = apr_psprintf(p, "%s\tmailaddr=%s", value, session->validmail);
	}

	/* ユニークIDがあれば追加 */
	if (IS_FILLED(session->uniqueid)) {
		value = apr_psprintf(p, "%s\tuniqueid=%s", value, session->uniqueid);
	}

	/* authcodeがあれば追加 */
	if (IS_FILLED(session->authcode)) {
		value = apr_psprintf(p, "%s\tauthcode=%s" , value, session->authcode);
	}

	if (IS_EMPTY(session->sid)) {
		session->sid = make_random_string(p);
	}

	/* 暗号化する */
	value = (char*)divy_encipher_text(p, value);

	st = divy_memcache_set(p, dconf->memd, prefix, session->sid, value,
												 strlen(value),
												 timeout, 0);
	if (st != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to insert memcached [user=%s]."
				"Please checked to memcached daemon.", session->uid);
		session->sid = NULL;
		return 1;
	}

	return 0;

}

/**
 * 指定されたセッションをリフレッシュする。内容の書き換えはない。
 */
DIVY_DECLARE(int)
divy_util_auth_refresh_session(request_rec *r, const char* sid)
{
	divy_rdbo_session *session;

	TRACE(r->pool);

	if (IS_EMPTY(sid)) return 1;
	session = divy_util_auth_get_memcache_session(r, sid, 1 /* update */);
	divy_pcache_set_data(r->pool, sid, DIVY_PCACHE_DAT_SES_SID);

	return (session != NULL);
}

/**
 * 指定されたセッションを削除します。無い場合はリクエストから取得します
 */
DIVY_DECLARE(int)
divy_util_auth_remove_session(request_rec *r, const char* sid)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char* root_uri = dav_divy_get_root_uri(r);
	
	/* sidがない場合は自分のリクエストから取得して見る */
	if (IS_EMPTY(sid)) {
		sid = divy_util_auth_get_cookie_value(r, TF_COOKIE_SES_NAME);
	}

	if (IS_EMPTY(sid) || IS_EMPTY(root_uri)) return 0;

	if (divy_enable_memcache(r)) {
		(void)divy_memcache_delete(r->pool, dconf->memd, root_uri, sid, 0);
	}
	else {
		(void)divy_rdbo_delete_session(r, NULL, NULL, sid);
	}

	return 0;
}

/**
 * 2FA認証を行ったユーザか？
 * @param r 		request_rec *	リクエスト構造体
 * @param id		const char*		ID文字列 (ユーザIDやセッションID）
 * @return int 		認証をパスしているかどうか（-1:期限切れ/1:パス/0:していない）
 */
DIVY_DECLARE(int)
divy_util_confirm_2FA_passed(request_rec *r, const char* id)
{
	apr_status_t st;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const divy_rdbo_usr *usr_pr = NULL;
	int timeout = 300; /* 秒 */
	char *key = NULL;
	char *token = NULL;
	char *value = NULL;
	time_t now = dav_divy_get_now_epoch();
	char *now_str = apr_psprintf(r->pool, "%"APR_SIZE_T_FMT, now);
	int is_guest = (strcmp(DIVY_GUEST_ID, r->user) == 0);
	apr_size_t length = 0;
	apr_uint16_t flags = 0;
	char *header_val = NULL;
	divy_auth_session session = { 0 };
	const char *mailaddr = NULL;
	const char *sid = NULL;
	const char *keyid = NULL;
	time_t period_epoch = 0;

	/* 2FAを有効にしないのならそのままパス */
	if (divy_support_2FA(r) == 0) return 1;

	TRACE(r->pool);

		
	/* 対象ユーザの情報をすべて取得する */
	usr_pr = divy_get_usrprop(r);

	/* 有効期限取得 */
	if (is_guest) {
		/* ゲストは有効期限は現在時刻より300秒 */
		period_epoch = now + timeout;
	}
	else {
		/* 2段階認証の有効期限をディレクティブより取得 */
		period_epoch = divy_util_auth_get_2faperiod_epoch(r);
	}
	
	/*
	 * ブラウザの場合でIDがNULLの場合はは、cookieよりSIDを取得し
	 * memcacheに記録されている情報からメールアドレスを導き出します
	 */
	if (id == NULL && divy_util_auth_request_is_browser(r)) {
		sid = divy_util_auth_get_cookie_value(r, TF_COOKIE_SES_NAME);
		if (IS_EMPTY(sid)) {
			return 0;	/* エラー */
		}
		session.sid = (char*)sid;
		if (divy_util_auth_get_memcache_userinfo(r, &session)) {
			/* 失敗 */
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"tf-session memcache not found");
			return 0;	/* エラー */
		}
		mailaddr = session.validmail;
		keyid = apr_pstrdup(r->pool, sid);

	}
	else {
		/* ユーザアカウントよりメールを取得 */
		mailaddr = (char *)divy_get_mailaddr(r);
		keyid = id;
	}

	/*
	 * 2要素認証を行っているかを調べます。
	 * キーに``@permit``が含まれている必要があります。
	 *
	 * 引数のID + @ + 登録日(ISO-8601 UTC) + "@permit"
	 */
	key = apr_psprintf(r->pool, "%s@%s@permit", keyid, usr_pr->registdt);

	/* IDの2FA情報を取得 */
	st = divy_memcache_get(r->pool, dconf->memd, dconf->root_uri,
											key, &value, &length, &flags);

	if (st == APR_SUCCESS) {
		/* レコードが存在した 
		 * 有効期限をチェックする */
		if (apr_atoi64(value) < period_epoch) {
			/* ゲスト接続のみ有効期限を更新する */
			if (is_guest) {
				divy_memcache_set(r->pool, dconf->memd, dconf->root_uri, key,
						value, strlen(now_str),
						timeout, 0);
			}
			return 1;	/* 有効期限内 */
		}
		else {
			return -1; /* 期限切れ */
		}
	}

	/*
	 * memcached を調べて対象ユーザがいないということは二要素認証を
	 * クリアしていないことになります。裏を返すとmemcacheを消すと
	 * 全員二要素認証やり直しということになります。
	 *
	 * 同様に@pendingのmemcacheレコードを作成します
	 */

	/* @pendingのレコードがあるかを調べる */
	key = apr_psprintf(r->pool, "%s@%s@pending", keyid, usr_pr->registdt);
	st = divy_memcache_get(r->pool, dconf->memd, dconf->root_uri,
											key, &value, &length, &flags);
	/* @pendingレコードがない */
	if (st == APR_NOTFOUND) {

		if (IS_FILLED(mailaddr)) {
			/* トークンを作成して@pendingレコードを期限付きで作る */
			token = apr_psprintf(r->pool, "%06d", rand() % 1000000 + 1);
			/* メールアドレスに対してトークンを送信 無条件に期限は5分*/
			divy_ml_2FA_token(r, mailaddr, token, now + 300);
			value = apr_psprintf(r->pool, "%s:%"APR_SIZE_T_FMT,
														token, now + 300);
			/* memcacheの生存期間は3時間 60 * 60 * 3*/
			divy_memcache_set(r->pool, dconf->memd, dconf->root_uri, key,
											value, strlen(value), 10800, 0);
		}

		if (dconf->type2fa == DIVY_2FA_TYPE_OTP_ONLY) {
			header_val = apr_pstrdup(r->pool, "otp-only");
		}
		else {
			header_val = apr_pstrdup(r->pool, "default");
		}
		apr_table_set(r->err_headers_out, "X-Tf-2FA-Required", header_val);

		return 0; /* パスしてない */
	}
	else {
		/* @pendingキーがあった */
		if (length) {

			/* valueのパース */
			char *authcode = ap_getword_nulls(r->pool,
													(const char**)&value, ':');
			time_t expired  = atoi(value);


			/* 有効期限のチェック */
			if (now > (time_t)expired) {
				/* レコードはあったが期限が切れていた */
				divy_memcache_delete(r->pool, dconf->memd,
												dconf->root_uri, key, 0);
				ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"expired code. [now epoch=%"APR_SIZE_T_FMT 
						" / authcode epoch= %"APR_SIZE_T_FMT"]",
						now, expired);
				return -1; /* 期限切れエラー */
			}
			
			/*
			 * 認証コードチェック
			 *
			 * 送信された認証コードと@pendingされているデータの認証コード
			 * と比較を行う
			 */
			if (IS_FILLED(session.authcode) && strncmp(session.authcode,
									authcode, strlen(authcode)) == 0) {

				/* pendingレコードを消去 */
				divy_memcache_delete(r->pool, dconf->memd,
												dconf->root_uri, key, 0);
				/* 許可された証拠を記録 */
				key = apr_psprintf(r->pool, "%s@%s@permit",
											keyid, usr_pr->registdt);
				/* TODO 無期限では困るよね */
				divy_memcache_set(r->pool, dconf->memd, dconf->root_uri, key,
										now_str, strlen(now_str),
										timeout, 0);
				return 1;	/* パス */
			}
		}
	}



	return 0; /* パスしてない */

/*
	int totp = 0;
	unsigned char secret[255] = { 0 };
	int secretlen = 255;

	divy_get_TOTP((const unsigned char*)r->user,
								30, &totp, secret, secretlen);
	ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"TOTP = %06d / secret = %s", totp, secret);

	if (length == 0) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"not found [%s]", value);
		return 1;
	}

	ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"pass !");

	return 1;	// passed !!
*/
}

/**
 * 指定されたcookieが送られてきたか?
 * @param r request_rec * リクエスト構造体
 * @param name	const char*	Cookie名
 * @return int (1: ある / 0: ない)
 */
DIVY_DECLARE(int)
divy_util_auth_has_cookie(request_rec *r, const char* cookiename)
{
	apr_table_t* jar;
	char *sess_cookie = NULL;

	jar = apr_table_make(r->pool, APREQ_DEFAULT_NELTS);
	if (jar) {
		sess_cookie = (char *)apr_table_get(jar, TF_COOKIE_SES_NAME);
	}

	return IS_FILLED(sess_cookie);
}

/*------------------------------------------------------------------------------
  Define private functions 
  ----------------------------------------------------------------------------*/
/*
 * Cookieヘッダからnameに該当するデータを返す
 * @param r request_rec *
 * @param name const char* 項目名
 * @return value 値 ( NULL = なし )
 */
static const char*
divy_util_auth_get_cookie_value(request_rec* r, const char* name)
{

	const char* header;
	apr_table_t* jar;

	TRACE(r->pool);

	/* Cookieヘッダを調べる */
	header = dav_divy_get_cookie(r);
	if (IS_EMPTY(header)) {
		/* Cookieヘッダがない場合 */
		/* 何もLOGを出さない */
		return NULL;
	}

	/* Cookie パース */
	jar = apr_table_make(r->pool, APREQ_DEFAULT_NELTS);
	if (apreq_parse_cookie_header(r->pool, jar, header) != APR_SUCCESS) {
		/* パース失敗 */
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
				"Failed to parse cookie. cookie value = %s", header);
		return NULL;
	}

	return apr_table_get(jar, name);

}

/**
 * memcacheからセッションIDに該当するデータを取得する
 *
 * @param r					request_rec *	リクエスト構造体
 * @param sessid 			const char *	検索するセッションID
 * @param withupdatetime	withupdatetime	1でタイムアウトを更新
 * @return session構造体 / NULL = 失敗
 */
static divy_rdbo_session*
divy_util_auth_get_memcache_session(request_rec *r, const char* sessid, int withupdatetime)
{

	apr_status_t st;
	divy_rdbo_session *session = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	apr_pool_t *p = r->pool;
	apr_size_t length = 0;
	apr_uint16_t flags = 0;
	char *prefix = apr_pstrdup(p, dconf->root_uri);
	char *value = NULL;
	char *parts = NULL;
	char *key, *val, *ctx;
	char *uid , *password, *mailaddr, *uniqueid,  *authcode;
	char *savevalue = NULL;
	int is_guest = 0;	/* ゲストではない */
	int timeout = 0;

	TRACE(p);

	st = divy_memcache_get(p, dconf->memd, prefix,
								 sessid, &value, &length, &flags);
	if (st != APR_SUCCESS) return NULL;

	/* memcacheの値を退避 */
	savevalue = apr_pstrdup(p, value);

	/* 複合化する */
	value = (char*)divy_decipher_text(p, value);
	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "[value] = %s", value);

	uid = password = mailaddr = uniqueid = authcode = NULL;
	if (st == APR_SUCCESS && IS_FILLED(value)) {
		while(IS_FILLED(value)) {
			parts = ap_getword_nulls(p, (const char**)&value, '\t');
			key = apr_strtok(parts, "=", &ctx);
			val = apr_strtok(NULL, "=", &ctx);
			if (strcmp(key, "uid") == 0) {
				uid = val;
			}
			else if (strcmp(key, "password") == 0) {
				password = val;
			}
			else if (strcmp(key, "mailaddr") == 0) {
				mailaddr = val;
			}
			else if (strcmp(key, "uniqueid") == 0) {
				uniqueid = val;
			}
			else if (strcmp(key, "authcode") == 0) {
				authcode = val;
			}
			
			ERRLOG3(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"[%s] %s = %s", parts, key, val);
		}

		/* udiが取れない場合は失敗 */
		if (IS_EMPTY(uid)) return NULL;

		is_guest = (strcmp(DIVY_GUEST_ID, uid) == 0);
		timeout = (is_guest) ? 300 : dconf->sessiontimeout; 
	}

	session = apr_pcalloc(p, sizeof(divy_rdbo_session));
	session->sid       = (char*)sessid;
	session->userid    = uid;
	session->logindate = 0;
	session->password  = password; 
	session->mailaddr  = mailaddr;
	session->authcode  = authcode;
	session->uniqueid  = uniqueid;

	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"get memcache session password = %s", session->password);

	if (withupdatetime) {
		/* 更新する */
		st = divy_memcache_set(p, dconf->memd, prefix, sessid, savevalue,
								strlen(savevalue), timeout, 0);	
		if (st != APR_SUCCESS) return NULL;
	}

	return session;
}

static time_t 
divy_util_auth_get_2faperiod_epoch(request_rec *r)
{
	apr_pool_t *p = r->pool;
	apr_status_t rv;
	time_t now = dav_divy_get_now_epoch();
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	char *value = NULL;
	time_t period = 0;
	char *date_str = NULL;
	apr_os_exp_time_t *tms = NULL;
	apr_time_exp_t apr_expt = { 0 };
	apr_time_t apr_time;
	int base_time = 0;
	time_t result_time = 0;
	int tmp_month = 0;
	int add_year = 0;

	tms = apr_pcalloc(p, sizeof(apr_os_exp_time_t));

	/* 月 */
	value = dav_divy_find_prestr(p, dconf->cycleperiod2fa, "m");

	if (IS_FILLED(value)) {
		divy_format_time_t(p, now, DIVY_TIME_STYLE_ISO8601, &date_str);

		date_str[4] = '\0';
		date_str[7] = '\0';
		date_str[10] = '\0';
		date_str[13] = '\0';
		date_str[16] = '\0';
		date_str[19] = '\0';

		/* 年 ポインタをずらしていくので取得しておく*/
		tms->tm_year = atoi(date_str) - 1900;

		/* 月 単純に月にプラスする */
		tmp_month = atoi(date_str+=5) - 1 + atoi(value);

		if (tmp_month > 12) {
			/* 12月をオーバーしていたら年数を算出 */
			add_year = tmp_month / 12;
			/* 月を算出 */
			tmp_month = tmp_month - (12 * add_year);
		}

		tms->tm_year += add_year;
		tms->tm_mon  = tmp_month;
		tms->tm_mday = atoi(date_str+=3);
		tms->tm_hour = 0;
		tms->tm_min  = 0;
		tms->tm_sec  = 0;

		apr_os_exp_time_put(&apr_expt, &tms, p);
		/* GMT では以下は必要なし */
		apr_expt.tm_gmtoff = 0;
		/* 戻りがtime_tであるならu_secの値は無意味 */
		apr_expt.tm_usec = 0;
		/* apr_time_exp_t --> apr_time */
		rv = apr_time_exp_get(&apr_time, &apr_expt);

		if (rv != APR_SUCCESS) {
			/* 存在しない日付だったので1日引く */
			tms->tm_mday--;
			apr_os_exp_time_put(&apr_expt, &tms, p);
			/* GMT では以下は必要なし */
			apr_expt.tm_gmtoff = 0;
			/* 戻りがtime_tであるならu_secの値は無意味 */
			apr_expt.tm_usec = 0;
			/* apr_time_exp_t --> apr_time */
			rv = apr_time_exp_get(&apr_time, &apr_expt);
		};

		period = (time_t) apr_time_sec(apr_time);

	}

	if (period == 0) {

		/* 日 */
		value = dav_divy_find_prestr(p, dconf->cycleperiod2fa, "d");
		if (IS_EMPTY(value)) {
			return 1;	/* どちらも空白 */
		}

		/* 単純日数計算 */
		period = now + atoi(value) * 60 * 60 * 24;
	}

	/* UTC => JST */
	period += 9 * 60 * 60;

	divy_format_time_t(p, period, DIVY_TIME_STYLE_ISO8601, &date_str);

	date_str[4] = '\0';
	date_str[7] = '\0';
	date_str[10] = '\0';
	date_str[13] = '\0';
	date_str[16] = '\0';
	date_str[19] = '\0';

	/* 開始時間 時単位 */
	base_time = dconf->cyclebasetime2fa;

	tms->tm_year = atoi(date_str) - 1900;
	tms->tm_mon  = atoi(date_str+=5) - 1;
	tms->tm_mday = atoi(date_str+=3);
	tms->tm_hour = base_time;
	tms->tm_min  = 0;
	tms->tm_sec  = 0;

	apr_os_exp_time_put(&apr_expt, &tms, p);
	/* GMT では以下は必要なし */
	apr_expt.tm_gmtoff = 0;
	/* 戻りがtime_tであるならu_secの値は無意味 */
	apr_expt.tm_usec = 0;
	/* apr_time_exp_t --> apr_time */
	rv = apr_time_exp_get(&apr_time, &apr_expt);
	if (rv != APR_SUCCESS) {
		return (time_t) 0;
	}

	/* apr_time -> time_t */
	result_time = (time_t) apr_time_sec(apr_time);

	/* JST => UTC */
	result_time -= 9 * 60 * 60;

	return result_time;

}

