/* $id:$ */

#include "util_crypt.h"
#include "util.h"
#include "apr_strings.h"

#include "openssl/sha.h"
#include "openssl/evp.h"	/* openssl evp	*/ 

#ifdef _WIN32
#include <string.h>
#include <stdlib.h>
#else
#ifdef HAVE_STRING_H
#include <string.h>			/* strdup	*/
#endif
#ifdef HAVE_MALLOC
#include <stdlib.h>			/* malloc, free	*/
#endif
#endif

/* default iv */
static unsigned const char def_iv[] = {'f', '8', 'e', 'd', '1', '0', '7', 'f', 'd', '0', '5', '4', 'f', 'b', '5', '7', '9', 'f', '6', '3', '7', '0', 'f', '5', '6', '2', '5', '0', '5', '9', '7', '8', '9', 'a', '0', '0', '3', '6', 'b', '9', '\0'};

/* define default buffer */
#define TF_CIPHER_DEF_BUFFER_SIZE 4096

/*
 *	暗号化を行うコンテキスト
 */
struct __tf_crypt_info_t
{
	int				mode;       /* モード 1: 暗号化 / 2: 複合化 */
	char			*iv;		/* IV */
	EVP_CIPHER_CTX	*ctx;		/* OpenSSLの暗号化コンテキスト */
	const EVP_CIPHER *cipher;	/* OpenSSLの暗号種類 */
};

/* private function */
static int _tf_crypt_set_cipher(tf_crypt_info_t* info, int type);
static const char* _tf_cipher_file(request_rec *r, const char* filepath, int mode, int cipher, const char* key, unsigned char* iv);

/* public function */
DIVY_DECLARE(const char*)
tf_encrypt_file(request_rec *r, const char *filepath, int type, const char* key, unsigned char* iv)
{
	return _tf_cipher_file(r, filepath, TFCRYPT_CIPHERMODE_ENCRYPT,
							type, key, iv);
}

DIVY_DECLARE(const char*)
tf_decrypt_file(request_rec *r, const char *filepath, int type, const char* key, unsigned char* iv)
{
	return _tf_cipher_file(r, filepath, TFCRYPT_CIPHERMODE_DECRYPT, 
							type, key, iv);
}

DIVY_DECLARE(int)
tf_cipher_file2sha1(apr_pool_t* p, const char *filepath, unsigned char* digest)
{
	apr_finfo_t finfo = {0};
	apr_file_t *fd = NULL;
	apr_status_t rv;
	apr_size_t nbytes = 0;
	char *buf = NULL;

	SHA_CTX ctx;

	/* ファイルの存在 */
	rv = apr_stat(&finfo, filepath, APR_FINFO_TYPE, p);
	if (rv || finfo.filetype != APR_REG) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to open reguler file."
				"(filepath = %s, code = %d)", filepath, finfo.filetype);
		return 0;
	}

	/* SHA1の初期化 */
	if (!SHA_Init(&ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to SHA1 initialize context.");
		return 0;
	}

	rv = apr_file_open(&fd, filepath, APR_READ | APR_BINARY, APR_OS_DEFAULT, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to open file. "
				"(filepath = %s, code = %d)", filepath, rv);
		return 0;
	}

	buf = apr_pcalloc(p, sizeof(char) * TF_CIPHER_DEF_BUFFER_SIZE);
	while(1) {
		nbytes = TF_CIPHER_DEF_BUFFER_SIZE;
		rv = apr_file_read(fd, buf, &nbytes);
		if (rv == APR_EOF) {
			if (nbytes) {
				if (!SHA_Update(&ctx, (const void*)buf, nbytes)) {
					ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"Failed to SHA1 Update Error.");

					(void)apr_file_close(fd);
					return 0;
				}
			}
			break;
		}
		else if (rv != APR_SUCCESS) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to open file for digest operation. (code = %"APR_SIZE_T_FMT"d)",
					nbytes);

			(void) apr_file_close(fd);
			return 0;
		}

		if (nbytes == 0) break;

		if (!SHA_Update(&ctx, (const void*)buf, nbytes)) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to SHA1 Update Error.");

			(void) apr_file_close(fd);
			return 0;
		}
	}

	return SHA_Final(digest, &ctx);
}

DIVY_DECLARE(int)
tf_cipher_str2sha1(apr_pool_t* p, const char* string, unsigned char* digest)
{
	SHA_CTX ctx;
	char *buf = NULL;
	apr_size_t len = 0;

	if (p == NULL || IS_EMPTY(string)) return 0;

	len = strlen(string);

	/* SHA1の初期化 */
	if (!SHA_Init(&ctx)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to SHA1 initialize context.");
		return 0;
	}

	buf = apr_pcalloc(p, sizeof(char) * TF_CIPHER_DEF_BUFFER_SIZE);
	if (!SHA_Update(&ctx, (const void*)buf, len)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to SHA1 Update Error.");
		return 0;
	}

	return SHA_Final(digest, &ctx);
}

const char*
_tf_cipher_file(request_rec *r, const char* filepath, int mode, int cipher, const char* key, unsigned char* iv)
{
	int ret = 0;
	apr_pool_t *p = r->pool;
	apr_status_t rv;
	divy_fstorage_t *fstorage = NULL;
	apr_file_t *fd = NULL;
	apr_file_t *fd2 = NULL;
	char *buf = NULL; 
	char *buf2= NULL;
	char *retpath = NULL; /* 戻り値のパス */
	apr_size_t nbytes;
	divy_pfile_t *pfile = NULL;
	tf_crypt_info_t *cinfo = NULL;

	/* ストレージのopen */
	ret = divy_fstorage_open(r, p, &fstorage);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to open storage.");
		return NULL;
	}

	/* 一時ファイルパスを作成する */
	ret = divy_pfile_mktemp(fstorage, p, DIVY_TMP_FILE_TEMPLATE_NAME, &pfile);
	if (ret != DIVY_FS_ST_OK) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to make temporary file path.(code = %d) ", ret);
		return NULL;
	}
	retpath = apr_pstrdup(p, divy_pfile_get_fullpath(pfile));

	/* ストレージを閉じる */
	(void) divy_fstorage_close(fstorage);

	/* 一時ファイルを開く */
	rv = apr_file_mktemp(&fd2, retpath, APR_WRITE | APR_CREATE | APR_TRUNCATE | APR_BINARY, p);
	if (rv != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open temporary file for decrypt operation."
			"(code = %d / tmppath = %s)", rv, retpath);

		return NULL;
	}

	ERRLOG1(p, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG, "retpath = %s", retpath);

	/* 元のファイルを開く */
	rv = apr_file_open(&fd, filepath, APR_READ | APR_BINARY, APR_OS_DEFAULT, p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open file for decryption operation. (code = %d)", rv);
		goto cleanup;
	}

	/* 暗号化コンテキスト初期化 
	 * ここのmodeで暗号(1)/復号(2)が決まります
	 */
	if (!tf_crypt_stream_init(&cinfo, mode, cipher)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to initialize cipher context.");

		goto cleanup;
	}

	/* キーと初期化ベクトル(iv)の登録 */
	if (iv != NULL) {
		cinfo->iv = apr_pstrdup(p, (const char*)iv);
	}
	else {
		cinfo->iv = &def_iv;
	}
	if (!tf_crypt_stream_set_key(cinfo, key, (int)strlen(key))) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to cipher set key or iv.");

		goto cleanup;
	}

	/* ファイルの読み込みと暗号・復号化処理 */
	buf = apr_pcalloc(p, sizeof(char) * TF_CIPHER_DEF_BUFFER_SIZE);
	buf2= apr_pcalloc(p, sizeof(char) * TF_CIPHER_DEF_BUFFER_SIZE);
	while(1) {
		nbytes = TF_CIPHER_DEF_BUFFER_SIZE;
		rv = apr_file_read(fd, buf, &nbytes);
		if (rv == APR_EOF) {
			if (nbytes) {
				/* 暗号/復号実処理 */
				if (!tf_crypt_stream(cinfo, buf2, buf, nbytes)) {
					ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"Failed to cipher stream (mode = %d)", mode);

					/* 後始末 */
					(void) apr_file_remove(retpath, p);
					goto cleanup;
				}

				rv = apr_file_write_full(fd2, buf2, nbytes, NULL);
				if (rv != APR_SUCCESS) {
					ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
							"Failed to write encription file"
							"(file = %s, code = %d)", retpath, rv);

					/* 後始末 */
					(void) apr_file_remove(retpath, p);
					goto cleanup;
				}
			}
				
		}
		else if (rv != APR_SUCCESS) {
			/* 読み込みの失敗 */
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to read file (file = %s, code = %d)", filepath, rv);

			/* 後始末 */
			(void) apr_file_remove(retpath, p);
			goto cleanup;
		}

		if (nbytes == 0) break;

		/* 暗号/復号実処理 */
		if (!tf_crypt_stream(cinfo, buf2, buf, nbytes))
		{
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to cipher stream (mode = %d)", mode);

			/* 後始末 */
			(void) apr_file_remove(retpath, p);
			goto cleanup;
		}
		rv = apr_file_write_full(fd2, buf2, nbytes, NULL);
		if (rv != APR_SUCCESS) {
			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to write encription file"
					"(file = %s, code = %d)", retpath, rv);

			/* 後始末 */
			(void) apr_file_remove(retpath, p);
			goto cleanup;
		}
	}

cleanup:

	/* 後始末 */
	(void) apr_file_close(fd);
	(void) apr_file_close(fd2);

	if (cinfo) tf_crypt_stream_cleanup(cinfo);

	return retpath;

}

int
tf_crypt_stream_init(tf_crypt_info_t** info, int mode, int cipher)
{
	tf_crypt_info_t     *tmpinfo;
	EVP_CIPHER_CTX		*ctx;
	int                 ret = 0; /* default failure */
	
	if (mode < 1 || mode > 2) return 0;
	
	tmpinfo = (tf_crypt_info_t*)malloc(sizeof(tf_crypt_info_t));
	if (!tmpinfo) return 0;

	ctx = (EVP_CIPHER_CTX*)malloc(sizeof(EVP_CIPHER_CTX));
	if (!ctx) return 0;
	EVP_CIPHER_CTX_init(ctx);

	/* cipherを設定する */
	if (!_tf_crypt_set_cipher(tmpinfo, cipher))
		return 0;

	tmpinfo->mode	= mode;
	tmpinfo->ctx    = ctx;
	tmpinfo->iv     = NULL;

	*info = tmpinfo;

	/* 初期化する keyとivは後で設定する */
	if (mode == 1)
		ret = EVP_EncryptInit((*info)->ctx, (*info)->cipher, NULL, NULL);
	else
		ret = EVP_DecryptInit((*info)->ctx, (*info)->cipher, NULL, NULL);
		
	return ret;

}

int
tf_crypt_stream_cleanup(tf_crypt_info_t* info)
{
	if (info == NULL) return 1;

	/* コンテキストの削除 */
	if (!info->ctx) {
		EVP_CIPHER_CTX_cleanup(info->ctx);
		free(info->ctx);
	}

	free(info);
	info = 0;

	return 1;
}

int
tf_crypt_stream_set_key(tf_crypt_info_t* info, const char* key, int keylen)
{

	int ret = 0; /* default failure */

	if (info == NULL) return 0;
	if (key == NULL || keylen == 0) return 0;

	if (info->mode == 1)
		ret = EVP_EncryptInit(info->ctx, info->cipher, (const unsigned char*)key, (unsigned char*)info->iv);
	else
		ret = EVP_DecryptInit(info->ctx, info->cipher, (const unsigned char*)key, (unsigned char*)info->iv);

	return ret;
}

int
tf_crypt_stream(tf_crypt_info_t* info, char* out, const char* in, int buflen)
{
	int ret = 0; /* default failure */
	int outlen = 0;

	if (info == NULL || out == NULL || in == NULL || buflen <= 0) return 0;

	if (info->mode == 1) {
		ret = EVP_EncryptUpdate(info->ctx, (unsigned char*)out, &outlen,
								(const unsigned char*)in,
								buflen);
	}
	else if (info->mode == 2)
	{
		ret = EVP_DecryptUpdate(info->ctx, (unsigned char*)out, &outlen,
								(const unsigned char*)in,
								buflen);
	}
	else {
		return 0;
	}

	/* 入力と出力は同じでなければならない */
	if (ret && outlen != buflen) return 0;

	return ret;
}


int
tf_crypt_string(char* out, int* outlen, const char* in, int inlen, const char* key, int mode, int cipher)
{
	int ret = 0;
	tf_crypt_info_t	*info = NULL;

	if (mode < 1 || mode > 2) return 0;

	if (tf_crypt_stream_init(&info, mode, cipher))
	{
		if (tf_crypt_stream_set_key(info, key, (int)strlen(key)))
		{
			ret = tf_crypt_stream(info, out, in, inlen);
			if (ret) *outlen = inlen;
		}
	}

	if (info)
		tf_crypt_stream_cleanup(info);
	
	return ret;
}

int
tf_crypt_get_ciphertype(const char* string)
{
	int i = 0;
	int match = 0;

	if (IS_EMPTY(string)) return TFCRYPT_CIPHER_NULL;

	for(i = 0; TFCRYPT_CIPHERTYPE_TABLE[i].string != NULL; i++) {
		match = strcmp(string, TFCRYPT_CIPHERTYPE_TABLE[i].string);
		if (match == 0)  return TFCRYPT_CIPHERTYPE_TABLE[i].type;
	}

	return TFCRYPT_CIPHER_NULL;
}

/* private function */
int
_tf_crypt_set_cipher(tf_crypt_info_t* info, int type)
{

	const EVP_CIPHER	*cipher;

	if (info == NULL) return 0;

	switch(type)
	{
		case TFCRYPT_CIPHER_CAST5_OFB:
			cipher = EVP_cast5_ofb();
			break;
#if 0
		case TFCRYPT_CIPHER_CAST5_CBC:
			cipher = EVP_cast5_cbc();
			break;
#endif
		case TFCRYPT_CIPHER_CAST5_CFB:
			cipher = EVP_cast5_cfb();
			break;
#if 0
		case TFCRYPT_CIPHER_AES128_CBC:
			cipher = EVP_aes_128_cbc();
			break;
#endif
		case TFCRYPT_CIPHER_AES128_CFB:
			cipher = EVP_aes_128_cfb();
			break;

		case TFCRYPT_CIPHER_AES128_OFB:
			cipher = EVP_aes_128_ofb();
			break;
#if 0
		case TFCRYPT_CIPHER_AES192_CBC:
			cipher = EVP_aes_192_cbc();
			break;
#endif

		case TFCRYPT_CIPHER_AES192_CFB:
			cipher = EVP_aes_192_cfb();
			break;

		case TFCRYPT_CIPHER_AES192_OFB:
			cipher = EVP_aes_192_ofb();
			break;
#if 0
		case TFCRYPT_CIPHER_AES256_CBC:
			cipher = EVP_aes_256_cbc();
			break;
#endif

		case TFCRYPT_CIPHER_AES256_CFB:
			cipher = EVP_aes_256_cfb();
			break;

		case TFCRTPY_CIPHER_AES256_OFB:
			cipher = EVP_aes_256_ofb();
			break;
#if 0
		case TFCRYPT_CIPHER_BF_CBC:
			cipher = EVP_bf_cbc();
			break;
#endif
		case TFCRYPT_CIPHER_BF_CFB:
			cipher = EVP_bf_cfb();
			break;

		case TFCRYPT_CIPHER_BF_OFB:
			cipher = EVP_bf_ofb();
			break;

		case TFCRYPT_CIPHER_DES_CFB:
			cipher = EVP_des_cfb();
			break;

		case TFCRYPT_CIPHER_DES_OFB:
			cipher = EVP_des_ofb();
			break;

		case TFCRYPT_CIPHER_DES3_CFB:
			cipher = EVP_des_ede3_cfb();
			break;

		case TFCRYPT_CIPHER_DES3_OFB:
			cipher = EVP_des_ede3_ofb();
			break;

		default:
			cipher = EVP_enc_null();  /* does nothing */
			break;
	}

	info->cipher = cipher;
	return 1; /* 正常 */
}

