/**
 * $Id$
 *
 * util_db.c
 *
 * TeamFileモジュールがDBプロバイダにアクセスするのに必要な手続き
 * (関数、構造体)を定義する。
 * 永続/一時プールマネージャの実施もここで行われます。
 *
 * (note)
 * 	ここで定義された関数や構造体をDBプロバイダが使用してはなりません。
 * 	使用してもよいのはTeamFileモジュールだけです。
 * 	そのような用途があるのなら、それはtf_db.hとtf_db.c に定義すべきです。
 * 	例えここに定義しないことで余計なレイヤが入ろうとも、この原理原則は
 * 	絶対です。
 *
 * 2004/02/08 Sun takehara NEW
 */
/* Apache header files */
#include "httpd.h"
#include "http_log.h"
#include "apr.h"
#include "apr_portable.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "apr_strings.h"
#include "apr_thread_proc.h"
#include "apr_thread_mutex.h"
#include "apr_thread_cond.h"
#include "apr_md5.h"

/* document management headers */
#include "mod_dav_tf.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "tf_rdbo_dbms.h"
#include "util.h"
#include "util_thread.h"
#include "util_common.h"
#include "tf_cset.h"
#include "tf_linkedlist.h"

APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define static values and macro
  ----------------------------------------------------------------------------*/
/*
 * デバッグ用フラグ
 * これを定義するとデバッグ出力されるようになります。
 */
//#define DIVY_DBCPOOL_DEBUG

/**
 * DBコネクションプール管理スレッドが動作する時間間隔[sec]
 * (note) 指定可能sec
 * 	0 < DB_CONNMGR_PROCESS_INTERVAL < apr_interval_time_t型の最大値
 * (note)
 * 	この時間毎に管理スレッドは動作します。但し、前の処理が
 * 	長引いた場合にはその限りではありません。
 * (note)
 * 	この時間を短くし過ぎると大量アクセスが発生したときにリクエストが
 * 	拒絶されてしまう可能性があります。管理スレッドによるロック間隔が
 * 	短くなることと関係しています。60[sec]以下にするのは危険です。
 */
#define DB_CONNMGR_PROCESS_INTERVAL 120

/**
 * １回のDBコネクションプールリフレッシュ作業でリフレッシュ対象
 * となるDBコネクションの割合(%)
 *
 * (note) 指定可能パーセント
 * 	0 < DB_CONNMGR_DBREFRESH_RATE < 100
 * 	0%の場合にはリフレッシュ作業が実施されません。
 *	100%の場合には、全てのアクティブなDBコネクションが対象になります。
 * (note)
 * 	プール中でアクティブなDBコネクションの数: navail
 * 	リフレッシュさせたいDBコネクションの数  : n_refresh
 *
 * 	DB_CONNMGR_DBREFRESH_RATE = (n_refresh / navail) * 100 (%)
 * 	但し、小数点第1位を切り上げします。
 */
#define DB_CONNMGR_DBREFRESH_RATE 10

/**
 * １回のDBコネクションアクティブチェックでチェック対象
 * となるDBコネクションプール中のDBコネクションの割合(%)
 *
 * (note) 指定可能パーセント
 * 	0 < DB_CONNMGR_CHKCONN_RATE < 100
 * 	0%の場合にはアクティブチェックが実施されません。
 *	100%の場合には、全てのアクティブなDBコネクションがチェック対象になる。
 * (note)
 * 	プール中でアクティブなDBコネクションの数  : navail
 * 	アクティブチェックしたいDBコネクションの数: n_chkactive
 *
 * 	DB_CONNMGR_CHKCONN_RATE = (n_chkactive / navail) * 100 (%)
 * 	但し、小数点第1位を切り上げします。
 */
#define DB_CONNMGR_CHKCONN_RATE 25

/**
 * DBプロバイダ接続情報に変化がないかどうかを検証する頻度 (回数)
 * メンテナンス作業が何回繰り返されたら検証動作を実視するのか決定する数
 * ということです。
 * (note) 指定可能回数
 * 	1 < DB_CONNMGR_CHKDBMSINFO_CYCLE < INT_MAX
 * 	この数が少ければ少い程、DBプロバイダのパフォーマンスを劣化させる
 * 	ことになります。DBMSの情報が変更されることは滅多にないと予想される
 * 	ので、数を減らしすぎないようにすべきです。
 * 	滅多にない作業に備えて通常パフォーマンスが劣化するのはナンセンスなので。
 */
#define DB_CONNMGR_CHKDBMSINFO_CYCLE 5

/**
 * DBコネクション管理スレッドが作業用に使用するapr_pool_t を
 * メンテナンス作業が何回繰り返されたらリフレッシュ(apr_pool_clear)する
 * のかを決定する数 (回数)
 * (note) 指定可能回数
 * 	1 < DB_CONNMGR_POOLREFRESH_RATE < INT_MAX
 * (note)
 * 	管理スレッドが作業用に使用するプールはr->pool のようにクリアされる
 * 	タイミングがないため、メモリを無尽蔵に消費してしまう可能性があります。
 * 	ただ、毎回クリアするとパフォーマンス劣化に繋がるため、定期的に
 * 	リフレッシュする目的でこの定数及び機構を設けました。
 */
#define DB_CONNMGR_POOLREFRESH_CYCLE 30

/**
 * デフォルトの最小保持コネクション数
 * ディレクティブが省略されたときに設定される値
 *
 * (note) 指定数
 * 	0 < DB_MIN_SPARECONN < INT_MAX
 */
#define DB_MIN_SPARECONN 2

/* デフォルト値を使用するかどうか判定するマクロ */
#define GET_MIN_SPARECONN(val)	\
		(((val) == DIVY_INT32_C_UNSET || (val) == 0) ? \
		 DB_MIN_SPARECONN : val)

/**
 * デフォルトの最大保持コネクション数
 * ディレクティブが省略されたときに設定される値
 *
 * (note) 指定数
 * 	0 < DB_MIN_SPARECONN <= DB_MAX_SPARECONN < INT_MAX
 */
#define DB_MAX_SPARECONN 5

/* デフォルト値を使用するかどうか判定するマクロ */
#define GET_MAX_SPARECONN(val)	\
		(((val) == DIVY_INT32_C_UNSET || (val) == 0) ? \
		 DB_MAX_SPARECONN : val)

/**
 * 一時プールが格納されているapr_pool_t* を取得するマクロ
 * (note)
 * 	一時プールはMainリクエストのr->pool に入っています
 * @param r request_rec *
 */
#define TEMPORARY_CPOOL_P(r)   ((r)->main ? (r)->main->pool : (r)->pool)

/**
 * 2つの数値のうち小さいほうを取得するマクロ
 */
#define MIN_VAL(a,b)	((a) < (b) ? (a) : (b))

/**
 * 2つの数値のうち大きいほうを取得するマクロ
 */
#define MAX_VAL(a,b)	((a) > (b) ? (a) : (b))

/* 拡張プロパティにアクセスするためのマクロ */
#define CPOOL_CTX(cn)	((cn)->dbcpoolctx)

/* 永続プールを利用するかどうかを判定するマクロ */
#define USING_CPOOL(flag)	((flag) == DIVY_DBPOOL_ON || (flag) == DIVY_DBPOOL_NOTHREAD)

/*------------------------------------------------------------------------------
  Define structures and enum
  ----------------------------------------------------------------------------*/
typedef enum __db_ConnMgrState		DbConnMgrState;
typedef enum __db_ConnPoolType		DbConnPoolType;
typedef enum __db_ConnState		DbConnState;
typedef enum __db_ProviderCntxtState	DbProviderCntxtState;

typedef struct __db_ProviderId		DbProviderId;
typedef struct __db_ProviderGrpCntxt	DbProviderGrpCntxt;
typedef struct __db_ProviderCntxt	DbProviderCntxt;
typedef struct __db_ConnPool		DbConnPool;
typedef struct __db_ConnList		DbConnList;

typedef struct __db_ConnMgrCntxt	DbConnMgrCntxt;
typedef struct __db_MaintenanceCntx	DbMaintenanceCntx;
typedef struct __db_ConnCreateList	DbConnCreateList;
typedef struct __db_DbmsInfo		DbDbmsInfo;

/**
 * DBコネクション管理スレッドの状態を表す値
 */
enum __db_ConnMgrState {
	DB_CONNMGR_ST_INIT = 0,		/* 初期状態(停止状態) */
	DB_CONNMGR_ST_STARTED,		/* 開始状態 */
	DB_CONNMGR_ST_TAKE1,		/* 1度起動に失敗した(停止状態) */
	DB_CONNMGR_ST_TAKE2,		/* 2度起動に失敗した(停止状態) */
	DB_CONNMGR_ST_ABNORMAL		/* 起動異常状態(停止状態) */
};

/**
 * DBコネクションプールの種類を表す値
 */
enum __db_ConnPoolType {
	DB_PTYPE_TEMPORARY_POOL = 0,	/* 一時(1次)プール */
	DB_PTYPE_PERMANENT_POOL		/* 永続(2次)プール */
};

/**
 * DbConn の状態(使用状況)を管理する列挙型
 */
enum __db_ConnState {
	DB_CONNSTAT_UNUSE = 0,	/* 未使用 */
	DB_CONNSTAT_INUSE,	/* 使用中 */
	DB_CONNSTAT_DISUSE	/* 使用不可 */
};

/**
 * DBプロバイダコンテキストの現在の状態を表す値
 */
enum __db_ProviderCntxtState {
	DB_PCNTXT_INIT = 0,		/* 初期状態             */
	DB_PCNTXT_ACTIVE,		/* 利用可能で有効な状態 */
	DB_PCNTXT_DESTROYED		/* 破棄されている状態   */
};


/**
 * [ 不完全型の実装 ]
 * DBコネクションプールの管理に必要な情報を表す構造体。
 * この構造体は、DbConnの中に保持されます。
 * (DbConnの拡張プロパティです。)
 */
struct __db_ConnPoolCntxt {
	/*
	 * 所属プールの種類
	 * (note) 存在意義
	 * 	この変数は、mutexロックなしで所属プールを知る目的のためだけに
	 * 	導入されました。本来であればproviderid だけで十分なのです。
	 */
	DbConnPoolType usepooltype;

	/*
	 * このコンテキストを持つコネクションの使用状況 (永続プール用)
	 */
	volatile DbConnState state;

	/*
	 * コネクション識別タグ (一時プール)
	 * (note) ctag のライフサイクル(終了)
	 * 	一時プールのライフサイクルが終わると例え永続プールに
	 * 	所属しているコネクションであってもctag はNULLになります。
	 */
	const char *ctag;

	/*
	 * コネクションが所属するDBプロバイダID (永続プール用)
	 */
	const DbProviderId *providerid;

	/*
	 * コネクションが生成された時刻 [nsec] (永続プール用)
	 */
	apr_time_t creationtm;
};

/**
 * DBプロバイダコンテキストを一意に識別するID
 * (DBプロバイダID)
 */
struct __db_ProviderId {
	/*
	 * DBプロバイダコンテキストが所属するDBプロバイダグループを
	 * 一意に識別するDBプロバイダグループID
	 */
	const char *provider_gid;

	/*
	 * DBプロバイダ名称
	 */
	const char *providerName;
};

/**
 * DBプロバイダ群(DBプロバイダグループ) を表す構造体
 * (note)
 * 	DBプロバイダグループとは、ある1つのリポジトリDBプロバイダと
 * 	それがdivy_dbms テーブルからローディングする複数のDBプロバイダから
 * 	構成される集合を指します。
 * 	この構造体は、DBプロバイダグループという概念が持つ全ての構成要素を
 * 	持つコンテナとしての役割を持っています。
 *
 * 	DBプロバイダグループにはそれぞれDBプロバイダグループIDが付加され
 * 	それによって他のDBプロバイダグループから一意に識別されます。
 * 	別のリポジトリDBプロバイダが別のディレクティブで定義されていれば
 * 	それも別のDBプロバイダグループを持っていることになります。
 */
struct __db_ProviderGrpCntxt {
	/*
	 * DBプロバイダグループを一意に識別するID
	 */
	const char *provider_gid;

	/*
	 * DBプロバイダグループを生成したリポジトリDBのプロバイダID
	 */
	const DbProviderId *repos_providerid;

	/*
	 * DBプロバイダグループの構成要素であるDBプロバイダの集合を
	 * 持つハッシュ。
	 * (note) 検索の高速化を図る目的で、DBMS識別名称をキーとして
	 *        DBプロバイダを保持しています。ハッシュという汎用コンテナに
	 *        格納することは本来望ましくないのですが仕方ありません。
	 *
	 * key : DBMS識別名称 (char *)
	 * val : DbProviderCntxt *
	 */
	apr_hash_t *dbprv_hash;

	/*
	 * 適切なライセンスを持っていて、DBプロバイダとしてロード可能な
	 * DBプロバイダの種類を持つSet。
	 *
	 * (note) dbprv_hash が表すDBプロバイダ集合との相違
	 * 	dbprv_hash に格納されているDBプロバイダの種類はこのハッシュにも
	 * 	必ず登録されています。
	 * 	dbprv_hash になく、このハッシュにあるDBプロバイダの種類は、
	 * 	"DBプロバイダとしてライセンス、実装上の観点からロードすることは
	 * 	可能であるが、接続情報が与えられなかったため現時点では使用できない"
	 * 	ことを表しています。
	 */
	apr_hash_t *loadable_type_set;

	/*
	 * DBプロバイダグループの初期化が終了しているかどうかを表す
	 * (1: 初期化済み / 0: 未初期化)
	 */
	int init_finished;
};

/**
 * ひとつのDBプロバイダが持つ情報をあらわすコンテキスト構造体
 * あるDBに接続するのに必要な情報を管理しています。
 * (note)
 * 	この構造体は、DBプロバイダグループ毎かつDBMS識別名称毎に
 * 	インスタンス化されます。
 */
struct __db_ProviderCntxt {
	/*
	 * DBプロバイダコンテキストを識別するID
	 */
	const DbProviderId *providerid;

	/*
	 * このコンテキストの状態
	 */
	DbProviderCntxtState pc_state;

	/*
	 * このコンテキストが表すDBプロバイダのデータソース
	 */
	DbDataSource *dbds;

	/*
	 * このコンテキストの永続プール
	 * (note)
	 * 	このDBプロバイダコンテキストが永続プールを使用しないのであれば
	 * 	cpool は常にNULLになります。
	 */
	DbConnPool *cpool;

	/*
	 * DBプロバイダの接続先情報を識別する文字列
	 */
	const char *conn_digest;

	/*
	 * このコンテキストがリポジトリDBプロバイダのものかどうか
	 * 1: リポジトリである / 0: リポジトリではない
	 */
	int is_reposdb;
};

/**
 * DB接続オブジェクトを格納管理する構造体
 */
struct __db_ConnPool {
	/* DBコネクションプールの種類 */
	DbConnPoolType pooltype;

	/* DbConn からなるリスト */
	DbConnList *dbconnlist;

	apr_int32_t nelts;	/* プール内の総数 */
	apr_int32_t navail;	/* 現在使用可能なDbConnの数 */
	apr_int32_t nmax;	/* 最大数 */
	apr_int32_t nmin;	/* 最小数 */
};

/**
 * DbConn の双方向リスト
 */
struct __db_ConnList {
	DbConn *dbconn;

	DbConnList *next;
	DbConnList *prev;
};


/**
 * DBコネクション管理スレッドに渡されるデータを保持する構造体
 */
struct __db_ConnMgrCntxt {
	apr_pool_t *owner_p;	/* 管理スレッドとライフサイクルを共にするプール */
	apr_threadattr_t *attr;	/* スレッド属性 */
};

/**
 * DBコネクション管理スレッドがメンテナンスすべきDbConnの集合を保持する構造体
 */
struct __db_MaintenanceCntx {

	/* 新規作成すべきDbConnのリスト */
	DbConnCreateList *create_list;

	/* 破棄すべきDbConnのリスト     */
	DbConnList *delete_list;

	/* アクティブチェックすべきDbConnのリスト */
	DbConnList *actchk_list;

	/*
	 * DBMS接続情報の確認に使用するハッシュ
	 * key: _create_db_provider_gid(p,r) で生成したプロバイダグループID
	 * val: DbDbmsInfo *
	 */
	apr_hash_t *dbmsinfo_hash;

	/* 今迄にメンテナンス処理を実施した回数 */
	unsigned int exec_cnt;
};

/**
 * 新規作成すべきDbConnの集合を保持する構造体
 * (note)
 * 	リストの１つの要素は、あるDBMSのDbConnを作成するのに必要な
 * 	情報を保持しています。
 * 	DBコネクション管理スレッドが使用します。
 * 	それ以外の用途には適していません。
 */
struct __db_ConnCreateList {
	apr_int32_t n_create;	/* 生成するコネクションの数 */
	DbDataSource *dbds;	/* 生成するコネクションのデータソース */
	const DbProviderId *providerid;	/* 生成するコネクションのDBプロバイダID */
	DbConnList *cnlist;	/* 上記の情報を使って生成したDbConnのリスト */

	DbConnCreateList *next;
};

/**
 * DBMS接続情報の変更チェックに使用する情報を持つ構造体
 * (note)
 * 	この構造体は、DBコネクション管理スレッドが使用します。
 * 	それ以外の用途には適していません。使わないこと。
 */
struct __db_DbmsInfo {
	/*
	 * リポジトリDBプロバイダのDbConnを1つ持つリスト
	 */
	DbConnList *repos_alist;

	/*
	 * ダイジェスト値とDBMS情報を持つハッシュ
	 * key: ダイジェスト値
	 * val: divy_rdbo_dbms *
	 */
	apr_hash_t *digest_hash;
};

/**
 * [ 不完全型の定義 ]
 */
struct __divy_db_bind_ctx {
	/* 以下入力値 */
	int unitnum;		/* 単位数 */
	divy_cset_t *src_set;	/* 文字列集合へのポインタ */

	apr_pool_t *p;

	/* src_set の要素をリスト上に並べたもの */
	divy_linkedlist_t *src_list;
};

/**
 * [ 不完全型の定義 ]
 */
struct __divy_db_bind_ctx_idx {
	divy_db_bind_ctx *bindctx;
	int num;
	divy_linkedlist_t *part_list;
	const char *bindstr;
};


/*------------------------------------------------------------------------------
  Define Global values
  ----------------------------------------------------------------------------*/
/*
 * DBプロバイダ管理コンテキスト専用のプール
 * (note)
 * 	pconf (process->pconf) のサブプールとなります。
 */
static apr_pool_t *prvmng_p = NULL;

/**
 * DB コネクションプール(永続プール)環境が初期化されているかどうか
 * 0: 未初期化 / 1: 初期化済み
 */
static volatile int is_initial = 0;

/**
 * DBコネクション管理スレッドが終了すべきかどうか決定するフラグ
 * 1: 終了すべき / 0: 終了しない
 */
static volatile int exit_connmgr_th = 0;

/**
 * DBプロバイダグループを記録するハッシュ
 * (note) 存在意義
 * 	DBプロバイダグループの検索の高速化を図る目的で導入されました。
 * 	本来であれば、DBプロバイダグループをこのような見通しの悪い汎用
 * 	コンテナ(ハッシュ) に格納すべきではありません。
 *
 * key: _create_db_provider_gid(p,r) で生成したプロバイダグループID
 * val: DbProviderGrpCntxt *
 */
static apr_hash_t *dbprv_grp_hash = NULL;

/**
 * DB コネクションリフレッシュ間隔[microsec]
 */
static apr_time_t dbrefreshinterval = DIVY_DBCPOOL_REFRESH_INTERVAL;

/**
 * DB コネクション管理スレッド
 */
static apr_thread_t *connmgr_th = NULL;

/**
 * DB コネクション管理スレッドの起動ステータス
 */
static DbConnMgrState connmgr_state = DB_CONNMGR_ST_INIT;

/**
 * DB コネクションプール(永続プール)に対するスレッド
 * アクセスをシリアルにする目的で使用するmutex
 */
static apr_thread_mutex_t *cpool_mutex = NULL;

/**
 * DBコネクション管理スレッドの監視間隔と停止動作を
 * 制御するために使用する条件変数
 */
static apr_thread_cond_t *cpool_cond = NULL;

/*------------------------------------------------------------------------------
  Declare private prototype function
  ----------------------------------------------------------------------------*/
static const DbProviderId * _create_db_providerid(apr_pool_t *p,
					const char *provider_gid,
					const char *providerName);
static const DbProviderId * _clone_db_providerid(apr_pool_t *p,
					const DbProviderId *providerid);
static const char * _create_db_provider_gid(apr_pool_t *p, request_rec *r);
static DbDataSource * _lookup_db_provider(request_rec *r,
					const DbProviderId *providerid);
static RET_STAT _init_db_providers(request_rec *r,
					const char *provider_gid);
static RET_STAT _register_db_provider(DbProviderGrpCntxt *grp_ctx,
					DbProviderCntxt *p_ctx);
static int _is_dbenv_ready(const char *provider_gid);
static int _validate_reposdb_config(apr_pool_t *p, dav_divy_dir_conf *dconf);
static RET_STAT _has_license(apr_pool_t *p,
					dav_divy_server_conf *sconf,
					const char *providerType);
static DbConnList * _create_DbConnList(apr_pool_t *p, DbConn *dbconn);
static void _append_DbConnList(DbConnList *l1, DbConnList *l2);
static void _append_DbConnList_to_DbConnPool(DbConnPool *cpool,
					DbConnList *lelts);
static DbConnList * _remove_DbConnList_from_DbConnPool(DbConnPool *cpool,
					DbConnList *lelts);
static DbConnPoolCntxt * _create_cpoolctx(apr_pool_t *p,
					const DbProviderId *providerid,
					DbConnPoolType usepooltype);
static DbProviderGrpCntxt * _create_prvgrp_ctx(apr_pool_t *p,
					const char *provider_gid,
					const DbProviderId *repos_providerid);
static DbProviderGrpCntxt * _get_prvgrp_ctx(const char *provider_gid);
static RET_STAT _register_prvgrp_ctx(DbProviderGrpCntxt *grp_ctx);
static DbProviderCntxt * _create_prv_ctx(apr_pool_t *p,
					const DbProviderId *providerid,
					int is_reposdb,
					DbDataSource *dbds);
static DbProviderCntxt * _get_prv_ctx(const DbProviderId *providerid);
#define DIVY_DB_DATA_DBDS 1	/* DbDataSource   へのポインタ */
#define DIVY_DB_DATA_DBMS 2	/* divy_rdbo_dbms へのポインタ */
static const char * _create_conn_digest(apr_pool_t *p,
					int kind_of_data, void *data);
static apr_status_t _cleanup_used_dbconn(void *ctx);
static void _close_dbconn(DbConn *dbconn);
static void _clear_globalval(void);
/*-----------------------------------------------------------------------------
 * Declare "Temporary DbConn Pool" functions
 -----------------------------------------------------------------------------*/
static DbConn * _trygetDbConn_from_temp_DbConnPool(apr_pool_t *p,
						const char *ctag,
						const DbProviderId *providerid);
static void _store_DbConn_to_temp_DbConnPool(apr_pool_t *p,
						const char *ctag,
						const DbProviderId *providerid,
						DbConn *dbconn);
static DbConnPool * _create_temp_DbConnPool(apr_pool_t *p);
static DbConnPool * _lookup_temp_DbConnPool(apr_pool_t *p,
						const DbProviderId *providerid);
static RET_STAT _register_temp_DbConnPool(apr_pool_t *p,
						const DbProviderId *providerid,
						DbConnPool *cpool);
/*-----------------------------------------------------------------------------
 * Declare "Permanent DbConn Pool" functions
 -----------------------------------------------------------------------------*/
static DbConn * _trygetDbConn_from_perm_DbConnPool(apr_pool_t *p,
						const DbProviderId *providerid);
static void _restore_DbConn_to_perm_DbConnPool(DbConn *dbconn);
static DbConnPool * _create_perm_DbConnPool(DbDataSource *dbds);
static DbConnPool * _lookup_perm_DbConnPool(const DbProviderId *providerid);
static RET_STAT _register_perm_DbConnPool(const DbProviderId *providerid,
						DbConnPool *cpool);
static void _destroy_all_perm_DbConnPool(void);
static RET_STAT _execute_connnmgr(apr_pool_t *owner_p);
/*-----------------------------------------------------------------------------
 * Declare DbConn manager functions
 -----------------------------------------------------------------------------*/
static void * APR_THREAD_FUNC _run_connmgr(apr_thread_t *thd, void *data);
#define DB_MAINTENANCE_NEED   0	/* メンテナンスが必要である */
#define DB_MAINTENANCE_NONEED 1	/* メンテナンス不要 */
#define DB_MAINTENANCE_ERR    2	/* エラーが発生した */
static int _find_maintenance(apr_pool_t *p, DbMaintenanceCntx *mctx);
static void _do_maintenance(apr_pool_t *p, DbMaintenanceCntx *mctx);
static void _do_maintenance_with_lock(apr_pool_t *p, DbMaintenanceCntx *mctx);
static void _restore_maintenance(apr_pool_t *p, DbMaintenanceCntx *mctx);
static void _cleanup_thread_mutex(void *mutex);

/*-----------------------------------------------------------------------------
 * Declare utility private functions
 -----------------------------------------------------------------------------*/
static int _create_transaction_ctx(request_rec *r,
					DbConn *dbconn,
					divy_db_transaction_ctx **ts_ctx);
static int _devide_list(divy_linkedlist_t **src, divy_linkedlist_t **dst, int unitnum);
static char * _make_bindstr(apr_pool_t *p, int num);

/*------------------------------------------------------------------------------
  Define Public Functions
  ----------------------------------------------------------------------------*/
/**
 * DBプロバイダ管理コンテキストの初期化する。
 * この関数は複数のスレッドからアクセスされてはなりません。
 * この関数を複数回呼び出さないようコール元で保証して下さい。
 * (note)
 * 	この関数は、DBコネクションを使用するどの全ての関数よりも先に呼び出される
 * 	必要があります。Child_initステージが適切です。
 * 	この関数は、１度コールされると、destroy_dbprvmng_env がコールされるまで
 * 	何も実行しなくなります。
 */
DIVY_DECLARE(void) init_dbprvmng_env(apr_pool_t *pchild, server_rec *s)
{
	apr_status_t rv;
	dav_divy_server_conf *sconf;

	TRACE_S(s);

	/* 既に初期化されていたら何もしない */
	if (prvmng_p && is_initial) return;

	/*
	 * DBプロバイダ管理コンテキストが使用するプールの生成
	 *
	 * (note) pchild ではなくpconfのサブプールにする訳
	 * 	Apacheのchildプロセスがexitするとき、pchild以下のサブプールは
	 * 	深さ優先検索されて最も下位層にあるものから順に親に向かって
	 * 	破棄されていきます。最も用途に適しているように見えるのですが、
	 * 	今回はサブプールとその親プールの破棄タイミングを同一にする
	 * 	必要が生じました。これは、管理スレッドの待ち合わせがあったため
	 * 	です。pchild のサブプールでは、非常に繊細な終了フェーズにおいて
	 * 	しばしば問題を引き起こしました。
	 * 	(pchildが先に破棄されたが、スレッドがとまらないなど)
	 * 	仕方がないので、ApacheのChildプロセスのdestroy対象にならない
	 * 	pconfから確保することにしました。破棄のタイミングを自分で制御
	 * 	するためです。
	 */
	apr_pool_create(&prvmng_p, s->process->pconf);
	apr_pool_tag(prvmng_p, "tf_dbcpool_main");	/* タグ名を付ける */

	/*
	 * mutex の生成
	 * (note) nest 出来ないmutex を生成する
	 */
	rv = apr_thread_mutex_create(&cpool_mutex,
					APR_THREAD_MUTEX_UNNESTED, prvmng_p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to create cpool_mutex. (code = %d)", rv);
		is_initial = 0;	/* 初期化が完了していない */
		return;
	}

	/*
	 * 条件変数の生成
	 */
	rv = apr_thread_cond_create(&cpool_cond, prvmng_p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to create cpool_cond. (code = %d)", rv);
		is_initial = 0;
		/* mutex を破棄する */
		if (cpool_mutex) {
			(void) apr_thread_mutex_destroy(cpool_mutex);
			cpool_mutex = NULL;
		}
		return;
	}

	/*
	 * DB プロバイダ管理コンテキストの生成と初期化
	 */
	exit_connmgr_th = 0;
	dbprv_grp_hash  = apr_hash_make(prvmng_p);
	connmgr_th      = NULL;		/* 必要になったら作る */
	connmgr_state   = DB_CONNMGR_ST_INIT;

	sconf = dav_divy_get_server_config(s);
	if (sconf) {
		if (sconf->dbrefreshinterval == 0 ||
		    sconf->dbrefreshinterval == DIVY_INT32_C_UNSET) {
			/* デフォルト値を使用する */
			dbrefreshinterval =
				apr_time_from_sec(DIVY_DBCPOOL_REFRESH_INTERVAL);
		}
		else {
			dbrefreshinterval =
				apr_time_from_sec(sconf->dbrefreshinterval);
		}
	}
	else {
		/* デフォルト値を使用する */
		dbrefreshinterval =
			apr_time_from_sec(DIVY_DBCPOOL_REFRESH_INTERVAL);
	}

	is_initial = 1;	/* 初期化が完了した */
}

/**
 * DBプロバイダ管理コンテキストの終了処理
 * (note)
 * 	この関数を複数回呼び出さないようコール元で保証して下さい。
 */
DIVY_DECLARE(void) destroy_dbprvmng_env(void)
{
	apr_status_t rv;
	int ret;

	if (!is_initial) return;	/* 初期化されていなければ以下は不要 */

	/*
	 * 管理スレッドに終了を命じる
	 */
	if (connmgr_th) {
		/* 停止フラグを立てる */
		exit_connmgr_th = 1;

		/* 取り消しを要求する */
		ret = dav_divy_thread_cancel(connmgr_th);
		if (ret) {
			ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to cancel the connection manager thread."
				"(code = %d)", ret);
		}

		/*
		 * スレッドの終了を待つ(join)
		 */
		(void) apr_thread_join(&rv, connmgr_th);
#ifdef DIVY_DBCPOOL_DEBUG
		ERRLOG1(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"join thread. (%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DBCPOOL_DEBUG */
	}

	/* mutex の破棄 */
	if (cpool_mutex) {
		apr_thread_mutex_destroy(cpool_mutex);
	}

	/* 条件変数の破棄 */
	if (cpool_cond) {
		apr_thread_cond_destroy(cpool_cond);
	}

	/* 永続プールの全DBコネクションを解放し、プールを破棄する */
	_destroy_all_perm_DbConnPool();

	/* サブプールの破棄 */
	if (prvmng_p) {
		apr_pool_destroy(prvmng_p);
	}

	/* 全てのグローバル変数を初期化しておく */
	_clear_globalval();
}


/**
 * r->server のvirtual host に存在するDB プロバイダの中から、providerNmae が示す
 * DB プロバイダ構造体(DbDataSource)取得して返却する。
 *
 */
DIVY_DECLARE(DbDataSource *) lookup_db_provider(request_rec *r,
						const char *providerName)
{
	const char *provider_gid = _create_db_provider_gid(r->pool, r);

	if (IS_EMPTY(providerName)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The providerName is NULL. ");
		return NULL;
	}

	/* DbDataSource * の取得 */
	return _lookup_db_provider(r,
		_create_db_providerid(r->pool, provider_gid, providerName));
}

/**
 * r->server のvirtual host に存在するDB プロバイダの中から、リポジトリDBの
 * DB プロバイダ構造体(DbDataSource)取得して返却する。
 *
 */
DIVY_DECLARE(DbDataSource *) lookup_reposdb_provider(request_rec *r)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	const char *provider_gid = _create_db_provider_gid(r->pool, r);

	/* DbDataSource * の取得 */
	return _lookup_db_provider(r,
		_create_db_providerid(r->pool, provider_gid, conf->dbmsname));
}


/**
 * 指定したproviderType が示すDBプロバイダ種類がリポジトリDBプロバイダとして
 * 登録されているかどうか。
 *
 */
DIVY_DECLARE(int) is_reposdb_provider(request_rec *r, const char *providerType)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	
	/* conf の検証 */
	if (_validate_reposdb_config(r->pool, conf)) return DB_FALSE;
	
	if (strcmp(conf->dbmstype, providerType) == 0) {
		return DB_TRUE;
	}
	else {
		return DB_FALSE;
	}
}

/**
 * リポジトリDBとの接続がアクティブなDbConn構造体を取得する。
 * この関数は、post_read_request以降からリクエストプールが破棄される
 * 直前までのステージの中で使用することが出来ます。
 * 
 */
DIVY_DECLARE(void) lookup_reposdb_DbConn(request_rec *r, DbConn **dbconn)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);

	return lookup_tagged_DbConn(r, NULL, conf->dbmsname, dbconn);
}

/**
 * 指定されたproviderNameが示すDBプロバイダからアクティブなDbConn を取得する。
 *
 * この関数は、リクエストプールが破棄される直前までのステージの中で
 * 使用することが出来ます。
 * (note) !!注意!!
 * 	この関数を使って取得したDbConnのclose関数はコールしてはなりません。
 * 	closeのコールは、リクエストプールが破棄される際、自動的に行われます。
 *
 * @param request_rec *r request_rec構造体へのポインタ
 * @param providerName const char * DBプロバイダの名称
 * @param dbconn DbConn ** 取得したDbConnへのポインタ
 * 			何らかの原因でDbConnが取得できない時にはNULLを返却
 */
DIVY_DECLARE(void) lookup_cached_DbConn(request_rec *r,
					const char *providerName, DbConn **dbconn)
{
	return lookup_tagged_DbConn(r, NULL, providerName, dbconn);
}

/**
 * 指定されたDbConn識別タグctag が示すリポジトリDBとの接続がアクティブな
 * DbConn構造体を取得する。
 * ptag が付いたDbConnが無ければ新たにDbConnを取得して返却します。
 */
DIVY_DECLARE(void) lookup_tagged_DbConn(request_rec *r, const char *ctag,
					const char *providerName, DbConn **dbconn)
{
	DbDataSource *dbds = NULL;
	apr_pool_t *p      = NULL;
	const char *provider_gid;
	const DbProviderId *providerid;

	*dbconn = NULL;	/* 初期化 */

	if (IS_EMPTY(providerName)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The providerName is EMPTY.");
		return;
	}

	/* 一時プールが格納されているapr_pool_t を取得する */
	p = TEMPORARY_CPOOL_P(r);

	/* プロバイダID の生成 */
	provider_gid = _create_db_provider_gid(p, r);
	providerid   = _create_db_providerid(p, provider_gid, providerName);

	/*
	 * 一時プール(1次プール)からDbConnを探す
	 */
	*dbconn = _trygetDbConn_from_temp_DbConnPool(p, ctag, providerid);

	/* (note) 1次プールにコネクションがある間は、dbdsが破棄されても
	 *        放置します。つまり破棄チェックはしないということです。*/
	if (*dbconn) return;	/* キャッシュ値を利用する */

	/*
	 * DbConn がキャッシュされていなかったので、新規作成も視野に入れて
	 * DBプロバイダを探す
	 */
	dbds = _lookup_db_provider(r, providerid);
	if (dbds == NULL || dbds->isFinishedInitEnv != DBENV_INIT_FINISHED) {

		/* 初期化に失敗した場合 */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"The database-provider was not initialized."
			"(providerName = %s)", providerName);
		*dbconn = NULL;
		return;	/* 諦める */
	}

	/* 永続プール(2次プール)を利用する場合、2次プールにあるかどうか調べる */
	if (USING_CPOOL(dbds->dbpool)) {
		/* (note) 第1引数のp はprvmng_pでは駄目. */
		*dbconn = _trygetDbConn_from_perm_DbConnPool(p, providerid);
		if (*dbconn) {

			/* 永続プールに格納されていた *dbconn を一時プールに登録する */
			_store_DbConn_to_temp_DbConnPool(p, ctag,
							providerid, *dbconn);
			return;
		}
	}

	/*
	 * 1次、2次プールに無かったので新しいDbConnを取得する
	 */
	if (USING_CPOOL(dbds->dbpool)) {
		*dbconn = dbds->getDbConn(dbds, prvmng_p);
	}
	else {
		*dbconn = dbds->getDbConn(dbds, p);
	}
	if ((*dbconn)->getCode(*dbconn) != DB_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get DbConn. (providerName = %s) "
			"Reason: %s", providerName,
			REPLACE_NULL((*dbconn)->getMsg(*dbconn)));
		if (*dbconn != NULL) (*dbconn)->close(*dbconn);
		*dbconn = NULL;
		return;
	}

	/* dbconn にDBコネクションプール管理用コンテキストを付加する */
	if (USING_CPOOL(dbds->dbpool)) {
		CPOOL_CTX(*dbconn) = _create_cpoolctx(prvmng_p, providerid,
							DB_PTYPE_PERMANENT_POOL);
	}
	else {
		CPOOL_CTX(*dbconn) = _create_cpoolctx(p, providerid,
							DB_PTYPE_TEMPORARY_POOL);
	}

	/* *dbconn を一時プールに登録する */
	_store_DbConn_to_temp_DbConnPool(p, ctag, providerid, *dbconn);

	return;
}

/**
 * リポジトリDBのDBMS接続情報をディレクトリコンフィグから取り出し
 * DB プロバイダの環境を初期化する。
 * (note)
 * 	テスト用。
 *
 */
DIVY_DECLARE(RET_STAT) init_reposdb_provider(request_rec *r, DbDataSource *repos)
{
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	
	/* リポジトリDBを初期化する */
	repos->isFinishedInitEnv = DBENV_INIT_FINISHED;
	repos->dbmstype          = apr_pstrdup(prvmng_p, conf->dbmstype);
	repos->dbmsname          = apr_pstrdup(prvmng_p, conf->dbmsname);
	repos->hostname          = apr_pstrdup(prvmng_p, conf->hostname);
	repos->hostport          = apr_pstrdup(prvmng_p, conf->hostport);
	repos->dbname            = apr_pstrdup(prvmng_p, conf->dbname);
	repos->username          = apr_pstrdup(prvmng_p, conf->username);
	repos->password          = apr_pstrdup(prvmng_p, conf->password);
	repos->dbpool            = conf->dbpool;
	repos->dbminspareconn    = GET_MIN_SPARECONN(conf->dbminspareconn);
	repos->dbmaxspareconn    = GET_MAX_SPARECONN(conf->dbmaxspareconn);

	return DB_SUCCESS;
}

/**
 * dbconn が使用不可であるとマークする。
 *
 * @param dbconn DbConn * マーク対象のDbConn
 */
DIVY_DECLARE(void) mark_inactive_dbconn(DbConn *dbconn)
{
	CPOOL_CTX(dbconn)->state = DB_CONNSTAT_DISUSE;
}

/**
 * 指定されたproviderType のDBプロバイダにライセンスがあるかどうかを判定する.
 *
 */
DIVY_DECLARE(RET_STAT) divy_db_has_license(apr_pool_t *p, server_rec *s,
		 				const char *providerType)
{
	return _has_license(p, dav_divy_get_server_config(s), providerType);
}

/**
 * ライセンスがあって使用可能な(ロードされている)DBプロバイダの種類一覧を返却する。
 * (note)
 * 	この関数はmutex を使用します。他のmutexロックスコープの中から
 * 	コールしてはなりません。
 *
 */
DIVY_DECLARE(RET_STAT) divy_db_get_licensed_dbmstype(request_rec *r,
						apr_hash_t **type_set)
{
	apr_status_t rv;
	apr_pool_t *p           = r->pool;
	const char *provider_gid;
	DbProviderGrpCntxt *grp_ctx;

	*type_set = NULL;	/* 初期化 */

	/* プロバイダグループIDの取得 */
	provider_gid = _create_db_provider_gid(p, r);

	/* mutex ロック */
	rv = apr_thread_mutex_lock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex lock.(code = %d)", rv);
		return DB_ERROR;
	}

	/* DBプロバイダグループコンテキストの取得 */
	grp_ctx = _get_prvgrp_ctx(provider_gid);
	if (grp_ctx && grp_ctx->loadable_type_set &&
			apr_hash_count(grp_ctx->loadable_type_set) > 0) {
		/* DBMS種類一覧をコピーする */
		*type_set = apr_hash_copy(p, grp_ctx->loadable_type_set);
	}

	/* mutex ロックを解除 */
	rv = apr_thread_mutex_unlock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex unlock. (code = %d)", rv);
	}

	return DB_SUCCESS;
}

/**
 * リポジトリDBのDBMS識別名称を生成して返却する。
 * この関数によって得られる名称が同じになるのは次の条件を全て満たす
 * 場合のみとなります。
 *  ・Location パス名称が等しいこと
 *  ・DB接続情報が等しいこと
 *  ・DBコネクションプール設定が等しいこと
 */
DIVY_DECLARE(char *) divy_db_build_reposdbmsname(apr_pool_t *p,
						dav_divy_dir_conf *dconf)
{
	if (p == NULL || dconf == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"apr_pool_t or dav_divy_dir_conf is NULL.");
		return NULL;
	}

	return apr_psprintf(p,  "%s:%s:%s:%s:%s:%s:%d:%d:%d@%s",
				dconf->dbmstype, dconf->hostname, dconf->hostport,
				dconf->dbname, dconf->username, dconf->password,
				dconf->dbpool,
				GET_MIN_SPARECONN(dconf->dbminspareconn),
				GET_MAX_SPARECONN(dconf->dbmaxspareconn),
				dconf->root_uri);
}

/**
 * トランザクションコンテキストを生成して、トランザクション開始前の状態に
 * 設定して渡す。
 */
DIVY_DECLARE(int) divy_db_create_transaction_ctx(request_rec *r, 
                                     divy_db_transaction_ctx **ts_ctx)
{
	DbConn *dbconn = NULL;

	/* 第2引数にNULLがしていされたら? (It may be bug..) */
	if (ts_ctx == NULL) return 1;	/* 何もできません */

	/* キャッシュされたリポジトリDB のDbConn を取得する */
	lookup_reposdb_DbConn(r, &dbconn);
	if (dbconn == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get DbConn using transaction ctx.");
		*ts_ctx = NULL;
		return 1;
	}

	/* トランザクションコンテキストの作成 */
	_create_transaction_ctx(r, dbconn, ts_ctx);

	return 0;
}

/**
 * トランザクションコンテキストを生成して、新しいDbConn を作成し、
 * トランザクション開始前の状態に設定して渡す。
 * (note)
 * 	ここで作成されたDBコネクションはこの関数を呼び出したスレッドが
 * 	１リクエストの間だけ"独占的"に使用することが保証されています。
 * 	但し、この関数から得られたts_ctx を破棄または忘れてしまうと
 * 	同じDBコネクションは２度と取得出来ません。
 * 	独占的に使用したいのであれば、ts_ctx を１リクエストの間、ずっと
 * 	保持するよう呼び出し側で対応して下さい。
 */
DIVY_DECLARE(int) divy_db_create_transaction_ctx_new(request_rec *r, 
					divy_db_transaction_ctx **ts_ctx)
{
	DbConn *dbconn     = NULL;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	const char *ctag;

	/* 第2引数にNULLがしていされたら? (It may be bug..) */
	if (ts_ctx == NULL) return 1;	/* 何もできません */

	*ts_ctx = NULL;		/* 初期化 */

	/*
	 * 現在時刻をctagとしたDbConnを取得する
	 */
	ctag = apr_psprintf(r->pool, "%"APR_TIME_T_FMT, apr_time_now());

	lookup_tagged_DbConn(r, ctag, conf->dbmsname, &dbconn);
	if (dbconn == NULL) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to tagged repository db connection.(repos = %s)",
			conf->dbmsname);
		return 1;	/* 失敗 */
	}

	/* トランザクションコンテキストの作成 */
	_create_transaction_ctx(r, dbconn, ts_ctx);

	return 0;
}

/**
 * 指定されたdbconn を持つトランザクションコンテキストを生成して
 * トランザクション開始前の状態に設定して渡す。
 *
 */
DIVY_DECLARE(int) divy_db_create_transaction_ctx_by_dbconn(request_rec *r,
					DbConn *dbconn,
					divy_db_transaction_ctx **ts_ctx)
{
	return _create_transaction_ctx(r, dbconn, ts_ctx);
}

/**
 * 関数divy_db_create_transaction_ctxまたは 
 * divy_db_create_transaction_ctx_newによって生成されたトランザクション
 * コンテキストのトランザクションを開始する。
 * (note)
 * 	既にトランザクションが開始していれば何もしません。
 */
DIVY_DECLARE(int) divy_db_start_transaction(divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn;
	if (ts_ctx == NULL) return 1;

	dbconn = ts_ctx->dbconn;

	/* トランザクションが開始していないときのみ開始する */
	if (!(ts_ctx->status & DIVY_TRANS_START)) {
		/* 開始 */
		dbconn->startTrans(dbconn, 0);
		if (dbconn->getCode(dbconn) != DB_SUCCESS) {
			ERRLOG2(ts_ctx->p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to start transaction. (providerName = %s) "
				"Reason: %s", dbconn->dbds->dbmsname,
				REPLACE_NULL(dbconn->getMsg(dbconn)));

			/* コネクションが壊れているとマークしておく */
			mark_inactive_dbconn(dbconn);
			return 1;
		}

		ts_ctx->status |= DIVY_TRANS_START;
	}

	return 0;
}

/**
 * 関数divy_db_create_transaction_ctxまたは 
 * divy_db_create_transaction_ctx_newによって開始されたトランザクションを
 * 確定する。
 */
DIVY_DECLARE(int) divy_db_commit_transaction(divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn = NULL;

	/* これは恐らく正しい状態ではないはず */
	if (ts_ctx == NULL) return 1;

	/* 既に終わっている or 開始されていない
	 * トランザクションに対しては何もしない */
	if (!(ts_ctx->status & DIVY_TRANS_START)) return 0;
	if (ts_ctx->status & DIVY_TRANS_END) return 0;

	/* 少なくとも、DB のコミットはできないのでやめる */
	if (ts_ctx->dbconn == NULL) return 0;
	dbconn = ts_ctx->dbconn;

	/* トランザクションを確定する */
	dbconn->commit(dbconn);
	if (dbconn->getCode(dbconn) != DB_SUCCESS) {
		ERRLOG2(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to commit transaction. code = %d msg = %s", 
			dbconn->getCode(dbconn), dbconn->getMsg(dbconn));
		ts_ctx->status |= DIVY_TRANS_ABORT;

		/* コミットに失敗したDBコネクションは使用不可とすべき */
		mark_inactive_dbconn(dbconn);
		return 1;
	}

	/* トランザクションが完了したことを記録する */
	ts_ctx->status |= DIVY_TRANS_END;
	return 0;
}

/**
 * 関数divy_db_create_transaction_ctxまたは 
 * divy_db_create_transaction_ctx_newによって開始されたトランザクションを
 * 破棄する。
 *
 */
DIVY_DECLARE(int) divy_db_rollback_transaction(divy_db_transaction_ctx *ts_ctx)
{
	DbConn *dbconn = NULL;

	/* これは恐らく正しい状態ではないはず */
	if (ts_ctx == NULL) return 1;

	/* 既に終わっている or 開始されていない
	 * トランザクションに対しては何もしない */
	if (!(ts_ctx->status & DIVY_TRANS_START)) return 0;
	if (ts_ctx->status & DIVY_TRANS_END) return 0;

	/* 少なくとも、DB のrollback はできないのでやめる */
	if (ts_ctx->dbconn == NULL) return 0;
	dbconn = ts_ctx->dbconn;

	/* トランザクションを破棄して変更を元に戻す */
	dbconn->rollback(dbconn);
	if (dbconn->getCode(dbconn) != DB_SUCCESS) {
		ERRLOG2(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to rollback db. code = %d msg = %s", 
			dbconn->getCode(dbconn), dbconn->getMsg(dbconn));
		ts_ctx->status |= DIVY_TRANS_ABORT;

		/* 失敗したDBコネクションは使用不可とすべき */
		mark_inactive_dbconn(dbconn);
		return 1;
	}

	/* トランザクションが完了したことを記録する */
	ts_ctx->status |= DIVY_TRANS_END;
	return 0;
}

/**
 * 指定されたts_ctx が示すトランザクションは、処理を継続できる正しい状態
 * であるかどうかを調べる。
 *
 */
DIVY_DECLARE(int) divy_db_is_transaction_valid_state(divy_db_transaction_ctx *ts_ctx)
{
	/* 判断のしようがありません。
	 * NULLのため新規作成は可能です。*/
	if (ts_ctx == NULL) return 1;

	/*
	 * トランザクションが異常終了している or
	 * 終了していれば処理を継続出来ない.
	 */
	if (ts_ctx->status & (DIVY_TRANS_ABORT | DIVY_TRANS_END)) return 0; 
	return 1;
}

/**
 * 渡されたハッシュのキーに指定された文字列をcondcnt の数だけ','区切りで
 * 文字列合成し、condcnt の単位で divy_linkedlist_t * につめて返却する。
 *
 */
DIVY_DECLARE(divy_linkedlist_t *) divy_db_make_incond(apr_pool_t *p,
					apr_hash_t *cond_h, int condcnt)
{
	apr_hash_index_t *hidx;
	const char	 *key;
	divy_linkedlist_t  *list = NULL, *first = NULL;
	divy_sbuf *sbuf = NULL;
	int i = 0;

	/* ハッシュそのものが存在しなかった場合 */
	if (cond_h == NULL) return NULL;

	/* ハッシュをループして 接続文字列 作成 */
	for (hidx = apr_hash_first(p, cond_h); hidx; hidx = apr_hash_next(hidx)) {

		apr_hash_this(hidx, (const void **)&key, NULL, NULL);

		/* 文字列を接続していく */
		if (i == 0) {
			if (sbuf == NULL) {
				/* バッファの作成 */
				divy_sbuf_create(p, &sbuf, 512);
			}
			else {
				/* 使用済みバッファの初期化 */
				divy_sbuf_clear(sbuf);
			}
			/* 先頭の場合 */
			divy_sbuf_appendbyte(sbuf, 1, "'");
			divy_sbuf_append(sbuf, key);
			divy_sbuf_appendbyte(sbuf, 1, "'");
		}
		else {
			/* 2番目以降の場合 */
			divy_sbuf_appendbyte(sbuf, 2, ",'");
			divy_sbuf_append(sbuf, key);
			divy_sbuf_appendbyte(sbuf, 1, "'");
		}

		i++;

		/* カウンタ = 指定数の場合は構造体に追加しカウンタを初期化 */
		if (i == condcnt) {
			if (list == NULL) {
				first = list = apr_pcalloc(p, sizeof(divy_linkedlist_t));
			}
			else {
				list->next = apr_pcalloc(p, sizeof(divy_linkedlist_t));
				list = list->next;
			}
			list->val = apr_pstrdup(p, divy_sbuf_tostring(sbuf));
			list->next = NULL;

			i = 0;
		}
	}

	/* ループ終了後カウンタが初期値以外の場合は最後の1件を追加 */
	if (i != 0) {
		if (list == NULL) {
			first = list = apr_pcalloc(p, sizeof(divy_linkedlist_t));
		}
		else {
			list->next = apr_pcalloc(p, sizeof(divy_linkedlist_t));
			list = list->next;
		}
		list->val = apr_pstrdup(p, divy_sbuf_tostring(sbuf));
		list->next = NULL;
	}

	return first;
}

/**
 * 指定された文字列集合str_set からバインド変数コンテキストを生成し返却する。
 *
 */
DIVY_DECLARE(divy_db_bind_ctx *) divy_db_bindcontext_make(apr_pool_t *p,
						divy_cset_t *str_set, int unitnum)
{
	divy_db_bind_ctx *bindctx;

	/* 何もする必要なし */
	if (str_set == NULL || divy_cset_count(str_set) == 0 || unitnum == 0) {
		return NULL;
	}

	if (unitnum == -1) {
		unitnum = divy_cset_count(str_set);
	}

	bindctx = apr_pcalloc(p, sizeof(divy_db_bind_ctx));
	bindctx->unitnum = unitnum;
	bindctx->src_set = str_set;
	bindctx->p = p;

	return bindctx;
}

/**
 * バインドコンテキストbindctx から最初のイテレータへのポインタを取り出す。
 *
 */
DIVY_DECLARE(divy_db_bind_ctx_idx *) divy_db_bindcontext_first(divy_db_bind_ctx *bindctx)
{
	apr_pool_t *p;
	divy_db_bind_ctx_idx *idx;
	divy_linkedlist_t *list = NULL;
	divy_cset_index_t *ci;

	if (bindctx == NULL) return NULL;	/* 何も返せない */

	p = bindctx->p;

	bindctx->src_list = NULL;	/* 初期化 */

	/* src_set をリンクリストに変換 */
	for (ci = divy_cset_first(bindctx->src_set); ci; ci = divy_cset_next(ci)) {
		if (bindctx->src_list == NULL) {
			bindctx->src_list = list = apr_palloc(p, sizeof(divy_linkedlist_t));
		}
		else {
			list->next = apr_palloc(p, sizeof(divy_linkedlist_t));
			list = list->next;
		}
		divy_cset_this(ci, (const char **)&list->val);
		list->next = NULL;
	}

	/* イテレータの作成 */
	idx = apr_pcalloc(bindctx->p, sizeof(divy_db_bind_ctx_idx));
	idx->bindctx   = bindctx;
	idx->num       = _devide_list(&bindctx->src_list, &idx->part_list, bindctx->unitnum);
	idx->bindstr   = _make_bindstr(p, idx->num);

	return idx;
}

/**
 * イテレータbindctx_idx から次のイテレータへのポインタを取り出す。
 *
 */
DIVY_DECLARE(divy_db_bind_ctx_idx *) divy_db_bindcontext_next(divy_db_bind_ctx_idx *bindctx_idx)
{
	apr_pool_t *p;
	divy_db_bind_ctx_idx *idx;
	divy_db_bind_ctx *bindctx;
	divy_linkedlist_t *ll = NULL;
	int num;

	if (bindctx_idx == NULL) return NULL;	/* もうない */
	p       = bindctx_idx->bindctx->p;
	bindctx = bindctx_idx->bindctx;

	num = _devide_list(&bindctx->src_list, &ll, bindctx->unitnum);
	if (num == 0) {
		/* 有効なリストが1つもなければこのイテレータには何も入っていない */
		return NULL;
	}

	/* イテレータの作成 */
	idx = apr_pcalloc(p, sizeof(divy_db_bind_ctx_idx));
	idx->bindctx   = bindctx_idx->bindctx;
	idx->num       = num;
	idx->part_list = ll;
	idx->bindstr   = _make_bindstr(p, idx->num);

	return idx;
}

/**
 * 現在のイテレータbindctx_idx に格納されたバインド変数の集合をリストとして取り出す。
 *
 */
DIVY_DECLARE(void) divy_db_bindcontext_get_list(divy_db_bind_ctx_idx *bindctx_idx, divy_linkedlist_t **list)
{
	*list = NULL;	/* 初期化 */

	if (bindctx_idx == NULL) return;

	*list = bindctx_idx->part_list;
}

/**
 * 指定されたイテレータbindctx_idx に格納されたバインド変数と同数のバインド変数指示文字"?" を","で連結し
 * 文字列として返却する。
 *
 */
DIVY_DECLARE(void) divy_db_bindcontext_get_bindstr(divy_db_bind_ctx_idx *bindctx_idx, const char **bindstr)
{
	*bindstr = "";	/* 初期化 */

	if (bindctx_idx == NULL) return;

	*bindstr = bindctx_idx->bindstr;
}

/*--------------------------------------------------------------
  Define Private Functions
  --------------------------------------------------------------*/
/**
 * DB プロバイダを一意に識別するIDを取得する。
 * (note)
 * 	プロバイダグループIDであるprovider_gid がNULLの場合
 * 	全てのDBプロバイダは、providerNameだけで識別されるように
 * 	なります。
 *
 * @param p apr_pool_t * 返却する値を割り当てるプール
 * @param provider_gid const char * DBプロバイダグループ識別子
 * @param providerName const char * DBプロバイダの識別名称
 * @return const DbProviderId * DBプロバイダID
 */
static const DbProviderId * _create_db_providerid(apr_pool_t *p,
					const char *provider_gid,
					const char *providerName)
{
	DbProviderId *providerid = apr_pcalloc(p, sizeof(DbProviderId));

	/* (note) p から文字列の領域を再割り当てします */
	providerid->provider_gid = apr_pstrdup(p, provider_gid);
	providerid->providerName = apr_pstrdup(p, providerName);

	return providerid;
}


/**
 * providerid が示すDBプロバイダIDをpの領域を使って複製する。
 *
 * @param p apr_pool_t * DBプロバイダIDを割り当てるプール
 * @param providerid const DbProviderId * 複製元のプロバイダID
 */
static const DbProviderId * _clone_db_providerid(apr_pool_t *p,
					const DbProviderId *providerid)
{
	if (providerid == NULL) return NULL;

	/* 再作成する */
	return _create_db_providerid(p, providerid->provider_gid,
					providerid->providerName);
}

/**
 * リポジトリDBとそこから生成されたDBプロバイダをグループ化するための
 * 識別IDを生成する。
 *
 * (note) 背景
 * -------------
 * 	異なるVirtualHost、異なるLocationにおいて同じプロバイダ名称を
 * 	用いてはならないと強要する機構は存在しないし、導入することも
 * 	アーキテクチャ上、難しい。
 * 	場合によっては同名が設定されてしまう可能性があることになるのだが、
 * 	この場合、設定が互いに上書きされることになり、意図したリポジトリ
 * 	に接続することが出来なくなる。これは恐らく重大な影響を及ぼすだろう。
 * 	また、リポジトリDBは、その機構上、単一のLocationに関する情報しか
 * 	持つことが出来ないため、他のLocationと共有されると破綻してしまう。
 *
 * 	この関数は、これらの問題を発生を未然に防ぐため、本来共用されては
 * 	ならないリポジトリDBプロバイダおよび関連するDBプロバイダを"まと
 * 	あげた"実体(DBプロバイダグループ)を識別するIDを採番する目的で導入
 * 	された。
 * 	グループ化では以下の点が留意されている。
 *
 * 	 (1) Locationの異なるリポジトリDBプロバイダは共有されてはならない
 * 	 (2) 異なるリポジトリDBプロバイダによって生成されたDBプロバイダは
 * 	     一意に識別できるべき。(例え、プロバイダ名が同じでも。)
 * 	 (3) Locationが等しく、"同一の"リポジトリDBプロバイダである場合には
 * 	     常に同じグループにマッピングされるべき。
 * 	     なお、同一であることの保証は、dconf->dbmsname の自動採番機構
 * 	     によって保証されることになりました。
 *
 * 	関数_create_db_providerid のbaseid にこの関数から得られた識別文字列
 * 	を与えることでDBプロバイダを安全に識別することが出来るように
 * 	なります。
 *
 * @param p apr_pool_t * DBプロバイダグループIDを割り当てるプール
 * @param r request_rec * Location, VirtualHost, リポジトリDB情報を持つ構造体
 * @return const char * DBプロバイダグループID
 */
static const char * _create_db_provider_gid(apr_pool_t *p, request_rec *r)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	/* 2004.08.23 Mon
	 * リポジトリDBのDBMS識別名称が自動採番されるようになったので、
	 * その情報を利用してグループIDを生成することにしました。*/
	return apr_psprintf(p, "GRP.%s", dconf->dbmsname);
}

/**
 * provider_gid によって識別されるプロバイダグループの中から
 * providerNmaeが示すDB プロバイダ構造体(DbDataSource)取得して返却する。
 *
 * @param r request_rec *
 * @param providerid const DbProviderId* DBプロバイダID
 * @return DbDataSource* DB プロバイダ構造体へのポインタ
 */
static DbDataSource * _lookup_db_provider(request_rec *r,
					const DbProviderId *providerid)
{
	apr_status_t rv;
	DbDataSource *dbds = NULL;
	RET_STAT ret;
	DbProviderCntxt *p_ctx;

	/* mutex ロック */
	rv = apr_thread_mutex_lock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex lock.(code = %d)", rv);
		return NULL;
	}

	/*
	 * DB プロバイダが初期化されていなければ初期化する
	 */
	if (!_is_dbenv_ready(providerid->provider_gid)) {

		ret = _init_db_providers(r, providerid->provider_gid);
		if (ret != DB_SUCCESS) {
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to initialize db providers.");
			dbds = NULL;
		}
		else {
			/* DbDataSource の取得 */
			p_ctx = _get_prv_ctx(providerid);
			if (p_ctx && p_ctx->pc_state != DB_PCNTXT_DESTROYED) {
				/* 有効なdbds が見つかった場合 */
				dbds = p_ctx->dbds;
			}
		}
	}
	else {
		/* DbDataSource の取得 */
		p_ctx = _get_prv_ctx(providerid);
		if (p_ctx && p_ctx->pc_state != DB_PCNTXT_DESTROYED) {
			/* 有効なdbds が見つかった場合 */
			dbds = p_ctx->dbds;
		}

		/* 途中でライセンスが切れたことを報告したければ
		 * ここにチェックを入れなければならない */
	}

	/* mutex ロックを解除 */
	rv = apr_thread_mutex_unlock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex unlock. (code = %d)", rv);
		return dbds;	/* 大目に見る */
	}

	return dbds;
}

/**
 * provider_gid が示すDBプロバイダグループの環境を初期化し、
 * DBコネクションが取得可能な状態にする。
 * ただし、ライセンスの無いDBMSプロバイダは初期化されません。
 * (note)
 *	登録されたDBプロバイダを取得するには、lookup_db_provider関数を
 *	使用して下さい。
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param r request_rec*
 * @param provider_gid const char* DBプロバイダグループID
 * @return RET_STAT 処理終了ステータスコード
 * 		DB_SUCCESS: 成功 / DB_ERROR: 失敗
 */
static RET_STAT _init_db_providers(request_rec *r, const char *provider_gid)
{
	DbDataSource *repos = NULL, *dbds = NULL;
	DbConn *dbconn = NULL;
	divy_rdbo_dbms *dbms_pr_list = NULL, *dbms_pr = NULL;
	int ret;
	RET_STAT retstat;
	const DbProviderId *providerid;
	DbProviderGrpCntxt *grp_ctx;
	DbProviderCntxt *p_ctx;
	apr_hash_index_t *hi;
	const char *providerType;
	DbProviderLicense *license;
	divy_cset_t *doubtful_type_set = NULL;
	divy_cset_index_t *ci;

	dav_divy_dir_conf    *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	apr_pool_t *p               = r->pool;

	TRACE(p);

	/*
	 * ディレクトリコンフィグの検証
	 */
	if (_validate_reposdb_config(p, dconf)) {
		return DB_ERROR;
	}

	/*
	 * (0) ライセンスのあるDBプロバイダの種類をdoubtful_type_setに記録する
	 * (note)
	 * 	"doubtful"というprefixには、ライセンスのあるDBプロバイダが必ずしも
	 * 	ロード可能であるとは限らないという意味が込められています。
	 */
	if (sconf->dblicense_h == NULL || apr_hash_count(sconf->dblicense_h) == 0) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The database provider did not have any valid license. "
			"Please contact the administrator and "
			"setup license key. (providerType = %s)",
			dconf->dbmstype);
		return DB_ERROR;
	}

	for (hi = apr_hash_first(p, sconf->dblicense_h); hi; hi = apr_hash_next(hi)) {
		/* ライセンス情報とプロバイダタイプの取得 */
		apr_hash_this(hi, (const void **) &providerType, NULL,
							(void **) &license);
		if (license == NULL) continue;

		/* ライセンスの検証 */
		retstat = divy_db_validate_dblicense(p, license);
		if (retstat == DB_LICENSE_EXPIRE) {
			/* ライセンス切れ */
			continue;
		}

		if (doubtful_type_set == NULL) {
			doubtful_type_set = divy_cset_make(p);
		}

		/* DBMSの種類をdoubtful_type_set に記録 */
		divy_cset_set(doubtful_type_set, providerType);
	}

	/* 全部有効期限切れだった場合 */
	if (divy_cset_count(doubtful_type_set) == 0) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The db provider did not have any valid license. "
			"Maybe all license had been expired."
			"Please contact the administrator and setup license key. ");
		return DB_ERROR;
	}

	/*
	 * (1) リポジトリDB の初期化
	 */
	/* リポジトリDB プロバイダのライセンスを確認 */
	retstat = _has_license(p, sconf, dconf->dbmstype);
	if (retstat == DB_LICENSE_NOTHING) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"This repository db provider did not have valid license. "
			"Please contact the administrator and "
			"setup license key. (providerType = %s)",
			dconf->dbmstype);
		return DB_ERROR;
	}
	/* ライセンス切れ */
	else if (retstat == DB_LICENSE_EXPIRE) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
			"The license of repository db provider was expired. "
			"Please contact the administrator and "
			"setup new license key. (providerType = %s)",
			dconf->dbmstype);
		return DB_ERROR;
	}

	/* リポジトリDB プロバイダ探してDbDataSourceを取得する */
	ret = divy_run_create_dbdatasource(prvmng_p, dconf->dbmstype, &repos);
	if (ret == DB_DECLINED) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get repository db provider. (type = %s) "
			"Please check httpd.conf for dbmstype.",
			dconf->dbmstype);
		return DB_ERROR;
	}

	/* ロードできたので"疑わしい"種類集合から削除する */
	divy_cset_remove(doubtful_type_set, dconf->dbmstype);

	/* dconf の中身をrepos にコピーする */
	repos->isFinishedInitEnv = DBENV_INIT_FINISHED;	/* 初期化終了 */
	repos->dbmstype          = apr_pstrdup(prvmng_p, dconf->dbmstype);
	repos->dbmsname          = apr_pstrdup(prvmng_p, dconf->dbmsname);
	repos->hostname          = apr_pstrdup(prvmng_p, dconf->hostname);
	repos->hostport          = apr_pstrdup(prvmng_p, dconf->hostport);
	repos->dbname            = apr_pstrdup(prvmng_p, dconf->dbname);
	repos->username          = apr_pstrdup(prvmng_p, dconf->username);
	repos->password          = apr_pstrdup(prvmng_p, dconf->password);
	repos->dbpool            = dconf->dbpool;
	repos->dbminspareconn    = GET_MIN_SPARECONN(dconf->dbminspareconn);
	repos->dbmaxspareconn    = GET_MAX_SPARECONN(dconf->dbmaxspareconn);

	/* DBプロバイダIDの生成 */
	providerid = _create_db_providerid(prvmng_p, provider_gid, repos->dbmsname);

	/* DBプロバイダグループコンテキストを新規作成 */
	grp_ctx = _create_prvgrp_ctx(prvmng_p, provider_gid, providerid);

	/* DBプロバイダコンテキストの生成 */
	p_ctx = _create_prv_ctx(prvmng_p, providerid, 1, repos);

	/* DBプロバイダコンテキストをDBプロバイダグループコンテキストに登録 */
	if (_register_db_provider(grp_ctx, p_ctx) != DB_SUCCESS) {
		return DB_ERROR;
	}

	/* DBを検索したいので、リポジトリDBのDbConnを取得する */
	if (USING_CPOOL(repos->dbpool)) {
		dbconn = repos->getDbConn(repos, prvmng_p);
	}
	else {
		dbconn = repos->getDbConn(repos, p);
	}
	if (dbconn->getCode(dbconn) != DB_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
			"Failed to get repos-DbConn. Reason: %s",
			REPLACE_NULL(dbconn->getMsg(dbconn)));
		if (dbconn != NULL) dbconn->close(dbconn);	/* closeが必要 */
		return DB_ERROR;
	}
	/* 管理用コンテキストを付加する(永続プールだけ) */
	if (USING_CPOOL(repos->dbpool)) {
		CPOOL_CTX(dbconn) = _create_cpoolctx(prvmng_p, providerid,
							DB_PTYPE_PERMANENT_POOL);
		/* 永続プールの生成と登録 */
		p_ctx->cpool = _create_perm_DbConnPool(repos);

		/* 永続プールにdbconnを登録 */
		_append_DbConnList_to_DbConnPool(p_ctx->cpool,
					_create_DbConnList(prvmng_p, dbconn));
			
		/* DbConn のステータスを"未使用"に変更 */
		CPOOL_CTX(dbconn)->state = DB_CONNSTAT_UNUSE;
		p_ctx->cpool->navail++;	/* 有効なDBコネクションが1つ増えた */
	}

	/*
	 * リポジトリDB に接続してDBMS一覧を取得する
	 *
	 * (note)
	 * 	自身(dbds)がリポジトリDBプロバイダ(repos)であっても、
	 * 	通常のDBプロバイダであることもあり得る.	(顧客DBと
	 * 	リポジトリDBが同じだった場合など)。そのため、リポジトリ
	 * 	DBの検索をTryしなくてはならない
	 */
	if (divy_rdbo_get_dbmsinfo(p, dbconn, &dbms_pr_list)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to get db provider information from repos db.");
		if (!USING_CPOOL(repos->dbpool)) {
			/* 一時プールのDbConnならば閉じてしまう */
			if (dbconn != NULL) dbconn->close(dbconn);
		}
		return DB_ERROR;
	}

	/* 永続プールでなければdbconnを捨てる */
	if (!USING_CPOOL(repos->dbpool)) {
		if (dbconn != NULL) dbconn->close(dbconn);
		dbconn = NULL;
	}

	/* 検索結果の取得 */
	for (dbms_pr = dbms_pr_list; dbms_pr; dbms_pr = dbms_pr->next) {

		/* DB プロバイダのライセンスを確認 */
		retstat = _has_license(p, sconf, dbms_pr->type);
		if (retstat == DB_LICENSE_NOTHING) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"This db provider did not have valid license. "
				"Please contact the administrator and "
				"setup license key. (providerType = %s)",
				dbms_pr->type);
			continue;	/* 初期化をしない */
		}
		/* ライセンス切れ */
		else if (retstat == DB_LICENSE_EXPIRE) {
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
				"The license of db provider was expired. "
				"Please contact the administrator and "
				"setup new license key. (providerType = %s)",
				dbms_pr->type);
			continue;	/* 初期化をしない */
		}

		/*
		 * DBプロバイダ構造体の生成を行う。
		 * (note)
		 * 	DBプロバイダは、DBMSの種類(dbmstype)ごとではなく、
		 *	接続先情報毎、即ち、DBMS接続先別名称(dbmsname)毎に
		 * 	存在する。なので結果セットの数だけ作る必要がある。
		 */
		ret = divy_run_create_dbdatasource(prvmng_p, dbms_pr->type, &dbds);
		if (ret == DB_DECLINED) {
			/*
			 * リポジトリDB にはエントリが記録されているが、対応する
			 * DB プロバイダはLoadModuleされていなかった場合。
			 * 何らかの設定ミスであると思われるので警告しておく。
			 * でもエラーにはしない。意図的かもしれないからだ。
			 */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"Could not get db provider(type = %s). "
				"If this provider was not used, Please ignore.",
				dbms_pr->type);
			continue;
		}

		/* ロードできたので"疑わしい"種類集合から削除する */
		divy_cset_remove(doubtful_type_set, dbms_pr->type);

		/* dbms_pr の値をdbds にコピーする */ 

		/* リポジトリDB擬似エントリの場合、repos からのコピー */
		if (strcmp(dbms_pr->dbmsid, DIVY_REPOS_DBMSID) == 0) {
			dbds->dbmstype = apr_pstrdup(prvmng_p, repos->dbmstype);
			dbds->dbmsname = apr_pstrdup(prvmng_p, dbms_pr->name);	/* ここだけは自分のもの */
			dbds->hostname = apr_pstrdup(prvmng_p, repos->hostname);
			dbds->hostport = repos->hostport;
			dbds->dbname   = apr_pstrdup(prvmng_p, repos->dbname);
			dbds->username = apr_pstrdup(prvmng_p, repos->username);
			dbds->password = apr_pstrdup(prvmng_p, repos->password);
		}
		else {
			dbds->dbmstype = apr_pstrdup(prvmng_p, dbms_pr->type);
			dbds->dbmsname = apr_pstrdup(prvmng_p, dbms_pr->name);
			dbds->hostname = apr_pstrdup(prvmng_p, dbms_pr->hostname);
			dbds->hostport = apr_itoa(prvmng_p,    dbms_pr->port);
			dbds->dbname   = apr_pstrdup(prvmng_p, dbms_pr->dbname);
			dbds->username = apr_pstrdup(prvmng_p, dbms_pr->username);
			dbds->password = apr_pstrdup(prvmng_p, dbms_pr->password);
		}
		if (USING_CPOOL(repos->dbpool)) {
			/* (note) 歴史的(?)な経緯からDB上のフラグは0(off)/1(on)
			 * 	なので、それを置き換えておく必要がある */
			dbds->dbpool = (dbms_pr->dbpool) ? DIVY_DBPOOL_ON : DIVY_DBPOOL_OFF;
		}
		else {
			dbds->dbpool = DIVY_DBPOOL_OFF;	/* 常にoff */
		}
		dbds->dbminspareconn = GET_MIN_SPARECONN(dbms_pr->dbminspareconn);
		dbds->dbmaxspareconn = GET_MAX_SPARECONN(dbms_pr->dbmaxspareconn);

		/* 初期化が完了しました */
		dbds->isFinishedInitEnv = DBENV_INIT_FINISHED;

		/* DBプロバイダIDの生成 */
		providerid = _create_db_providerid(prvmng_p, provider_gid,
								dbds->dbmsname);

		/* DBプロバイダコンテキストを新規作成 */
		p_ctx = _create_prv_ctx(prvmng_p, providerid, 0, dbds);

		/* DBプロバイダコンテキストをDBプロバイダグループコンテキストに登録 */
		if (_register_db_provider(grp_ctx, p_ctx) != DB_SUCCESS) {
			return DB_ERROR;
		}
	}

	/*
	 * 接続情報がなかった"疑わしい"DBプロバイダがロードできるかどうか検証する。
	 */
	if (divy_cset_count(doubtful_type_set) > 0) {
		for (ci = divy_cset_first(doubtful_type_set);
						ci; ci = divy_cset_next(ci)) {
			/* "疑わしい"DBプロバイダの種類を取得 */
			divy_cset_this(ci, &providerType);

			/* ロードしてみる(最後には捨てるのでp に確保する) */
			ret = divy_run_create_dbdatasource(p, providerType, &dbds);
			if (ret == OK) {
				/* loadable_type_set に登録する */
				apr_hash_set(grp_ctx->loadable_type_set,
						apr_pstrdup(prvmng_p, providerType),
						APR_HASH_KEY_STRING, "");
			}
		}
	}

	/* 初期化処理完了の状態を記録する */
	grp_ctx->init_finished = 1;

	/* DBプロバイダグループコンテキストを登録 */
	(void) _register_prvgrp_ctx(grp_ctx);

	return DB_SUCCESS;
}

/**
 * DBプロバイダが提供するハンドラ構造体(DbDataSource) をDBプロバイダ
 * グループコンテキストに登録する。
 * 登録されたDBプロバイダ構造体は、lookup_db_provider、または
 * lookup_reposdb_provider 関数を利用して取得することが出来ます。
 *
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param grp_ctx DbProviderGrpCntxt * DBプロバイダグループコンテキスト
 * @param p_ctx DbProviderCntxt * DBプロバイダコンテキスト
 * @return RET_STAT 処理終了ステータスコード
 * 		DB_SUCCESS: 成功 / DB_ERROR: 失敗
 */
static RET_STAT _register_db_provider(DbProviderGrpCntxt *grp_ctx,
					DbProviderCntxt *p_ctx)
{
	if (grp_ctx == NULL || p_ctx == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The grp_ctx or p_ctx is NULL. ");
		return DB_ERROR;
	}

	/* プロバイダ名称をキーとしてDBプロバイダコンテキストを登録する */
	apr_hash_set(grp_ctx->dbprv_hash, p_ctx->providerid->providerName,
					APR_HASH_KEY_STRING, p_ctx);

	/* DBMS種類を登録する */
	apr_hash_set(grp_ctx->loadable_type_set, p_ctx->dbds->dbmstype,
						APR_HASH_KEY_STRING, "");

	return DB_SUCCESS;
}

/**
 * DBプロバイダ群の初期化が完了しているかどうか.
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param provider_gid const char* DBプロバイダグループID
 * @return int 1: 初期化されている / 0: 初期化されていない
 */
static int _is_dbenv_ready(const char *provider_gid)
{
	DbProviderGrpCntxt *grp_ctx;

	if (!is_initial || apr_hash_count(dbprv_grp_hash) == 0) {
		return 0;
	}

	/* DBプロバイダの初期化状態を取得する */
	grp_ctx = apr_hash_get(dbprv_grp_hash, provider_gid, APR_HASH_KEY_STRING);
	if (grp_ctx && grp_ctx->init_finished) {
		return 1;
	}
	else {
		return 0;
	}
}

/**
 * 指定されたサーバコンフィグsconfの内、リポジトリDBプロバイダに関する
 * 情報が正しいかどうかを検証する。
 *
 * [ 検証にパスする条件 ]
 *
 * ・dconf がNULL ではないこと
 * ・dbmstype, dbmsname がNULLまたは空文字ではないこと
 * ・hostport がNULL でなければ、その値は数値であること
 *
 * @param p apr_pool_t *
 * @param dconf dav_divy_dir_conf * ディレクトリコンフィグ
 * @return int 処理結果(1: 誤り / 0: 正しい)
 */
static int _validate_reposdb_config(apr_pool_t *p, dav_divy_dir_conf *dconf)
{
	if (dconf == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"dconf is NULL.");
		return 1;
	}

 	/* dbmstype がNULLまたは空文字ではないこと */
	if (IS_EMPTY(dconf->dbmstype)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The value of \"TfDbDbmstype\" is missing. "
			"Please set these value.");
		return 1;
	}

 	/* hostport がNULL でなければ、その値は数値であること */
	if (dconf->hostport && !dav_divy_isdigit_str(dconf->hostport)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The value of \"TfDbHostpor\" must be a positive digit."
			"Please check httpd.conf.");
		return 1;

	}

	/* dbmsname がNULLまたは空文字ではないこと
	 * もし空ならばバグである。dbmsname が自動生成するので。*/
	if (IS_EMPTY(dconf->dbmsname)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The value of \"TfDbDbmsname\" is missing. ");
		return 1;
	}

	return 0;
}

/**
 * 指定されたproviderType のDBプロバイダにライセンスがあるかどうかを判定する。
 * この関数は、divy_db_has_licenseに実装を提供するプライベート関数です。
 *
 * (note) ライセンスがあると判断する条件
 * 	(1) DBプロバイダライセンスキーが登録されていること
 * 	(2) ライセンス期限内もしくは、無制限ライセンスであること
 *
 * @param p apr_pool_t *
 * @param sconf dav_divy_server_conf *
 * @param providerType const char * プロバイダの種類
 * @return int ライセンスがあるかどうか
 * 	DB_LICENSE_VALID   : 持っている
 * 	DB_LICENSE_NOTHING : 持っていない
 * 	DB_LICENSE_EXPIRE  : 持っていたが切れてしまった
 */
static RET_STAT _has_license(apr_pool_t *p,
					dav_divy_server_conf *sconf,
					const char *providerType)
{
	DbProviderLicense *license;

	if (IS_EMPTY(providerType) || sconf->dblicense_h == NULL ||
			apr_hash_count(sconf->dblicense_h) == 0) {

		return DB_LICENSE_NOTHING;
	}

	/* ライセンスキーコンテキストの取得 */
	license = apr_hash_get(sconf->dblicense_h, providerType, APR_HASH_KEY_STRING);

	/* ライセンスの検証 */
	return divy_db_validate_dblicense(p, license);
	
}

/**
 * 新しいDbConnListを1つ作成して返却する。
 * (note)
 * 	リスト構造にはしません
 *
 * @param p apr_pool_t *
 * @param dbconn DbConn *
 * @return DbConnList * 作成したDbConnListへのポインタ
 */
static DbConnList * _create_DbConnList(apr_pool_t *p, DbConn *dbconn)
{
	DbConnList *list = apr_pcalloc(p, sizeof(DbConnList));
	list->dbconn = dbconn;
	list->next   = NULL;
	list->prev   = NULL;

	return list;
}

/**
 * l1の後ろにl2をつなげる
 */
static void _append_DbConnList(DbConnList *l1, DbConnList *l2)
{
	DbConnList *list;

	if (l1 == NULL || l2 == NULL) return;

	for (list = l1; list->next; list = list->next);
	list->next = l2;
	l2->prev   = list;
}

/**
 * cpool が示すDBコネクションプールにleltsのリストを追加する。
 * (note)
 * 	この関数をコールする前にlelts はどこかのリストに
 * 	所属していてはなりません。
 *
 * @param cpool DbConnPool * DBコネクションプール
 * @param lelts DbConnList * DBコネクションリスト
 */
static void _append_DbConnList_to_DbConnPool(DbConnPool *cpool,
						DbConnList *lelts)
{
	DbConnList *list;
	if (cpool == NULL || lelts == NULL) return;

	/* リストに追加する */
	if (cpool->dbconnlist == NULL) {
		cpool->dbconnlist = lelts;
	}
	else {
		/* cpool->dbconnlist の末尾を探す */
		_append_DbConnList(cpool->dbconnlist, lelts);
	}

	/* lelts に入っている要素数を加算する */
	for (list = lelts; list; list = list->next) {
		cpool->nelts++;
	}
}

/**
 * cpool が示すDBコネクションプールからlelts のリストを取り外す。
 * この関数が終了するとleltsからリストを辿ることが出来なくなります。
 * 注意して下さい。
 * (note)
 * 	DbConnList は双方向リストの実装です。従って、leltsも正しい
 * 	双方向リストとして生成されているものとして処理します。
 *
 * @param cpool DbConnPool * DBコネクションプール
 * @param lelts DbConnList * DBコネクションリスト
 * @return DbConnList * leltsの次のDbConnList *
 */
static DbConnList * _remove_DbConnList_from_DbConnPool(DbConnPool *cpool,
						DbConnList *lelts)
{
	DbConnList *prev_lelts, *next_lelts;
	if (cpool == NULL || lelts == NULL) return NULL;

	/*
	 * (note)
	 * lelts ==  (lelts->prev)->next == (lelts->next)->prev
	 */
	prev_lelts = lelts->prev;
	next_lelts = lelts->next;
	if (prev_lelts == NULL && next_lelts == NULL) {
		cpool->dbconnlist = NULL;	/* 1個しかなかったリストが無くなった */
	}
	else if (prev_lelts == NULL && next_lelts != NULL) {
		next_lelts->prev = NULL;	/* next_lelts が先頭になる */
		cpool->dbconnlist = next_lelts;
	}
	else if (prev_lelts != NULL && next_lelts == NULL) {
		prev_lelts->next  = NULL;	/* prev_lelts が末尾になる */
	}
	else {
		prev_lelts->next  = next_lelts;
		next_lelts->prev = prev_lelts;
	}

	cpool->nelts--;		/* 総数をデクリメント */
	lelts->prev = NULL;	/* 他のリスト要素とのリンクを消す */
	lelts->next = NULL;

	return next_lelts;	/* 1つ隣の要素を返す */
}

/**
 * DBコネクションを管理するためのコンテキストを生成する。
 *
 * @param p apr_pool_t * DbConnPoolCntxt とライフサイクルを共にするプール
 * @param providerid const DbProviderId * DBプロバイダID
 * @param usepooltype DbConnPoolType プールの種類
 * @return DbConnPoolCntxt *
 */
static DbConnPoolCntxt * _create_cpoolctx(apr_pool_t *p,
					const DbProviderId *providerid,
					DbConnPoolType usepooltype)
{
	DbConnPoolCntxt *ctx = apr_pcalloc(p, sizeof(DbConnPoolCntxt));

	ctx->usepooltype = usepooltype;
	ctx->state       = DB_CONNSTAT_UNUSE;	/* 未使用 */
	ctx->ctag        = NULL;
	ctx->providerid  = _clone_db_providerid(p, providerid);
	ctx->creationtm  = apr_time_now();	/* タイムスタンプの付加 */

	return ctx;
}

/**
 * DBプロバイダグループコンテキストを生成する。
 * (note)
 * 	引数repos_providerid はp から割り当てられていなければ
 * 	なりません。
 *
 * @param p apr_pool_t * DBプロバイダグループとライフサイクルを共にするプール
 * @param provider_gid const char * DBプロバイダグループ識別子
 * @param repos_providerid const DbProviderId *
 * 			コンテキストを生成したリポジトリDBプロバイダのID
 * @return DbProviderGrpCntxt * 生成したDBプロバイダグループコンテキスト
 * 				確保に失敗したらNULLを返却
 */
static DbProviderGrpCntxt * _create_prvgrp_ctx(apr_pool_t *p,
					const char *provider_gid,
					const DbProviderId *repos_providerid)
{
	DbProviderGrpCntxt *grp_ctx;

	if (IS_EMPTY(provider_gid) || repos_providerid == NULL) {

		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The provider_gid or repos_providerid is EMPTY. ");
		return NULL;
	}

	grp_ctx = apr_pcalloc(p, sizeof(DbProviderGrpCntxt));

	/* (note) スコープを合わせるため、文字列はp から再確保します */
	grp_ctx->provider_gid     = apr_pstrdup(p, provider_gid);
	grp_ctx->repos_providerid = repos_providerid;
	grp_ctx->loadable_type_set= apr_hash_make(p);
	grp_ctx->dbprv_hash       = apr_hash_make(p);
	grp_ctx->init_finished    = 0;

	return grp_ctx;
}

/**
 * DBプロバイダグループ用コンテキストを取得する
 * (note)
 * 	この関数は同期アクセスされるよう呼び出し元にて保証して下さい。
 *
 * @param provider_gid const char * DBプロバイダグループ識別ID
 * @return DbProviderGrpCntxt * 取得したDBプロバイダグループコンテキスト
 */
static DbProviderGrpCntxt * _get_prvgrp_ctx(const char *provider_gid)
{
	if (IS_EMPTY(provider_gid)) {
		return NULL;
	}

	return apr_hash_get(dbprv_grp_hash, provider_gid, APR_HASH_KEY_STRING);
}

/**
 * 指定されたgrp_ctx が示すDBプロバイダグループコンテキストを登録する。
 * (note)
 * 	この関数は同期アクセスされるよう呼び出し元にて保証して下さい。
 *
 * @param grp_ctx DbProviderGrpCntxt * DBプロバイダコンテキスト
 * @return RET_STAT 処理ステータス
 */
static RET_STAT _register_prvgrp_ctx(DbProviderGrpCntxt *grp_ctx)
{
	if (grp_ctx == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The grp_ctx is NULL. ");
		return DB_ERROR;
	}

	/*
	 * グローバルハッシュにDBプロバイダグループIDをキーとして登録する
	 */
	apr_hash_set(dbprv_grp_hash, grp_ctx->provider_gid,
					APR_HASH_KEY_STRING, grp_ctx);

	return DB_SUCCESS;
}

/**
 * DBプロバイダコンテキストを生成する。
 * (note)
 * 	providerid はp から割り当てられていなければなりません。
 *
 * @param p apr_pool_t * このコンテキスを割り当てるプール
 * @param providerid const DbProviderId * DBプロバイダID
 * @param is_reposdb int リポジトリDBプロバイダかどうか
 * @param dbds DbDataSource * DBプロバイダハンドラ構造体
 * @return DbProviderCntxt * 生成に失敗したらNULL
 */
static DbProviderCntxt * _create_prv_ctx(apr_pool_t *p,
					const DbProviderId *providerid,
					int is_reposdb,
					DbDataSource *dbds)
{
	DbProviderCntxt *p_ctx;

	if (dbds == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The dbds is NULL. ");
		return NULL;
	}

	p_ctx = apr_pcalloc(p, sizeof(DbProviderCntxt));
	p_ctx->providerid  = providerid;
	p_ctx->pc_state    = DB_PCNTXT_INIT;
	p_ctx->dbds        = dbds;
	p_ctx->cpool       = NULL;	/* 初期状態 */
	p_ctx->conn_digest = _create_conn_digest(p, DIVY_DB_DATA_DBDS, dbds);
	p_ctx->is_reposdb  = is_reposdb;

	return p_ctx;
}

/**
 * grp_ctx が示すDBプロバイダグループで管理されているproviderName の
 * DBプロバイダコンテキストを取得する。
 * (note)
 * 	この関数は同期アクセスされるよう呼び出し元にて保証して下さい。
 *
 * @param providerid const DbProviderId * DBプロバイダID
 * @return DbProviderCntxt * 取得したDBプロバイダコンテキスト
 *                           なければNULLを返却
 */
static DbProviderCntxt * _get_prv_ctx(const DbProviderId *providerid)
{
	DbProviderGrpCntxt *grp_ctx;

	if (providerid == NULL) return NULL;

	/* DBプロバイダグループコンテキストの取得 */
	grp_ctx = _get_prvgrp_ctx(providerid->provider_gid);

	return apr_hash_get(grp_ctx->dbprv_hash,
				providerid->providerName, APR_HASH_KEY_STRING);
}

/**
 * DBMS接続情報の要約メッセージ(ダイジェスト)を生成して返却する。
 * (note)
 * 	ダイジェストメッセージには、MD5を使用しています。
 * 	頻繁に呼び出すと性能劣化に繋がります。
 *
 * @param p apr_pool_t *
 * @param kind_of_data int data に指定されるオブジェクトの種類
 * 		DIVY_DB_DATA_DBDS: DbDataSource *
 * 		DIVY_DB_DATA_DBMS: divy_rdbo_dbms *
 * @param data void * DbDataSource またはdivy_rdbo_dbms へのポインタ
 * @return const char * ダイジェストメッセージ
 */
static const char * _create_conn_digest(apr_pool_t *p,
					int kind_of_data, void *data)
{
	apr_status_t rv;
	const unsigned char *pre_str;
	unsigned char digest[APR_MD5_DIGESTSIZE];

	/* ダイジェストの元となるメッセージを作成 */
	if (kind_of_data == DIVY_DB_DATA_DBDS) {
		DbDataSource *dbds = data;
		pre_str = apr_psprintf(p, "%s#%s#%s#%s#%s#%s#%d#%s",
					dbds->dbmstype, dbds->hostname,
					dbds->hostport, dbds->dbname,
					dbds->username, dbds->password,
					dbds->dbpool,   dbds->dbmsname);
	}
	else if (kind_of_data == DIVY_DB_DATA_DBMS) {
		divy_rdbo_dbms *dbms = data;
		pre_str = apr_psprintf(p, "%s#%s#%d#%s#%s#%s#%d#%s",
					dbms->type,	dbms->hostname,
					dbms->port,	dbms->dbname,
					dbms->username, dbms->password,
					dbms->dbpool,   dbms->name);
	}
	else {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"The kind of data is invalid.(kind_of_data = %d)",
			kind_of_data);
		return NULL;
	}

	/* MD5ダイジェストメッセージを生成してこれを返却 */
	rv = apr_md5(digest, pre_str, strlen((const char*)pre_str));
	if (rv != APR_SUCCESS) return "";

	return dav_divy_btohex(p, APR_MD5_DIGESTSIZE, digest);
}

/**
 * 一時プールを破棄し、
 * 一時プールにのみ所属していたDBコネクションを切断して開放し、
 * 永続プールに所属していたコネクションをプールに戻す。
 * DBコネクションに付加されていたctag はここでなくなります。
 * (note)
 * 	この関数を複数回アクセスしないようコール元で保証して下さい。
 *
 * @param ctx void * (一時プールのDbConnPool へのポインタ)
 * @return 終了ステータス (APR_SUCCESS)
 */
static apr_status_t _cleanup_used_dbconn(void *ctx)
{
	DbConnPool *cpool = ctx;	/* 一時プールを取得 */
	DbConnList *list;

	if (cpool == NULL) return APR_SUCCESS;

	/*
	 * 一時プールに所属したDbConn: 破棄する
	 * 永続プールに所属したDbConn: 永続プールに戻す
	 */
	for (list = cpool->dbconnlist; list; list = list->next) {
		if (list->dbconn == NULL || CPOOL_CTX(list->dbconn) == NULL) {
			continue;
		}

		/* ctag をdbconn から剥がす */
		CPOOL_CTX(list->dbconn)->ctag = NULL;

		/* 永続プールの場合 */
		if (CPOOL_CTX(list->dbconn)->usepooltype == DB_PTYPE_PERMANENT_POOL) {

			/* 永続プールに戻す */
			_restore_DbConn_to_perm_DbConnPool(list->dbconn);
		}
		/* 一時プールの場合 */
		else {
			/* DBコネクションを閉じる */
			_close_dbconn(list->dbconn);
			list->dbconn = NULL;
		}
	}

	/* 一時プール自体の破棄はApacheにお任せ */
	cpool = NULL;

	return APR_SUCCESS;
}

/**
 * 指定されたdbconn が示すDBコネクションを閉じる
 * (note)
 * 	エラーが発生しても無視します。
 *
 * @param dbconn DbConn * close 対象のdbconn
 */
static void _close_dbconn(DbConn *dbconn)
{
	if (dbconn == NULL) return;	/* 何も出来ない */

	/* DBコネクションを閉じる */
	dbconn->close(dbconn);
	if (dbconn->getCode(dbconn) != DB_SUCCESS) {
		/* 失敗したらログだけ出しておく */
		ERRLOG2(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to close DbConn. "
			"(providerName = %s) Reason: %s",
			dbconn->dbds->dbmsname, dbconn->getMsg(dbconn));
	}
}

/**
 * 使用しているグローバル変数の値を全て初期化する
 */
static void _clear_globalval(void)
{
	prvmng_p          = NULL;
	is_initial        = 0;
	exit_connmgr_th   = 0;
	dbprv_grp_hash    = NULL;
	dbrefreshinterval = APR_TIME_C(0);
	connmgr_th        = NULL;
	connmgr_state     = DB_CONNMGR_ST_INIT;
	cpool_mutex       = NULL;
	cpool_cond        = NULL;
}

/*-----------------------------------------------------------------------------
 * Define "Temporary DbConn Pool" functions
 -----------------------------------------------------------------------------*/
/**
 * 指定されたproviderid の一時プール(1次プール)にアクティブなDbConn が
 * あれば、そのDbConnを"使用中"にしてから返却する。
 * なければNULLを返却します。
 *
 * @param p apr_pool_t * 一時プールとライフサイクルを共にするプール
 * @param ctag const char * 取得したいDbConnを識別するタグ名称
 * @param providerid const DbProviderId * DBプロバイダID
 * @return DbConn * アクティブなDbConn / NULL
 */
static DbConn * _trygetDbConn_from_temp_DbConnPool(apr_pool_t *p,
						const char *ctag,
						const DbProviderId *providerid)
{
	DbConnPool *cpool = NULL;
	DbConn *dbconn    = NULL;
	DbConnList *list;

	/* 一時プールを取得する */
	cpool = _lookup_temp_DbConnPool(p, providerid);
	if (cpool == NULL) {
		/* 有効なDbConnが無かった */
		return NULL;
	}

	/* 一時プールがあればそこからDbConnを探す */
	for (list = cpool->dbconnlist; list; list = list->next) {
		dbconn = list->dbconn;

		/*
		 * "未使用"または"使用中"のDbConnを探す
		 * (note)
		 * 	一時プールでは、"使用中"であっても再利用できます。
		 */
		if (CPOOL_CTX(dbconn) &&
			(CPOOL_CTX(dbconn)->state == DB_CONNSTAT_UNUSE ||
			 CPOOL_CTX(dbconn)->state == DB_CONNSTAT_INUSE)) {

			/* dbconn にctag があった場合 */
			if (CPOOL_CTX(dbconn)->ctag && ctag &&
				strcmp(ctag, CPOOL_CTX(dbconn)->ctag) == 0) {

				/* "使用中"に変更する */
				CPOOL_CTX(dbconn)->state = DB_CONNSTAT_INUSE;

				break;
			}
			/* dbconn にctag がなかった場合 */
			else if (CPOOL_CTX(dbconn)->ctag == NULL && ctag == NULL) {

				/* "使用中"に変更する */
				CPOOL_CTX(dbconn)->state = DB_CONNSTAT_INUSE;

				break;
			}
		}
		dbconn = NULL;	/* 再初期化 */
	}

	return dbconn;
}

/**
 * 一時プール(1次プール) にDB コネクションdbconn を格納する。
 * 一時プールが存在しなければ新たに生成します。
 *
 * (note)
 * 	p がr->pool である限り、mutex ロックは不要です。
 *
 * @param p apr_pool_t * 一時プールとライフサイクルを共にするプール
 * @param ctag const char * 取得したいDbConnを識別するタグ名称
 * @param providerid const DbProviderId * DBプロバイダID
 * @param dbconn DbConn *
 */
static void _store_DbConn_to_temp_DbConnPool(apr_pool_t *p,
						const char *ctag,
						const DbProviderId *providerid,
						DbConn *dbconn)
{
	DbConnPool *cpool = NULL;
	RET_STAT st;

	if (dbconn == NULL) return;

	/* 一時プールを取得する */
	cpool = _lookup_temp_DbConnPool(p, providerid);
	if (cpool == NULL) {
		/* 生成 */
		cpool = _create_temp_DbConnPool(p);

		/* 登録 */
		st = _register_temp_DbConnPool(p, providerid, cpool);
		if (st != DB_SUCCESS) {
			cpool = NULL;	/* 登録しない */
			return;
		}
	}

	if (ctag != NULL) {
		/* dbconn にctag をつける */
		CPOOL_CTX(dbconn)->ctag = apr_pstrdup(p, ctag);
	}

	/* cpoolにリストを繋げる */
	_append_DbConnList_to_DbConnPool(cpool, _create_DbConnList(p, dbconn));
}

/**
 * 一時プール(1次プール)を生成して返却する。
 *
 * @param p apr_pool_t * 一時プールの領域が確保されるapr_pool_t
 * @return DbConnPool * 生成した一時プールへのポインタ
 */
static DbConnPool * _create_temp_DbConnPool(apr_pool_t *p)
{
	DbConnPool *cpool = apr_pcalloc(p, sizeof(DbConnPool));

	cpool->pooltype   = DB_PTYPE_TEMPORARY_POOL;
	cpool->nmax       = -1;
	cpool->nmin       = -1;
	cpool->dbconnlist = NULL;
	cpool->nelts      = 0;
	cpool->navail     = 0;

	return cpool;
}

/**
 * providerid を使って、一時プール(1次プール)を取得する。
 * (note)
 * 	一時プールは、p の中に格納された"一時的に"使用するDbConnのプールです.
 *
 * @param p apr_pool_t *
 * @param providerid const DbProviderId * DBプロバイダID
 * @return DbConnPool * 一時プールへのポインタ。取得に失敗したらNULL
 */
static DbConnPool * _lookup_temp_DbConnPool(apr_pool_t *p,
						const DbProviderId *providerid)
{
	DbConnPool *cpool = NULL;

	/* キャッシュp からcpool を取得する */
	cpool = divy_pcache_vget_data(p, DIVY_PCACHE_DAT_REQ_DBPROVIER,
					providerid->providerName, NULL);

	return cpool;
}

/**
 * providerid を使って、一時プール(1次プール)をp に登録する
 *
 * @param p apr_pool_t * 一時プールとライフサイクルと共にするプール
 * @param providerid const DbProviderId * DBプロバイダID
 * @param DbConnPool * 一時プールへのポインタ
 * @return RET_STAT 処理ステータス
 */
static RET_STAT _register_temp_DbConnPool(apr_pool_t *p,
						const DbProviderId *providerid,
						DbConnPool *cpool)
{
	if (cpool == NULL || providerid == NULL) {
		return DB_ERROR;
	}

	/* r->pool にcpool をキャッシュする */
	divy_pcache_vfset_data(p, cpool, _cleanup_used_dbconn,
					DIVY_PCACHE_DAT_REQ_DBPROVIER,
					providerid->providerName, NULL);

	/*
	 * Child用クリーンアップハンドラをアンレジスタする
	 * (note)
	 * 	これをしないと、apr_proc_create してChildプロセスを起動した時に
	 * 	DBコネクションが切られてしまいます。
	 */
	apr_pool_child_cleanup_set(p, cpool, _cleanup_used_dbconn,
						apr_pool_cleanup_null);

	return DB_SUCCESS;
}


/*-----------------------------------------------------------------------------
 * Define "Permanent DbConn Pool" functions
 -----------------------------------------------------------------------------*/
/**
 * providerid が示す永続プール(2次プール)にアクティブなDbConn が既にあれば、
 * そのDbConnを取得する。
 * 無ければNULLを返却します。
 *
 * (note)
 * 	DbConnは"使用中"に変更してから返却されます。
 * 	providerid のDBプロバイダが永続プールを利用することは、
 * 	呼び出し元にて保証して下さい。
 *
 * (note) この関数は、mutexロックを利用します。
 *
 * @param p apr_pool_t * エラーログ用のプール
 * @param providerid const DbProviderId * DBプロバイダID
 * @return DbConn *
 * 	NULL    : アクティブなDbConnを持っていない
 * 	NULL以外: アクティブなDbConn
 */
static DbConn * _trygetDbConn_from_perm_DbConnPool(apr_pool_t *p,
						const DbProviderId *providerid)
{
	apr_status_t rv;
	DbConn *dbconn = NULL;
	DbProviderCntxt *p_ctx;
	DbConnPool *cpool = NULL;

	/* mutex ロック */
	rv = apr_thread_mutex_lock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex lock.(code = %d)", rv);
		return NULL;
	}

	/* 永続プールを取得する */
	p_ctx = _get_prv_ctx(providerid);
	if (p_ctx && p_ctx->pc_state != DB_PCNTXT_DESTROYED) { 
		cpool = p_ctx->cpool;
	}

	if (cpool && cpool->navail > 0) {
		DbConnList *list;

		/* 永続プールがあればそこからDbConnを探す */
		for (list = cpool->dbconnlist; list; list = list->next) {

			dbconn = list->dbconn;
			/*
			 * "未使用"なDbConnを探す
			 * (note)
			 * 	永続プールでは、"使用中"のDbConnを再利用
			 * 	することは出来ません。(一時プールとの違い)
			 */
			if (CPOOL_CTX(dbconn) &&
			    CPOOL_CTX(dbconn)->state == DB_CONNSTAT_UNUSE) {

				/* "使用中"に変更する */
				CPOOL_CTX(dbconn)->state = DB_CONNSTAT_INUSE;
				cpool->navail--;	/* 1つ使用中 */
				break;
			}
			dbconn = NULL;	/* 再初期化 */
		}
	}
	
	/* mutex ロックを解除 */
	rv = apr_thread_mutex_unlock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex unlock. (code = %d)", rv);
		return dbconn;	/* 大目に見る */
	}

	return dbconn;
}

/**
 * 永続プール(2次プール) にDB コネクションdbconn を戻す。
 * 永続プールが無ければ生成します。
 *
 * (note) この関数は、mutexロックを利用します。
 *
 * @param dbconn DbConn * 永続プールに格納されるDBコネクション
 */
static void _restore_DbConn_to_perm_DbConnPool(DbConn *dbconn)
{
	apr_status_t rv;
	DbConnPool *cpool;
	DbConnList *list = NULL;
	RET_STAT st;
	int is_pool_entry = 0;
	DbProviderCntxt *p_ctx;

	if (dbconn == NULL || CPOOL_CTX(dbconn) == NULL) return;

	/* mutex ロック */
	rv = apr_thread_mutex_lock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex lock.(code = %d)", rv);
		return;
	}

	/* 永続プールを取得する */
	cpool = _lookup_perm_DbConnPool(CPOOL_CTX(dbconn)->providerid);
	if (cpool == NULL) {
		/* 生成 */
		cpool = _create_perm_DbConnPool(dbconn->dbds);

		/* 登録 */
		(void) _register_perm_DbConnPool(CPOOL_CTX(dbconn)->providerid, cpool);
	}

	/*
	 * DBコネクション管理スレッドが起動していなければ起動をかける
	 */
	if (dbconn->dbds->dbpool != DIVY_DBPOOL_NOTHREAD &&
			(connmgr_state == DB_CONNMGR_ST_INIT ||
			 connmgr_state == DB_CONNMGR_ST_TAKE1 ||
			 connmgr_state == DB_CONNMGR_ST_TAKE2)) {

		/* DBコネクション管理スレッドを起動 */
		st = _execute_connnmgr(prvmng_p);
		if (st == DB_SUCCESS) {
			connmgr_state = DB_CONNMGR_ST_STARTED;	/* 起動した */
		}
		else {
			if (connmgr_state == DB_CONNMGR_ST_INIT) {
				connmgr_state = DB_CONNMGR_ST_TAKE1;	/* 1回失敗 */
			}
			else if (connmgr_state == DB_CONNMGR_ST_TAKE1) {
				connmgr_state = DB_CONNMGR_ST_TAKE2;	/* 2回失敗 */
			}
			else {	/* もう諦める */
				connmgr_state = DB_CONNMGR_ST_ABNORMAL;
				ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"The server give up to start "
					"connection-manager thread. Please check "
					"machine resource."
					"(pid = %"APR_PID_T_FMT")", getpid());
			}
		}
	}

	/*
	 * DbConn が既にプールに格納済みかどうかを調べる
	 * (note) ポインタ比較で探します(これで十分)
	 */
	is_pool_entry = 0;
	if (cpool->dbconnlist) {
		for (list = cpool->dbconnlist; list; list = list->next) {
			if (list->dbconn == dbconn) {
				is_pool_entry = 1;	/* プールに入っていた */
				break;
			}
		}
	}


	/* DBプロバイダコンテキストの取得 */
	p_ctx = _get_prv_ctx(CPOOL_CTX(dbconn)->providerid);
	if (CPOOL_CTX(dbconn)->state == DB_CONNSTAT_DISUSE ||
		(p_ctx && p_ctx->pc_state == DB_PCNTXT_DESTROYED)) {

		/* dbconn が"使用不可"またはdbds が"破棄"とマークされていた場合 */
		if (is_pool_entry) {
			/* プールから取り除く(navail をデクリメントしてはならない) */
			(void) _remove_DbConnList_from_DbConnPool(cpool, list);
		}
		/* dbconn を閉じて破棄する */
		_close_dbconn(dbconn);
	}
	else {
		/*
		 * プールに追加する
		 * (note)
		 * 	最大値に達しているかどうかはここでは見ないことにします。
		 * 	最大・最小値の確認は管理スレッドの責務だからです。
		 */
		if (!is_pool_entry) {
			/* cpoolにリストを追加する */
			_append_DbConnList_to_DbConnPool(cpool,
						_create_DbConnList(prvmng_p, dbconn));
		}

		/* DbConn のステータスを"未使用"に変更 */
		CPOOL_CTX(dbconn)->state = DB_CONNSTAT_UNUSE;
		cpool->navail++;	/* 有効なDBコネクションが1つ増えた */
	}

	/* mutex ロックを解除 */
	rv = apr_thread_mutex_unlock(cpool_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex unlock. (code = %d)", rv);
		return;
	}
}


/**
 * 永続プール(2次プール)を生成して返却する。
 *
 * @param dbds DbDataSource * 初期化されたDBプロバイダへのポインタ
 * @return DbConnPool * 生成した一時プールへのポインタ
 */
static DbConnPool * _create_perm_DbConnPool(DbDataSource *dbds)
{
	DbConnPool *cpool = apr_pcalloc(prvmng_p, sizeof(DbConnPool));

	cpool->pooltype   = DB_PTYPE_PERMANENT_POOL;
	cpool->nmin       = dbds->dbminspareconn;
	cpool->nmax       = dbds->dbmaxspareconn;
	cpool->dbconnlist = NULL;
	cpool->nelts      = 0;
	cpool->navail     = 0;

	return cpool;
}

/**
 * providerid が示す永続プール(2次プール)を取得する。
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param p apr_pool_t *
 * @param providerid const DbProviderId * DBプロバイダID
 * @return DbConnPool * 一時プールへのポインタ。取得に失敗したらNULL
 */
static DbConnPool * _lookup_perm_DbConnPool(const DbProviderId *providerid)
{
	DbProviderCntxt *p_ctx;

	if (providerid == NULL) return NULL;

	/* DBプロバイダコンテキストの取得 */
	p_ctx = _get_prv_ctx(providerid);

	return (p_ctx != NULL) ? p_ctx->cpool : NULL;
}

/**
 * cpool を永続プール(2次プール) としてproviderid で登録する。
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param providerid const DbProviderId * DBプロバイダID
 * @param cpool DbConnPool * 永続プールへのポインタ
 * @return RET_STAT 処理ステータス
 * 	DB_SUCCESS: 成功
 * 	DB_ERROR  : 失敗
 */
static RET_STAT _register_perm_DbConnPool(const DbProviderId *providerid,
						DbConnPool *cpool)
{
	DbProviderCntxt *p_ctx;

	if (providerid == NULL || cpool == NULL) {
		return DB_ERROR;
	}

	/* DBプロバイダコンテキストを取得する */
	p_ctx = _get_prv_ctx(providerid);
	if (p_ctx == NULL) {
		return DB_ERROR;
	}

	/* DBプロバイダコンテキストにcpoolを設定する */
	p_ctx->cpool = cpool;
	return DB_SUCCESS;
}

/**
 * 存在する全ての永続プールを破棄し、格納していたコネクションを切断する。
 * (note)
 * 	この関数が同時にコールされないよう呼び出し元で保証して下さい。
 */
static void _destroy_all_perm_DbConnPool()
{
	apr_hash_index_t *hi1, *hi2;
	DbProviderGrpCntxt *grp_ctx;
	DbProviderCntxt *p_ctx;
	DbConnList *list;
	const void *dummy;

	if (dbprv_grp_hash == NULL || apr_hash_count(dbprv_grp_hash) == 0) {
		return;
	}

	/*
	 * 全てのコネクションプールを取り出し、
	 * 使用中ではないコネクションを切断する
	 */
	for (hi1 = apr_hash_first(prvmng_p, dbprv_grp_hash);
						hi1; hi1 = apr_hash_next(hi1)) {
		/* DBプロバイダグループコンテキストの取得 */
		apr_hash_this(hi1, &dummy, NULL, (void**) &grp_ctx);
		if (grp_ctx == NULL) continue;

		for (hi2 = apr_hash_first(prvmng_p, grp_ctx->dbprv_hash);
						hi2; hi2 = apr_hash_next(hi2)) {
			/* DBプロバイダコンテキストの取得 */
			apr_hash_this(hi2, &dummy, NULL, (void**) &p_ctx);
			if (p_ctx == NULL || p_ctx->cpool == NULL) continue;

			for (list = p_ctx->cpool->dbconnlist; list; list = list->next) {
				if (CPOOL_CTX(list->dbconn)->state != DB_CONNSTAT_INUSE) {
					/* close する */
					_close_dbconn(list->dbconn);
					CPOOL_CTX(list->dbconn)->state = DB_CONNSTAT_DISUSE;
				}
			}
		}
	}
}

/**
 * DBコネクション管理用スレッドを起動する。
 * (note)
 * 	同時にコールされないよう呼び出し元にて保証して下さい。
 *
 * @param owner_p apr_pool_t * 起動するスレッドと同じライフサイクルを持つプール
 * @return RET_STAT 起動に成功したかどうか
 * 	DB_SUCCESS: 成功
 * 	DB_ERROR  : 失敗
 */
static RET_STAT _execute_connnmgr(apr_pool_t *owner_p)
{
	apr_status_t rv;
	DbConnMgrCntxt *th_ctx;

	/* 2度呼び禁止 */
	if (connmgr_th || connmgr_state == DB_CONNMGR_ST_STARTED) {
		return DB_SUCCESS;	/* もう起動していますよ */
	}

	/*
	 * スレッド専用のコンテキスを生成
	 */
	th_ctx = apr_pcalloc(owner_p, sizeof(DbConnMgrCntxt));
	th_ctx->owner_p = owner_p;

	/*
	 * スレッド属性の作成
	 * (note) スレッドの属性
	 *	ノンデタッチ (join 可能)属性を付けて生成します。
	 */
	apr_threadattr_create(&th_ctx->attr, owner_p);
	apr_threadattr_detach_set(th_ctx->attr, 0 /* join 可能 */);

	/*
	 * スレッドの作成と起動
	 * (note)
	 * 	スレッドが内部的に使用するプールはperm_pのサブプールです。
	 */
	rv = apr_thread_create(&connmgr_th, th_ctx->attr,
				_run_connmgr, th_ctx, owner_p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to execute DB connection management thread."
			"(code = %d)", rv);
		connmgr_th = NULL;
		return DB_ERROR;
	}

	return DB_SUCCESS;
}

/*-----------------------------------------------------------------------------
 * Define DbConn manager functions
 -----------------------------------------------------------------------------*/
/**
 * DBコネクションの管理を行うスレッドのハンドラ関数
 * (機能)
 *	(1) DBコネクションプールの最大値をチェック
 * 	(2) DBコネクションのリフレッシュ期間をチェック
 * 	(3) DBコネクションがアクティブであるかどうかをチェック
 * 	(4) 不要になったDBコネクションを閉じて破棄する
 *	(5) DBコネクションを適正値まで閉じて破棄する
 *	(6) DBコネクションを適正値まで作成する
 * (note)
 * 	上記の処理をDB_CONNMGR_PROCESS_INTERVAL[sec]毎に実施しますが
 * 	前の処理に時間がかかった場合、余計に掛かった時間分だけ後ろに
 * 	シフトしていきます。
 * (note)
 * 	この関数は、APRに存在しないネイティブな機能(pthread)を使っています。
 * 	プラットフォームの変更には注意が必要です。
 *
 * @param thd apr_thread_t * このハンドラを起動したスレッドの識別子
 * @param data void *	DbConnMgrCntxt へのポインタが入っています
 */
static void * APR_THREAD_FUNC _run_connmgr(apr_thread_t *thd, void *data)
{
	apr_status_t rv;
	int ret;

	DbConnMgrCntxt *ctx    = (DbConnMgrCntxt *) data; /* 専用コンテキスト */
	apr_pool_t *wk_p       = NULL;
	pid_t pid              = getpid();	/* このスレッドのLWP 番号 */
	DbMaintenanceCntx mctx = { 0 };		/* メンテナンスリスト */
	apr_interval_time_t pollingtm =
				apr_time_from_sec(DB_CONNMGR_PROCESS_INTERVAL);

	/* 作業用に使用する専用のプールを生成 */
	apr_pool_create(&wk_p, ctx->owner_p);
	apr_pool_tag(wk_p, "tf_dbcpool_thread_work");


	/* 起動メッセージを表示 */
	ERRLOG1(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"connection manager started (%"APR_PID_T_FMT ")", pid);

	/*
	 * このスレッドが全てのシグナルを受け付けないようにする
	 * (note)
	 * 	シグナルを受けたときに、親のシグナルハンドラを実行してしまわない
	 * 	ようにするための措置です。Apacheの停止手順以外では止まらなく
	 * 	なりますが、それは正当です。
	 */
	rv = apr_setup_signal_thread();
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to setup signal thread.(code = %d)", rv);
	}

	/* 終了フラグが立たない間、以下を実行する */
	while (!exit_connmgr_th) {

		/* クリーンアップハンドラの登録 */
		DAV_DIVY_THREAD_CLEANUP_PUSH(_cleanup_thread_mutex, cpool_mutex);

		/* mutex ロック */
		rv = apr_thread_mutex_lock(cpool_mutex);
		if (rv != APR_SUCCESS) {
			ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to mutex lock.(code = %d)", rv);
			break;	/* どうしようもありません */
		}

		/*
		 * (0) DB_CONNMGR_PROCESS_INTERVAL [sec] だけwaitする
		 * (note)
		 * 	apr_sleepよりも軽いのでwaitを使います。
		 */
		rv = apr_thread_cond_timedwait(cpool_cond, cpool_mutex, pollingtm);
		if (rv != APR_TIMEUP) {

			/* (note)
			 * 	apr_thread_cond_timedwait において起き得るエラーは
			 * 	シグナルにより割り込まれた場合だけであるが、
			 * 	シグナルはブロックしているのでここにくるのは非常に
			 * 	稀である。
			 */
			ERRLOG1(NULL, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_OS,
				"cond_timewait is abort. (code = %d)", rv);
			break;	/* もう何も出来ない */
		}

		/* メンテナンス回数カウントをインクリメント */
		mctx.exec_cnt++;

		/* 作業用プールのリフレッシュ間隔を調べる */
		if (mctx.exec_cnt % DB_CONNMGR_POOLREFRESH_CYCLE == 0) {
			apr_pool_clear(wk_p);
#ifdef DIVY_DBCPOOL_DEBUG
		ERRLOG0(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"refresh working pool");
#endif	/* DIVY_DBCPOOL_DEBUG */
		}

		/*
		 * メンテナンスが必要なDbConnの集合を取得する
		 */
		ret = _find_maintenance(wk_p, &mctx);

		/* mutex ロックを解除 */
		rv = apr_thread_mutex_unlock(cpool_mutex);

		/* 何より先にクリーンアップハンドラの解除 */
		DAV_DIVY_THREAD_CLEANUP_POP(0);

		/* mutex unlock のステータスチェック */
		if (rv != APR_SUCCESS) {
			ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to mutex unlock. (code = %d)", rv);
			break;	/* どうしようもありません */
		}
		if (exit_connmgr_th) break;


		/* メンテナンスが必要な時だけ実施 */
		if (ret == DB_MAINTENANCE_NEED) {

			/* メンテナンス */
			_do_maintenance(wk_p, &mctx);
			if (exit_connmgr_th) break;
		}
		else {
			continue;	/* see you.. */
		}

		/* クリーンアップハンドラの登録 */
		DAV_DIVY_THREAD_CLEANUP_PUSH(_cleanup_thread_mutex, cpool_mutex);

		/* mutex ロック */
		rv = apr_thread_mutex_lock(cpool_mutex);
		if (rv != APR_SUCCESS) {
			ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to mutex lock.(code = %d)", rv);
			break;	/* どうしようもありません */
		}

		/*
		 * ロックと共に実施しなければならないメンテナンス処理の実施
		 */
		_do_maintenance_with_lock(wk_p, &mctx);

		/*
		 * 結果をプールに書き戻す
		 */
		_restore_maintenance(wk_p, &mctx);

		/* mutex ロックを解除 */
		rv = apr_thread_mutex_unlock(cpool_mutex);

		/* 何より先にクリーンアップハンドラの解除 */
		DAV_DIVY_THREAD_CLEANUP_POP(0);

		/* mutex unlock のステータスチェック */
		if (rv != APR_SUCCESS) {
			ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
				"Failed to mutex unlock. (code = %d)", rv);
			break;
		}
	}

	/* 終了メッセージを表示 */
	ERRLOG1(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"connection manager end (%"APR_PID_T_FMT ")", pid);

	/*
	 * スレッドを終了する
	 * (note)
	 * 	このスレッドが持っているプール(apr_thread_createした時に
	 * 	渡すプールのサブプール) が直接・間接的にpchild のサブプール
	 * 	であれば、apr_thread_exit関数を呼び出してはなりません。
	 * 	理由は以下の通り。
	 *
	 * 	・Childプロセス終了時には、最下層にあるサブプールから順番に
	 * 	  破棄されていく(poolのdestroyのシーケンス)
	 * 	・TeamFileはpchild のpoolのdestroyシーケンスをトリガとして
	 * 	  動作する
	 * 	・apr_thread_exit はスレッド専用に持っていたサブプール
	 * 	  (apr_thread_create で渡されたプールのサブプール)を破棄する
	 *
	 * 	以上により、pchildのサブプールであるときには、apr_thread_exit
	 * 	が呼び出される前に、スレッドが持っているサブプール(pchild の
	 * 	最下層になります) は破棄されてしまいます。
	 * 	ところが、apr_thread_exitは内部で自身のサブプールの破棄を
	 * 	実施しますので、これがダングリングポインタのfreeとなり、Child
	 * 	プロセスが最後にCoreDumpするという現象が発生します。
	 * 	どうしてもpchild のサブプールにしたいのならば pthread_exit(NULL)
	 * 	をapr_thread_exitの代わりに記述してください。
	 */
	apr_thread_exit(thd, APR_SUCCESS);

	return NULL;
}

/**
 * DBコネクション管理スレッドがメンテナンスしなければならないDbConnの
 * 集合を検索して返却する。
 *
 * (機能)
 * 	(1) DBMS接続情報変更チェック
 *	(2) DBコネクションプールの最大値をチェック
 * 	(3) DBコネクションのリフレッシュ期間をチェック
 * 	(4) DBコネクションがアクティブであるかどうかをチェック
 * 	(5) DBコネクションプールの最小値をチェック
 *
 * (note) 競合した時の処理優先順位
 *	DBMS接続情報チェック > リフレッシュチェック
 *				> 最大値 > アクティブチェック > 最小値
 *
 * 	例えば、リフレッシュチェックするコネクションの数が2個で
 * 	アクティブチェックする数が3個の場合、アクティブチェック対象は
 *	既存のコネクション(3-1) 個に対してアクティブチェックを実施
 *	します。
 * (note)
 * 	同期アクセスを呼び出し側で保障してください。
 *
 * @param p apr_pool_t * DBコネクション管理スレッドとライフサイクルを共にするプール
 * @param mctx DbMaintenanceCntx * 検索して見つかったDbConnを格納するコンテキスト
 * @return int メンテナンスすべきリストが見つかったかどうか
 * 	DB_MAINTENANCE_NEED   : メンテナンスの必要がある(リストが見つかった)
 * 	DB_MAINTENANCE_NONEED : メンテナンスの必要がない(リストがなかった)
 * 	DB_MAINTENANCE_ERR    : エラーが見つかった
 */
static int _find_maintenance(apr_pool_t *p, DbMaintenanceCntx *mctx)
{
	DbConnPool *cpool;
	DbProviderGrpCntxt *grp_ctx;
	DbProviderCntxt *p_ctx;
	DbConnCreateList *l_create = NULL;
	DbConnList       *l_delete = NULL;
	DbConnList       *l_actchk = NULL;
	DbDbmsInfo       *dbmsinfo = NULL;
	apr_hash_index_t *hi1, *hi2;
	apr_time_t now;
	apr_int32_t n_refresh = 0, n_refreshed = 0;
	apr_int32_t n_actchk  = 0;
	apr_int32_t n_delete  = 0;
	apr_int32_t n_create  = 0;
	DbConnList *list, *nextlist;
	DbDataSource *dbds;
	const void *dummy;
	int exec_reposdb = (mctx->exec_cnt % DB_CONNMGR_CHKDBMSINFO_CYCLE == 0) ? 1: 0;

	if (mctx == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"mctx is NULL or empty.");
		return DB_MAINTENANCE_ERR;
	}

	/* 各種変数の初期化 */
	mctx->create_list   = NULL;
	mctx->delete_list   = NULL;
	mctx->actchk_list   = NULL;
	mctx->dbmsinfo_hash = NULL;

	now = apr_time_now();	/* 現時点の時刻を取得 */

	/* 存在する全プールを探す */
	for (hi1 = apr_hash_first(p, dbprv_grp_hash); hi1; hi1 = apr_hash_next(hi1)) {

		/* DBプロバイダグループコンテキストの取得 */
		apr_hash_this(hi1, &dummy, NULL, (void**) &grp_ctx);
		if (grp_ctx == NULL) continue;

		/* DBMS 情報の変更確認を実施する場合 */
		if (exec_reposdb) {

			/* リポジトリDBのプロバイダコンテキストを取得 */
			p_ctx = _get_prv_ctx(grp_ctx->repos_providerid);

			cpool = (p_ctx != NULL) ? p_ctx->cpool : NULL;
			list  = (cpool != NULL) ? cpool->dbconnlist : NULL;
			while (list != NULL) {
				if (CPOOL_CTX(list->dbconn)->state == DB_CONNSTAT_UNUSE) {
					/* このリストを外す */
					_remove_DbConnList_from_DbConnPool(cpool, list);
					if (mctx->dbmsinfo_hash == NULL) {
						mctx->dbmsinfo_hash = apr_hash_make(p);
					}

					dbmsinfo = apr_pcalloc(p, sizeof(DbDbmsInfo));
					dbmsinfo->repos_alist = list;
					/* グループIDをキーとしてdbmsinfoを格納 */
					apr_hash_set(mctx->dbmsinfo_hash,
							grp_ctx->provider_gid,
							APR_HASH_KEY_STRING, dbmsinfo);
					cpool->navail--;
#ifdef DIVY_DBCPOOL_DEBUG
	ERRLOG2(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"dbms(%s, 0x%pp)", list->dbconn->dbds->dbmsname, (void *)list->dbconn);
#endif	/* DIVY_DBCPOOL_DEBUG */
					break;
				}
				list = list->next;	/* 次へ */
			}
		}

		for (hi2 = apr_hash_first(p, grp_ctx->dbprv_hash);
					hi2; hi2 = apr_hash_next(hi2)) {

			/* DBプロバイダコンテキストの取得 */
			apr_hash_this(hi2, &dummy, NULL, (void**) &p_ctx);
			if (p_ctx == NULL || p_ctx->cpool == NULL) {
				continue;
			}
			cpool = p_ctx->cpool;
			dbds  = p_ctx->dbds;

			/* (3) リフレッシュ対象チェックの数 */
			if (cpool->navail > 1) {
				n_refresh = (cpool->navail * DB_CONNMGR_DBREFRESH_RATE) / 100;
				n_refresh = MIN_VAL(MAX_VAL(n_refresh, 1), cpool->navail);
			}
			else {
				n_refresh = cpool->navail;	/* 0 or 1 */
			}

#ifdef DIVY_DBCPOOL_DEBUG
	ERRLOG2(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"(nelts,navail)=(%d,%d)", cpool->nelts, cpool->navail);
#endif	/* DIVY_DBCPOOL_DEBUG */

			/*
			 * リフレッシュ対象を探す
			 */
			list = cpool->dbconnlist;	/* 処理対象リストの先頭 */
			n_refreshed = 0;		/* リフレッシュされた数 */
			while (list != NULL && n_refresh > 0) {

				/* (2) 未使用でリフレッシュ期間を過ぎていた場合 */
				if (CPOOL_CTX(list->dbconn)->state == DB_CONNSTAT_UNUSE &&
					now > dbrefreshinterval +
						CPOOL_CTX(list->dbconn)->creationtm) {
#ifdef DIVY_DBCPOOL_DEBUG
	ERRLOG2(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"refresh(%s, 0x%pp)", list->dbconn->dbds->dbmsname, (void *)list->dbconn);
#endif	/* DIVY_DBCPOOL_DEBUG */
					/* 破棄対象をプールから外して削除リストに入れておく */
					nextlist = _remove_DbConnList_from_DbConnPool(cpool, list);
					if (mctx->delete_list == NULL) {
						mctx->delete_list = l_delete = list;
					}
					else {
						l_delete->next = list;
						l_delete = l_delete->next;
					}
					cpool->navail--;
					n_refresh--;
					n_refreshed++;
					l_delete->next = NULL;
					list = nextlist;        /* 次のループのために元に戻す */
				}
				else {
					/* 使用中だった、リフレッシュ期間を過ぎていなかった */
					list = list->next;
				}
			}

			/*
			 * リフレッシュ動作の結果を受けて、アクティブチェック数
			 * 最大値リミット、最小値リミット数を算出する
			 */

			/* (2) 最大値リミットチェック */
			n_delete = MIN_VAL(MAX_VAL(cpool->nelts - cpool->nmax, 0), cpool->navail);

			/* (4) アクティブチェック(最大値、リフレッシュ分を省く) */
			if (cpool->navail > 1) {
				n_actchk  = (cpool->navail * DB_CONNMGR_CHKCONN_RATE) / 100;
				n_actchk  = MIN_VAL(MAX_VAL(n_actchk, 1), cpool->navail);
			}
			else {
				n_actchk  = cpool->navail;
			}
			n_actchk = MIN_VAL(MAX_VAL(n_actchk - n_refreshed - n_delete, 0), cpool->navail);

			/* (5) 最小値リミットチェック
			 *    アクティブチェックは'成功する'だろうと仮定してしまいます */
			if (n_delete == 0) {
				n_create = MAX_VAL(cpool->nmin - cpool->nelts, 0);
				/* リポジトリDBプロバイダの場合 */
				if (exec_reposdb && p_ctx->is_reposdb) {
					/* DBMS接続情報チェックに回った分を考慮 */
					n_create--;
				}

				if (n_create > 0 && dbds) {
					/* dbdsへのポインタを別の場所にコピーする */
					if (mctx->create_list == NULL) {
						mctx->create_list = l_create =
							apr_pcalloc(p, sizeof(DbConnCreateList));
					}
					else {
						l_create->next =
							apr_pcalloc(p, sizeof(DbConnCreateList));
						l_create = l_create->next;
					}
					l_create->dbds       = dbds;
					l_create->n_create   = n_create;
					l_create->providerid = p_ctx->providerid;
					l_create->cnlist     = NULL;
					l_create->next       = NULL;
				}
			}

			list = cpool->dbconnlist;	/* 処理対象リストの先頭 */
			while (list != NULL && (n_delete > 0 || n_actchk > 0)) {

				/* 未使用だけが対象 */
				if (CPOOL_CTX(list->dbconn)->state != DB_CONNSTAT_UNUSE) {
					list = list->next;
					continue;
				}

				/* (2) 最大値を超えていた場合 */
				if (n_delete > 0) {
#ifdef DIVY_DBCPOOL_DEBUG
	ERRLOG2(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"max(%s, 0x%pp)", list->dbconn->dbds->dbmsname, (void *)list->dbconn);
#endif	/* DIVY_DBCPOOL_DEBUG */
					/* プールから外して破棄対象を記録しておく */
					nextlist = _remove_DbConnList_from_DbConnPool(cpool, list);
					if (mctx->delete_list == NULL) {
						mctx->delete_list = l_delete = list;
					}
					else {
						l_delete->next = list;
						l_delete = l_delete->next;
					}
					cpool->navail--;
					n_delete--;
					l_delete->next = NULL;
					list = nextlist;        /* 次のループのために元に戻す */
				}
				/* (4) アクティブチェック対象が存在した */
				else if (n_actchk > 0) {

					/* プールから外してアクティブチェック対象を記録しておく */
					nextlist = _remove_DbConnList_from_DbConnPool(cpool, list);
					if (mctx->actchk_list == NULL) {
						mctx->actchk_list = l_actchk = list;
					}
					else {
						l_actchk->next = list;
						l_actchk = l_actchk->next;
					}

					cpool->navail--;
					n_actchk--;
					l_actchk->next = NULL;
					list = nextlist;	/* 次のループのために元に戻す */
				}
				else {
					list = list->next;
				}
			}
		}
	}

	/* メンテナンスリストは存在していたか? */
	if (mctx->create_list != NULL || mctx->delete_list != NULL ||
	    mctx->actchk_list != NULL || mctx->dbmsinfo_hash != NULL) {
		return DB_MAINTENANCE_NEED;	/* メンテナンスが必要 */
	}

	return DB_MAINTENANCE_NONEED;	/* メンテナンスは不要 */
}

/**
 * DBコネクション管理スレッドがメンテナンスしなければならないDbConnの集合を
 * 処理する。
 *
 * (機能)
 * 	(1) DBMS情報の取得とダイジェストの生成
 * 	(2) 不要になったDBコネクションを閉じて破棄する
 *	(3) DBコネクションを適正値まで閉じて破棄する
 *	(4) DBコネクションを適正値まで作成する
 *
 * @param p apr_pool_t * 作業用のプール
 * @param mctx DbMaintenanceCntx * 処理対象のDbConnを格納するコンテキスト
 */
static void _do_maintenance(apr_pool_t *p, DbMaintenanceCntx *mctx)
{
	DbConnList *list;
	DbConn *dbconn;
	DbDataSource *dbds;
	int ret;
	apr_int32_t n_create, i;
	DbConnCreateList *cctx;
	DbDbmsInfo *dbmsinfo;
	divy_rdbo_dbms *dbms_pr;
	const char *conn_digest;
	apr_hash_index_t *hi;
	const void *dummy;

	/*
	 * 破棄
	 */
	for (list = mctx->delete_list; list; list = list->next) {
		/* dbconn のclose */
		_close_dbconn(list->dbconn);
		if (exit_connmgr_th) return;	/* 終了チェック */
	}

	/*
	 * アクティブチェック
	 */
	for (list = mctx->actchk_list; list; list = list->next) {
#ifdef DIVY_DBCPOOL_DEBUG
		ERRLOG2(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"actchk(%s, 0x%pp)", list->dbconn->dbds->dbmsname,
			(void *) list->dbconn);
#endif	/* DIVY_DBCPOOL_DEBUG */

		/* DbConn のアクティブチェック */
		ret = divy_rdbo_validate_activedbconn(p, list->dbconn);
		if (ret) {
			/* 非アクティブならば捨てておく */
			CPOOL_CTX(list->dbconn)->state = DB_CONNSTAT_DISUSE;
			_close_dbconn(list->dbconn);
		}
		if (exit_connmgr_th) return;	/* 終了チェック */
	}

	/*
	 * 新規作成
	 */
	for (cctx = mctx->create_list; cctx; cctx = cctx->next) {
#ifdef DIVY_DBCPOOL_DEBUG
		ERRLOG2(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"create%d(%s)", cctx->n_create, cctx->dbds->dbmsname);
#endif	/* DIVY_DBCPOOL_DEBUG */
		n_create = cctx->n_create;
		dbds     = cctx->dbds;
		cctx->n_create = 0;
		for (i = 0; i < n_create; i++) {
			/* DbConn の取得 */
			dbconn = dbds->getDbConn(dbds, prvmng_p);
			if (dbconn->getCode(dbconn) != DB_SUCCESS) {
				ERRLOG2(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get DbConn. (providerName = %s) "
					"Reason: %s", dbds->dbmsname,
					REPLACE_NULL(dbconn->getMsg(dbconn)));
				if (dbconn != NULL) dbconn->close(dbconn);
				dbconn = NULL;
				continue;
			}
			/* 管理用コンテキストの付加 */
			CPOOL_CTX(dbconn) = _create_cpoolctx(prvmng_p,
							cctx->providerid,
							DB_PTYPE_PERMANENT_POOL);
			if (cctx->cnlist == NULL) {
				cctx->cnlist = list =
					_create_DbConnList(prvmng_p, dbconn);
			}
			else {
				_append_DbConnList(list,
						_create_DbConnList(prvmng_p, dbconn));
			}
			cctx->n_create++;
		}
		if (exit_connmgr_th) return;	/* 終了チェック */
	}

	/* dbmsinfo_hash がなければ何もする必要なし */
	if (mctx->dbmsinfo_hash == NULL || apr_hash_count(mctx->dbmsinfo_hash) == 0) {
		return;
	}

	/*
	 * DBMS接続情報の取得とダイジェストの作成
	 */
	for (hi = apr_hash_first(p, mctx->dbmsinfo_hash);
						hi; hi = apr_hash_next(hi)) {
		/* DBMS接続情報を取得する */
		apr_hash_this(hi, &dummy, NULL, (void**) &dbmsinfo);

		dbms_pr = NULL;
		dbconn  = dbmsinfo->repos_alist->dbconn;
		ret = divy_rdbo_get_dbmsinfo(p, dbconn, &dbms_pr);
		if (ret) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get db provider information from repos db."
				"(providerName = %s)", dbconn->dbds->dbmsname);
			continue;
		}

		if (dbms_pr == NULL) continue;
		if (strcmp(dbms_pr->dbmsid, DIVY_REPOS_DBMSID) == 0) {
			continue;	/* リポジトリDB用擬似エントリは含めない
						   (新規情報は絶対にないから、これからチェックする必要なし) */
		}

		/* ダイジェスト用ハッシュの生成 */
		dbmsinfo->digest_hash = apr_hash_make(p);
		for (; dbms_pr; dbms_pr = dbms_pr->next) {
			/* ダイジェストを作成 */
			conn_digest = _create_conn_digest(p,
						DIVY_DB_DATA_DBMS, dbms_pr);

			/* ダイジェストとdbms_pr をハッシュに格納 */
			apr_hash_set(dbmsinfo->digest_hash, conn_digest,
					APR_HASH_KEY_STRING, dbms_pr);
		}
	}
}

/**
 * mutex ロックと共にDBコネクション管理スレッドがメンテナンスしなければならない
 * DbConnの集合を処理する。
 * (note) 機能
 * 	(1) DBMS 接続情報が変更されているかどうか
 * (note)
 * 	同期化は呼び出し元で保証して下さい。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param mctx DbMaintenanceCntx * 処理対象のDbConnを格納するコンテキスト
 */
static void _do_maintenance_with_lock(apr_pool_t *p, DbMaintenanceCntx *mctx)
{
	DbProviderGrpCntxt *grp_ctx;
	DbProviderCntxt *p_ctx, *new_p_ctx;
	apr_hash_index_t *hi, *hi2;
	divy_rdbo_dbms *dbms_pr;
	const char *provider_gid;
	const void *dummy;
	DbDbmsInfo *dbmsinfo;
	DbDataSource *dbds, *repos_dbds;
	const DbProviderId *providerid;
	DbConnList *list;
	int ret;

	if (mctx == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"mctx is NULL or empty.");
		return;
	}

	if (mctx->dbmsinfo_hash == NULL || apr_hash_count(mctx->dbmsinfo_hash) == 0) {
		return;	/* 以下を実施する必要なし */
	}

	/* dbmsinfo_hash をコピーしてから使用する */

	/* 取得した全DBMS 情報を順番に取得する */
	for (hi = apr_hash_first(p, mctx->dbmsinfo_hash); hi; hi = apr_hash_next(hi)) {
		/* DBMS 情報の取得 */
		apr_hash_this(hi, (const void **) &provider_gid,
						NULL, (void**) &dbmsinfo);
		/* 接続情報が取れていなかった場合 */
		if (dbmsinfo == NULL || dbmsinfo->digest_hash == NULL) continue;

		/* DBプロバイダグループコンテキストの取得 */
		grp_ctx = _get_prvgrp_ctx(provider_gid);
		if (grp_ctx == NULL) continue;

		/* グループコンテキスト内のプロバイダを順番に調べる */
		for (hi2 = apr_hash_first(p, grp_ctx->dbprv_hash); hi2;
							hi2 = apr_hash_next(hi2)) {

			/* リポジトリDBプロバイダ以外のDBプロバイダコンテキストの取得 */
			apr_hash_this(hi2, &dummy, NULL, (void**) &p_ctx);
			if (p_ctx == NULL || p_ctx->is_reposdb ||
				p_ctx->pc_state == DB_PCNTXT_DESTROYED) {
				continue;
			}

			/* conn_digest が等しいdbms_pr は存在したか？ */
			dbms_pr = apr_hash_get(dbmsinfo->digest_hash,
						p_ctx->conn_digest, APR_HASH_KEY_STRING);
			/* 同じだった --> プロバイダ名 & 接続情報は同じだった */
			if (dbms_pr != NULL) {
				/* 何も変わっていないので後始末だけ行う */
				apr_hash_set(dbmsinfo->digest_hash, p_ctx->conn_digest,
							APR_HASH_KEY_STRING, NULL);
			}
			/* 異なっていた --> 同じプロバイダ名 & 接続情報のエントリなし */
			else {
				/* 今まで持っていたp_ctx を破棄する */
				p_ctx->pc_state = DB_PCNTXT_DESTROYED;

				if (p_ctx->cpool == NULL) continue;

				/* 不要になったDbConnを閉じる */
				for (list = p_ctx->cpool->dbconnlist;
							list; list = list->next) {
					/* 使用中のコネクションはスキップ */
					if (CPOOL_CTX(list->dbconn)->state ==
								DB_CONNSTAT_INUSE) {
						continue;
					}
					/* close する */
					_close_dbconn(list->dbconn);
					CPOOL_CTX(list->dbconn)->state = DB_CONNSTAT_DISUSE;
				}
			}
		}

		/* 既に処理済になっているか ? */
		if (apr_hash_count(dbmsinfo->digest_hash) == 0) continue;

		/*
		 * 新しく増えたDBMS接続情報が無いかどうか検証する
		 */
		repos_dbds = dbmsinfo->repos_alist->dbconn->dbds;
		for (hi2 = apr_hash_first(p, dbmsinfo->digest_hash);
						hi2; hi2 = apr_hash_next(hi2)) {

			/* DBMS接続情報の取得 */
			apr_hash_this(hi2, &dummy, NULL, (void**) &dbms_pr);
			if (dbms_pr == NULL) continue;	/* 処理済み */

			dbds = NULL;
			
			/* 新しいDBMSを登録する */
			ret = divy_run_create_dbdatasource(prvmng_p, dbms_pr->type, &dbds);
			if (ret == DB_DECLINED) {
				/* リポジトリDBにエントリはあったが、プロバイダモジュールが
				 * なかった場合 */
				ERRLOG1(NULL, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to load db provider(type = %s)."
					"If this provider was not used, Please ignore.",
					dbms_pr->type);
				continue;
			}

			/* DBプロバイダのライセンスを確認する */
			if (apr_hash_get(grp_ctx->loadable_type_set, dbms_pr->type,
						APR_HASH_KEY_STRING) == NULL) {
				ERRLOG1(NULL, APLOG_WARNING, DIVY_FST_CERR + DIVY_SST_DATA,
					"This db provider did not have valid license. "
					"If this provider was not used, Please ignore."
					"(type = %s)", dbms_pr->type);
				continue;
			}

			/* dbms_pr の値をdbds にコピーする */ 
			dbds->dbmstype       = apr_pstrdup(prvmng_p, dbms_pr->type);
			dbds->dbmsname       = apr_pstrdup(prvmng_p, dbms_pr->name);
			dbds->hostname       = apr_pstrdup(prvmng_p, dbms_pr->hostname);
			dbds->hostport       = apr_itoa(prvmng_p,    dbms_pr->port);
			dbds->dbname         = apr_pstrdup(prvmng_p, dbms_pr->dbname);
			dbds->username       = apr_pstrdup(prvmng_p, dbms_pr->username);
			dbds->password       = apr_pstrdup(prvmng_p, dbms_pr->password);
			if (USING_CPOOL(repos_dbds->dbpool)) {
				/* (note) 歴史的(?)な経緯からDB上のフラグは0(off)/1(on)
				 * 	なので、それを置き換えておく必要がある */
				dbds->dbpool = (dbms_pr->dbpool) ? DIVY_DBPOOL_ON : DIVY_DBPOOL_OFF;
			}
			else {
				dbds->dbpool = DIVY_DBPOOL_OFF;	/* 常にoff */
			}
			dbds->dbminspareconn = GET_MIN_SPARECONN(dbms_pr->dbminspareconn);
			dbds->dbmaxspareconn = GET_MAX_SPARECONN(dbms_pr->dbmaxspareconn);
			dbds->isFinishedInitEnv = DBENV_INIT_FINISHED;

			/* DBプロバイダIDの生成 */
			providerid = _create_db_providerid(prvmng_p, provider_gid,
								dbds->dbmsname);

			/* DBプロバイダコンテキストを新規作成 */
			new_p_ctx = _create_prv_ctx(prvmng_p, providerid, 0, dbds);

			/* DBプロバイダグループコンテキストに登録 */
			if (_register_db_provider(grp_ctx, new_p_ctx) != DB_SUCCESS) {
				continue;
			}
		}
	}
}

/**
 * mctx に格納されたDbConnのリストをDBコネクションプールに書き戻す。
 * (note)
 * 	同期化は呼び出し元で保証して下さい。
 *
 * @param p apr_pool_t *
 * @param mctx DbMaintenanceCntx *
 */
static void _restore_maintenance(apr_pool_t *p, DbMaintenanceCntx *mctx)
{
	DbConnList *list, *nextlist;
	DbConnCreateList *cctx;
	DbProviderCntxt *p_ctx;
	DbConnPool *cpool;
	const void *dummy;
	DbDbmsInfo *dbmsinfo;

	/*
	 * リポジトリDBのDbConnをプールに戻すこと
	 */

	/*
	 * 新規作成したDbConnをプールに格納する
	 */
	for (cctx = mctx->create_list; cctx; cctx = cctx->next) {
		/* DBプロバイダコンテキストを取得する */
		p_ctx = _get_prv_ctx(cctx->providerid);
		if (p_ctx == NULL || p_ctx->pc_state == DB_PCNTXT_DESTROYED) {
			continue;
		}

		cpool = p_ctx->cpool;

		/* プールに格納 */
		_append_DbConnList_to_DbConnPool(cpool, cctx->cnlist);
		cpool->navail+= cctx->n_create;

		if (exit_connmgr_th) return;	/* 終了チェック */
	}

	/*
	 * アクティブだったDbConnをプールに戻す
	 * (note) actlist には色々なDBMSのDbConnListが入っているので
	 * 	ひとつひとつを単独でcpool にappend していく必要がある
	 */
	list = mctx->actchk_list;
	while (list != NULL) {
		if (CPOOL_CTX(list->dbconn)->state == DB_CONNSTAT_DISUSE) {
			list = list->next;
			continue;	/* 非アクティブは無視する */
		}

		/* DBプロバイダコンテキストを取得する */
		p_ctx = _get_prv_ctx(CPOOL_CTX(list->dbconn)->providerid);
		if (p_ctx == NULL || p_ctx->pc_state == DB_PCNTXT_DESTROYED) {
			list = list->next;
			continue;
		}	

		cpool = p_ctx->cpool;

		nextlist = list->next;	/* 次のリストをswap */
		list->next = NULL;
		list->prev = NULL;

		/* プールに戻す */
		_append_DbConnList_to_DbConnPool(cpool, list);
		cpool->navail++;
		list = nextlist;	/* 戻す */

		if (exit_connmgr_th) return;	/* 終了チェック */
	}

	/*
	 * 使い終わったリポジトリDBのコネクションを戻す
	 */
	if (mctx->dbmsinfo_hash != NULL) {
		apr_hash_index_t *hi;
		for (hi = apr_hash_first(p, mctx->dbmsinfo_hash);
						hi; hi = apr_hash_next(hi)) {
			/* DBMS 接続情報を取得 */
			apr_hash_this(hi, &dummy, NULL, (void**) &dbmsinfo);

			/* DBプロバイダコンテキストを取得する */
			p_ctx = _get_prv_ctx(CPOOL_CTX(dbmsinfo->repos_alist->dbconn)->providerid);
			if (p_ctx == NULL || p_ctx->cpool == NULL ||
					p_ctx->pc_state == DB_PCNTXT_DESTROYED) {
				continue;
			}

			/* プールに戻す */
			_append_DbConnList_to_DbConnPool(p_ctx->cpool, dbmsinfo->repos_alist);
			p_ctx->cpool->navail++;
			if (exit_connmgr_th) return;	/* 終了チェック */
		}
	}
}

/**
 * DBコネクション管理スレッドのクリーンアップハンドラに登録する
 * mutex解除用のハンドラ関数。
 *
 * @param mutex void * mutex オブジェクト
 */
static void _cleanup_thread_mutex(void *mutex)
{
	apr_status_t rv;

	/* mutex ロックを解除 */
	rv = apr_thread_mutex_unlock((apr_thread_mutex_t *) mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex unlock. (code = %d)", rv);
	}
#ifdef DIVY_DBCPOOL_DEBUG
	ERRLOG1(NULL, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		"unlock handler(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DBCPOOL_DEBUG */
}

/*-----------------------------------------------------------------------------
 * Define utility private functions
 -----------------------------------------------------------------------------*/
/**
 * トランザクションコンテキストを生成してトランザクション開始前の状態
 * に設定して渡す。
 *
 * @param r request_rec *
 * @param dbconn DbConn* データベースとの接続状態を表すコネクションオブジェクト
 * @param ts_ctx divy_db_transaction_ctx ** トランザクションコンテキスト
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _create_transaction_ctx(request_rec *r,
					DbConn *dbconn,
					divy_db_transaction_ctx **ts_ctx)
{
	/* トランザクションコンテキストの作成 */
	*ts_ctx = apr_pcalloc(r->pool, sizeof(divy_db_transaction_ctx));

	(*ts_ctx)->dbconn = dbconn;
	(*ts_ctx)->status = DIVY_TRANS_NOTREADY;	/* 開始されていない */
	(*ts_ctx)->data   = NULL;
	(*ts_ctx)->p      = r->pool;

	return 0;
}

/**
 * *src のリンクリストの先頭から辿ってunitnum 個までのリンクリストを
 * *dst として返却する。
 * 実際にunitnum 個の要素が無ければ切り出し可能だったリンクリストを
 * 返します。リストの終端はNULLです。
 * (note)
 * 	この関数が終わるとsrc の値も変更され、新しいリストの先頭を
 * 	示すように変更されます。ご注意ください。
 *
 * @param src divy_linkedlist_t ** src リンクリストへのポインタ
 * @param dst divy_linkedlist_t ** dst リンクリストへのポインタ
 * @param unitnum int 切り出したいリンクリストの要素数
 * @return int 実際に切り出したリンクリストの要素数
 * 	src がNULLならば0
 */
static int _devide_list(divy_linkedlist_t **src, divy_linkedlist_t **dst, int unitnum)
{
	divy_linkedlist_t *ll;
	int i;

	*dst = NULL;	/* 初期化 */

	if (*src == NULL) return 0;

	/* unitnum 個だけリストに入れる */
	for (i = 0, ll = *dst = *src; i < unitnum - 1; i++) {
		if (ll->next == NULL) break;
		ll = ll->next;	/* リンクを辿る */
	}
	/* 先頭を入れ替える */
	*src = ll->next;

	ll->next = NULL;	/* dst の末尾をNULLにする */

	return (i + 1);
}

/**
 * num 個のバインド変数指示文字"?" を","で連結し文字列として返却する。
 *
 * @param p apr_pool_t *
 * @param num int "?"の数。0ならば空文字列を返却
 * @return char * 組み立てた文字列
 */
static char * _make_bindstr(apr_pool_t *p, int num)
{
	int i;
	divy_sbuf *sbuf = NULL;

	if (num == 0) return "";

	/* バインド変数指示文字列の組み立て */
	for (i = 0; i < num; i++) {
		if (sbuf == NULL) {
			divy_sbuf_create(p, &sbuf, 256);
		}

		if (i == 0) {
			divy_sbuf_appendbyte(sbuf, 1, "?");
		}
		else {
			divy_sbuf_appendbyte(sbuf, 2, ",?");
		}
	}

	if (divy_sbuf_getlength(sbuf) > 0) {
		return divy_sbuf_tostring(sbuf);
	}
	else {
		return "";
	}
}


