/**
 * $Id$
 *
 * util_ldap.c
 *
 * LDAPユーティリティ関数
 *
 *
 */
#include "httpd.h"
#include "http_core.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_tables.h"
#include "apr_global_mutex.h"
#ifndef AP_NEED_SET_MUTEX_PERMS
#define AP_NEED_SET_MUTEX_PERMS
#endif
#include "unixd.h"
#include "util_md5.h"

#include "mod_dav_tf.h"
#include "util.h"
#include "util_ldap.h"
#include "util_common.h"
#include "tf_rdbo.h"
#include "tf_rdbo_user.h"
#include "liveprop.h"
#include "util_db.h"

#include "util_auth.h"
#include "tf_folder.h"
#include "tf_validator.h"
#include "tf_valuecache.h"

#if defined(DIVY_SUPPORT_LDAP)

/*--------------------------------------------------------------
  declare static value
  --------------------------------------------------------------*/
/**
 * LDAPの共有メモリを扱う為のロック構造体
 */
static apr_global_mutex_t	*divy_ldap_mutex;

/*--------------------------------------------------------------
  Define fixed values and macro
  --------------------------------------------------------------*/
APLOG_USE_MODULE(dav_tf);

/**
 * セマフォを利用する為のユニークファイルパス名
 */
#define DIVY_LDAP_SEM_FILE		"/tmp/ldap_sem."

#define LDAP_NOMAP	"DIVY.NOMAP.ITEM"

/**
 * LDAPConfigの各デフォルト値
 * configに設定されていなかった場合に利用されます
 */
#define DIVY_LDAP_CACHE_MARK		50
#define DIVY_LDAP_CACHE_PURGE		85
#define DIVY_LDAP_SEARCH_CACHE_TTL	30
#define DIVY_LDAP_MAXMEM_BLOCKS		10000

/*--------------------------------------------------------------
  Declare  private functions
  --------------------------------------------------------------*/
static TFLDAPConf * _divy_util_ldap_create_config(request_rec *r, apr_pool_t *wp);
static int _divy_util_ldap_sync_property(request_rec *r, const char *userid, const char *passwd);
static void _divy_util_ldap_lock(TFLULOCKTYPE enuLockType, void *pUserData);
static int _divy_util_ldap_init_lock(apr_pool_t *pconf, server_rec *s);
static int _divy_util_ldap_user_notfound_action(request_rec *r, const char* userid, int action);
static apr_status_t _divy_util_ldap_cleanup(void* not_used);
#endif
static int _divy_util_ldap_allow_user(request_rec *r, const char *userid);
/**
 * LDAPを利用しているユーザかを調べる
 *
 * @param	r request_rec *
 * @return  int TF_LDAP_TRUE(LDAPを利用) | TF_LDAP_FALSE(LDAP利用しない)
 */
LDAP_DECLARE(int) divy_util_ldap_use_ldap(request_rec *r)
{
	int ret = TF_LDAP_FALSE;

	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);

	if (sconf->use_ldap_opt && conf->ldap == TF_LDAP_ON) {
		if (_divy_util_ldap_allow_user(r, divy_get_userid(r)) == TF_LDAP_TRUE) 
			ret = TF_LDAP_FALSE;
		else
			ret = TF_LDAP_TRUE;
	}

	return ret;
}

/**
 * LDAPの初期化を行う
 * このコードはPostconfigから呼ばれることを意識しています。
 * その他での利用はしないでください。
 *
 * @param s server_rec*                 サーバリクエスト構造体
 * @param pconf apr_pool_t              プール
 */
LDAP_DECLARE(int) divy_util_ldap_initialize_config(server_rec *s, apr_pool_t *pconf)
{
#if defined(DIVY_SUPPORT_LDAP)
	int flag;
	TFLDAPOptions *ldap_options = NULL;
	TFLDAPUtil    *ldap_util = NULL;

	dav_divy_server_conf *conf = dav_divy_get_server_config(s);

	/* LDAPが利用されない場合処理を行なわない */
	if (conf->ldapcachebyte == TF_LDAP_UNSET) return TF_LDAP_TRUE;

	flag = divy_pcache_get_flag(s->process->pool, DIVY_PCACHE_FLG_GL_SHM);

	/*
	 * 一度目の読み出し時は何も設定を行なわずそのまま正常終了する
	 */
	if (!flag) {
		divy_pcache_set_flag(s->process->pool, 1, DIVY_PCACHE_FLG_GL_SHM);
		return TF_LDAP_TRUE;
	}

	TRACE_S(s)

	/*
	 * 共有メモリを利用する為のロックを取得する
	 */
	if ((_divy_util_ldap_init_lock(pconf, s)) != TF_LDAP_TRUE) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to Semaphore.");
		_divy_util_ldap_cleanup(NULL);
		return TF_LDAP_FALSE;
	}

	/* 共有メモリの設定 */
	ldap_options = apr_pcalloc(pconf, sizeof(TFLDAPOptions));
	if (ldap_options == NULL) return TF_LDAP_FALSE;

	ldap_options->pszShmPath          = conf->ldapshmpath;
	ldap_options->unCacheBytes        = conf->ldapcachebyte;
	ldap_options->pszCAFile           = conf->ldapcert_auth_file;
	ldap_options->enuCAType           = conf->ldapcert_file_type;

	/*
	 * 値が設定されていなかった場合は強制的にデフォルトの値とする。
	 */
	if (conf->ldap_maxmemBlocks != 0) {
		ldap_options->unMaxMemBlocks = conf->ldap_maxmemBlocks;
	}
	else {
		ldap_options->unMaxMemBlocks = DIVY_LDAP_MAXMEM_BLOCKS;
	}

	if (conf->ldapsearch_cache_ttl != 0) {
		ldap_options->unTTL = conf->ldapsearch_cache_ttl;
	}
	else {
		ldap_options->unTTL = DIVY_LDAP_SEARCH_CACHE_TTL;
	}

	if (conf->ldapMarkPercentage != 0) {
		ldap_options->fMarkPercentage = conf->ldapMarkPercentage;
	}
	else {
		ldap_options->fMarkPercentage = DIVY_LDAP_CACHE_MARK;
	}

	if (conf->ldapPurgePercentage != 0) {
		ldap_options->fPurgePercentage    = conf->ldapPurgePercentage;
	}
	else {
		ldap_options->fPurgePercentage = DIVY_LDAP_CACHE_PURGE;
	}

	ldap_options->unURLHashTblSize    = conf->ldapURLHashTblSize;
	ldap_options->unSearchHashTblSize = conf->ldapSearchHashTblSize;

	/* LDAP共有メモリの作成 */
	/* TFLDAPUtil_Createの内部でクリーンアップレジスターを登録しているので
	 *　ここ関数では何もしてはいけません */
	ldap_util = TFLDAPUtil_Create(pconf, ldap_options, _divy_util_ldap_lock, NULL);
	if (ldap_util == NULL) {
			ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"Failed to creating the shared memory.");
			_divy_util_ldap_cleanup(NULL);
			return TF_LDAP_FALSE;
	}

	/* poolに共有メモリ状態構造体をセットする */
	divy_pcache_set_data(s->process->pool, ldap_util, DIVY_PCACHE_DAT_GL_LDAPUTIL);

#endif
	return TF_LDAP_TRUE;
}


/**
 * LDAPが利用するキャッシュ領域（共有メモリ）を作成する
 * (note)
 *      この内部で共有メモリをプールを利用して作成します。
 *      共有メモリの存続はプールの存続期間に縛られる為、
 *      引数のプールには注意して利用してください。
 * この関数はpost_configから呼ばれることを意識して作られています。
 *
 * @param pchild apr_pool_t *           プール
 * @param s server_rec *                サーバリクエスト構造体
 *
 * @return TF_LDAP_TRUE 成功 | TF_LDAP_FALSE 失敗
 */
LDAP_DECLARE(int) divy_util_ldap_initialize(apr_pool_t *pchild, server_rec *s)
{

#if defined(DIVY_SUPPORT_LDAP)

		dav_divy_server_conf *conf = dav_divy_get_server_config(s);
		const char *fname;
		apr_status_t rv;

		/* LDAPが利用されない場合処理を行なわない */
		if (conf->ldapcachebyte == TF_LDAP_UNSET) return TF_LDAP_TRUE;

		/* グローバルmutex が取得できていなかった場合 */
		if (divy_ldap_mutex == NULL) {
				_divy_util_ldap_cleanup(NULL);
				return TF_LDAP_FALSE;
		}

		/* グローバルmutex の再オープン */
		fname = apr_pstrcat(pchild, DIVY_LDAP_SEM_FILE, ap_unixd_config.user_name, NULL);
		rv = apr_global_mutex_child_init(&divy_ldap_mutex, fname, pchild);
		if (rv != APR_SUCCESS) {
				ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
								"Failed to init global mutex for child.(code = %d)", rv);
				_divy_util_ldap_cleanup(NULL);
				return TF_LDAP_FALSE;
		}

#endif

		return TF_LDAP_TRUE;

}


/**
 * LDAPへユーザIDとパスワードのチェックを行なう
 *
 * @param r request_rec *		リクエスト構造体
 * @param userid const char *		ユーザID
 * @param passwd const char *		パスワード
 *
 * @return int
 *
 */
LDAP_DECLARE(int) divy_util_ldap_check_userid(request_rec *r,
					       	const char *userid,
					       	const char *passwd)
{
#if defined(DIVY_SUPPORT_LDAP)

	int 	     result = TFLROK;	/* LDAPライブラリの戻り値 */
	TFLDAPConf   *ldap_config = NULL;
	TFLDAPUtil   *ldap_util = NULL;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	apr_pool_t *ka_p = r->connection->pool;
	char *cached_pw;

	/* LDAPを利用しない場合[OFF||UNSET]、ライセンスが無い場合はそのまま正常終了 */
	if (conf->ldap != TF_LDAP_ON || !sconf->use_ldap_opt) return TF_LDAP_TRUE;

	/* パラメータチェック */
	if (IS_EMPTY(userid) || IS_EMPTY(passwd)) return TF_LDAP_FALSE;

	/*
	 * LDAPで管理する必要がないユーザがいた場合はそのまま
	 * 正常終了とする
	 */
	if (_divy_util_ldap_allow_user(r, userid) == TF_LDAP_TRUE) 
		return TF_LDAP_TRUE;

	/*
	 * パスワード無視モードの場合パスワードは対象のLDAPに
	 * 登録されているユーザのパスワードと不一致になります。
	 * 調べるだけ無駄ですのでここで終了とする
	 */
	if (divy_util_auth_allow_ignore_login(r, passwd))
		return TF_LDAP_TRUE;

	TRACE(r->pool)

	/* LDAPユーティリティの取得 */
	ldap_util = divy_pcache_get_data(r->server->process->pool, DIVY_PCACHE_DAT_GL_LDAPUTIL);
	/* LDAPユーティリティが取得できないはずはない */
	if (ldap_util == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
		"Failed to get LDAP Utility. It is necessary to use it in the LDAP function. check \"TfLDAPCacheByte\" directive. or Server prosess pool is broken ?");
		return TF_LDAP_FALSE;
	}

	/* tf_ldap_config構造体の作成 */
	ldap_config = _divy_util_ldap_create_config(r, r->pool);
	if (ldap_config == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
		"Failed to create LDAP configuration."); 
		return TF_LDAP_FALSE;
	}

	/*
	 * LDAP認証キャッシュを使って認証されたuserid であったかどうか調べる
	 * (note)
	 *   このコードは、LDAPサーバに対する接続・切断があまりにも多すぎるため
	 *   それを軽減する目的で入れたものです.
	 *   本来はあまり正しい方法ではありません. 何故なら、KeepAliveの単位でしか
	 *   認証が行われないからです. トレードオフです.
	 */
	cached_pw = divy_pcache_vget_data(ka_p, DIVY_PCACHE_DAT_KA_LDAPAUTH, userid, NULL);
	if (IS_FILLED(cached_pw) && strcmp(cached_pw, passwd) == 0) {
		/* キャッシュ済みであればもう何もしない */
		return TF_LDAP_TRUE;
	}

	/*
	 * LDAPライブラリに問合せ
	 * ここで共有メモリに情報を全てキャッシュします
	 */
	 result = TFLDAPUtil_CheckUserID(ldap_util, ldap_config, r->pool,
		       					userid, passwd);
	if (result == TFLROK) {

		/*
		 * LDAPの情報とTeamFileのユーザ情報(divy_usr)を同期する
		 *
		 * (note) 2006/12/01 Fri takehara
		 *  以前、ここではr->connection->notes を利用して、ここに値が
		 *  あれば同期化処理をしないようにしていました.
		 *  ですが、同一コネクションの中で値が変わることが現実に起きて、
		 *  これを回避するには、毎回DBを引いてチェックするしかありません.
		 *  という訳で、キャッシュのコードは除去します.
		 *
		 *  DB 値を常に書き込むかどうかの考慮は、下記の同期化関数の
		 *  中身を見てください.
		 */
		if (conf->ldapsyncproperty) {
			/* 同期化を実施 */
			(void)_divy_util_ldap_sync_property(r, userid, passwd);
		}
		/* 認証情報をキャッシュ
		 * (note) ka_p から文字列データは確保しておくこと */
		divy_pcache_vset_data(ka_p, apr_pstrdup(ka_p, passwd), DIVY_PCACHE_DAT_KA_LDAPAUTH,
								apr_pstrdup(ka_p, userid), NULL);
	}
	else if (result == TFLRSERVERDOWN) {
		/* サーバダウン (継続) */
		ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_NET,
			"The inquiry failed in LDAP. Processing is continued without using LDAP."
			"[result = %s]", TFLDAPUtil_FormatMessage(result));
		/* 継続させます */
	}
	else if (result == TFLRAUTHFAILED) {
		/* LDAPを利用しての認証エラーは必ずエラーとする */
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to LDAP authenticate. user=%s . [result = %s]",
			REPLACE_NULL(userid), TFLDAPUtil_FormatMessage(result));
		return TF_LDAP_FAIL_AUTHFAIL;
	}
	else if (result == TFLROBJNOTFOUND) {
		/* ユーザが存在せずに許可されたユーザでもなかった */
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_DATA,
			"Failed to LDAP user %s not found. [retult = %s]",
			REPLACE_NULL(userid), TFLDAPUtil_FormatMessage(result));

		/* LDAPにユーザがいない取り扱い */
		if (conf->ldapusernotfoundaction != DIVY_INT32_C_UNSET) {
			(void)_divy_util_ldap_user_notfound_action(r, userid, conf->ldapusernotfoundaction);
		}
		return TF_LDAP_FAIL_OBJNOTFOUND;
	}
	else {
		/* その他のエラーは致命的なのでエラー */
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_PROC,
			"Failed to LDAP Server. [result = %s]", TFLDAPUtil_FormatMessage(result))
		return TF_LDAP_FALSE;
	}

#endif
	return TF_LDAP_TRUE;
}


/**
 * LDAPからユーザの情報を取得する
 * divy_rdbo_get_user_property()と同じく構造体に値を埋めて返す。
 * この関数はBANDするユーザはLDAPのBindユーザで行います
 * そうしなければTeamFileの管理者がユーザ一覧を行うことができないからです
 *
 * @param r request_rec *
 * @param userid const char *
 * @param usr_pr divy_rdbo_usr **
 *
 */
LDAP_DECLARE(int) divy_util_ldap_get_user_property(request_rec *r,
					      apr_pool_t *wp,
					      const char *userid,
					      divy_rdbo_usr *usr_pr)
{
#if defined(DIVY_SUPPORT_LDAP)

	char *dn, **attributes;

	dav_divy_dir_conf *conf  = dav_divy_get_dir_config(r);		/* dir conf */
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	TFLDAPConf *ldap_config = NULL;
	TFLDAPUtil *ldap_util = NULL;
	int result = TFLROK;

	/* LDAPを利用しない場合[OFF||UNSET]、ライセンスが無い場合 */
	if (conf->ldap != TF_LDAP_ON || !sconf->use_ldap_opt) return TF_LDAP_FALSE;

	if (IS_EMPTY(userid)) return TF_LDAP_FALSE;

	/*
	 * LDAPで管理する必要がないユーザがいた場合はそのまま
	 * なにもしないで失敗とする
	 */
	if (_divy_util_ldap_allow_user(r, userid) == TF_LDAP_TRUE) 
		return TF_LDAP_FALSE;

	TRACE(r->pool)

	ldap_util = divy_pcache_get_data(r->server->process->pool, DIVY_PCACHE_DAT_GL_LDAPUTIL);
	if (ldap_util == NULL || IS_EMPTY(userid))
	       	return TF_LDAP_FALSE;

	/* tf_ldap_config構造体の作成 */
	ldap_config = _divy_util_ldap_create_config(r, wp);
	if (conf->ldap == TF_LDAP_ON && ldap_config == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
		"Failed to create LDAP configuration."); 
		return TF_LDAP_FALSE;
	}

	/* ライブラリに属性の取得を行なう */
	result = TFLDAPUtil_GetAttributes(ldap_util, ldap_config, wp,
			       			userid, &dn, &attributes);

	/*
	 * configに値が設定されていて尚且つアトリビュートとしても
	 * 設定されていた場合だけに値をコピーする
	 */
	if (result == TFLROK) {

		/*
		   LDAPより情報を入手する為に必須項目が正確にあるか
		   を確認します。
		   パスワードは取得できない可能性があるのでチェックしない
		   ユーザ名（ログインIDではない）は取得できなければ
		   ログを出力してユーザIDを設定する
		*/
		/* ユーザID */
		if (IS_FILLED(ldap_config->ppszAttributes[0]) &&
			       		IS_FILLED(attributes[0])) {
			usr_pr->usrid = apr_pstrdup(wp, attributes[0]);
		}
		else {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_NET,
				   "LDAP server returned by \"OK\"."
				   "but LDAP \"userid\" in attribute is NULL."
				   "LDAP service is maintaining? (userid=%s)", userid);
			/* ユーザIDは必須なのでエラーとする */
			return TF_LDAP_FALSE;
		}

		/* パスワード */
		if (IS_FILLED(ldap_config->ppszAttributes[1]) &&
			       		IS_FILLED(attributes[1])) {
			usr_pr->password = apr_pstrdup(wp, attributes[1]);
		}
		/* チェックしない */

		/* 名称(名前) */
		if (strcmp(ldap_config->ppszAttributes[2], LDAP_NOMAP) != 0) {
			if (IS_FILLED(attributes[2])) {
				usr_pr->fullname = apr_pstrdup(wp, attributes[2]);
			}
			else {
				ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_NET,
						"LDAP server returned by \"OK\"."
						"but LDAP \"name\" in attribute is NULL."
						"replace of name -> userid. (userid=%s)", userid
					   );
				/* ユーザIDを代わりに入れる */
				usr_pr->fullname = apr_pstrdup(wp, usr_pr->usrid);
			}
		}
		else {
			/* なにもしない */
		}

		/* メールアドレス */
		if (IS_FILLED(ldap_config->ppszAttributes[3]) &&
			       		IS_FILLED(attributes[3])) {
			usr_pr->mailaddr = apr_pstrdup(wp, attributes[3]);
		}
		/* チェックはしない(なしでもいい) */

		/* コメント(description) */
		if (IS_FILLED(ldap_config->ppszAttributes[4]) && 
						IS_FILLED(attributes[4])) {
			usr_pr->comment = apr_pstrdup(wp, attributes[4]);
		}
		/* チェックはしない(なしでもいい) */

	}
	/* 対象ユーザが見つからなかった場合 */
    else if (result == TFLROBJNOTFOUND || result == TFLRAUTHFAILED) {
		/* エラーとまではいかないので警告しておく */
		ERRLOG2(r->pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"There is not found user entry of LDAP. (userid = %s)"
				"LDAP result = [%s]", userid, TFLDAPUtil_FormatMessage(result));
#if 0
		if (conf->ldapusernotfoundaction != DIVY_INT32_C_UNSET) {
			(void)_divy_util_ldap_user_notfound_action(r, userid, conf->ldapusernotfoundaction);
		}
#endif
		return TF_LDAP_FALSE;
	}
	else if (result == TFLRSERVERDOWN) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_NET,
				"The inquiry failed of LDAP. continued without LDAP. (userid = %s)"
				"LDAP result = [%s]", userid, TFLDAPUtil_FormatMessage(result));
		/* サーバダウンは継続 */
	}
	else {
		/* その他のエラーは致命的なのでエラー */
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_CERR + DIVY_SST_PROC,
				"Failed to LDAP Server. [result = %s]", TFLDAPUtil_FormatMessage(result))
		return TF_LDAP_FALSE;
	}

	return TF_LDAP_TRUE;

#else
	return TF_LDAP_FALSE;
#endif
	
}

/**
 * LDAPの情報を元にTeamFileにユーザを新規作成する
 * (note)
 * 	LDAPから取得できる項目はuserid, password, fullname, mailaddress
 * 	それ以外の項目はすべてデフォルト値を設定するようにする
 * (note)
 * 	LDAPから取ったパスワードは利用しないことにしました。(2004/12/14 Tue)
 *
 * @param r		request_rec *
 * @param userid	const char *
 * @param password	const char * ユーザが入力したパスワード
 *
 * @return TF_LDAP_TRUE | TF_LDAP_FALSE
 */
LDAP_DECLARE(int) divy_util_ldap_found_create_user(request_rec *r, const char *userid, const char *password)
{
#if defined(DIVY_SUPPORT_LDAP)
	apr_pool_t *p  = r->pool;
	char *user_uri;		/* プライベートコレクションのURI */
	divy_user_iscreen iscreen = { 0 };
	divy_rdbo_usr *dst_usr_pr = NULL;
	divy_rdbo_usr *usr_pr = apr_pcalloc(r->pool, sizeof(divy_rdbo_usr));
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);

	if ((divy_util_ldap_get_user_property(r, r->pool, userid, usr_pr)) != TF_LDAP_TRUE) {
		return TF_LDAP_FALSE;
	}

	/* LDAP からとったパスワードは利用しない (無視する) */
	usr_pr->password = apr_pstrdup(p, password);

	/* 管理者ユーザのコレクションURIを生成する */
	user_uri = divy_build_m_user_uri(p, dav_divy_get_root_uri(r), usr_pr->usrid);
	divy_parse_uri(p, dav_divy_get_root_uri(r), user_uri, &iscreen.u_spec);

	/* iscreen に中身を詰める */
	iscreen.usrid      = usr_pr->usrid;
	iscreen.password   = usr_pr->password;
	iscreen.fullname   = usr_pr->fullname;
	iscreen.adminmode  = DIVY_ADMINMODE_NORMAL;	/* 一般ユーザ */
	if (conf->ldapmaxresquota != DIVY_INT64_C_UNSET) {
		iscreen.smaxst     = apr_psprintf(p, "%"APR_INT64_T_FMT, conf->ldapmaxresquota);
	}
	else {
		iscreen.smaxst     = apr_psprintf(p, "%"APR_INT64_T_FMT, DIVY_QUOTA_MAX_STORAGE);
	}
	iscreen.smaxres    = apr_psprintf(p, "%"APR_INT64_T_FMT, DIVY_QUOTA_MAX_RESOURCE);
	/* アクセス制限を設定 */
	if (conf->ldapuseroptions != DIVY_INT32_C_UNSET)
	{
		iscreen.accessdeny = apr_pcalloc(p, sizeof(int) * DIVY_FOLDER_ID_END);	
		/* ユーザフォルダ */
		if (conf->ldapuseroptions & DIVY_LDAP_OPT_NOTUSERFOLDER)
			iscreen.accessdeny[DIVY_FOLDER_ID_user] = 1;
		/* グループフォルダ */
		if (conf->ldapuseroptions & DIVY_LDAP_OPT_NOTGROUPFOLDER)
			iscreen.accessdeny[DIVY_FOLDER_ID_group] = 1;
		/* データベースフォルダ */
		if (conf->ldapuseroptions & DIVY_LDAP_OPT_NOTDBFOLDER)
			iscreen.accessdeny[DIVY_FOLDER_ID_dblink] = 1;
		/* リポジトリデータベース */
		if (conf->ldapuseroptions & DIVY_LDAP_OPT_NOTREPOSDBFOLDER)
			iscreen.accessdeny[DIVY_FOLDER_ID_reposdblink] = 1;
	} else {
		iscreen.accessdeny = NULL;	/* アクセス制限なし */
	}
	if (divy_support_extenduserstatus(r)) {
		iscreen.extstatus = divy_rdbo_create_default_extstatus(p, EXTSTATUS_TYPE_USR);	/* デフォルト */
	}
	else {
		iscreen.extstatus = NULL;
	}
	iscreen.src_usr_pr = NULL;

	/*
	 * 新規作成するユーザが問題がないかチェックをしてから作成する
	 */
	if ((divy_validate_user_property(r, DIVY_VALIDATE_INSERT, &iscreen)) != NULL) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to validate user property via LDAP.(userid = %s)", iscreen.usrid);
		return TF_LDAP_FALSE;
	}

	/* iscreen -> rdbo */
	(void) divy_map_user_property(p, &iscreen, &dst_usr_pr);

#ifdef DIVY_SUPPORT_PASSPOLICY
	/* ユーザのパスワードポリシー状態を作る */
	usr_pr->passpolicy_status = apr_pcalloc(p, sizeof(divy_rdbo_passpolicystatus));
	usr_pr->passpolicy_status->policyid        = DIVY_DEFAULT_POLICY_ID;
	usr_pr->passpolicy_status->usrid           = usr_pr->usrid;
	usr_pr->passpolicy_status->lastchange      = 0L;	/* 自動的にいれます */
	usr_pr->passpolicy_status->sendexpiredmail = 0L;
	usr_pr->passpolicy_status->firstlogin      = 0L;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

	/* DBに登録する */
	if ((divy_rdbo_insert_user_property(r, dst_usr_pr, NULL)) != 0) {
		return TF_LDAP_FALSE;
	}

	ERRLOG1(p, APLOG_INFO, DIVY_FST_NORMAL,
		"Attestation succeeded in LDAP. However, since no account existed and there was in TeamFile, it created. (user = %s)", r->user);
#endif
	return TF_LDAP_TRUE;
}

/*--------------------------------------------------------------
  Declare  private functions
  --------------------------------------------------------------*/
#if defined(DIVY_SUPPORT_LDAP)
/**
 * コンフィグの内容を取得してLDAP用configの構造体を作成する
 * 既にリクエスト内でconfig構造体を作成していた場合はそのまま再利用
 * なかった場合だけ作成して以後利用できるようにプールに設定する。
 *
 * @param r request_rec *
 * @param wp apr_pool_t * TFLDAPConf を割り当てるプール
 *
 * @return ldap_config TFLDAPConf *
 *
 */
static TFLDAPConf * _divy_util_ldap_create_config(request_rec *r, apr_pool_t *wp)
{


	TFLDAPConf	  *ldap_config = NULL;
	dav_divy_dir_conf *conf = NULL;
	dav_divy_server_conf *sconf = NULL;

	TRACE(r->pool)

	if (wp == NULL) return NULL;

	/*
	 * ロケーション内でのLDAPの利用を行なわない場合と
	 * LDAPライセンスが無い場合と
	 * 共有メモリの設定漏れなどの場合はNULLとする
	 */
	conf  = dav_divy_get_dir_config(r);		/* dir    conf */
	sconf = dav_divy_get_server_config(r->server);	/* server conf */
	if (conf->ldap != TF_LDAP_ON || !sconf->use_ldap_opt)
	       	return NULL;
	if (sconf->ldapcachebyte == TF_LDAP_UNSET) return NULL;

	/*
	 * poolにconfigデータが設定されているかチェックし
	 * 存在した場合はそのままconfigを返す
	 */
	/* (note) 2006/11/30 Thu takehara
	 * r->pool と wpのライフサイクルは異なるので、r->pool にキャッシュしてはならない */
	//ldap_config = divy_pcache_get_data(r->pool, DIVY_PCACHE_DAT_REQ_LDAPCONFIG);
	ldap_config = divy_pcache_get_data(wp, DIVY_PCACHE_DAT_REQ_LDAPCONFIG);
	if (ldap_config != NULL) return ldap_config;

	/* LDAP構造体の作成 */
	ldap_config = TFLDAPConf_Create(wp);
	if (ldap_config == NULL) return NULL;

	if (IS_EMPTY(conf->ldapurl)) {
		/* URIが設定されていない場合は強制的にApacheより取得 */
		ldap_config->pszURL = ap_construct_url(wp, conf->root_uri, r);
	}
	else {
		ldap_config->pszURL = conf->ldapurl;
	}

	ldap_config->pszBaseDN    = conf->ldapbasedn;
	ldap_config->pszBindDN    = conf->ldapbinddn;
	ldap_config->pszBindPW    = conf->ldapbindpw;
	ldap_config->pszAttribute = conf->ldapattr;
	ldap_config->scope        = conf->ldapscope;
	ldap_config->pszFilter    = conf->ldapfilter;
	ldap_config->enuDeref     = always;

	ldap_config->pszHost      = conf->ldaphost;
	ldap_config->nPort        = conf->ldapport;
	ldap_config->bSSL         = conf->ldapsslsupport;

	/* LDAPの属性に対してマッピングする項目をセットする */
	ldap_config->ppszAttributes = apr_pcalloc(wp, sizeof(char*) * 6);
	ldap_config->ppszAttributes[0] = (IS_EMPTY(conf->ldapmapuserattr)) ?
	       				LDAP_NOMAP : conf->ldapmapuserattr;
	ldap_config->ppszAttributes[1] = (IS_EMPTY(conf->ldapmappasswordattr)) ?
						LDAP_NOMAP : conf->ldapmappasswordattr;
	ldap_config->ppszAttributes[2] = (IS_EMPTY(conf->ldapmapfullnameattr)) ?
	       				LDAP_NOMAP : conf->ldapmapfullnameattr;
	ldap_config->ppszAttributes[3] = (IS_EMPTY(conf->ldapmapmailaddrattr)) ?
	       				LDAP_NOMAP : conf->ldapmapmailaddrattr;
	ldap_config->ppszAttributes[4] = (IS_EMPTY(conf->ldapmapcommentattr)) ?
	       				LDAP_NOMAP : conf->ldapmapcommentattr;

	//divy_pcache_set_data(r->pool, ldap_config, DIVY_PCACHE_DAT_REQ_LDAPCONFIG);
	divy_pcache_set_data(wp, ldap_config, DIVY_PCACHE_DAT_REQ_LDAPCONFIG);

	return ldap_config;
}

/**
 * 指定されたユーザが認証がLDAPに存在しないでも許可するかを確認する
 *
 * @param r request_rec *
 * @param userid const char *
 *
 * @return int TF_LDAP_TRUE: 許可ユーザ | TF_LDAP_FALSE: 非許可ユーザ
 *
 */
static int _divy_util_ldap_allow_user(request_rec *r, const char *userid)
{

	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	int length = 0;
	int i = 0;
	int ret = TF_LDAP_FALSE;
	const divy_rdbo_extstatus *extstatus = NULL;
	ap_regex_t *preg;

	TRACE(r->pool)

	if (conf->ldapallowuser == NULL || IS_EMPTY(userid)) return TF_LDAP_FALSE;

	/* TeamFile上のシステムユーザはLDAP管理対象外 */
	if (divy_support_extenduserstatus(r)) {
		extstatus = divy_get_extstatus(r);
		if (extstatus && divy_rdbo_has_sysexec_privilege(extstatus)) {
			return TF_LDAP_TRUE;
		}
	}

	length = divy_array_getlength(conf->ldapallowuser);
	for (i = 0; i < length; i++) {
		char *allowid;
		allowid = divy_array_get(conf->ldapallowuser, i);
		if (strcmp(userid, allowid) == 0) {
			return TF_LDAP_TRUE;
		}
	}

	/* 正規表現ルールにヒットしたらLDAP管理対象外 */
	if (conf->ldapallowuserregex == NULL) return TF_LDAP_FALSE;
	preg = ap_pregcomp(r->pool, conf->ldapallowuserregex, 0);
	ret = ap_regexec(preg, userid, 0, NULL, 0);
	ap_pregfree(r->pool, preg);

	if (!ret)
		return TF_LDAP_TRUE;

	return TF_LDAP_FALSE;
}

/**
 * LDAPの情報とTeamFileのユーザテーブルを同期する
 *
 * @param r		request_rec *
 * @param userid	const char *
 * @param passwd	const char *
 *
 * @return int
 */
static int _divy_util_ldap_sync_property(request_rec *r, const char *userid,
					 		    const char *passwd)
{

	divy_rdbo_usr	*usr_pr = NULL;
	divy_rdbo_usr	*usr_pr_ldap = NULL;
	int update = 0;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	int support_groupleader = divy_support_groupleader(r);

	if (IS_EMPTY(userid)) return TF_LDAP_FALSE;

	TRACE(r->pool)

	/* TeamFileのデータベース(divy_usr)から情報を取得する */
	if ((divy_rdbo_get_user_property(r, userid, &usr_pr)) != 0) {
		return TF_LDAP_FALSE;
	}
	if (usr_pr == NULL) return TF_LDAP_FALSE;

	/* LDAPから情報を取得する */
	usr_pr_ldap = apr_pcalloc(r->pool, sizeof(divy_rdbo_usr));
	if ((divy_util_ldap_get_user_property(r, r->pool, userid, usr_pr_ldap))
		       					!= TF_LDAP_TRUE) {
		return TF_LDAP_FALSE;
	}
	if (usr_pr_ldap == NULL) return TF_LDAP_FALSE;

	/*
	 * TeamFileユーザデータベースとLDAPのエントリを比較する
	 * (note)
	 * 	上記usr_prとusr_pr_ldapとの比較
	 * 	httpd.confのTfldapMAPXXX系に値が含まれている項目のみを比較する
	 *
	 * 	passwordだけはLDAPから検索された値は暗号化されている
	 * 	可能性があります。例：{CRYPT}TNupllH/594SU
	 * 	そのまま返却されさらに暗号化方式もまちまち{CRYPT}のほかに{SHA}
	 * 	もあります。その為にTeamFileでは元に戻すことができません。
	 * 	認証が成功したことによる同期処理ですからパスワードは引数で
	 * 	渡すようになりました。
	 */

	/* LDAPから受け取ったパスワードは利用しない */
	usr_pr_ldap->password = apr_pstrdup(r->pool, passwd);
	update = 0;
 
	if (update == 0 && IS_FILLED(conf->ldapmappasswordattr)) {
		/* 完全一致した時のみ更新は不要 */
		if (!(IS_FILLED(usr_pr->password) && IS_FILLED(usr_pr_ldap->password) &&
		      strcmp(usr_pr->password, usr_pr_ldap->password) == 0)) {
			update++;
		}
	}
	if (update == 0 && IS_FILLED(conf->ldapmapfullnameattr)) {
		/* 完全一致した時のみ更新は不要 */
		if (!(IS_FILLED(usr_pr->fullname) && IS_FILLED(usr_pr_ldap->fullname) &&
		      strcmp(usr_pr->fullname, usr_pr_ldap->fullname) == 0)) {
			update++;
		}
	}
	if (update == 0 && IS_FILLED(conf->ldapmapmailaddrattr)) {
		/* 完全一致した時のみ更新は不要 */
		if (!(IS_FILLED(usr_pr->mailaddr) && IS_FILLED(usr_pr_ldap->mailaddr) &&
		      strcmp(usr_pr->mailaddr, usr_pr_ldap->mailaddr) == 0)) {
			update++;
		}
	}
	if (update == 0 && IS_FILLED(conf->ldapmapcommentattr)) {
		/* 完全一致した時のみ更新は不要 */
		if (!(IS_FILLED(usr_pr->comment) && IS_FILLED(usr_pr_ldap->comment) &&
		      strcmp(usr_pr->comment, usr_pr_ldap->comment) == 0)) {
			update++;
		}
	}


	/*
	 * 更新処理を行なう
	 * (note)
	 * 	divy_rdbo_update_user_property()はトランザクションを
	 * 	受け取らないと処理ができない為、自前でトランザクションを
	 * 	作成して処理を進めます。
	 */
	if (update > 0) {
		apr_pool_t *p = r->pool;
		char *user_uri;
		divy_user_iscreen iscreen = { 0 };
		divy_db_transaction_ctx *ts_ctx = NULL;
		divy_rdbo_usr *dst_usr_pr = NULL;

		/*
		 * 更新する内容に不備がないか調べる
		 * 内容に不備があった場合はエラーを出力して終了
		 */

		/* iscreen に中身を詰める */
		iscreen.usrid      = NULL;	/* 意図的にNULL にする */
		iscreen.password   = usr_pr_ldap->password;
		/* MAPする指定の場合のみLDAPの値を埋めます。*/
		iscreen.fullname   = IS_FILLED(conf->ldapmapfullnameattr) ? usr_pr_ldap->fullname : usr_pr->fullname;
		iscreen.mailaddr   = IS_FILLED(conf->ldapmapmailaddrattr) ? usr_pr_ldap->mailaddr : usr_pr->mailaddr;
		iscreen.comment    = IS_FILLED(conf->ldapmapcommentattr) ? usr_pr_ldap->comment : usr_pr->comment;

		/* (note) DBに格納された値は信じますので、検証を通すのに必要な値だけを詰める */
		iscreen.adminmode  = usr_pr->adminmode;
		iscreen.smaxst     = apr_psprintf(p, "%"APR_INT64_T_FMT, usr_pr->maxst);
		iscreen.smaxres    = apr_psprintf(p, "%"APR_INT64_T_FMT, usr_pr->maxres);
		iscreen.accessdeny = usr_pr->accessdeny;
		iscreen.susedst    = NULL;	/* 更新時には指定しない */
		iscreen.susedres   = NULL;	/* 更新時には指定しない */
		iscreen.extstatus  = usr_pr->extstatus;
		divy_format_time_t(p, usr_pr->expiration, DIVY_TIME_STYLE_ISO8601, &iscreen.sexpiration);

		iscreen.registdt   = NULL;	/* 更新時には指定しない */
		iscreen.updatedt   = NULL;	/* 更新時には指定しない */
		iscreen.is_old_client = 0;	/* 関係無し */
		iscreen.src_usr_pr = usr_pr;

		/* 管理者ユーザのコレクションURIを生成する */
		user_uri = divy_build_m_user_uri(p, dav_divy_get_root_uri(r), usr_pr_ldap->usrid);
		divy_parse_uri(p, dav_divy_get_root_uri(r), user_uri, &iscreen.u_spec);

		/* iscreen の検証 */
		if ((divy_validate_user_property(r,
						(DIVY_VALIDATE_UPDATE | DIVY_VALIDATE_LDAP_UPDATE), &iscreen)) != NULL) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The \"userdiscovery\" property is wrong. (userid = %s)", userid);
			return TF_LDAP_FALSE;
		}

		/* iscreen -> rdbo */
		(void) divy_map_user_property(p, &iscreen, &dst_usr_pr);

		/* 足りない情報の詰め直し */
		dst_usr_pr->usrid      = (char *)userid;
		dst_usr_pr->rsid       = usr_pr->rsid;
		dst_usr_pr->usrseq     = usr_pr->usrseq;
		dst_usr_pr->comment    = usr_pr->comment;
		dst_usr_pr->maxst      = usr_pr->maxst;
		dst_usr_pr->maxres     = usr_pr->maxres;
#ifdef DIVY_SUPPORT_PASSPOLICY
		dst_usr_pr->passpolicy_status = usr_pr->passpolicy_status;
#endif	/* DIVY_SUPPORT_PASSPOLICY */

		/* 暗号化パスワードの考慮 */
		if (strcmp(dst_usr_pr->password, iscreen.src_usr_pr->password) != 0) {
			/* このロジックの意味はliveprop.c のコメントを参照 */
			if (conf->encryptpassword == DIVY_ENCRYPTPASSWORD_ON) {
				dst_usr_pr->password = ap_md5(p, (const unsigned char*)dst_usr_pr->password);
			}
		}

		/* グループリーダ周りの設定は昔の値を継承する(LDAP更新では変わりようがないから) */
		if (support_groupleader) {
			dst_usr_pr->maxusercreation = usr_pr->maxusercreation;
			dst_usr_pr->ownerid         = usr_pr->ownerid;
		}

		/* トランザクションコンテキストの生成 */
		if (divy_db_create_transaction_ctx(r, &ts_ctx)) {
			/* トランザクションが作れなかったらエラーとする */
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. LDAP Syncproperty Failed."
				"Transaction_ctx is NULL.");
			return TF_LDAP_FALSE;
		}

		/* (note)
		 * LDAP連携ではユーザの種別は変わらないので、制限ユーザに変化することによる
		 * 開封通知の除去は不要.
		 */

		/* ユーザ情報の更新 */
		if (divy_rdbo_update_user_property(r, dst_usr_pr, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
		}
		else {
			divy_db_commit_transaction(ts_ctx);
		}
	}

	return TF_LDAP_TRUE;
}

static void _divy_util_ldap_lock(TFLULOCKTYPE enuLockType, void *pUserData)
{
	if (divy_ldap_mutex == NULL) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to lock for ldap. divy_ldap_mutex is NULL.");
		return;	/* 何も出来ず */
	}

	switch(enuLockType) {
		case TFLUREADLOCK:
			 (void)apr_global_mutex_lock(divy_ldap_mutex);
			break;

		case TFLUWRITELOCK:
			(void)apr_global_mutex_lock(divy_ldap_mutex);	
			break;

		case TFLUUNLOCK:
			(void)apr_global_mutex_unlock(divy_ldap_mutex);	
			break;
		default:
			break;
	}

}

/**
 * セマフォの取得を行なう。
 * これはpost_configステージから呼ばれることを前提に作られています。
 * それ以外のところでは使わないようにすること。
 *
 * @param pconf	apr_pool_t*	configのプール
 * @param s	server_rec*	サーバリクエスト
 *
 * @return int TF_LDAP_TRUE | TF_LDAP_FALSE
 */
static int _divy_util_ldap_init_lock(apr_pool_t *pconf, server_rec *s)
{

	const char *fname;
	divy_ldap_mutex = NULL;

	fname = apr_pstrcat(pconf, DIVY_LDAP_SEM_FILE,
		       			ap_unixd_config.user_name, NULL);

	/* セマフォを取得する */
	if (apr_global_mutex_create(&divy_ldap_mutex, fname,
		       		    APR_LOCK_DEFAULT, pconf) != APR_SUCCESS) {
		return TF_LDAP_FALSE;
	}

#ifdef AP_NEED_SET_MUTEX_PERMS
	if ((ap_unixd_set_global_mutex_perms(divy_ldap_mutex) != APR_SUCCESS)) {
		ERRLOG0(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS, "LDAP: failed to set permissions");
		return TF_LDAP_FALSE;
	}
#endif

	apr_pool_cleanup_register(pconf, NULL, _divy_util_ldap_cleanup, apr_pool_cleanup_null);
	
	return TF_LDAP_TRUE;
}

/*
 * LDAPに存在しなかったユーザでTeamFileに登録されている場合に削除
 * 若しくは非活性にする
 * 
 * @param r request_rec
 * @param userid const char*
 * @param action int
 *
 * @return int (成功: TF_LDAP_TRUE / 失敗: TF_LFAP_FALSE)
 */
static int _divy_util_ldap_user_notfound_action(request_rec *r, const char* userid, int action)
{

	divy_rdbo_usr	*usr_pr = NULL;
	divy_linkedlist_t *del_filelist = NULL;
	divy_db_transaction_ctx *ts_ctx = NULL;

	/* TeamFileのユーザを取得する */
	if ((divy_rdbo_get_user_property(r, userid, &usr_pr)) != 0) {
		return TF_LDAP_TRUE;
	}

	if (usr_pr == NULL) return TF_LDAP_TRUE;

	if (action == DIVY_LDAP_NOTFOUNDUSERACTION_DISABLE) {

		if (!divy_rdbo_is_active_user(usr_pr->extstatus)) return TF_LDAP_TRUE;

		/* activeフラグを落とす */
		divy_rdbo_set_active_user(usr_pr->extstatus, 0);

		if (divy_db_create_transaction_ctx(r, &ts_ctx)) {
			/* トランザクションが作れなかったらエラーとする */
			ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DB,
				"Could not operation. LDAP Syncproperty Failed."
				"Transaction_ctx is NULL.");
			return TF_LDAP_FALSE;
		}

		/* 更新する */
		if (divy_rdbo_update_user_property(r, usr_pr, ts_ctx)) {
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to disabled user. (userid = %s)", usr_pr->usrid);

			return TF_LDAP_FALSE;
		}

		/* ログの出力 */
		ERRLOG1(r->pool, APLOG_NOTICE, DIVY_FST_NORMAL + DIVY_SST_DATA,
				"User disabled. user=%s.", userid);

		/* DBへの反映を確定する */
		divy_db_commit_transaction(ts_ctx);

	} else if (action == DIVY_LDAP_NOTFOUNDUSERACTION_DELETE) {
		/* ユーザの削除 */
		if (divy_rdbo_remove_user_property(r, usr_pr, &del_filelist, ts_ctx)) {
			/* ユーザ削除の失敗はログを出力 */
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to delete user.(userid = %s)", usr_pr->usrid);
			return TF_LDAP_FALSE;
		}

		/* ログの出力 */
		ERRLOG1(r->pool, APLOG_NOTICE, DIVY_FST_NORMAL + DIVY_SST_DATA,
				"User deleted. user=%s.", userid);

		/* DBへの反映を確定する */
		divy_db_commit_transaction(ts_ctx);
	} else {
		/* 種別不明 ここには通常来ない*/
		return TF_LDAP_FALSE;
	}

	return TF_LDAP_TRUE;
}

/*
 * LDAPのクリーンナップ
 * apr_pool_cleanup_registerで呼ばれることも考えている為引数は
 * つけていません
 *
 * @param *void
 * @return apr_status_t 
 */
static apr_status_t _divy_util_ldap_cleanup(void* not_used)
{
	if (divy_ldap_mutex) {
		apr_global_mutex_destroy(divy_ldap_mutex);
		divy_ldap_mutex = NULL;
	}

	return APR_SUCCESS;
}

#endif	/* static */
