/**
 * $Id$
 *
 * メールユーティリティ関数
 *
 */
#include "httpd.h"
#include "http_core.h"
#include "http_protocol.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_strings.h"
#include "apr_time.h"
#include "apr_network_io.h"
#include "apr_base64.h"
#include "apr_file_info.h"
#include "apr_file_io.h"
#include "apr_hash.h"
#include "apr_thread_mutex.h"

#include "mod_dav_tf.h"
#include "tf_ml.h"
#include "util_ml.h"
#include "util.h"
#include "util_common.h"
#include "util_auth.h"
#include "tf_db.h"
#include "util_db.h"
#include "tf_rdbo.h"
#include "tf_rdbo_user.h"
#include "tf_rdbo_group.h"
#include "tf_folder.h"
#include "tf_storage.h"
#include "tf_vscan.h"
#include "tf_validator.h"
#include "tf_confirmreading.h"

/*--------------------------------------------------------------
  Define fixed values and macro
  --------------------------------------------------------------*/
APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define static values
  ----------------------------------------------------------------------------*/
/**
 * メール機能が利用できるかどうかをチェックする。
 * (1: 利用できない / 0: 利用できる)
 *
 * @param conf	dav_divy_dir_conf*	ディレクトリコンフィグ	
 * @param sconf	dav_divy_server_conf*	サーバコンフィグ
 *
 */
#define IS_MAIL(conf, sconf) \
		((sconf)->use_mail_opt == 0 || IS_EMPTY((conf)->mail) || \
	 	strcmp((conf)->mail, DIVY_ML_OFF) == 0)

/*
 * 通知メールの本文に利用される"ごみ箱"フォルダの名称
 */
#define DIVY_ML_TRASHFOLDER_NAME_JA	"\xE3\x81\x94\xE3\x81\xBF\xE7\xAE\xB1"
#define DIVY_ML_TRASHFOLDER_NAME_EN	"Trash"

/**
 * クライアントから送られてくる固定のBodyデータを識別するための文字列
 * (note) この対応は、クライアントを誰も入れ替えてくれないので、仕方なく入れた対応です.
 * 
 * (日本語)
 *
 * ========== 以下の文章はTeamFileサーバが付加したメッセージです==========
 * (英語)
 *
 * ========== The TeamFile server added this message ==========
 *
 * 以下では共通部分のみを抜き出しています. TeamFile という単語もNGです.
 */
#define DIVY_ML_FIXED_BODYSTR	"\x0D\x0A========== "
#define DIVY_ML_FIXED_BODYSTR2	"========== "

/**
 * (サーバ主体メール通知用)
 */

/*
 * メール雛型ファイル
 */
/* 監視メール用 */
#define DIVY_SVRML_TEMPLATEFILE_PREFIX	"template_ml"

/* ユーザ作成通知用 */
#define DIVY_SVRML_NOTIFYACT_TEMPL_PREFIX	"notifyaction_ml"

/* ユーザ更新通知用 */
#define DIVY_SVRML_NOTIFYACT_UPDATE_USER_PREFIX "updateuser_ml"

/* 2FAトークン通知用 */
#define DIVY_SVRML_NOTIFY_2FA_TOKEN_PREFIX "notify2fatoken_ml"

/* ウィルス感染報告用 */
#define DIVY_SVRML_VIRUS_DETECTION_PREFIX	"infectalert_ml"

/* ウィルス感染の検索失敗用 */
#define DIVY_SVRML_VIRUS_DETECTION_FAIL_PREFIX	"infectalert2_ml"

#ifdef DIVY_SUPPORT_PASSPOLICY

/* パスワード変更通知 */
#define DIVY_SVRML_NOTIFY_CHANGED_PASSWORD_PREFIX "changedpassword_ml"

/* パスワード猶予期間用 */
#define DIVY_SVRML_PASS_PROBATION_PREFIX	"password_probation_ml"

/* パスワードロックアウト用 */
#define DIVY_SVRML_PASS_EXPIRED_PREFIX	"password_expired_ml"

/* 初回ログインパスワード猶予期間用 */
#define DIVY_SVRML_FLOGIN_PROBATION_PREFIX	"firstlogin_probation_ml"

/* 初回ログインパスワードロックアウト用 */
#define DIVY_SVRML_FLOGIN_EXPIRED_PREFIX	"firstlogin_expired_ml"

/* 初回ログインパスワード即時変更用 */
#define DIVY_SVRML_FLOGIN_CHANGENOW_PREFIX	"firstlogin_changenow_ml"

#endif	/* DIVY_SUPPORT_PASSPOLICY */

/* オーナー変更通知 */
#define DIVY_SVRML_OWNER_CHANGE_NOTIFICATION_PREFIX	"owner_notification_ml"

/* 開封通知 */
#define DIVY_SVRML_OWNER_CONFIRMREADING_PREFIX	"confirmreading_ml"

/* ログイン失敗によるアカウントロック通知 */
#define DIVY_SVRML_FAILED_LOGIN_ACCOUNT_LOCK_PREFIX "failedloginlock_ml"

/* BOX参照通知 */
#define DIVY_SVRML_BOX_OPEN_NOTIFICATION_PREFIX "box_opennotification_ml"

/* BOX公開レポート */
#define DIVY_SVRML_BOX_OPEN_REPORT_PREFIX "box_open_report_ml"

/* BOX招待通知 */
#define DIVY_SVRML_BOX_INVITATION_PREFIX "box_invitation_ml"

/* アップロードポリシー適用通知 */
#define DIVY_SVRML_GROUP_POLICY_APPLY_NOTICE_PREFIX "upload_policy_apply_notice_ml"

/* アップロードポリシールール接触通知 */
#define DIVY_SVRML_GROUP_POLICY_VIOLATION_PREFIX "upload_policy_violation_ml"

/*------------------------------------------------------------------------------
  Define structures
  ----------------------------------------------------------------------------*/
/**
 * メールの送信先のフォルダ情報を保持する
 */
typedef struct __collectionInfo collectionInfo;
struct __collectionInfo {
	char *uri;
	char *etag;
	char *displayname;
};

/*
 * メール監視をしているフォルダの情報構造体
 */
typedef struct __urilist urilist;

/*
 * URIのメール監視情報が含まれている
 */
struct __urilist {
	char    *uri;		 /* メールのURI */
	MlWatchPlace watchplace; /* このURIの場所（SRCなのかDSTなのか？) */
	int method;		 /* メソッド番号(M_PUT, M_GETなど) */
	MlAddress *address;	 /* このURIを監視しているユーザアドレス */
	urilist *next;		 /* 次へのポインタ */
};

typedef	struct __userlist userlist;

struct __userlist {
	MlWatchPlace 	watchplace;
	MlAddress 	*address;
	userlist 	*next;
};

/**
 * 文字列置換処理で使用する列挙型、構造体
 * (note)
 * 	キーワードは「@」で始まり「@」で閉じられていなければ
 * 	なりません。
 */
/* 置換パラメータの数値表現 */
typedef enum __divy_replace_key_id	divy_replace_key_id;
enum __divy_replace_key_id {

	DIVY_ML_REP_ADMIN = 0,		/* 管理者のフルネーム     */
	DIVY_ML_REP_USERNAME,		/* ユーザのフルーネーム   */
	DIVY_ML_REP_USERID,		/* ユーザのユーザID       */
	DIVY_ML_REP_PASSWORD,		/* ユーザのパスワード     */
	DIVY_ML_REP_MAILADDR,		/* ユーザのメールアドレス */
	DIVY_ML_REP_LIMITSIZE,		/* ユーザQuota(size)      */
	DIVY_ML_REP_LIMITCOUNT,		/* ユーザQuota(file)      */
	DIVY_ML_REP_REGDAY,		/* 登録日                 */
	DIVY_ML_REP_USERCOMMENT,	/* ユーザのコメント       */
	DIVY_ML_REP_USERTYPE,		/* ユーザの種類           */
	DIVY_ML_REP_USERPRIVILEGE,	/* ユーザの権限           */
	DIVY_ML_REP_LASTACCESS,		/* ユーザの最終アクセス日 */
	DIVY_ML_REP_PUBLISHERPRIVILEGE,	/* ユーザのリソース公開権限 */
	DIVY_ML_REP_GRPCONSTRAINTSIGNORE,	/* グループ制約の適用を受けるかどうか */
	DIVY_ML_REP_USERSTATE,		/* ユーザの状態           */
	DIVY_ML_REP_USEREXPIRATION,	/* ユーザの有効期限       */
	DIVY_ML_REP_USEREOWNERNAME,	/* ユーザのオーナ名       */
	DIVY_ML_REP_MAXUSERCREATION,	/* 作成可能ユーザ数       */
	DIVY_ML_REP_CONTROLOTHERUSER,	/* Otherユーザの管理を許可するかどうか */
	DIVY_ML_REP_ADMINMAIL,		/* 管理者のメールアドレス */
	DIVY_ML_REP_LOCATIONUTIL,	/* クライアント更新フォルダのURL */
	DIVY_ML_REP_ROTURL,		/* ロケーションまでのURL  */
	DIVY_ML_REP_FILENAME,		/* ウィルス感染ファイルの名前   */
	DIVY_ML_REP_FILESIZE,		/* ウィルス感染ファイルのサイズ */
	DIVY_ML_REP_FILETYPE,		/* ウィルス感染ファイルの種類   */
	DIVY_ML_REP_SRCURI,		/* メールのsource URI      */
	DIVY_ML_REP_DSTURI,		/* メールのdestination URI */
	DIVY_ML_REP_FROM,		/* 操作者のユーザID    */
	DIVY_ML_REP_METHOD,		/* 操作メソッド        */
	DIVY_ML_REP_OPERATION_JA,	/* 操作の種類 (日本語) */
	DIVY_ML_REP_OPERATION_EN,	/* 操作の種類 (英語)   */
	DIVY_ML_REP_WATCHFOLDER,	/* 監視フォルダの名前  */
	DIVY_ML_REP_SRC_WATCHFOLDER,	/* 監視フォルダの名前(src) */
	DIVY_ML_REP_DST_WATCHFOLDER,	/* 監視フォルダの名前(dst) */
	/* (2007.09.14 Fri) 廃止 */
#if 0
	DIVY_ML_REP_ETAG,		/* E-tag */ /* (2007.09.14 Fri) 廃止 */
#endif
	DIVY_ML_REP_GETLASTMODIFIED,	/* getlastmodified     */
	DIVY_ML_REP_BRANDNAME,		/* ブランド名          */
#ifdef DIVY_SUPPORT_PASSPOLICY
	 DIVY_ML_REP_PASSWORD_PERIOD,	/* パスワードの有効期限が切れた日付 */
	 DIVY_ML_REP_PASSWORD_PROBATION,	/* 変更までの猶予期限 */
	 DIVY_ML_REP_FIRSTLOGIN_PROBATION,	/* 初回ログインのパスワード有効期限が切れる日付 */
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	 DIVY_ML_REP_APPOINTEDLEADER,	/* グループリーダに任命されているかどうか */
	 DIVY_ML_REP_GROUPNAME,			/* グループ名称 */
	 DIVY_ML_REP_HAS_GROUPLEADER,	/* 対象ユーザがグループリーダ管理下かどうか (1/0) */
	 DIVY_ML_REP_TICKETURL,			/* ticket URL */
	 DIVY_ML_REP_SRC_TICKETURL,		/* src のticket URL (MOVE, COPY) */
	 DIVY_ML_REP_DST_TICKETURL,		/* dst のticket URL (MOVE, COPY) */
	 DIVY_ML_REP_SUPPORTED_LEADER,	/* グループリーダ機能のサポート有無 (1/0) */
	 DIVY_ML_REP_PARENT_FOLDER,		/* (開封通知) (監視メール)親フォルダ名 */
	 DIVY_ML_REP_DOWNLOAD_DATE,		/* (開封通知) ダウンロード日付 */
	 DIVY_ML_REP_DOWNLOAD_USERNAME,	/* (開封通知) ダウンロードユーザ名 */
	 DIVY_ML_REP_REMOTE_ADDR,       /* (開封通知)(アカウントロック) リモートアドレス */
	 DIVY_ML_REP_FAILELOGIN_COUNT,	/* (アカウントロック) ログイン失敗回数 */
	 DIVY_ML_REP_BOX_GREETING,		/* (BOX)あいさつ文 */
	 DIVY_ML_REP_BOX_SHORTEN_URL,	/* (BOX)BOX短縮URL */
	 DIVY_ML_REP_BOX_OPEN_FILE_LIST, /* (BOX)公開したファイル一覧(改行複数件) */
	 DIVY_ML_REP_BOX_OPEN_FILE_COUNT,/* (BOX)公開したフォルダにあるファイル数 */
	 DIVY_ML_REP_BOX_OPEN_PUB_DATE, /* (BOX)公開した日 */
	 DIVY_ML_REP_BOX_OPEN_PUB_EXPIRED,  /* (BOX)有効期限 */
	 DIVY_ML_REP_BOX_HAS_ACCESSCODE, /* (BOX)アクセスコードの有無(0:無/1:有) */
	 DIVY_ML_REP_BOX_TO_MAILADDRESS, /* (BOX)受信先メールアドレス一覧 */
	 DIVY_ML_REP_GRP_POLICY_UPDATEDATE, /* (アップデートポリシー更新日 */
	 DIVY_ML_REP_GRP_POLICY_MATCHTYPE, /* (アップデートポリシー)マッチタイプ */
	 DIVY_ML_REP_GRP_POLICY_FILE,	/* (アップデートポリシー)ルールファイル名 */
	 DIVY_ML_REP_GRP_POLICY_SUFFIX,	/* (アップデートポリシー)ルール拡張子 */
	 DIVY_ML_REP_GRP_POLICY_LENGTH,	/* (アップデートポリシー)ルール長さ */
	 DIVY_ML_REP_GRP_POLICY_CHAR,	/* (アップデートポリシー)ルール文字種 */
	 DIVY_ML_REP_GRP_POLICY_HIDDEN,	/* (アップデートポリシー)非公開 */
	 DIVY_ML_REP_GRP_POLICY_ACTIVE, /* (アップデートポリシーポリシーアクティブ */
	 DIVY_ML_REP_GRP_POLICY_PREFIX_PASS, /* (アップデートポリシー)ファイル名はパス */
	 DIVY_ML_REP_GRP_POLICY_SUFFIX_PASS, /* (アップデートポリシー)拡張子はパス */
	 DIVY_ML_REP_GRP_POLICY_CHAR_PASS, /* (アップデートポリシー)文字種はパス */
	 DIVY_ML_REP_GRP_POLICY_LEN_PASS, /* (アップデートポリシー)長さはパス */
	 DIVY_ML_REP_2FA_TOKEN,			/* (2FA認証）トークン文字列 */
	 DIVY_ML_REP_2FA_EXPIRATION,	/* (2FA認証）トークン期限 */
	DIVY_ML_REP_END			/* sentinel */
};

/* 置換パラメータと値を持つ構造体 */
typedef struct __divy_replace_keydata	divy_replace_keydata;
struct __divy_replace_keydata {
	divy_replace_key_id id;	/* key の列挙型 */
	const char *key;	/* 置換対象のキーワード */
	const char *value;	/* 置換する値           */
	apr_size_t key_len;	/* key の文字列長(パフォーマンス向上のため) */

	divy_replace_keydata *next;
};

/* 置換パラメータとその数値表現(列挙型)の関係を保持する構造体 */
typedef struct __divy_replace_keydata_elem	divy_replace_keydata_elem;
struct __divy_replace_keydata_elem {
	divy_replace_key_id ikey;	/* 置換パラメータの数値表現 */
	const char *skey;		/* 置換パラメータ  */
};

/* 置換パラメータとその数値表現を記録する配列 */
static const divy_replace_keydata_elem _replace_keydata[] =
{
	{ DIVY_ML_REP_ADMIN,		"@admin@"		},
	{ DIVY_ML_REP_USERNAME,		"@username@"		},
	{ DIVY_ML_REP_USERID,		"@userid@"		},
	{ DIVY_ML_REP_PASSWORD,		"@password@"		},
	{ DIVY_ML_REP_MAILADDR,		"@mailaddr@"		},
	{ DIVY_ML_REP_LIMITSIZE,	"@limitsize@"		},
	{ DIVY_ML_REP_LIMITCOUNT,	"@limitcount@"		},
	{ DIVY_ML_REP_REGDAY,		"@regday@"		},
	{ DIVY_ML_REP_USERCOMMENT,	"@user-comment@"	},
	{ DIVY_ML_REP_USERTYPE,		"@user-type@"		},
	{ DIVY_ML_REP_USERPRIVILEGE,	"@user-privilege@"	},
	{ DIVY_ML_REP_LASTACCESS,   "@lastaccess@" },
	{ DIVY_ML_REP_PUBLISHERPRIVILEGE,"@publisher-privilege@"},
	{ DIVY_ML_REP_GRPCONSTRAINTSIGNORE,"@groupconstraints-ignore@"},
	{ DIVY_ML_REP_USERSTATE,	"@user-state@"		},
	{ DIVY_ML_REP_USEREXPIRATION,	"@user-expiration@"	},
	{ DIVY_ML_REP_USEREOWNERNAME,	"@user-owner@"	},
	{ DIVY_ML_REP_MAXUSERCREATION,	"@user-maxcreation@"	},
	{ DIVY_ML_REP_CONTROLOTHERUSER,	"@user-controlother@"	},
	{ DIVY_ML_REP_ADMINMAIL,	"@adminmail@"		},
	{ DIVY_ML_REP_LOCATIONUTIL,	"@location_util@"	},
	{ DIVY_ML_REP_ROTURL,		"@rooturl@"		},
	{ DIVY_ML_REP_FILENAME,		"@filename@"		},
	{ DIVY_ML_REP_FILESIZE,		"@filesize@"		},
	{ DIVY_ML_REP_FILETYPE,		"@filetype@"		},
	{ DIVY_ML_REP_SRCURI,		"@srcuri@"		},
	{ DIVY_ML_REP_DSTURI,		"@dsturi@"		},
	{ DIVY_ML_REP_FROM,			"@from@"		},
	{ DIVY_ML_REP_METHOD,		"@method@"		},
	{ DIVY_ML_REP_OPERATION_JA,	"@operation_ja@"	},
	{ DIVY_ML_REP_OPERATION_EN,	"@operation_en@"	},
	{ DIVY_ML_REP_WATCHFOLDER,		"@watchfolder@"		},
	{ DIVY_ML_REP_SRC_WATCHFOLDER,	"@src_watchfolder@"		},
	{ DIVY_ML_REP_DST_WATCHFOLDER,	"@dst_watchfolder@"		},
#if 0
	{ DIVY_ML_REP_ETAG,		"@etag@"		},	/* (2007.09.14 Fri) 廃止 */
#endif
	{ DIVY_ML_REP_GETLASTMODIFIED,	"@getlastmodified@"	},
	{ DIVY_ML_REP_BRANDNAME,		"@brandname@"		},
#ifdef DIVY_SUPPORT_PASSPOLICY
	{ DIVY_ML_REP_PASSWORD_PERIOD,		"@password_period@"	},
	{ DIVY_ML_REP_PASSWORD_PROBATION,	"@password_probation@" },
	{ DIVY_ML_REP_FIRSTLOGIN_PROBATION,	"@firstlogin_probation@" },
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	{ DIVY_ML_REP_APPOINTEDLEADER,	"@is-appointed-groupleader@"	},
	{ DIVY_ML_REP_GROUPNAME,		"@group-name@"	},
	{ DIVY_ML_REP_HAS_GROUPLEADER,	"@has-groupleader@"	},
	{ DIVY_ML_REP_TICKETURL,		"@ticketurl@"	},
	{ DIVY_ML_REP_SRC_TICKETURL,	"@src_ticketurl@"	},
	{ DIVY_ML_REP_DST_TICKETURL,	"@dst_ticketurl@"	},
	{ DIVY_ML_REP_SUPPORTED_LEADER,	"@is-supported-groupleader@"	},
	{ DIVY_ML_REP_PARENT_FOLDER,		"@parent_folder@"	},
	{ DIVY_ML_REP_DOWNLOAD_DATE,		"@download_date@"	},
	{ DIVY_ML_REP_DOWNLOAD_USERNAME,	"@download_username@"	},
	{ DIVY_ML_REP_REMOTE_ADDR,          "@remote_addr@"   },
	{ DIVY_ML_REP_FAILELOGIN_COUNT,     "@failedlogincount@" },
	{ DIVY_ML_REP_BOX_GREETING,		"@boxgreeting@" },
	{ DIVY_ML_REP_BOX_SHORTEN_URL,  "@box_shorten_url@" },
	{ DIVY_ML_REP_BOX_OPEN_FILE_LIST,  "@box_open_file_list@" },
	{ DIVY_ML_REP_BOX_OPEN_FILE_COUNT, "@box_open_file_count@" },
	{ DIVY_ML_REP_BOX_OPEN_PUB_DATE, "@box_open_pub_date@" },
	{ DIVY_ML_REP_BOX_OPEN_PUB_EXPIRED, "@box_open_pub_expired@"},
	{ DIVY_ML_REP_BOX_HAS_ACCESSCODE, "@box_has_accesscode@" },
	{ DIVY_ML_REP_BOX_TO_MAILADDRESS, "@box_to_mailaddress@" },
	{ DIVY_ML_REP_GRP_POLICY_UPDATEDATE, "@updatedate@" },
	{ DIVY_ML_REP_GRP_POLICY_MATCHTYPE, "@rule_matchtype@" },
	{ DIVY_ML_REP_GRP_POLICY_FILE, "@rule_file@" },
	{ DIVY_ML_REP_GRP_POLICY_SUFFIX, "@rule_suffix@" },
	{ DIVY_ML_REP_GRP_POLICY_LENGTH, "@rule_length@" },
	{ DIVY_ML_REP_GRP_POLICY_CHAR, "@rule_char@" },
	{ DIVY_ML_REP_GRP_POLICY_HIDDEN, "@rule_hidden@" },
	{ DIVY_ML_REP_GRP_POLICY_ACTIVE, "@uploadpolicyactive@" },
	{ DIVY_ML_REP_GRP_POLICY_PREFIX_PASS, "@prefix_pass@" },
	{ DIVY_ML_REP_GRP_POLICY_SUFFIX_PASS, "@suffix_pass@" },
	{ DIVY_ML_REP_GRP_POLICY_CHAR_PASS, "@char_pass@" },
	{ DIVY_ML_REP_GRP_POLICY_LEN_PASS, "@len_pass@" },
	{ DIVY_ML_REP_2FA_TOKEN, "@2fa_token@" },
	{ DIVY_ML_REP_2FA_EXPIRATION, "@2fa_expiration@"},
};

/* _replace_keydata の要素数 */
#define DIVY_REPLACE_KEYDATA_LEN sizeof(_replace_keydata)/sizeof(divy_replace_keydata_elem)

/**
 * サーバ主体メールメッセージの日本語、英語対応表
 */
typedef enum __divy_svrml_id	divy_svrml_id;
enum __divy_svrml_id {
	DIVY_SVRML_GET = 0,
	DIVY_SVRML_PUT,
	DIVY_SVRML_DELETE,
	DIVY_SVRML_MOVE,
	DIVY_SVRML_COPY,
	DIVY_SVRML_NOTIFYCL_SUBJECT,
	DIVY_SVRML_NOTIFYCL_USERTYPE_ADMIN,
	DIVY_SVRML_NOTIFYCL_USERTYPE_NORMAL,
	DIVY_SVRML_NOTIFYCL_USERTYPE_LIMITED,
	DIVY_SVRML_NOTIFYCL_USERTYPE_GROUPLEADER,
	DIVY_SVRML_NOTIFYCL_PRIVILEGE_READWRITE,
	DIVY_SVRML_NOTIFYCL_PRIVILEGE_UPLOAD,
	DIVY_SVRML_NOTIFYCL_PRIVILEGE_READ,
	DIVY_SVRML_NOTIFYCL_PRIVILEGE_PUBLISHER,
	DIVY_SVRML_NOTIFYCL_PRIVILEGE_NOPUBLISHER,
	DIVY_SVRML_NOTIFYCL_PRIVILEGE_GRPCONSTRAINTSIGNORE,
	DIVY_SVRML_NOTIFYCL_PRIVILEGE_GRPCONSTRAINTSAPPLY,
	DIVY_SVRML_NOTIFYCL_STATE_ACTIVE,
	DIVY_SVRML_NOTIFYCL_STATE_INACTIVE,
	DIVY_SVRML_NOTIFYCL_STATE_EXPIRE,
	DIVY_SVRML_NOTIFYCL_ALLOW,
	DIVY_SVRML_NOTIFYCL_NOT_ALLOW,
	DIVY_SVRML_NOTIFYCL_UNLIMITED_USER_CREATION,
	DIVY_SVRML_VIRUS_DETECTION_SUBJECT,
#ifdef DIVY_SUPPORT_PASSPOLICY
	DIVY_SVRML_NOTIFY_PASS_PROBATION_SUBJECT,
	DIVY_SVRML_NOTIFY_PASS_EXPIRED_SUBJECT,
	DIVY_SVRML_NOTIFY_FLOGIN_PROBATION_SUBJECT,
	DIVY_SVRML_NOTIFY_FLOGIN_EXPIRED_SUBJECT,
	DIVY_SVRML_NOTIFY_FLOGIN_CHANGENOW_SUBJECT,
#endif	/* DIVY_SUPPORT_PASSPOLICY */
	DIVY_SVRML_NOTIFY_CHANGELEADER_SUBJECT,
	DIVY_SVRML_NOTIFY_CONFIRMREADING_SUBJECT,

	DIVY_SVRML_END	/* sentinel */
};

/* メッセージIDとメッセージの関係を保持する構造体の定義 */
typedef struct __divy_svrml_msg	divy_svrml_msg;
struct __divy_svrml_msg {
	divy_svrml_id id;
	const char *en;
	const char *ja;
};

/* メッセージIDとメッセージの関係を保持する配列 */
static const divy_svrml_msg _svrml_msgs[] =
{
	/* オペレーション名 */
	{ DIVY_SVRML_GET,
	  "GET",
	  "\xE5\x8F\x96\xE5\xBE\x97"
	},
	{ DIVY_SVRML_PUT,
	  "PUT",
	  "\xE7\x99\xBB\xE9\x8C\xB2\xE5\x8F\x88\xE3\x81\xAF\xE5\xA4\x89\xE6\x9B\xB4"
	},
	{ DIVY_SVRML_DELETE,
	  "DELETE",
	  "\xE5\x89\x8A\xE9\x99\xA4"
	},
	{ DIVY_SVRML_MOVE,
	  "MOVE",
	  "\xE7\xA7\xBB\xE5\x8B\x95"
	},
	{ DIVY_SVRML_COPY,
	  "COPY",
	  "\xE3\x82\xB3\xE3\x83\x94\xE3\x83\xBC"
	},
	/* (ユーザ作成通知メール用) */
	{ DIVY_SVRML_NOTIFYCL_SUBJECT,
	  "Account creation complete",
	  "\xE3\x82\xA2\xE3\x82\xAB\xE3\x82\xA6\xE3\x83\xB3\xE3\x83\x88\xE4\xBD\x9C"
	  "\xE6\x88\x90\xE5\xAE\x8C\xE4\xBA\x86\xE5\x8F\x8A\xE3\x81\xB3\xE5\x88\xA9"
	  "\xE7\x94\xA8\xE9\x96\x8B\xE5\xA7\x8B\xE3\x81\xAE\xE3\x81\x94\xE6\xA1\x88"
	  "\xE5\x86\x85"
	},
	/* 管理者 */
	{ DIVY_SVRML_NOTIFYCL_USERTYPE_ADMIN,
	  "Administrator",
	  "\xE7\xAE\xA1\xE7\x90\x86\xE8\x80\x85"
	},
	/* 一般ユーザー */
	{ DIVY_SVRML_NOTIFYCL_USERTYPE_NORMAL,
	  "General User",
	  "\xE4\xB8\x80\xE8\x88\xAC\xE3\x83\xA6\xE3\x83\xBC\xE3\x82\xB6\xE3\x83\xBC"
	},
	/* 制限ユーザー */
	{ DIVY_SVRML_NOTIFYCL_USERTYPE_LIMITED,
	  "Limited Access User",
	  "\xE5\x88\xB6\xE9\x99\x90\xE3\x83\xA6\xE3\x83\xBC\xE3\x82\xB6\xE3\x83\xBC"
	},
	/* グループリーダー */
	{ DIVY_SVRML_NOTIFYCL_USERTYPE_GROUPLEADER,
	  "Group leader",
	  "\xE3\x82\xB0\xE3\x83\xAB\xE3\x83\xBC\xE3\x83\x97\xE3\x83\xAA\xE3\x83\xBC\xE3\x83\x80\xE3\x83\xBC"
	},
	/* 読み込み・書き込み */
	{ DIVY_SVRML_NOTIFYCL_PRIVILEGE_READWRITE,
	  "Read/Write",
	  "\xE8\xAA\xAD\xE3\x81\xBF\xE8\xBE\xBC\xE3\x81\xBF\xE3\x83\xBB\xE6\x9B\xB8"
	  "\xE3\x81\x8D\xE8\xBE\xBC\xE3\x81\xBF"
	},
	/* アップロード専用 */
	{ DIVY_SVRML_NOTIFYCL_PRIVILEGE_UPLOAD,
	  "Upload Only",
	  "\xE3\x82\xA2\xE3\x83\x83\xE3\x83\x97\xE3\x83\xAD\xE3\x83\xBC\xE3\x83\x89"
	  "\xE5\xB0\x82\xE7\x94\xA8"
	},
	/* 読み込み専用 */
	{ DIVY_SVRML_NOTIFYCL_PRIVILEGE_READ,
	  "Read Only",
	  "\xE8\xAA\xAD\xE3\x81\xBF\xE8\xBE\xBC\xE3\x81\xBF\xE5\xB0\x82\xE7\x94\xA8"
	},
	/* リソース公開を許可 */
	{ DIVY_SVRML_NOTIFYCL_PRIVILEGE_PUBLISHER,
	  "Permit publishing resources",
	  "\xE3\x83\xAA\xE3\x82\xBD\xE3\x83\xBC\xE3\x82\xB9\xE5\x85\xAC\xE9\x96\x8B"
	  "\xE3\x82\x92\xE8\xA8\xB1\xE5\x8F\xAF"
	},
	/* リソース公開を非許可 */
	{ DIVY_SVRML_NOTIFYCL_PRIVILEGE_NOPUBLISHER,
	  "No permit publishing resources",
	  "\xE3\x83\xAA\xE3\x82\xBD\xE3\x83\xBC\xE3\x82\xB9\xE5\x85\xAC\xE9\x96\x8B"
	  "\xE3\x82\x92\xE9\x9D\x9E\xE8\xA8\xB1\xE5\x8F\xAF"
	},
	/* グループ制約を無視 */
	{ DIVY_SVRML_NOTIFYCL_PRIVILEGE_GRPCONSTRAINTSIGNORE,
	  "Impervious to constraints of group",
	  "\xE3\x82\xB0\xE3\x83\xAB\xE3\x83\xBC\xE3\x83\x97\xE5\x88\xB6"
	  "\xE7\xB4\x84\xE3\x82\x92\xE7\x84\xA1\xE8\xA6\x96"
	},
	/* グループ制約を適用 */
	{ DIVY_SVRML_NOTIFYCL_PRIVILEGE_GRPCONSTRAINTSAPPLY,
	  "Apply to constraints of group",
	  "\xE3\x82\xB0\xE3\x83\xAB\xE3\x83\xBC\xE3\x83\x97\xE5\x88\xB6"
	  "\xE7\xB4\x84\xE3\x82\x92\xE9\x81\xA9\xE7\x94\xA8"
	},
	/* 有効 */
	{ DIVY_SVRML_NOTIFYCL_STATE_ACTIVE,
	  "Active",
	  "\xE6\x9C\x89\xE5\x8A\xB9"
	},
	/* 無効 */
	{ DIVY_SVRML_NOTIFYCL_STATE_INACTIVE,
	  "Not Active",
	  "\xE7\x84\xA1\xE5\x8A\xB9"
	},
	/* 有効期限切れ */
	{ DIVY_SVRML_NOTIFYCL_STATE_EXPIRE,
	  "Expired",
	  "\xE6\x9C\x89\xE5\x8A\xB9\xE6\x9C\x9F\xE9\x99\x90\xE5\x88\x87\xE3\x82\x8C"
	},
	/* 許可 */
	{ DIVY_SVRML_NOTIFYCL_ALLOW,
	  "Allow",
	  "\xE8\xA8\xB1\xE5\x8F\xAF"
	},
	/* 非許可 */
	{ DIVY_SVRML_NOTIFYCL_NOT_ALLOW,
	  "Not allow",
	  "\xE9\x9D\x9E\xE8\xA8\xB1\xE5\x8F\xAF"
	},
	/* 無制限 */
	{ DIVY_SVRML_NOTIFYCL_UNLIMITED_USER_CREATION,
	  "Unlimited",
	  "\xE7\x84\xA1\xE5\x88\xB6\xE9\x99\x90"
	},
	/* (ウィルス感染報告用, ウィルス検索失敗用) */
	{ DIVY_SVRML_VIRUS_DETECTION_SUBJECT,
	  "Anti Virus Engine delete uploaded file.",
	  "\xE3\x82\xA6\xE3\x82\xA3\xE3\x83\xAB\xE3\x82\xB9\xE6\xA4\x9C\xE7\xB4\xA2"
	  "\xE3\x82\xA8\xE3\x83\xB3\xE3\x82\xB8\xE3\x83\xB3\xE3\x81\x8C\xE3\x83\x95"
	  "\xE3\x82\xA1\xE3\x82\xA4\xE3\x83\xAB\xE3\x82\x92\xE5\x89\x8A\xE9\x99\xA4"
	  "\xE3\x81\x97\xE3\x81\xBE\xE3\x81\x97\xE3\x81\x9F"
	},
#ifdef DIVY_SUPPORT_PASSPOLICY
	/* パスワードの有効期限が切れました */
	{ DIVY_SVRML_NOTIFY_PASS_PROBATION_SUBJECT,
	  "The user's password was expired",
	  "\xE3\x83\x91\xE3\x82\xB9\xE3\x83\xAF\xE3\x83\xBC\xE3\x83\x89\xE3\x81\xAE"
	  "\xE6\x9C\x89\xE5\x8A\xB9\xE6\x9C\x9F\xE9\x99\x90\xE3\x81\x8C\xE5\x88\x87"
	  "\xE3\x82\x8C\xE3\x81\xBE\xE3\x81\x97\xE3\x81\x9F"
	},
	/* パスワードがロックされました */
	{ DIVY_SVRML_NOTIFY_PASS_EXPIRED_SUBJECT,
	  "The user's password was locked",
	  "\xE3\x83\x91\xE3\x82\xB9\xE3\x83\xAF\xE3\x83\xBC\xE3\x83\x89\xE3\x81\x8C"
	  "\xE3\x83\xAD\xE3\x83\x83\xE3\x82\xAF\xE3\x81\x95\xE3\x82\x8C\xE3\x81\xBE"
	  "\xE3\x81\x97\xE3\x81\x9F"
	},
	/* 初期パスワード変更のお願い */
	{ DIVY_SVRML_NOTIFY_FLOGIN_PROBATION_SUBJECT,
	  "Please change the default password.",
	  "\xE5\x88\x9D\xE6\x9C\x9F\xE3\x83\x91\xE3\x82\xB9\xE3\x83\xAF\xE3\x83\xBC"
	  "\xE3\x83\x89\xE5\xA4\x89\xE6\x9B\xB4\xE3\x81\xAE\xE3\x81\x8A\xE9\xA1\x98"
	  "\xE3\x81\x84"
	},
	/* 初期パスワードがロックされました */
	{ DIVY_SVRML_NOTIFY_FLOGIN_EXPIRED_SUBJECT,
	  "The default password was locked.",
	  "\xE5\x88\x9D\xE6\x9C\x9F\xE3\x83\x91\xE3\x82\xB9\xE3\x83\xAF\xE3\x83\xBC"
	  "\xE3\x83\x89\xE3\x81\x8C\xE3\x83\xAD\xE3\x83\x83\xE3\x82\xAF\xE3\x81\x95"
	  "\xE3\x82\x8C\xE3\x81\xBE\xE3\x81\x97\xE3\x81\x9F"
	},
	/* 初期パスワード変更のお願い */
	{ DIVY_SVRML_NOTIFY_FLOGIN_CHANGENOW_SUBJECT,
	  "Please change the default password now.",
	  "\xE5\x88\x9D\xE6\x9C\x9F\xE3\x83\x91\xE3\x82\xB9\xE3\x83\xAF\xE3\x83\xBC"
	  "\xE3\x83\x89\xE5\xA4\x89\xE6\x9B\xB4\xE3\x81\xAE\xE3\x81\x8A\xE9\xA1\x98"
	  "\xE3\x81\x84"
	},
	/* オーナー変更通知 */
	{ DIVY_SVRML_NOTIFY_CHANGELEADER_SUBJECT,
	  "Group owner change notification.",
	  "\xE3\x82\xAA\xE3\x83\xBC\xE3\x83\x8A\xE3\x83\xBC\xE5\xA4\x89\xE6\x9B\xB4"
	  "\xE9\x80\x9A\xE7\x9F\xA5"
	},
	/* ファイル開封通知 */
	{ DIVY_SVRML_NOTIFY_CONFIRMREADING_SUBJECT,
	  "The notification of confirm-reading",
	  "\xE3\x83\x95\xE3\x82\xA1\xE3\x82\xA4\xE3\x83\xAB\xE9\x96\x8B\xE5\xB0\x81"
	  "\xE9\x80\x9A\xE7\x9F\xA5"
	},
#endif	/* DIVY_SUPPORT_PASSPOLICY */
};

#define DIVY_SVRML_MSGS_LEN	sizeof(_svrml_msgs)/sizeof(divy_svrml_msg)



/*--------------------------------------------------------------
  Define Global values
  --------------------------------------------------------------*/
/*
 * メールプロバイダ専用のプール
 * (note)
 * 	pchildのサブプールとなります。
 */
static apr_pool_t *ml_p;

/*
 * グローバル変数を排他的に使用する際に使うmutex
 */
static apr_thread_mutex_t *gth_ml_mutex;

/*
 * 初期化完了状態を記録するハッシュ
 *
 * key : _get_ml_provider_id() の値
 * val : ML_PROVIDER_INIT_FINISH / ML_PROVIDER_INIT_UNFINISH
 */
#define ML_PROVIDER_INIT_FINISH   "ml_f"	/* 完了した */
#define ML_PROVIDER_INIT_UNFINISH "ml_u"	/* 完了していない */
static apr_hash_t *ml_initfinish_hash;

/*
 * メールプロバイダ構造体を保持するハッシュ
 * key : _get_ml_provider_id() の値
 * val : MlDataSource *
 */
static apr_hash_t *mlprv_hash;

/*------------------------------------------------------------------------------
  Declare private prototype function
  ----------------------------------------------------------------------------*/
static int _divy_ml_alloc_address(request_rec *r, urilist *l, userlist **usrlist);
static int _divy_ml_uri_parse(apr_pool_t *pool, urilist** list,
	       						MlWatchPlace w,
	       						const char *uri,
							int method);

static const char * _get_ml_provider_id(request_rec *r, const char *providerType);
static int _init_ml_provider(request_rec *r,
					const char *providerid);
static int _registe_ml_provider(request_rec *r, 
					const char *providerid,
					MlDataSource *mlds);
static int _is_mlenv_ready(const char *providerid);
static void _set_mlenv_is_ready(const char *providerid);
static apr_status_t _cleanup_ml_env(void *data);
static int _divy_ml_encodetext(apr_pool_t *p, const char *str,
	       		       const char *charset, char **to_str,
			       int folding);
static char *_divy_ml_address_strtok(apr_pool_t *pool, char *address, char **token_ctx);
static char *_get_oneclick_url(request_rec *r, divy_rdbo_resource *rdb_r);
static char *_get_ml_watchfolder(request_rec *r, divy_rdbo_resource *rdb_r);
static int _replace_ml_keyword(apr_pool_t *p, const char *src,
					divy_replace_keydata *keydata,
					divy_sbuf *sbuf);
static void _push_replace_param(apr_pool_t *p,
					divy_replace_key_id ikey,
					const char *value,
					divy_replace_keydata **first);
static const char * _get_svrml_string(divy_svrml_id id, const char *lang);
static int _make_svrml_data(apr_pool_t *wp, const char *mltemp_prefix,
					const char *lang, request_rec *r,
					divy_replace_keydata *keydata, MlMessage *msg);
static int _open_svrml_template(apr_pool_t *wp, const char *mltemp_prefix,
					const char *lang, request_rec *r, char **path, apr_file_t **fd_r);
static const char * _get_ml_language_param(request_rec *r);
static int _send_watchmail(request_rec *r, apr_pool_t *wp, divy_ml_mailwatch_data *mldata);
static char * _extract_inputed_body(request_rec *r, apr_pool_t *wp, char *body);
static int _xmldat2mlmsg(request_rec *r, apr_pool_t *wp, divy_ml_mailwatch_data *mldata, MlMessage **msg);
static int _is_range_start(request_rec *r);
static const char * _choose_brand_name(request_rec *r, const char *msgbrandname);
static void _build_user_notification_body(request_rec *r, apr_pool_t *wp,
						divy_replace_keydata *keydata, divy_rdbo_usr *usr_pr);
/*--------------------------------------------------------------
  Define public function
  --------------------------------------------------------------*/
/**
 * メールプロバイダの環境を初期化する。
 * この関数は複数のスレッドからアクセスされてはなりません。
 */
DIVY_DECLARE(void) init_mlprv_env(apr_pool_t *pchild, server_rec *s)
{
	apr_status_t rv;

	TRACE_S(s);

	/* メールプロバイダが利用するプールの生成 */
	apr_pool_create(&ml_p, pchild);

	/* クリーンアップ関数を登録する */
	apr_pool_cleanup_register(ml_p, NULL, _cleanup_ml_env,
					apr_pool_cleanup_null);
	/* mutexの生成 */
	rv = apr_thread_mutex_create(&gth_ml_mutex,
					APR_THREAD_MUTEX_UNNESTED, ml_p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to create gth_ml_mutex. (code = %d)", rv);
	}

	ml_initfinish_hash = apr_hash_make(ml_p);
	mlprv_hash         = apr_hash_make(ml_p);
}

/**
 * 指定されたproviderTypeのメールプロバイダ(MlDataSource)を取得する。
 *
 * @param providerType const char *
 *
 */
DIVY_DECLARE(MlDataSource *) lookup_ml_provider(request_rec *r,
						const char *providerType)
{
	MlDataSource *mlds;
	apr_status_t rv;
	apr_pool_t *p = r->pool;
	const char *providerid;
	int ret;

	if (providerType == NULL || *providerType == '\0') {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"providerType is NULL.");
		return NULL;
	}

	/* プロバイダIDを取得 */
	providerid = _get_ml_provider_id(r, providerType);

	/* mutex ロックをかける */
	rv = apr_thread_mutex_lock(gth_ml_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex lock. (code = %d)", rv);
		return NULL;
	}

	/*
	 * メールプロバイダが初期化されていなければ初期化する
	 */
	if (!_is_mlenv_ready(providerid)) {

		ret = _init_ml_provider(r, providerid);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to initialize mail provider.");
			mlds = NULL;
		}
		else {
			/* MlDataSource の取得 */
			mlds = apr_hash_get(mlprv_hash, providerid,
						APR_HASH_KEY_STRING);
		}
	}
	else {
		/* MlDataSource の取得 */
		mlds = apr_hash_get(mlprv_hash, providerid, APR_HASH_KEY_STRING);
	}

	/* mutex ロックを解除 */
	rv = apr_thread_mutex_unlock(gth_ml_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex unlock. (code = %d)", rv);
		return mlds;	/* 大目に見る */
	}

	return mlds;
}

/**
 * MlMessage構造体の内容が正しく送信できるかチェックする
 *
 * @param	pool	apr_pool_t*	プール
 * @param	msg	MlMessage**	メールメッセージ構造体
 * @return	0: 正常 / 1: 失敗
 *
 */
DIVY_DECLARE(int) divy_ml_validate_mlmessage(apr_pool_t *pool, MlMessage *mlmsg)
{

	if (pool == NULL || mlmsg == NULL) return 1;

	for (;mlmsg != NULL; mlmsg = mlmsg->next) {
		/* fromの確認 */
		if (mlmsg->from_addr == NULL) return 1;
		if (IS_EMPTY(mlmsg->from_addr->addr)) return 1;

		/* toの確認 */
		if (mlmsg->to_addr  == NULL) return 1;
		if (IS_EMPTY(mlmsg->to_addr->addr)) return 1;

		/* priorityの確認 */
		switch(mlmsg->priority) {
			case ML_PRIORITY_HIGH:
			case ML_PRIORITY_NORMAL:
			case ML_PRIORITY_LOW:
				break;
			default:
				return 1;

		}

		/* header_encodeの確認 */
		switch(mlmsg->header_encoding) {
			case ML_ENCODING_NOP:
			case ML_ENCODING_B:
			case ML_ENCODING_Q:
				break;
			default:
				return 1;
		}

		/* mime_typeの確認 */
		if (mlmsg->mime_type == NULL) return 1;

		/* charsetの確認 */
		if (mlmsg->charset == NULL) return 1; 
	}

	return 0;
}

/**
 * filename が示す監視メールXMLをパースして*mldata に解析結果を格納する.
 *
 */
DIVY_DECLARE(int) divy_ml_parse_mailrequest_xml(request_rec *r, apr_pool_t *wp, const char *filename,
												int ignore_body, divy_ml_mailwatch_data **mldata)
{
	int ret = 0;
	apr_pool_t *p = r->pool;
	apr_status_t rv;
	apr_file_t *fd = NULL;

	*mldata = NULL;	/* 初期化 */

	if (IS_EMPTY(filename)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"Failed to Filename is NULL");
		return 1;
	}

	/* XML ファイルを開く */
	rv = apr_file_open(&fd, filename, APR_READ, APR_OS_DEFAULT, p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to open File ( file = %s )", filename);
		return 1;
	}

	/* ファイルをパースする */
	ret = divy_ml_parse_mailrequest_xml_by_fd(r, wp, fd, ignore_body, mldata);

	/* ファイルを閉じる */
	(void) apr_file_close(fd);

	return ret;
}

/**
 * fd が示す監視メールのディスクリプタからXMLをパースして*mldata に解析結果を格納する.
 * (note)
 *    divy_ml_parse_mailrequest_xml のディスクリプタ版です.
 */
DIVY_DECLARE(int) divy_ml_parse_mailrequest_xml_by_fd(request_rec *r, apr_pool_t *wp, apr_file_t *fd,
												int ignore_body, divy_ml_mailwatch_data **mldata)
{
	apr_pool_t *p = r->pool;
	apr_status_t rv;
	apr_xml_doc	*doc;
	apr_xml_parser  *parser;
	apr_xml_elem	*root_ele;
	apr_xml_elem    *ce;

	*mldata = NULL;	/* 初期化 */

	if (fd == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA, "fd is NULL");
		return 1;
	}

	/* XML ファイルのパース */
	rv = apr_xml_parse_file(wp, &parser, &doc, fd, 2048);
	if (rv != APR_SUCCESS || doc == NULL) {
		/* 512 バイト以上のメッセージは切捨て。。*/
		/* parserがエラーの場合でparserがNULLの場合がある */
		char *errbuf = apr_pcalloc(wp, 512);

		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"Failed to XML Parser: Reason: (%s)",
				(parser == NULL) ? "XML Parser Error." :
					apr_xml_parser_geterror(parser, errbuf, 512));
		return 1;
	}

	/*
	 * XML内部をチェックして必要な項目を抜き取る
	 */
	root_ele = doc->root;
	if (root_ele == NULL || strcmp(root_ele->name, "mailrequest")) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"Failed to XML Parser.");
		return 1;
	}

	/* 構造体のインスタンスを生成 */
	*mldata = apr_pcalloc(wp, sizeof(divy_ml_mailwatch_data));

	/*
	 * ループでまわしてはいますが一件だけしか取得しません。
	 */
	for (ce = root_ele->first_child; ce != NULL; ce = ce->next) {
		if ((strcmp(ce->name, "mailproperty")) == 0) {
			apr_xml_elem *e;

			for (e = ce->first_child; e != NULL; e = e->next) {
				if ((strcmp(e->name, "srcuri")) == 0) {
					(*mldata)->srcuri = (char *) divy_xml_get_cdata(e, wp, 1);
				}
				else if ((strcmp(e->name, "dsturi")) == 0) {
					(*mldata)->dsturi = (char *) divy_xml_get_cdata(e, wp, 1);
				}
				else if ((strcmp(e->name, "from")) == 0) {
					(*mldata)->from = (char *) divy_xml_get_cdata(e, wp, 1);
				}
				else if ((strcmp(e->name, "subject")) == 0) {
					(*mldata)->subject = (char *) divy_xml_get_cdata(e, wp, 1);
				}
				else if ((strcmp(e->name, "body")) == 0) {
					(*mldata)->body = (char *) divy_xml_get_cdata(e, wp, 0/* whilte spaceを維持 */);

					/* 不要な部分を切り捨てる (= ユーザ入力部だけを取り出す) */
					if (ignore_body) {
						(*mldata)->body = _extract_inputed_body(r, wp, (*mldata)->body);
					}
				}
				else if ((strcmp(e->name, "bodyfodder")) == 0 && ignore_body == 0) {
					(*mldata)->fbody = (char *) divy_xml_get_cdata(e, wp, 0/* whilte spaceを維持 */);
				}
				else if ((strcmp(e->name, "filelist")) == 0) {
					dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
					apr_xml_elem *fe;
					char *enc_filename, *fname;
					const char *charset = dconf->mailencoding;
					divy_sbuf *sbuf = NULL;

					divy_sbuf_create(wp, &sbuf, 1024);

					for (fe = e->first_child; fe != NULL; fe = fe->next) {
						if ((strcmp(fe->name, "filename")) == 0) { 

							fname = (char *)divy_xml_get_cdata(fe, wp, 1);
							if (IS_FILLED(charset) &&
								!(strcasecmp(charset, ML_CHARSET_UTF_8) == 0 ||
								  strcasecmp(charset, ML_CHARSET_UTF8) == 0)) {

								(void) _divy_ml_encodetext(wp, (const char*)fname, charset, &enc_filename, 0);

								fname = enc_filename;
							}

							divy_sbuf_append(sbuf, fname);
							divy_sbuf_append(sbuf, DIVY_CRLF);
						}
						else {	/* 存在しない. 無視 */
						}
					}
					(*mldata)->filelist = divy_sbuf_tostring(sbuf);
				}
				else if ((strcmp(e->name, "method")) == 0) {
					(*mldata)->method = (char *)divy_xml_get_cdata(e, wp, 1);
				}
				else if ((strcmp(e->name, "brand")) == 0) {
					/* brand名 SNIサーバ対応*/
					(*mldata)->brandname = (char *)divy_xml_get_cdata(e, wp, 1);
				}
				else {	/* 無視 */
				}
			}
		}
		else {	/* ありえません. 無視 */
		}
	}

	return 0;
}

/**
 * mlmsgの内容をそのままメール送信する
 *
 * @param r	request_rec*
 * @param mlmsg	MlMessage*
 * @return int	0: 成功/ 1:失敗
 *
 */
DIVY_DECLARE(int) divy_ml_sendmaildirect(request_rec *r, MlMessage *mlmsg)
{

	MlDataSource	*mlds	   = NULL;
	MlProperty	*mlprop    = NULL;
	MlSession	*session   = NULL;
	int		rt	   = ML_SUCCESS;
	apr_pool_t	*pool	   = r->pool;


	dav_divy_dir_conf    *conf  = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	if (IS_MAIL(conf, sconf)) {
		/* メール機能が使えない場合はそのまま正常終了 */
		return 0;
	}
	
	/* SMTPヘッダの設定 */
	if (divy_ml_setSMTPHeader(pool, mlmsg) != 0)
	       	return 1;

	/* メールプロバイダの取得 */
	mlds = lookup_activeml_provider(r);
	if (mlds == NULL) {
		ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to lookup mail provider. Check configuration.");
		return 1;
	}

	/* メールプロパティの生成 */
	mlprop = divy_ml_createMlProperty(r, r->user);
	if (mlprop == NULL) {
		ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get mail property.");
		return 1;
	}

	/* メールセッションの取得 */
	session = mlds->getMlSession(mlds, mlprop, r->pool);
	if (session == NULL) {
		ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get mailsession property.");
		return 1;
	}

	/* メール送信 */
	if ((rt = session->send(session, mlmsg)) != 0) {
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
			"Mail transmission went wrong. Message: %s",
			session->__msg);
		return 1;
	}

	/*
	 * プロバイダから戻ってきたセッションにはメッセージが含まれている
	 * ことがあります。
	 */
	if (session->__msg != NULL && strlen(session->__msg)) {
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Mail transmission went wrong. Message: %s",
			       	session->getMsg(session));
	}
	
	return 0;

}

/**
 * 指定されたファイルネームを元にメールを送信する。
 * メールの機能は全てこの関数を利用すれば勝手にメールが送られます。
 * メール機能が利用することができない場合(ライセンス不足)も正常終了とします。
 *
 * @param r request_rec *	リクエスト構造体
 * @param filename const char *	絶対パスのファイル名
 * @return 処理ステータス (ML_ERROR | ML_SUCCESS)
 *
 */
DIVY_DECLARE(int) divy_ml_send_clientmail(request_rec *r, const char *filename)
{
	apr_pool_t *p  = r->pool;
	apr_pool_t *wp = NULL;
	divy_ml_mailwatch_data *mldata = NULL, *templ_mldata = NULL;
	int ret = 1;
	apr_file_t *fd_r = NULL;
	const char *lang = _get_ml_language_param(r);
	char *templ_filepath = NULL;
	dav_divy_dir_conf    *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	divy_replace_keydata *keydata  = NULL;
	const char *val, *sOperation = NULL;
	divy_sbuf *sbuf  = NULL;
	divy_rdbo_grp *grp_pr = NULL;
	int mnum;

	/*
	 * メール使用ライセンスが存在しない、またはメール機能が
	 * Offにされている場合にはそのまま正常終了させる
	 * エラーは起動時に出力されている為、ここではなにもいいません。
	 */
	if (IS_MAIL(dconf, sconf)) {
		return ML_SUCCESS;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* XMLをパースする */
	if (divy_ml_parse_mailrequest_xml(r, wp, filename, 1/* bodyを無視する */, &mldata)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse mail-template (userid = %s)", divy_get_userid(r));
		goto cleanup_temp_resource;
	}

	/* メールテンプレートを読み込んでbody の一部とbodyfodderの全てを置き換える */
	if (_open_svrml_template(wp, DIVY_SVRML_TEMPLATEFILE_PREFIX, lang, r, &templ_filepath, &fd_r)) {
		goto cleanup_temp_resource;
	}

	if (divy_ml_parse_mailrequest_xml_by_fd(r, wp, fd_r, 0/* bodyを無視しない */, &templ_mldata)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse mail-template (userid = %s)", divy_get_userid(r));
		goto cleanup_temp_resource;
	}

	/* (note) 手動入力されたbody は生かす必要あり */
	mldata->body  = apr_psprintf(wp, "%s"DIVY_CRLF"%s", mldata->body, templ_mldata->body);
	mldata->fbody = templ_mldata->fbody;
	/* 件名が空白の場合、デフォルトメッセージを埋め込みます */
	if (IS_EMPTY(mldata->subject)) {
		keydata = NULL;
		/* メソッド名をメソッド番号に変換 */
		mnum = ap_method_number_of(mldata->method);
		/* グループ名を取得する */
		(void) divy_rdbo_get_group_property_by_resuri(r,
													mldata->srcuri, &grp_pr);
		/* 置換パラメータによる置き換え */
		if (mnum == M_GET) {
			sOperation = _get_svrml_string(DIVY_SVRML_GET, lang);
		}
		else if (mnum == M_PUT) {
			sOperation = _get_svrml_string(DIVY_SVRML_PUT, lang);
		}
		else if (mnum == M_DELETE) {
			sOperation = _get_svrml_string(DIVY_SVRML_DELETE, lang);
		}
		else if (mnum == M_MOVE) {
			sOperation = _get_svrml_string(DIVY_SVRML_MOVE, lang);
		}
		else if (mnum == M_COPY) {
			sOperation = _get_svrml_string(DIVY_SVRML_COPY, lang);
		}
		_push_replace_param(wp, DIVY_ML_REP_OPERATION_JA, sOperation, &keydata);
		_push_replace_param(wp, DIVY_ML_REP_OPERATION_EN, sOperation, &keydata);
		_push_replace_param(wp, DIVY_ML_REP_FILENAME, 
													mldata->filelist, &keydata);
		_push_replace_param(wp, DIVY_ML_REP_PARENT_FOLDER,
													grp_pr->name, &keydata);

		val = _choose_brand_name(r, mldata->brandname);
		_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

		divy_sbuf_create(wp, &sbuf, strlen(templ_mldata->subject));
		(void) _replace_ml_keyword(p, templ_mldata->subject, keydata, sbuf);
		mldata->subject = divy_sbuf_tostring(sbuf);	/* 置換済みの文字列と入れ替え */
	}

	/*
	 * メール送信実行
	 */
	ret = _send_watchmail(r, wp, mldata);

cleanup_temp_resource:
	/* 終了処理 */
	if (fd_r != NULL) (void) apr_file_close(fd_r);
	if (wp != NULL) apr_pool_destroy(wp);

	return ret;
}

/**
 * Mailプロバイダで利用するプロパティ構造体を作成する。
 * uidは今後の対応を考え予約しています。
 * 基本的にuri = r->userと同じですが、便宜を考え分けて利用するように
 * しています。
 *
 * @param  r	request_rec *
 * @param  uid	const char *
 * @return メールプロパティ構造体 MlProperty *
 *
 */
DIVY_DECLARE(MlProperty *) divy_ml_createMlProperty(request_rec *r,
	       					   const char *uid)
{
	apr_pool_t		*p      = r->pool;
	MlProperty 		*mlprop = NULL;
	dav_divy_dir_conf	*dconf  = NULL;

	TRACE(p);

	dconf = dav_divy_get_dir_config(r);

	mlprop            = apr_pcalloc(p, sizeof(*mlprop));
        mlprop->hostname  = apr_pstrdup(p, dconf->mailhost);
        mlprop->hostport  = apr_pstrdup(p, dconf->mailport);
        mlprop->adminaddr = apr_pstrdup(p, dconf->mailadminaddr);
        mlprop->username  = dconf->mlusername ? apr_pstrdup(p, dconf->mlusername) : NULL;
        mlprop->password  = dconf->mlpassword ? apr_pstrdup(p, dconf->mlpassword) : NULL;
        mlprop->localhost = (char *)ap_get_server_name(r);
        mlprop->interval  = dconf->mlsendinterval == DIVY_INT32_C_UNSET ? 0 : dconf->mlsendinterval;
	
	/*
	 * 以下の二つはdefaultでは利用しません。
	 * 引数のuidを元になんらかの方法で設定すればいいでしょう。
	 */


	return mlprop;
}

/**
 * MlMessage構造体のスケルトンを作成する。
 * (note)
 * 	MlMessage構造体はリスト構造になっているが、この関数では一件しか
 * 	作りません。
 *
 * @param pool 	apr_pool_t *
 * @param mlmsg	MlMessage **
 * @return int 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_ml_create_mlmsg(apr_pool_t* pool, MlMessage **mlmsg)
{

	MlMessage	*msg  = NULL;

	msg = apr_pcalloc(pool, sizeof(MlMessage));
	if (msg == NULL) return 1;

	msg->subject         = NULL;
	msg->priority        = ML_PRIORITY_HIGH;
	msg->header_encoding = ML_ENCODING_B;
	msg->mime_type       = ML_MIMETYPE_TEXT_PLAIN;
	msg->charset         = ML_CHARSET_UTF_8;
	msg->to_addr         = NULL;
	msg->from_addr	     = NULL;
	msg->replyto_addr    = NULL;
	msg->body            = NULL;
	msg->bodyfodder      = NULL;
	msg->next            = NULL;

	*mlmsg = msg;

	return 0;
}

/**
 * MlAddress構造体を作成する。
 * useridへはNULLをはめるとディレクティブのサーバ管理者のユーザを作成します。
 *
 * @return 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_ml_create_mladdress(request_rec *r,
	       				const char* userid, MlAddress **addr)
{
	divy_rdbo_usr	  *usr_pr = NULL;
	MlAddress	  *address = NULL;

	if (r == NULL || userid == NULL) return 1;

	address = apr_pcalloc(r->pool, sizeof(MlAddress));
	if ((divy_rdbo_get_user_property(r, userid, &usr_pr)) != 0)
		return 1;
	address->userid = apr_pstrdup(r->pool, usr_pr->usrid);
	address->addr   = apr_pstrdup(r->pool, usr_pr->mailaddr);
	address->displayname = apr_pstrdup(r->pool, usr_pr->fullname);
	address->next   = NULL;

	*addr = address;

	return 0;
}

/**
 * サーバ管理者へユーザからメールを送るMlMessage構造体を作成する。
 *
 * @param r request_rec *
 * @param mlmsg	MlMessage **
 * @return 0: 成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_ml_create_admin2user_msg(request_rec *r,
	       						MlMessage **mlmsg)
{

	apr_pool_t	*pool = r->pool;
	MlMessage	*msg  = NULL;
	MlAddress	*from_addr = NULL;
	MlAddress	*to_addr = NULL;
	dav_divy_dir_conf *dconf  = dav_divy_get_dir_config(r);
	const char	*admin_addr = NULL;

	if (r == NULL) return 1;

	/* メッセージ構造体の作成 */
	if ((divy_ml_create_mlmsg(pool, &msg)) != 0) return 1;
	if (msg == NULL) return 1;

	if ((admin_addr = dconf->mailadminaddr) == NULL) return 1;
	
	/* 送信元の作成(from) */
	from_addr = apr_pcalloc(pool, sizeof(MlAddress));
	from_addr->userid = NULL;
	from_addr->addr   = apr_pstrdup(r->pool, admin_addr);
	from_addr->displayname = "admin";
	from_addr->next   = NULL;

	/* 送信先の作成(to) */
	if ((divy_ml_create_mladdress(r, divy_get_userid(r), &to_addr) != 0))
			return 1;

	/* 題名の追加(subject) */
	msg->subject = apr_psprintf(r->pool, "[%s] Server Information", 
			ap_get_server_name(r));
	msg->from_addr = from_addr;

	msg->to_addr = to_addr;
	(void)divy_ml_setMultiAddress(r->pool, &msg->to_addr, &msg->cc_addr);
	*mlmsg = msg;

	return 0;
}


/**
 * SMTPヘッダにヘッダを追加する。
 * リプレース引数により追加するか置き換えるかを設定できる
 *
 * @param headers apr_table_t *
 * @param key   const char *
 * @param value const char *
 * @param replace int 
 * @return 処理ステータス
 *
 */
DIVY_DECLARE(int) divy_ml_setHeader(apr_table_t *headers, const char *key,
	       			   const char *value, int replace)
{

	if (key == NULL || value == NULL) {
		return ML_ERROR;
	}

	/*
	 * リプレースの場合はヘッダを置き換える。
	 * 違う場合は追加する
	 */
	if (replace == 0) {
		apr_table_set(headers, key, value);
	}
	else {
		apr_table_add(headers, key, value);
	}

	return ML_SUCCESS;
}

/**
 * MlMessageに基づいてSMTPのヘッダを作成する。
 * MlMessage内のheaderを一度クリアしてから設定します。
 * 何度流しても同じ結果になります。
 *
 * @param p apr_pool_t *
 * @param msg MlMessage *
 * @return 0:成功 / 1: 失敗
 */
DIVY_DECLARE(int) divy_ml_setSMTPHeader(apr_pool_t *p, MlMessage *mlmsg)
{
	MlAddress *tmp_addr;
#if 0
	time_t short_time  = dav_divy_get_now_epoch();
	char *date_buff    = NULL;
#endif

	if (p == NULL || mlmsg == NULL) return 1;

	/* MlMessage構造体のチェック */
	if (divy_ml_validate_mlmessage(p, mlmsg) != 0) return ML_ERROR;

	/* dateフィールドの日付を取得 */
#if 0
#### TODO: RFC2822のフォーマットができたときに実装します。
	divy_format_time_t(p, short_time, DIVY_TIME_STYLE_RFC2822, &date_buff);
#endif
	for (; mlmsg != NULL; mlmsg = mlmsg->next) {
		char *ctrans 	= NULL;
		char *ctype 	= NULL;
		char *from 	= NULL;
		char *val       = NULL;

		if (mlmsg->headers != NULL) {
			/* テーブルが存在したらすべてクリアする */
			apr_table_clear(mlmsg->headers);
		}
		else {
			/* 新規作成 */
			mlmsg->headers = apr_table_make(p,  5);
		}

		/*
		 * SMTPヘッダの作成
		 */
#if 0
#### TODO: RFC2822のフォーマットができたときに実装します。
		/* Date */
		  if (date_buff != NULL) {
			  divy_ml_setHeader(mlmsg->headers,
					 		 "Date", date_buff,0);
		  }
#endif

		/* Content-Transfer-Encoding */
		if (strcasecmp(mlmsg->charset, ML_CHARSET_UTF_8) == 0 || 
	            strcasecmp(mlmsg->charset, ML_CHARSET_UTF8) == 0 || 
		    strcasecmp(mlmsg->charset, ML_CHARSET_SHIFT_JIS) == 0) {
			ctrans = "8bit";
		}
		else {
			ctrans = "7bit";
		}
		divy_ml_setHeader(mlmsg->headers,
				"Content-Transfer-Encoding", ctrans, 0);

		/* X-Mailer */
		divy_ml_setHeader(mlmsg->headers,
			       	"X-Mailer", ML_HEADER_X_MAILER, 0);

		/* X-Priority */
		if (mlmsg->priority != 0) {
			divy_ml_setHeader(mlmsg->headers, "X-Priority",
				       	apr_psprintf(p, "%d", mlmsg->priority), 0);
		}

		/* MIME-version */
		divy_ml_setHeader(mlmsg->headers,
			       	"MIME-Version", ML_MIME_VERSION, 0);

		/* Subject */
		if (mlmsg->subject == NULL) mlmsg->subject = '\0';
		divy_ml_setHeader(mlmsg->headers,
			      	"Subject", mlmsg->subject, 0);

		/* content-type */
		ctype = apr_psprintf(p, "%s; charset=%s",
				mlmsg->mime_type, mlmsg->charset);
		divy_ml_setHeader(mlmsg->headers,
			       	"Content-Type", (const char*)ctype, 0);

		/* from */
		/* fromは複数レコードの可能性があるため始めの一件だけを
		 * 適応するようにする
		 */
		mlmsg->from_addr->addr =
		       	divy_ml_get_first_mail_address(p,
				       	mlmsg->from_addr->addr);

		if (mlmsg->from_addr->displayname != NULL) {
			from = apr_psprintf(p, "%s <%s>",
				       	mlmsg->from_addr->displayname,
				       	mlmsg->from_addr->addr);
		}
		else {
			from = apr_psprintf(p, "<%s>", 
					mlmsg->from_addr->addr);
		}

		divy_ml_setHeader(mlmsg->headers, "From", from, 0);

		/* to */
		for (tmp_addr = mlmsg->to_addr; tmp_addr != NULL;
				tmp_addr = tmp_addr->next) {
			if (tmp_addr->displayname != NULL) {
				val = apr_psprintf(p, "%s <%s>",
					       	tmp_addr->displayname,
						tmp_addr->addr);
			}
			else {
				val = apr_psprintf(p, "<%s>",
						tmp_addr->addr);
			}

			divy_ml_setHeader(mlmsg->headers, "To", val, 1);
		}

		/* cc */
		for (tmp_addr = mlmsg->cc_addr; tmp_addr != NULL;
				tmp_addr = tmp_addr->next) {
			if (tmp_addr->displayname != NULL) {
				val = apr_psprintf(p, "%s <%s>",
					       	tmp_addr->displayname,
						tmp_addr->addr);
			}
			else {
				val = apr_psprintf(p, "<%s>",
						tmp_addr->addr);
			}

			divy_ml_setHeader(mlmsg->headers, "Cc", val, 1);
		}

#if 0
/*
 * --- BCCをヘッダに含めない理由 ---
 *  BCCでもエンベローブとしてRCPTコマンド投げてメールの送信を行ってもらうが
 *  どうも、BCCヘッダを追加してもTOのユーザからBCCのユーザが存在している動き
 *  をしてしまいます。BCCの本来の意味からしてはこれは問題です。
 *  では、BCCのユーザだけはヘッダを付与して見せようかと思いましたが、
 *  BCCユーザが複数合った場合がまた問題で、BCC同士がお互いの存在を分かって
 *  しまうという新たな問題も発生しました。
 *  どうすればいいかちょっと検討する必要があるが、BCCは見えてはいけない
 *  という結論でこの部分をコメント扱いにしています。
 *
 *  本機能としては基本的にBCCユーザへはRCPTのエンベローブだけにしてヘッダは
 *  送出しないようにしています。
 *
 */
		/* bcc */
		for (tmp_addr = mlmsg->bcc_addr; tmp_addr != NULL;
				tmp_addr = tmp_addr->next) {
			if (tmp_addr->displayname == NULL)
				tmp_addr->displayname = '\0';
			val = apr_psprintf(p, "%s <%s>", tmp_addr->displayname,
					tmp_addr->addr);
			divy_ml_setHeader(mlmsg->headers, "Bcc", val, 1);
		}
#endif

		/* reply-to */
		for (tmp_addr = mlmsg->replyto_addr; tmp_addr != NULL;
				tmp_addr = tmp_addr->next) {
			if (tmp_addr->displayname != NULL) {
				val = apr_psprintf(p, "%s <%s>",
					       	tmp_addr->displayname,
						tmp_addr->addr);
			}
			else {
				val = apr_psprintf(p, "<%s>",
						tmp_addr->addr);
			}

			divy_ml_setHeader(mlmsg->headers, "Reply-To", val, 1);
		}
	}


	return ML_SUCCESS;

}

/**
 * SMTPヘッダの内容を取得する。
 * 同じキーが複数あった場合は全てを引数のデリミタで区切って返却する
 *
 * ##### 実装も完全ではなく、使っていません。
 *
 * @param pool apr_pool_t *
 * @param mlmsg MlMessage *
 * @param key   const char *
 * @param delimiter char * (default カンマ)
 * @return 設定されているヘッダ値
 *
 */
DIVY_DECLARE(const char *)divy_ml_getHeader(apr_pool_t *pool, MlMessage *mlmsg,
	       				   const char *key, char *delimiter)
{

#if 0
	apr_array_header_t	*array;

	if (mlmsg == NULL || key == '\0' || key == NULL) return NULL;
	if (mlmsg->headers == NULL) return NULL;

	/* デリミタの設定 */
	if (delimiter == NULL) delimiter = ",";

	array = apr_array_make(pool, 5, sizeof(char *));

	apr_table_do(_divy_ml_getHeader_callback, (void *)array,
		       				mlmsg->headers, key, NULL);
#endif

	return NULL;
}

/**
 * コンフィグファイルより利用できるメールプロバイダの種類を取得する。
 *
 * @param r request_rec *
 * @return データソース MlDataSource
 *
 */
DIVY_DECLARE(MlDataSource *)lookup_activeml_provider(request_rec *r)
{

	dav_divy_server_conf	*sconf;
	dav_divy_dir_conf	*dconf;

	/* configからメールの種類を取得する */
	sconf = dav_divy_get_server_config(r->server);
	dconf = dav_divy_get_dir_config(r);
	if (sconf == NULL || dconf == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"dav_divy_server_conf or dav_divy_dir_conf is NULL");
		return NULL;
	}

	/*
	 * メール使用ライセンスが存在しない、またはメール機能が
	 * Offにされている場合には、プロバイダは取得出来なかったことにする
	 */
	if (IS_MAIL(dconf,sconf)) {
		return NULL;
	}

	return lookup_ml_provider(r, dconf->mail);

}

/**
 * メールアドレスの情報を取得した後（データベースからとか）
 * 複数はいっている可能性のあるメールアドレスのうち一番初めの
 * 一件だけを返す。
 *
 * @param	pool		apr_pool_t* プール
 * @param	address		char *	調べたいメールアドレス
 * @return	mailaddress	char *	一番先頭のメールアドレス
 */
DIVY_DECLARE(char *) divy_ml_get_first_mail_address(apr_pool_t *pool, char *address)
{
	if (IS_EMPTY(address)) return NULL;	/* 空ならば何もしない */
	return _divy_ml_address_strtok(pool, address,  NULL);
}



/**
 * 文字列を指定されたcharsetでエンコードしてフォールディングを行った個所
 * をカンマで区切って返却する。
 *
 * @param p apr_pool_t *	プール
 * @param str const char *	エンコードしたい文字列
 * @param charset const char *	char set ex. utf-8など
 * @param to_str char *		エンコードされた文字列 NULLでもよい
 * @param int folding		foldingするか否か (1:する、0:しない)
 * @return int *		ML_SUCCESS: 成功, ML_ERROR:失敗
 *
 */
static int _divy_ml_encodetext(apr_pool_t *p, const char *str,
	       						const char *charset,
							char **to_str,
							int folding)
{

	char *base64_str  = NULL;
	int   base64_len  = 0;
	char *folding_str = NULL;
	char *token_str   = NULL;
	char *offset      = NULL;
	char *buffer      = NULL;
	int   len         = 0;
	int   other_len   = 0;
	int   total_len   = 0;
	char *enc_str     = NULL;
	

	if (p == NULL || str == NULL || charset == NULL) return 0;
	if (*str == '\0' || *charset == '\0') return 0;

	/*
	 * UTF8以外のエンコードの指定があった場合は指定された
	 * エンコードに変換してから進める。
	 */
	if (strcasecmp(charset, ML_CHARSET_UTF_8)) {
		int ret = 0;
		ret = divy_misc_encode_str(p, ML_CHARSET_UTF_8,
			       		     charset, (char*)str, &enc_str);
		if (ret != 0) {
			/* 失敗した場合はNULL */
			if (*to_str != NULL) *to_str = NULL;
			return ML_ERROR;
		}
	}
	else {
		/* UTF8の場合はそのまま文字列を設定する */
		enc_str = apr_pstrdup(p, str);
	}

	/* foldingを行わない場合はそのままエンコードした状態で返却 */
	if (folding == 0) {
		*to_str = enc_str;
		return ML_SUCCESS;
	}

	len = strlen(enc_str);
	other_len = strlen(charset) + 7; /* 8 = "=?" + "?B?" + "?=" */
	/*
	 * 76はヘッダの一行あたりの文字数(CRLF)除く
	 * ##### FIXME
	 * 本来は76文字であるがfoldingの方式が理解不能な為、今は256文字で
	 * foldingするようにしている。これはRFC違反ですけど今は保留する
	 */
	if ((total_len = 256 - other_len) < 0) return 0;

	/* 引数の文字列を全てbase64エンコードする */
	base64_str = apr_pcalloc(p, sizeof(char) * (apr_base64_encode_len(len) + 1));
	base64_len = apr_base64_encode(base64_str, enc_str, len);
	base64_str[base64_len + 1] = '\0';

	/* foldingを行う */
	while ((folding_str = divy_ml_folding_str(p, base64_str, total_len, &offset)) != NULL) {
		base64_str = NULL;

		/* foldingした文字列とその他の連結する文字列をallocする */
		buffer = apr_pcalloc(p, sizeof(char) * total_len+other_len+1);
		apr_snprintf(buffer, total_len+other_len+1, "=?%s?B?%s?=",
			       			charset, folding_str);
		buffer[total_len+other_len] = '\0';
		if (token_str != NULL) {
			token_str = apr_pstrcat(p, token_str, ",", buffer, NULL);
		}
		else {
			token_str = apr_pstrdup(p, buffer);
		}

	}

	*to_str = token_str;
	return ML_SUCCESS;
}

/**
 * 文字列を指定の長さでフォールディングする。
 *
 * strにlenだけフォールでリングして次の文字のポインタをoffsetにセットして
 * フォールディングした文字列を返します。
 *
 * @param p apr_pool_t *
 * @param str char *
 * @param len int
 * @param offset char **
 *
 */
DIVY_DECLARE(char *) divy_ml_folding_str(apr_pool_t *p, char *str,
	       						int len, char **offset) 
{

	char	*first_ch = NULL;
	char    *retstr   = NULL;
	int	 tmp_len  = 0;

	if (((str     == NULL || *str == '\0') &&
      	     (*offset == NULL || **offset == '\0' )) || (len <= 0))
	       		return NULL;

	if (str == NULL || *str == '\0') {
		first_ch = *offset;
	}
	else {
		first_ch = str;
	}

	tmp_len = strlen(first_ch);
	if (len >= tmp_len) {
		*offset = NULL;
		return apr_pstrcat(p, first_ch, '\0', NULL);
	}

	retstr = apr_pcalloc(p, sizeof(char) * len + 1);
	apr_snprintf(retstr, len + 1, "%s", first_ch);
	retstr[len] = '\0';

	/* 次の文字にポインタを移す */
	*offset = &first_ch[len];

	return retstr;

}

/**
 * カンマで区切られた複数このメールアドレス文字列address のフォーマットを検証する。
 *
 */
DIVY_DECLARE(int) divy_ml_validate_mailaddr(apr_pool_t *p, const char *address)
{
	ap_regex_t *preg;
	char *token, *token_cntx, *addrs;

	if (IS_EMPTY(address)) return ML_ERROR;	/* チェックできません */

	/* 正規表現パターンの作成 */
	preg = ap_pregcomp(p, "^[-A-Za-z0-9_.]+@[-A-Za-z0-9_]+[.][-A-Za-z0-9._]+$",
				0);
	if (preg == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to compile matching pattern.");
		return ML_ERROR;
	}

	/* メールアドレスを","(カンマ)で分けてそれぞれを検証 */
	addrs = apr_pstrdup(p, address);	/* address に影響を与えないようコピー */
	while ((token = apr_strtok(addrs, ",", &token_cntx)) != NULL) {
		addrs = NULL;

		/* 前後のwhiteスペースを取り除く */
		token = (char *) dav_divy_trim_white(p, token);

		/* マッチングを行う */
		if (ap_regexec(preg, token, 0, NULL, 0)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The mailaddress format is invalid."
				"(mailaddr = %s)", token);
			ap_pregfree(p, preg);
			return ML_ERROR;
		}
	}
	ap_pregfree(p, preg);

	return ML_SUCCESS;
}

/**
 * Addrのユーザのメールアドレスが複数だった場合(カンマ区切り）にaddrの内容から
 * もうひとつのアドレス構造体へ対象者を分割する。
 *
 * @param p		apr_pool_t *
 * @param addr		MlAddress **
 * @param addr2		MlAddress **
 * @return void
 */
DIVY_DECLARE(void) divy_ml_setMultiAddress(apr_pool_t *p,
	       			MlAddress **addr, MlAddress **addr2)
{

	char *token     = NULL;
	char *token_ctx = NULL;
	char *address   = NULL;
	int   first     = 0;
	MlAddress *a = NULL;
	MlAddress *c = NULL;

	if (p == NULL || *addr == NULL || IS_EMPTY((*addr)->addr)) return;

	a = *addr;
	address = apr_pstrdup(p, a->addr);
	while ((token = _divy_ml_address_strtok(p, address, &token_ctx))
		       	!= NULL) {
		if (token == NULL) break;
		address = NULL;

		if (first != 1) {
			/* 一回目はTOに設定 */
			a->addr = apr_pstrdup(p, token);
			first = 1;
		}
		else {
			/* 二件目以降はすべてCCとして処理 */
			MlAddress *new_cc = apr_pcalloc(p, sizeof(MlAddress));
			new_cc->userid = apr_pstrdup(p, a->userid);
			new_cc->addr   = apr_pstrdup(p, token);
			new_cc->displayname = NULL;
			new_cc->next   = NULL;

			if (c == NULL) {
				c = new_cc;
			}
			else {
				new_cc->next = c;
				c            = new_cc;
			}
		}
	}

	/* 設定されたCCを返却 */
	*addr2 = c;

}

/**
 * サーバ主体のメール送信機能が有効かどうかを判定する。
 *
 */
DIVY_DECLARE(int) divy_ml_enable_svrsendmail(request_rec *r,
						divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p               = r->pool;
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	divy_uri_spec *u_spec;
	const char *ch;

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"rdb_r is NULL.");
		return 0;
	}

	if (rdb_r->u_spec == NULL) {
		divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &u_spec);
	}
	else {
		u_spec = rdb_r->u_spec;
	}

	/*
	 * ハイブリッドのメール監視でのサーバ主体のメールを飛ばさないケース
	 */
	if (dconf->mlserversend == DIVY_MLSERVERSEND_MIDDLE) {
		/*
		 * TeamFileクライアントの場合はサーバ主体のメールは送らない
		 */
		if (apr_table_get(r->headers_in, DIVY_HEADER_REQ_CLIENT_INFO) != NULL) {
			return 0;
		}

		/*
		 * ブラウザからのPUT/DELETEはメール送らない
		 * 実際のPOST処理はぷトロコルゲートウェイにてWEBDAVメソッドに
		 * 変換されるため
		 */
		if (IS_FILLED(r->the_request) && !strncmp(r->the_request, "POST", 4)) {
			return 0;
		}
	}

	/*
	 * (note) GETのRangeリクエスト(_is_range_startを参照)
	 */
	if (!_is_range_start(r)) return 0;

	/* 環境変数の取得 */
	ch = apr_table_get(r->subprocess_env, DIVY_APENV_NOSVRMAIL);

	/* (note) ごみ箱に対してメール監視フラグが有効であるのは、
	 * 	ごみ箱直下のリソースと更にしたのリソースのみ。
	 * 	ごみ箱自身はNG */
	if (!IS_MAIL(dconf, sconf) &&
	    (dconf->mlserversend == DIVY_MLSERVERSEND_ON || 
		 dconf->mlserversend == DIVY_MLSERVERSEND_MIDDLE) &&
	    IS_FILLED(dconf->mltemplateroot) && !DIVY_APENV_IS_ON(ch) &&
	    (u_spec->infotype == DIVY_INFOTYPE_group_e        ||
	     u_spec->infotype == DIVY_INFOTYPE_group_trash_e0 ||
	     u_spec->infotype == DIVY_INFOTYPE_group_trash_ex ||
	     u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {
		return 1;
	}

	return 0;
}

/**
 * 指定されたuriにおいて、クライアント主体のメール送信機能が有効かどうかを判定する。
 *
 */
DIVY_DECLARE(int) divy_ml_enable_clientsendmail(request_rec *r, const char *uri)
{
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	char *mwatch_uri = NULL;

	if (!IS_MAIL(dconf, sconf) &&
	    dconf->mlserversend != DIVY_MLSERVERSEND_ON) {
		/* メール監視フラグを設定可能なURIを取り出してみる */
		(void) divy_extract_mailwatch_uri(r->pool, dav_divy_get_root_uri(r),
						  uri, &mwatch_uri);
		if (IS_EMPTY(mwatch_uri)) {
			return 0;	/* 有効ではないURIだった */
		}
		return 1;
	}

	return 0;
}

/**
 * ユーザ作成メール通知が有効かどうか判定する
 *
 */
DIVY_DECLARE(int) divy_ml_enable_notify_usercreation(request_rec *r,
						divy_rdbo_usr *usr_pr)
{
	apr_pool_t *p               = r->pool;
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	if (usr_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"usr_pr is NULL.");
		return 0;
	}

	if (!IS_MAIL(dconf, sconf) && IS_FILLED(usr_pr->mailaddr) &&
	    IS_FILLED(dconf->mltemplateroot) &&
	    (dconf->mlnotifyact == DIVY_MLNOTIFYACT_UNSET ||
	     dconf->mlnotifyact == DIVY_MLNOTIFYACTD_ON)) {
		return 1;
	}

	return 0;
}

/**
 * Virus detection alert メール通知が有効かどうか判定する
 *
 */
DIVY_DECLARE(int) divy_ml_enable_virul_detection_alert(request_rec *r)
{
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	if (!IS_MAIL(dconf, sconf) && IS_FILLED(dconf->mltemplateroot)) {
		return 1;
	}

	return 0;
}

/**
 * subject にブランディング用プレフィックスを付けて返す
 *
 */
DIVY_DECLARE(char *) divy_ml_append_subject_prefix(apr_pool_t *wp, request_rec *r,
                                                    const char *subject)
{
    dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

    if (IS_FILLED(sconf->brandname)) {
        return apr_psprintf(wp, "[%s] %s", sconf->brandname, subject);
    }
    else {
        return apr_pstrdup(wp, subject);
    }
}

/**
 * サーバ主体のメールを送信する.
 *
 */
DIVY_DECLARE(void) divy_ml_send_servermail(request_rec *r,
					divy_rdbo_resource *rdb_r,
					divy_rdbo_resource *dst_rdb_r)
{
	apr_status_t rv;
	int ret;
	apr_pool_t *p  = r->pool;
	apr_pool_t *wp = NULL;
	apr_file_t *fd_r = NULL;
	const char *lang = NULL, *val, *sOperation = NULL;
	divy_sbuf *sbuf  = NULL;
	divy_sbuf *sbuf_body = NULL;
	int flag;
	divy_replace_keydata *keydata  = NULL;
	int mnum = divy_get_method_number(r);
	divy_ml_mailwatch_data *mldata = NULL;
	char *templ_filepath = NULL;
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	dav_divy_dir_conf    *dconf = dav_divy_get_dir_config(r);
	const char *charset = dconf->mailencoding;
	char *grpname = "-";
	divy_rdbo_grp *grp_pr = NULL;
	divy_rdbo_usr *usr_pr = NULL;

	/*
	 * メール使用ライセンスが存在しない、またはメール機能が
	 * Offにされている場合にはそのまま正常終了させる
	 * エラーは起動時に出力されている為、ここではなにもいいません。
	 */
	if (IS_MAIL(dconf, sconf)) {
		return;
	}

	/*
	 * ._で始まるファイル名若しくは.DS_Storeはメール送信の対象にしない
	 */
	if ((strncmp("._", rdb_r->displayname, 2)) == 0) {
		return;
	}
	if ((strcmp(".DS_Store", rdb_r->displayname)) == 0) {
		return;
	}

	/* 同一KeepAlive であれば再送信は行わない */
	flag = divy_pcache_get_flag(r->connection->pool, DIVY_PCACHE_FLG_KA_SVRMAILSEND);
	if (flag) {
		return;	/* 1度通っているので何もしない */
	}
	/* COPY, MOVE の場合 */
	else if (mnum == M_COPY || mnum == M_MOVE) {
		/* COPY, MOVE の場合、"Overwrite: T" または未指定だったら
		 * remove が先に走る。よって、このremove の時にメールを
		 * 送信しないよう細工しなければならない */
		
		if (dst_rdb_r == NULL) {
			return;	/* 何もさせません */
		}
		else if (mnum == M_MOVE) {
			const char *p_src_uri = ap_make_dirstr_parent(p, rdb_r->uri);
			const char *p_dst_uri = ap_make_dirstr_parent(p, dst_rdb_r->uri);

			/* final path segment を比較して異なっていたら
			 * リネームであると判断する */
			if (p_src_uri && p_dst_uri && strcmp(p_src_uri, p_dst_uri) == 0 &&
			    strcmp(rdb_r->u_spec->final_path_segment, dst_rdb_r->u_spec->final_path_segment) != 0) {
				return;	/* 何もしない */
			}
		}
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* メールテンプレートファイルの読み込み */
	if (_open_svrml_template(wp, DIVY_SVRML_TEMPLATEFILE_PREFIX, lang, r, &templ_filepath, &fd_r)) {
		if (wp != NULL) {
			apr_pool_destroy(wp);
			wp = NULL;
		}
		return;
	}

	if (divy_rdbo_get_user_property(r, divy_get_userid(r), &usr_pr) != 0)
		return;

	/* メールテンプレートのXMLをパースする */
	if (divy_ml_parse_mailrequest_xml_by_fd(r, wp, fd_r, 0/* bodyを無視しない */, &mldata)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to parse mail-template (userid = %s)", divy_get_userid(r));
		goto cleanup_temp_resource;
	}

	/* パラメータの置き換え
	 * (note)
	 * 		サーバ主体メールでは、XML に含まれているbody 以外にもいくつかが
	 * 		置換パラメータ化されているのでそれを置き換えます.
	 */
	mldata->srcuri   = ap_make_dirstr_parent(wp, rdb_r->uri);
	/* グループ名を取得する */
	(void) divy_rdbo_get_group_property_by_resuri(r, mldata->srcuri, &grp_pr);
	/* mldata->srcuriがユーザフォルダの場合はgrp_prがNULLになる */
	if (grp_pr != NULL) {
		grpname = grp_pr->name;
	}
	mldata->dsturi   = (dst_rdb_r != NULL) ?  ap_make_dirstr_parent(wp, dst_rdb_r->uri) : NULL;
	mldata->from     = (char *)divy_get_userid(r);
	mldata->method   = divy_get_method_name(r);

	/* ファイル名はエンコードする */
	if (IS_FILLED(charset) &&
		!(strcasecmp(charset, ML_CHARSET_UTF_8) == 0 ||
		  strcasecmp(charset, ML_CHARSET_UTF8) == 0)) {
		(void) _divy_ml_encodetext(wp, (const char*)rdb_r->displayname, charset, &mldata->filelist, 0);
	}
	else {
		mldata->filelist = rdb_r->displayname;
	}

	if (mnum == M_GET) {
		sOperation = _get_svrml_string(DIVY_SVRML_GET, lang);
	}
	else if (mnum == M_PUT) {
		sOperation = _get_svrml_string(DIVY_SVRML_PUT, lang);
	}
	else if (mnum == M_DELETE) {
		sOperation = _get_svrml_string(DIVY_SVRML_DELETE, lang);
	}
	else if (mnum == M_MOVE) {
		sOperation = _get_svrml_string(DIVY_SVRML_MOVE, lang);
	}
	else if (mnum == M_COPY) {
		sOperation = _get_svrml_string(DIVY_SVRML_COPY, lang);
	}

	/* subject 内の置換パラメータの処理 */
	keydata = NULL;
	/* 置換キーワード一覧を作成する */
	_push_replace_param(wp, DIVY_ML_REP_OPERATION_JA, sOperation, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_OPERATION_EN, sOperation, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_FILENAME, mldata->filelist, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_PARENT_FOLDER, grpname, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_USERNAME,
						 REPLACE_NULL((char*)divy_get_fullname(r)), &keydata);
	_push_replace_param(wp, DIVY_ML_REP_USERCOMMENT, usr_pr->comment, &keydata);

	val = _choose_brand_name(r, mldata->brandname);
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* 置換済みの文字列と入れ替え */
	divy_sbuf_create(wp, &sbuf, strlen(mldata->subject));
	(void) _replace_ml_keyword(p, mldata->subject, keydata, sbuf);
	mldata->subject = divy_sbuf_tostring(sbuf);	

	divy_sbuf_create(wp, &sbuf_body, strlen(mldata->body));
	(void) _replace_ml_keyword(p, mldata->body, keydata, sbuf_body);
	mldata->body = divy_sbuf_tostring(sbuf_body);

	/*
	 * メール送信処理
	 */
	ret = _send_watchmail(r, wp, mldata);
	if (ret != ML_SUCCESS) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to send mail from \"%s\" user.", mldata->from);
	}

	/* 送信が完了したことを示すフラグを立てておく */
	divy_pcache_set_flag(r->connection->pool, 1, DIVY_PCACHE_FLG_KA_SVRMAILSEND);

cleanup_temp_resource:

	/* メールテンプレートファイルを閉じる */
	if (fd_r != NULL && (rv = apr_file_close(fd_r)) != APR_SUCCESS) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to close mail-template file.(filepath = %s, code = %d)", templ_filepath, rv);
		/* 続ける */
	}

	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return;
}

/**
 * ユーザ作成通知を送信する。
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * ユーザ情報
 */
DIVY_DECLARE(void) divy_ml_notify_usercreation(request_rec *r, divy_rdbo_usr *usr_pr)
{
	apr_pool_t *p            = r->pool;
	apr_pool_t *wp           = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	const char *lang = NULL;
	int ret;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg           = NULL;
	char *admin_mailaddr, *admin_fullname;

	if (usr_pr == NULL || IS_EMPTY(usr_pr->mailaddr)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"usr_pr or mailaddr is NULL.");
		return;
	}

	TRACE(p);

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* 管理者のアドレスを取得する fromになる */
	admin_mailaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;
	admin_fullname = IS_FILLED(dconf->mlrepfromname) ?
							dconf->mlrepfromname : dconf->mailadminaddr;

	/* 置換キーワード一覧を作成する */
	_push_replace_param(wp, DIVY_ML_REP_ADMIN,    admin_fullname, &keydata);

	/* ユーザ情報をかき集める */
	_build_user_notification_body(r, wp, keydata, usr_pr);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBody と Subject の取得して置換パラメータで置換する */
	if (_make_svrml_data(wp, DIVY_SVRML_NOTIFYACT_TEMPL_PREFIX, lang, r, keydata, msg)) {
		goto cleanup_wk_resource;
	}

	/* Subjectが空だった場合 */
	if (IS_EMPTY(msg->subject)) {
		msg->subject = divy_ml_append_subject_prefix(wp, r,
					_get_svrml_string(DIVY_SVRML_NOTIFYCL_SUBJECT, lang));
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信先(to) */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->to_addr->userid      = usr_pr->usrid;
	msg->to_addr->addr        = usr_pr->mailaddr;
	msg->to_addr->displayname = usr_pr->fullname;
	msg->to_addr->next        = NULL;

	(void) divy_ml_setMultiAddress(wp, &msg->to_addr, &msg->cc_addr);

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	msg->from_addr->addr        = admin_mailaddr;
	msg->from_addr->displayname = admin_fullname;
	msg->from_addr->next        = NULL;

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret =divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");

			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		/* エラー報告は行うがエラーにはしない */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send User-creation-notify mail."
			"If you need it, Please contact new user."
			"(creator = %s, new user = %s)",
			divy_get_userid(r), usr_pr->usrid);
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return;
}

/**
 * ユーザ変更通知を送信する。
 *
 * @param r request_rec *
 * @param usr_pr divy_rdbo_usr * ユーザ情報
 */
DIVY_DECLARE(void) divy_ml_notify_update_user(request_rec *r, divy_rdbo_usr *usr_pr)
{
	apr_pool_t *p            = r->pool;
	apr_pool_t *wp           = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	char *admin_mailaddr, *admin_fullname;
	const char *lang;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg           = NULL;
	int ret = 0;

	if (IS_MAIL(dconf, sconf)) {
		return ;	/* メール機能がOFFなので送信できない */
	}

	TRACE(p);

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);
	
	/* 管理者のアドレスを取得する fromになる */
	admin_mailaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;
	admin_fullname = IS_FILLED(dconf->mlrepfromname) ?
							dconf->mlrepfromname : dconf->mailadminaddr;

	/* 置換キーワード一覧を作成する */
	_push_replace_param(wp, DIVY_ML_REP_ADMIN,    admin_fullname, &keydata);

	/* ユーザ情報をかき集める */
	_build_user_notification_body(r, wp, keydata, usr_pr);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBody と Subject の取得して置換パラメータで置換する */
	if (_make_svrml_data(wp, DIVY_SVRML_NOTIFYACT_UPDATE_USER_PREFIX,
													lang, r, keydata, msg)) {
		goto cleanup_wk_resource;
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信先(to) */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->to_addr->userid      = usr_pr->usrid;
	msg->to_addr->addr        = usr_pr->mailaddr;
	msg->to_addr->displayname = usr_pr->fullname;
	msg->to_addr->next        = NULL;

	(void) divy_ml_setMultiAddress(wp, &msg->to_addr, &msg->cc_addr);

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	msg->from_addr->addr        = admin_mailaddr;
	msg->from_addr->displayname = admin_fullname;
	msg->from_addr->next        = NULL;

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret =divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");

			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		/* エラー報告は行うがエラーにはしない */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send User-update-notify mail."
			"If you need it, Please contact update user."
			"(creator = %s, update user = %s)",
			divy_get_userid(r), usr_pr->usrid);
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return;
}

/**
 * ウィルス感染したことメールで通知する。
 * [ 送信先 ]
 *   * TeamFileサーバ管理者
 *   * アップロード実行者
 *
 * [ 送信元 ]
 *   * TeamFile 管理者のメールアドレス(表示名なし)
 *
 * @param r request_rec *
 * @param infect_status int 感染状況
 * @param rdb_r divy_rdbo_resource * 感染したファイルの情報
 */
DIVY_DECLARE(void) divy_ml_notify_virusdetection(request_rec *r, int infect_status,
					divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p            = r->pool;
	apr_pool_t *wp           = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const char *mltemplate   = NULL, *lang = NULL, *val;
	int ret;
	char *datestr            = NULL;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg           = NULL;
	char *own_mailaddr, *admin_mailaddr;
	int support_groupleader = divy_support_groupleader(r);
	const char *ownerid = divy_get_userownerid(r), *owner_mailaddr = NULL;
	divy_rdbo_usr *owner_pr = NULL;

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is NULL.");
		return;
	}

	own_mailaddr   = (char *) divy_get_mailaddr(r);
	admin_mailaddr = dconf->mailadminaddr;
	if (support_groupleader && IS_FILLED(ownerid)) {
		/* オーナーのメールアドレスを取得する */
		if (divy_rdbo_get_user_property(r, ownerid, &owner_pr) || owner_pr == NULL) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get ownerinformation. (ownerid = %s) "
					"We could not send virus-detected mail", ownerid);
			return;
		}
		owner_mailaddr = owner_pr->mailaddr;
	}
	else {
		owner_mailaddr = NULL;
	}
	
	/* 送信元メールアドレスが存在するか？ */
	if (IS_EMPTY(admin_mailaddr)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"We could not send virus-detection mail, "
			"because sender mailaddress was missing."
			"Please check configuration file.");
		return;
	}

	/* 少なくとも1つ以上の送信先メールアドレスが存在するか？ */
	if (IS_EMPTY(own_mailaddr) && IS_EMPTY(owner_mailaddr)) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"We could not send virus-detection mail, "
			"because \"To\" or \"Cc\" mailaddress was missing."
			"Please check it.(user = %s)", divy_get_userid(r));
		return;
	}

	TRACE(p);

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* 置換キーワード一覧を作成する */
	_push_replace_param(wp, DIVY_ML_REP_FILENAME, rdb_r->displayname,    &keydata);
	_push_replace_param(wp, DIVY_ML_REP_FILESIZE,
			divy_byte2displaysize(wp, rdb_r->getcontentlength, 0),  &keydata);
	_push_replace_param(wp, DIVY_ML_REP_FILETYPE, rdb_r->getcontenttype, &keydata);

	(void) divy_format_time_t(wp, dav_divy_get_now_epoch(), DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_REGDAY,   datestr, &keydata);

	_push_replace_param(wp, DIVY_ML_REP_USERID,    divy_get_userid(r),   &keydata);
	_push_replace_param(wp, DIVY_ML_REP_USERNAME,  divy_get_fullname(r), &keydata);
	_push_replace_param(wp, DIVY_ML_REP_MAILADDR,  own_mailaddr,         &keydata);
	_push_replace_param(wp, DIVY_ML_REP_ADMINMAIL, admin_mailaddr,       &keydata);
	if (support_groupleader && IS_FILLED(ownerid)) {
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "1", &keydata);
	}
	else {
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "0", &keydata);
	}

	if (IS_FILLED(sconf->brandname)) {
		val = sconf->brandname;
	}
	else {
		val = DIVY_PRODUCT_NAME;
	}
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBody と Subject の取得して置換パラメータで置換する */
	if (infect_status == DIVY_VSC_ST_INFECTED) {
		mltemplate = DIVY_SVRML_VIRUS_DETECTION_PREFIX;
	}
	else {
		mltemplate = DIVY_SVRML_VIRUS_DETECTION_FAIL_PREFIX;
	}
	if (_make_svrml_data(wp, mltemplate, lang, r, keydata, msg)) {
		goto cleanup_wk_resource;
	}

	/* Subjectが空だった場合 */
	if (IS_EMPTY(msg->subject)) {
		msg->subject = divy_ml_append_subject_prefix(wp, r,
					_get_svrml_string(DIVY_SVRML_VIRUS_DETECTION_SUBJECT, lang));
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信先 */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->to_addr->userid = NULL;
	if (IS_FILLED(own_mailaddr)) {
		/* 本人、管理者、グループリーダ */
		if (IS_FILLED(owner_mailaddr)) {
			msg->to_addr->addr = apr_psprintf(wp, "%s,%s,%s", own_mailaddr,
							admin_mailaddr, owner_mailaddr);
		}
		/* 本人、管理者 */
		else {
			msg->to_addr->addr = apr_psprintf(wp, "%s,%s", own_mailaddr,
							admin_mailaddr);
		}
		msg->to_addr->displayname = (char *) divy_get_fullname(r);
	}
	else {
		/* 管理者、グループリーダ */
		if (IS_FILLED(owner_mailaddr)) {
			msg->to_addr->addr = apr_psprintf(wp, "%s,%s", admin_mailaddr, owner_mailaddr);
		}
		/* 管理者 */
		else {
			msg->to_addr->addr = admin_mailaddr;
		}
		msg->to_addr->displayname = NULL;
	}
	msg->to_addr->next = NULL;

	(void) divy_ml_setMultiAddress(wp, &msg->to_addr, &msg->cc_addr);

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	/* mlrepfromaddrがある場合は管理者のアドレスは*送信者*としない */
	msg->from_addr->addr        = IS_FILLED(dconf->mlrepfromaddr) ?
									dconf->mlrepfromaddr : admin_mailaddr;
	msg->from_addr->displayname = NULL;
	msg->from_addr->next        = NULL;

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");

			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		/* エラー報告は行うがエラーにはしない */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send virus-detection mail."
			"If you need it, Please contact sender."
			"(userid = %s)", divy_get_userid(r));
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return;
}

#ifdef DIVY_SUPPORT_PASSPOLICY
/**
 * パスワード期限切れ状態をメールで通知する.
 *
 */
DIVY_DECLARE(int) divy_ml_notify_passwordexpired(request_rec *r,
								int mltype, const divy_rdbo_usr *usr_pr,
								divy_ml_passwordexpired_data *data)
{
	apr_pool_t *p            = r->pool;
	apr_pool_t *wp           = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const char *lang = NULL, *val, *prefix;
	int ret, st = 0;
	char *datestr = NULL;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg           = NULL;
	char *own_mailaddr, *admin_mailaddr;
	divy_svrml_id subjectid;
	int support_groupleader = divy_support_groupleader(r);
	const char *ownerid = divy_get_userownerid(r);
	divy_rdbo_usr *owner_pr = NULL;

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	if (usr_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"usr_pr is NULL.");
		return 1;
	}

	own_mailaddr   = (char *) divy_get_mailaddr(r);
	if (support_groupleader && IS_FILLED(ownerid) && IS_EMPTY(dconf->mlrepfromaddr)) {

		/* オーナ情報の取得 */
		if (divy_rdbo_get_user_property(r, ownerid, &owner_pr) || owner_pr == NULL) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to get ownerinformation. (ownerid = %s) "
					"We could not send password-policy error mail", ownerid);
			return 1;
		}
		admin_mailaddr = owner_pr->mailaddr;
	}
	else {
		admin_mailaddr = IS_FILLED(dconf->mlrepfromaddr) ?
									dconf->mlrepfromaddr:dconf->mailadminaddr;
	}
	
	/* 送信元メールアドレスが存在するか？ */
	if (IS_EMPTY(admin_mailaddr)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"We could not send password-expired mail, "
			"because sender mailaddress was missing."
			"Please check configuration file.");
		return 1;
	}

	/* 送信先メールアドレスが存在するか？ */
	if (IS_EMPTY(own_mailaddr)) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
			"We could not send password-expired mail, "
			"because \"To\" or \"Cc\" mailaddress was missing."
			"Please check it.(user = %s)", divy_get_userid(r));
		return 1;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* 置換キーワード一覧を作成する */

	if (data != NULL) {
		/* パスワードの期限が切れた日時 */
		if (data->period != 0L) {
			(void) divy_format_time_t(wp, data->period,
									  DIVY_TIME_STYLE_JAPANESE, &datestr);
			_push_replace_param(wp, DIVY_ML_REP_PASSWORD_PERIOD, datestr, &keydata);
		}

		/* 変更までに猶予された期間の最終日 */
		if (data->probation_last != 0L) {
			(void) divy_format_time_t(wp, data->probation_last,
									  DIVY_TIME_STYLE_JAPANESE, &datestr);
			_push_replace_param(wp, DIVY_ML_REP_PASSWORD_PROBATION, datestr, &keydata);
		}

		/* 初回ログインパスワードが利用できなくなる最終日時 */
		if (data->flprobation_last != 0L) {
			(void) divy_format_time_t(wp, data->flprobation_last,
									  DIVY_TIME_STYLE_JAPANESE, &datestr);
			_push_replace_param(wp, DIVY_ML_REP_FIRSTLOGIN_PROBATION, datestr, &keydata);
		}
	}

	/* ユーザID */
	_push_replace_param(wp, DIVY_ML_REP_USERID, divy_get_userid(r), &keydata);

	/* 管理者 / グループリーダ のメールアドレス */
	_push_replace_param(wp, DIVY_ML_REP_ADMINMAIL, admin_mailaddr, &keydata);

	/* オーナーを持っているかどうか(グループリーダが管理するユーザかどうか) */
	if (support_groupleader && IS_FILLED(ownerid)) {
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "1", &keydata);
	}
	else {
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "0", &keydata);
	}

	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* $schema://$root/ を算出する */
	val =  divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, dav_divy_get_root_uri(r), r, NULL);
	_push_replace_param(wp, DIVY_ML_REP_ROTURL, apr_psprintf(wp, "%s/", val), &keydata);

	/* メールテンプレートファイルの読み込み */
	prefix = "";
	switch (mltype) {
		/* 猶予期間中 */
		case DIVY_ML_NOTIFY_PASS_PROBATION:
			prefix    = DIVY_SVRML_PASS_PROBATION_PREFIX;
			subjectid = DIVY_SVRML_NOTIFY_PASS_PROBATION_SUBJECT;
			break;
		/* 完全期限切れ */
		case DIVY_ML_NOTIFY_PASS_EXPIRED:
			prefix    = DIVY_SVRML_PASS_EXPIRED_PREFIX;
			subjectid = DIVY_SVRML_NOTIFY_PASS_EXPIRED_SUBJECT;
			break;
		/* 初回ログイン時の猶予期間中 */
		case DIVY_ML_NOTIFY_FLOGIN_PROBATION:
			prefix    = DIVY_SVRML_FLOGIN_PROBATION_PREFIX;
			subjectid = DIVY_SVRML_NOTIFY_FLOGIN_PROBATION_SUBJECT;
			break;
		/* 初回ログイン時の完全期限切れ */
		case DIVY_ML_NOTIFY_FLOGIN_EXPIRED:
			prefix    = DIVY_SVRML_FLOGIN_EXPIRED_PREFIX;
			subjectid = DIVY_SVRML_NOTIFY_FLOGIN_EXPIRED_SUBJECT;
			break;
		/* パスワードを直ぐに変更しなさいメール */
		case DIVY_ML_NOTIFY_FLOGIN_CHANGENOW:
			prefix    = DIVY_SVRML_FLOGIN_CHANGENOW_PREFIX;
			subjectid = DIVY_SVRML_NOTIFY_FLOGIN_CHANGENOW_SUBJECT;
			break;
		default:
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to send password-notification. "
					"Invalid mailtype found (%d)", mltype);
			return 1;
	}

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBody と Subject の取得して置換パラメータで置換する */
	if (_make_svrml_data(wp, prefix, lang, r, keydata, msg)) {
		st = 1;
		goto cleanup_wk_resource;
	}

	/* Subjectが空だった場合 */
	if (IS_EMPTY(msg->subject)) {
		msg->subject = divy_ml_append_subject_prefix(wp, r, _get_svrml_string(subjectid, lang));
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信先 */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->to_addr->userid = NULL;

	/* 完全期限切れの場合のみ管理者 or グループリーダにも通知する */
	if (mltype == DIVY_ML_NOTIFY_PASS_EXPIRED ||
		mltype == DIVY_ML_NOTIFY_FLOGIN_EXPIRED) {
		msg->to_addr->addr = apr_psprintf(wp, "%s,%s", own_mailaddr, admin_mailaddr);
	}
	else {
		msg->to_addr->addr = own_mailaddr;
	}
	msg->to_addr->displayname = (char *) divy_get_fullname(r);
	msg->to_addr->next = NULL;

	(void) divy_ml_setMultiAddress(wp, &msg->to_addr, &msg->cc_addr);

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	msg->from_addr->addr        = admin_mailaddr;
	msg->from_addr->displayname = NULL;
	msg->from_addr->next        = NULL;

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			st = 1;
			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send password-expired mail."
			"(userid = %s)", divy_get_userid(r));
		st = 1;
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return st;
}
#endif	/* DIVY_SUPPORT_PASSPOLICY */

/**
 * オーナーの変更をメールで通知する
 *
 */
DIVY_DECLARE(int) divy_ml_notify_ownerchange(request_rec *r, int appointed_groupleader,
											const divy_rdbo_grp *grp_pr, divy_rdbo_usr *owner_pr)
{
	apr_pool_t *p               = r->pool;
	apr_pool_t *wp              = NULL;
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const char *lang = NULL, *val;
	int ret, st = 0;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg              = NULL;
	char *own_mailaddr;

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	if (grp_pr == NULL || owner_pr == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"grp_pr or owner_pr is NULL.");
		return 1;
	}

	own_mailaddr = (char *) divy_get_mailaddr(r);
	
	/* 送信元メールアドレスが存在するか？ */
	if (IS_EMPTY(own_mailaddr)) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send changeowner mail, "
				"because sender mailaddress was missing."
				"Please check configuration file. (action user = %s)", divy_get_userid(r));
		return 1;
	}

	/* 送信先メールアドレスが存在するか？ */
	if (IS_EMPTY(owner_pr->mailaddr)) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send changeowner mail, "
				"because \"To\" mailaddress was missing."
				"Please check it.(user = %s)", owner_pr->usrid);
		return 1;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/*
	 * 置換キーワード一覧を作成する
	 */

	/* グループ名 */
	_push_replace_param(wp, DIVY_ML_REP_GROUPNAME, grp_pr->name, &keydata);

	/* 管理者メールアドレス */
	_push_replace_param(wp, DIVY_ML_REP_ADMINMAIL, own_mailaddr, &keydata);

	/* 任命かどうか */
	if (appointed_groupleader) {
		_push_replace_param(wp, DIVY_ML_REP_APPOINTEDLEADER, "1", &keydata);
	}
	else {
		_push_replace_param(wp, DIVY_ML_REP_APPOINTEDLEADER, "0", &keydata);
	}

	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* $schema://$root/ を算出する */
	val =  divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, dav_divy_get_root_uri(r), r, NULL);
	_push_replace_param(wp, DIVY_ML_REP_ROTURL, apr_psprintf(wp, "%s/", val), &keydata);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBody と Subject の取得して置換パラメータで置換する */
	if (_make_svrml_data(wp, DIVY_SVRML_OWNER_CHANGE_NOTIFICATION_PREFIX, lang, r, keydata, msg)) {
		st = 1;
		goto cleanup_wk_resource;
	}

	/* Subjectが空だった場合 */
	if (IS_EMPTY(msg->subject)) {
		msg->subject = divy_ml_append_subject_prefix(wp, r, _get_svrml_string(DIVY_SVRML_NOTIFY_CHANGELEADER_SUBJECT, lang));
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信先 */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->to_addr->userid = NULL;
	msg->to_addr->addr        = owner_pr->mailaddr;
	msg->to_addr->displayname = owner_pr->fullname;
	msg->to_addr->next = NULL;

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	msg->from_addr->addr        = own_mailaddr;
	msg->from_addr->displayname = (char *) divy_get_fullname(r);
	msg->from_addr->next        = NULL;

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			st = 1;
			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send owner-change mail.(userid = %s)", divy_get_userid(r));
		st = 1;
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return st;
}

/**
 * 開封通知をメールで送信する.
 *
 * [ 送信先 ]
 *   * rdb_r->uri に開封通知を仕掛けている人 (N人)
 *
 * [ 送信元 ]
 *   * TeamFile 管理者のメールアドレス(表示名なし)
 */
DIVY_DECLARE(int) divy_ml_notify_confirmreading(request_rec *r, divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p               = r->pool;
	apr_pool_t *wp              = NULL;
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const char *lang = NULL, *val, *admin_mailaddr;
	int ret, st = 0, found_mailaddr = 0, isFirst = 1;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg              = NULL, *m;
	divy_rdbo_usr *to_usr_pr = NULL, *usr;
	char *parent_folder = NULL, *enc_uri, *datestr = NULL;
	char *keycode = NULL;
	char *sid = NULL;
	char *fullname = NULL;

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	if (!divy_support_confirmreading(r)) {
		return 0;	/* 未サポート */
	}

	if (!_is_range_start(r)) {
		return 0;	/* rangeの始まりではない */
	}

	if (rdb_r == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"rdb_r is NULL.");
		return 1;
	}

	/* 送信先のメールアドレスを取得する */
	if (divy_rdbo_get_confirmreading_users(r, rdb_r->uri, &to_usr_pr, NULL)) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"We could not send confirmreading mail "
				"because we could not get to-mailaddr (action user = %s)", divy_get_userid(r));
		return 1;
	}

	/* ユーザが一人もいなかった場合のみ次のチェックを行う
	 * 該当した場合はメールが開封通知が送られます
	 *
	 * 1. グループフォルダの下
	 * 2. BOX機能が有効
	 * 3. BOXが公開中
	 * 4. BOX設定者ではないユーザのダウンロード
	 */
	if (to_usr_pr == NULL && 
			rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular
												&& divy_support_tfbox(r)) {
		divy_rdbo_resourcestate *p_rstate_pr = NULL;
		if (divy_rdbo_get_parent_resourcestate_property(r, rdb_r->uri,
									DIVY_RSTATE_TYPE_BOX, &p_rstate_pr) == 0) {
			if (p_rstate_pr != NULL
					&& p_rstate_pr->optional.box_pr != NULL
					&& p_rstate_pr->optional.box_pr->flag & BOX_FLAG_OPEN
					&& strcmp(p_rstate_pr->optional.box_pr->creator_usr_id,
																	r->user)) {

				/* ユーザ情報を設定する */
				divy_rdbo_get_user_property(r,
					p_rstate_pr->optional.box_pr->creator_usr_id, &to_usr_pr);
			}
		}
	}

	/* 開封通知しているユーザが1人も居なかった -> 開封通知を送る必要なし */
	if (to_usr_pr == NULL) {
		return 0;
	}

	/* mlrepfromaddrがある場合は管理者のアドレスは*送信者*としない */
	admin_mailaddr = IS_FILLED(dconf->mlrepfromaddr)?
								dconf->mlrepfromaddr : dconf->mailadminaddr;

	/* 送信元メールアドレスが存在するか？ */
	if (IS_EMPTY(admin_mailaddr)) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send confirmreading mail "
				"because the sender mailaddress entry (\"TfMlAdminaddr\") was missing."
				"Please check configuration file. (action user = %s)", divy_get_userid(r));
		return 1;
	}

	/* 送信先メールアドレスが存在するか？ */
	found_mailaddr = 0;
	for (usr = to_usr_pr; usr; usr = usr->next) {
		if (IS_FILLED(usr->mailaddr)) {
			found_mailaddr = 1;
			break;
		}
	}
	if (!found_mailaddr) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send confirmreading mail "
				"because ALL \"To\" mailaddress were missing."
				"Please check it.(user = %s)", divy_get_userid(r));
		return 1;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/*
	 * 置換キーワード一覧を作成する
	 */

	/* ファイル名 */
	_push_replace_param(wp, DIVY_ML_REP_FILENAME, rdb_r->displayname, &keydata);

	/* 親フォルダ名 */
	if (rdb_r->u_spec == NULL) {
		(void) divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec);
	}
	enc_uri = divy_get_parenturi(p, rdb_r->u_spec->other_part);
	(void) dav_divy_unescape_uri(p, enc_uri, &parent_folder);
	_push_replace_param(wp, DIVY_ML_REP_PARENT_FOLDER, parent_folder, &keydata);

	/* ダウンロード日付 */
	(void) divy_format_time_t(wp, dav_divy_get_now_epoch(), DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_DOWNLOAD_DATE, datestr, &keydata);

	/* ダウンロードしたユーザ名 */
	if (divy_util_auth_parse_cookie(r, &keycode, &sid)
												== DIVY_AUTH_SESSION_OK) {
		divy_auth_session session = { 0 };
		session.sid = sid;
		if (divy_util_auth_get_memcache_userinfo(r, &session) == 0) {
			fullname = session.validmail;
		}
	}

	if (IS_EMPTY(fullname)) {
		fullname = (char*)divy_get_fullname(r);
	}
	_push_replace_param(wp, DIVY_ML_REP_DOWNLOAD_USERNAME, fullname, &keydata);
	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* リモートアドレス */
	_push_replace_param(wp, DIVY_ML_REP_REMOTE_ADDR, r->useragent_ip, &keydata);

	/* $schema://$root/ を算出する */
	val =  divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, dav_divy_get_root_uri(r), r, NULL);
	_push_replace_param(wp, DIVY_ML_REP_ROTURL, apr_psprintf(wp, "%s/", val), &keydata);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBody と Subject の取得して置換パラメータで置換する */
	if (_make_svrml_data(wp, DIVY_SVRML_OWNER_CONFIRMREADING_PREFIX, lang, r, keydata, msg)) {
		st = 1;
		goto cleanup_wk_resource;
	}

	/* Subjectが空だった場合 */
	if (IS_EMPTY(msg->subject)) {
		msg->subject = divy_ml_append_subject_prefix(wp, r, _get_svrml_string(DIVY_SVRML_NOTIFY_CONFIRMREADING_SUBJECT, lang));
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	msg->from_addr->addr        = apr_pstrdup(p, admin_mailaddr);
	msg->from_addr->displayname = NULL;
	msg->from_addr->next        = NULL;

	/* 送信先(To) 毎にmsg 構造体を生成する */
	isFirst = 1;
	m = NULL;
	for (usr = to_usr_pr; usr; usr = usr->next) {
		if (IS_EMPTY(usr->mailaddr)) {
			/* 念のため手短にログに出しておく */
			ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
					"Skip sending confirmreading mail. (\"%s\" user did not have mailaddress)",
					usr->usrid);
			continue;
		}

		if (isFirst) {
			m = msg;
		}
		else {
			m->next = apr_pcalloc(wp, sizeof(MlMessage));
			m = m->next;

			m->subject         = msg->subject;
			m->priority        = msg->priority;
			m->header_encoding = msg->header_encoding;
			m->mime_type       = msg->mime_type;
			m->charset         = msg->charset;
			m->from_addr       = msg->from_addr;
			m->body            = msg->body;

			m->from_addr              = apr_pcalloc(wp, sizeof(MlAddress));
			m->from_addr->userid      = NULL;
			m->from_addr->addr        = apr_pstrdup(p, msg->from_addr->addr);
			m->from_addr->displayname = NULL;
			m->from_addr->next        = NULL;
		}

		m->to_addr              = apr_pcalloc(wp, sizeof(MlAddress));
		m->to_addr->userid      = apr_pstrdup(p, usr->usrid);
		m->to_addr->addr        = apr_pstrdup(p, usr->mailaddr);
		m->to_addr->displayname = apr_pstrdup(p, usr->fullname);
		m->to_addr->next        = NULL;

		/* アドレスを切り分ける */
		(void) divy_ml_setMultiAddress(r->pool, &m->to_addr, &m->cc_addr);
		isFirst = 0;
	}
	
	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			st = 1;
			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send confirmreading mail.(userid = %s)", divy_get_userid(r));
		st = 1;
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return st;
}

/**
 * 存在する限りのロックされた本人、オーナー、管理者にメールを送ります
 */
DIVY_DECLARE(int) divy_ml_notify_failed_login(request_rec *r)
{
	apr_pool_t *p	= r->pool;
	apr_pool_t *wp  = NULL;
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const char *failedaddr = NULL , *adminaddr = NULL, *lang, *val;
	const char *ownerid = NULL;
	int support_groupleader       = divy_support_groupleader(r);
	MlMessage *msg = NULL, *m=NULL;
	divy_replace_keydata *keydata = NULL;
	int ret = 0;
	divy_rdbo_usr *owner_pr = NULL;
	char *datestr = NULL;

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	/*  ログイン失敗制御の機能の有効をチェックする */
	if (!divy_support_failedlogin_lockout(r)) {
		return 0;
	}

	/* 操作者のメールアドレスを取得する */
	failedaddr = divy_get_mailaddr(r);

	/* 管理者のアドレスを取得する */
	/* mlrepfromaddrがある場合は管理者のアドレスは*送信者*としない */
	adminaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;

	/* 送信元メールアドレスが存在しなければ処理は継続しない */
	if (IS_EMPTY(adminaddr)) {
		ERRLOG1(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send login failed mail "
				"because the sender mailaddress entry (\"TfMlAdminaddr\") was missing."
				"Please check configuration file. (action user = %s)", divy_get_userid(r));
		return 1;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);
			
	/* グループリーダーをサポートしていればオーナを所得してみる */
	if (support_groupleader) {
		ownerid = divy_get_userownerid(r);
	}
	/* 
	 * 置換パラメータの作成
	 */
	_push_replace_param(wp, DIVY_ML_REP_DOWNLOAD_USERNAME, divy_get_fullname(r), &keydata);

	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* $schema://$root/ を算出する */
	val =  divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, dav_divy_get_root_uri(r), r, NULL);
	_push_replace_param(wp, DIVY_ML_REP_ROTURL, apr_psprintf(wp, "%s/", val), &keydata);

	/* ロックアウトされたユーザID */
	_push_replace_param(wp, DIVY_ML_REP_USERID, r->user, &keydata);

	/* ロックアウトされたユーザ名 */
	_push_replace_param(wp, DIVY_ML_REP_USERNAME, divy_get_fullname(r), &keydata);

	/* 最終アクセス日（ロックされた日ではない) */
	(void) divy_format_time_t(wp, divy_get_lastaccess(r),
									  DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_LASTACCESS, datestr, &keydata);

	/* グループリーダー機能による通知者の変更 */
	if (support_groupleader && IS_FILLED(ownerid)) {
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "1", &keydata);
	}
	else { 
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "0", &keydata);
	}

	/* ログイン失敗回数 */
	_push_replace_param(wp, DIVY_ML_REP_FAILELOGIN_COUNT, apr_psprintf(wp, "%d", dconf->lockoutlimit), &keydata);

	/* リモートアドレス */
	_push_replace_param(wp, DIVY_ML_REP_REMOTE_ADDR,
								r->useragent_ip, &keydata);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートから本文を作成する */
	if (_make_svrml_data(wp, DIVY_SVRML_FAILED_LOGIN_ACCOUNT_LOCK_PREFIX, lang, r, keydata, msg)) {
		goto cleanup_wk_pool;

	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid		= NULL;
	msg->from_addr->addr		= apr_pstrdup(p, adminaddr);
	msg->from_addr->displayname = NULL;
	msg->from_addr->next		= NULL;

	/* 送信先(To) 管理者と本人 */
	msg->to_addr = msg->from_addr;	/* 管理者はfromと同じ */

	m = msg;

	if (failedaddr) {
		m->next = apr_pcalloc(wp, sizeof(MlMessage));
		m = m->next;
		m->subject         = msg->subject;
		m->priority        = msg->priority;
		m->header_encoding = msg->header_encoding;
		m->mime_type       = msg->mime_type;
		m->charset         = msg->charset;
		m->from_addr       = msg->from_addr;
		m->body            = msg->body;

		m->to_addr         = apr_pcalloc(wp, sizeof(MlAddress));
		m->to_addr->userid      = NULL;
		m->to_addr->addr        = apr_pstrdup(p, failedaddr);
		m->to_addr->displayname = NULL;
		m->to_addr->next        = NULL;
		/* アドレスを切り分ける */
		(void) divy_ml_setMultiAddress(r->pool, &m->to_addr, &m->cc_addr);
	}

	/* グループリーダーがいればToに追加する */
	if (support_groupleader) {
		if ((ownerid &&
				divy_rdbo_get_user_property(r, ownerid, &owner_pr) == 0) &&
															owner_pr != NULL) {
			if (owner_pr->mailaddr != NULL) {
				m->next = apr_pcalloc(wp, sizeof(MlMessage));
				m = m->next;
				m->subject         = msg->subject;
				m->priority        = msg->priority;
				m->header_encoding = msg->header_encoding;
				m->mime_type       = msg->mime_type;
				m->charset         = msg->charset;
				m->from_addr       = msg->from_addr;
				m->body            = msg->body;

				m->to_addr              = apr_pcalloc(wp, sizeof(MlAddress));
				m->to_addr->userid      = NULL;
				m->to_addr->addr        = apr_pstrdup(p, owner_pr->mailaddr);
				m->to_addr->displayname = NULL;
				m->to_addr->next        = NULL;
				/* アドレスを切り分ける */
				(void) divy_ml_setMultiAddress(r->pool, &m->to_addr,
											   					&m->cc_addr);
			}
		}
	}

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			goto cleanup_wk_pool;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send account lock notify  mail.(userid = %s)", 
														divy_get_userid(r));
	}
cleanup_wk_pool:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return 0;
}

/**
 * BOX画面を開いたときに送信するメール
 * @param r request_rec *
 * @param type int (メールの種類 DIVY_ML_BOX_TYPE_XXXX)
 * @param box divy_rdbo_box *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_box(request_rec *r, int type, divy_rdbo_box *box)
{
	apr_pool_t *p	= r->pool;
	apr_pool_t *wp  = NULL;
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	const char *adminaddr = NULL, *boxowner = NULL, *val, *lang;
	divy_rdbo_usr	  *usr_pr = NULL;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg = NULL;
	char *datestr = NULL;
	int ret = 0;
	char *template_prefix = NULL;
	apr_uint32_t propflag     = 0;
	divy_rdbo_resource *rdb_r = NULL, *tmp_rdb_r = NULL;
	divy_sbuf *sbuf = NULL;
	int fcnt = 0;
	char* boxfolder = NULL;

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	/* メールのテンプレートを決定する */
	switch (type)
	{
		case DIVY_ML_BOX_TYPE_OPEN_NOTIFY:
			template_prefix = DIVY_SVRML_BOX_OPEN_NOTIFICATION_PREFIX;
			break;

		case DIVY_ML_BOX_TYPE_OPEN_REPORT:
			template_prefix = DIVY_SVRML_BOX_OPEN_REPORT_PREFIX;
			break;

		default:
			template_prefix = DIVY_SVRML_BOX_OPEN_NOTIFICATION_PREFIX;
			break;
	}

	/* BOX作成者のメールアドレスを取得する Toになる*/
	if ((divy_rdbo_get_user_property(r, box->creator_usr_id, &usr_pr)) != 0)
		return 1;

	boxowner = apr_pstrdup(p, usr_pr->mailaddr);

	/* 管理者のアドレスを取得する fromになる */
	adminaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;

	/* 送信元メールアドレスが存在しなければ処理は継続しない */
	if (IS_EMPTY(adminaddr) || IS_EMPTY(boxowner)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send Notify open BOX mail."
				"box owner or adminaddress is NULL");
		return 1;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/*
	 * 置換キーワード一覧を作成する
	 */

	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* $schema://$root/ を算出する */
	val =  divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, dav_divy_get_root_uri(r), r, NULL);
	_push_replace_param(wp, DIVY_ML_REP_ROTURL, apr_psprintf(wp, "%s/", val), &keydata);

	/* リモートアドレス */
	/* BOXはブラウザからも行うことが可能になっているがその場合のIPアドレスは
	 * リクエスト元の127.0.0.1になってしまう。
	 * CGIにてリクエスト元のIPアドレスをヘッダで渡すことによりこの問題を
	 * 回避します。
	 */
	val = apr_table_get(r->headers_in, "X-Tf-RemoteAddr");
	if (IS_EMPTY(val)) {
		val = r->useragent_ip;
	}
	_push_replace_param(wp, DIVY_ML_REP_REMOTE_ADDR, val, &keydata);

	/* アクセスパス(shorten path) */
	_push_replace_param(wp, DIVY_ML_REP_BOX_SHORTEN_URL,
			apr_psprintf(wp, "%s?" DIVY_URIPARAM_CMD "=" DIVY_CMD_BOX, 
						divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, 
						divy_build_shorten_url(wp, dav_divy_get_root_uri(r),
							box->shorten), r, NULL)), &keydata);

	/* 実フォルダ */
	boxfolder = dav_divy_extract_finalpath_segment(p, box->uri);
	_push_replace_param(wp, DIVY_ML_REP_PARENT_FOLDER,  boxfolder, &keydata);

	/* BOXの件名 */
	_push_replace_param(wp, DIVY_ML_REP_BOX_GREETING, box->greeting, &keydata);

	/* BOX設定されているフォルダ名 */

	/* アクセス日 */
	(void) divy_format_time_t(wp, dav_divy_get_now_epoch(),
									  DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_LASTACCESS, datestr, &keydata);

	/* BOX公開日 */
	(void) divy_format_time_t(wp, box->creationdate,
									  DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_BOX_OPEN_PUB_DATE, datestr, &keydata);

	/* BOX有効期限 */
	if (box->expirationdate > 0) {
		(void) divy_format_time_t(wp, box->expirationdate,
										DIVY_TIME_STYLE_JAPANESE, &datestr);
	}
	else {
		datestr = apr_pstrdup(wp, "-\0");
	}
	_push_replace_param(wp, DIVY_ML_REP_BOX_OPEN_PUB_EXPIRED,
													datestr, &keydata);

	/* 相手先メールアドレス */
	val = IS_FILLED(box->tomailaddr) ? box->tomailaddr : "-";

	_push_replace_param(wp, DIVY_ML_REP_BOX_TO_MAILADDRESS,
											val, &keydata);

	/* レポートの場合はBOX公開されているファイルの一覧を作成する */ 
	if (type == DIVY_ML_BOX_TYPE_OPEN_REPORT) {
		propflag |= DIVY_GET_PROP_fullname;
		rdb_r        = apr_pcalloc(p, sizeof(divy_rdbo_resource));
		rdb_r->uri   = box->uri;
		rdb_r->depth = divy_count_dirs(rdb_r->uri);
		rdb_r->p     = wp;
		divy_sbuf_create(wp, &sbuf, 2048); 

		(void)divy_rdbo_get_hierarchy_property(r, rdb_r, rdb_r->depth, propflag, NULL);
		/*
		 * BOX内部のファイル名を取得する
		 * rdb_rの一番初めはBOXそのものなので除外します
		 */
		for(tmp_rdb_r = rdb_r->next;
				tmp_rdb_r != NULL; tmp_rdb_r = tmp_rdb_r->next) {
			divy_sbuf_append(sbuf, apr_psprintf(wp, "%d: %s"CRLF"  ",
										++fcnt, tmp_rdb_r->displayname));
		}
		_push_replace_param(wp, DIVY_ML_REP_BOX_OPEN_FILE_LIST,
								divy_sbuf_tostring(sbuf), &keydata);
		/* ファイル件数 */
		_push_replace_param(wp, DIVY_ML_REP_BOX_OPEN_FILE_COUNT,
												apr_itoa(wp, fcnt), &keydata);

	}

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートから本文を作成する */
	if (_make_svrml_data(wp, template_prefix, lang, r, keydata, msg)) {
		goto cleanup_wk_pool;
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid		= NULL;
	msg->from_addr->addr		= apr_pstrdup(p, adminaddr);
	msg->from_addr->displayname = NULL;
	msg->from_addr->next		= NULL;

	/* 送信先 */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->to_addr->addr          = apr_pstrdup(p, boxowner);

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			goto cleanup_wk_pool;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);

cleanup_wk_pool:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return 0;

}

/**
 * アップロードポリシーを適用した時に所属グループメンバに送られるメール
 * @param r request_rec *
 * @param grp_pr const divy_rdbo_grp *
 * @param usr_pr divy_rdbo_usr *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_group_policy_apply_notify(request_rec *r,
												const divy_rdbo_grp *grp_pr,
												divy_rdbo_usr *usr_pr)
{

	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	divy_rdbo_upload_policy *policy = grp_pr->policy;
	const char *adminaddr = NULL;
	apr_pool_t *p   = r->pool;
	apr_pool_t *wp  = NULL;
	divy_replace_keydata *keydata = NULL;
	const char *lang, *val;
	char *datestr = NULL;
	divy_rdbo_usr *usr = NULL;
	MlMessage *msg , *m;
	int isFirst = 0;
	int st = 0;
	int ret = 0;

	if (IS_MAIL(dconf, sconf)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"a mail function is NULL");
		return 0;	/* メール機能がOFFなので送信できない */
	}

	/* 管理者のアドレスを取得する fromになる */
	adminaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;

	/* 送信元メールアドレスが存在しなければ処理は継続しない */
	if (IS_EMPTY(adminaddr)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send Notify group policy apply mail."
				"adminaddress is NULL");
		return 1;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* 置換キーワード一覧を作成する */
	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);
	/* グループ名 */
	_push_replace_param(wp, DIVY_ML_REP_GROUPNAME, grp_pr->name, &keydata);
	/* ルール更新日 */
	(void)divy_format_time_t(wp, policy->updatedate,
									DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_UPDATEDATE,
														datestr, &keydata);
	/* マッチタイプ */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_MATCHTYPE,
					apr_psprintf(wp, "%d", policy->rules_matchtype), &keydata);
	/* ファイル名ルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_FILE,
										policy->rules_fnames, &keydata);
	/* 拡張子ルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_SUFFIX,
										policy->rules_suffix, &keydata);
	/* 長さルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_LENGTH,
										policy->rules_length, &keydata);
	/* 文字種ルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_CHAR,
					apr_psprintf(wp, "%d", policy->rules_chartype), &keydata);
	/* ルール非公開 */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_HIDDEN,
					apr_psprintf(wp, "%d", policy->hidden), &keydata);
	/* 有効/無効 */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_ACTIVE,
					apr_psprintf(wp, "%d", policy->active), &keydata);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートから本文を作成する */
	if (_make_svrml_data(wp, DIVY_SVRML_GROUP_POLICY_APPLY_NOTICE_PREFIX,
														lang, r, keydata, msg)) {
		goto cleanup_wk_pool;
	}

	/* 優先度(高)にする */
	msg->priority = ML_PRIORITY_HIGH;

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid		= NULL;
	msg->from_addr->addr		= apr_pstrdup(p, adminaddr);
	msg->from_addr->displayname = NULL;
	msg->from_addr->next		= NULL;

	isFirst = 1;
	for (usr = usr_pr; usr; usr = usr->next) {
		if (IS_EMPTY(usr->mailaddr)) {
			/* TODO ログだそう */
			continue;
		}

		if (isFirst) {
			m = msg;
		}
		else {
			m->next = apr_pcalloc(wp, sizeof(MlMessage));
			m = m->next;

			m->subject         = msg->subject;
			m->priority        = msg->priority;
			m->header_encoding = msg->header_encoding;
			m->mime_type       = msg->mime_type;
			m->charset         = msg->charset;
			m->from_addr       = msg->from_addr;
			m->body            = msg->body;

			m->from_addr              = apr_pcalloc(wp, sizeof(MlAddress));
			m->from_addr->userid      = NULL;
			m->from_addr->addr        = apr_pstrdup(p, msg->from_addr->addr);
			m->from_addr->displayname = NULL;
			m->from_addr->next        = NULL;
		}

		m->to_addr              = apr_pcalloc(wp, sizeof(MlAddress));
		m->to_addr->userid      = apr_pstrdup(p, usr->usrid);
		m->to_addr->addr        = apr_pstrdup(p, usr->mailaddr);
		m->to_addr->displayname = apr_pstrdup(p, usr->fullname);
		m->to_addr->next        = NULL;

		/* アドレスを切り分ける */
		(void) divy_ml_setMultiAddress(r->pool, &m->to_addr, &m->cc_addr);
		isFirst = 0;
	}

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			st = 1;
			goto cleanup_wk_pool;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send upload policy apply notify mail.(userid = %s)",
															divy_get_userid(r));
		st = 1;
	}

cleanup_wk_pool:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return st;

}

/**
 * アップデートポリシーに違反したときに飛ばされるメール
 * @param r request_rec *
 * @param filename const char *		対象となったファイル
 * @param grp_pr	divy_rdbo_grp *
 * @param prefix	int				ファイル名のパス(1: pass / 0: NG)
 * @param char		int				文字種のパス(1: pass / 0: NG)
 * @param len		int				長さのパス(1: pass / 0: NG)
 * @param suffix	int				拡張子のパス(1: pass / 0: NG)
 *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_group_policy_inspection_result(request_rec *r,
												const char* filename,
												divy_rdbo_grp *grp_pr,
												int prefix_pass, int char_pass,
												int len_pass, int suffix_pass)
{

	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	divy_rdbo_upload_policy *policy = grp_pr->policy;
	apr_pool_t *p   = r->pool;
	apr_pool_t *wp  = NULL;
	int ret = 0;
	divy_replace_keydata *keydata = NULL;
	const char* addr = divy_get_mailaddr(r);
	const char* adminaddr = NULL;
	const char* lang, *val;
	char *datestr            = NULL;
	MlMessage *msg              = NULL;

	if (IS_MAIL(dconf, sconf)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_DATA,
				"a mail function is NULL");
		return 0;	/* メール機能がOFFなので送信できない */
	}

	/* 管理者のアドレスを取得する fromになる */
	adminaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;

	/* 送信元メールアドレスが存在しなければ処理は継続しない */
	if (IS_EMPTY(adminaddr)) {
		ERRLOG0(p, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"We could not send Notify group violation mail."
				"adminaddress is NULL");
		return 1;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* 置換キーワード一覧を作成する */
	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);
	/* ファイル名 */
	_push_replace_param(wp, DIVY_ML_REP_FILENAME, filename, &keydata);
	/* 登録日 */
	(void) divy_format_time_t(wp, dav_divy_get_now_epoch(), DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_REGDAY, datestr, &keydata);
	/* グループ名 */
	_push_replace_param(wp, DIVY_ML_REP_GROUPNAME, grp_pr->name, &keydata);
	/* ユーザID */
	_push_replace_param(wp, DIVY_ML_REP_USERID, divy_get_userid(r), &keydata);
	/* ユーザ名 */
	_push_replace_param(wp, DIVY_ML_REP_USERNAME,  divy_get_fullname(r), &keydata);
	/* ルール更新日 */
	(void)divy_format_time_t(wp, policy->updatedate,
									DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_UPDATEDATE,
														datestr, &keydata);
	/* ポリシー有効無効 */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_ACTIVE,
					apr_psprintf(wp, "%d", policy->active), &keydata);
	/* マッチタイプ */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_MATCHTYPE,
					apr_psprintf(wp, "%d", policy->rules_matchtype), &keydata);
	/* ファイル名ルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_FILE,
										policy->rules_fnames, &keydata);
	/* 拡張子ルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_SUFFIX,
										policy->rules_suffix, &keydata);
	/* 長さルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_LENGTH,
										policy->rules_length, &keydata);
	/* 文字種ルール */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_CHAR,
					apr_psprintf(wp, "%d", policy->rules_chartype), &keydata);
	/* ルール非公開 */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_HIDDEN,
					apr_psprintf(wp, "%d", policy->hidden), &keydata);
	/* prefix_pass */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_PREFIX_PASS,
					apr_psprintf(wp, "%d", prefix_pass), &keydata);
	/* suffix_pass */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_SUFFIX_PASS,
					apr_psprintf(wp, "%d", suffix_pass), &keydata);
	/* char_pass */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_CHAR_PASS,
					apr_psprintf(wp, "%d", char_pass), &keydata);
	/* len_passs */
	_push_replace_param(wp, DIVY_ML_REP_GRP_POLICY_LEN_PASS,
					apr_psprintf(wp, "%d", len_pass), &keydata);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートから本文を作成する */
	if (_make_svrml_data(wp, DIVY_SVRML_GROUP_POLICY_VIOLATION_PREFIX,
														lang, r, keydata, msg)) {
		goto cleanup_wk_pool;
	}

	/* 優先度(高)にする */
	msg->priority = ML_PRIORITY_HIGH;

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid		= NULL;
	msg->from_addr->addr		= apr_pstrdup(p, adminaddr);
	msg->from_addr->displayname = NULL;
	msg->from_addr->next		= NULL;

	/* 送信先 */
	if (IS_FILLED(addr)) {
		msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
		msg->to_addr->addr          = apr_pstrdup(p, addr);
	}
	else {
		/* 一旦fromと同じにする */
		msg->to_addr = msg->from_addr;

		/* 本人のメールアドレスがないときはオーナー */
		const char* ownerid = divy_get_userownerid(r);
		divy_rdbo_usr* owner_pr = NULL;

		if (ownerid && 
			divy_rdbo_get_user_property(r, ownerid, &owner_pr) == 0) {
			if (owner_pr != NULL && IS_FILLED(owner_pr->mailaddr)) {
				msg->to_addr->addr = owner_pr->mailaddr;
			}
		}
		/* オーナーいないときはfromと同じユーザになります */
	}

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			goto cleanup_wk_pool;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);

cleanup_wk_pool:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return 0;

}

/**
 * パスワード変更通知を送信する
 * 
 * @param r request_rec *
 * @param usr_pr const divy_rdbo_usr * ユーザ情報
 */
DIVY_DECLARE(int) divy_ml_notify_changed_password(request_rec *r, const divy_rdbo_usr *usr_pr)
{
	apr_pool_t *p            = r->pool;
	apr_pool_t *wp           = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	char *admin_mailaddr, *admin_fullname;
	const char *lang;
	divy_replace_keydata *keydata = NULL;
	MlMessage *msg           = NULL;
	int ret = 0;

	TRACE(p);

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);
	
	/* 管理者のアドレスを取得する fromになる */
	admin_mailaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;
	admin_fullname = IS_FILLED(dconf->mlrepfromname) ?
							dconf->mlrepfromname : dconf->mailadminaddr;

	/* 置換キーワード一覧を作成する */
	_push_replace_param(wp, DIVY_ML_REP_ADMIN,    admin_fullname, &keydata);

	/* ユーザ情報をかき集める */
	_build_user_notification_body(r, wp, keydata, (divy_rdbo_usr*)usr_pr);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBody と Subject の取得して置換パラメータで置換する */
	if (_make_svrml_data(wp, DIVY_SVRML_NOTIFY_CHANGED_PASSWORD_PREFIX,
													lang, r, keydata, msg)) {
		goto cleanup_wk_resource;
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信先(to) */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->to_addr->userid      = usr_pr->usrid;
	msg->to_addr->addr        = usr_pr->mailaddr;
	msg->to_addr->displayname = usr_pr->fullname;
	msg->to_addr->next        = NULL;

	(void) divy_ml_setMultiAddress(wp, &msg->to_addr, &msg->cc_addr);

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	msg->from_addr->addr        = admin_mailaddr;
	msg->from_addr->displayname = admin_fullname;
	msg->from_addr->next        = NULL;

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret =divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");

			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		/* エラー報告は行うがエラーにはしない */
		ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send changed-password-notify mail."
			"If you need it, Please contact update user."
			"(creator = %s, changed password user = %s)",
			divy_get_userid(r), usr_pr->usrid);
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return 0;
}

/**
 * 二段階認証のトークンを送る
 * @param r			request_rec *		リクエスト構造体
 * @param mailaddr	const char * 		メールアドレス
 * @param token		char *				トークン文字列
 * @param expire	apr_time_t *		有効期限
 *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_2FA_token(request_rec *r, const char *mailaddr,
											char* token, apr_time_t expire)
{
	apr_pool_t *p            = r->pool;
	apr_pool_t *wp           = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	char *admin_mailaddr;
	const char *lang;
	char *datetext = NULL;
	divy_replace_keydata *keydata = NULL;
	divy_rdbo_usr *usr_pr = NULL;
	MlMessage *msg           = NULL;
	int ret = 0;

	TRACE(p);

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* 送信元管理者アカウントの作成 */
	if (IS_FILLED(dconf->mlrepfromaddr)) {
		admin_mailaddr = apr_pstrdup(wp, dconf->mlrepfromaddr);
	}
	else {
		admin_mailaddr = apr_pstrdup(wp, dconf->mailadminaddr);
	}

	/* 日付を作成 */
	divy_format_time_t(wp, expire, DIVY_TIME_STYLE_JAPANESE, &datetext);

	/* 置換キーワード一覧を作成する */
	_push_replace_param(wp, DIVY_ML_REP_2FA_TOKEN,      token,    &keydata);
	_push_replace_param(wp, DIVY_ML_REP_2FA_EXPIRATION, datetext, &keydata);

	usr_pr = divy_get_usrprop(r);
	/* ユーザ情報をかき集める */
	_build_user_notification_body(r, wp, keydata, (divy_rdbo_usr*)usr_pr);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);

	/* メールテンプレートからBodyとSubject の取得して置換パラメータで置換する */
	if (_make_svrml_data(wp, DIVY_SVRML_NOTIFY_2FA_TOKEN_PREFIX,
													lang, r, keydata, msg)) {
		goto cleanup_wk_resource;
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		msg->charset = dconf->mailencoding;
	}

	/* 送信先(to) */
	msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
	if (IS_FILLED(mailaddr)) {
		msg->to_addr->userid      = "";
		msg->to_addr->addr        = (char*)mailaddr;
		msg->to_addr->displayname = "";
		msg->to_addr->next        = NULL;
	}
	else {
		msg->to_addr->userid      = usr_pr->usrid;
		msg->to_addr->addr        = usr_pr->mailaddr;
		msg->to_addr->displayname = usr_pr->fullname;
		msg->to_addr->next        = NULL;
	}

	(void) divy_ml_setMultiAddress(wp, &msg->to_addr, &msg->cc_addr);

	/* 送信元(from) */
	msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	msg->from_addr->userid      = NULL;
	msg->from_addr->addr        = admin_mailaddr;
	msg->from_addr->displayname = "";
	msg->from_addr->next        = NULL;

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret =divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");

			goto cleanup_wk_resource;
		}
	}

	/* メール送信の実施 */
	ret = divy_ml_sendmaildirect(r, msg);
	if (ret) {
		/* エラー報告は行うがエラーにはしない */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to send 2FA token mail. (user = %s)", usr_pr->usrid);
	}

cleanup_wk_resource:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;
	
	return 0; /* 成功 */
}

/**
 * BOXを渡す利用者に対しての通知を送る
 *
 * @param r request_rec *
 * @param box divy_rdbo_box *
 * @return int 処理ステータス (0: 成功 / 1: 失敗)
 */
DIVY_DECLARE(int) divy_ml_box_invitation(request_rec *r, divy_rdbo_box *box)
{
	apr_pool_t *p            = r->pool;
	apr_pool_t *wp           = NULL;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	MlMessage *msg = NULL, *tmp_msg = NULL;
	divy_replace_keydata *keydata = NULL;
	const char *lang, *val;
	const char *adminaddr = NULL;
	char *datestr = NULL;
	char *addrs = NULL, *token = NULL, *ctx = NULL;
	int	first_flg = 0;
	int ret = 0;
	ap_regex_t *preg;

	TRACE(p);

	if (IS_MAIL(dconf, sconf)) {
		return 0;	/* メール機能がOFFなので送信できない */
	}

	/* boxがない。もしくはメールアドレスがない場合は処理しない */
	if (box == NULL || IS_EMPTY(box->tomailaddr)) {
		return 0;
	}

	/* 作業用のサブプール作成 */
	apr_pool_create(&wp, p);

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	/* 管理者のアドレスを取得する fromになる */
	adminaddr = IS_FILLED(dconf->mlrepfromaddr) ?
							dconf->mlrepfromaddr : dconf->mailadminaddr;

	/*
	 * 置換キーワード一覧を作成する
	 */

	/* ブランド名 */
	val = (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	/* $schema://$root/ を算出する */
	val =  divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, dav_divy_get_root_uri(r), r, NULL);
	_push_replace_param(wp, DIVY_ML_REP_ROTURL, apr_psprintf(wp, "%s/", val), &keydata);

	/* アクセスパス(shorten path) */
	_push_replace_param(wp, DIVY_ML_REP_BOX_SHORTEN_URL,
			apr_psprintf(wp, "%s?" DIVY_URIPARAM_CMD "=" DIVY_CMD_BOX, 
						divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, 
						divy_build_shorten_url(wp, dav_divy_get_root_uri(r),
							box->shorten), r, NULL)), &keydata);

	/* BOXの件名 */
	_push_replace_param(wp, DIVY_ML_REP_BOX_GREETING, box->greeting, &keydata);

	/* BOX公開日 */
	(void) divy_format_time_t(wp, box->creationdate,
									  DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_BOX_OPEN_PUB_DATE, datestr, &keydata);

	/* BOX有効期限 */
	if (box->expirationdate > 0) {
		(void) divy_format_time_t(wp, box->expirationdate,
										DIVY_TIME_STYLE_JAPANESE, &datestr);
	}
	else {
		datestr = apr_pstrdup(wp, "-\0");
	}
	_push_replace_param(wp, DIVY_ML_REP_BOX_OPEN_PUB_EXPIRED,
													datestr, &keydata);

	/* アクセスコードの有無 */
	val = IS_FILLED(box->password) ? "1" : "0";
	_push_replace_param(wp, DIVY_ML_REP_BOX_HAS_ACCESSCODE, val, &keydata);

	/* メールメッセージの作成 */
	divy_ml_create_mlmsg(wp, &msg);
	tmp_msg = msg;

	/* メールテンプレートから本文を作成する */
	if (_make_svrml_data(wp,
				DIVY_SVRML_BOX_INVITATION_PREFIX, lang, r, keydata, msg)) {
		goto cleanup_wk_pool;
	}

	/* charset はコンフィグの値があればそれを使う */
	if (IS_FILLED(dconf->mailencoding)) {
		tmp_msg->charset = dconf->mailencoding;
	}

	/* 送信元(from) */
	tmp_msg->from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	tmp_msg->from_addr->userid		= NULL;
	tmp_msg->from_addr->addr		= apr_pstrdup(wp, adminaddr);
	tmp_msg->from_addr->displayname = NULL;
	tmp_msg->from_addr->next		= NULL;

	/* メールアドレスの正規表現パターン作成 */
	preg = ap_pregcomp(wp,
				"^[-A-Za-z0-9_.]+@[-A-Za-z0-9_]+[.][-A-Za-z0-9._]+$", 0);

	/* 送信先(to) */
	addrs = apr_pstrdup(wp, box->tomailaddr);
	while ((token = apr_strtok(addrs, ",", &ctx)) != NULL) {
		addrs = NULL;
		if (token == NULL) break;

		token = (char *) dav_divy_trim_white(wp, token);

		/* メールアドレスのルールになっていない場合はスキップ */
		if (ap_regexec(preg, token, 0, NULL, 0)) {
			continue;
		}

		if (first_flg) {
			divy_ml_create_mlmsg(wp, &tmp_msg->next);
			tmp_msg = tmp_msg->next;

			tmp_msg->subject			= msg->subject;
			tmp_msg->priority 			= msg->priority;
			tmp_msg->header_encoding	= msg->header_encoding;
			tmp_msg->mime_type			= msg->mime_type;
			tmp_msg->charset			= msg->charset;
			tmp_msg->from_addr			= msg->from_addr;
			tmp_msg->body				= msg->body;

		}

		tmp_msg->to_addr = apr_pcalloc(wp, sizeof(MlAddress));
		tmp_msg->to_addr->userid      = NULL;
		tmp_msg->to_addr->addr        = apr_pstrdup(wp, token);
		tmp_msg->to_addr->displayname = NULL;
		first_flg = 1;
	}

	ap_pregfree(wp, preg);

	/* エンコーディング変換 */
	ret = divy_ml_encode_msg(wp, msg);
	if (ret != ML_SUCCESS) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to encode mail-data. continue utf8 encoding.");

		msg->charset = ML_CHARSET_UTF_8;
		ret = divy_ml_encode_msg(wp, msg);
		if (ret != ML_SUCCESS) {
			ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Wnn.. Failed to encoding by UTF8. give up.");
			goto cleanup_wk_pool;
		}
	}

	/* first_flgが`1`にならないとメールを送るユーザがいないことを意味します */
	if (first_flg) {
		/* メール送信の実施 */
		ret = divy_ml_sendmaildirect(r, msg);
		/* result-stateヘッダを付与する */
		divy_set_resultstatus(r, DIVY_RESULTSTATUS_TYPE_SENDMAIL);
	}

cleanup_wk_pool:
	/* サブプールを破棄する */
	if (wp != NULL) apr_pool_destroy(wp); wp = NULL;

	return 0;
}

/**
 * メッセージ全体をエンコードする。
 * 失敗した場合は全てUTF8（変換しない）で設定を戻す。
 *
 * @param pool apr_pool_t *	プール
 * @param msg  MlMessage * 	メッセージ構造体
 * @return int (ML_SUCCCESS: 成功 | ML_ERROR: 失敗)
 */
DIVY_DECLARE(int) divy_ml_encode_msg(apr_pool_t *pool, MlMessage *msg)
{

	MlMessage *top_msg = msg;
	int encret = 0;
	char *from_name = NULL;
	char *replyto_name = NULL;
	char *to_name = NULL;
	char *cc_name = NULL;
	char *bcc_name = NULL;
	char *subject = NULL;
	char *body = NULL;
	char *bodyfodder = NULL;
	char *charset = NULL;

	if (pool == NULL || msg == NULL) return ML_ERROR;

	/* 一度だけエンコードかければいいもの */
	/* FROMの名前 */
	if (msg->from_addr->displayname != NULL) {
		encret = _divy_ml_encodetext(pool,
				msg->from_addr->displayname, msg->charset,
				&from_name,
				 1);

		if (encret != 0) {
			msg->encretry = 1;
			return ML_ERROR;
		}
	}

	/* reply-toの名前
 	 * reply-toは設定されていないケースもあるため、NULLチェックをする。
 	 */
	if (msg->replyto_addr != NULL && msg->replyto_addr->displayname != NULL) {
		encret = _divy_ml_encodetext(pool,
				msg->replyto_addr->displayname, msg->charset,
				&replyto_name,
				1);
	}

	/* 題名 */
	if (msg->subject != NULL) {
		encret = _divy_ml_encodetext(pool, msg->subject,
				msg->charset,
				&subject,
				1);

		if (encret != 0) {
			msg->encretry = 1;
			return ML_ERROR;
		}
	}

	/* 本文 */
	if (msg->body != NULL) {
		encret = _divy_ml_encodetext(pool, msg->body,
				msg->charset,
				&body,
				0);

		if (encret != 0) {
			msg->encretry = 1;
			return ML_ERROR;
		}
	}

	/* 全体で利用するcharset */
	charset = msg->charset;

	/* ループしなければならないもの */
	for (;msg != NULL; msg = msg->next) {
		/* ここですべてのメッセージをエンコードする */

		/*
		 * 先に決定されたcharsetで移行すべてを行います。
		 * 先頭のcharsetをすべてのmsgへ投入しているわけは
		 * エンコードが失敗したときに置き換えるcharsetは
		 * 先頭のmsg->charのみであるためです。
		 * 
		 * msg->charset = ML_CHARSET_UTF_8
		 * このようにしているだけでは次の場所には入らない
		 * msg->next->charset
		 */
		msg->charset = charset;

		/* TOの名前
		 * 変換に失敗したら表示名（displayname)は使わない
		 * この変換はエラーにならない
		 */
		if (msg->to_addr != NULL) {
			MlAddress *tmp_addr = NULL;
			for (tmp_addr = msg->to_addr; tmp_addr != NULL;
					tmp_addr = tmp_addr->next) 
			{
				encret = _divy_ml_encodetext(pool,
						tmp_addr->displayname,
						msg->charset,
						&to_name, 1);
				if (encret != 0) {
					tmp_addr->displayname = '\0';
				}
				else {
					tmp_addr->displayname = to_name;
				}
			}

		}

		/*
		 * CCの名前
		 * 変換に失敗したら表示名（displayname)は使わない
		 * この変換はエラーにならない
		 */

		if (msg->cc_addr != NULL) {
			MlAddress *tmp_addr = msg->cc_addr;
			for (; tmp_addr != NULL; tmp_addr = tmp_addr->next)
			{
				encret = _divy_ml_encodetext(pool,
						tmp_addr->displayname,
						msg->charset,
						&cc_name, 1);
				if (encret != 0)
				{
					tmp_addr->displayname = '\0';
				}
				else {
					tmp_addr->displayname = cc_name;
				}
			}
		}

		/*
		 * BCCの名前
		 * 変換に失敗したら表示名（displayname)は使わない
		 * この変換はエラーにならない
		 */

		if (msg->bcc_addr != NULL) {
			MlAddress *tmp_addr = msg->bcc_addr;
			for (; tmp_addr != NULL; tmp_addr = tmp_addr->next)
			{
				encret = _divy_ml_encodetext(pool,
						tmp_addr->displayname,
						msg->charset,
						&bcc_name, 1);
				if (encret != 0)
				{
					tmp_addr->displayname = '\0';
				}
				else {
					tmp_addr->displayname = bcc_name;
				}
			}
		}

		/* 本文フッダ */
		if (msg->bodyfodder != NULL) {
			encret = _divy_ml_encodetext(pool,
					(const char *)msg->bodyfodder,
					msg->charset, &bodyfodder, 0);

			if (encret != 0) {
				msg->encretry = 1;
				return ML_ERROR;
			}
			msg->bodyfodder = bodyfodder;
		}
	}


	/* 変換した内容を全て適用する */
	for (; top_msg != NULL; top_msg = top_msg->next) {
		top_msg->from_addr->displayname = from_name;
		if (IS_FILLED(replyto_name)) {
			top_msg->replyto_addr->displayname = replyto_name;
		}
		top_msg->subject = subject;
		if (IS_FILLED(top_msg->bodyfodder)) {
			/* (note) CRLF は送信側が付けてくれるので要りません。*/
			//top_msg->body = apr_psprintf(pool, "%s%s%s", body, DIVY_CRLF,
			//				top_msg->bodyfodder); 
			top_msg->body = apr_psprintf(pool, "%s%s", body,
							top_msg->bodyfodder); 
		}
		else {
			//top_msg->body = apr_psprintf(pool, "%s%s", body, DIVY_CRLF);
			top_msg->body = apr_pstrdup(pool, body);
		}
	}

	return ML_SUCCESS;

}

/*--------------------------------------------------------------
  Define private function
  --------------------------------------------------------------*/
/**
 * URIを利用してメールウォッチを行っているユーザをMlAddress構造体へセットする
 *
 * @param r 	request_rec *
 * @param list	tf_ml_UriList *
 * @param usrlist	userlist*
 * @return int  ML_SUCCESS / ML_ERROR
 *
 */
static int _divy_ml_alloc_address(request_rec *r, urilist *l, userlist **usrlist)
{

	apr_pool_t	*p	    = r->pool;
	divy_rdbo_usr	*dbusr      = NULL;
	const char 	*key 	    = NULL;
	urilist		*tmplist    = NULL;
	int		rt	    = ML_ERROR;
	apr_hash_t	*usrlist_h  = apr_hash_make(p);	/* ユーザID - userlist */
	userlist	*tmpusrlist = NULL, *a_usrlist;

	TRACE(p);

	*usrlist = NULL;	/* 初期化 */

	for (tmplist = l; tmplist != NULL; tmplist = tmplist->next) {

		/* URIの後ろの/を切り取る */
		key = dav_divy_remove_endslash(p, tmplist->uri);

		/* ユーザテーブルよりユーザ名とアドレスを取得する */
		if ((dbusr = divy_rdbo_ml_get_addr(r, key, tmplist->method)) == NULL)
			continue;

		/* ここにきたならユーザが一人はいたということ */
		rt = ML_SUCCESS;

		/*
		 * ユーザのメールアドレスとフルネームを作成する
		 * srcとdst共に取得したアドレスを格納する
		 * src,dstとも含まれる可能性もあります。
		 */
		for (; dbusr != NULL; dbusr = dbusr->next) {
			if (*usrlist == NULL) {
				*usrlist = tmpusrlist = apr_pcalloc(p, sizeof(userlist));
			}
			else {
				/*
				 * ユーザは複数のグループをメール監視している
				 * 可能性があります。
				 * その為、グループをまたがったメール送信処理
				 * （COPY,MOVE)はユーザに複数同じメールを送信
				 * する可能性がある。
				 * ユーザが既に一件でもリストに入っていた場合は
				 * ユーザの重複を避ける為リストを全検索する
				 * ただし、MOVE,COPYはお互い監視していた場合は
				 * 報告する必要があるために、送り先（移動先）、
				 * 送り元（移動元）の管理をしているかを保存する
				 */
				a_usrlist = apr_hash_get(usrlist_h, dbusr->usrid, APR_HASH_KEY_STRING);
				/* userid が重複していた場合 */
				if (a_usrlist != NULL) {
					/* 昔取得したwatchplace をOR する */
					a_usrlist->watchplace |= tmplist->watchplace;
					continue;	/* 重複禁止。次へ */
				}
				/* userid が重複していなかった場合 */
				else {
					tmpusrlist->next = apr_pcalloc(p, sizeof(userlist));
					tmpusrlist = tmpusrlist->next;
				}
			}

			tmpusrlist->address = apr_pcalloc(p, sizeof(MlAddress));
			tmpusrlist->address->userid	 = dbusr->usrid;
			tmpusrlist->address->addr        = dbusr->mailaddr;
			tmpusrlist->address->displayname = dbusr->fullname;
			tmpusrlist->watchplace           = tmplist->watchplace;
			tmpusrlist->next                 = NULL; /* sentinel */

			/* ユーザIDをキーにしてtmpusrlist を記録する */
			apr_hash_set(usrlist_h, tmpusrlist->address->userid, APR_HASH_KEY_STRING, tmpusrlist);
		
		} /* for(; dbusr */
	}

	return rt;
}

/**
 * URIをハッシュにいれることにより重複URIをなくす
 *
 * @param pool	apr_pool_t *
 * @param w     MlWatchPlace *
 * @param uri   const char *
 * @return int 
 */
static int _divy_ml_uri_parse(apr_pool_t *pool, urilist **list,
	       						MlWatchPlace w,
	       						const char *uri,
							int method)
{

	const char *value   = NULL;
	urilist    *tmp_list = NULL;

	if (pool == NULL) return ML_ERROR;
	if (uri == NULL || *uri == '\0') return ML_SUCCESS;

	if ((value = dav_divy_escape_uri(pool, uri)) == NULL) {
		ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX, "Faied to URLEncode.");
		*list = NULL;
		return ML_ERROR;
	}

	if (value != NULL) {
		if (*list == NULL) {
			*list = apr_pcalloc(pool, sizeof(urilist));
			(*list)->uri = apr_pstrdup(pool, uri);
			(*list)->watchplace = w;
		}
		else {
			tmp_list = apr_pcalloc(pool, sizeof(urilist));
			tmp_list->uri = apr_pstrdup(pool, uri);
			tmp_list->watchplace = w;
			tmp_list->next = *list;
			*list = tmp_list;
		}
		(*list)->method = method;
	}

	return ML_SUCCESS;

}


/**
 * メールプロバイダを一意に識別するIDを取得する。
 *
 * @param p apr_pool_t * 返却する値を割り当てるプール
 * @param r request_rec *
 * @param providerName const char * DBプロバイダの識別名称
 */
static const char * _get_ml_provider_id(request_rec *r, const char *providerType)
{
	const char *vhostid;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);

	if (r->server->addrs) {
		vhostid = apr_psprintf(r->pool, "%s:%d:%s:%d",
					r->server->server_hostname,
					r->server->port,
					r->server->addrs->virthost,
					r->server->addrs->host_port);
	}
	else {
		vhostid = apr_psprintf(r->pool, "%s:%d::0",
					r->server->server_hostname,
					r->server->port);
	}

	return apr_psprintf(r->pool, "tf.mlprovider.%s#%s:%s:%s:%s@%s&%s",
				providerType,
				conf->mailhost, conf->mailport,
				conf->mailadminaddr, 
				conf->mailencoding,
				conf->root_uri, vhostid);
}

/**
 * メールプロバイダ環境を初期化して利用できる状態にする。
 * (note)
 * 	同時アクセスが起きないよう呼び出し元にて保証して下さい。
 *
 * @param r request_rec*
 * @param providerid const char * メールプロバイダID
 * @return int 処理終了ステータスコード (0: 成功 / 1: 失敗)
 */
static int _init_ml_provider(request_rec *r, const char *providerid)
{
	MlDataSource *mlds = NULL;
	int ret;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	
	TRACE(r->pool);

	/*
	 * メールプロバイダを探してMlDataSourceを取得する
	 */
	ret = divy_run_create_mldatasource(ml_p, conf->mail, &mlds);
	if (ret == ML_DECLINED) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Failed to get mail provider. (name = %s) "
			"Please check httpd.conf for mail.", conf->mail);
		return ML_ERROR;
	}

	/*
	 * メールプロバイダIDをキーとしてmlds を登録する
	 * (note)
	 * 	providerid はmlds と同じライフサイクルにするため
	 * 	再確保しています。
	 */
	if (_registe_ml_provider(r, apr_pstrdup(ml_p, providerid),
						mlds) != ML_SUCCESS) {
		return ML_ERROR;
	}

	/* 初期化処理完了の状態を記録する */
	_set_mlenv_is_ready(providerid);

	return ML_SUCCESS;
}

/**
 * provideridが示す名称でメールプロバイダ構造体(MlDataSource)を登録する。
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param r request_rec *
 * @param providerid const char* メールプロバイダID
 * @param mlds MlDataSource* プロバイダ提供のMlDataSource構造体へのポインタ
 * @return int 実行ステータス。(ML_SUCCESS: 成功 / ML_ERROR: 失敗)
 */
static int _registe_ml_provider(request_rec *r, 
					const char *providerid,
					MlDataSource *mlds)
{
	if (providerid == NULL || *providerid == '\0' || mlds == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"providerid or mlds is NULL.");
		return ML_ERROR;
	}

	/* プロバイダIDをキーとしてMlDataSource * を登録する */
	apr_hash_set(mlprv_hash, providerid, APR_HASH_KEY_STRING, mlds);
	return ML_SUCCESS;
}

/**
 * メールプロバイダの接続環境が初期化されているかどうか.
 *
 * @param r request_rec *
 * @return int 1: 初期化されている / 0: 初期化されていない
 */
static int _is_mlenv_ready(const char *providerid)
{
	char *str;

	if (apr_hash_count(ml_initfinish_hash) == 0) {
		return 0;
	}

	/* メールプロバイダの初期化状態を取得する */
	str = apr_hash_get(ml_initfinish_hash, providerid, APR_HASH_KEY_STRING);
	if (str) {
		return 1;
	}
	else {
		return 0;
	}
}

/*
 * providerid で識別されるメールプロバイダグループの初期化が完了した
 * ことを記録する。
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param providerid const char * メールプロバイダID
 */
static void _set_mlenv_is_ready(const char *providerid)
{
	/* (note)
	 * 	ml_initfinish_hash のスコープはグローバルなので、
	 * 	providerid の領域はml_p から再確保します。
	 */
	apr_hash_set(ml_initfinish_hash,
			apr_pstrdup(ml_p, providerid),
			APR_HASH_KEY_STRING, ML_PROVIDER_INIT_FINISH);
}

/**
 * メールプロバイダの環境を破棄する。
 *
 * @param data void *
 */
static apr_status_t _cleanup_ml_env(void *data)
{
	/* mutex の破棄 */
	if (gth_ml_mutex) {
		(void) apr_thread_mutex_destroy(gth_ml_mutex);
	}
	ml_initfinish_hash = NULL;
	mlprv_hash         = NULL;
	gth_ml_mutex       = NULL;
	ml_p = NULL;	/* サブプールは自動的に削除されるので何もしない */

	return APR_SUCCESS;
}

/**
 * デリミタで区切られているメールアドレスをtokenで取得する
 * apr_strtk()のメール用のラッパーです。
 * **token_ctxにNULLを設定すればはじめの一件だけが取得できます。
 *
 * @param	pool		apr_pool_t *
 * @param	address		char*
 * @param	token_ctx	char **
 * @return	token		char *
 * 		address がNULLまたは空文字であれば、NULLを返却します。
 */
static char *_divy_ml_address_strtok(apr_pool_t* pool, char *address, char **token_ctx)
{
	char *token;

	if (token_ctx != NULL) {
		token = apr_strtok(address, ",",  token_ctx);
	}
	else if (token_ctx == NULL && IS_EMPTY(address)) {
		/* token_ctx もNULLで、address も空ならば得られる値はない */
		return NULL;
	}
	else {
		char *tmp_ctx = NULL;
		token = apr_strtok(address, ",", &tmp_ctx);
	}

	return (char *)dav_divy_trim_white(pool, token);
}

/**
 * rdb_r が示すリソースのチケットURL (one-click url) を取得して返却する.
 */
static char *_get_oneclick_url(request_rec *r, divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;
	char *ticket  = "";

	if (rdb_r->u_spec == NULL) {
		(void) divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec);
		if (rdb_r->u_spec == NULL) {
			return "";
		}
	}

	/* グループフォルダの下だけ対象 */
	if (rdb_r->u_spec &&
		(rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e ||
		 rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_e_regular)) {

		ticket = divy_construct_oneclick_url(r, p, rdb_r);
	}
	else { /* 何もしない */
	}

	return ticket;
}

/**
 * rdb_r->uri が示すリソースの"監視フォルダ"を算出して返却する.
 *
 * [ フォーマット ]
 * 	/$groupname/$foldername1/.../$foldernameN
 */
static char *_get_ml_watchfolder(request_rec *r, divy_rdbo_resource *rdb_r)
{
	apr_pool_t *p = r->pool;
	char *pfolder = "", *enc_uri = "";

	if (rdb_r == NULL || IS_EMPTY(rdb_r->uri)) {
		return "";
	}

	if (rdb_r->u_spec == NULL) {
		(void) divy_parse_uri(p, dav_divy_get_root_uri(r), rdb_r->uri, &rdb_r->u_spec);
		if (rdb_r->u_spec == NULL) {
			return "";
		}
	}

	/* ごみ箱以下のリソースの場合には特別な処理を行う
	 * (note)
	 * 	rdb_r はごみ箱自身を示しています. メール通知のフォーマットが
	 * 	親フォルダを示めすURIを持っているからです */
	if (rdb_r->u_spec->infotype == DIVY_INFOTYPE_group_trash) {
		char *c;
		const char *lang = _get_ml_language_param(r);

		enc_uri = dav_divy_remove_endslash(p, rdb_r->u_spec->other_part);
		c = divy_strrchr_c(enc_uri, '/');
		if (c != NULL) *c = '\0';

		if (IS_FILLED(lang) && strcmp(lang, "ja") == 0) {
			enc_uri = apr_psprintf(p, "%s/%s", enc_uri, DIVY_ML_TRASHFOLDER_NAME_JA);
		}
		else {
			enc_uri = apr_psprintf(p, "%s/%s", enc_uri, DIVY_ML_TRASHFOLDER_NAME_EN);
		}
	}
	else {
		enc_uri = (char *)rdb_r->u_spec->other_part;
	}

	/* URLデコードする */
	(void) dav_divy_unescape_uri(p, enc_uri, &pfolder);

	return pfolder;
}

/**
 * src に含まれるkeydata->key をkeydata->value の値で置換し、
 * sbuf に結果を詰めて返却する。
 *
 * @param p apr_pool_t * 作業用のプール
 * @param src const char * 置換対象の文字列
 * @param keydata divy_replace_keydata * キーワードデータ
 * @param sbuf divy_sbuf * 置換結果を格納する可変長文字列バッファへのポインタ
 * @param int 処理ステータス (0: 成功 / 1: 失敗)
 */
static int _replace_ml_keyword(apr_pool_t *p, const char *src,
					divy_replace_keydata *keydata,
					divy_sbuf *sbuf)
{
	char *str, *prev, *end, *else_str, *line, *true_val, *false_val, *first, *rval;
	int match_keyword;
	divy_replace_keydata *key;
	divy_sbuf *tmp_sbuf = NULL;
	int endlen = strlen("</ML_IF>"), linelen;

	if (keydata == NULL || IS_EMPTY(src) || sbuf == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"Parameter is NULL or EMPTY.");
		return 1;
	}

	prev = str = (char *) src;

	divy_sbuf_create(p, &tmp_sbuf, 2048); 
	/*
	 * 文法の解析
	 * <ML_IF @xxxx@ = "rval">(TRUE_STR)<ML_ELSE>(FALSE_STR)</ML_IF>
	 *
	 * 	TRUE_STR  = @xxxx@ が0以外の時に表示される文字列
	 * 	FALSE_STR = @xxxx@ が空 or 0の時に表示される文字列
	 * 	右辺値は省略可能.
	 */
	while (1) {
		str = divy_strstr(str, "<ML_IF");
		if (str == NULL) {
			divy_sbuf_append(tmp_sbuf, prev);	/* 残り全部を足して解析終了 */
			break;
		}
		else {
			/* str の直前までをコピー */
			divy_sbuf_appendbyte(tmp_sbuf, str - prev, prev);
		}

		/* 終了タグを探す */
		end = divy_strstr(str, "</ML_IF>");
		if (end == NULL) {
			/* 終了タグがなければ、以降、文法解析は無駄である */
			divy_sbuf_append(tmp_sbuf, str);	/* 残り全部を足して解析終了 */
			break;
		}
		else {
			/* <ML_IF> から </ML_IF> までをコピーしてline に入れる*/
			first = line = apr_pcalloc(p, (end + endlen - str) + 1);
			memcpy(line, str, end + endlen - str);
			linelen = strlen(first);
		}

		line += strlen("<ML_IF");
		while (*++line == ' ');

		/* key (@xxxxx@) とのマッチング */
		for (key = keydata; key; key = key->next) {
			if (strncmp(line, key->key, key->key_len) == 0) {
				line += key->key_len;
				break;
			}
		}
		/* マッチしなければ解析しないので次へ */
		if (key == NULL) {
			divy_sbuf_append(tmp_sbuf, first);
			str += linelen; prev = str;
			continue;
		}
		if (*line == ' ') {
			while (*++line == ' ');
		}

		/* = が見つかった */
		if (*line == '=') {
			char *c;

			/* 右辺値の解析 */
			while (*++line == ' ');
			if (*line != '"') {
				divy_sbuf_append(tmp_sbuf, first);
				str += linelen; prev = str;
				continue;
			}
			line++;

			c = divy_strchr(line, '"');	/* " を探す */
			if (c == NULL) {
				divy_sbuf_append(tmp_sbuf, first);
				str += linelen; prev = str;
				continue;
			}
			/* 右辺値の取得 */
			rval = apr_pcalloc(p, c - line + 1);
			memcpy(rval, line, c - line);
			line = c;

			while (*++line == ' ');
			if (*line != '>') {
				divy_sbuf_append(tmp_sbuf, first);
				str += linelen; prev = str;
				continue;
			}
		}
		/* > が見つかった */
		if (*line == '>') {
			rval = NULL;
			line++;
		}
		else {
			divy_sbuf_append(tmp_sbuf, first);
			str += linelen; prev = str;
			continue;
		}

		/* ML_ELSE を探す */
		else_str = divy_strstr(line, "<ML_ELSE>");
		if (else_str == NULL) {
			/* line = "xxxxx</ML_IF>" */
			true_val  = line; true_val[strlen(line) - endlen] = '\0';
			false_val = "";
		}
		else {
			/* line = "xxxxx<ELSE>yyyyy</ML_IF>" */
			char *c;

			*else_str = '\0';
			true_val = line;

			false_val = else_str + strlen("<ML_ELSE>");
			c = divy_strstr(false_val, "</ML_IF>");
			if (c != NULL) *c = '\0';
		}

		/* 判定 */
		if (IS_FILLED(rval)) {
			if (IS_FILLED(key->value) && strcmp(rval, key->value) == 0) {
				divy_sbuf_append(tmp_sbuf, true_val);
			}
			else {
				divy_sbuf_append(tmp_sbuf, false_val);
			}
		}
		else {
			if (IS_FILLED(key->value) && strcmp(key->value, "0") != 0) {
				divy_sbuf_append(tmp_sbuf, true_val);
			}
			else {
				divy_sbuf_append(tmp_sbuf, false_val);
			}
		}

		str += linelen;
		prev = str;
	}
	prev = str = divy_sbuf_tostring(tmp_sbuf);

	/*
	 * 値の置換処理
	 */
	/* '@' を探す */
	while ((str = divy_strchr(str, '@')) != NULL) {
		divy_sbuf_appendbyte(sbuf, str - prev, prev);

		match_keyword = 0;
		for (key = keydata; key; key = key->next) {

			/* パラメータの切り出し */
			if (strncmp(str, key->key, key->key_len) == 0) {
				divy_sbuf_append(sbuf, key->value);
				str += key->key_len;
				match_keyword = 1;
			}
		}

		/* キーワード集合に含まれていなかった場合 */
		if (!match_keyword) {
			/* これはパラメータではないのでそのまま埋め込む */
			divy_sbuf_appendbyte(sbuf, 1, "@");
			str++;	/* 1つだけ進めて同じ位置を検索しないようにする */
		}
		prev = str;
	}
	divy_sbuf_append(sbuf, prev);

	return 0;
}

/**
 * key, value のデータを*first の末尾に詰める。
 */

static void _push_replace_param(apr_pool_t *p,
					divy_replace_key_id ikey,
					const char *value,
					divy_replace_keydata **first)
{
	divy_replace_keydata *data;
	int len = DIVY_REPLACE_KEYDATA_LEN, i;

	if (*first == NULL) {
		*first = data = apr_pcalloc(p, sizeof(divy_replace_keydata));
	}
	else {
		for (data = *first; data->next; data = data->next);
		data->next = apr_pcalloc(p, sizeof(divy_replace_keydata));
		data = data->next;
	}

	data->id  = ikey;
	data->value = value;

	/* 数値のkey -> 文字列のkey */
	for (i = 0; i < len; i++) {
		if (_replace_keydata[i].ikey == ikey) {
			data->key     = _replace_keydata[i].skey;
			data->key_len = strlen(data->key);
			break;
		}
	}

	data->next = NULL;
}

/**
 * id とlangに対応したメッセージを取得して返却する.
 *
 */
static const char * _get_svrml_string(divy_svrml_id id, const char *lang)
{
	int i, len = DIVY_SVRML_MSGS_LEN;
	const divy_svrml_msg *msg = NULL;

	for (i = 0; i < len; i++) {
		msg = &_svrml_msgs[i];
		if (id == msg->id) {
			if (IS_FILLED(lang) && strcmp(lang, "ja") == 0) {
				return msg->ja;
			}
			else {
				return msg->en;
			}
		}
	}

	return "";
}

/**
 * 指定されたメールテンプレートプレフィックス名mltemp_prefix をベースに
 * テンプレートファイルを読み、msg データを組み立てる.
 * [ msg に詰める内容 ]
 *   * body データ
 *   * サブジェクト
 *
 * @param wp apr_pool_t *
 * @param mltemp_prefix const char *
 * @param lang const char * 言語環境を表す文字列
 * @param r request_rec *
 * @param keydata divy_replace_keydata * 置換パラメータ
 * @param msg MlMessage * メールメッセージ (外でアロケートして下さい) 
 * @return int 処理ステータス(0: 成功 / 1: 失敗)
 */
static int _make_svrml_data(apr_pool_t *wp, const char *mltemp_prefix,
					const char *lang, request_rec *r,
					divy_replace_keydata *keydata, MlMessage *msg)
{
	apr_status_t rv;
	apr_file_t *fd_r    = NULL;
	divy_sbuf *sbuf = NULL;
	char *path, *data, *subject = NULL, *body = NULL;
	apr_xml_parser  *parser = NULL;
	apr_xml_doc *doc        = NULL;
	apr_xml_elem *elem;

	if (IS_EMPTY(mltemp_prefix) || IS_EMPTY(lang) || r == NULL || msg == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"The value of param were EMPTY.");
		return 1;
	}

	/* メールテンプレートファイルの読み込み */
	if (_open_svrml_template(wp, mltemp_prefix, lang, r, &path, &fd_r)) {
		return 1;
	}

	/* XML ファイルのパース */
	rv = apr_xml_parse_file(wp, &parser, &doc, fd_r, 2048);
	if (rv != APR_SUCCESS) {
		char errbuf[512] = { 0 };
		if (parser != NULL) {
			apr_xml_parser_geterror(parser, errbuf, sizeof(errbuf));

			ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse mailtemplate XML. (file = %s)"
					"Reason: (%s)", path, errbuf);
		}
		else {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to parse mailtemplate XML. (file = %s)", path);
		}
	}
	if (fd_r != NULL) (void) apr_file_close(fd_r);	/* テンプレートファイルを閉じる */

	/* XML エレメントのチェック */
	if (doc == NULL || doc->root == NULL || doc->root->first_child == NULL ||
		doc->root->first_child->first_child == NULL) {
		ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"Invalid mailtemplate XML. (file = %s)", path);
		return 1;
	}

	/* XML データの取り出し */
	for (elem = doc->root->first_child->first_child; elem != NULL; elem = elem->next) {
		data = (char *) divy_xml_get_cdata(elem, wp, 0);	/* white space は除去しない */
		if (strcmp(elem->name, "subject") == 0) {
			subject = data;
		}
		else if (strcmp(elem->name, "body") == 0) {
			body = data;
		}
		else {
			/* 無視 */
		}
	}

	/*
	 * 置換パラメータによる置換処理
	 */
	divy_sbuf_create(wp, &sbuf, 2048);

	_replace_ml_keyword(wp, subject, keydata, sbuf);
	msg->subject = apr_pstrdup(wp, divy_sbuf_tostring(sbuf));
	divy_sbuf_clear(sbuf);

	_replace_ml_keyword(wp, body, keydata, sbuf);
	msg->body = apr_pstrdup(wp, divy_sbuf_tostring(sbuf));

	return 0;
}

static int _open_svrml_template(apr_pool_t *wp, const char *mltemp_prefix,
					const char *lang, request_rec *r, char **path, apr_file_t **fd_r)
{
	apr_status_t rv;
	char *name;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	*fd_r = NULL;
	*path = NULL;

	if (IS_EMPTY(mltemp_prefix) || IS_EMPTY(lang)) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"There were EMPTY value.");
		return 1;
	}

	/*
	 * メールテンプレート適用の優先順位
	 *
	 * 1. (ロケーション名).(mltemp_prefixの値).(langの値)
	 * 2. (mltemp_prefixの値).(langの値) を利用する
	 */
	name = apr_psprintf(wp, "%s.%s.%s", dav_divy_get_root_uri(r), mltemp_prefix, lang);
	*path = ap_make_full_path(wp, dconf->mltemplateroot, name);

	/* (1) */
	rv = apr_file_open(fd_r, *path, APR_READ | APR_BINARY, 0, wp);
	if (rv == APR_ENOENT) {
		if (*fd_r != NULL) (void) apr_file_close(*fd_r);
		name = apr_psprintf(wp, "%s.%s", mltemp_prefix, lang);
		*path = ap_make_full_path(wp, dconf->mltemplateroot, name);

		/* (2) */
		rv = apr_file_open(fd_r, *path, APR_READ | APR_BINARY, 0, wp);
		if (rv == APR_ENOENT) {
			ERRLOG1(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
					"The mail-template file was missing."
					"Please check this file.(filepath = %s)", *path);
		}
	}

	if (rv != APR_SUCCESS) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to open mail-template file."
			"Please check this file.(filepath = %s, code = %d)", *path, rv);
		return 1;
	}

	return 0;
}

/**
 * メール専用の言語パラメータ名を取得する.
 *
 * @param r request_rec *
 */
static const char * _get_ml_language_param(request_rec *r)
{
	const char *lang = divy_get_language_param(r);
	const char *ml_lang = apr_table_get(r->subprocess_env, DIVY_APENV_MLLANGUAGE);

	/* メール専用の言語パラメータがあればそれを優先する */
	if (IS_FILLED(ml_lang)) {
		lang = ml_lang;
	}

	return lang;
}

/**
 * 監視メールを送信する.
 * (note)
 *   関数 divy_ml_send_clientmail で定義されていない内容をここに移動しました.
 *
 * @param r request_rec *	リクエスト構造体
 * @param wp apr_pool_t * 作業用のプール
 * @param mldata divy_ml_mailwatch_data * 送信するメール情報
 * @return 処理ステータス (ML_ERROR | ML_SUCCESS)
 */
static int _send_watchmail(request_rec *r, apr_pool_t *wp, divy_ml_mailwatch_data *mldata)
{
	apr_pool_t	*p       = r->pool;
	MlDataSource	*mlds    = NULL;
	MlProperty	*mlprop  = NULL;
	MlSession	*session = NULL;
	MlMessage	*msg     = NULL;
	int		rt       = ML_SUCCESS;

	/* メールプロバイダの取得 */
	mlds = lookup_activeml_provider(r);
	if (mlds == NULL) {
		return ML_ERROR;
	}

	/* (note) サーバ主体メールで使用する場合、メールテンプレートの内容は既に構成済み
	 * 		だから、第3引数は0になります.
	 * 		クライアント主体のメールでは、メールテンプレートの内容を読まなければ
	 * 		なりません. クライアントから送信されてくるbodyfodder を無視するためです.
	 */
	rt = _xmldat2mlmsg(r, wp, mldata, &msg);
	if (rt != ML_SUCCESS) {
		return rt;
	}
	else if (mlds == NULL || msg == NULL) {
		/* 成功なのにmsg がNULLであることもありえる */
		if (rt == ML_SUCCESS) {
			return ML_SUCCESS;	/* 成功である */
		}
		else {
			return ML_ERROR;
		}
	}

	mlprop = divy_ml_createMlProperty(r, r->user);
	if (mlprop == NULL) {
		return ML_ERROR;
	}

	session = mlds->getMlSession(mlds, mlprop, p);
	if (session == NULL) {
		return ML_ERROR;
	}

	ERRLOG1(p, APLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DATA,
			"Send watch-mail. from user=%s", r->user);

	if ((rt = session->send(session, msg)) != 0) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
			"Mail transmission went wrong. Message: %s",
			session->__msg);
	}

	return rt;
}

/**
 * body データの中からエンドユーザが手入力したデータ列だけを抽出して
 * 返却する.
 *
 */
static char * _extract_inputed_body(request_rec *r, apr_pool_t *wp, char *body)
{
	char *ch, *prev = NULL;
	divy_sbuf *sbuf = NULL;
	int len = strlen(DIVY_ML_FIXED_BODYSTR);

	if (IS_EMPTY(body)) {
		return "";
	}

	ch = body;
	while ((ch = divy_strstr(ch, DIVY_ML_FIXED_BODYSTR)) != NULL) {
		prev = ch;
		ch += len;
	}

	if (prev == NULL) {
		/* prev==NULLのケースはユーザが入力したものが一件もないことを
		 * 示します。(つまりサーバメッセージだけしかない）
		 * このコードはこのケースの時にクライアントソフトが
		 * 今までのへんてこりんなメッセージをつけてきたのか、それとも
		 * 単に文字を追加して（サーバメッセージはサーバにお願いする）
		 * いるのかを評価する為に追加されました。
		 * 文字列の比較は、先頭が======という文字ならばという危険な
		 * ことをしています。(DIVY_ML_FIXED_BODYSTR2)
		 * しかし、これを行わないと/.mailに対してクライアントソフトから
		 * メールXMLを投げることが難しくなる（=====を付与しないと従来
		 * のコードでは削除されてしまうため。
		 */
		ch = body;
		if ((ch = divy_strstr(ch, DIVY_ML_FIXED_BODYSTR2)) != NULL) {
			if (ch == body) {
				return "";
			}
		}
		/* DIVY_ML_FIXED_BODYSTR2がないのでリクエストのBODYをそのまま利用する */
		return body;
	}

	divy_sbuf_create(wp, &sbuf, strlen(body));
	divy_sbuf_appendbyte(sbuf, prev - body, body);

	return divy_sbuf_tostring(sbuf);

}

/**
 * 監視メール用XMLデータmldata からメール送信用のメッセージ構造体に変換する.
 * (note)
 *   divy_ml_XML_parser の実装をこちらに移動しました.bodyとbodyfodderを
 *   サーバ主体メールとクライアント主体メールの両方で利用するためです
 *
 * @param r request_rec *
 * @param mldata divy_ml_mailwatch_data *
 * @param msg MlMessage **
 * @return int 処理ステータス (ML_XXX)
 */
static int _xmldat2mlmsg(request_rec *r, apr_pool_t *wp, divy_ml_mailwatch_data *mldata, MlMessage **msg)
{

	apr_pool_t	*p	    = r->pool;
	const char	*uid    = NULL;
	const char	*srcuri = NULL;
	const char	*dsturi = NULL;
	MlMessage	*tmp_msg    = NULL;
	MlAddress	*from_addr  = NULL;
	MlAddress   *replyto_addr = NULL;
	divy_rdbo_mailwatch *mw     = NULL;
	dav_divy_dir_conf    *conf  = dav_divy_get_dir_config(r);
	divy_rdbo_resource *srcrdb_r = NULL;
	divy_rdbo_resource *dstrdb_r = NULL;
	urilist		*list       = NULL;
	collectionInfo	*srcinfo, *dstinfo;
	char		*mime_type, *charset, *subject, *body, *bodyfodder = NULL, *filelist;
	int		priority, mnum;
	userlist	*usrlist;
	int		first_flg = 0;
	divy_replace_keydata *keydata  = NULL;
	char *datestr = NULL;
	divy_sbuf *body_buf = NULL;
	const char *val, *sOperation = NULL, *lang;

	TRACE(p);

	if (mldata == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA, "mldata is NULL");
		return ML_ERROR;
	}

	/*
	 * セキュリティチェック
	 *
	 * requestのユーザとXML内のユーザIDが異なる場合は
	 * 不正にメールXMLをPUTされた恐れがある為、エラー
	 * を出力する。
	 */
	uid = (char *) divy_get_userid(r);
	if (uid == NULL || strcmp(uid, mldata->from) != 0) {
		ERRLOG2(p, APLOG_ERR, DIVY_FST_INFO + DIVY_SST_NET,
				"A transmitting user has a problem. "
				"Mail transmission was interrupted."
				"Mismatch UserID: (loginUser = %s <=> XML From User %s)",
				REPLACE_NULL(uid), REPLACE_NULL(mldata->from));
		return ML_ERROR;
	}

	/* メッセージ構造体を作成する */
	srcinfo = apr_pcalloc(wp, sizeof(collectionInfo));
	dstinfo = apr_pcalloc(wp, sizeof(collectionInfo));

	/*
	 * メッセージ構造体作成
	 *
	 * (1) MIMEタイプとcharsetパラメータの設定 (固定値 see divy_ml.h)
	 * (2) 送信者のアドレスとディスプレイ名を取得
	 * (3) 受信者のアドレスとディスプレイ名を取得
	 * (4) 件名を設定
	 * (5) 優先度を設定 (固定値 see divy_ml.h)
	 * (6) メールの本文を設定
	 * (7) SMTPヘッダの生成
	 */

	/* (1) MIMEタイプとcharsetパラメータの設定
	 * configファイルにエンコードが指定されていなかった場合は
	 * UTF（デフォルトとする）
	 */
	mime_type = ML_MIMETYPE_TEXT_PLAIN;
	if (IS_EMPTY(conf->mailencoding)) {
		charset   = ML_CHARSET_UTF_8;
	}
	else {
		charset   = apr_pstrdup(p, conf->mailencoding);
	}

	/* メソッド名をメソッド番号に変換 */
	mnum = ap_method_number_of(mldata->method);

	/* (2) 送信者のアドレスとディスプレイ名を取得	*/
	from_addr = apr_pcalloc(wp, sizeof(MlAddress));
	if (IS_FILLED(conf->mlrepfromaddr)) {
		from_addr->addr        = apr_pstrdup(p, conf->mlrepfromaddr);
		/* 表示名を置き換えを調べる */ 
		if (IS_FILLED(conf->mlrepfromname)) {
			from_addr->displayname = apr_pstrdup(p, conf->mlrepfromname);
		}
		else {
			from_addr->displayname = REPLACE_NULL((char *)divy_get_fullname(r));
		}
		/* reply-toに実際の担当者を設定する */
		replyto_addr = apr_pcalloc(wp, sizeof(MlAddress));
		replyto_addr->addr = (char *)divy_get_mailaddr(r);
		replyto_addr->displayname = REPLACE_NULL((char *)divy_get_fullname(r));
	}
	else {
		from_addr->addr        = (char *)divy_get_mailaddr(r);
		from_addr->displayname = REPLACE_NULL((char *)divy_get_fullname(r));
	}
	from_addr->next        = NULL;

	/*
	 * FROMのユーザのメールアドレスが存在しない場合はサーバの管理者
	 * のメールアドレスを利用して送るようにします。
	 * 管理者のメールアドレスが存在しない場合はあきらめてエラーとする
	 */
	if (from_addr->addr == NULL) {
		if (conf->mailadminaddr == NULL) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to admin mail address. "
					"Please check a \"TfMlAdminaddr\" directive.");
			return ML_ERROR;
		}
		from_addr->addr = apr_pstrdup(p, conf->mailadminaddr);
		from_addr->displayname = apr_psprintf(wp,
			       	"Unknown Mail Address (%s)", from_addr->displayname);
	}

	/*
	 * メールアドレスは複数（カンマ区切り）の可能性がある為一番初めを
	 * メールアドレスとして適用します。
	 */
	from_addr->addr = divy_ml_get_first_mail_address(wp, from_addr->addr);

	/*
	 * (3) 受信者のアドレスとディスプレイ名を取得
	 */

	/*
	 * srcリソースの情報を取得
	 */
	if (IS_EMPTY(mldata->srcuri)) {
		/* 送信先のユーザをURIから取得する為に送信先のURIは
		 * メソッドに関わらず必ず存在しなければならない */
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"Failed to mail xml Parser (source uri Not Found)");
		return ML_ERROR;
	}

	srcuri = mldata->srcuri;
	srcuri = dav_divy_remove_endslash(wp, srcuri);
	ap_no2slash((char*)srcuri);
	srcinfo->uri = apr_pstrdup(p, srcuri);

	srcrdb_r = apr_pcalloc(wp, sizeof(divy_rdbo_resource));
	srcrdb_r->uri    = (char *)srcuri;
	srcrdb_r->depth  = divy_count_dirs(srcuri);
	srcrdb_r->u_spec = NULL;

	(void) divy_rdbo_get_hierarchy_property(r, srcrdb_r, 0, 0, NULL);

	/* srcuri のURLデコード */
	if ((_divy_ml_uri_parse(wp, &list, ML_SRCWATCH, srcuri, mnum) != ML_SUCCESS)) {
		return ML_ERROR;
	}

	/*
	 * dstリソースの情報を取得  (COPY, MOVE)
	 */
	if ((mnum == M_MOVE || mnum == M_COPY) && IS_FILLED(mldata->dsturi)) {

		dsturi = mldata->dsturi;
		dsturi = dav_divy_remove_endslash(p, dsturi);
		ap_no2slash((char *)dsturi);
		dstinfo->uri = apr_pstrdup(p, dsturi);

		dstrdb_r = apr_pcalloc(wp, sizeof(divy_rdbo_resource));
		dstrdb_r->uri    = (char *)dsturi;
		dstrdb_r->depth  = divy_count_dirs(dsturi);
		dstrdb_r->u_spec = NULL;

		(void) divy_rdbo_get_hierarchy_property(r, dstrdb_r, 0, 0, NULL);

		/* dsturi のURLデコード */
		if ((_divy_ml_uri_parse(wp, &list, ML_DSTWATCH, dsturi, mnum) != ML_SUCCESS)) {
			return ML_ERROR;
		}
	}

	/*
	 * 受信者を探して管理者が存在しない場合は何もしないでぬける
	 */
	if (_divy_ml_alloc_address(r, list, &usrlist) == ML_ERROR) {
		*msg = NULL;
		return ML_SUCCESS;
	}

	/*
	 * メール監視フラグを物理削除する
	 * (削除条件)
	 *    ・srcuriが監視URIと一致している
	 *    ・論理削除フラグdelflag がセットされている
	 *    ・対象srcuriのリソースが存在すれば、監視ユーザとuid が一致している
	 */
	/* メール監視フラグを取得 */
	if (divy_rdbo_get_mailwatch_property(r, srcuri, NULL, &mw, NULL)) {
		/* 続けてしまう */
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to get merged mailwatch. (uri = %s) "
				"But server continue process sending mail. "
				"Please check divy_mailwatch entry and remove disused "
				"the entry.", srcuri);
	}

	/* 論理削除フラグがセットされているか? */
	if (mw && mw->delflag) {
		divy_db_transaction_ctx *ts_ctx = NULL;

		/* トランザクションコンテキストの取得 */
		(void) divy_db_create_transaction_ctx(r, &ts_ctx);

		if (srcrdb_r->rsid == NULL || *srcrdb_r->rsid == '\0') {
			mw->usrid = NULL;	/* ユーザIDは無視する */
		}
		else {
			mw->usrid = (char *) uid;	/* ユーザIDを使って削除する */
		}
		mw->delflag = 0;	/* 物理削除を要請 */

		/* メール監視フラグエントリを物理削除 */
		if (divy_rdbo_remove_mailwatch_property(r, mw, ts_ctx)) {
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
					"Failed to remove mailwatch while sending "
					"mail operation.(uri = %s)", srcuri);
			ts_ctx->status |= DIVY_TRANS_ABORT;
			divy_db_rollback_transaction(ts_ctx);
			/* でも続ける */
		}
		else {
			divy_db_commit_transaction(ts_ctx);
		}
	}

	/* (4) 件名を設定				*/
	subject = mldata->subject;

	/* 題名がもし設定されていない場合はサーバ側で設定する */
	if (IS_EMPTY(subject))	subject = "Server Information";

    /* 題名にプレフィックス[BRAND]を付ける */
	/* 但し、すでに[があるときはプレフィックスがついていると仮定する */
	if (IS_FILLED(subject) && *subject != '[') {
			const char *subject_prefix = _choose_brand_name(r, mldata->brandname);
			subject = apr_psprintf(wp, "[%s] %s", subject_prefix, subject);
	}

	/* (5) 優先度を設定				*/
	priority  = ML_PRIORITY_NORMAL;

	/* (6) メールの本文を設定			*/
	lang = _get_ml_language_param(r);

	/* 置換パラメータによる置き換え */
	if (mnum == M_GET) {
		sOperation = _get_svrml_string(DIVY_SVRML_GET, lang);
	}
	else if (mnum == M_PUT) {
		sOperation = _get_svrml_string(DIVY_SVRML_PUT, lang);
	}
	else if (mnum == M_DELETE) {
		sOperation = _get_svrml_string(DIVY_SVRML_DELETE, lang);
	}
	else if (mnum == M_MOVE) {
		sOperation = _get_svrml_string(DIVY_SVRML_MOVE, lang);
	}
	else if (mnum == M_COPY) {
		sOperation = _get_svrml_string(DIVY_SVRML_COPY, lang);
	}
	/* 置換処理 */
	_push_replace_param(wp, DIVY_ML_REP_OPERATION_JA, sOperation, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_OPERATION_EN, sOperation, &keydata);

	val = _choose_brand_name(r, mldata->brandname);
	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

	divy_sbuf_create(wp, &body_buf, strlen(mldata->body));
	(void) _replace_ml_keyword(wp, mldata->body, keydata, body_buf);
	body = divy_sbuf_tostring(body_buf);

	/* (6.1)メールの本文のフッダを設定 */
	bodyfodder = mldata->fbody;

	/*
	 * --------------------------------------------------------------
	 * この時点でメッセージ情報は全てそろった。
	 * 送信するアドレス分のメッセージ構造体を作成する。
	 * --------------------------------------------------------------
	 */
	*msg = tmp_msg = apr_pcalloc(wp, sizeof(MlMessage));

	/* 送信者によって変更する必要のない項目 */
	/* ただし、subjectは後で内部を置換される可能性があります */
	(*msg)->subject         = apr_pstrdup(wp, subject);
	(*msg)->priority        = priority;
	(*msg)->header_encoding = ML_ENCODING_B;
	(*msg)->mime_type       = apr_pstrdup(wp, mime_type);
	(*msg)->charset         = apr_pstrdup(wp, charset);

	(*msg)->from_addr              = apr_pcalloc(wp, sizeof(MlAddress));
	(*msg)->from_addr->userid      = apr_pstrdup(wp, from_addr->userid);
	(*msg)->from_addr->addr        = apr_pstrdup(wp, from_addr->addr);
	(*msg)->from_addr->displayname = apr_pstrdup(wp, from_addr->displayname);

	/* replytoがあるとNULLにならない */
	if (replyto_addr != NULL) {
		(*msg)->replyto_addr           = apr_pcalloc(wp, sizeof(MlAddress));
		(*msg)->replyto_addr->addr     = apr_pstrdup(wp, replyto_addr->addr);
		(*msg)->replyto_addr->displayname
								 = apr_pstrdup(wp, replyto_addr->displayname);
	}

	(*msg)->body                   = apr_pstrdup(wp, body);

	/* bodyfoddr 用の置換パラメータ生成(日付) */
	divy_format_time_t(wp, dav_divy_get_now_epoch(), DIVY_TIME_STYLE_JAPANESE, &datestr);

	for (;usrlist != NULL; usrlist = usrlist->next) {

		/* 送信先メールアドレスは存在するのか? */
		if (IS_EMPTY(usrlist->address->addr)) {
			/* このユーザにはメールアドレスが無いので無視 */
			continue;
		}

		if (first_flg) {
			tmp_msg->next = apr_pcalloc(wp, sizeof(MlMessage));
			tmp_msg = tmp_msg->next;
		}
		tmp_msg->subject         = (*msg)->subject;
		tmp_msg->priority        = (*msg)->priority;
		tmp_msg->header_encoding = (*msg)->header_encoding;
		tmp_msg->mime_type       = (*msg)->mime_type;
		tmp_msg->charset         = (*msg)->charset;
		tmp_msg->from_addr       = (*msg)->from_addr;
		tmp_msg->replyto_addr    = (*msg)->replyto_addr;

		/* 送信先の設定 */
		tmp_msg->to_addr              = apr_pcalloc(wp, sizeof(MlAddress));
		tmp_msg->to_addr->userid      = apr_pstrdup(wp, usrlist->address->userid);
		tmp_msg->to_addr->addr        = apr_pstrdup(wp, usrlist->address->addr);
		tmp_msg->to_addr->displayname = apr_pstrdup(wp, usrlist->address->displayname);

		/* 複数アドレスある可能性がある為、切り分ける */
		(void)divy_ml_setMultiAddress(wp, &tmp_msg->to_addr, &tmp_msg->cc_addr);
		tmp_msg->body = (*msg)->body;

		/* bodyfodder の置換パラメータを生成 */
		keydata = NULL;
		_push_replace_param(wp, DIVY_ML_REP_GETLASTMODIFIED, datestr, &keydata);

		/* dst 情報を持たない = MOVE, COPY 以外の場合 */
		if (dstinfo != NULL && srcinfo != NULL && dstinfo->uri == NULL) {

			if (usrlist->watchplace & ML_SRCWATCH) {
				_push_replace_param(wp, DIVY_ML_REP_WATCHFOLDER,     _get_ml_watchfolder(r, srcrdb_r), &keydata);
				_push_replace_param(wp, DIVY_ML_REP_SRC_WATCHFOLDER, "", &keydata);
				_push_replace_param(wp, DIVY_ML_REP_DST_WATCHFOLDER, "", &keydata);
			}
			/* ticket(src) */
			if (conf->mloneclickurl == DIVY_MLONECLICKURL_ON) {
				_push_replace_param(wp, DIVY_ML_REP_TICKETURL,     _get_oneclick_url(r, srcrdb_r), &keydata);
				_push_replace_param(wp, DIVY_ML_REP_SRC_TICKETURL, "", &keydata);
				_push_replace_param(wp, DIVY_ML_REP_DST_TICKETURL, "", &keydata);
			}
		}
		/* COPY, MOVE の場合 (srcとdstを共に持つ場合) */
		else if (dstinfo != NULL && srcinfo != NULL) {

			/* srcとdst両方にアクセスできるユーザの場合 */
			if ((usrlist->watchplace & ML_SRCWATCH) && (usrlist->watchplace & ML_DSTWATCH)) {

				_push_replace_param(wp, DIVY_ML_REP_WATCHFOLDER, "", &keydata);
				_push_replace_param(wp, DIVY_ML_REP_SRC_WATCHFOLDER, _get_ml_watchfolder(r, srcrdb_r), &keydata);
				_push_replace_param(wp, DIVY_ML_REP_DST_WATCHFOLDER, _get_ml_watchfolder(r, dstrdb_r), &keydata);

				/* ticket (src, dst) */
				if (conf->mloneclickurl == DIVY_MLONECLICKURL_ON) {
					_push_replace_param(wp, DIVY_ML_REP_TICKETURL,     "", &keydata);
					_push_replace_param(wp, DIVY_ML_REP_SRC_TICKETURL, _get_oneclick_url(r, srcrdb_r), &keydata);
					_push_replace_param(wp, DIVY_ML_REP_DST_TICKETURL, _get_oneclick_url(r, dstrdb_r), &keydata);
				}
			}
			/* dst のみにアクセスできるユーザの場合 */
			else if (usrlist->watchplace & ML_DSTWATCH) {
				_push_replace_param(wp, DIVY_ML_REP_WATCHFOLDER,     "", &keydata);
				_push_replace_param(wp, DIVY_ML_REP_SRC_WATCHFOLDER, "**********", &keydata);
				_push_replace_param(wp, DIVY_ML_REP_DST_WATCHFOLDER, _get_ml_watchfolder(r, dstrdb_r), &keydata);

				/* ticket (---, dst) */
				if (conf->mloneclickurl == DIVY_MLONECLICKURL_ON) {
					_push_replace_param(wp, DIVY_ML_REP_TICKETURL,     "", &keydata);
					_push_replace_param(wp, DIVY_ML_REP_SRC_TICKETURL, "", &keydata);
					_push_replace_param(wp, DIVY_ML_REP_DST_TICKETURL, _get_oneclick_url(r, dstrdb_r), &keydata);
				}
			}
			/* src のみにアクセスできるユーザの場合 */
			else if (usrlist->watchplace & ML_SRCWATCH) {
				_push_replace_param(wp, DIVY_ML_REP_WATCHFOLDER,     "", &keydata);
				_push_replace_param(wp, DIVY_ML_REP_SRC_WATCHFOLDER, _get_ml_watchfolder(r, srcrdb_r), &keydata);
				_push_replace_param(wp, DIVY_ML_REP_DST_WATCHFOLDER, "**********", &keydata);

				/* ticket (src, ---) */
				if (conf->mloneclickurl == DIVY_MLONECLICKURL_ON) {
					_push_replace_param(wp, DIVY_ML_REP_TICKETURL,     "", &keydata);
					_push_replace_param(wp, DIVY_ML_REP_SRC_TICKETURL, _get_oneclick_url(r, srcrdb_r), &keydata);
					_push_replace_param(wp, DIVY_ML_REP_DST_TICKETURL, "", &keydata);
				}
			}
			else { /* 何もしない */
			}
		}
		else { /* 何もしない */
		}

		/* 置換パラメータの置き換え処理 */
		if (keydata != NULL) {
			divy_sbuf *fbody_buf = NULL;
			divy_sbuf_create(wp, &fbody_buf, strlen(bodyfodder));

			(void) _replace_ml_keyword(wp, bodyfodder, keydata, fbody_buf);

			tmp_msg->bodyfodder = divy_sbuf_tostring(fbody_buf);
			divy_sbuf *sbuf = NULL;
			divy_sbuf_create(wp, &sbuf, 2048);

			(void) _replace_ml_keyword(wp, subject, keydata, sbuf);
			tmp_msg->subject = divy_sbuf_tostring(sbuf);
		}

		first_flg = 1;
	}

	/*
	 * msgを全てエンコードかける
	 * ここの成否で以降のエンコードが決まります。
	 */
	if (strcasecmp((*msg)->charset, ML_CHARSET_UTF_8) != 0 &&
		strcasecmp((*msg)->charset, ML_CHARSET_UTF8) != 0) {

		if (divy_ml_encode_msg(wp, *msg) != ML_SUCCESS) {
			/*
			 * エンコードがエラーとなった場合は問題が発生した証拠
			 * UTF8で以降処理を行う。
			 */
			ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"Failed to encode. Processing is continued by UTF8. (%s -> UTF-8)",
					(*msg)->charset);

			(*msg)->charset = ML_CHARSET_UTF_8;
			/*
			 * 再度UTF8でエンコードの処理を行なう
			 * エラーが発生しても続ける。
			 */
			divy_ml_encode_msg(p, *msg);
		}
	}
	else {
		/* UTF-8のエンコード */
		(*msg)->charset = ML_CHARSET_UTF_8;
		divy_ml_encode_msg(p, *msg);
		/* 失敗しても無視 */
	}

	/* ファイルの一覧を作成する */
	filelist = mldata->filelist;

	/* ファイルリストをすべてのメッセージへ適用する */
	if (IS_FILLED(filelist)) {
		for (tmp_msg = *msg; tmp_msg != NULL; tmp_msg = tmp_msg->next) {
			tmp_msg->body = apr_psprintf(wp, "%s%s", tmp_msg->body, filelist);
		}
	}
	
	/* (7) SMTPヘッダの生成 */
	if (divy_ml_setSMTPHeader(p, *msg) != 0) return ML_ERROR;

	return ML_SUCCESS;
}

/*
 * リクエストがRangeの場合に１を返す
 * メール専用のため、Rangeの場合でも以下のケースは0を返す（つまりRangeとはみなさない)
 * 
 * Range=bytes=0-100はRangeである
 * Range=bytes=101-200などのように始まりが0以外の場合はRangeとは言わない
 *
 * @param r request_rec *
 * @return int 1: RangeGETである / 0: 違う
 */
static int _is_range_start(request_rec *r) {

	const char *range = NULL;

	if (M_GET != divy_get_method_number(r)) return 1; /* GET以外はチェック不要 */

	if (!(range = apr_table_get(r->headers_in, "Range"))) {
		range = apr_table_get(r->headers_in, "Request-Range");
	}

	if (range && strncasecmp(range, "bytes=0-", 8) != 0) {
		/* 先頭取得0-出ない場合は対象外 */
		return 0;	/* rangeとしない */
	}

	return 1;
}

/*
 * 複数あるブランド名のどれを利用するのか選択する
 *
 * @param r request_rec *
 * @return const char * ブランド名文字列
 *
 */
static const char * _choose_brand_name(request_rec *r, const char *msgbrandname)
{
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	/* メールのメッセージにBRAND名が入っている場合は最優先利用 */
	if (IS_FILLED(msgbrandname)) return msgbrandname;
	return (IS_FILLED(sconf->brandname)) ? sconf->brandname : DIVY_PRODUCT_NAME;
}

/*
 * ユーザ情報の値を詰める
 * @param p apr_pool_t *
 * @param keydata divy_replace_keydata *
 */
static void _build_user_notification_body(request_rec *r, apr_pool_t *wp,
						divy_replace_keydata *keydata, divy_rdbo_usr *usr_pr) 
{
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);
	int support_grpconstraints = divy_support_grpconstraints(r);
	int support_groupleader    = divy_support_groupleader(r);
	const divy_rdbo_extstatus *own_extstatus = divy_get_extstatus(r);
	const char *val, *lang, *root_uri, *uri_part;
	char *datestr;

	if (r == NULL || wp == NULL || usr_pr == NULL || keydata == NULL) 
		return ;

	/* 送信言語の取得 */
	lang = _get_ml_language_param(r);

	_push_replace_param(wp, DIVY_ML_REP_USERNAME, usr_pr->fullname, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_USERID,   usr_pr->usrid, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_PASSWORD, usr_pr->password, &keydata);
	_push_replace_param(wp, DIVY_ML_REP_MAILADDR, usr_pr->mailaddr, &keydata);

	if (usr_pr->maxst == DIVY_UNLIMITED_USRQUOTA_BYTES) {
		val = "Unlimited";
	}
	else {
		val = divy_byte2displaysize(wp, usr_pr->maxst, 0);
	}
	_push_replace_param(wp, DIVY_ML_REP_LIMITSIZE, val, &keydata);

	if (usr_pr->maxres == DIVY_UNLIMITED_USRQUOTA_FILES) {
		val = "Unlimited";
	}
	else {
		val = divy_i2displaynum(wp, usr_pr->maxres);
	}
	_push_replace_param(wp, DIVY_ML_REP_LIMITCOUNT, val, &keydata);

	(void) divy_format_time_t(wp, dav_divy_get_now_epoch(), DIVY_TIME_STYLE_JAPANESE, &datestr);
	_push_replace_param(wp, DIVY_ML_REP_REGDAY, datestr, &keydata);
	if (IS_FILLED(usr_pr->comment)) {
		/* 先頭がハイフンのコメントは置換対象にしない */
		if (strncmp(usr_pr->comment, "-", 1) != 0) {
			_push_replace_param(wp, DIVY_ML_REP_USERCOMMENT,
												usr_pr->comment, &keydata);
		}
		else {
			/* 空白にする */
			_push_replace_param(wp, DIVY_ML_REP_USERCOMMENT, "", &keydata);
		}
	}
	else {
			/* 空白にする */
			_push_replace_param(wp, DIVY_ML_REP_USERCOMMENT, "", &keydata);
	}

	/* ユーザの種類 */
	if (usr_pr->adminmode == DIVY_ADMINMODE_ADMIN) {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_USERTYPE_ADMIN, lang);
	}
	else {
		if (divy_rdbo_is_groupleader(usr_pr->extstatus)) {
			val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_USERTYPE_GROUPLEADER, lang);
		}
		else if (divy_rdbo_is_trusty_user(usr_pr->extstatus)) {
			val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_USERTYPE_NORMAL, lang);
		}
		else {
			val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_USERTYPE_LIMITED, lang);
		}
	}
	_push_replace_param(wp, DIVY_ML_REP_USERTYPE, val, &keydata);

	/* ユーザの権限 */
	if (divy_rdbo_has_readwrite_privilege(usr_pr->extstatus)) {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_PRIVILEGE_READWRITE, lang);
	}
	else if (divy_rdbo_has_upload_privilege(usr_pr->extstatus)) {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_PRIVILEGE_UPLOAD, lang);
	}
	else {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_PRIVILEGE_READ, lang);
	}
	_push_replace_param(wp, DIVY_ML_REP_USERPRIVILEGE, val, &keydata);

	/* リソース公開権限 */
	if (divy_rdbo_has_setview_privilege(usr_pr->extstatus)) {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_PRIVILEGE_PUBLISHER, lang);
	}
	else {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_PRIVILEGE_NOPUBLISHER, lang);
	}
	_push_replace_param(wp, DIVY_ML_REP_PUBLISHERPRIVILEGE, val, &keydata);

	/* グループ制約を無視できるかどうか */
	if (support_grpconstraints) {
		/* グループ制約を無視できるユーザ or 管理者の場合 */
		if (divy_rdbo_has_user_groupconstraints_ignore(usr_pr->extstatus) ||
			usr_pr->adminmode == DIVY_ADMINMODE_ADMIN) {
			val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_PRIVILEGE_GRPCONSTRAINTSIGNORE, lang);
		}
		else {
			val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_PRIVILEGE_GRPCONSTRAINTSAPPLY, lang);
		}
	}
	else {
		/* グループ制約がサポートされていなければ空文字 */
		val = "";
	}
	_push_replace_param(wp, DIVY_ML_REP_GRPCONSTRAINTSIGNORE, val, &keydata);

	/* ユーザの状態 */
	if (divy_rdbo_is_active_user(usr_pr->extstatus)) {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_STATE_ACTIVE, lang);
	}
	else {
		val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_STATE_INACTIVE, lang);
	}
	if (usr_pr->expiration != 0 && usr_pr->expiration < apr_time_sec(r->request_time)) {
		val = apr_pstrcat(wp, val, ", ",
				_get_svrml_string(DIVY_SVRML_NOTIFYCL_STATE_EXPIRE, lang), NULL);
	}
	_push_replace_param(wp, DIVY_ML_REP_USERSTATE, val, &keydata);

	/* ユーザの有効期限 */
	if (usr_pr->expiration > 0) {
		(void) divy_format_time_t(wp, usr_pr->expiration,
					  DIVY_TIME_STYLE_JAPANESE, &datestr);
	}
	else {
		datestr = "0";
	}
	_push_replace_param(wp, DIVY_ML_REP_USEREXPIRATION, datestr, &keydata);

	/* グループ管理者機能サポートの有無 */
	if (support_groupleader) {
		_push_replace_param(wp, DIVY_ML_REP_SUPPORTED_LEADER, "1", &keydata);
	}
	else {
		_push_replace_param(wp, DIVY_ML_REP_SUPPORTED_LEADER, "0", &keydata);
	}

	/* オーナ名 */
	val = (support_groupleader) ? usr_pr->ownername : "";
	_push_replace_param(wp, DIVY_ML_REP_USEREOWNERNAME, val, &keydata);

	/* 作成可能ユーザ数 */
	if (support_groupleader) {
		/* グループリーダの場合だけ意味を持ちます */
		if (divy_rdbo_is_groupleader(usr_pr->extstatus)) {
			if (usr_pr->maxusercreation == 0) {
				/* 0 は無制限を意味します */
				val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_UNLIMITED_USER_CREATION, lang);
			}
			else {
				val = apr_psprintf(wp, "%d", usr_pr->maxusercreation);
			}
		}
		else {
			val = "";
		}
	}
	else {
		val = "";	/* 未サポートの場合 */
	}
	_push_replace_param(wp, DIVY_ML_REP_MAXUSERCREATION, val, &keydata);

	/* Other ユーザの管理 */
	if (support_groupleader) {
		if (divy_rdbo_has_control_otheruser(usr_pr->extstatus)) {
			val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_ALLOW, lang);
		}
		else {
			val = _get_svrml_string(DIVY_SVRML_NOTIFYCL_NOT_ALLOW, lang);
		}
	}
	else {
		val = "";
	}
	_push_replace_param(wp, DIVY_ML_REP_CONTROLOTHERUSER, val, &keydata);

	/* グループリーダかどうか */
	if (support_groupleader && divy_rdbo_is_groupleader(own_extstatus)) {
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "1", &keydata);
	}
	else {
		_push_replace_param(wp, DIVY_ML_REP_HAS_GROUPLEADER, "0", &keydata);
	}

	/* Location を算出する */
	root_uri = dav_divy_get_root_uri(r);
	uri_part  = dav_divy_escape_uri(wp, (const char *)
				divy_build_u_update_uri(wp, root_uri));
	_push_replace_param(wp, DIVY_ML_REP_LOCATIONUTIL, divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, uri_part, r, NULL), &keydata);

	/* $schema://$root/ を算出する */
	val =  divy_construct_url2(wp, DIVY_URLTYPE_PUBLIC, root_uri, r, NULL);
	_push_replace_param(wp, DIVY_ML_REP_ROTURL, apr_psprintf(wp, "%s/", val), &keydata);

	if (IS_FILLED(sconf->brandname)) {
		val = sconf->brandname;
	}
	else {
		val = DIVY_PRODUCT_NAME;
	}

	_push_replace_param(wp, DIVY_ML_REP_BRANDNAME, val, &keydata);

}
