/**
 * $Id$
 *
 * util_saml.c
 *
 * SAMLユーティリティ関数
 *
 */

#include "apr_xml.h"
#include "apr_strings.h"
#include "apr_uuid.h"
#include "apr_base64.h"

#include "httpd.h"

#include "libxml/tree.h"
#include "libxml/parser.h"
#include "xmlsec/xmlsec.h"
#include "xmlsec/xmltree.h"
#include "xmlsec/xmldsig.h"
#include "xmlsec/templates.h"
#include "xmlsec/crypto.h"

#include "zlib.h"

#include "liveprop.h"
#include "util.h"
#include "util_saml.h"
#include "tf_logger.h"
#include "tf_valuecache.h"
#include "tf_extmap.h"

/*--------------------------------------------------------------
  Define fixed values and macro
  --------------------------------------------------------------*/
APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Declare and Define structure
  ----------------------------------------------------------------------------*/

/*--------------------------------------------------------------
  Declare  private functions
  --------------------------------------------------------------*/
static int divy_saml_build_samlrequest(request_rec *r, char **samlrequest);
static int divy_saml_build_entity_descriptor(apr_pool_t *pool, apr_xml_elem *elem, divy_saml_idp_metadata **meta);
static int divy_saml_get_xml_attribute(apr_pool_t *pool, apr_xml_elem *elem, const char *key, char **value);
static int divy_saml_set_binding_location(apr_pool_t *pool, int issignon, apr_xml_elem *elem, const char* urn, const char* location, divy_saml_idp_metadata **meta);
static int divy_saml_parse_xml_signon_logout(apr_pool_t *pool, apr_xml_elem *elem, divy_saml_idp_metadata **meta);
static int divy_saml_set_x509_certificate(apr_pool_t *pool, apr_xml_elem *elem, divy_saml_idp_metadata **meta);
static int divy_saml_xml_deflate(apr_pool_t *pool, const char *src, int srclen,
													char **dst, int *dstlen);
static void _divy_saml_parse_assertion_element(apr_pool_t *pool,
							apr_xml_elem *elem, divy_saml_assertion *assertion);

/*------------------------------------------------------------------------------
  Define Public Functions
  ----------------------------------------------------------------------------*/

/**
 * IdPメタデータを開いて構造体に詰める
 */
DIVY_DECLARE(int)
divy_saml_open_metadata(apr_pool_t* pool, const char* filepath, divy_saml_idp_metadata **idpmeta)
{
	apr_status_t rv;
	apr_file_t *fd = NULL;
	apr_xml_parser  *parser;
	apr_xml_doc	*doc;
	apr_xml_elem *root_elem;
	apr_xml_elem *e;
	divy_saml_idp_metadata *tmp_meta;

	rv = apr_file_open(&fd, filepath, APR_READ, APR_OS_DEFAULT, pool);
	if (rv == APR_ENOENT) {
		ERRLOG1(pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_DATA,
				"The SAML META data file was missing. "
				"( filepath = %s )", filepath);
		return 1;
	}
	else if (rv != APR_SUCCESS) {
		ERRLOG2(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_FS,
				"Failed to open SAML META data."
				"( filepath = %s, code = %d )" , filepath, rv);
		return 1;
	}

	/* XML ファイルのパース */
	rv = apr_xml_parse_file(pool, &parser, &doc, fd, 2048);
	if (rv != APR_SUCCESS || doc == NULL) {

		/* 512 バイト以上のメッセージは切捨て。。*/
		/* parserがエラーの場合でparserがNULLの場合がある */
		char *errbuf = apr_pcalloc(pool, 512);
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"Failed to XML Parser: Reason: (%s)",
				(parser == NULL) ? "XML Parser Error." :
					apr_xml_parser_geterror(parser, errbuf, 512));
		return 1;
	}

	/*
	 * XML内部をチェックして必要な項目を抜き取る
	 */
	root_elem = doc->root;
	if (root_elem == NULL) {
		ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"Failed to XML Parser.");
		return 1;
	}

	 *idpmeta = apr_pcalloc(pool, sizeof(divy_saml_idp_metadata));
	(*idpmeta)->signon = apr_pcalloc(pool, sizeof(divy_saml_idp_binding));
	(*idpmeta)->logout = apr_pcalloc(pool, sizeof(divy_saml_idp_binding));

	/* IdPのメタデータを記録 */
	(*idpmeta)->filepath = filepath;

	/* メタデーターを解析して構造体に詰め込みます */
	if ((strcmp(root_elem->name, "EntitiesDescriptor") == 0)) {
		/* 複数のEntityDescriptor の場合 */
		for (e = root_elem->first_child; e != NULL; e = e->next) {
			if (divy_saml_build_entity_descriptor(pool, e, idpmeta)) {
				return 1;
			}
		}
	}
	else if ((strcmp(root_elem->name, "EntityDescriptor") == 0)) {
		/* 一件のEntityDescriptor の場合 */
		if (divy_saml_build_entity_descriptor(pool, root_elem, idpmeta)) {
			return 1;
		}
	}
	else {
		/* parse error */
		ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
				"EntitiesDescriptor or EntityDescriptor not found");
		return 1;
	}

	/**
	 * IdPデーターの内容確認
	 * 必要最小限あるべきデータを調べる
	 */
	if ((*idpmeta)->signon == NULL || IS_EMPTY((*idpmeta)->entityID)) {
		return 1;
	}

	/* 利用すべきBINDINGを決定する */
	for (tmp_meta = *idpmeta; tmp_meta != NULL; tmp_meta = tmp_meta->next) {

		if (IS_EMPTY(tmp_meta->entityID)) {
			continue;
		}

		if (IS_FILLED(tmp_meta->signon->http_post)) {
			tmp_meta->choose_binding_type = SAML_URN_BINDING_POST;
		}
		else if (IS_FILLED(tmp_meta->signon->http_redirect)) {
			tmp_meta->choose_binding_type = SAML_URN_BINDING_REDIRECT;
		}
		else {
			/* ここに来てはSAMLそのものが動かない為エラーとする */
			tmp_meta->choose_binding_type = SAML_URN_BINDING_NONE;

			ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
					"IdP ``Binding Location`` not found. "
					"Please check IdP meta data. [file=%s]", filepath);

			return 1;
		}
	}

	return 0;

}

/**
 * SAML認証画面で利用する独自XMLデータを作成する
 */
DIVY_DECLARE(int)
divy_saml_build_autoindex_xml(request_rec *r, divy_samlinfo_iscreen* screen, char **xml)
{
	apr_pool_t *pool = r->pool;
	divy_sbuf *sbuf             = NULL;
	dav_divy_server_conf *sconf;
	dav_divy_dir_conf *dconf;
	char *brandname;
	const divy_cmap_elem *map_e = NULL;
	apr_hash_t *mime_map  = NULL;
	char *token = NULL;
	divy_saml_tfuser *usr_pr = NULL;
	int useSAML = divy_support_saml(r);
	const char* str = NULL;

	dconf = dav_divy_get_dir_config(r);
	sconf = dav_divy_get_server_config(r->server);

	TRACE(pool);

	/* 拡張子、Content-Typeマップを読み込む */
	mime_map = divy_read_mimetype_map(r);

	divy_sbuf_create(pool, &sbuf, 8192);
	divy_sbuf_append(sbuf,
		DAV_XML_HEADER CRLF
		"<?xml-stylesheet type=\"text/xsl\"?>"CRLF
		"<"DIVY_NS_PREFIX":samldiscovery "
		"xmlns:TF=\""DIVY_NS_DEFAULT_URI"\">"CRLF);

	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":pageinfo>" CRLF);

	/* brandname を付加 */
	if (IS_EMPTY(sconf->brandname)) {
		/* デフォルトのプロダクト名を入れる */
		brandname = apr_pstrdup(pool, DIVY_PRODUCT_NAME);
	}
	else {
		brandname = apr_pstrdup(pool, sconf->brandname);
	}

	divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":brandname>");
	divy_sbuf_append(sbuf,	dav_divy_escape_xmlstr(pool, brandname,
													DIVY_XML_T2T_QUOTE));
	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":brandname>");

	/* ブランド画像 */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":brandimg>");
	str = divy_get_custom_image_path(r, "logo.png");
	if (IS_EMPTY(str)) {
		map_e = divy_get_extmapinfo(pool, mime_map,
									"application/x-tf-toppage", NULL);
		if (map_e != NULL && IS_FILLED(map_e->img_path)) {
			str = map_e->img_path;
		}
	}
	if (IS_FILLED(str)) {
		divy_sbuf_append(sbuf,
				dav_divy_escape_xmlstr(pool, (char*)str, DIVY_XML_T2T_QUOTE));
	}
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":brandimg>");

	/* ロケーションURI */
	divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":locationuri>");
	divy_sbuf_append(sbuf, dav_divy_escape_uri(pool, dav_divy_get_root_uri(r)));
	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":locationuri>");

	/* ブラウザスタイル */
	divy_sbuf_append(sbuf,	"<"DIVY_NS_PREFIX":browserstyle>");
	if (screen->browserstyle & DIVY_BROWSER_STYLE_INVISIBLE_BRAND) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":invisiblebrand/>");
	}
	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":browserstyle>");

	/* 表示言語 */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":lang>");
	divy_sbuf_append(sbuf,	divy_get_language_param(r));
	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":lang>");

	/* ユニークID */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":uniqueid>");
	divy_sbuf_append(sbuf, screen->uniqueid);
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":uniqueid>");

	/* TeamFileバージョン */
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":tfversion>");
	divy_sbuf_append(sbuf, dav_divy_escape_xmlstr(pool,
								(char*)sconf->tfversion, DIVY_XML_T2T_QUOTE));
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":tfversion>");

	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":pageinfo>" CRLF);

	/* SAML情報 */

	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":samlinfo>");

	/* SAMLを利用するか?*/
	divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":useSAML>");
	divy_sbuf_append(sbuf, useSAML?"on":"off");
	divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":useSAML>");

	if (useSAML) {

		/* POST先 */
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":actionurl>");
		divy_sbuf_append(sbuf, screen->actionurl);
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":actionurl>");

		/* SAMLRequest */
		if (IS_EMPTY(screen->samlxml)) {
			if (divy_saml_build_samlrequest(r, &screen->samlxml)) {
				/* TODO エラー */
				/* ここで終わらせないとね */
			}
		}
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":samlrequest>");
		divy_sbuf_append(sbuf, screen->samlxml);
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":samlrequest>");

		/* Relay State */
		if (IS_EMPTY(screen->relaystate)) {
			if (divy_saml_save_relaystate(r, screen->next, 60 * 5, &token)) {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":relaystate/>");
			}
			else {
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":relaystate>");
				divy_sbuf_append(sbuf, token);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":relaystate>");
			}
		}
		else {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":relaystate>");
			divy_sbuf_append(sbuf, screen->relaystate);
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":relaystate>");
		}

		/* 署名 */
		if (dconf->idpmeta->wantauthnreqsigned
				&& IS_FILLED(dconf->saml_x509certificatepath)) {
			/* 証明書 */
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":signature>");
			divy_sbuf_append(sbuf, screen->x509path);
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":signature>");

			/* 署名アルゴリズム RSA-SHA1 固定 */
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":sigalg>");
			divy_sbuf_append(sbuf, "http://www.w3.org/2000/09/xmldsig#rsa-sha1");
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":sigalg>");
		}
		if (screen->usr_pr != NULL) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":tfuserlist>");

			for (usr_pr = screen->usr_pr; usr_pr != NULL;
													usr_pr = usr_pr->next) {
				divy_sbuf_append(sbuf, apr_psprintf(pool,
						"<%s:user id=\"%s\">", DIVY_NS_PREFIX, usr_pr->usrid));
				divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":fullname>");
				divy_sbuf_append(sbuf, usr_pr->fullname);
				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":fullname>");

				divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":user>");
			}

			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":tfuserlist>");
		}
		else {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":tfuserlist/>");
		}

		/* selectedtfuser */
		if (IS_FILLED(screen->selectedtfuser)) {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":selectedtfuser>");
			divy_sbuf_append(sbuf, screen->selectedtfuser);
			divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":selectedtfuser>");
		}
		else {
			divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":selectedtfuser/>");
		}
	}

	/* status code */
	if (IS_FILLED(screen->statuscode)) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":statuscode>");
		divy_sbuf_append(sbuf, screen->statuscode);
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":statuscode>");
	}

	/* message */
	if (IS_FILLED(screen->msg)) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":msg>");
		divy_sbuf_append(sbuf, screen->msg);
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":msg>");
	}

	/* token */
	if (IS_FILLED(screen->token)) {
		divy_sbuf_append(sbuf, "<"DIVY_NS_PREFIX":token>");
		divy_sbuf_append(sbuf, screen->token);
		divy_sbuf_append(sbuf, "</"DIVY_NS_PREFIX":token>");
	}

	divy_sbuf_append(sbuf,	"</"DIVY_NS_PREFIX":samlinfo>"
							"</"DIVY_NS_PREFIX":samldiscovery>");

	*xml = divy_sbuf_tostring(sbuf);
	/* debug */
	ERRLOG1(pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"xml = %s" , *xml);

	return 0;
}

/**
 * SAMLResponseデータをパースする
 */
DIVY_DECLARE(int)
divy_saml_parse_samlresponse(apr_pool_t *pool, const char* xml, divy_saml_assertion **data)
{
	apr_status_t rv;
	char *samlresponse = NULL;
	int base64_len = 0;
	apr_xml_parser *parser = NULL;
	apr_xml_doc *doc =NULL;
	apr_xml_elem *elem, *e;
	char *line;

	divy_saml_assertion* assertion;

	TRACE(pool);

	samlresponse = apr_pcalloc(pool,
							sizeof(char) * (apr_base64_decode_len(xml) + 1));

	base64_len = apr_base64_decode(samlresponse, xml);
	samlresponse[base64_len + 1] = '\0';

	/* debug */
	ERRLOG1(pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
			"SAMLResponse = %s" , samlresponse);
	
	parser = apr_xml_parser_create(pool);
	rv = apr_xml_parser_feed(parser, samlresponse, base64_len);
	if (rv != APR_SUCCESS) {
		line = apr_pcalloc(pool, 512);
		apr_xml_parser_geterror(parser, line, 512);
		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
							"error XML document for SAML Response %s", line);
		return 1;
	}
	apr_xml_parser_done(parser, &doc);

	/* XMLのタグを解析する */
	if (dav_validate_root(doc, "Response")) {
		/* TODO エラーメッセージを */
		return 1;
	}

	assertion = apr_pcalloc(pool, sizeof(divy_saml_assertion));
	assertion->xml = apr_pstrdup(pool, samlresponse);

	for (elem = doc->root->first_child; elem != NULL; elem = elem->next) {

		/* Issuer */
		if ((strcmp(elem->name, "Issuer")) == 0) {
			assertion->issuer = (char*)divy_xml_get_cdata(elem, pool, 1);
			ERRLOG1(pool, APLOG_DEBUG, DIVY_FST_IERR + DIVY_SST_DEBUG,
											"issuer = %s", assertion->issuer);
		}

		/* Status */
		if ((strcmp(elem->name, "Status")) == 0) {
			e = elem->first_child;
			if (e != NULL && (strcmp(e->name, "StatusCode")) == 0) {
				divy_saml_get_xml_attribute(pool, e, "Value", &assertion->status);
				/* debug */
				ERRLOG1(pool, APLOG_DEBUG, DIVY_FST_IERR + DIVY_SST_DEBUG,
											"status = %s", assertion->status);
			}
			else {
				/* TODO statusがあるのに中身がない.エラーにしないと */
				/* TODO エラーメッセージを */
				return 1;
			}
		}

		/* Assertion */
		if ((strcmp(elem->name, "Assertion")) == 0) {
			/* debug */
			ERRLOG0(pool, APLOG_DEBUG, DIVY_FST_IERR + DIVY_SST_DEBUG, "assertion");
			(void)_divy_saml_parse_assertion_element(pool, elem, assertion);
		}
	}

	/* 解析したデータを渡す */
	*data = assertion;

	return 0;
}

/**
 * SAMLResponseの内容をチェックします
 *
 */
DIVY_DECLARE(dav_error *)
divy_saml_validate_samlresponse(request_rec *r, divy_saml_assertion *assertion, const char* token)
{

	apr_pool_t *pool = r->pool;

	time_t comp = 0;
	time_t now = 0;
	now = dav_divy_get_now_epoch();

	TRACE(pool);

	/* SAMLステータスがSuccessではない場合はエラー */
	if (IS_FILLED(assertion->status)) {
		if (strcmp(assertion->status, SAML_STATUS_SUCCESS) != 0) {

			ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"IdP Result Error status = %s", assertion->status);

			apr_table_setn(r->headers_out, "Location",
					apr_psprintf(pool, "%s?token=%s&st=401",
						divy_build_login_uri(pool,
							dav_divy_get_root_uri(r)), token));
			return dav_new_error(pool, HTTP_MOVED_TEMPORARILY, 0, 0, "");
		}
	}

	/* SubjectのNotOnOrAfterが本日時間より古ければエラー */
	comp = dav_divy_iso8601totime_t(pool, assertion->subject->notonorafter);
	if (comp < now) {

		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"SAML Response assertion error. expired subject %s",
				assertion->subject->notonorafter );

		apr_table_setn(r->headers_out, "Location",
				apr_psprintf(pool, "%s?st=408",
					divy_build_login_uri(pool, dav_divy_get_root_uri(r))));
		return dav_new_error(pool, HTTP_MOVED_TEMPORARILY, 0, 0, "");
	}

	/* Conditions のNotBeforeが本日時間より新しいならエラー */
	comp = dav_divy_iso8601totime_t(pool, assertion->cond->notbefore);
	if (comp > now) {
		apr_table_setn(r->headers_out, "Location",
				apr_psprintf(pool, "%s?st=408",
					divy_build_login_uri(pool, dav_divy_get_root_uri(r))));
		return dav_new_error(pool, HTTP_MOVED_TEMPORARILY, 0, 0, "");
	}

	/* Conditions のNotOnOrAfterが本日時間より古いならエラー */
	comp = dav_divy_iso8601totime_t(pool, assertion->cond->notonorafter);
	if (comp < now) {

		ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
				"SAML Response assertion error. expired subject %s",
				assertion->cond->notonorafter );

		apr_table_setn(r->headers_out, "Location",
				apr_psprintf(pool, "%s?st=408",
					divy_build_login_uri(pool, dav_divy_get_root_uri(r))));
		return dav_new_error(pool, HTTP_MOVED_TEMPORARILY, 0, 0, "");
	}

	/* 署名があれば検証する */
	if (assertion->x509certificate != NULL) {
		divy_saml_verifying_xml_signed(pool, assertion);
	}

	return NULL;
}

/**
 * XML署名を検証する
 * 
 */
DIVY_DECLARE(int)
divy_saml_verifying_xml_signed(apr_pool_t *pool, divy_saml_assertion *assertion) 
{

	/* TODO 実装をちゃんとしましょう */

    xmlSecKeysMngrPtr mngr;

    xmlInitParser();
    LIBXML_TEST_VERSION
    xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
    xmlSubstituteEntitiesDefault(1);
	xmlSecDSigCtxPtr dsigCtx = NULL;
	int ret = 0;
	xmlDocPtr doc = NULL;
	xmlNodePtr root_node;
	xmlNodePtr assertion_node;
	xmlNodePtr sig_node;

	TRACE(pool);

    if(xmlSecInit() < 0) {
		/* TODO エラー */
        return 1;
    }

    /* Check loaded library version */
    if(xmlSecCheckVersion() != 1) {
		/* TODO エラー */
        return 1;
    }

	xmlSecCryptoDLLoadLibrary(BAD_CAST XMLSEC_CRYPTO);
	xmlSecCryptoAppInit(NULL);
	xmlSecCryptoInit();

	/* SSL証明書を読み込む */
    mngr = xmlSecKeysMngrCreate();
    if(mngr == NULL) {
		/* TODO エラー */
		return 1;
    }
    if(xmlSecCryptoAppDefaultKeysMngrInit(mngr) < 0) {
		/* TODO エラー */
        return 1;
    }  

	if (xmlSecCryptoAppKeysMngrCertLoadMemory(mngr,
										(xmlSecByte*)assertion->x509certificate,
										strlen(assertion->x509certificate),
										xmlSecKeyDataFormatPem,
										xmlSecKeyDataTypeTrusted) < 0) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
	}

	dsigCtx = xmlSecDSigCtxCreate(mngr);
	if (dsigCtx == NULL) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
	}

	//dsigCtx->enabledReferenceUris = xmlSecTransformUriTypeEmpty;

    /* limit allowed transforms for signature and reference processing */
	if((xmlSecDSigCtxEnableSignatureTransform(dsigCtx,
											 xmlSecTransformInclC14NId) < 0) ||
		(xmlSecDSigCtxEnableSignatureTransform(dsigCtx,
											 xmlSecTransformExclC14NId) < 0) ||
		(xmlSecDSigCtxEnableSignatureTransform(dsigCtx,
											 xmlSecTransformSha1Id) < 0) ||
		(xmlSecDSigCtxEnableSignatureTransform(dsigCtx,
											 xmlSecTransformRsaSha1Id) < 0)) {

		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;

    }

	if((xmlSecDSigCtxEnableReferenceTransform(dsigCtx,
											 xmlSecTransformInclC14NId) < 0) ||
       (xmlSecDSigCtxEnableReferenceTransform(dsigCtx,
											 xmlSecTransformExclC14NId) < 0) ||
       (xmlSecDSigCtxEnableReferenceTransform(dsigCtx,
											 xmlSecTransformSha1Id) < 0) ||
       (xmlSecDSigCtxEnableReferenceTransform(dsigCtx,
											 xmlSecTransformEnvelopedId) < 0)) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
    }

	if (xmlSecPtrListAdd(&(dsigCtx->keyInfoReadCtx.enabledKeyData),
										BAD_CAST xmlSecKeyDataX509Id) < 0) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
	}

	doc = xmlParseMemory(assertion->xml, strlen(assertion->xml));
	if (doc == NULL) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
	}

	root_node = xmlDocGetRootElement(doc);
	if (root_node == NULL) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
	}

	/* Assertionエレメントを取得する */
	assertion_node = xmlSecFindNode(root_node, 
										(const xmlChar*)"Assertion",
										(const xmlChar*)SAML_URN_ASSERTION);
	if (assertion_node == NULL) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
	}

	/* Assertonエレメントのシグネチャを取得する */
	sig_node = xmlSecFindNode(assertion_node,
											xmlSecNodeSignature, xmlSecDSigNs);

	if (sig_node == NULL) {
		/* TODO エラー */
		ret = 1;
		goto cleanup_verify_sign;
	}

	/* この部分が必ずエラーをはきだしてしまう。作り方間違ってるんだろうね */
	if (xmlSecDSigCtxVerify(dsigCtx, sig_node) < 0) {
		/* TODO エラー */
		ret = 1;
		//ERRLOG0(pool , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC, "verify fail");
		goto cleanup_verify_sign;
	}

	ERRLOG0(pool , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC, "verify ok");

cleanup_verify_sign:

	if (mngr != NULL) 
		xmlSecKeysMngrDestroy(mngr);

	return ret;
}

/**
 * SAMLのAuth認証で利用するRelayStateをmemcahedに記録する
 */
DIVY_DECLARE(int)
divy_saml_save_relaystate(request_rec *r, const char* data, 
												int timeout, char **token)
{
	apr_status_t st;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	apr_pool_t *p = r->pool;
	char *prefix = apr_pstrdup(p, conf->root_uri);

	TRACE(p);

	*token = NULL; /* 初期化 */

	if (IS_EMPTY(data)) {
		return 1;
	}

	*token = make_random_string(p);
	st = divy_memcache_set(p, conf->memd, prefix, *token, (char*)data,
												 strlen(data),
												 timeout, 0);
	if (st != APR_SUCCESS) return 1;

	return 0;
}

/**
 * 記録されているRelayStateの情報を取得する
 */
DIVY_DECLARE(int)
divy_saml_load_relaystate(request_rec *r, const char* token,
										unsigned int updatetime, char **data)
{
	apr_status_t st;
	apr_pool_t *p = r->pool;
	apr_size_t length = 0;
	apr_uint16_t flags = 0;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	char *prefix = apr_pstrdup(p, conf->root_uri);

	TRACE(p);

	*data = NULL;	/* 初期化 */
	if (IS_FILLED(token)) {
		st = divy_memcache_get(p, conf->memd, prefix, token,
													data, &length, &flags);
		if (updatetime > 1 && IS_FILLED(*data)) {
			(void)divy_memcache_set(p, conf->memd, prefix, token,
									*data, strlen(*data), (int)updatetime, 0);
		}
		else {
			(void)divy_memcache_delete(p, conf->memd, prefix, token, 0);
		}

		if (st != APR_SUCCESS) return 1;
	}

	return 0;
}

/**
 * SAMLのユーザを新規作成する
 * @param r	request_rec *
 * @param assertion divy_saml_assertion *
 *
 * @return int 0: 正常 / 1: エラー
 */
DIVY_DECLARE(int)
divy_saml_provisioning_account(request_rec *r, divy_saml_assertion *assertion)
{

	/* TODO まだ途中なので何もしてません */
# if 0
	divy_rdbo_usr *dst_usr_pr = NULL;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	apr_pool_t *p = r->pool;

	divy_saml_attribute *attr;

	divy_rdbo_usr *usr_pr = apr_pcalloc(r->pool, sizeof(divy_rdbo_usr));

	TRACE(p);

	if (assertion == NULL) {
		/* TODO エラーを */
		return 1;
	}

	for (attr = assertion->attr; attr != NULL; attr = attr->next) {
		ERRLOG2(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"name=[%s] value=[%s]", attr->name, attr->value);
	}
#endif


	return 0;
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/

/**
 * SAMLResponseXML内のAssertionエレメント内をパースする
 *
 */
static void _divy_saml_parse_assertion_element(apr_pool_t *pool,
							apr_xml_elem *elem, divy_saml_assertion *assertion)
{

	apr_xml_elem *e;
	for (e = elem->first_child; e != NULL; e = e->next) {

		/* Subject */
		if ((strcmp(e->name, "Subject")) == 0) {
			assertion->subject = apr_pcalloc(pool,
											sizeof(struct divy_saml_subject));
			apr_xml_elem *se;
			for (se = e->first_child; se != NULL; se = se->next) {
				/* NameID */
				if ((strcmp(se->name, "NameID")) == 0) {
					assertion->subject->nameid =
									(char*)divy_xml_get_cdata(se, pool, 1);
				}
				/* SubjectConfirmation */
				if ((strcmp(se->name, "SubjectConfirmation")) == 0) {
					/* Method */
					divy_saml_get_xml_attribute(pool, se, "Method",
												&assertion->subject->method);
					apr_xml_elem *de = se->first_child;
					if ((strcmp(de->name, "SubjectConfirmationData")) == 0) {
						/* Recipient */
						divy_saml_get_xml_attribute(pool, de, "Recipient",
											&assertion->subject->recipient);
						/* NotOnOrAfter */
						divy_saml_get_xml_attribute(pool, de, "NotOnOrAfter",
											&assertion->subject->notonorafter);
						/* InReponseTo */
						divy_saml_get_xml_attribute(pool, de, "InResponseTo",
											&assertion->subject->inresponseto);
					}
				}
			}
		}

		/* Signature(署名) */
		if ((strcmp(e->name, "Signature")) == 0) {
			apr_xml_elem *es, *ec;
			for (es = e->first_child; es != NULL; es = es->next) {
				if ((strcmp(es->name, "KeyInfo")) == 0) {
					ec = es->first_child;
					if (ec != NULL && (strcmp(ec->name, "X509Data")) == 0) {
						ec = ec->first_child;
						if (ec != NULL &&
								(strcmp(ec->name, "X509Certificate")) == 0) {

							assertion->x509certificate = 
								apr_psprintf(pool, "%s"CRLF"%s"CRLF"%s",
										"-----BEGIN CERTIFICATE-----",
										divy_xml_get_cdata(ec, pool, 1),
										"-----END CERTIFICATE-----");
						}
					}
				}
			}

		}

		/* Conditions */
		if ((strcmp(e->name, "Conditions")) == 0) {
			assertion->cond = apr_pcalloc(pool,
										sizeof(struct divy_saml_conditions));
			/* NotBefore */
			divy_saml_get_xml_attribute(pool, e, "NotBefore",
									&assertion->cond->notbefore);
			/* NotOnOrAfter */
			divy_saml_get_xml_attribute(pool, e, "NotOnOrAfter",
									&assertion->cond->notonorafter);
		}

		/* AttributeStatement */
		if ((strcmp(e->name, "AttributeStatement")) == 0) {
			apr_xml_elem *ae;
			for (ae = e->first_child; ae != NULL; ae = ae->next) {
				if ((strcmp(ae->name, "Attribute")) == 0) {
					divy_saml_attribute *attr = apr_pcalloc(pool,
												sizeof(divy_saml_attribute));
					if (assertion->attr == NULL) {
						assertion->attr = attr;
					}
					else {
						attr->next = assertion->attr;
						assertion->attr = attr;
					}
					divy_saml_get_xml_attribute(pool, ae,
												"Name", &attr->name);
					apr_xml_elem *aev;
					if ((aev = ae->first_child) != NULL) {
						if ((strcmp(aev->name, "AttributeValue")) == 0) {
							attr->value = (char*)divy_xml_get_cdata(aev, pool, 1);
//ERRLOG1(pool , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC, "attr->value = %s", attr->value);


						} /* if strcmp(aev->name, "AttributeValue" */
					}
				}
			}
		}

		/* AuthnStatement */
		if ((strcmp(e->name, "AuthnStatement")) == 0) {
			/* TODO 解析せよ */
			/* AuthnInstant 属性 や SessionIndex 属性*/
		}
	}
	
}

/**
 * SAMLRequst文字列を作成する
 * 
 * @param	r				request_rec *	リクエスト構造体
 * @param	samlrequest		char *			SAMLRequest文字列	
 *
 * @return 0: 成功 / 1: エラー
 */
static int divy_saml_build_samlrequest(request_rec *r, char **samlrequest)
{
	int ret = 0;	/* デフォルト成功 */
	apr_pool_t *p = r->pool;
	dav_divy_dir_conf *conf = dav_divy_get_dir_config(r);
	const char *urn;
	char *location = NULL;
	const char *saml_uri, *acs;
	apr_uuid_t uuid;
    char char_uuid[APR_UUID_FORMATTED_LENGTH + 1];
	const char *id = NULL;
	char *issueinstant = NULL;
	char *authn_xml = NULL;
	int length = 0;

	divy_sbuf *sbuf = NULL;

	xmlDocPtr doc = NULL;
	xmlNodePtr signNode = NULL;
	xmlNodePtr refNode = NULL;
	xmlNodePtr keyInfoNode = NULL;
	xmlSecDSigCtxPtr dsigCtx = NULL;
	xmlChar *signed_xml = NULL;
	//xsltSecurityPrefsPtr xsltSecPrefs = NULL;
	int signed_xml_size = 0;

	char *comp_xml = NULL;
	int comp_xml_len = 0;

	char *base64_str = NULL;
	int base64_len = 0;

	if (conf->idpmeta == NULL) {
		/* メタデーターがない. あってはならないエラー */
		ERRLOG0(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Fail to IdP metadata. Please check configuration file.");
		return 1;
	}

	TRACE(p);

	/* http postが優先 */
	if (conf->idpmeta->choose_binding_type == SAML_URN_BINDING_POST) {
		urn = SAML_URN_BINDING_POST_STR;
		location = (char *)conf->idpmeta->signon->http_post;
	}
	else {
		/* TODO HTTP Post以外ならどうするの?? */
	}

	/* Assertion Consumer Service(ACS) urlを作成する */
	saml_uri = apr_psprintf(p, "%s%s", dav_divy_get_root_uri(r), "/.saml/acs");
	acs = divy_construct_url2(p, DIVY_URLTYPE_PUBLIC, saml_uri, r, NULL);
/* DEBUG */
//ERRLOG1(p , APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC, "acs = %s", acs);

	/* idを作成 */
	apr_uuid_get(&uuid);
	apr_uuid_format(char_uuid, &uuid);
	id = apr_psprintf(p, "teamfile_%s", char_uuid);

	/* IssueInstantを作成 */
	divy_format_time_t(p, dav_divy_get_now_epoch(),
								DIVY_TIME_STYLE_ISO8601, &issueinstant);
							
	divy_sbuf_create(p, &sbuf, 1024);

	divy_sbuf_append(sbuf, apr_psprintf(p, "<samlp:AuthnRequest xmlns:samlp="
					"\"urn:oasis:names:tc:SAML:2.0:protocol\""
					" AssertionConsumerServiceURL=\"%s\""
					" Destination=\"%s\""
					" ID=\"%s\""
					" IssueInstant=\"%s\""
					" ProtocolBinding=\"%s\""
					" Version=\"%s\""
					">", acs, location, id, issueinstant, urn, "2.0"));

	divy_sbuf_append(sbuf, apr_psprintf(p, "<saml:Issuer xmlns:saml="
					"\"urn:oasis:names:tc:SAML:2.0:assertion\">"
					"%s</saml:Issuer>",
					divy_construct_url2(p, DIVY_URLTYPE_PUBLIC,
										dav_divy_get_root_uri(r), r, NULL)));

	divy_sbuf_append(sbuf, "</samlp:AuthnRequest>");

	authn_xml = divy_sbuf_tostring(sbuf);
/* DEBUG */
#if 0
ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA, "AuthnRequest = %s", authn_xml);
#endif

	/*
	 * XML署名をする
	 * 署名が要求されている場合、XML署名のキーファイル、証明書ファイルが
	 * 設定されている必要があります
	 */
	if (conf->idpmeta->wantauthnreqsigned) {
//	if (0) {

		if (IS_EMPTY(conf->saml_privatekeypath) ||
				IS_EMPTY(conf->saml_x509certificatepath))
		{
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"SAML IdP metadata ``WantAuthnReqestSigned`` is true"
					"However private key or public key is missing"
					"Please checked SAML directive settings.");
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		xmlInitParser();
		LIBXML_TEST_VERSION
		xmlLoadExtDtdDefaultValue = XML_DETECT_IDS | XML_COMPLETE_ATTRS;
		xmlSubstituteEntitiesDefault(1);

		xmlSecInit();
		xmlSecCryptoDLLoadLibrary(BAD_CAST XMLSEC_CRYPTO);
		xmlSecCryptoAppInit(NULL);
		xmlSecCryptoInit();

		doc = xmlParseMemory(authn_xml, strlen(authn_xml));
		if ((doc == NULL) || (xmlDocGetRootElement(doc) == NULL)){
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}
		signNode = xmlSecTmplSignatureCreate(doc, xmlSecTransformExclC14NId,
											  xmlSecTransformRsaSha1Id, NULL);
		if(signNode == NULL) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* add <dsig:Signature/> node to the doc */
		xmlAddChild(xmlDocGetRootElement(doc), signNode);
		refNode = xmlSecTmplSignatureAddReference(signNode,
													xmlSecTransformSha1Id,
													NULL, NULL, NULL);
		if (refNode == NULL) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* add enveloped transform */
		if(xmlSecTmplReferenceAddTransform(refNode,
									xmlSecTransformEnvelopedId) == NULL) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* add <dsig:KeyInfo/> and <dsig:X509Data/> */
		keyInfoNode = xmlSecTmplSignatureEnsureKeyInfo(signNode, NULL);
		if(keyInfoNode == NULL) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		if(xmlSecTmplKeyInfoAddX509Data(keyInfoNode) == NULL) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* create signature context, * we don't need keys manager
		 * in this example
		 */
		dsigCtx = xmlSecDSigCtxCreate(NULL);
		if(dsigCtx == NULL) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* load private key, assuming that there is not password */
		dsigCtx->signKey = xmlSecCryptoAppKeyLoad(conf->saml_privatekeypath,
									xmlSecKeyDataFormatPem, NULL, NULL, NULL);
		if(dsigCtx->signKey == NULL) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}
		/* load certificate and add to the key */
		if(xmlSecCryptoAppKeyCertLoad(dsigCtx->signKey, 
									conf->saml_x509certificatepath,
									xmlSecKeyDataFormatPem) < 0) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* set key name to the file name, this is just an example! */
		if(xmlSecKeySetName(dsigCtx->signKey,
								(xmlChar *)conf->saml_privatekeypath) < 0) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* sign the template */
		if(xmlSecDSigCtxSign(dsigCtx, signNode) < 0) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		/* 署名したXMLをバッファ(authn_xml)へ戻す */
		xmlDocDumpMemory(doc, &signed_xml, &signed_xml_size);
		authn_xml = apr_pstrdup(p, (char*)signed_xml);

	} /* 署名が必要な場合 */

	length = strlen(authn_xml);
	/* zlib圧縮を行う HTTP REDIRECTの場合のみ */
	if (conf->idpmeta->choose_binding_type == SAML_URN_BINDING_REDIRECT) {
		if (divy_saml_xml_deflate(p, authn_xml, length,
					&comp_xml, &comp_xml_len)) {
			/* TODO error */
			ret = 1;	/* エラー */
			goto cleanup_buildxml;
		}

		authn_xml = comp_xml;
		length    = comp_xml_len;
	}

	/* base64 エンコード */
	base64_str = apr_pcalloc(p, sizeof(char) *
								(apr_base64_encode_len(length) + 1));
	base64_len = apr_base64_encode(base64_str, authn_xml, length);
	base64_str[base64_len + 1] = '\0';
	authn_xml = apr_pstrdup(p, base64_str);

	/* urlエンコード HTTP REDIRECTの場合のみ */
	if (conf->idpmeta->choose_binding_type == SAML_URN_BINDING_REDIRECT) {
		*samlrequest = divy_url_encode(p, authn_xml);
	}
	else {
		*samlrequest = authn_xml;
	}

	/* DEBUG */
	ERRLOG2(p, APLOG_DEBUG, DIVY_FST_IERR + DIVY_SST_DEBUG,
		"SAMLRequest = %s / orignal len = %d", *samlrequest,
		(int)strlen(authn_xml));
	ret = 0;

cleanup_buildxml:

    /* cleanup */
	if (signed_xml != NULL) {
		xmlFree(signed_xml);
	}
    if(dsigCtx != NULL) {
		xmlSecDSigCtxDestroy(dsigCtx);
    }
    
    if(doc != NULL) {
        xmlFreeDoc(doc); 
    }

	return ret;

}

/**
 * 一つのEntityDescriptorを解析する
 *
 * @param pool apr_pool_t * プール
 * @param elem apr_xml_elem * XMLエレメント
 * @param meta	divy_saml_idp_metadata ** IdPメタデータ構造体
 *
 * @return 0: 成功 / 1: エラー
 */
static int divy_saml_build_entity_descriptor(apr_pool_t *pool, apr_xml_elem *elem, divy_saml_idp_metadata **meta)
{
	apr_xml_elem *e, *e1;
	char *value;
	divy_saml_idp_metadata *cur_meta = NULL;
	int foundIDP = 0;	/* IDPSSODescriptor が見つかったか? */

	cur_meta = apr_pcalloc(pool, sizeof(divy_saml_idp_metadata));
	cur_meta->signon = apr_pcalloc(pool, sizeof(divy_saml_idp_binding));
	cur_meta->logout = apr_pcalloc(pool, sizeof(divy_saml_idp_binding));

	if (divy_saml_get_xml_attribute(pool, elem, "entityID", &value) == 0) {
		cur_meta->entityID = apr_pstrdup(pool, value);
	}
	else {
		/* TODO
		 * saml-metadata-2.0-os 2.3.2 
		 * EntityDescriptor に entityIDは必須のはず 
		 * エラーにすべきだよね
		 */
	}

	for (e = elem->first_child; e != NULL; e = e->next) {
		if ((strcmp(e->name, "IDPSSODescriptor")) == 0) {

			foundIDP = 1;	/* 見つかった */
			if (divy_saml_get_xml_attribute(pool, e,
								"WantAuthnRequestsSigned", &value) == 0) {
				if ((strcmp(value, "true")) == 0) {
					cur_meta->wantauthnreqsigned = 1;	/* 要求する */
				}
			}

			for (e1 = e->first_child; e1 != NULL; e1 = e1->next) {

				/* SingleLogoutService or SingleSignOnService */
				if (divy_saml_parse_xml_signon_logout(pool, e1, &cur_meta)) {
					/* TODO エラーメッセージを入れること */
					return 1;
				}

				/* KeyDescriptor */
				if (divy_saml_set_x509_certificate(pool, e1, &cur_meta)) {
					/*
					 * TODO エラーメッセージを入れること
					 * divy_saml_set_x509_certificateでは
					 * 詳細エラーを出しているけど
					 */
					return 1;
				}

				/* ArtifactResolutionService */
				/* TODO 実装を */
			}
		}
		else if ((strcmp(e->name, "AttributeAuthorityDescriptor")) == 0) {
			/* TODO 実装を */
		}
		else if ((strcmp(e->name, "ContactPerson")) == 0) {
			/* TODO 実装を */
			/* 使い道がないかも */
		}
	}

	/* IDPSSODescriptorが見つかった場合だけリストにする */
	if (foundIDP) {
		cur_meta->filepath = (*meta)->filepath;
		cur_meta->next = *meta;
		*meta = cur_meta;
	}

	return 0;
}

/**
 * エレメント内の属性を指定して取得する
 *
 * @param pool apr_pool_t * プール
 * @param elem apr_xml_elem * XMLエレメント
 * @param key  const char * 取得したい属性名
 * @param value char ** 取得した値
 *
 * @return 0: 成功 / 1: エラー
 */
static int divy_saml_get_xml_attribute(apr_pool_t* pool, apr_xml_elem *elem, const char *key, char **value)
{
	apr_xml_attr *attr;

	if (elem == NULL || IS_EMPTY(key)) return 1;
	*value = NULL;

	for (attr = elem->attr; attr != NULL; attr = attr->next) {
		if ((strcmp(key, attr->name)) == 0) {
			(*value) = apr_pstrdup(pool, attr->value);
			return 0; /* 見つかったら終わる */
		}
	}

	return 1; /* 見つからなかった */
}

/**
 * SingleSignOn / SingleLogoutをパースしてメタデータ構造体へ保存する
 * 
 * @param	pool	apr_pool_t*
 * @param elem	apr_xml_elem *	XMLエレメント
 * @param meta	divy_saml_idp_metadata ** IdPメタデータ構造体
 *
 * @return 0: 成功 / 1: エラー
 */
static int divy_saml_parse_xml_signon_logout(apr_pool_t *pool, apr_xml_elem *elem, divy_saml_idp_metadata **meta)
{

	char *binding, *location;
	int issignon = 0;

	if (elem == NULL || *meta == NULL) return 1;

	if ((strcmp(elem->name, "SingleSignOnService")) == 0) {
		issignon = 1;
	}
	else if ((strcmp(elem->name, "SingleLogoutService")) == 0) {
		issignon = 0;
	}
	else {
		/* 関係ない内容の場合はスキップ */
		return 0;
	}

	TRACE(pool);
	
	/* Binding 属性を取得する */
	if (divy_saml_get_xml_attribute(pool, elem, "Binding", &binding)) {
		/* TODO Error */
		return 1;
	}

	/* Location 属性を取得する */
	if (divy_saml_get_xml_attribute(pool, elem, "Location", &location)) {
		/* TODO Error */
		return 1;
	}

	/* DEBUG */
	ERRLOG2(pool, APLOG_DEBUG, DIVY_FST_INFO + DIVY_SST_DEBUG,
							"binding=%s / location=%s", binding, location);

	divy_saml_set_binding_location(pool, issignon, elem, binding,
														location, meta);
	
	return 0;
}

/**
 * binding locationを解析してメタデータ構造体に保存する
 *
 * @param pool	apr_pool_t *　プール
 * @param issignon	int SignOn用なら1 / logout用なら0
 * @param elem	apr_xml_elem *	XMLエレメント
 * @param urn const char * キーとなるurn	
 * @paarm location const char* 保存するURL
 * @param meta	divy_saml_idp_metadata ** IdPメタデータ構造体
 * 
 * @return 0: 成功 / 1: エラー
 *
 */
static int divy_saml_set_binding_location(apr_pool_t *pool, int issignon, apr_xml_elem *elem, const char* urn, const char* location, divy_saml_idp_metadata **meta)
{
	divy_saml_idp_binding *binding;

	if (issignon == 1) {
		binding = (*meta)->signon;
	}
	else {
		binding = (*meta)->logout;
	}

	if (strcmp(urn,SAML_URN_BINDING_REDIRECT_STR) == 0) {
		binding->http_redirect = location;
	}
	else if (strcmp(urn, SAML_URN_BINDING_POST_STR) == 0) {
		binding->http_post = location;
	}
	else if (strcmp(urn, SAML_URN_BINDING_ARTIFACT_STR) == 0) {
		binding->http_artifact = location;
	}
	else if (strcmp(urn, SAML_URN_BINDING_SOAP_STR) == 0) {
		binding->soap = location;
	}
	else {
		/* TODO error */
		return 1;
	}

	return 0;
}

/**
 * IdPメタデータXML内のKeyDescriptorを展開してメタデータ構造体に保存する
 *
 * @param pool	apr_pool_t *　プール
 * @param elem	apr_xml_elem *	XMLエレメント
 * @param meta	divy_saml_idp_metadata ** IdPメタデータ構造体
 *
 * @return 0: 成功 / 1: エラー
 *
 */
static int divy_saml_set_x509_certificate(apr_pool_t *pool, apr_xml_elem *elem, divy_saml_idp_metadata **meta)
{
	if (elem == NULL) return 1;
	apr_xml_elem *e = elem;
	char *use = NULL;
	char *x509data = NULL;

	if ((strcmp(e->name, "KeyDescriptor")) == 0) {

		/* 用途を取得（無い場合もある) */
		divy_saml_get_xml_attribute(pool, elem, "use", &use);
		
		e = e->first_child;
		if ((strcmp(e->name, "KeyInfo")) == 0) {
			e = e->first_child;
			if ((strcmp(e->name, "X509Data")) == 0) {
				e = e->first_child;
				x509data = (char *)divy_xml_get_cdata(e, pool, 1);
#if 0
ERRLOG1(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_SYNTAX,
						"IdP metadata x509 data = %s", x509data);
#endif

				/* 証明書の用途が署名と暗号化用二種類ある */
				if (IS_EMPTY(use) || strcmp(use, "signing") == 0) {
					(*meta)->signedx509data = x509data;
				}
				else {
					/* 未使用 */
					(*meta)->encryptionx509data = x509data;
				}

				return 0;	/* 見つかった正常 */
			}
			else {
				ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
						"Failed to Parse KeyDescriptor/KeyInfo element."
						"Not found <X509Data> Element.");
				return 1;	/* 失敗 */
			}
		}
		else {
			ERRLOG0(pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
					"Failed to Parse KeyDescriptor element."
					"Not found <KeyInfo> Element.");
			return 1;	/* 失敗 */
		}
	}

	return 0; /* 存在しなかった*/

}

/**
 * SAMLの内容(src)を圧縮する SAML 要件
 *
 */
static int divy_saml_xml_deflate(apr_pool_t *pool, const char *src, int srclen,
														char **dst, int *dstlen)
{

	int status;
	uLongf len = 0;
	z_stream z  = {0};
	Bytef *dest = NULL;

	len = compressBound(srclen);
	dest = apr_pcalloc(pool, len);

	status = deflateInit2(&z, Z_DEFAULT_COMPRESSION, Z_DEFLATED,
											-15, 9, Z_DEFAULT_STRATEGY);
	if (status != Z_OK) {
		return 1;	/* 失敗 */
	}

	z.next_in = (Bytef*)src;
	z.avail_in = srclen;

	z.next_out = dest;
	z.avail_out = len;

	status = deflate(&z, Z_FINISH);
	if (status != Z_OK && status != Z_STREAM_END) {
		deflateEnd(&z);
		return 1;	/* 失敗 */
	}

	*dst = (char*)dest;
	*dstlen = z.total_out;

	deflateEnd(&z);

	return 0;
}


