/**
 * $Id$
 *
 * util_vscan.c
 *
 * TeamFile モジュールがウィルス検索プロバイダにアクセスするのに必要な手続きを
 * 定義する。tf_vscan.[ch] で定義された関数、構造体群へのラッパ関数です。
 *
 * 2004/10/04 Mon takehara NEW
 */
/* Apache header files */
#include "httpd.h"
#include "util_script.h"
#include "apr.h"
#include "apr_pools.h"
#include "apr_hash.h"
#include "apr_strings.h"
#include "apr_time.h"

#include "util_common.h"
#include "mod_dav_tf.h"
#include "util_vscan.h"
#include "util.h"

/*------------------------------------------------------------------------------
  Define static values and macro
  ----------------------------------------------------------------------------*/
APLOG_USE_MODULE(dav_tf);

/*------------------------------------------------------------------------------
  Define Global values
  ----------------------------------------------------------------------------*/
/**
 * ウィルス検索プロバイダ専用のプール
 * (note)
 * 	pchild のサブプールです。
 */
static apr_pool_t *vsc_p = NULL;

/**
 * グローバル変数を排他的に使用する際に使うmutex
 */
static apr_thread_mutex_t *gth_vsc_mutex = NULL;

/*
 * 初期化完了状態を記録するハッシュ
 *
 * key : _get_vsc_provider_id() の値
 * val : NULL / VSC_PROVIDER_INIT_FINISH
 */
#define VSC_PROVIDER_INIT_FINISH	"vsc_finish"
static apr_hash_t *vsc_initfinish_hash = NULL;

/**
 * ウィルス検索プロバイダをキャッシュするハッシュ
 * key :
 * val : VscDataSource *
 */
static apr_hash_t *vscprv_hash = NULL;

/*------------------------------------------------------------------------------
  Declare private prototype function
  ----------------------------------------------------------------------------*/
static apr_status_t _cleanup_vsc_env(void *data);
static const char * _get_vsc_provider_id(request_rec *r, const char *providerType);
static int _is_vscenv_ready(const char *providerid);
static int _init_vsc_provider(request_rec *r, const char *providerid);
static int _registe_vsc_provider(request_rec *r, 
					const char *providerid,
					VscDataSource *vscds);
static void _set_vscenv_is_ready(const char *providerid);
static apr_status_t _cleanup_vscsession(void *data);

/*------------------------------------------------------------------------------
  Define Public Functions
  ----------------------------------------------------------------------------*/
/**
 * ウィルス検索プロバイダ管理コンテキストを初期化する。
 * この関数は複数のスレッドからアクセスされてはなりません。
 *
 */
DIVY_DECLARE(void) init_vscprv_env(apr_pool_t *pchild, server_rec *s)
{
	apr_status_t rv;

	TRACE_S(s);

	/* ウィルス検索プロバイダが利用するプールの生成 */
	apr_pool_create(&vsc_p, pchild);

	/* クリーンアップ関数を登録する */
	apr_pool_cleanup_register(vsc_p, NULL, _cleanup_vsc_env,
					apr_pool_cleanup_null);
	/* mutexの生成 */
	rv = apr_thread_mutex_create(&gth_vsc_mutex,
					APR_THREAD_MUTEX_UNNESTED, vsc_p);
	if (rv != APR_SUCCESS) {
		ERRLOG1(NULL, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to create gth_vsc_mutex. (code = %d)", rv);
	}

	vsc_initfinish_hash = apr_hash_make(vsc_p);
	vscprv_hash         = apr_hash_make(vsc_p);
}

/**
 * 指定されたproviderTypeのウィルス検索プロバイダ(VscDataSource)を取得する。
 *
 */
DIVY_DECLARE(VscDataSource *) lookup_vsc_provider(request_rec *r,
						const char *providerType)
{
	apr_status_t rv;
	VscDataSource *vscds = NULL;
	apr_pool_t *p        = r->pool;
	const char *providerid;
	int ret;

	if (IS_EMPTY(providerType)) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"providerType is NULL.");
		return NULL;
	}

	/* プロバイダIDを取得 */
	providerid = _get_vsc_provider_id(r, providerType);

	/* mutex ロックをかける */
	rv = apr_thread_mutex_lock(gth_vsc_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex lock. (code = %d)", rv);
		return NULL;
	}

	/*
	 * プロバイダが初期化されていなければ初期化する
	 */
	if (!_is_vscenv_ready(providerid)) {

		ret = _init_vsc_provider(r, providerid);
		if (ret != DIVY_VSC_ST_OK) {
			ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to initialize virus scan provider.");
			vscds = NULL;
		}
		else {
			/* VscDataSource の取得 */
			vscds = apr_hash_get(vscprv_hash, providerid,
						APR_HASH_KEY_STRING);
		}
	}
	else {
		/* VscDataSource の取得 */
		vscds = apr_hash_get(vscprv_hash, providerid, APR_HASH_KEY_STRING);
	}

	/* mutex ロックを解除 */
	rv = apr_thread_mutex_unlock(gth_vsc_mutex);
	if (rv != APR_SUCCESS) {
		ERRLOG1(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_OS,
			"Failed to mutex unlock. (code = %d)", rv);
		return vscds;	/* 大目に見る */
	}

	return vscds;
}

/**
 * ウィルス検索プロバイダとの接続を行い、ウィルス検索可能な状態にする。
 *
 */
DIVY_DECLARE(VscSession *) divy_get_vscsession(request_rec *r,
						VscDataSource *vscds)
{
	VscProperty *vscprop     = NULL;
	VscSession *vscsession   = NULL;
	dav_divy_dir_conf *dconf = NULL;
	apr_pool_t *p            = r->pool;
	int ret;

	if (vscds == NULL) {
		ERRLOG0(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"vscds is NULL.");
		return NULL;
	}

	/*
	 * ウィルス検索プロバイダに渡す接続情報の収集
	 */
	dconf = dav_divy_get_dir_config(r);
	vscprop = apr_pcalloc(p, sizeof(VscProperty));
	vscprop->env = (const char * const *) ap_create_environment(p,
							r->subprocess_env);
	if (IS_FILLED(dconf->vsccmd)) {
		/* コマンドラインをパースする */
		vscprop->argv = (const char * const *)
				divy_parse_commandline(p, dconf->vsccmd);
	}
	vscprop->hostname = apr_pstrdup(p, dconf->vschostname);
	if (IS_FILLED(dconf->vschostport)) {
		vscprop->port = atoi(dconf->vschostport);
	}
	vscprop->refreshinterval = DIVY_VSC_DEFAULT_REFRESH_INTERVAL;

	/*
	 * ウィルスパターンデータベースの更新
	 */
	if (divy_vsc_canLoadVirusData(vscds) == DIVY_VSC_ST_OK) {
		/* パターンデータベースの読み込み */
		ret = divy_vsc_loadVirusData(vscds, vscprop);
		if (ret != DIVY_VSC_ST_OK) {
			int ret2 = divy_vsc_getCode(vscds, DIVY_VSC_TYPE_DATASOURCE);
			char *msg = divy_vsc_getMsg(vscds, DIVY_VSC_TYPE_DATASOURCE);

			ERRLOG2(p, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
				"Failed to load virus pattern database."
				"(code = %d, msg = %s)", ret2, msg);
			return NULL;
		}
	}

	/*
	 * セッションを取得
	 */
	vscsession = divy_vsc_getVscSession(vscds, vscprop, p);
	if (vscsession != NULL) {
		/* クリーンアップハンドラに登録する */
		apr_pool_cleanup_register(p, vscsession, _cleanup_vscsession,
						apr_pool_cleanup_null);
	}

	return vscsession;
}

/**
 * ウィルス検索プロバイダを利用できるかどうか。
 *
 * 	DIVY_VSC_ST_ACTIVE         : 有効だった
 * 	DIVY_VSC_ST_INACTIVE       : OFFだった
 */
DIVY_DECLARE(int) divy_vsc_enable(request_rec *r)
{
	dav_divy_dir_conf *dconf    = dav_divy_get_dir_config(r);
	dav_divy_server_conf *sconf = dav_divy_get_server_config(r->server);

	/* ウィルス検索プロバイダが利用できる状態かどうか */
	if (dconf->vscan == NULL ||
			strcasecmp(dconf->vscan, DIVY_VSC_OFF) == 0) {
		return DIVY_VSC_ST_INACTIVE;	/* 利用できません */
	}

	/* 適切なライセンスを持っているか？ */
	if (sconf->use_vsc_opt == 0) {
		return DIVY_VSC_ST_INACTIVE;
	}

	return DIVY_VSC_ST_ACTIVE;
}

/*------------------------------------------------------------------------------
  Define private function
  ----------------------------------------------------------------------------*/
/**
 * クリーンアップハンドラ
 */
static apr_status_t _cleanup_vsc_env(void *data)
{
	/* mutex の破棄 */
	if (gth_vsc_mutex != NULL) {
		(void) apr_thread_mutex_destroy(gth_vsc_mutex);
	}

	gth_vsc_mutex = NULL;
	vscprv_hash   = NULL;
	vsc_p         = NULL;	/* サブプールは自動的に削除されるので何もしない */

	return APR_SUCCESS;
}


/**
 * ウィルス検索プロバイダのプロバイダIDを取得する。
 *
 * @param r request_rec *
 * @param providerType const char * プロバイダタイプ
 * @return const char * プロバイダIDを表す文字列
 */
static const char * _get_vsc_provider_id(request_rec *r, const char *providerType)
{
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);
	return apr_psprintf(r->pool, "tf.vsc.%s:%s:%d:%s#%s:%s@%s",
				providerType,
				REPLACE_NULL(dconf->vscan),
				dconf->streammethod,
				REPLACE_NULL(dconf->vsccmd),
				REPLACE_NULL(dconf->vschostname),
				REPLACE_NULL(dconf->vschostport),
				dconf->root_uri);

}

/**
 * ウィルス検索プロバイダの接続環境が初期化されているかどうか.
 *
 * @param r request_rec *
 * @return int 1: 初期化されている / 0: 初期化されていない
 */
static int _is_vscenv_ready(const char *providerid)
{
	char *str;

	if (vsc_initfinish_hash == NULL ||
	    apr_hash_count(vsc_initfinish_hash) == 0) {
		return 0;
	}

	/* プロバイダの初期化状態を取得する */
	str = apr_hash_get(vsc_initfinish_hash, providerid, APR_HASH_KEY_STRING);
	if (str) {
		return 1;
	}
	else {
		return 0;
	}
}

/**
 * ウィルス検索プロバイダ環境を初期化して利用できる状態にする。
 * (note)
 * 	同時アクセスが起きないよう呼び出し元にて保証して下さい。
 *
 * @param r request_rec*
 * @param providerid const char * プロバイダID
 * @return int 処理終了ステータスコード (0: 成功 / 1: 失敗)
 */
static int _init_vsc_provider(request_rec *r, const char *providerid)
{
	VscDataSource *vscds = NULL;
	int ret;
	dav_divy_dir_conf *dconf = dav_divy_get_dir_config(r);

	TRACE(r->pool);

	/*
	 * VscDataSourceを取得する
	 */
	ret = divy_vsc_lookup_vscdatasource(vsc_p, dconf->vscan, &vscds);
	if (ret == DIVY_VSC_ST_NOTFOUND_VSCDS) {
		ERRLOG1(r->pool, APLOG_WARNING, DIVY_FST_IERR + DIVY_SST_PROC,
			"The specified virus scan provider not found. (name = %s) "
			"Please check configuration file.", dconf->vscan);
		return DIVY_VSC_ST_NOTFOUND_VSCDS;
	}
	else if (ret != DIVY_VSC_ST_OK) {
		ERRLOG2(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_PROC,
			"Failed to lookup virus scan provider.(name = %s, "
			"code = %d)", dconf->vscan, ret);
		return DIVY_VSC_ST_ERR;
	}

	/*
	 * プロバイダIDをキーとしてvscds を登録する
	 * (note)
	 * 	providerid はvscds と同じライフサイクルにするため再確保
	 */
	if (_registe_vsc_provider(r, apr_pstrdup(vsc_p, providerid),
						vscds) != DIVY_VSC_ST_OK) {
		return DIVY_VSC_ST_ERR;
	}

	/* 初期化処理完了の状態を記録する */
	_set_vscenv_is_ready(providerid);

	return DIVY_VSC_ST_OK;
}

/**
 * provideridが示す名称でウィルス検索ルプロバイダ(VscDataSource)を登録する。
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param r request_rec *
 * @param providerid const char* プロバイダID
 * @param vscds VscDataSource* プロバイダ提供のVscDataSource構造体へのポインタ
 * @return int 実行ステータス。(DIVY_VSC_ST_OK: 成功)
 */
static int _registe_vsc_provider(request_rec *r, 
					const char *providerid,
					VscDataSource *vscds)
{
	if (IS_EMPTY(providerid) || vscds == NULL) {
		ERRLOG0(r->pool, APLOG_ERR, DIVY_FST_IERR + DIVY_SST_DATA,
			"providerid or vscds is NULL.");
		return DIVY_VSC_ST_INVALID_PARAMS;
	}

	/* プロバイダIDをキーとしてVscDataSource * を登録する */
	apr_hash_set(vscprv_hash, providerid, APR_HASH_KEY_STRING, vscds);

	return DIVY_VSC_ST_OK;
}

/*
 * providerid で識別されるプロバイダグループの初期化が完了したことを記録する。
 * (note)
 * 	この関数を同時アクセスしないようコール元で保証して下さい。
 *
 * @param providerid const char * プロバイダID
 */
static void _set_vscenv_is_ready(const char *providerid)
{
	/* (note)
	 * 	vsc_initfinish_hash のスコープはグローバルなので、
	 * 	providerid の領域はvsc_p から再確保します。
	 */
	apr_hash_set(vsc_initfinish_hash,
			apr_pstrdup(vsc_p, providerid),
			APR_HASH_KEY_STRING, VSC_PROVIDER_INIT_FINISH);

}

/**
 * VscSession のクリーンアップハンドラ
 * data は VscSession へのポインタです。
 */
static apr_status_t _cleanup_vscsession(void *data)
{
	VscSession *vscsession = (VscSession *)data;

	if (vscsession != NULL) {
		/* セッションを閉じる */
		divy_vsc_closeSession(vscsession);
	}

	return APR_SUCCESS;
}


