/**
 * $Id$
 * 
 * tf_vsc_clmv.c
 *
 * ClamAV AntiVirus Engine 用ウィルス検索プロバイダの実装。
 *
 * 2004/10/05 Tue takehara NEW
 */
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"
#include "apr_time.h"
#include "apr_thread_proc.h"
#include "apr_file_info.h"
#include "apr_file_io.h"
#include "apr_network_io.h"
#include "apr_time.h"

#include "util_common.h"
#include "tf_provider.h"

#if defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H
#include <unistd.h>
#endif	/* defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H */

/* 各種構造体の定義を見えるようにする */
#define DIVY_VSC_PROVIDER_PRIVATE
#include "tf_vscan.h"

/*------------------------------------------------------------------------------
  Define fixed value and macro
  ----------------------------------------------------------------------------*/
/**
 * ClamAV 用ウィルス検索プロバイダの種類を表す文字列
 *
 * [ フレームワークからの要請 ]
 * 	この名称は同一種のプロバイダ間で一意の名称でなければなりません。
 * 	可能であれば、全てのプロバイダ間で一意に識別できる名称であるべき。
 */
#define CLMV_VSC_PROVIDER_TYPE	"clmv"

#define CLAMAV_CMD_STREAM	"STREAM"

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
struct __vsc_sessionCntxt {
	apr_socket_t	*control_socket;	/* clamdへのデータポート */
	apr_socket_t	*data_socket;		/* clamdからもらうデータポート */
};

struct __vsc_StreamFile {
};

/*------------------------------------------------------------------------------
  Declare provider hook function
  ----------------------------------------------------------------------------*/
static int _clmv_vscds_canLoadVirusData(VscDataSource *vscds);
static int _clmv_vscds_loadVirusData(VscDataSource *vscds,
					const VscProperty *vscprop);
static VscSession * _clmv_vscds_getVscSession(VscDataSource *vscds,
					const VscProperty *vscprop, apr_pool_t *p);
static int _clmv_vscds_canScanStream(VscDataSource *vscds);
static int _clmv_vscsession_canCacheSession(VscSession *vscsession);
static void _clmv_vscsession_closeSession(VscSession *vscsession);
static int _clmv_vscsession_scanFile(VscSession *vscsession,
					VscAction action, const char *filepath);
static int _clmv_vscsession_startStreamScan(VscSession *vscsession,
					VscAction action, VscStreamFile **sfile);
static int _clmv_vscsession_writeStreamBytes(VscSession *vscsession,
					VscStreamFile *sfile,
					const char *bytes, apr_size_t len);
static int _clmv_vscsession_endStreamScan(VscSession *vscsession,
					VscStreamFile *sfile);
static void _clmv_vscsession_abortStreamScan(VscSession *vscsession,
					VscStreamFile *sfile);

static int _vsc_clmv_connect(apr_socket_t** socket, const char* host, apr_port_t port, apr_pool_t* pool);
static int _vsc_clmv_data_send(apr_socket_t *socket, const char* data, apr_size_t len);
/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
/* 構造体の関数へのポインタを初期化する関数群のプロトタイプ */
static void _init_clmv_vscds_method(VscDataSource *vscds);
static void _init_clmv_vscsession_method(VscSession *vscsession);

/* プロバイダフレームワークライブラリからの要請で生成する関数群のプロトタイプ */
#if defined(DIVY_LIB_DYNAMIC_EXPORT) && defined(DIVY_USE_GNU_LD)
static void _vsc_clmv_init(void) __attribute__ ((constructor));
static void _vsc_clmv_fini(void) __attribute__ ((destructor));
#endif	/* DIVY_LIB_DYNAMIC_EXPORT && DIVY_USE_GNU_LD */
static int _vsc_clmv_init_func(apr_pool_t *pconf);

/* プロバイダフレームワークに渡される関数群のプロトタイプ */
static int _vsc_clmv_create_vscdatasource(apr_pool_t *p,
						const char *providerType,
						VscDataSource **vscds);
/*------------------------------------------------------------------------------
  Define provider library hook structure
  ----------------------------------------------------------------------------*/
/*
 * プロバイダ管理機構にClamAV ウィルス検索プロバイダを登録する
 * (note)
 * 	ウィルス検索プロバイダの第1引数は常に"vsc"です。(文字列リテラルにはしない)
 * 	第2引数には、プロバイダの種類を表す名称を入れること。define 値は絶対にNG
 */
DIVY_PROVIDER_HOOK_STRUCT(vsc,clmv,
		_vsc_clmv_init_func,		/* 初期化ハンドラ */
		divy_lib_nop_fini_func);	/* 終了ハンドラ(nop) */

/*------------------------------------------------------------------------------
  Define provider hook function
  ----------------------------------------------------------------------------*/
static int _clmv_vscds_canLoadVirusData(VscDataSource *vscds)
{
	DIVY_VSC_INIT_STATUS(vscds);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscds, DIVY_VSC_ST_OK);
	DIVY_VSC_SET_STATUS_NCD(vscds, "");
	DIVY_VSC_SET_STATUS_MSG(vscds, "");

	return DIVY_VSC_ST_OK;	/* サポートしている */
}

static int _clmv_vscds_loadVirusData(VscDataSource *vscds,
					const VscProperty *vscprop)
{
	DIVY_VSC_INIT_STATUS(vscds);

	/* ##### FIXME 足りない実装を */
	/* ステータスコードを設定 */

	return DIVY_VSC_ST_OK;
}

static VscSession * _clmv_vscds_getVscSession(VscDataSource *vscds,
					const VscProperty *vscprop, apr_pool_t *p)
{
	VscSession *vscsession = NULL;
	apr_socket_t	*socket = NULL;

	/* セッションの生成 */
	vscsession = apr_pcalloc(p, sizeof(VscSession));
	vscsession->__p = p;
	vscsession->vscscntxt = apr_pcalloc(p, sizeof(VscSessionCntxt));

	/* 関数の実体を登録する */
	_init_clmv_vscsession_method(vscsession);

	/* ステータスメッセージの初期化 */
	DIVY_VSC_NEW_STATUS(vscsession, p);

	/* 引数チェック */
	if (vscprop == NULL || vscds == NULL) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_INVALID_PARAMS);
		DIVY_VSC_SET_STATUS_NCD(vscsession, "");
		DIVY_VSC_SET_STATUS_MSG(vscsession, "Invalid arguments.");

		return NULL;
	}
	vscsession->vscprop = (VscProperty *) vscprop;

	/* コントロールポート(daemon)へ接続する */
	if (_vsc_clmv_connect(&socket, vscprop->hostname,
							(apr_port_t)vscprop->port, p) != DIVY_VSC_ST_OK) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_NO_CONNECTED);
		DIVY_VSC_SET_STATUS_NCD(vscsession, "");
		DIVY_VSC_SET_STATUS_MSG(vscsession, "connection failed to clamd.clamd daemon not found.");
		return NULL;
	}

	vscsession->vscscntxt->control_socket = socket;
	return vscsession;
}

static int _clmv_vscds_canScanStream(VscDataSource *vscds)
{
	DIVY_VSC_INIT_STATUS(vscds);

	DIVY_VSC_SET_STATUS_CD(vscds, DIVY_VSC_ST_SUPPORTED_STREAM);
	DIVY_VSC_SET_STATUS_NCD(vscds, 0);
	DIVY_VSC_SET_STATUS_MSG(vscds, "");

	/* ストリームベースの検索APIをサポートです */
	return DIVY_VSC_ST_SUPPORTED_STREAM;
}

static int _clmv_vscsession_canCacheSession(VscSession *vscsession)
{
	DIVY_VSC_INIT_STATUS(vscsession);

	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_SUPPORTED_CACHE);

	/* キャッシュはサポートしています */
	return DIVY_VSC_ST_SUPPORTED_CACHE;
}

static void _clmv_vscsession_closeSession(VscSession *vscsession)
{

	apr_status_t	st = -1;
	DIVY_VSC_INIT_STATUS(vscsession);

	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_ERR);
	DIVY_VSC_SET_STATUS_NCD(vscsession, 0);
	DIVY_VSC_SET_STATUS_MSG(vscsession, "Failed to close clamAV daemon connection.");

	if (vscsession == NULL && vscsession->vscscntxt == NULL) return;



	/*
	 * コントロールポートを閉じる。エラーを調べる必要はない。
	 */
	if (vscsession->vscscntxt->control_socket != NULL) {
		st = apr_socket_close(vscsession->vscscntxt->control_socket);
	}

	return;
}

static int _clmv_vscsession_scanFile(VscSession *vscsession,
					VscAction action, const char *filepath)
{
	return DIVY_VSC_ST_ERR;	/* 取り敢えずのステータス */
}

static int _clmv_vscsession_startStreamScan(VscSession *vscsession,
					VscAction action, VscStreamFile **sfile)
{
	apr_socket_t	*socket = NULL;
	apr_socket_t	*dsocket = NULL;
	apr_status_t	st;
	char			buffer[256];
	apr_size_t		len, cmdlen;
	char			*sp = NULL;
	char			*str_port;
	int				data_port = -1;

	*sfile = NULL;	/* 初期化しておく */
	*sfile = apr_pcalloc(vscsession->__p, sizeof(VscStreamFile));

	DIVY_VSC_INIT_STATUS(vscsession);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_NO_CONNECTED);
	DIVY_VSC_SET_STATUS_NCD(vscsession, "");
	DIVY_VSC_SET_STATUS_MSG(vscsession, "failed to start stream.");

	if (vscsession->vscscntxt->control_socket == NULL) {
		DIVY_VSC_SET_STATUS_MSG(vscsession, "losted clamd control socket.");
		return DIVY_VSC_ST_ERR;
	}

	/* コントロールポートを通じてデーモンよりデータポートを取得する */
	socket = vscsession->vscscntxt->control_socket;
	cmdlen = strlen(CLAMAV_CMD_STREAM);
	if (_vsc_clmv_data_send(socket, CLAMAV_CMD_STREAM, cmdlen) != DIVY_VSC_ST_OK) {
		/* 送信失敗 */
		DIVY_VSC_SET_STATUS_MSG(vscsession, "It was not present.... stage1 (present: dataport)");
		return DIVY_VSC_ST_NO_CONNECTED;
	}

	memset(buffer, 0, sizeof(buffer));
	len = sizeof(buffer);
	st = apr_socket_recv(socket, buffer, &len);

	if (st != APR_SUCCESS) {
		DIVY_VSC_SET_STATUS_MSG(vscsession, "It was not present.... stage2 (present: dataport)");
		return DIVY_VSC_ST_NO_CONNECTED;
	}

	/* コントロールポートから戻ってくる文字列による新しいセッション
	 * を作成する
	 * サンプル
	 * PORT XXXX
	 */
	sp = divy_strstr(buffer, "PORT ");
	if (sp == NULL) {
		DIVY_VSC_SET_STATUS_MSG(vscsession, "It was not present.... stage3 (present: dataport)");
		return DIVY_VSC_ST_NO_CONNECTED;
	}

	str_port = apr_pstrdup(vscsession->__p, &buffer[5]);
	data_port = atoi(str_port);

	/* データポートが戻ってこない場合はエラーとする */
	if (data_port == -1) {
		DIVY_VSC_SET_STATUS_MSG(vscsession, "It was not present.... stage4 (present: dataport)");
		return DIVY_VSC_ST_NO_CONNECTED;
	}

	/* データポートへの接続 */
	if (DIVY_VSC_ST_OK != _vsc_clmv_connect(&dsocket,
											vscsession->vscprop->hostname,
											(apr_port_t)data_port, vscsession->__p)) {
		DIVY_VSC_SET_STATUS_MSG(vscsession, 
						apr_psprintf(vscsession->__p,
							"connection failed to dataport.(%d)", data_port));
		return DIVY_VSC_ST_ERR;
	}
	vscsession->vscscntxt->data_socket = dsocket;

	/* 成功 */
	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_OK);
	DIVY_VSC_SET_STATUS_NCD(vscsession, 0);
	DIVY_VSC_SET_STATUS_MSG(vscsession, "");

	return DIVY_VSC_ST_OK;
}

static int _clmv_vscsession_writeStreamBytes(VscSession *vscsession,
					VscStreamFile *sfile,
					const char *bytes, apr_size_t len)
{
	apr_status_t	st;

	DIVY_VSC_INIT_STATUS(vscsession);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_ERR);
	DIVY_VSC_SET_STATUS_NCD(vscsession, "");
	DIVY_VSC_SET_STATUS_MSG(vscsession, "");

	st = apr_socket_send(vscsession->vscscntxt->data_socket, bytes, &len);
	if (st != APR_SUCCESS) {
		return DIVY_VSC_ST_ERR;
	}

	/* 成功 */
	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_OK);
	DIVY_VSC_SET_STATUS_NCD(vscsession, 0);
	DIVY_VSC_SET_STATUS_MSG(vscsession, "");

	return DIVY_VSC_ST_OK;
}

static int _clmv_vscsession_endStreamScan(VscSession *vscsession,
					VscStreamFile *sfile)
{
	int 			ret = DIVY_VSC_ST_ERR;
	apr_status_t	st;
	apr_pool_t		*p  = vscsession->__p;

	DIVY_VSC_INIT_STATUS(vscsession);

	/* データポートを閉じる */
	if (vscsession->vscscntxt->data_socket != NULL) {
		st = apr_socket_close(vscsession->vscscntxt->data_socket);
		/* 閉じるのが失敗しても突き進む */
	}

	if (vscsession->vscscntxt->control_socket != NULL) {
		apr_socket_t	*socket = vscsession->vscscntxt->control_socket;
		char			buff[255] = {0};
		apr_size_t		len = sizeof(buff);

		if ((st = apr_socket_recv(socket, buff, &len)) == APR_SUCCESS) {
			if (divy_strstr(buff, "FOUND\n")) {
				DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_INFECTED);
				DIVY_VSC_SET_STATUS_NCD(vscsession, 0);
				DIVY_VSC_SET_STATUS_MSG(vscsession,
								apr_psprintf(p, "clamd returnd %s.", buff));
				/* ウィルスが見つかった */
				ret = DIVY_VSC_ST_INFECTED;
			}
			else if (divy_strstr(buff, "ERROR\n")){
				DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_ERR);
				DIVY_VSC_SET_STATUS_NCD(vscsession, 0);
				DIVY_VSC_SET_STATUS_MSG(vscsession, 
								apr_psprintf(p, "clamd returnd %s.", buff));
				ret = DIVY_VSC_ST_ERR;
			}
			else {
				DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_OK);
				DIVY_VSC_SET_STATUS_NCD(vscsession, 0);
				DIVY_VSC_SET_STATUS_MSG(vscsession, "");
				ret = DIVY_VSC_ST_OK;
			}
		}
		else {
			DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_UNSUPPORTED_STREAM);
			DIVY_VSC_SET_STATUS_NCD(vscsession, apr_psprintf(p, "%d", st));
			DIVY_VSC_SET_STATUS_MSG(vscsession, "failed to clamd control port.");
		}

	}


	return ret; 
}

static void _clmv_vscsession_abortStreamScan(VscSession *vscsession,
					VscStreamFile *sfile)
{
	DIVY_VSC_INIT_STATUS(vscsession);

	/* 何もすることなし */
	return;
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * VscDataSource の中で宣言された関数へのポインタに実体を提供する。
 *
 * @param vscds VscDataSource *
 */
static void _init_clmv_vscds_method(VscDataSource *vscds)
{
	vscds->canLoadVirusData = _clmv_vscds_canLoadVirusData;
	vscds->loadVirusData    = _clmv_vscds_loadVirusData;
	vscds->getVscSession    = _clmv_vscds_getVscSession;
	vscds->canScanStream    = _clmv_vscds_canScanStream;
}

/**
 * VscSession の中で宣言された関数へのポインタに実体を提供する。
 *
 * @param vscsession VscSession *
 */
static void _init_clmv_vscsession_method(VscSession *vscsession)
{
	vscsession->canCacheSession  = _clmv_vscsession_canCacheSession;
	vscsession->closeSession     = _clmv_vscsession_closeSession;
	vscsession->scanFile         = _clmv_vscsession_scanFile;
	vscsession->startStreamScan  = _clmv_vscsession_startStreamScan;
	vscsession->writeStreamBytes = _clmv_vscsession_writeStreamBytes;
	vscsession->endStreamScan    = _clmv_vscsession_endStreamScan;
	vscsession->abortStreamScan  = _clmv_vscsession_abortStreamScan;
}

/*-----------------------------------------------------------------------------
  Define provider library handler
 *---------------------------------------------------------------------------*/
#if defined(DIVY_LIB_DYNAMIC_EXPORT)
/**
 * スタートアップハンドラ (コンストラクタ関数)
 * (note)
 * 	このハンドラはdlopen 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	コールの直前に呼び出されます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlopen された時だけ実施する特別な初期化処理
 * 	    * 関数divy_register_lib_provider のコール
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _vsc_clmv_init(void)
#else
void _init(void)
#endif	/* DIVY_USE_GNU_LD */
{
	/* プロバイダライブラリ構造体の登録 */
	divy_register_lib_provider(&DIVY_PROVIDER_HOOK_STRUCT_NAME(vsc,clmv));
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		CLMV_VSC_PROVIDER_TYPE"(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
}

/**
 * 終了ハンドラ (デストラクタ関数)
 * (note)
 * 	このハンドラはdlclose 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	が終了した後、コールされます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlclose された時だけ実施する特別な終了処理
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _vsc_clmv_fini(void)
#else
void _fini(void)
#endif	/* DIVY_USE_GNU_LD */
{
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		CLMV_VSC_PROVIDER_TYPE"(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return;
}
#endif	/* DIVY_LIB_DYNAMIC_EXPORT */

/**
 * DIVY_PROVIDER_HOOK_STRUCT の初期化ハンドラに登録された関数
 *
 * @pconf apr_pool_t * コンフィグプール
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
static int _vsc_clmv_init_func(apr_pool_t *pconf)
{
	/* create_vscdatasource ハンドラへの登録 */
	divy_hook_create_vscdatasource(_vsc_clmv_create_vscdatasource,
					NULL, NULL, APR_HOOK_MIDDLE);
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		CLMV_VSC_PROVIDER_TYPE"(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return DIVY_LIB_ST_OK;
}

/**
 * 指定されたproviderType が自身のプロバイダを示していた時、
 * 自身のデータソースオブジェクトを生成して返却する。
 *
 * @param p apr_pool_t *
 * @param providerType const char * プロバイダタイプ
 * @param vscds VscDataSource ** 作成したVscDataSource構造体
 * @return
 * 	VSC_OK       : providerType が示すプロバイダは自分自身、かつ処理が成功した
 * 	VSC_DECLINED : providerType が示すプロバイダは自分ではなかった
 */
static int _vsc_clmv_create_vscdatasource(apr_pool_t *p,
						const char *providerType,
						VscDataSource **vscds)
{
	*vscds = NULL;	/* 初期化 */

	/* ステータスメッセージの初期化 */
	DIVY_VSC_NEW_STATUS(*vscds, p);

	/*
	 * providerType は自分を示しているか？
	 */
	if (providerType == NULL ||
	    strcmp(providerType, CLMV_VSC_PROVIDER_TYPE) != 0) {
		return VSC_DECLINED;
	}

	/*
	 * データソースの生成
	 */
	*vscds = apr_pcalloc(p, sizeof(VscDataSource));
	(*vscds)->__p         = p;
	(*vscds)->type        = CLMV_VSC_PROVIDER_TYPE;
	(*vscds)->loaded_time = APR_TIME_C(0);

	/* 関数の実体を登録する */
	_init_clmv_vscds_method(*vscds);

	return VSC_OK;
}

/**
 * 指定したポートに接続する
 * @param	socket	apr_socket_t**
 * @param	host	const char*
 * @param	port	apr_port_t
 * @param	pool	apr_pool_t*
 * @return
 * 	DIVY_VSC_ST_OK  : 正常
 * 	DIVY_VSC_ST_ERR : 失敗
 * 	ここでエラーを設定はしません。必ず呼び出し元でエラーが起きた原因を報告してください。
 */
static int _vsc_clmv_connect(apr_socket_t** socket, const char* host, apr_port_t port, apr_pool_t* pool)
{

	apr_status_t	st;
	apr_socket_t	*s = NULL;
    apr_sockaddr_t  *sockaddr_dest  = NULL;

	if (host == NULL || *host == '\0') return DIVY_VSC_ST_INVALID_PARAMS;

	st = apr_sockaddr_info_get(&sockaddr_dest, host,
							   APR_UNSPEC, port, 0, pool);
	if (st != APR_SUCCESS) return DIVY_VSC_ST_NO_CONNECTED;

	/* ソケットの作成  */
	st = apr_socket_create(&s, sockaddr_dest->family, SOCK_STREAM, APR_PROTO_TCP, pool);

	if (st != APR_SUCCESS) return DIVY_VSC_ST_NO_CONNECTED;

	/* 接続する */
	st = apr_socket_connect(s, sockaddr_dest);

	if (st != APR_SUCCESS) return DIVY_VSC_ST_NO_CONNECTED;
	*socket = s;

	return DIVY_VSC_ST_OK;
}

/**
 * ソケットに対してデータを送信する
 *
 * @param 	socket	apr_socket_t*		ソケット
 * @param	data	const char*			データ長(\0で終わる必要はない）
 * @param	len		apr_size_t			dataの長さ
 * @return 
 *  DIVY_VSC_ST_OK	:	成功
 *  DIVY_VSC_ST_ERR	:	失敗
 * 	ここでエラーを設定はしません。必ず呼び出し元でエラーが起きた原因を報告してください。
 */
static int _vsc_clmv_data_send(apr_socket_t *socket, const char* data, apr_size_t len)
{
	apr_status_t	st;

	if (socket == NULL) return DIVY_VSC_ST_ERR;
	if (data == NULL || *data == '\0') return DIVY_VSC_ST_ERR;

	st = apr_socket_send(socket, data, &len);

	return (st != APR_SUCCESS) ? DIVY_VSC_ST_ERR : DIVY_VSC_ST_OK;
}

/**
 * ソケットよりデータを取得する
 *
 * @param	socket	apr_socket_t*
 * @return
 *  DIVY_VSC_ST_OK  : 成功
 *  DIVY_VSC_ST_ERR : 失敗
 */
/*
static int _vsc_clmv_data_recv(apr_socket_t *socket, char** buffer, int **len, apr_pool_t* pool)
{
	char	tmp[254];
	int		tmplen = 1;
	if (socket == NULL) return DIVY_VSC_ST_ERR;

	apr_socket_recv(socket, &tmp, &tmplen);
	*buffer = apr_pstrdup(pool, tmp);

	return DIVY_VSC_ST_OK;
}
*/

