/**
 * $Id$
 * 
 * tf_vsc_fsec.c
 *
 * F-Secure AntiVirus Engine 用ウィルス検索プロバイダの実装。
 *
 * 2004/10/05 Tue takehara NEW
 */
#include "apr.h"
#include "apr_strings.h"
#include "apr_pools.h"
#include "apr_time.h"
#include "apr_thread_proc.h"
#include "apr_file_info.h"
#include "apr_file_io.h"

#if defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H
#include <unistd.h>
#endif	/* defined(DIVY_DEBUG_PROVIDER_LOADER) && APR_HAVE_UNISTD_H */

#include "util_common.h"
#include "tf_provider.h"

/* 各種構造体の定義を見えるようにする */
#define DIVY_VSC_PROVIDER_PRIVATE
#include "tf_vscan.h"

/*------------------------------------------------------------------------------
  Define fixed value and macro
  ----------------------------------------------------------------------------*/
/**
 * F-Secure 用ウィルス検索プロバイダの種類を表す文字列
 *
 * [ フレームワークからの要請 ]
 * 	この名称は同一種のプロバイダ間で一意の名称でなければなりません。
 * 	可能であれば、全てのプロバイダ間で一意に識別できる名称であるべき。
 */
#define FSEC_VSC_PROVIDER_TYPE	"fsec"

/* F-Secure 依存 */

/**
 * F-Secure ウィルス検索エンジンがウィルス検索した時に返却するステータスコード
 */
#define FSEC_ST_VIRUS_DETECTION	3

/**
 * F-Secure ウィルス検索エンジンが正常に返却するステータスコード
 */
#define FSEC_ST_NORMAL	0

/**
 * F-Secure デフォルト検索用コマンドのパス
 */
#define FSEC_DEFAULT_CMD	"/usr/bin/fsav"

/**
 * F-Secure ウィルス検索エンジンが認識可能な最大サイズ(byte)
 */
#define FSEC_MAXFILE_SIZE	APR_INT64_C(2147483647)

#if 0
/**
 * ZIPファイルのマジック番号
 */
#define FSEC_ZIP_MAGIC	0x4b50

/**
 * ZIPのGeneral Purpose bit の位置
 */
#define FSEC_GENERAL_PURPOSE_BIT	6

/**
 * ZIPにパスワードが付いているかどうかを表すビット値
 * (FSEC_GENERAL_PURPOSE_BIT の位置にあるデータを見る)
 */
#define FSEC_FLG_ENCRYPTED	0x01
#endif

/*------------------------------------------------------------------------------
  Define incomplete type structure
  ----------------------------------------------------------------------------*/
/* (note) このプロバイダは不完全型のコンテキストを利用しませんでした。*/


/*------------------------------------------------------------------------------
  Declare provider hook function
  ----------------------------------------------------------------------------*/
static int _fsec_vscds_canLoadVirusData(VscDataSource *vscds);
static int _fsec_vscds_loadVirusData(VscDataSource *vscds,
					const VscProperty *vscprop);
static VscSession * _fsec_vscds_getVscSession(VscDataSource *vscds,
					const VscProperty *vscprop, apr_pool_t *p);
static int _fsec_vscds_canScanStream(VscDataSource *vscds);
static int _fsec_vscsession_canCacheSession(VscSession *vscsession);
static void _fsec_vscsession_closeSession(VscSession *vscsession);
static int _fsec_vscsession_scanFile(VscSession *vscsession,
					VscAction action, const char *filepath);
static int _fsec_vscsession_startStreamScan(VscSession *vscsession,
					VscAction action, VscStreamFile **sfile);
static int _fsec_vscsession_writeStreamBytes(VscSession *vscsession,
					VscStreamFile *sfile,
					const char *bytes, apr_size_t len);
static int _fsec_vscsession_endStreamScan(VscSession *vscsession,
					VscStreamFile *sfile);
static void _fsec_vscsession_abortStreamScan(VscSession *vscsession,
					VscStreamFile *sfile);

/*------------------------------------------------------------------------------
  Declare private functions
  ----------------------------------------------------------------------------*/
static char * _errcode2msg(apr_pool_t *p, int code);

/* 構造体の関数へのポインタを初期化する関数群のプロトタイプ */
static void _init_fsec_vscds_method(VscDataSource *vscds);
static void _init_fsec_vscsession_method(VscSession *vscsession);

/* プロバイダフレームワークライブラリからの要請で生成する関数群のプロトタイプ */
#if defined(DIVY_LIB_DYNAMIC_EXPORT) && defined(DIVY_USE_GNU_LD)
static void _vsc_fsec_init(void) __attribute__ ((constructor));
static void _vsc_fsec_fini(void) __attribute__ ((destructor));
#endif	/* DIVY_LIB_DYNAMIC_EXPORT && DIVY_USE_GNU_LD */
static int _vsc_fsec_init_func(apr_pool_t *pconf);

/* プロバイダフレームワークに渡される関数群のプロトタイプ */
static int _vsc_fsec_create_vscdatasource(apr_pool_t *p,
						const char *providerType,
						VscDataSource **vscds);
/*------------------------------------------------------------------------------
  Define provider library hook structure
  ----------------------------------------------------------------------------*/
/*
 * プロバイダ管理機構にF-Secureウィルス検索プロバイダを登録する
 * (note)
 * 	ウィルス検索プロバイダの第1引数は常に"vsc"です。(文字列リテラルにはしない)
 * 	第2引数には、プロバイダの種類を表す名称を入れること。define 値は絶対にNG
 */
DIVY_PROVIDER_HOOK_STRUCT(vsc,fsec,
		_vsc_fsec_init_func,		/* 初期化ハンドラ */
		divy_lib_nop_fini_func);	/* 終了ハンドラ(nop) */

/*------------------------------------------------------------------------------
  Define provider hook function
  ----------------------------------------------------------------------------*/
static int _fsec_vscds_canLoadVirusData(VscDataSource *vscds)
{
	DIVY_VSC_INIT_STATUS(vscds);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscds, DIVY_VSC_ST_UNSUPPORTED_STREAM);
	DIVY_VSC_SET_STATUS_NCD(vscds, "");
	DIVY_VSC_SET_STATUS_MSG(vscds, "Unsupported virus pattern loading.");

	return DIVY_VSC_ST_UNSUPPORTED_LOAD;	/* 未サポート */
}

static int _fsec_vscds_loadVirusData(VscDataSource *vscds,
					const VscProperty *vscprop)
{
	DIVY_VSC_INIT_STATUS(vscds);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscds, DIVY_VSC_ST_UNSUPPORTED_STREAM);
	DIVY_VSC_SET_STATUS_NCD(vscds, "");
	DIVY_VSC_SET_STATUS_MSG(vscds, "Unsupported virus pattern loading.");

	return DIVY_VSC_ST_OK;	/* だが正常扱い(取り決めです) */
}

static VscSession * _fsec_vscds_getVscSession(VscDataSource *vscds,
					const VscProperty *vscprop, apr_pool_t *p)
{
	VscSession *vscsession = NULL;

	/* セッションの生成 */
	vscsession = apr_pcalloc(p, sizeof(VscSession));
	vscsession->__p = p;

	/* 関数の実体を登録する */
	_init_fsec_vscsession_method(vscsession);

	/* ステータスメッセージの初期化 */
	DIVY_VSC_NEW_STATUS(vscsession, p);

	/* 引数チェック */
	if (vscprop == NULL || vscds == NULL) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_INVALID_PARAMS);
		DIVY_VSC_SET_STATUS_NCD(vscsession, "");
		DIVY_VSC_SET_STATUS_MSG(vscsession, "Invalid arguments.");

		return NULL;
	}
	vscsession->vscprop = (VscProperty *) vscprop;

	return vscsession;
}

static int _fsec_vscds_canScanStream(VscDataSource *vscds)
{
	DIVY_VSC_INIT_STATUS(vscds);

	DIVY_VSC_SET_STATUS_CD(vscds, DIVY_VSC_ST_UNSUPPORTED_STREAM);

	/* ストリームベースの検索APIは未サポートです */
	return DIVY_VSC_ST_UNSUPPORTED_STREAM;
}

static int _fsec_vscsession_canCacheSession(VscSession *vscsession)
{
	DIVY_VSC_INIT_STATUS(vscsession);

	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_SUPPORTED_CACHE);

	/* キャッシュはサポートしています(意味ないけど) */
	return DIVY_VSC_ST_SUPPORTED_CACHE;
}

static void _fsec_vscsession_closeSession(VscSession *vscsession)
{
	DIVY_VSC_INIT_STATUS(vscsession);

	/* 特にすることは無いはず */
	return;
}

static int _fsec_vscsession_scanFile(VscSession *vscsession,
					VscAction action, const char *filepath)
{
	apr_status_t rv;
	int ret;
	char *msg;
	int exitcode = FSEC_ST_NORMAL;
	int exceptcd_set[] = {FSEC_ST_VIRUS_DETECTION, FSEC_ST_NORMAL};
	apr_file_t *fd    = NULL;
	apr_finfo_t finfo = { 0 };

	DIVY_VSC_INIT_STATUS(vscsession);

	/* 引数チェック */
	if (vscsession == NULL || IS_EMPTY(filepath)) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_INVALID_PARAMS);
		DIVY_VSC_SET_STATUS_NCD(vscsession, "");
		DIVY_VSC_SET_STATUS_MSG(vscsession, "filepath is EMPTY.");

		return DIVY_VSC_ST_ERR;
	}

	/* 現状、DIVY_VSC_ACT_NOTICE 以外の動作はサポートしていない */
	if (action != DIVY_VSC_ACT_NOTICE) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_UNSUPPORTED_ACTION);
		DIVY_VSC_SET_STATUS_NCD(vscsession, "");
		msg = apr_psprintf(vscsession->__p,
				"Unsupported action specified.(action = %d)", action);
		DIVY_VSC_SET_STATUS_MSG(vscsession, msg);

		return DIVY_VSC_ST_ERR;

	}

	/* ファイルサイズの取得 */
	rv = apr_file_open(&fd, filepath, APR_READ | APR_BINARY, 0, vscsession->__p);
	if (rv != APR_SUCCESS) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_CANNOT_OPEN);

		msg = apr_psprintf(vscsession->__p, "%d", rv);
		DIVY_VSC_SET_STATUS_NCD(vscsession, msg);

		msg = apr_psprintf(vscsession->__p,
				"Cannot open file.(code = %d, filepath = %s)",
				rv, filepath);
		DIVY_VSC_SET_STATUS_MSG(vscsession, msg);

		return DIVY_VSC_ST_ERR;
	}

	rv = apr_file_info_get(&finfo, APR_FINFO_SIZE, fd);
	if (rv != APR_SUCCESS) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_CANNOT_READ_FILEST);

		msg = apr_psprintf(vscsession->__p, "%d", rv);
		DIVY_VSC_SET_STATUS_NCD(vscsession, msg);

		msg = apr_psprintf(vscsession->__p,
				"Failed to get file length.(code = %d, filepath = %s)",
				rv, filepath);
		DIVY_VSC_SET_STATUS_MSG(vscsession, msg);

		if (fd != NULL) apr_file_close(fd); fd = NULL;

		return DIVY_VSC_ST_ERR;

	}

	/* 最大サイズを超えているか？ */
#if 0
	/* supported --skipleage option. F-Secure version 4.63 or later */

	if (finfo.size > FSEC_MAXFILE_SIZE) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_TOOLARGE_CONTENT);
		DIVY_VSC_SET_STATUS_NCD(vscsession, "");
		msg = apr_psprintf(vscsession->__p,
				"This file was exceed the maximum size limit of "
				"F-Secure AntiVirus Engine.(size = %"APR_INT64_T_FMT")",
				finfo.size);
		DIVY_VSC_SET_STATUS_MSG(vscsession, msg);

		if (fd != NULL) apr_file_close(fd); fd = NULL;

		return DIVY_VSC_ST_ERR;
	}
#endif

#if 0
	/* zip ファイルなのか? */
	if (finfo.size > 8) {
		unsigned char zip_fhdr[8] = { 0 };
		int encrypted = 0;	/* パスワード付きかどうか */
		apr_size_t len = 8;

		/* ヘッダを読む */
		rv = apr_file_read(fd, (char *)zip_fhdr, &len);
		if (rv == APR_SUCCESS && len == 8) {
			unsigned short zip_magic = (unsigned short) zip_fhdr[0];
			zip_magic |= (unsigned short) zip_fhdr[1] << 8;
			if (zip_magic == FSEC_ZIP_MAGIC) {
				/* パスワード付きか? */
				encrypted = zip_fhdr[FSEC_GENERAL_PURPOSE_BIT] & FSEC_FLG_ENCRYPTED;
			}
			LERRLOG3(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"zip_magic = 0x%x, zip_fhdr[6] = %d, zip_fhdr[4] = %d",
				zip_magic, zip_fhdr[6], zip_fhdr[4]);
		}

		if (fd != NULL) apr_file_close(fd); fd = NULL;

		if (encrypted) {
			LERRLOG0(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
				"Skip virus check.");
			return DIVY_VSC_ST_OK;
		}
	}
#endif

	if (fd != NULL) apr_file_close(fd); fd = NULL;

	/* 引数の作成 */
	if (vscsession->vscprop->argv != NULL) {
		int i, cnt = 0;
		const char **argv = NULL;
		char **src_argv = (char **) vscsession->vscprop->argv;

		/* 引数の数を数えながら
		 * サポートされていないオプションがないかどうか調べる */
		for (i = 0; src_argv[i] != NULL; i++) {
			/* 禁止オプション"--delete", "--disinf", "--rename" */
			if (divy_strstr(src_argv[i], "--delete") != NULL ||
			    divy_strstr(src_argv[i], "--disinf") != NULL ||
			    divy_strstr(src_argv[i], "--rename") != NULL) {

				DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_UNSUPPORTED_ACTION);
				DIVY_VSC_SET_STATUS_NCD(vscsession, "");
				msg = apr_psprintf(vscsession->__p,
					"Unsupported argument specified.(arg = \"%s\")",
					src_argv[i]);
				DIVY_VSC_SET_STATUS_MSG(vscsession, msg);

				return DIVY_VSC_ST_ERR;
			}
			cnt++;	/* 数える */
		}

		argv = apr_pcalloc(vscsession->__p, sizeof(char *) * (cnt + 2));
		for (i = 0; i < cnt; i++) {
			argv[i] = src_argv[i];
		}
		argv[cnt]     = filepath;
		argv[cnt + 1] = NULL;

		/* コマンドの実行 */
		ret = divy_execute_cmd(vscsession->__p, argv[0], argv,
				vscsession->vscprop->env, APR_PROGRAM,
				NULL, &exitcode, exceptcd_set);
	}
	else {
		/* コマンドラインが省略された場合 */
		const char *sargv[] = { FSEC_DEFAULT_CMD, "--dumb=on",
					"--silent=on", "--archive=on",
					filepath, NULL };

		/* コマンドの実行 */
		ret = divy_execute_cmd(vscsession->__p, sargv[0], sargv,
				vscsession->vscprop->env, APR_PROGRAM,
				NULL, &exitcode, exceptcd_set);
	}

	msg = _errcode2msg(vscsession->__p, ret);

	/* ウィルス感染を検出した */
	if (ret == DIVY_UTIL_COMMON_EXCEPTCD_SET &&
	    exitcode == FSEC_ST_VIRUS_DETECTION) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_INFECTED);
		DIVY_VSC_SET_STATUS_NCD(vscsession,
				apr_psprintf(vscsession->__p, "%d", ret));
		DIVY_VSC_SET_STATUS_MSG(vscsession, msg);

		return DIVY_VSC_ST_ERR;
	}
	else if (ret) {
		DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_ERR);
		DIVY_VSC_SET_STATUS_NCD(vscsession,
				apr_psprintf(vscsession->__p, "%d", ret));
		DIVY_VSC_SET_STATUS_MSG(vscsession, msg);

		return DIVY_VSC_ST_ERR;
	}

	return DIVY_VSC_ST_OK;
}

static int _fsec_vscsession_startStreamScan(VscSession *vscsession,
					VscAction action, VscStreamFile **sfile)
{
	*sfile = NULL;	/* 初期化しておく */

	DIVY_VSC_INIT_STATUS(vscsession);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_UNSUPPORTED_STREAM);
	DIVY_VSC_SET_STATUS_NCD(vscsession, "");
	DIVY_VSC_SET_STATUS_MSG(vscsession, "Unsupported stream-based method.");

	/* このコールはエラー */
	return DIVY_VSC_ST_ERR;
}

static int _fsec_vscsession_writeStreamBytes(VscSession *vscsession,
					VscStreamFile *sfile,
					const char *bytes, apr_size_t len)
{
	DIVY_VSC_INIT_STATUS(vscsession);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_UNSUPPORTED_STREAM);
	DIVY_VSC_SET_STATUS_NCD(vscsession, "");
	DIVY_VSC_SET_STATUS_MSG(vscsession, "Unsupported stream-based method.");

	/* このコールはエラー */
	return DIVY_VSC_ST_ERR;
}

static int _fsec_vscsession_endStreamScan(VscSession *vscsession,
					VscStreamFile *sfile)
{
	DIVY_VSC_INIT_STATUS(vscsession);

	/* ステータスコードを設定 */
	DIVY_VSC_SET_STATUS_CD(vscsession, DIVY_VSC_ST_UNSUPPORTED_STREAM);
	DIVY_VSC_SET_STATUS_NCD(vscsession, "");
	DIVY_VSC_SET_STATUS_MSG(vscsession, "Unsupported stream-based method.");

	/* このコールはエラー */
	return DIVY_VSC_ST_ERR;
}

static void _fsec_vscsession_abortStreamScan(VscSession *vscsession,
					VscStreamFile *sfile)
{
	DIVY_VSC_INIT_STATUS(vscsession);

	/* 何もすることなし */
	return;
}

/*------------------------------------------------------------------------------
  Define private functions
  ----------------------------------------------------------------------------*/
/**
 * F-Secure のウィルス検索コマンドが出力したエラーコードをメッセージに変換して
 * 返却する。
 *
 */
static char * _errcode2msg(apr_pool_t *p, int code)
{
	char *msg = "";

	switch (code) {
		case 0:
			msg = "";
			break;
		case 1:
			msg = "Fatal error; unrecoverable error.";
			break;
		case 3:
			msg = "File virus found.";
			break;
		case 6:
			msg = "At least one virus was removed and "
				"no infected files left.";
			break;
		case 7:
			msg = "Out of memory.";
			break;
		case 8:
			msg = "Suspicious files found; "
				"these are not necessarily infected by a virus.";
			break;
		case 9:
			msg = "Scan failed.";
			break;
		case 130:
			msg = "Program was terminated by a sigterm or suspend event.";
			break;
	}

	return msg;
}

/**
 * VscDataSource の中で宣言された関数へのポインタに実体を提供する。
 *
 * @param vscds VscDataSource *
 */
static void _init_fsec_vscds_method(VscDataSource *vscds)
{
	vscds->canLoadVirusData = _fsec_vscds_canLoadVirusData;
	vscds->loadVirusData    = _fsec_vscds_loadVirusData;
	vscds->getVscSession    = _fsec_vscds_getVscSession;
	vscds->canScanStream    = _fsec_vscds_canScanStream;
}

/**
 * VscSession の中で宣言された関数へのポインタに実体を提供する。
 *
 * @param vscsession VscSession *
 */
static void _init_fsec_vscsession_method(VscSession *vscsession)
{
	vscsession->canCacheSession  = _fsec_vscsession_canCacheSession;
	vscsession->closeSession     = _fsec_vscsession_closeSession;
	vscsession->scanFile         = _fsec_vscsession_scanFile;
	vscsession->startStreamScan  = _fsec_vscsession_startStreamScan;
	vscsession->writeStreamBytes = _fsec_vscsession_writeStreamBytes;
	vscsession->endStreamScan    = _fsec_vscsession_endStreamScan;
	vscsession->abortStreamScan  = _fsec_vscsession_abortStreamScan;
}

/*-----------------------------------------------------------------------------
  Define provider library handler
 *---------------------------------------------------------------------------*/
#if defined(DIVY_LIB_DYNAMIC_EXPORT)
/**
 * スタートアップハンドラ (コンストラクタ関数)
 * (note)
 * 	このハンドラはdlopen 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	コールの直前に呼び出されます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlopen された時だけ実施する特別な初期化処理
 * 	    * 関数divy_register_lib_provider のコール
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _vsc_fsec_init(void)
#else
void _init(void)
#endif	/* DIVY_USE_GNU_LD */
{
	/* プロバイダライブラリ構造体の登録 */
	divy_register_lib_provider(&DIVY_PROVIDER_HOOK_STRUCT_NAME(vsc,fsec));
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		FSEC_VSC_PROVIDER_TYPE"(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
}

/**
 * 終了ハンドラ (デストラクタ関数)
 * (note)
 * 	このハンドラはdlclose 終了直後に呼び出されます。また、このライブラリが
 * 	ダイナミックリンクライブラリとして使用された時には、リンク先のmain 関数
 * 	が終了した後、コールされます。
 * (note)
 * 	この関数はプロバイダライブラリがstatic コンパイルされると自動的に
 * 	デバックアウトされます。従って、この関数には、
 * 	    * dlclose された時だけ実施する特別な終了処理
 * 	以外の処理を"書いてはなりません"。
 */
#if defined(DIVY_USE_GNU_LD)
static void _vsc_fsec_fini(void)
#else
void _fini(void)
#endif	/* DIVY_USE_GNU_LD */
{
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		FSEC_VSC_PROVIDER_TYPE"(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return;
}
#endif	/* DIVY_LIB_DYNAMIC_EXPORT */

/**
 * DIVY_PROVIDER_HOOK_STRUCT の初期化ハンドラに登録された関数
 *
 * @pconf apr_pool_t * コンフィグプール
 * @return int 処理ステータス
 * 	DIVY_LIB_ST_OK  : 正常
 * 	DIVY_LIB_ST_ERR : 失敗
 */
static int _vsc_fsec_init_func(apr_pool_t *pconf)
{
	/* create_vscdatasource ハンドラへの登録 */
	divy_hook_create_vscdatasource(_vsc_fsec_create_vscdatasource,
					NULL, NULL, APR_HOOK_MIDDLE);
#if defined(DIVY_DEBUG_PROVIDER_LOADER)
	LERRLOG1(LLOG_NOTICE, DIVY_FST_INFO + DIVY_SST_DEBUG,
		FSEC_VSC_PROVIDER_TYPE"(%"APR_PID_T_FMT")", getpid());
#endif	/* DIVY_DEBUG_PROVIDER_LOADER */
	return DIVY_LIB_ST_OK;
}

/**
 * 指定されたproviderType が自身のプロバイダを示していた時、
 * 自身のデータソースオブジェクトを生成して返却する。
 *
 * @param p apr_pool_t *
 * @param providerType const char * プロバイダタイプ
 * @param vscds VscDataSource ** 作成したVscDataSource構造体
 * @return
 * 	VSC_OK       : providerType が示すプロバイダは自分自身、かつ処理が成功した
 * 	VSC_DECLINED : providerType が示すプロバイダは自分ではなかった
 */
static int _vsc_fsec_create_vscdatasource(apr_pool_t *p,
						const char *providerType,
						VscDataSource **vscds)
{
	*vscds = NULL;	/* 初期化 */

	/*
	 * providerType は自分を示しているか？
	 */
	if (providerType == NULL ||
	    strcmp(providerType, FSEC_VSC_PROVIDER_TYPE) != 0) {
		return VSC_DECLINED;
	}

	/*
	 * データソースの生成
	 */
	*vscds = apr_pcalloc(p, sizeof(VscDataSource));
	(*vscds)->__p         = p;
	(*vscds)->type        = FSEC_VSC_PROVIDER_TYPE;
	(*vscds)->loaded_time = APR_TIME_C(0);

	/* 関数の実体を登録する */
	_init_fsec_vscds_method(*vscds);

	/* ステータスメッセージの初期化 */
	DIVY_VSC_NEW_STATUS(*vscds, p);

	return VSC_OK;
}


