/*
 * コマンドラインのメインとして利用するベースクラス
 *
 * $Id$
 */
#ifndef __TF_CMD_PKCMDBASE_H_
#define __TF_CMD_PKCMDBASE_H_

#include <stdarg.h>
#include <iostream>
#ifdef WIN32
#include <list>
#endif
#include "libteamfile/TFUtils.h"
#include "libteamfile/TFServerManager.h"
#include "libteamfile/TFServerResource.h"
#include "libteamfile/TFTransaction.h"

#include "PKTransactionHandler.h"
#include "PKSystemInformationHandler.h"
#include "PKNotificationHandler.h"
#include "PKDefine.h"

const char	CMD_ENV_HOST[]			= "PKHOST";
const char	CMD_ENV_USER[]			= "PKUSER";
const char	CMD_ENV_PASS[]			= "PKPASS";
const char	CMD_ENV_COMP[]			= "PKCOMP";
const char	CMD_ENV_DEBUG[]			= "PKDEBUG";

const char	CMD_ENV_HTTP_PROXY[]	= "HTTP_PROXY";

typedef struct _tagCPKPROGINFO {
	OI_STRING_A strProgName;	///< プログラム名(本体)
	OI_STRING_A	strVersion;		///< バージョン
	OI_STRING_A	strDisc;		///< 詳細説明
} CPKPROGINFO;

enum CMDRESULT {
	TF_CMD_OK,							///< 正常
	TF_CMD_OK_ERROR,					///< 問題があったが正常終了
	TF_CMD_OK_ALLSKIP,					///< 全てスキップされた正常終了
	TF_CMD_PRM_ENCODE_ERROR = 10,		///< パラメータ:qのエンコードが失敗した
	TF_CMD_PRM_HOST_UNKNOWN,			///< パラメータのホストが未入力
	TF_CMD_PRM_PORT_UNKNOWN,			///< パラメータのポートが未入力
	TF_CMD_PRM_USER_UNKNOWN,			///< パラメータのユーザが未入力
	TF_CMD_PRM_PASS_UNKNOWN,			///< パラメータのパスワードが未入力
	TF_CMD_PRM_UID_UNKNOWN,				///< パラメータのUID（ユーザID)が未入力
	TF_CMD_PRM_NAME_UNKNOWN,			///< パラメータの名前が未入力
	TF_CMD_PRM_USERTYPE_UNKNOWN,		///< パラメータのユーザタイプが不明
	TF_CMD_PRM_CONTRADICTION,			///< パラメータの内容に矛盾がある（同時に利用できないパラメータを利用したなど）
	TF_CMD_PRM_LIMITATION,				///< 未定義(予約）
	TF_CMD_PRM_INVALID_PARAM,			///< パラメータが不正（詳細なエラーでは無い場合すべてこれになる）
	TF_CMD_PRM_AUTH_FAIL,				///< 認証エラー
	TF_CMD_PRM_AUTH_PROXY_FAIL,			///< プロキシ認証エラー
	TF_CMD_PRM_REGEX_ERROR,				///< 正規表現エラー

	TF_CMD_HST_NOT_FOUND = 50,			///< ホストが見つかりません
	TF_CMD_USR_NOT_FOUND,				///< ユーザが存在しません
	TF_CMD_USR_EXPIRED,					///< ユーザの有効期限がきれている
	TF_CMD_USR_INACTIVE,				///< ユーザが利用不可能
	TF_CMD_USR_EXIST,					///< ユーザが既に存在している
	TF_CMD_USR_INACTIVE_USR,			///< ユーザを利用不可能にした

	TF_CMD_FILE_NOTFOUND = 60,			///< ファイルが存在しない
	TF_CMD_FILE_ACCESSDENIED,			///< アクセスが拒否された

	TF_CMD_GRP_NOT_FOUND = 70,			///< グループが存在しない
	TF_CMD_GRP_DEL_FAIL,				///< グループが削除できなかった。
	TF_CMD_GRP_GROUPOWNERFOUND,			///< 既にオーナーが存在していました
	TF_CMD_GRP_RELEASE_USER_FAIL,		///< ユーザの所属解除に失敗した
	TF_CMD_GRP_EXIST,					///< グループが既に存在している。

	TF_CMD_CANTCREATEFOLDER = 80,		///< フォルダの作成に失敗しました
	TF_CMD_FAILDDEPEND,					///< 予約
	TF_CMD_PARENTFOLDER_NOTFOUND,		///< 親フォルダが存在しない
	TF_CMD_DOWNLOADFOLDER_NOTFOUND,		///< ダウンロードフォルダが存在しない
	TF_CMD_SERVER_ERROR,				///< サーバエラー（Internal server error)
	TF_CMD_CON_FAILED,					///< 接続エラー（socket error)

	TF_CMD_SSL_SSLHANDSHAKEFAILED,		///< SSLハンドシェイクエラー

	TF_CMD_FAIL = 99,					///< エラー
	TF_CMD_EXE_EXPIRED = 999			///< ライセンス切れ
};

enum CMDMODE {
	TFCMD_NONE		= 0,
	TFCMD_ADD		= 1,
	TFCMD_DELETE	= 2,
	TFCMD_UPDATE	= 3,
	TFCMD_VIEW		= 4
};

// 出力フォーマット
enum CMDOUTFMT {
	PKCMDOUTFMT_NONE	= 0,
	PKCMDOUTFMT_PLAIN	= 1,
	PKCMDOUTFMT_XML		= 2
};

extern void printDate(time_t gmttime);


class CPKCmdBase : public CPKTransactionHandler
{
public:
	CPKCmdBase(void);
	virtual ~CPKCmdBase(void);

	/**
	 *	コマンドラインを実行する
	 *	@param	nArgc		int
	 *	@param	pszArgv[]	char* const
	 *	@return int
	 */
	int Execute(int nArgc, char* const pszArgv[]);

	/**
	 *	利用するプログラム情報を設定するハンドラ
	 *	純粋なので必ず実装してください。
	 *
	 *	@param	info	CPKPROGINFO&
	 *	@return void
	 *
	 */
	virtual void OnQueryProgramInfo(CPKPROGINFO &info)=0;

	/**
	 *	パラメータが見つかった単位で戻ってくるハンドラ
	 *	@param	nKey		const int
	 *	@param	pszValue	const TFXMLByte*
	 *	@return boolean
	 *	パラーメータによって問題の場合はfalseを返却してください。
	 *	処理がそこで中断します。
	 *	ここではアルファベットの大文字は戻ってきません。
	 *	基本パラメータとして予約されているからです。
	 */
	virtual bool OnFoundParam(const int nKey, const TFXMLByte* pszValue);

	/**
	 *	使い方（usage)の文字列を戻す為に必要な内容を問い合わせてくるハンドラ
	 *	falseを返すと無視するようになります。
	 *	@param	nOpt		const int
	 *	@param	strUsage	OI_STRING_A&
	 *	@return boolean
	 */
	virtual bool OnQueryUsage(const int nOpt, OI_STRING_A& strUsage);

	/**
	 *	このコマンドで理解できるオプション文字列を指定する。
	 *
	 *	@param	strOpt	OI_STRING_A&	ASCIIだけ指定できます。?は予約されている為利用できません。
	 *	@return void
	 *	(note)
	 *	この文字列にはフォーマットがあります。
	 *	== 例 ==
	 *     strOpt = "ABC:DE:"
	 *
	 *	AとBとDは引数を持たないオプションです。
	 *	CとEは:があるため引数があることを示します。
	 */
	virtual void OnKnownOptions(OI_STRING_A& strOpt);

	/**
	 *	コマンドを実行（Run)する直前に呼び出される
	 *	コマンドを実行することができるかどうかをここでチェックしてください。
	 *	CMDRESULTがTF_CMD_OKの場合だけRunを実行する事ができます。
	 *
	 *	@param enuMode	CMDMODE	実行するモード
	 *	@return CMDRESULT
	 */
	virtual CMDRESULT OnIsValid(CMDMODE enuMode);

	/**
	 *	実行される本体です。
 	 *	純粋なので必ず実装してください。
	 *	この内部で必要な処理全てを終了させてください。
	 *	@param enuMode	CMDMODE
	 *	@param pSvrItem	CTFServerResource*
	 *
	 *	@return int	エラーコード
	 */
	virtual int OnExecute(CMDMODE enuMode, CTFServerResource* pSvrItem)=0;


protected:
	bool cmdBaseValidate(void);
	void printHELP(void);
	void getEnv(CTFServerResource& cSvrItem);
	void setServerInfo(CTFServerResource& cSvrItem, const OI_STRING_A strURL);
	bool setDebugMode(const char* path);
#ifdef WIN32
	bool ParseHostNameList(LPWSTR szHostNames, std::list<TF_STRING_W>& lpHost);
	bool ParseProtocolAndHostname(TF_STRING_W szName, TF_STRING_W& szProtocol, TF_STRING_W &szHostname);
	bool ParseHostnameAndPortNumber(TF_STRING_W szName, TF_STRING_W &szHostName, int &iPort);
#endif
	CMDRESULT				m_enuResult;		///< コマンドラインの結果

	CTFTransaction			m_cTrans;			///< トランザクション
	bool					m_bVerbose;			///< Verboseモードの場合TRUE
	CMDOUTFMT				m_enuOutFmt;		///< 出力形式

private:
	CPKPROGINFO					m_info;
	OI_STRING_A					m_strOpt;
	CPKSystemInformationHandler	m_cSysHandler;	///< システムハンドラ
	CPKNotificationHandler		m_cNotifyHandler;

#ifdef _WIN32
	static BOOL WINAPI SignalHandler(DWORD dwCtrlType)
	{
		switch(dwCtrlType)
		{
			case CTRL_C_EVENT:	// CTRL + C
				break;

			default:
				break;
		}

		// カーソルを元に戻す
		CONSOLE_CURSOR_INFO lpinfo;
		HANDLE hHandle = ::GetStdHandle(STD_OUTPUT_HANDLE);
		if (::GetConsoleCursorInfo(hHandle, &lpinfo))
		{
			lpinfo.bVisible = true;	//　カーソル表示を元に戻す
			::SetConsoleCursorInfo(hHandle, &lpinfo);
		}

		// FALSEを渡しことによりExitProsessへの実行になる（デフォルト）
		return FALSE;
	}
#endif

};

/*
void
printDate(time_t gmttime)
{
	struct tm *s_tm;

	s_tm = localtime(&gmttime);

	printf("%d/%d/%d %d:%d:%d,"
			,s_tm->tm_year+1900
			,s_tm->tm_mon+1
			,s_tm->tm_mday
			,s_tm->tm_hour
			,s_tm->tm_min
			,s_tm->tm_sec);
};
*/


#endif
