#include "PKCmdPut.h"
#include "libteamfile/TFStringTokenizer.h"
#include "libteamfile/TFResourceItem.h"
CPKCmdPut::CPKCmdPut(void)
{
	m_strTargetURI.empty();
	m_strFiles.empty();
	m_bRecarsive = true;
	m_bCreateFolder = false;
	m_tjst = (time_t)-1;

	CTFPlatformsFactory* pFactory = new CTFPlatformsFactory();
	m_pLocalFileUtils = pFactory->CreateLocalFileUtils();
	m_pCmpItem = pFactory->CreateLocalFileItem();
	delete pFactory;
}

CPKCmdPut::~CPKCmdPut(void)
{
	delete m_pLocalFileUtils;
	delete m_pCmpItem;
}

void
CPKCmdPut::OnQueryProgramInfo(CPKPROGINFO &info)
{
	info.strDisc		= "Uploading command line using WebDAV protocol.";
	info.strProgName	= "tfput";
	info.strVersion		= TFCMDVERSION;
}

bool
CPKCmdPut::OnFoundParam(const int nKey, const TFXMLByte* pszValue)
{
	bool bResult = true;

	TF_STRING_W strValue;
	OI_STRING_A strAscii;

	strValue = (const TFXMLCh*)X(pszValue);
	strAscii = (const char*)X(pszValue);

	switch(nKey)
	{
		// アップロード条件
		case 'd':
			m_strDateTime = strValue.c_str();
			break;

		// ファイル一覧
		case 'f':
			m_strFiles = strValue.c_str();
			break;

		// ターゲットフォルダを作成
		case 'm':
			m_bCreateFolder = true;
			break;

		//　ターゲットフォルダ
		case 't':
			m_strTargetURI = strValue.c_str();
			break;

		// 階層処理をしない
		case '1':
			m_bRecarsive = false;
			break;

		default:
			m_enuResult = TF_CMD_PRM_INVALID_PARAM;
			bResult = false;
	}

	return bResult;
}

bool
CPKCmdPut::OnQueryUsage(const int nOpt, OI_STRING_A& strUsage)
{
	bool bResult = true;

	switch(nOpt)
	{
		case 'd':
			strUsage = "condition date YYYY/mm/dd HH:MM:ss";
			break;

		case 'f':
			strUsage = "file or directory (ex: file1 file2 dir)";
			break;

		case 'm':
			strUsage = "create a new target folder";
			break;

		case 't':
			strUsage = "target URI";
			break;
			
		case '1':
			strUsage = "not recarsive call";
			break;

		default:
			bResult = false;
			break;
	}

	return bResult;
}

void
CPKCmdPut::OnKnownOptions(OI_STRING_A& strOpt)
{
	strOpt = "d:1f:mt:";
}

CMDRESULT
CPKCmdPut::OnIsValid(CMDMODE enuMode)
{

	// ファイルが一件もないならエラー
	if (m_strFiles.empty()) return TF_CMD_PRM_INVALID_PARAM;

	// ターゲットURIがないとエラー
	if (m_strTargetURI.empty()) return TF_CMD_PRM_INVALID_PARAM;

	// ターゲットフォルダの最後にスラッシュがない場合は追加する
	if (m_strTargetURI[m_strTargetURI.size()-1] != '/')
		m_strTargetURI += C2W("/");

	if (!m_strDateTime.empty()) 
	{
		if ( (m_tjst = ParseTimeJapan((const char*)X(m_strDateTime.c_str()))) == -1)
			return TF_CMD_PRM_INVALID_PARAM;
	}

	return TF_CMD_OK;
}

int
CPKCmdPut::OnExecute(CMDMODE enuMode, CTFServerResource* pSvrItem)
{

	CTFURL cURL;
	CTFResourceProp *pProp;
	TF_STRING_W	strURI = pSvrItem->GetURI();
	strURI += m_strTargetURI.c_str();
	pSvrItem->SetURI(strURI.c_str());

	// PROPFINDを流します（ターゲットフォルダのチェック）
	if (!m_cTrans.List(pSvrItem, &pProp))
	{
		switch(m_cTrans.GetLastError())
		{
			case TF_RESULT_WRONGUSERPASSWORD:
				return TF_CMD_PRM_AUTH_FAIL;
				break;

			case TF_RESULT_UNEXPECTED:
				return TF_CMD_FAIL;
				break;

			case TF_RESULT_INTERNALSERVERERROR:	// サーバエラー
				return TF_CMD_SERVER_ERROR;
				break;

			case TF_RESULT_SSLHANDSHAKEFAILED:	// SSLハンドシェイクエラー
				return TF_CMD_SSL_SSLHANDSHAKEFAILED;

			default:
				break;
		}

		return TF_CMD_PARENTFOLDER_NOTFOUND;		
	}
 
	// 調べるだけなのでdeleteしていい
	if (pProp) delete pProp;

	// ここにきたのならファイルは必ず指定されている
	// ただし、そのファイルが実在するかどうかはわからない
	CTFStringTokenizer	cToken(m_strFiles.c_str(), C2W("!"));
	TF_STRING_W			strFile;
	TF_STRVECTOR_W		vFiles;
	while(cToken.GetNextToken(strFile))
	{
		vFiles.push_back(strFile.c_str());
	}

	bool bResult = false;
	if (vFiles.size() > 0)
	{
		CTFResourceItem* pItem = TFLIB_DYNAMIC_CAST(CTFResourceItem, pSvrItem);
		bResult = m_cTrans.Upload(pSvrItem->GetServerID(), pItem, vFiles);

		// トランザクションで失敗したがコマンドラインのエラーが正常だった場合
		if (!bResult && m_enuResult == TF_CMD_OK)
		{
			switch(m_cTrans.GetLastError())
			{
				case TF_RESULT_UNEXPECTED:
					m_enuResult = TF_CMD_FAIL;
					break;

				case TF_RESULT_LSYS_NOTFOUND:
					m_enuResult = TF_CMD_FILE_NOTFOUND;
					break;

				case TF_RESULT_SKIP:
					m_enuResult = TF_CMD_OK_ALLSKIP;
					break;

				default:
					m_enuResult = TF_CMD_FAIL;
			}
		}

	}

	// ファイル（フォルダ）が存在しなかったがフォルダ作成オプションがついていた
	if (m_enuResult == TF_CMD_FILE_NOTFOUND && m_bCreateFolder)
	{
		TF_STRVECTOR_W::iterator it = vFiles.begin();
		for(; it != vFiles.end(); )
		{
			TF_STRING_W strDir = *it;

			bResult = m_cTrans.MakeCol(pSvrItem->GetServerID(), pSvrItem->GetURI(), strDir.c_str());
			if (bResult == false)
			{
				switch(m_cTrans.GetLastError())
				{
					case TF_RESULT_FOUNDFOLDER:
						m_enuResult = TF_CMD_CANTCREATEFOLDER;	// フォルダの作成に失敗
						break;

					default:
						m_enuResult = TF_CMD_CANTCREATEFOLDER;	// フォルダの作成に失敗
					
				}
			}

			it++;

		}
	}

	// メソッドまで正常なら正常として返す
	if (bResult) m_enuResult = TF_CMD_OK;

	return m_enuResult;

}

void
CPKCmdPut::OnResultLocalResource( const TFTRANSMODE enuMode,
								  const TFDAVMETHOD enuMethod, 
										CTFLocalFileItem* pItem,
								  const TFRESULT nResult)
{

	TF_STRING_A str;

#ifdef _WIN32
	// ラインクリア出力には影響を与えない
	CONSOLE_SCREEN_BUFFER_INFO lpinfo;
	HANDLE h = ::GetStdHandle(STD_OUTPUT_HANDLE);
	if (h)
	{
		::GetConsoleScreenBufferInfo(h, &lpinfo);
		COORD cr;
		cr.X = lpinfo.dwCursorPosition.X;
		cr.Y = lpinfo.dwCursorPosition.Y;
		DWORD dwWritten;
		BOOL bResult = ::FillConsoleOutputCharacter(h, ' ', lpinfo.dwSize.X, cr, &dwWritten);
	}
#else
	::fprintf(stderr, "\x1b[2K"); // ラインクリア
#endif

	// MKCOL
	if (enuMethod == TF_DAV_MKCOL)
	{
		switch(nResult)
		{
			case TF_RESULT_OK:					//< 成功
				if (m_enuResult == TF_CMD_OK)	m_enuResult = TF_CMD_OK;
				str = "OK   ";
				break;

			case TF_RESULT_FOUNDFOLDER:			//< フォルダが既にあった
				m_enuResult = TF_CMD_OK_ERROR;
				str = "OK-  ";
				break;

			case TF_RESULT_SKIP:				//< 正常だがスキップされた
				m_enuResult = TF_CMD_OK;
				str = "SKIP ";
				break;

			default:
				if (m_enuResult != TF_CMD_OK_ERROR) m_enuResult = TF_CMD_FAIL;
				str = "FALSE";
				break;
		}
	}

	// PUT
	if (enuMethod == TF_DAV_PUT)
	{
		switch(nResult)
		{
			case TF_RESULT_OK:					//< 正常
				if (m_enuResult == TF_CMD_OK) m_enuResult = TF_CMD_OK;
				str = "OK   ";
				break;

			case TF_RESULT_RESOURCE_NOCONTENT:	//< 上書き
				m_enuResult = TF_CMD_OK_ERROR;
				str = "OK-  ";
				break;

			case TF_RESULT_SKIP:				//< 正常だがスキップされた
				m_enuResult = TF_CMD_OK;
				str = "SKIP ";
				break;

			default:
				if (m_enuResult != TF_CMD_OK_ERROR) m_enuResult = TF_CMD_FAIL;
				str = "FALSE";
		}
	}

#ifdef _WIN32
	std::cout << str.c_str() << " " << (const char*)X(pItem->GetURI()) << std::endl;
#else
	std::cout << str.c_str() << " " << (const char*)X(pItem->GetURI()).UTF8().c_str() << std::endl;
#endif

}

bool
CPKCmdPut::OnQueryRecursiveLocalResource(TF_SIZE_T stratum, CTFLocalFileItem* pItem)
{
	return m_bRecarsive;
}

/**
 *	リソースアイテムの詳細を埋める為の最後のハンドラ
 *	ここでCTFLocalFileItemの内容を変更や全ての項目を埋めることができます。
 */
 bool
CPKCmdPut::OnQueryDetailLocalResource(CTFLocalFileItem* pLocalItem, TFTRANSMODE enuMode)
 {
	 /* フォルダは対象外 */
	 if (pLocalItem->isFolder()) return true;

	 if (m_tjst != -1)
	 {
		 m_pLocalFileUtils->SetTimeTemporary(m_pCmpItem, m_tjst, m_tjst);

		 /* 引数（アップロード対象)が引数の日付より小さい場合はfalseとして処理を継続させない */
		 if (m_pLocalFileUtils->CompareDateTime(pLocalItem, m_pCmpItem) < 1)
			 return false;
	 }

	 return true;
 }
