/**
 *  Copyrigth(c) 2004 Pertner Co., Ltd. TeamFile Team
 *  $Id$
 *
 *	ファイル名	:	TFLocalFileUtils.h
 *	クラス名	:	CTFLocalFileUtils
 *	継承		:
 *
 *	機能　		:	ローカルファイルのユーティリティの抽象クラス
 *
 *
 */

#ifndef	TF_LIB_TFLOCALFILEUTILS_H
#define TF_LIB_TFLOCALFILEUTILS_H	1

#include "TFLocalFileItem.h"
#include "TFLocalFileItemList.h"
#include "TFWsTransactionHandler.h"

enum TFFSTAT
{
	e_NOERROR = 0,		// 正常
	e_EEXIST,			// 既に存在していた
	e_EISDIR,			// ディレクトリを指定して尚且つ書き込みが指定されていた
	e_EACCES,			// アクセス拒絶
	e_ENAMETOOLONG,		// パスが長すぎる
	e_ENOENT,			// ディレクトリが存在しないかシンボリックリンクが壊れている
	e_ENOTDIR,			// ディレクトリではない
	e_EINVAL,			// ハンドルが無効(ファイルが存在しない場合はこれを指定する）
	e_ENOMEM,			// カーネルに十分なメモリがない
	e_EMFILE,			// プロセスが開ける数を超えている
	e_EBADF,			// ディスクリプタがオープンされていない
	e_EIO,				// I/Oエラー
	e_EFAIL				// エラーだが不明な内容
};

TF_BEGIN_CLASSEXPORT

class TF_EXPORT CTFLocalFileUtils
{
public:
	CTFLocalFileUtils(void);
	virtual ~CTFLocalFileUtils(void);
	void SetHandler(CTFWsTransactionHandler* pHandler);

	/**
	 *	ローカルファイルアイテムを作成する
	 *	@param	rItem		CTFLocalFileItem&
	 *	@param	rHintItem	const CTFLocalFileItem&
	 *	@return			TFFSTAT	(0が正常）
	 *
	 *	(note)
	 *	rHintItemの内容を利用してローカルファイルアイテムを作成する
	 *	ここで作成されたローカルファイルアイテム（CTFLocalFileItem)では
	 *	ローカルのファイルシステムにアクセスできるようなパスの準備をしておく
	 *	必要があります。
	 *
	 */
	virtual TFFSTAT BuildLocalFileItem(CTFLocalFileItem& rItem, const CTFLocalFileItem& rHintItem) const = 0;

	enum TFOPENFLAGS
	{
		TF_F_NONE       = 0x0000,		// unknown file flags
		TF_F_RDONLY     = 0x0001,		// read only
		TF_F_WRONLY     = 0x0002,		// write only
		TF_F_RDWR       = 0x0004,		// read and write
		TF_F_NEW        = 0x0100,		// open the new file
		TF_F_EXISTING	= 0x0200,		// open the existing truncate file
		TF_F_APPEND		= 0x0400		// open the existing append file
	};

	/**
	 *	指定されたファイルをオープンする
	 *	@param cFile				CTFLocalFileItem&
	 *	@param nFlags				int
	 *
	 *	@return TFFSTAT			ファイルオープンの結果（0が正常）
	 *
	 *	(note)
	 *	TFOPENFLGSに基づきファイルを開いてファイルディスクリプタ－を
	 *	設定して返してください。
	 */
	virtual TFFSTAT Open(CTFLocalFileItem& cFile, int nFlags) const = 0;

	/**
	 *	ファイルをクローズする
	 *	@param	pItem		CTFLocalFileItem&
	 *
	 *	@return TFFSTAT
	 */
	virtual TFFSTAT Close(CTFLocalFileItem& cFile) const = 0;

	/**
	 *	ファイルを削除する
	 *	@param	pItem		CTFLocalFileItem&
	 *
	 *	@return TFFSTAT
	 */
	virtual TFFSTAT Unlink(CTFLocalFileItem& cFile) const = 0;

	/**
	 *	ディレクトリを作成する
	 *	@param	cFile		作成するローカルアイテム
	 *	@param	bParent		存在しない親フォルダも作る
	 *
	 *	@return TFFSTAT
	 *
	 */
	virtual TFFSTAT Mkdir(CTFLocalFileItem& cFile, bool bParent) const = 0;

	/**
	 *	ファイルの状態を取得する
	 *	@param pItem		CTFLocalFileItem&
	 *
	 *	@return TFFSTAT
	 */
	virtual TFFSTAT GetFileAttr(CTFLocalFileItem& cFile) const = 0;

	/**
	 *	ローカルファイルが存在するか調べる
	 *	rFileにはパス名が含まれて戻ってきますのでそのパス名で調べてください。
	 *	@param	rFile	CTFLocalFileItem&
	 *	@return boolean ( true: 存在する / false: 存在しない )
	 *	
	 */
	virtual bool ResourceExist(CTFLocalFileItem& rFile) const = 0;

	/**
	 *	渡されたpItemListのファイル以下（フォルダなら）全てを一覧として出力する
	 *	ここでの戻りのvFilesには必ず一件は存在するはず。
	 *	@param	pItemList	CTFLocalFileItemList*
	 *	@param	pUtil		CTFLocalFileUtil*
	 *
	 *	@return	TFFSTAT
	 */
	virtual TFFSTAT EnumLocalFiles(CTFLocalFileItemList* pItemList, CTFLocalFileUtils* pUtil, TF_SIZE_T& unTotalSize) = 0;

	/**
	 * 指定されたローカルファイルの日付を設定する
	 * @param pFile		CTFLocalFileItem*	対象ファイル
	 * @param at		time_t				作成日
	 * @param ut		time_t				更新日
	 *
	 * atime , utime共にNULLを指定した場合は現在時刻となる
	 */
	virtual TFFSTAT SetTime(CTFLocalFileItem* pFile, time_t at=NULL, time_t ut=NULL) = 0;

	/**
	 * 指定されたローカルファイルの日付を設定する
	 * SetTime()と異なるのはpFileが存在していることを確認せず単にItemの日付を設定するだけで
	 * あることです。
	 * @param pFile		CTFLocalFileItem*	対象ファイル
	 * @param at		time_t				作成日
	 * @param ut		time_t				更新日
	 */
	virtual TFFSTAT SetTimeTemporary(CTFLocalFileItem* pFile, time_t at=NULL, time_t ut=NULL) = 0;

	/**
	 *	指定されたディスクリプタに対して書き込みを行う
	 *	@param	nFD			int			ファイルディスクリプタ
	 *	@param	pszString	conat char*	出力内容
	 *	@param	size		TF_SIZE_T	pszStringのlength
	 */
	virtual TFFSTAT Write(int nFD, const char* pszString, TF_SIZE_T size) = 0;

	/**
	 *	渡されたエラーコードに関するメッセージを出力する
	 *	これはネイティブなコードの為必ずOS側で用意されたメッセージを返却の事
	 *
	 *	@param nErrorCode		TF_SIZE_T		エラーコード
	 *	@param strMessage		TF_STRING_A&	エラーメッセージ
	 */
	virtual void GetErrorMessage(TF_SIZE_T nErrorCode, TF_STRING_A& strMessage) = 0;

	/**
	 *
	 *	二つのファイルの日時を比較する
	 *	@param pItem1	CTFLocalFileItem*
	 *	@param pItem2	CTFLocalFileItem*
	 *	@return pItem1 > pItem2 => 1 / pItem == pItem2 => 0 / pItem1 < pItem2 => -1
	 */
	virtual int CompareDateTime(CTFLocalFileItem* pItem1, CTFLocalFileItem *pItem2) = 0;

/*
 * まだサポートはしない
 *
	// ファイルの読み込み
	virtual int Read()=0;

	// ファイルの書き込み
	virtual int Write()=0;
*/

protected:
	CTFWsTransactionHandler		m_cTransactionHandlerDef;
	CTFWsTransactionHandler*	m_pTransactionHandler;

};

TF_END_CLASSEXPORT

#endif

