/**
 *	Copyrigth(c) 2004 Pertner Co., Ltd. TeamFile Team
 *  $Id$
 *
 *	ファイル名	:	TfObject.h
 * クラス名	:	CTFObject
 *	継承		:
 *
 *	機能　		:	クラスの正体を動的に判断し、ダウンキャスト検証機能を提供する
 *
 */

#ifndef TF_LIB_TFOBJECT_H
#define TF_LIB_TFOBJECT_H	1

#include "TFClassInfo.h"

/// CTFObject派生クラスのクラス情報を取得するマクロ
#define TFLIB_GET_CLASSINFO(classname)	\
	(classname::m_cClass##classname)

// CTFObject派生クラスのクラス宣言の中に入れるマクロ
#define TFLIB_DECLARE_CLASS(classname)	\
	public:	\
		classname(const classname& source);	\
		classname& operator =(const classname& source);	\
		static CTFClassInfo m_cClass##classname; \
		static CTFObject* Create##classname##Object();

/// CTFObject派生クラスのコンストラクタに入れるマクロ
#define TFLIB_CONSTRUCT_CLASS(classname)	\
	RegisterClassInfo(&classname::m_cClass##classname);

/// CTFObjectを動的生成するためのマクロ（TFLIB_IMPLEMENT_CLASSマクロ内で使用する）
#define TFLIB_IMPLEMENT_DYNAMIC(classname)	\
		classname::classname(const classname& source) {	\
			TFLIB_CONSTRUCT_CLASS(classname);	\
			Import(source);	\
		}	\
		classname& classname::operator =(const classname& source) {	\
			if( &source != this ) Import(source);	\
			return *this;	\
		}	\
		CTFObject* classname::Create##classname##Object() {	\
			return new classname;	\
		}

/// CTFObject派生クラスのcppファイル内に入れるマクロ
#ifdef _DEBUG
	#define TFLIB_IMPLEMENT_CLASS(classname,classname_str)	\
		TFLIB_IMPLEMENT_DYNAMIC(classname) \
		CTFClassInfo classname::m_cClass##classname(classname::Create##classname##Object,classname_str);
#else
	#define TFLIB_IMPLEMENT_CLASS(classname,classname_str)	\
		TFLIB_IMPLEMENT_DYNAMIC(classname)	\
		CTFClassInfo classname::m_cClass##classname(classname::Create##classname##Object,0);
#endif

/// CTFObject派生クラスのImportメソッド実装の先頭に入れるマクロ
#define TFLIB_PREPARE_IMPORT(class,upperclass,source,psource)	\
	upperclass::Import(source);	\
	const class* psource = TFLIB_DYNAMIC_CAST(class, (CTFObject*)&source);	\
	if( !psource ) return;

/**
 *	リソースクラス用の基本クラス
 *
 *	リソース情報を保持するCTFDavResourceの基本クラスとして用いる。
 *	さらに、メソッドやプロパティ内容に合わせてCTFDavResourceから任意のクラスに派生させる。
 *	CTFTransactionクラスでリソース情報を扱う場合は、基本的にCTFDavResourceを使用する。
 *
 *	なお、このクラスは多重継承には対応していない。
 *
 *	このクラスからの派生を行うと、代入演算子とコピーコンストラクタの処理が自動的に追加される。
 *	その代わり、派生先のクラスでは、Importメソッドを実装し、メンバー変数コピー処理を行うこと。
 *
 *	【CTFObject派生クラスの実装方法】
 *<pre>
 *	// MyClass.h
 *
 *	class CMyClass : public CTFObject
 *	{
 *	public:
 *		CMyClass();
 *		virtual ~CMyClass();
 *		virtual void Import(const CTFObject& source);				// メンバー変数のコピー処理
 *
 *		int m_iA;
 *
 *		TFLIB_DECLARE_CLASS(CMyClass)
 *	};
 *</pre>
 *
 *<pre>
 *	// MyClass.cpp
 *
 *	#include "MyClass.h"
 *
 *	TFLIB_IMPLEMENT_CLASS(MyClass,"CMyClass")
 *
 *	CMyClass::CMyClass()
 *	{
 *		TFLIB_CONSTRUCT_CLASS(MyClass);
 *	}
 *	
 *	CMyClass::~CMyClass()
 *	{
 *	}
 *
 *	void CMyClass::Import(const CTFObject& source)
 *	{
 *		TFLIB_PREPARE_IMPORT(CMyClass,CTFObject,source,pSource);
 *
 *		m_iA = pSource->m_iA;	// メンバー変数のコピーを行う
 *	}
 *
 *</pre>
 *
 *	【チェック方法】
 *<pre>
 *	CMyClass *pa = new CMyClass();
 *
 *	CTFObject *po = pa;	// 下位→上位の場合は、TFLIB_DYNAMIC_CAST不要
 *	ASSERT(!po);
 *
 *	CMyClass *pb = TFLIB_DYNAMIC_CAST(CMyClass,po);
 *	ASSERT(!pb);
 *</pre>
 *
 *	【オブジェクトの複製方法】
 *
 *<pre>
 *	CMyClass *pa = new CMyClass();
 *
 *	CTFObject *po = pa;	// 下位→上位の場合は、TFLIB_DYNAMIC_CAST不要
 *	ASSERT(!po);
 *
 *	CTFObject *po2 = po->Clone();	// 複製する。この時生成されるオブジェクトはCMyClassである
 *
 *	CMyClass *pb = TFLIB_DYNAMIC_CAST(CMyClass,po2);
 *	ASSERT(!pb);
 *</pre>
 *
 *	【オブジェクト内容のコピー方法】
 *<pre>
 *	CMyClassA *pa = new CMyClassA();
 *	CMyClassA *pa2 = new CMyClassA();
 *	CMyClassB *pb = new CMyClassB();	// CMyClassBはCMyClassAの派生クラス
 *
 *	pa->Import(pb);		// パターン１：CMyClassAのメンバー変数だけが取り込まれる
 *	pb->Import(pa);		// パターン２：こちらもCMyClassAのメンバー変数だけが取り込まれる
 *	*pa = *pa2;			// パターン３：同一クラスであれば=を使っても良い
 *</pre>
 */
TF_BEGIN_CLASSEXPORT
class TF_EXPORT CTFObject
{

public:
	CTFObject(void);
	virtual ~CTFObject(void);					// デストラクタはvirtualにすること。virtualじゃないとデストラクタ起動時に暴走する
	bool IsKindOf(CTFClassInfo* pinfo);
	const char* GetTFClassName();
	CTFObject* CreateDerivedObject();
	CTFObject* Clone();
	void RegisterClassInfo(CTFClassInfo* pinfo);
	virtual void Import(const CTFObject& source);
private:
	CTFClassInfoPtrList m_cClassInfoPtrList;	///< クラス情報リスト
	
	TFLIB_DECLARE_CLASS(CTFObject)
};
TF_END_CLASSEXPORT

/**
 *	キャスト正当性検証用のインライン関数（動的チェック用）
 *
 *	TFLIB_DYNAMIC_CASTマクロから利用すること。このインライン関数を直接利用しないこと。
 *
 *	@param	classinfo	CTFClassInfo*	キャスト先クラスのクラス情報オブジェクトのポインタ(&TFLIB_GET_CLASSINFO(class))
 *	@param	obj			CTFObject*		対象オブジェクトのポインタ
 *	@return	CTFObject*	!=0:オブジェクトのポインタ(==obj)、==0:キャスト失敗
 */
inline CTFObject *TFCheckDynamicCast(CTFClassInfo *classinfo, CTFObject *obj)
{
    return (obj && obj->IsKindOf(classinfo)) ? obj : 0;
}

/**
 *	キャスト正当性検証用のインライン関数（静的チェック用）
 *
 *	TFLIB_STATIC_CASTマクロから利用すること。このインライン関数を直接利用しないこと。
 *
 *	@param	classinfo	CTFClassInfo*	キャスト先クラスのクラス情報オブジェクトのポインタ(&TFLIB_GET_CLASSINFO(class))
 *	@param	obj			CTFObject*		対象オブジェクトのポインタ
 *	@return	CTFObject*	!=0:オブジェクトのポインタ(==obj)、==0:キャスト失敗
 */
inline CTFObject *TFCheckStaticCast(CTFClassInfo *classinfo, CTFObject *obj)
{
	CTFObject* retobj = (obj && obj->IsKindOf(classinfo)) ? obj : 0;

	// TODO: アサートマクロを入れること
	//assert(retobj);

    return retobj;
}

/**
 *	キャスト正当性検証用のマクロ（動的チェック用）
 *
 *	ダウンキャストを行う場合、このマクロを使ってキャストの正当性をチェックしてください。
 *	ただし、このマクロはCTFObjectの派生クラスでないと使用できません。
 *
 *	@param	targetclass	キャスト先のクラス
 *	@param	obj			対象のオブジェクト
 *	@return				!=0:オブジェクトのポインタ(==obj)、==0:キャスト失敗
 */
#define TFLIB_DYNAMIC_CAST(targetclass,obj)	\
	((targetclass *)TFCheckDynamicCast(&TFLIB_GET_CLASSINFO(targetclass),obj))

/**
 *	キャスト正当性検証用のマクロ（静的チェック用）
 *
 *	ダウンキャストを行う場合、このマクロを使ってキャストの正当性をチェックしてください。
 *	ただし、このマクロはCTFObjectの派生クラスでないと使用できません。
 *
 *	@param	targetclass	キャスト先のクラス
 *	@param	obj			対象のオブジェクト
 *	@return				!=0:オブジェクトのポインタ(==obj)、==0:キャスト失敗
 */
#define TFLIB_STATIC_CAST(targetclass,obj)	\
	((targetclass *)TFCheckStaticCast(&TFLIB_GET_CLASSINFO(targetclass),obj))

#endif	// TFLIB_HEADER_TFOBJECT_H

