/**
 *	Copyrigth(c) 2004 Pertner Co., Ltd. TeamFile Team
 *	$Id$
 *	$Name$
 *
 *	ファイル名	:	TFTmplItemList.h
 *	クラス名	:	CTFTmplItemList
 *	継承		:	
 *
 *	機能　		:	リストを保持するテンプレートクラス
 *					リスト機能を付けたいクラスで多重継承して使用する
 *
 *	使用例	:
 *		class CTFDavResourceList :
 *				public CTFResourcePropTF,
 *				public CTFTmplItemList<CTFDavResource>
 *		{
 *			...
 *
 *			virtual CTFDavResource* OnCloneItem(CTFDavResource* source);
 *
 *			...
 *			
 *		}
 *
 */
#ifndef TF_LIB_TFTMPLITEMLIST_H
#define TF_LIB_TFTMPLITEMLIST_H 1

#include "TFLibPublic.h"

TF_BEGIN_CLASSEXPORT

template <typename T> class TF_EXPORT CTFTmplItemList 
{

private:
	typedef std::vector<T*> tItemList;
	typedef	typename tItemList::iterator tItemList_Iterator;

	tItemList	m_vItemList;
	tItemList_Iterator 	m_vItemList_Iterator;

public:
	/**
	 *
	 */
	CTFTmplItemList()
	{
		ResetItemPosition();
	};

	/**
	 *
	 */
	virtual ~CTFTmplItemList()
	{
		EmptyList();
	};
	
	/**
	 *	アイテムをリストに追加する。
	 *	ここで追加されたアイテムの所有権はリストに移る。
	 *	同一ポインタのアイテムを複数追加することはできない
	 *
	 *	内蔵iteratorはリセットされる。
	 *
	 *	OnDeleteItemメソッドをオーバーライドすることで所有権を移さないようにもできる
	 *
	 */
	bool AddItem(T* source)
	{
		if( source == NULL ) return false;

		m_vItemList.push_back(source);
		ResetItemPosition();
		return true;
	};
	
	/**
	 *	リストが空かどうかを調べる
	 *
	 */
	bool IsEmpty(void)
	{
		return m_vItemList.empty();
	}

	/**
	 *	リストを空にする。
	 *	リストと一緒にアイテムの実体も削除される
	 *
	 *	内蔵iteratorはリセットされる。
	 *
	 *	OnDeleteItemメソッドをオーバーライドすることで実体を消さないようにもできる
	 */
	bool EmptyList(void)
	{
		while( !m_vItemList.empty() )
		{
			OnDeleteItem(m_vItemList.back());
			m_vItemList.pop_back();
		}

		ResetItemPosition();
		return true;
	};

	/**
	 *	リストをコピーする。
	 *	実体をコピーするためには、OnCloneItemをオーバーライドする必要がある
	 *
	 *	内蔵iteratorはリセットされる。
	 */
	bool ImportList(CTFTmplItemList<T>* source)
	{
		tItemList_Iterator p;

		// 自分のリストを空にしておく
		EmptyList();

		// リストの内容をコピーする
		p = source->m_vItemList.begin();
		while( p != source->m_vItemList.end() )
		{
			// 新しいリストにオブジェクトをコピーする
			T* newItem = OnCloneItem(*p);
			if( newItem != NULL) m_vItemList.push_back(newItem);
			p++;
		}

		ResetItemPosition();
		return true;
	};

	/**
	 *	リストから特定のアイテムを探す。
	 *	このメソッドを利用するには、OnCheckItemメソッドをオーバーライドする必要がある
	 *	引数hintはOnCheckItemにそのまま渡されるので、クラスや構造体のポインタ等を渡すと良い
	 *
	 *	このメソッドは、内蔵iteratorには影響を及ぼさない。
	 */
	T* FindItem(int iHint, void* pHint)
	{
		tItemList_Iterator p;

		// リスト内を検索する
		p = m_vItemList.begin();
		while( p != m_vItemList.end() )
		{
			// アイテムをチェックして、探しているアイテムならばそのアイテムを返す
			if( OnCheckItem(*p,iHint,pHint) ) return *p;
			p++;
		}

		return NULL;
	};

	/**
	 *	リストから指定されたアイテムを探す。
	 *	このメソッドは単純にポインタを比較しているだけ。
	 *
	 *	このメソッドは、内蔵iteratorには影響を及ぼさない。
	 */
	T* FindItemByPtr(T* source)
	{
		tItemList_Iterator p;

		// リスト内を検索する
		p = m_vItemList.begin();
		while( p != m_vItemList.end() )
		{
			// ポインタを比較して、同じならばそのアイテムを返す
			if( *p == source ) return *p;
			p++;
		}

		return NULL;
	};

	/**
	 *	インデックス番号を用いてアイテムを得る。（０が先頭）
	 *	インデックス番号はGetItemCountメソッドで得られるアイテム数未満でなければならない
	 */
	T* GetItem(size_t nIndex)
	{
		// g++ 2.96では.at(index)がエラーとして処理されてしまう
//		return m_vItemList.at(nIndex);
		return m_vItemList[nIndex];
	};

	/**
	 *	リストに格納されているアイテム数を返す。
	 */
	size_t GetItemCount()
	{
		return m_vItemList.size();
	};

	/**
	 *	内蔵iteratorを使用して、次のアイテムを返す。
	 *	終点に到達した等で返すアイテムが無い場合はNULLを返す。
	 *
	 *	内蔵iteratorは、EmptyList/AddItem/ImportListメソッドを使用するとリセットされる
	 */
	T* GetNextItem()
	{
		if( m_vItemList_Iterator != m_vItemList.end() )
		{
			return *(m_vItemList_Iterator++);
		}
		else
		{
			return NULL;
		}
	};

	/**
	 *	内蔵iteratorをリセットし、ポジションを先頭にする。
	 */
	void ResetItemPosition()
	{
		m_vItemList_Iterator = m_vItemList.begin();
	};
	
	
protected:
	/**
	 *	アイテムの複製が必要な場合に呼び出されるハンドラ。
	 *	ImportItemListを使用する場合は、このメソッドをオーバーライドする必要がある。
	 *	派生クラスでは、new等でオブジェクトを生成したあとに中身をコピーする処理を行う
	 */
	virtual T* OnCloneItem(T* source)
	{
		// アイテムを複製して返す
		// デフォルトでは、NULLを返す（コピーされない）
		return NULL;
	};

	/**
	 *	アイテムの削除が必要な場合に呼び出されるハンドラ。
	 *	Emptyメソッド等で、リスト内のアイテムを消去する場合に使用される。
	 *	デフォルトではdeleteでアイテムを削除するようになっているが、
	 *	派生クラスでオーバーライドすることで削除しないようにすることもできる
	 */
	virtual void OnDeleteItem(T* target)
	{
		// デフォルトではdeleteでアイテムを削除する
		delete target;
	};

	/**
	 *	アイテムのチェックが必要な場合に呼び出されるハンドラ。
	 *	FindItemメソッドで、リスト内のアイテムの中身を調べるために使用される。
	 *	引数hintで示された条件と一致するアイテムの場合、trueを返す。
	 */
	virtual bool OnCheckItem(T* target, int iHint, void* pHint)
	{
		// デフォルトでは常に不一致を返す
		// 派生クラスでは比較結果に応じてtrue(一致)或いはfalse(不一致)を返す
		return false;
	};
	
};

TF_END_CLASSEXPORT

#endif	// TF_LIB_TFTMPLITEMLIST_H

