#include "TFLinuxLocalFileUtils.h"
#include "TFLinuxLocalFileItem.h"

#include "TFLibDefinition.h"
#include "TFURL.h"
#include "TFUtils.h"
#include "oimap.h"

#include <stdio.h>

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif
#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <time.h>
#include <utime.h>
#include <dirent.h>
#include <errno.h>

CTFLinuxLocalFileUtils::CTFLinuxLocalFileUtils(void)
{
}

CTFLinuxLocalFileUtils::~CTFLinuxLocalFileUtils(void)
{
}

TFFSTAT
CTFLinuxLocalFileUtils::BuildLocalFileItem(CTFLocalFileItem& rItem, const CTFLocalFileItem& rHintItem) const
{
	TFFSTAT enuStat = e_NOERROR;

	CTFLinuxLocalFileItem& rLinuxItem = (CTFLinuxLocalFileItem&)rItem;
	CTFLinuxLocalFileItem& rLinuxHintItem = (CTFLinuxLocalFileItem&)rHintItem;

	TF_STRING_W strPath = rLinuxHintItem.GetPathW();
	OI_ASSERT(!strPath.empty());

	TF_STRING_W strURI = rLinuxItem.GetURI();
	if (strURI.empty()) return e_EFAIL;
	CTFURL cURL(strURI.c_str());

	strPath += X("\\");
	strPath += cURL.m_strName;

	return rLinuxItem.SetPath(0, (void*)strPath.c_str(), sizeof(TFXMLCh)) ? e_NOERROR: e_EFAIL;
}

TFFSTAT
CTFLinuxLocalFileUtils::Open(CTFLocalFileItem& cItem, int nFlags) const
{

	int flags = 0;
	int original_mask, nmode;
	TFFSTAT enuStat = e_NOERROR;
	CTFLinuxLocalFileItem& rItem = (CTFLinuxLocalFileItem&)cItem;

	/* original mask μ */
	original_mask = umask(0);
	umask(original_mask);
	nmode = (S_IRWXU | S_IRWXG | S_IRWXO) & ~original_mask;

	if (nFlags & TF_F_RDONLY)
		flags |= O_RDONLY;

	if (nFlags & TF_F_WRONLY)
		flags |= O_WRONLY;

	if (nFlags & TF_F_RDWR)
		flags |= O_RDWR;

	if (nFlags & TF_F_NEW)
		flags |= O_CREAT;

	if (nFlags & TF_F_EXISTING)
		flags |= O_CREAT | O_TRUNC;

	// open
	errno = 0;
	int fd = open(rItem.GetPathA(), flags, nmode);
	if (fd < 0) 
	{
		int err = errno;
		TFFSTAT ret = e_EFAIL;
		m_pTransactionHandler->SetErrorCodeNative(errno, CTFWsTransactionHandler::TF_TRANS_ERROR_TYPE_LOCAL, &rItem);
		switch(err)
		{
			case ENOENT:	// file not found
				ret = e_EINVAL;
				break;
			default:
				break;
		}

		return ret;
	}

	rItem.SetFD(fd);
	return enuStat;
}

TFFSTAT
CTFLinuxLocalFileUtils::Close(CTFLocalFileItem& cFile) const
{

	TFFSTAT enuStat = e_NOERROR;

	CTFLinuxLocalFileItem& rLinuxItem = (CTFLinuxLocalFileItem&)cFile;

	if (rLinuxItem.GetFD() == -1) return e_EFAIL;

	int nRet = close(rLinuxItem.GetFD());
	if (nRet != 0)
	{
		OI_ASSERT(false);
		enuStat = e_EFAIL;
	}

	rLinuxItem.SetFD(-1);

	return enuStat;
}

TFFSTAT
CTFLinuxLocalFileUtils::Unlink(CTFLocalFileItem& cFile) const
{
	TFFSTAT enuStat = e_NOERROR;

	return enuStat;
}

TFFSTAT
CTFLinuxLocalFileUtils::GetFileAttr(CTFLocalFileItem& cFile) const
{

	TFFSTAT enuStat = e_NOERROR;
 
	return enuStat;
}

bool
CTFLinuxLocalFileUtils::ResourceExist(CTFLocalFileItem& rFile) const
{
	CTFLinuxLocalFileItem* pItem = (CTFLinuxLocalFileItem*)&rFile;

	struct stat st;
	if (stat(pItem->GetPathA(), &st) < 0)
		return false;

	return true;
}


TFFSTAT
CTFLinuxLocalFileUtils::EnumLocalFiles(CTFLocalFileItemList* pItemList, CTFLocalFileUtils *pUtils, TF_SIZE_T& unTotalSize)
{
	if (pUtils == NULL) return e_EFAIL;
	CTFLinuxLocalFileUtils* pLinuxUtils = (CTFLinuxLocalFileUtils*)pUtils;
	CTFLinuxLocalFileItem* pItem = NULL;

	if (pItemList->GetItemCount() < 1) return e_EFAIL;
	pItem = (CTFLinuxLocalFileItem*)pItemList->GetNextItem();
	pItemList->ResetItemPosition();

	TF_STRING_A strPath = (const char*)X(pItem->GetPathW()).UTF8().c_str();

	// open
	int fd = open(strPath.c_str(), O_RDONLY);
	if (fd < 0)
	{
		int err = errno;
		TFFSTAT ret = e_EFAIL;
		m_pTransactionHandler->SetErrorCodeNative(errno, CTFWsTransactionHandler::TF_TRANS_ERROR_TYPE_LOCAL, pItem);
		switch(err)
		{
			case ENOENT:	// file not found
				ret = e_EINVAL;
				break;
			default:
				break;
		}

		return  ret;
	}

	// եξ
	struct stat st;
	if ((fstat(fd, &st)) < 0) 
	{
		m_pTransactionHandler->SetErrorCodeNative(errno, CTFWsTransactionHandler::TF_TRANS_ERROR_TYPE_LOCAL, pItem);
		return e_EFAIL;
	}

	// URI
	// եʥѥȤǾ̥եˤʤ
	CTFURL cURL(pItem->GetPathW());
	cURL.m_strName;
	TF_STRING_W strURI = cURL.m_strName;
	pItem->SetURI(strURI.c_str());

	// file size
	pItem->SetSize(st.st_size);
	unTotalSize += pItem->GetSize();

	// °
	setFileAttribute(pItem, st.st_mode);
	// dir?
	if (pItem->isFolder() && !pItem->isSymLink()
					&& m_pTransactionHandler->OnQueryRecursiveLocalResource(0, pItem))
	{
		if (strPath.compare(0,1,".") == 0)
		{
			pItem->SetFolderType(TF_FOLDER_TYPE_TEMP);
			pItem->SetURI(C2W("/"));
		}
		else
		{
			// ΥեϰʥեȤ
			pItem->SetFolderType(TF_FOLDER_TYPE_SAFTY);
		}

		InternalEnumLocalFiles(pItem->GetPathW(), pItemList, pUtils, unTotalSize);
	}

	// ǥץƤĤ
	if (fd > -1) close(fd);

	return e_NOERROR;
															
}


void
CTFLinuxLocalFileUtils::setFileAttribute(CTFLocalFileItem* pItem, TF_SIZE_T unAttributes)
{
	TF_SIZE_T unAttr = 0;

	// ɤ߹
	if (S_IRUSR & unAttributes)
			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_READONLY;

	// ե(UNIXˤϤʤ)
//	if (FILE_ATTRIBUTE_HIDDEN & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_HIDDEN;

	// ƥեUNIXˤϤʤ
//	if (FILE_ATTRIBUTE_SYSTEM & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_SYSTEM;

	// ǥ쥯ȥ
	if (S_ISDIR(unAttributes))
			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_DIRECTORY;

	// ֥եUNIXˤϤʤ
//	if (FILE_ATTRIBUTE_ARCHIVE & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_ARCHIVE;

	// ǥХUNIXˤϤʤ
//	if (FILE_ATTRIBUTE_DEVICE & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_DEVICE;

	// ̾ե
	if (S_ISREG(unAttributes))
			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_NORMAL;

	// եUNIXˤϤʤ
//	if (FILE_ATTRIBUTE_TEMPORARY & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_TEMPORARY;

	// ʤä
//	if (FILE_ATTRIBUTE_SPARSE_FILE & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_TEMPORARY;

	// ʤä
//	if (FILE_ATTRIBUTE_REPARSE_POINT & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_REPARSE_POINT;

	// ̥եUNIXˤϤʤ
//	if (FILE_ATTRIBUTE_COMPRESSED & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_COMPRESSED;

	// OFF饤եUNIXˤϤʤ
//	if (FILE_ATTRIBUTE_OFFLINE & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_OFFLINE;

	// ʤä
//	if (FILE_ATTRIBUTE_NOT_CONTENT_INDEXED & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_NOT_CONTENT_INDEXED;

	// Ź沽եUNIXˤϤʤ
//	if (FILE_ATTRIBUTE_ENCRYPTED & unAttributes)
//			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_ENCRYPTED;

	// ܥå
	if (S_ISLNK(unAttributes))
			unAttr |= CTFLocalFileItem::TF_FILE_ATTRIBUTE_LINK;

	pItem->SetAttributes(unAttr);

}

TFFSTAT
CTFLinuxLocalFileUtils::InternalEnumLocalFiles(const TFXMLCh* strFolder, CTFLocalFileItemList* pItemList, CTFLocalFileUtils *pUtil, TF_SIZE_T& unTotalSize)
{

	CTFLinuxLocalFileUtils* pLinuxUtils = (CTFLinuxLocalFileUtils*)pUtil;

	CTFLocalFileItem* pItem = pItemList->GetItem(pItemList->GetItemCount()-1);
	TF_STRING_A strPath = (const char*)X(strFolder).UTF8().c_str();
	if (strPath.empty()) return e_EFAIL;

	// open directory
	DIR *dp = opendir(strPath.c_str());
	if (dp == NULL) return e_EFAIL;

	CTFLinuxLocalFileItem cItem;
	struct dirent *dir;
	while((dir = readdir(dp)) != NULL)
	{
		TF_STRING_A strFile = dir->d_name;
		// åפ
		if (strFile.compare(".") == 0) continue;
		if (strFile.compare("..") == 0) continue;

		// ƥν
		cItem.Initialize();

		// ѥ
		strFile  = (const char*)X(strFolder).UTF8().c_str();
		strFile += "/";
		strFile += dir->d_name;

		// URI
		TF_STRING_W strURI;
		if (!pItem->isTemporary())
			strURI = pItem->GetURI();

		strURI += C2W("/");
		strURI += C2W((const TFXMLByte*)dir->d_name);
		cItem.SetURI(strURI.c_str());
		//printf(">> parent = %s URI=%s strFile=%s\n", (const char*)X(pItem->GetURI()).UTF8().c_str(), (const char*)X(cItem.GetURI()).UTF8().c_str(), strFile.c_str());

		// ƥκ
		// ƥϥեOPEN˼ԤƤ
		// 褷Ȥʥ顼˸Ǥ뤫
		//cItem.SetPath(0, (void*)strFile.c_str(), sizeof(char));
		cItem.SetPath(0, (void*)strFile.c_str(), sizeof(char));

		// եξ
		int fd = open(strFile.c_str(), O_RDONLY);
		if (fd < 0) continue;

		struct stat st;
		if ((fstat(fd, &st)) != -1)
		{
			cItem.SetSize(st.st_size);
			unTotalSize += cItem.GetSize();
			// °
			setFileAttribute(&cItem, st.st_mode);
		}

		// եĤ
		close(fd);

		// եפꤹ
		if (cItem.isFolder()) cItem.SetFolderType(TF_FOLDER_TYPE_NORMAL);
		// ƥɲ
		pItemList->AddItem((CTFLocalFileItem*)cItem.Clone());

		if (cItem.isFolder()
						&& m_pTransactionHandler->OnQueryRecursiveLocalResource(1, pItem))
		{
			InternalEnumLocalFiles(cItem.GetPathW(), pItemList, pUtil, unTotalSize);
		}
	}

	closedir(dp);

	return e_NOERROR;
}


TFFSTAT
CTFLinuxLocalFileUtils::SetTime(CTFLocalFileItem* pFile, time_t at, time_t ut)
{
	TFFSTAT enuStat = e_NOERROR;
	if (pFile == NULL) return e_EINVAL;

	struct tm atm;
	struct tm utm;
	struct utimbuf ub;

	localtime_r(&at, &atm);
	localtime_r(&ut, &utm);

	ub.actime  = mktime(&atm);
	ub.modtime = mktime(&utm);

	TF_STRING_A strFile = pFile->GetPathA();
	
	int ret = utime(strFile.c_str(), &ub);

	return enuStat;
}

TFFSTAT
CTFLinuxLocalFileUtils::SetTimeTemporary(CTFLocalFileItem* pFile, time_t at, time_t ut)
{
	TFFSTAT enuStat = e_NOERROR;
	if (pFile == NULL) return e_EINVAL;
	
	struct tm atm;
	struct tm utm;
	struct utimbuf ub;

	localtime_r(&at, &atm);
	localtime_r(&ut, &utm);

	ub.actime  = mktime(&atm);
	ub.modtime = mktime(&utm);

	TF_STRING_A strFile = pFile->GetPathA();

	int ret = utime(strFile.c_str(), &ub);

	return enuStat;
}

/**
 *
 *	ĤΥեӤ
 *
 *	@param pItem1	CTFLocalFileItem*
 *	@param pItem2	CTFLocalFileItem*
 *	@return pItem1 > pItem2 => 1 / pItem == pItem2 => 0 / pItem1 < pItem2 => -1
 */
int 
CTFLinuxLocalFileUtils::CompareDateTime(CTFLocalFileItem* pItem1, CTFLocalFileItem *pItem2)
{
	struct stat st1, st2;
	CTFLinuxLocalFileItem *pLinuxItem1, *pLinuxItem2;

	pLinuxItem1 = (CTFLinuxLocalFileItem*)pItem1;
	pLinuxItem2 = (CTFLinuxLocalFileItem*)pItem2;

	if (pLinuxItem1 == NULL || pLinuxItem2 == NULL) return 0; // 顼ξƱˤ

	TF_STRING_A strPath1 = pLinuxItem1->GetPathA();
	TF_STRING_A strPath2 = pLinuxItem2->GetPathA();
	
	stat(strPath1.c_str(), &st1);
	stat(strPath2.c_str(), &st2);
	
	double diftime = difftime(st1.st_mtime, st2.st_mtime);
	
	if (diftime > 0) return 1;
	if (diftime == 0) return 0;
	if (diftime < 0) return -1;
	
}

void
CTFLinuxLocalFileUtils::GetErrorMessage(TF_SIZE_T nErrorCode, TF_STRING_A& strMessage)
{
	char *string = strerror((int)nErrorCode);
	if (string) strMessage = string;
}

TFFSTAT
CTFLinuxLocalFileUtils::Write(int nFD, const char* pszString, TF_SIZE_T size)
{
	if (nFD < 0 || pszString == NULL) return e_EINVAL;
	if (size < 1) return e_NOERROR;

	char* pszBuff = (char*)pszString;
	TF_SIZE_T nTotal   = size;
	unsigned int nReadLen = OI_GENBUFSIZE;
	TF_SIZE_T nWritten = 0;

	for (; nWritten < nTotal; )
	{
		if (nReadLen > nTotal - nWritten) nReadLen = nTotal - nWritten;
		int nOut = ::write(nFD, pszBuff, nReadLen);
		if (nOut == -1)
		{
			// error
			m_pTransactionHandler->SetErrorCodeNative(nOut, CTFWsTransactionHandler::TF_TRANS_ERROR_TYPE_LOCAL, NULL);
		}
		pszBuff = pszBuff + nOut;
		nWritten += nOut;
	}

	return (nWritten > 0)?e_NOERROR:e_EFAIL;

}


TFFSTAT
CTFLinuxLocalFileUtils::Mkdir(CTFLocalFileItem& cFile, bool bParent) const
{
	const char*	pch = 0;
	TF_STRING_A strPath;

	int unSize;
	TFFSTAT	enuStat = e_NOERROR;
	struct stat sb;
	int omode = S_IRWXU | S_IRWXG | S_IRWXO;	/* a=rwx */
	int nmode;
	int original_umask;
	int parent_mode;

	// get file path string
	strPath = cFile.GetPathA();
	// ֥륹åõ
	RemoveDoubleSlashA(strPath);
	// ǸΥåդ
	if (strPath[strPath.size()-1] != '/')
		strPath += '/';

	if (strPath.empty()) return e_EINVAL;

	/* Make the new mode */
	original_umask = umask(0);
	umask(original_umask);

	nmode = (S_IRWXU | S_IRWXG | S_IRWXO) & ~original_umask;
	parent_mode = nmode | (S_IWRITE|S_IEXEC);
	nmode &= omode;

	if (stat(strPath.c_str(), &sb) == 0)
	{
		if (S_ISDIR(sb.st_mode) == 0)
		{
			// exists file
			return e_EEXIST;
		}

		if (chmod(strPath.c_str(), nmode))
		{
			return e_EACCES;
		}

		return e_NOERROR;
	}

	char *npath = strndup(strPath.c_str(), strPath.length());
	char *p = npath;
	while(*p == '/')
		p++;

	while(p = strchr(p, '/'))
	{
		*p = '\0';
		if (stat(npath, &sb) != 0)
		{
			if (mkdir(npath, parent_mode))
			{
				m_pTransactionHandler->SetErrorCodeNative(errno, CTFWsTransactionHandler::TF_TRANS_ERROR_TYPE_LOCAL, &cFile);
				umask(original_umask);
				free(npath);
				return e_EACCES;
			}
		}
		else if (S_ISDIR(sb.st_mode) == 0)
		{
			umask(original_umask);
			free(npath);
			return e_EEXIST;
		}

		*p++ = '/';		/* resotre slash */
		while(*p == '/')
			p++;
	}

	/* final directory component */
	if (stat(npath, &sb) && mkdir(npath, nmode))
	{
		m_pTransactionHandler->SetErrorCodeNative(errno, CTFWsTransactionHandler::TF_TRANS_ERROR_TYPE_LOCAL, &cFile);
		umask(original_umask);
		free(npath);
	}

	umask(original_umask);
	free(npath);
	return enuStat;
}

