#include "TFBrowser.h"
#include "TFServerManager.h"
#include "TFResourceItem.h"
#include "TFResourceProp.h"
#include "TFTransaction.h"
#include "TFResourceTFRootTree.h"
#include "TFResourcePropTFMsg.h"
#include "TFResourcePropTFUser.h"
#include "TFResourcePropTFGroup.h"

CTFBrowser::CTFBrowser(void)
{
	// デフォルトハンドラの設定
	m_pTransHandler		= &m_cTransHandler;
	m_pNotifyHandler	= &m_cNotifyHandler;
	m_pSysInfoHandler	= &m_cSysInfoHandler;
}

CTFBrowser::~CTFBrowser(void)
{
}

bool
CTFBrowser::CheckServerExistence(CTFResourceItem* pItem, bool bAnswer)
{
	bool bResult = false;

	if (pItem == NULL) return false;

	CTFTransaction	cTrans;
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	int nAllowMethods;
	int nAllowDASL;

	TFDAVMETHOD enuMethod = cTrans.GetServerOptions(pItem, NULL, nAllowMethods, nAllowDASL);

	if (bAnswer)
		m_pNotifyHandler->OnResultServerType(nAllowMethods, nAllowDASL);

	return (enuMethod != TF_DAV_UNKNOWN);
}

bool
CTFBrowser::OpenResource(CTFResourceItem* pItem)
{
	bool bResult = false;

	void *handle = NULL;

	if (pItem == NULL) 
		return getServerItems();

	int nAllowMethod;			// 利用可能なDAVメソッド
	TFSEARCHTYPE enuAllowDASL;	// 利用可能なDASLメソッド
	int nDASLSubType;			// 利用可能なDASLメソッド詳細

	/*
	 * 対象のアイテムがどのようなメソッドを投げられるかチェック
	 */
	queryMethod(pItem, nAllowMethod, enuAllowDASL, nDASLSubType);

	if (nAllowMethod == TF_DAV_UNKNOWN)
		return true;

	if (nAllowMethod == TF_DAV_SEARCH)
		return openbySEARCH(pItem, enuAllowDASL, nDASLSubType);

	if (nAllowMethod & TF_DAV_GET)
		return true;

	if (nAllowMethod & TF_DAV_PROPFIND)
		return openbyPROPFIND(pItem);

	return false;
}

bool
CTFBrowser::CreateResource(CTFResourceItem* pParentItem, TFRESOURCETYPE enuType)
{
	OI_ASSERT(pParentItem);

	bool bResult = false;

	int nAllowMethod;			// 利用可能なDAVメソッド
	TFSEARCHTYPE enuAllowDASL;	// 利用可能なDASLメソッド
	int nDASLSubType;			// 利用可能なDASLメソッド詳細

	queryMethod(pParentItem, nAllowMethod, enuAllowDASL, nDASLSubType);

	CTFTransaction	cTrans;
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	//ファイルの新規作成
	if (enuType == TF_RES_FILE)
	{
		// TODO: ファイル新規作成の実装をいれなさい
		OI_ASSERT(false);
	}

	// コレクションの新規作成
	if (enuType == TF_RES_COLLECTION)
	{
		bResult = cTrans.MakeFolder(pParentItem);
	}

	return bResult;
}

bool
CTFBrowser::CreateSystemMessage(CTFResourceItem* pParentItem)
{
	OI_DEBUG("CTFBrowser::CreateSystemMessage()\n");

	CTFTransaction	cTrans;
	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	return cTrans.MakeColEx(pParentItem, TF_XML_BODY_MSG);

}

bool 
CTFBrowser::UploadResource(CTFResourceItem* pParentItem)
{
	bool	bResult = false;

	CTFTransaction	cTrans;
	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

// TODO: アップロード完成させなさい
	return bResult;
}

bool
CTFBrowser::RenameResource(CTFResourceItem* pItem)
{
	bool	bResult = false;

	CTFTransaction	cTrans;
	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	bResult = cTrans.ReName(pItem);

	return bResult;
}

bool
CTFBrowser::QueryResource(CTFResourceItem* pItem)
{
	OI_ASSERT(pItem);

	bool bResult = false;

	int nAllowMethod;			// 利用可能なDAVメソッド
	TFSEARCHTYPE enuAllowDASL;	// 利用可能なDASLメソッド
	int nDASLSubType;			// 利用可能なDASLメソッド詳細

	queryMethod(pItem, nAllowMethod, enuAllowDASL, nDASLSubType);

	TFITEMTYPE enuType = pItem->GetType();
	switch(enuType)
	{
		case TFITEM_SERVER:
		break;
	}

	if (nAllowMethod & TF_DAV_PROPFIND)
	{
		CTFTransaction cTrans;
		cTrans.SetHandler(m_pNotifyHandler);
		cTrans.SetHandler(m_pTransHandler);
		cTrans.SetHandler(m_pSysInfoHandler);

		// TODO: 一件取得するPROPFINDを追加して完成させなさい。
	}

	return bResult;
}

bool
CTFBrowser::DeleteResource(CTFResourceItem* ppItem)
{
	bool	bResult = false;

	CTFTransaction	cTrans;
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	return bResult;
}

bool 
CTFBrowser::CopyResource(CTFResourceItem* ppItem)
{
	return false;
}

bool
CTFBrowser::MoveResource(CTFResourceItem* ppItem)
{
	bool	bResult = false;

	CTFTransaction	cTrans;
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	// TODO: 移動を完成させなさい
	return bResult;
}

bool
CTFBrowser::OpenSSLCertificateList(CTFResourceItem* pItem)
{
	OI_DEBUG("CTFBrowser::OpenSSLCertificateList()\n");

	CTFServerResource* pSvrItem = GetServerManager().GetServerItem(pItem->GetServerID());
	if (pSvrItem == NULL)
	{
		return false;
	}

	CTFSSLCertificate* pCert =  pSvrItem->GetSSLCertificate();
	if (pCert)
	{
		m_pNotifyHandler->OnNotifySSLCerts(pCert);
	}

	return (pCert != NULL);
}

void
CTFBrowser::SetHandler(CTFWsSystemInformationHandler* pHander)
{
	m_pSysInfoHandler = pHander;
}

void
CTFBrowser::SetHandler(CTFWsNotificationHandler* pHandler)
{
	m_pNotifyHandler = pHandler;
}

void
CTFBrowser::SetHandler(CTFWsTransactionHandler* pHandler)
{
	m_pTransHandler = pHandler;
}

void
CTFBrowser::queryMethod(CTFResourceItem* pItem, int& nAllowMethod, TFSEARCHTYPE& enuAllowDASL, int& nDASLSubType)
{

	nAllowMethod	= TF_DAV_UNKNOWN;
	enuAllowDASL	= TF_DASL_UNKNOWN;
	nDASLSubType	= 0;

	TFDAVMETHOD	enuMethod = TF_DAV_UNKNOWN;

	int nType = pItem->GetType();

	// ROOTTREEの不明なエレメントが登場した場合は
	// CTFResourceItemクラスでは判断がつかないCASTして再度中身を調査します
	if (nType == TFITEM_MNG_MELEMENT)
	{
		CTFResourceTFRootTree* pRtItem = TFLIB_DYNAMIC_CAST(CTFResourceTFRootTree, pItem);
		if (pRtItem)
			nType = pRtItem->GetFolderType();
	}

	switch(nType)
	{
		case TFITEM_SERVER:			//< サーバアイテム
			enuMethod = openServer(pItem);
			if (enuMethod == TF_DAV_SEARCH)
			{
				nAllowMethod	= TF_DAV_SEARCH;
				enuAllowDASL	= TF_DASL_ROOTTREE;
				nDASLSubType	= TF_SEARCH_RT_ALL;
			}
			else {
				nAllowMethod	= TF_DAV_PROPFIND;
				enuAllowDASL	= TF_DASL_UNKNOWN;
				nDASLSubType	= 0;
			}
			break;

		case TFITEM_FOLDER:				//< フォルダ
			nAllowMethod = (TF_DAV_OPTIONS | TF_DAV_HEAD | TF_DAV_PROPFIND | TF_DAV_DELETE | TF_DAV_COPY | TF_DAV_MOVE | TF_DAV_LOCK | TF_DAV_UNLOCK | TF_DAV_SEARCH);
			break;

		case TFITEM_FILE:				//< ファイル
			nAllowMethod = (TF_DAV_GET | TF_DAV_OPTIONS | TF_DAV_HEAD | TF_DAV_PROPFIND | TF_DAV_DELETE | TF_DAV_COPY | TF_DAV_MOVE | TF_DAV_LOCK | TF_DAV_UNLOCK | TF_DAV_SEARCH);
			break;

		case TFITEM_USER:				//< ユーザフォルダ
			nAllowMethod = (TF_DAV_OPTIONS | TF_DAV_HEAD | TF_DAV_PROPFIND | TF_DAV_DELETE | TF_DAV_COPY | TF_DAV_MOVE | TF_DAV_LOCK | TF_DAV_UNLOCK | TF_DAV_SEARCH);
			break;

		case TFITEM_GROUP_ROOT:			//< グループフォルダ(ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_ROOTTREE;
			nDASLSubType	= TF_SEARCH_RT_GROUP;
			break;

		case TFITEM_GROUP:				//< グループフォルダのグループ
			nAllowMethod = (TF_DAV_OPTIONS | TF_DAV_HEAD | TF_DAV_PROPFIND | TF_DAV_DELETE | TF_DAV_COPY | TF_DAV_MOVE | TF_DAV_LOCK | TF_DAV_UNLOCK | TF_DAV_SEARCH);
			break;

		case TFITEM_DBLINK_ROOT:		//< データベース連携フォルダ(ルート)
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_ROOTTREE;
			nDASLSubType	= TF_SEARCH_RT_SQLSTMT;
			break;

		case TFITEM_REPOSDB_ROOT:	//< リポジトリデータベースフォルダ(ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_ROOTTREE;
			nDASLSubType	= TF_SEARCH_RT_REPOSDBSQLSTMT;
			break;

		case TFITEM_MNG_ROOT:			//< マネージメントフォルダ(ルート)
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_ROOTTREE;
			nDASLSubType	= TF_SEARCH_RT_MANAGEMENT;
			break;

		case TFITEM_MNG_STAT_ROOT:		//< マネージメントフォルダのステータス(ルート)
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_STATUSINFORMATION;
			break;

		case TFITEM_MNG_USR_ROOT:		//< マネージメントフォルダのユーザ(ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_USERINFORMATION;
			nDASLSubType	= TF_SEARCH_UI_DETAILLIST;
			break;

		case TFITEM_MNG_GRP_ROOT:		//< マネージメントフォルダのグループ(ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_GROUPINFORMATION;
			nDASLSubType	= TF_SEARCH_GI_DETAILLIST;
			break;

		case TFITEM_MNG_SQL_ROOT:		//< マネージメントフォルダのSQL(ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_ROOTTREE;
			nDASLSubType	= TF_SEARCH_RT_EXECSQLSTMT;
			break;

		case TFITEM_MNG_MSG_ROOT:		//< マネージメントフォルダのメッセージ(ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_SYSMSGINFORMATION;
			nDASLSubType	= TF_SEARCH_SMI_DETAILLIST;
			break;

		case TFITEM_MNG_CLIENT_ROOT:	//< マネージメントフォルダのクライアント更新（ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_UPDATEINFORMATION;
			break;

		case TFITEM_MNG_DBMS_ROOT:		//< マネージメントフォルダのDBMS（ルート）
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_DBMSINFORMATION;
			nDASLSubType	= TF_SEARCH_DI_DETAILLIST;
			break;

		case TFITEM_MNG_USR:		//< マネージメントフォルダのユーザ(各項目)
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_USERINFORMATION;
			nDASLSubType	= TF_SEARCH_UI_CONTENT | TF_SEARCH_UI_AVAILABLEGROUP;
			break;

		case TFITEM_MNG_MSG:		//< マネージメントフォルダのメッセージ(各項目)
			nAllowMethod	= TF_DAV_SEARCH;
			enuAllowDASL	= TF_DASL_SYSMSGINFORMATION;
			nDASLSubType	= TF_SEARCH_SMI_CONTENT | TF_SEARCH_SMI_CURRENTMSG;
			break;

		case TFITEM_MNG_GRP:		//< マネージメントフォルダのグループ(各項目)
		case TFITEM_MNG_SQL:		//< マネージメントフォルダのSQL(各項目)
		case TFITEM_MNG_CLIENT:		//< マネージメントフォルダのクライアント更新(各項目)

		case TFITEM_REPOSDB:		//< リポジトリデータベースフォルダ(各項目)
		case TFITEM_DBLINK:			//< データベース連携フォルダ(各項目)
		case TFITEM_MNG_STAT:		//< マネージメントフォルダの統計
		case TFITEM_MNG_DBMS:		//< マネージメントのDBMS（各項目）
		default:
			nAllowMethod	= TF_DAV_UNKNOWN;
			enuAllowDASL	= TF_DASL_UNKNOWN;
			nDASLSubType	= 0;
			break;
	}
}

/**
 *	サーバを開いてどのようなサーバかを調べる
 *	@param	pItem		CTFResourceItem
 *	@return TFDAVMETHOD
 */
TFDAVMETHOD 
CTFBrowser::openServer(CTFResourceItem* pItem)
{
	OI_ASSERT(pItem);

	CTFTransaction	cTrans;

	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	int nAllowMethod;
	int nAllowDASL;

	return cTrans.GetServerOptions(pItem, NULL, nAllowMethod, nAllowDASL);
}

/**
 *	リソースをPROPFINDで開く
 *	@param	pItem		CTFResourceItem*
 *	@return boolean
 */
bool 
CTFBrowser::openbyPROPFIND(CTFResourceItem* pItem)
{
	OI_ASSERT(pItem);

	bool bResult;
	CTFTransaction		cTrans;
	CTFResourceProp*	pPropItem = 0;

	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	bResult = cTrans.List(pItem, &pPropItem);

	CTFDavResourceList* pItemList = TFLIB_DYNAMIC_CAST(CTFDavResourceList, pPropItem);

	// TODO: pItemListはどうするの？
	delete pPropItem;
	return bResult;
}


bool 
CTFBrowser::openbySEARCH(CTFResourceItem* pItem, TFSEARCHTYPE enuAllowDASLType, int nDASLSubType)
{
	OI_ASSERT(pItem);

	CTFTransaction		cTrans;
	bool				bResult = false;
	CTFResourceProp*	pPropItem = 0;

	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	if (enuAllowDASLType == TF_DASL_ROOTTREE)
	{
		bResult = cTrans.SearchRootTree(pItem, &pPropItem, (TFSEARCH_RT)nDASLSubType);
	}

	if (enuAllowDASLType == TF_DASL_STATUSINFORMATION)
	{
		bResult = cTrans.SearchStatus(pItem, &pPropItem);
	}

	if (enuAllowDASLType == TF_DASL_USERINFORMATION)
	{
		CTFResourcePropTFUser* pUsrItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFUser, pItem);
		bResult = cTrans.SearchUser(pItem->GetServerID(), pUsrItem, &pPropItem, (TF_SEARCH_UI)nDASLSubType);
	}

	if (enuAllowDASLType == TF_DASL_GROUPINFORMATION)
	{
		CTFResourcePropTFGroup* pGrpItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFGroup, pItem);
		bResult = cTrans.SearchGroup(pItem->GetServerID(), pGrpItem, &pPropItem, (TF_SEARCH_GI)nDASLSubType);
	}

	// システムメッセージインフォメーションサーチ
	if (enuAllowDASLType == TF_DASL_SYSMSGINFORMATION)
	{
		OI_STRING_A strID = "";
		if (nDASLSubType == TF_SEARCH_SMI_DETAILLIST)
		{
			bResult = cTrans.SearchSysMsg(pItem->GetServerID(), &pPropItem, (TF_SEARCH_SMI)nDASLSubType, NULL);
		}
		else 
		{
			CTFResourcePropTFMsg* pMsgItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFMsg, pItem);
			OI_ASSERT(pMsgItem);
			strID = (const char*)X(pMsgItem->GetID());
//			bResult = cTrans.SearchSysMsg(pItem, &pPropItem, (TF_SEARCH_SMI)nDASLSubType, strID.c_str());
		}
	}

	if (enuAllowDASLType == TF_DASL_UPDATEINFORMATION)
	{
		bResult = cTrans.SearchUpdate(pItem, &pPropItem);
	}

	if (enuAllowDASLType == TF_DASL_DBMSINFORMATION)
	{
		bResult = cTrans.SearchDBMS(pItem, &pPropItem, (TF_SEARCH_DI)nDASLSubType, NULL);
	}

	if (pPropItem != NULL) delete pPropItem;

	return bResult;
}

bool
CTFBrowser::openbyGET(CTFResourceItem* pItem)
{
	OI_ASSERT(pItem);
	bool	bResult = false;

	CTFTransaction	cTrans;

	cTrans.SetHandler(m_pTransHandler);
	cTrans.SetHandler(m_pNotifyHandler);
	cTrans.SetHandler(m_pSysInfoHandler);

	return bResult;
}

bool
CTFBrowser::getServerItems(void)
{

	// GetServerItemList()は複製を渡す為必ず最後にdeleteする
	CTFDavResourceList* pItemList = GetServerManager().GetServerItemList();
	if (pItemList != NULL)
	{
		CTFDavResource* pItem = NULL;
		pItemList->ResetItemPosition();

		while(true)
		{
			pItem = pItemList->GetNextItem();
			if( pItem == NULL ) break;
			m_pTransHandler->OnShowResource(pItem);
		}

		delete pItemList;
	}

	return true;
}

int 
CTFBrowser::QueryAcceptableDropEffect(CTFResourceItem* pTargetItem, CTFResourceItemList* pItemList)
{
	OI_ASSERT(pTargetItem);

	int nDstType = pTargetItem->GetType();

	// ソースが一件も無いかターゲットアイテムのタイプが分からない場合
	if (pItemList->GetItemCount() == 0 || nDstType == TFITEM_UNKNOWN) return TF_DL_NONE;

	// ターゲットアイテムがファイルの場合
	if (nDstType == TFITEM_FILE) return TF_DL_NONE;

	//	アイテム一覧のタイプをすべて収集する
	CTFResourceItem* pItem = NULL;
	int	nSrcType = TF_DAV_UNKNOWN;
	for (; pItem = pItemList->GetNextItem(); )
	{
		nSrcType |= pItem->GetType();
	}

	int nNOTDrop = (TFITEM_SERVER			| TFITEM_USER			| TFITEM_GROUP_ROOT		| TFITEM_GROUP		|
					TFITEM_DBLINK_ROOT		| TFITEM_DBLINK			| TFITEM_REPOSDB_ROOT	| TFITEM_REPOSDB	|
					TFITEM_MNG_ROOT			| TFITEM_MNG_MELEMENT	| TFITEM_MNG_CLIENT_ROOT| TFITEM_MNG_CLIENT	|
					TFITEM_MNG_MSG_ROOT		| TFITEM_MNG_MSG		| TFITEM_MNG_USR_ROOT	| TFITEM_MNG_USR	|
					TFITEM_MNG_GRP_ROOT		| TFITEM_MNG_GRP		| TFITEM_MNG_STAT_ROOT	| TFITEM_MNG_STAT	|
					TFITEM_MNG_SQL_ROOT		| TFITEM_MNG_SQL		| TFITEM_MNG_DBMS_ROOT	| TFITEM_MNG_DBMS	|
					TFITEM_MNG_EXECSQL_ROOT | TFITEM_MNG_EXECSQL	| TFITEM_TRASH_FOLDER
					);

	// ターゲットがフォルダの場合で受け入れ不可能チェック
	if ( (nDstType == TFITEM_FOLDER) && (nSrcType & nNOTDrop) )
	{
		return TF_DL_NONE;
	}

	if ( (nDstType == TFITEM_FOLDER) && nSrcType & (TFITEM_FILE | TFITEM_FOLDER) )
	{
		return (TF_DL_MOVE | TF_DL_COPY);
	}

	// TODO: タイプによってpTargetItemが受け入れ可能かを調べなさい。

	return TF_DL_NONE;

}



