#include "TFDavResource.h"
#include "TFServerResource.h"
#include "TFServerManager.h"
#include "TFUtils.h"

TFLIB_IMPLEMENT_CLASS(CTFDavResource,"CTFDavResource")

CTFDavResource::CTFDavResource(void)
{
	TFLIB_CONSTRUCT_CLASS(CTFDavResource);

	m_bValid = false;
}

CTFDavResource::~CTFDavResource(void)
{
}

/**
 *	メンバー変数を取り込む
 *
 *	@param	source	const CTFObject&	コピー元のオブジェクト
 *
 */
void CTFDavResource::Import(const CTFObject& source)
{
	TFLIB_PREPARE_IMPORT(CTFDavResource, CTFResourceItem, source, psource);

	m_bValid = psource->m_bValid;
	m_strURI = psource->m_strURI;
	m_strURL = psource->m_strURL;
}

/**
 *	このリソースオブジェクトのURIを得る
 *
 *	@return	const TFXMLCh*	URI
 *
 */
const TFXMLCh*
CTFDavResource::GetURI(void)
{
	return m_strURI.c_str();
}

/**
 *	このリソースオブジェクトのを作ったリソースのURI
 *	@return TFXMLCh*		URI
 */
const TFXMLCh*
CTFDavResource::GetTrigerURI(void)
{
	return m_strTrigerURI.c_str();
}

const TFXMLCh*
CTFDavResource::GetParentURI(void)
{
	m_cURL.parseURL(m_strURI.c_str());
	return m_cURL.m_strDir.c_str();
}

const TFXMLCh* 
CTFDavResource::GetParentURL(void)
{
	m_cURL.parseURL(GetURL());
	m_cURL.BuildURL(true, true);
	return m_cURL.m_strDir.c_str();
}

void
CTFDavResource::SetTrigerURI(const TFXMLCh* pszURI)
{
	m_strTrigerURI = pszURI;
}

/**
 *	このリソースオブジェクトのURLを得る
 *
 *	@return	const TFXMLCh*	URL
 *
 */
const TFXMLCh*
CTFDavResource::GetURL(void)
{
	if (m_strURL.empty())
	{
		CTFServerResource* pSvrResource = GetServerManager().GetServerItem(GetServerID());
		OI_ASSERT(pSvrResource);

		bool bUseSSL = pSvrResource->GetUseSSL();

		if (bUseSSL)
			m_strURL = X("https://");
		else
			m_strURL = X("http://");

		m_strURL += pSvrResource->GetHost();

		int  uPort   = pSvrResource->GetPort();
		if ( (bUseSSL && uPort == 443) || (!bUseSSL && uPort == 80) )
		{
			// デフォルトのポートの場合は何もしない
		}
		else 
		{
			char chPort[30];
			sprintf(chPort, ":%d", uPort);
			m_strURL += X(chPort);
		}

		m_strURL += GetURI();
	}

	return m_strURL.c_str();
}

/**
 *	このリソースオブジェクトのURIを設定する
 *
 *	@param	pszURI	const TFXMLCh*	URI
 *
 */
void 
CTFDavResource::SetURI(const TFXMLCh* pszURI)
{
	m_strURI = pszURI;
	RemoveDoubleSlashW(m_strURI);
}

/**
 *	このリソースオブジェクトのURLを設定する
 *
 *	@param	pszURL	const TFXMLCh*	URL
 *
 */
void
CTFDavResource::SetURL(const TFXMLCh* pszURL)
{
	OI_ASSERT(pszURL);
	m_strURL = pszURL;
}

/**
 *	指定したURIがこのリソースのURIと同じであるかを調べる
 *
 *	@param	pszURI	const TFXMLCh*	URI
 *	@return	bool				URIが同じであればtrue
 *
 */
bool
CTFDavResource::IsURI(const TFXMLCh* pszURI)
{
	return (m_strURI.compare(pszURI) == 0);
}

/**
 *	指定したURLがこのリソースのURLと同じであるかを調べる
 *
 *	@param	pszURI	const TFXMLCh*	URL
 *	@return	bool				URLが同じであればtrue
 *
 */
bool
CTFDavResource::IsURL(const TFXMLCh* pszURL)
{
	return (m_strURI.compare(pszURL) == 0);
}


/**
 *	このリソースオブジェクトが有効であるかを調べる。
 *
 *	@return	bool	有効である場合はtrueを返す
 *
 */
bool
CTFDavResource::IsValid(void)
{
	return m_bValid;
}

/**
 *	このリソースオブジェクトを有効にする。
 *	パーサーにおいて、パース結果をこのリソースオブジェクトに格納し終えたときに呼び出す。
 *	これにより、リソースオブジェクトが有効な情報を保持していることがIsValid()メソッドで
 *	判断可能となる
 *
 *	@param	bValid	bool	このリソースを有効とする場合にtrueを与える
 *	@return	bool			常にtrueを返す
 *
 */
bool
CTFDavResource::Validate(bool bValid)
{
	m_bValid = bValid;
	return true;
}



