#include "TFLibTransactionBase.h"
#include "TFLibWorkSessionWorker.h"
#include "TFLibWorkSessionManager.h"
#include "TFServerManager.h"
#include "TFUtils.h"

CTFLibTransactionBase::CTFLibTransactionBase(void)
:m_enuLastResult(TF_RESULT_NOERROR)
{
	m_pNotifyHandler = &m_cNotifyHandlerDef;
	m_pTransHandler	 = &m_cTransHandlerDef;
	m_pSysHandler	 = &m_cSysHandler;

	m_pPFFactory	 = new CTFPlatformsFactory;
}

CTFLibTransactionBase::~CTFLibTransactionBase(void)
{
	delete m_pPFFactory;
}

/**
 *	SetHandler
 *	@param	Handler	CTFWsNotificationHandler
 *	@return void
 *
 *	(note)
 *	このクラスで利用する通知等につかうハンドラクラスを設定する
 *	このメソッドで設定したハンドラは呼び出し元が保証する必要があります。
 */
void
CTFLibTransactionBase::SetHandler(CTFWsNotificationHandler* pHandler)
{
	m_pNotifyHandler = pHandler;
}

void 
CTFLibTransactionBase::SetHandler(CTFWsTransactionHandler* pHandler)
{
	m_pTransHandler = pHandler;
}

void 
CTFLibTransactionBase::SetHandler(CTFWsSystemInformationHandler* pHandler)
{
	m_pSysHandler = pHandler;
}

/**
 *	サーバの情報を取得する
 *
 *	@param	pItem		CTFResourceItem*
 *	@param	pWsWorker*	CTFLibWorkSessionWorker
 *	@param	int&		nAllowMethods;
 *	@param	int&		nAllowDASL;
 *	@return boolean
 */
TFDAVMETHOD
CTFLibTransactionBase::GetServerOptions(CTFResourceItem* pItem, CTFLibWorkSessionWorker *pWsWorker, int& nAllowMethods, int& nAllowDASL)
{
	OI_RESULT	oResult;
	TFDAVMETHOD enuMethod = TF_DAV_UNKNOWN;
	bool bMyOpen = false;

	nAllowMethods = TF_DAV_UNKNOWN;
	nAllowDASL	  = TF_DASL_UNKNOWN;

	TFTRANSMODE enuMode = TF_TRANS_UNKNOWN;
	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	if (pWsWorker == NULL)
	{
		pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
		bMyOpen = true;
	}
	OI_ASSERT(pWsWorker);

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem != NULL) {
		pWsWorker->SetHandler(m_pTransHandler);
		pWsWorker->SetHandler(m_pNotifyHandler);
		pWsWorker->SetHandler(m_pSysHandler);

		oResult = pWsWorker->OPTIONS(pDavItem->GetURI());
		if (oResult == OI_OK)
		{
			int iAllowMethod = pWsWorker->GetAllowMethod();
			int iAllowDASL   = pWsWorker->GetAllowDASL();
			if ( (iAllowMethod & TF_DAV_SEARCH) && (iAllowDASL & TF_DASL_ROOTTREE) )
				enuMethod = TF_DAV_SEARCH;
			else if (iAllowMethod & TF_DAV_PROPFIND)
				enuMethod = TF_DAV_PROPFIND;
			else
				enuMethod = TF_DAV_UNKNOWN;

			nAllowMethods = iAllowMethod;
			nAllowDASL	  = iAllowDASL;
		}
	}
	else {
		enuMethod = TF_DAV_UNKNOWN;
	}

	if (bMyOpen)	CTFLibWorkSessionManager::CloseSession(pWsWorker);

	return enuMethod;
}

bool
CTFLibTransactionBase::GetLock(CTFResourceItem* pItem, CTFLibWorkSessionWorker* pWsWorker)
{
	OI_ASSERT(pItem);
	OI_ASSERT(pWsWorker);

	bool bResult = false;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem != NULL)
	{
		// TODO: PROPFINDの実装を入れなさい
	}

	return bResult;
}

OI_RESULT
CTFLibTransactionBase::GetLock(const TFXMLCh* pszURI, CTFLibWorkSessionWorker* pWsWorker, OI_STRING_A& strLockToken, long lTimeout)
{
	OI_ASSERT(!TF_STRING_W(pszURI).empty());
	OI_ASSERT(pWsWorker);

	OI_RESULT	oResult;
	strLockToken.erase();

	CTFServerResource* pSvrItem = GetServerManager().GetServerItem(pWsWorker->GetServerID());
	if (pSvrItem->GetOmitLockFlag())
	{
		// ロックを利用しない明示的な場合
		return OI_OK;
	}
	else
	{
		// DAVのCLASS2(LOCK系）を利用するか調べる
		if (m_pTransHandler->OnUseDAVCLASS2() == false) return OI_OK;
	}

	CDavLock	cLock;
	cLock.m_unMask   = OI_LIVF_TIMEOUT;
	cLock.m_lTimeout = lTimeout;
	cLock.m_enuType  = T_LOCK_WRITE;
	cLock.m_enuScope = SCP_LOCK_EXCLUSIVE;
	cLock.m_enuDepth = D_LOCK_INFINITE;
	// TODO: オーナーを入れなさい。
	cLock.m_pOwner   = NULL;
	cLock.m_strToken = "";

	oResult = pWsWorker->LOCK(pszURI, &cLock, NULL);
	if (oResult == OI_OK)
	{
		strLockToken = cLock.m_strToken.c_str();
	}

	return oResult;
}

OI_RESULT
CTFLibTransactionBase::UnLock(const TFXMLCh* pszURI, const char* pszLockToken, CTFLibWorkSessionWorker* pWsWorker)
{
	OI_ASSERT(!TF_STRING_W(pszURI).empty());
	OI_ASSERT(pWsWorker);

	// ロックトークンが存在しない場合メソッドを投げる必要がない。
	if (IS_EMPTY(pszLockToken)) return OI_OK;

	// DAVのCLASS2(LOCK系）を利用するか調べる
	if (m_pTransHandler->OnUseDAVCLASS2() == false) return OI_OK;

	return pWsWorker->UNLOCK(pszURI, pszLockToken);
}

bool
CTFLibTransactionBase::ExistResource(const TFXMLCh* pszURI, LOCKLIST& vLock, OI_RESULT& oResult, CTFLibWorkSessionWorker* pWsWorker)
{
	OI_ASSERT(!TF_STRING_W(pszURI).empty());
	OI_ASSERT(pWsWorker);

	vLock.clear();	// LOCKLISTの初期化
	bool	bResult = false;
	CTFResourceProp* ppItem = 0;

	oResult = pWsWorker->PROPFIND(pszURI, &ppItem, D_PFIND_ZERO, LP_EXISTCHECK);
	if (oResult == OI_OK)
	{
		vLock = ppItem->GetLocklist();
		bResult = true;

	}

	// 検索するだけなのでアイテムは全て不要
	if (ppItem) delete ppItem; ppItem = 0;

	return bResult;

}

bool
CTFLibTransactionBase::GetOneResource(const TFXMLCh* pszURI, CTFResourceProp** pItem, CTFLibWorkSessionWorker* pWsWorker)
{
	OI_ASSERT(!TF_STRING_W(pszURI).empty());

	bool bResult = false;
	OI_RESULT oResult;
	CTFResourceProp* ppItem=0;

	oResult = pWsWorker->PROPFIND(pszURI, &ppItem, D_PFIND_ZERO, LP_ALL);
	if (oResult == OI_OK)
	{
		//　一件だけを目的としている為、取得するのは始めの一件だけでよい（戻りも複数はない）
		CTFDavResourceList* pList = TFLIB_DYNAMIC_CAST(CTFDavResourceList, ppItem);
		if (pList->GetItemCount() == 0)
			*pItem = (CTFResourceProp*)ppItem->Clone();
		else
			*pItem = (CTFResourceProp*)pList->GetNextItem()->Clone();

		delete ppItem;
	}
	else {
		CTFDavResource cDavResource;
		cDavResource.SetURI(pszURI);
		QueryDAVError(TF_TRANS_GETRESOURCE, TF_DAV_GET, oResult, &cDavResource);
	}

	return (oResult == OI_OK);
}

bool
CTFLibTransactionBase::GetResourceListRecursive(const TFXMLCh* pszURI, CTFResourceProp** pItem, CTFLibWorkSessionWorker* pWsWorker)
{

	TF_STRING_W szURI = pszURI;
	OI_ASSERT(!szURI.empty());

	bool bResult = false;
	OI_RESULT oResult;
	CTFResourceProp* ppItem=0;
	CTFDavResourceList cList;

	// 余計なスラッシュを削る
	RemoveSlash(szURI);
	oResult = pWsWorker->PROPFIND(pszURI, &ppItem, D_PFIND_ONE, LP_ALL);
	if (oResult == OI_OK)
	{
		CTFDavResourceList* pList = TFLIB_DYNAMIC_CAST(CTFDavResourceList, ppItem);
		CTFDavResource* pDavItem;
		for (pDavItem = NULL; pDavItem = (CTFDavResource*)pList->GetNextItem(); )
		{
			// 追加
			cList.AddItem((CTFDavResource*)pDavItem->Clone());
			TF_STRING_W szDavURI = pDavItem->GetURI();
			RemoveSlash(szDavURI);
			// フォルダの場合さらに再帰的に取得する
			// 引数（pszURIと同じURIも戻ってしまう（自分自身）為対象外）
			if (((CTFResourceProp*)pDavItem)->IsCollection() && (szURI.compare(szDavURI.c_str()) != 0) )
			{
				CTFResourceProp* pInProp = 0;
				if (!GetResourceListRecursive(pDavItem->GetURI(), &pInProp, pWsWorker))
				{
					return false;
				}

				// 取得した内容をリストに追加する
				CTFDavResourceList* ptmpDavList = TFLIB_DYNAMIC_CAST(CTFDavResourceList, pInProp);
				if (ptmpDavList)
				{
					CTFDavResource* pInsItem=NULL;
					while( (pInsItem = (CTFDavResource*)ptmpDavList->GetNextItem() ))
					{
						cList.AddItem((CTFDavResource*)pInsItem->Clone());
					}

					delete pInProp;
				}
			}
		}

		delete ppItem;
	}

	if (cList.GetItemCount())
	{
		*pItem = (CTFDavResourceList*)cList.Clone();
	}

	return true;
}


void
CTFLibTransactionBase::SetDebugMode(bool bDebug, int nFD)
{
	m_bDebug   = bDebug;
	m_nDebugFD = nFD;
}

TFRESULT
CTFLibTransactionBase::GetLastError(void)
{
	return m_enuLastResult;
}

TFDLGNOTIFYTYPE
CTFLibTransactionBase::QueryDAVError(TFTRANSMODE enuMode, TFDAVMETHOD enuDAVMethod, OI_RESULT oResult, CTFObject* pObj)
{
	TFDLGNOTIFYTYPE enuType = TF_DLG_NOTIFY_NONE;

	m_enuLastResult = TF_RESULT_NOERROR;

	if (enuMode == TF_TRANS_GETRESOURCE)
	{
		if (enuDAVMethod == TF_DAV_GET)
		{
			switch(oResult)
			{
				case OI_OK:
					m_enuLastResult = TF_RESULT_OK;
					break;

				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}
	}

	// ファイルアップロード（PUT)
	if (enuMode == TF_TRANS_UPLOADRESOURCE)
	{
		if (enuDAVMethod == TF_DAV_PUT)
		{
			switch(oResult)
			{
				case OI_OK:				// 正常
					m_enuLastResult = TF_RESULT_OK;
					break;
				case OIDENOCONTENT:		// 上書き（正常）
					m_enuLastResult = TF_RESULT_RESOURCE_NOCONTENT;
					break;
				case OIDECONFLICT:		// 親がない
					m_enuLastResult = TF_RESULT_PARENT_NOTFOUND;
					break;
				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}

		if (enuDAVMethod == TF_DAV_MKCOL)
		{
			switch(oResult)
			{
				case OI_OK:				// 正常
					m_enuLastResult = TF_RESULT_OK;
					break;
				case OIDENOTALLOWED:	// 上書き（正常）
					m_enuLastResult = TF_RESULT_FOUNDFOLDER;
					break;
				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}
	}

	// ユーザ検索
	if (enuMode == TF_TRANS_USERSEARCH)
	{
		if (enuDAVMethod == TF_DAV_SEARCH)
		{
			switch(oResult)
			{
				case OIDENOTFOUND:
					m_enuLastResult = TF_RESULT_NORESULT;	// ユーザが存在しなかった
					break;

				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}
	}

	// ユーザ作成
	if (enuMode == TF_TRANS_MAKEUSER)
	{
		if (enuDAVMethod == TF_DAV_MKCOL)
		{
			switch(oResult)
			{
				case OIDEPAYMENTREQUIRED:
					m_enuLastResult = TF_RESULT_INSUFFICIENTLICENSE;
					break;

				case OIDEINSUFFSTG:
					m_enuLastResult = TF_RESULT_EXCEEDEDLICENSEDCAPACITY;
					break;

				case OIDEBADREQUEST:
					m_enuLastResult = TF_RESULT_BADREQUEST;
					break;

				case OIDENOTALLOWED:
					m_enuLastResult = TF_RESULT_USEREXIST;
					break;

				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}
	}

	// グループ作成
	if (enuMode == TF_TRANS_MAKEGROUP)
	{
		if (enuDAVMethod == TF_DAV_MKCOL)
		{
			switch(oResult)
			{
				case OIDEBADREQUEST:
					m_enuLastResult = TF_RESULT_BADREQUEST;
					break;

				case OIDENOTALLOWED:
					m_enuLastResult = TF_RESULT_GROUP_RESOURCE_EXIST;
					break;

				default:
					break;
			}
		}
	}

	// リソース削除
	if (enuMode == TF_TRANS_DELETE)
	{
		if (enuDAVMethod == TF_DAV_DELETE)
		{
			switch(oResult)
			{
				case OIDENOTFOUND:		// リソースが存在しなかった
					m_enuLastResult = TF_RESULT_NOTFOUND;
					break;

				case OIDEFAILEDDEPS:	// 削除しようとしたが依存されているリソースがあった
					m_enuLastResult = TF_RESULT_FAILDDEPEND;
					break;

				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}
	}

	// フォルダ作成
	if (enuMode == TF_TRANS_MAKEFOLDER)
	{
		// MKCOL
		if (enuDAVMethod == TF_DAV_MKCOL)
		{
			switch(oResult)
			{
				case OIDENOTFOUND:		// フォルダが存在しなかった
				case OIDENOTALLOWED:	// 既にフォルダが存在した
					m_enuLastResult = TF_RESULT_CANTCREATEFOLDER;
					break;

				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}

		// PROPFIND
		if (enuDAVMethod == TF_DAV_PROPFIND)
		{
			switch(oResult)
			{
				case OIDENOTFOUND:		// フォルダが存在しなかった
					m_enuLastResult = TF_RESULT_NOTFOUND;
					break;

				case OIDENOTALLOWED:
					/* 
					 * TODO:
					 * PROPFIND のエラーでNOTALLOWDはメソッドが禁止されているか
					 * ロケーションがないかわからない
					 *
					 */
					m_enuLastResult = TF_RESULT_NOTFOUND;
					break;

				default:
					// デフォルトは最後に処理をお願いする
					break;
			}
		}
	}

	// リネーム
	if (enuMode == TF_TRANS_RENAME)
	{
		if (enuDAVMethod == TF_DAV_MOVE)
		{
		}
	}

	// どのエラーコードの設定もされなかった場合はデフォルトを呼び出す
	if (m_enuLastResult == TF_RESULT_NOERROR)
	{
		QueryDAVError_Default(oResult);
		enuType = m_pNotifyHandler->OnNotifyMessage(m_enuLastResult, TF_DLG_NOTIFY_OK);

	}

	// デバッグを出力する
	m_pTransHandler->SetErrorCodeNative(oResult, CTFWsTransactionHandler::TF_TRANS_ERROR_TYPE_NET, pObj);

	return enuType;
}

TFDLGNOTIFYTYPE
CTFLibTransactionBase::QueryDAVError_Default(OI_RESULT oResult)
{

	TFDLGNOTIFYTYPE enuType = TF_DLG_NOTIFY_NONE;

	switch(oResult)
	{
		case OI_OK:				// 正常ケース
			// 何もしない
			break;

		case OI_USERCANCELED:	// User Calceled
			m_enuLastResult = TF_RESULT_USERCANCELD;
			break;

		case OIEEGENERIC:		//Generic SSL Error
		case OIEENOENTROPY:		//No entropy source found. could not seed PRNG
		case OIEELACKOFENTROPY:	//SSL disabled due to lack of entropy
		case OIEEDATAINITFAILED://SSL data initialization failed	
		case OIEENOSERVERCERT:	//SSL server certificate not present
		case OIEEINVALIDCERT:	//SSL server certificate invalid
		case OIEECERTCHANGED:	//Server certificate changed, connection may be intercepted
		case OIEECERTCHECKFAILED://Certificate check failed	
		case OIEECNMISMATCH:	//CN Mismatch
		case OIEETUNNELFAILED:	//proxy tunnel failed
			m_enuLastResult = TF_RESULT_SSLGENERICERROR;
			break;

		case OIEEHANDSHAKEFAILED://SSL handshake failed
			m_enuLastResult = TF_RESULT_SSLHANDSHAKEFAILED;
			break;

		case OIEEFATALCONNERROR://fatal error encountred while trying to connect via SSL
			m_enuLastResult = TF_RESULT_SSLCONNECTIONERROR;
			break;

		case OIHELINETOOLONG:		//HeaderLine Too Long
		case OIHETOOMANYHEADERS:	//Too many header fields
		case OIHEPROTOCOLPANIC:		//The response in not a Http status
		case OIHECHUNKSIZE:			//response chunk size parse error
		case OIHEMALFORMEDHEADER:	//malformed http header
			m_enuLastResult = TF_RESULT_HTTPPROTCOLERROR;
			break;

		case OIXEINITFAILED:		//Xerces init failed
		case OIXEPARSERCREATION:	//Parser creation failed
		case OIXEEXCEPTION:			//XML exception
			m_enuLastResult = TF_RESULT_XMLERROR;
			break;

		case OISEGEN:			// socket error
			m_enuLastResult = TF_RESULT_SOCKGENERICERROR;
			break;

		case OISETIMEOUT:		//Connection timed out
			m_enuLastResult = TF_RESULT_TIMEOUT;
			break;

		case OISECLOSED:		//Socket Closed
			m_enuLastResult = TF_RESULT_CLOSED;
			break;
			
		case OISERESET:			//Connection reset by server
			m_enuLastResult = TF_RESULT_RESETBYSERVER;
			break;

		case OISELINETOOLONG:	//Line Too Long
		case OISECONNABORTED:	//Connection aborted by server
		case OISEINITFAILED:	//socket version too low 
		case OISEINITIALIZED:	//socket already initialized
		case OISECLEANUPFAILED:	//socket clean up failed
			m_enuLastResult = TF_RESULT_SOCKGENERICERROR;
			break;

		case OISEHOSTNOTFOUND:	//host not found
			m_enuLastResult = TF_RESULT_HOSTNOTFOUND;
			break;

		case OISEINVALIDSOCK:	//can not get sock from system
		case OISESOCKVER:		//socket version not supported by OS
		case OISECONNFAILED:	//system connection failed
		case OISETRUNC:		    //connection truncated
		case OISEREADFAILED:	//unable to read from socket
			m_enuLastResult = TF_RESULT_SOCKGENERICERROR;
			break;

		case OIDENOTFOUND:		// 404 Not Found
			m_enuLastResult = TF_RESULT_NOTFOUND;
			break;

		case OIGEFILEOPENFAILED://ファイルオープン失敗
			m_enuLastResult = TF_RESULT_LSYS_NOTFOUND;
			break;

		case OIAEFAILEDTOGETCRED:	// 認証エラー（OnAuthenticate Handler return false)
			m_enuLastResult = TF_RESULT_WRONGUSERPASSWORD;
			break;

		case OIDEFORBIDDEN:		// アクセス拒否
			m_enuLastResult = TF_RESULT_FORBIDDEN;
			break;

		case OIDEINTRSVRERR:	//　サーバエラー(接続失敗もこれ）
			m_enuLastResult = TF_RESULT_INTERNALSERVERERROR;
			break;

		default:				// 予期せぬエラー
			m_enuLastResult = TF_RESULT_UNEXPECTED;
			break;
	}

	enuType = m_pNotifyHandler->OnNotifyMessage(m_enuLastResult, TF_DLG_NOTIFY_OK);
	return enuType;
}