#include "TFLibWorkSession.h"
#include "TFServerManager.h"
#include "onion/DavServerPolicy.h"

CTFLibWorkSession::CTFLibWorkSession(void)
{
	m_pNotifyHandler = &m_cNotifyHandlerDef;
	m_pTransHandler  = &m_cTransHandlerDef;
	m_pSysHandler	 = &m_cSysHandlerDef;

	m_pSvrResource = NULL;
	m_hProgress    = NULL;
}

CTFLibWorkSession::~CTFLibWorkSession(void)
{
	if (m_pSvrResource != NULL) delete m_pSvrResource;
}

/**
 *	SetHandler
 *	@param	Handler	CTFWsNotificationHandler
 *	@return void
 *
 *	(note)
 *	このクラスで利用する通知等につかうハンドラクラスを設定する
 *	このメソッドで設定したハンドラは呼び出し元が保証する必要があります。
 */
void
CTFLibWorkSession::SetHandler(CTFWsNotificationHandler* pHandler)
{
	m_pNotifyHandler = pHandler;
}

/**
 *	SetHandler
 *	@param	pHandler	CTFWsTransactionHandler*
 *	@return	void
 */
void
CTFLibWorkSession::SetHandler(CTFWsTransactionHandler* pHandler)
{
	m_pTransHandler = pHandler;
}

/**
 *	SetHandler
 *	@param pHandler		CTFWsSystemInformationHandler
 *	@return void
 */
void
CTFLibWorkSession::SetHandler(CTFWsSystemInformationHandler* pHandler)
{
	m_pSysHandler = pHandler;
}

/**
 *	Onion認証ハンドラ
 *
 *	@param pszReaml		X&
 *	@param strUsername	OI_STRING_A&
 *	@param	strPassws		OI_STRING_A&
 *	@param int				nRetry
 *	@param	enuAuthClass	OI_AUTH_CLASS
 */
bool
CTFLibWorkSession::OnAuthentication(X& pszRealm, OI_STRING_A& strUsername,
							OI_STRING_A& strPasswd, int nRetry, OI_AUTH_CLASS enuAuthClass)
{
	OI_DEBUG("TRACE: CTFLibWorkSession::OnAuthentication\n");
	
	OI_ASSERT(m_pSvrResource);
	bool bResult = false;		// falseで認証を失敗させる

	TF_AUTH_CLASS enuTFAuth = TF_C_AUTH_INVALID;

	switch(enuAuthClass)
	{
		case C_AUTH_SERVER:
			enuTFAuth = TF_C_AUTH_SERVER;
			break;

		case C_AUTH_PROXY:
			enuTFAuth = TF_C_AUTH_PROXY;
			break;

		default:
			OI_ASSERT(false);
			break;
	}

	// 認証回数が問題ないか確認する
	bResult = m_pNotifyHandler->OnQueryAuthRetryCount(nRetry, enuTFAuth);

	if (bResult == true) 
	{
		// 認証回数が0回以上だった場合は始めの認証は失敗しているのでユーザに要求
		if (nRetry > 0) 
		{
			bResult = m_pNotifyHandler->OnAuthenticateNotify(pszRealm, strUsername, strPasswd, nRetry, enuTFAuth);
			if (bResult)
			{
				// 設定された値を保存する
				saveUserProfile(enuAuthClass, strUsername.c_str(), strPasswd.c_str());
			}
		}
		else {
			switch(enuAuthClass)
			{
			case C_AUTH_SERVER:
				// 通常認証
				strUsername = X(m_pSvrResource->GetUserID());
				strPasswd   = X(m_pSvrResource->GetPassWord());
				break;

			case C_AUTH_PROXY:
				// プロキシ認証
				strUsername = X(m_pSvrResource->GetProxyUserID());
				strPasswd   = X(m_pSvrResource->GetProxyPassWord());
				break;

			default:
				bResult = false;
				break;
			}

		}	// if (nRetry > 1)
	}

	// 認証失敗処理
	if (bResult == false)
	{
		m_pNotifyHandler->OnNotifyMessage(TF_RESULT_WRONGUSERPASSWORD, TF_DLG_NOTIFY_OK);
	}

	return bResult;
}

/**
 * SSLの認証の際に呼ばれるハンドラ
 *
 *	@param	pUnauthenticCerts	CDavSSLCertificateList*
 *	@param	boolian				true: 成功 / false: 失敗
 *	(note)
 *	ONIONハンドラ。SSLの証明書でなにか問題があった証明書情報の場合
 *	によばれます。
 *
 */
bool
CTFLibWorkSession::OnVerifyCertificates(CDavSSLCertificateList* pUnauthenticCerts)
{
	bool bResult = false;

	unsigned int unCnt = pUnauthenticCerts->GetCount();
	CTFServerResource* pSvrItem = GetServerManager().GetServerItem(m_strServerID.c_str());
	if (pSvrItem == NULL) return false;

	// システムがSSL証明書をもっているかを調べることができるか問い合わせる
	TF_SSL_CERT_HANDLE_TYPE enuType = TF_SSL_CERT_UNKNOWN;
	m_pSysHandler->OnQueryHasSSLCertificate(enuType);
	CDavSSLCertificateData	cData;
	if (enuType == TF_SSL_CERT_UNKNOWN)
	{
		// 報告をして結果をそのまま戻り値とする
		// falseの場合は証明書に問題があるためエラーとする。
		CTFSSLCertificate cCert;
		cCert.SetCerts((void*)pUnauthenticCerts);
		return m_pNotifyHandler->OnNotifySSLCerts(&cCert);
	}

	CTFSSLCertificate* pCacheCerts = pSvrItem->GetSSLCertificate();
	if (pCacheCerts != NULL)
	{
		for (unsigned int unIndex=0; unIndex < unCnt; unIndex)
		{
		}
	}



	// TODO: キャッシュ（CTFSeverResource)との比較をしてからbooleanを返しなさい
	return bResult;
}

/**
 *	クライアント認証の際に呼ばれるハンドラ
 *	@param	pCretOut		CDavSSLClientCretificate*
 *	@param	pCAList			CDavSSLDName*
 *	@param	nCount			int
 *	@return boolean
 */
bool
CTFLibWorkSession::OnProvideClientCertificate(CDavSSLClientCertificate* pCertOut,
											CDavSSLDName* pCAList, int nCount)
{
	// TODO: 実装をいれなさい
	return true;
}

/*
 *　ポリシーファイルを検索する為のHostヘッダのサーバ比較をするハンドラ
 *
 *	@param pszHeaderValue	HTTPヘッダー
 *	@param pszConfigValue	configファイルの値
 *	@return boolean			true:ヒット /  false:違う 
 */
bool
CTFLibWorkSession::MatchServerString(const char* pszHeaderValue, const char* pszConfigValue)
{
	if (pszHeaderValue != NULL && pszConfigValue != NULL)
	{
		size_t pos = TF_STRING_A(pszHeaderValue).find(pszConfigValue);
		if (TF_STRING_A::npos != pos) return true;
	}

	return false;
}

/**
 *	サーバ情報を設定する
 *
 *	@param	pszServerID		const TFXMLCh*	サーバID
 *	@return void
 */
void
CTFLibWorkSession::SetServerInfo(const TFXMLCh* pszServerID)
{
	OI_DEBUG("TRACE: CTFLibWorkSession::SetServerInfo\n");

	OI_ASSERT(pszServerID);
	m_strServerID = pszServerID;

	if (m_pSvrResource == NULL)
	{
		// GetServerItemはポインタだけ渡す為に複製で保持する
		m_pSvrResource = (CTFServerResource*)GetServerManager().GetServerItem(pszServerID)->Clone();
	}
	OI_ASSERT(m_pSvrResource);

	// Onionの設定

	// Host情報
	m_Server.SetHostName(X(m_pSvrResource->GetHost()));
	m_Server.SetPort(m_pSvrResource->GetPort());

	// ユーザエージェントを問い合わせる
	m_strUserAgent.erase();
	m_pSysHandler->OnRequestUserAgentName(m_strUserAgent);

	// SSLの設定
	SetUseSSL(m_pSvrResource->GetUseSSL());

	// ソケットタイムアウト
	if (m_pSvrResource->GetSocketTimeout() > 0)
		SetSocketTimeout(m_pSvrResource->GetSocketTimeout());

	// プロキシサーバ
	if (!TF_STRING_W(m_pSvrResource->GetProxyHost()).empty())
	{
		SetUseProxy(true);
		m_Proxy.SetHostName((const char*)X(m_pSvrResource->GetProxyHost()));
		m_Proxy.SetPort(m_pSvrResource->GetProxyPort());
	}

	// read onion policy.xml
	TF_STRING_W strConfigPath;
	if (m_pSysHandler->OnRequestConfigPath(TFSETTINGXMLTAG_POLICY, strConfigPath))
	{
		CDavServerPolicy *pServerPolicy = GetServerPolicy();
		if (pServerPolicy) pServerPolicy->Load(X(strConfigPath.c_str()));
	}

	// 圧縮要求
#ifdef HAVE_LIBZ
	SetUseCompression(m_pSvrResource->GetUseCompression());
#endif
}

/**
 * ワークセッションで利用されるプログレスのハンドルをセットする
 * @param hProgress
 * @return void
 */
void
CTFLibWorkSession::SetProgressHandle(TFHANDLE hProgress)
{
	if (hProgress != NULL) m_hProgress = hProgress;
}

// --------------------------------------------------------------------------------------- 
// PRIVATE
// --------------------------------------------------------------------------------------- 
void
CTFLibWorkSession::saveUserProfile(OI_AUTH_CLASS enuAuthClass, const char* pszUsername, const char* pszPassWd)
{
	OI_ASSERT(pszUsername);
	OI_ASSERT(pszPassWd);

	switch(enuAuthClass)
	{
		case C_AUTH_SERVER:
			m_pSvrResource->SetUserID(X(pszUsername));
			m_pSvrResource->SetPassWord(X(pszPassWd));
			break;

		case C_AUTH_PROXY:
			m_pSvrResource->SetProxyUserID(X(pszUsername));
			m_pSvrResource->SetProxyPassWord(X(pszPassWd));
			break;

		default:
			break;
	}

	// #### TODO: サーバアイテムだけじゃなくて実体のXMLファイルにもセーブする機構を書きましょう
}
