#include "TFLibWorkSessionWorker.h"
#include "TFServerManager.h"
#include <time.h>

CTFLibWorkSessionWorker::CTFLibWorkSessionWorker()
								: m_iMethod(TF_DAV_UNKNOWN),
								  m_iDASL(TF_DASL_UNKNOWN),
								  m_enuState(TF_CONN_IDLE),
								  m_unRefCount(0)
{
	// デフォルトハンドラの設定
	m_pTransHandler	 = &m_cTransHandler;
	m_pNotifyHandler = &m_cNotifyHandler;
	m_pSysHandler	 = &m_cSysHandler;

	// ワークセッションの作成
	m_pSession = new CTFLibWorkSessionWrapper;

	m_enuState		= TF_CONN_IDLE;		// アイドルに設定
	m_tCreateTime	= time(NULL);		//	作成時間を保持
	m_pSvrItem		= NULL;				// サーバアイテム
	m_strServerID	= X("");			// サーバID
	m_hProgress		= NULL;				// プログレスハンドル
}

CTFLibWorkSessionWorker::~CTFLibWorkSessionWorker(void)
{
	if (m_pSession != NULL) delete m_pSession;
}

CTFLibWorkSessionWorker&
CTFLibWorkSessionWorker::operator =(const CTFLibWorkSessionWorker& source)
{
	if (&source != this)
	{
		m_pSession		= source.m_pSession;
		m_unSessionID	= source.m_unSessionID;
		m_strServerID	= source.m_strServerID;
		m_tCreateTime	= source.m_tCreateTime;
		*m_pSvrItem		= *source.m_pSvrItem;
		AddRef();
	}
	
	return *this;
}

int
CTFLibWorkSessionWorker::AddRef(void)
{
	return ++m_unRefCount;
}

void
CTFLibWorkSessionWorker::Release(void)
{
	if (this) delete this;
}

int
CTFLibWorkSessionWorker::GetRef(void)
{
	return m_unRefCount;
}

int
CTFLibWorkSessionWorker::Detach(void)
{
	return (--m_unRefCount < 0) ? 0 : m_unRefCount;
}

bool 
CTFLibWorkSessionWorker::CreateSession(void)
{
	if (m_pSession == NULL) return false;
	m_pSession->SetHandler(GetServerManager().GetSysHandler());
	m_pSession->SetServerInfo(m_strServerID.c_str());

	AddRef();

	// セッションIDを作成する
	// 中身はthisのアドレスです
	m_unSessionID = (TF_SIZE_T)this;
	
	return true;
}

void
CTFLibWorkSessionWorker::SetServerItem(CTFServerResource* pSvrItem)
{
	OI_ASSERT(pSvrItem);

	m_pSvrItem		= pSvrItem;
	m_strServerID	= pSvrItem->GetServerID();
}

CTFServerResource*
CTFLibWorkSessionWorker::GetServerItem(void)
{
	return m_pSvrItem;
}

TF_CONN_STATE
CTFLibWorkSessionWorker::GetStatus(void)
{
	// (note)
	// m_pSession->GetConnectState()は当てにならないらしい
	return m_enuState;
}

void 
CTFLibWorkSessionWorker::setHandler(void)
{
	m_pSession->SetHandler(m_pTransHandler);
	m_pSession->SetHandler(m_pNotifyHandler);
	m_pSession->SetHandler(m_pSysHandler);
}

/**
 *	利用可能なメソッドを文字列から利用しやすい形に変換する
 *	@param	strAllow	OI_STRING_A
 *	@return void
 */
void 
CTFLibWorkSessionWorker::setAvailableMethods(OI_STRLIST_A vecAllow)
{
	m_iMethod = TF_DAV_UNKNOWN;
	OI_STRLIST_A::iterator	it = vecAllow.begin();
	OI_STRING_A strMethod , strValue;

	for (; it != vecAllow.end(); )
	{
		strMethod = *it;
		CDavStringTokenizer cToken(strMethod.c_str(), ",");
		while(cToken.GetNextToken(strValue))
		{
			for (int i=0; TF_DAV_METHODTABLE[i].szMethod != NULL; i++)
			{
				TrimLeftA(strValue);
				TrimRightA(strValue);
				if (strValue.compare(TF_DAV_METHODTABLE[i].szMethod) == 0)
				{
					m_iMethod |= TF_DAV_METHODTABLE[i].iMethod;
					break;
				}
			}
		}

		it++;
	}
}

/**
 *	利用可能なサーチメソッドを文字列から利用しやすい形に変換する
 *	@param	vecURIs	OI_STRLIST_A
 *	@return void
 */
void
CTFLibWorkSessionWorker::setAvailableDASL(OI_STRLIST_A vecURLs)
{
	m_iDASL = TF_DASL_UNKNOWN;
	OI_STRLIST_A::iterator	it = vecURLs.begin();
	OI_STRING_A strURL;
	for (; it != vecURLs.end(); )
	{
		strURL = *it;
		for (int i=0; TF_DASL_METHOD_TABLE[i].szURL != NULL; i++)
		{
			if (strURL.find(TF_DASL_METHOD_TABLE[i].szURL) != OI_STRING_A::npos)
			{
				m_iDASL |= TF_DASL_METHOD_TABLE[i].iType;
				break;
			}
		}

		it++;
	}
}

void 
CTFLibWorkSessionWorker::parseOptionsHeaders(OI_HEADERMAP mapHeaders)
{
	OI_HEADERMAP::iterator it;
	OI_STRING_A	strHeaderName;

	// Allowヘッダ
	strHeaderName = OI_RSPHDR_ALLOW;
	MakeLowerA(strHeaderName);
	it = mapHeaders.find(strHeaderName);
	if (it != mapHeaders.end())
	{
		setAvailableMethods(it->second);
	}

	// DASLヘッダ
	strHeaderName = TF_OPHDR_DASL;
	MakeLowerA(strHeaderName);
	it = mapHeaders.find(strHeaderName);
	if (it != mapHeaders.end())
	{
		setAvailableDASL(it->second);
	}
}

void 
CTFLibWorkSessionWorker::SetHandler(CTFWsTransactionHandler* pHandler)
{
	m_pTransHandler	= pHandler;
}

void
CTFLibWorkSessionWorker::SetHandler(CTFWsNotificationHandler* pHandler)
{
	m_pNotifyHandler = pHandler;
}

void
CTFLibWorkSessionWorker::SetHandler(CTFWsSystemInformationHandler* pHandler)
{
	m_pSysHandler	= pHandler;
}

OI_RESULT 
CTFLibWorkSessionWorker::OPTIONS(const TFXMLCh* pszTargetURI)
{

	OI_RESULT	oResult;
	OI_STRING_A	strAllowDAV, strDAV, strDASL, strMSV;
	OI_HEADERMAP pHeaders;

	setHandler();

	oResult = m_pSession->DoTFOptions(pszTargetURI, &pHeaders);
	if (OI_OK == oResult)
	{
		parseOptionsHeaders(pHeaders);
	}

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::PROPFIND(const TFXMLCh*	pszTargetURI,
								  CTFResourceProp**	pItem,
								  OI_PFIND_DEPTH	enuDepth,
                                  TFLIVEPROP		enuLive,
                                  TFLIVEPROPTF		enuLiveEx,
                                  TFDEADPROP		enuDead)
{

	OI_RESULT	oResult;

	setHandler();

	if (enuLive == LP_NONE && enuLiveEx == LP_EX_NONE && enuDead == DP_NONE)
		oResult = m_pSession->DoTFPropfind(pszTargetURI, pItem, enuDepth);
	else
		oResult = m_pSession->DoTFPropfindCustom(pszTargetURI, pItem, enuDepth, enuLive, enuLiveEx, enuDead);

	// TODO: Proxy のエラーは特別扱い
	if (m_pSession->m_nLastStatusCode == 407) oResult = OIAEFAILEDTOGETCRED;

	m_pTransHandler->OnResultDAVMethod((const int)oResult, pszTargetURI, TF_DAV_PROPFIND);

	return oResult;	// TODO: Proxy のエラーは特別扱い
	if (m_pSession->m_nLastStatusCode == 407) oResult = OIAEFAILEDTOGETCRED;

}

OI_RESULT
CTFLibWorkSessionWorker::MKCOL(const TFXMLCh* pszTargetURI, const char* pszIfHeader, CRequestBodyProvider *pBodyProvider)
{
	OI_RESULT	oResult;

	setHandler();

	oResult = m_pSession->DoTFMkcol(pszTargetURI, pszIfHeader, pBodyProvider);

	// TODO: Proxy のエラーは特別扱い
	if (m_pSession->m_nLastStatusCode == 407) oResult = OIAEFAILEDTOGETCRED;

	m_pTransHandler->OnResultDAVMethod((const int)oResult, pszTargetURI, TF_DAV_MKCOL);

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::SEARCH(const TFXMLCh* pszTargetURI, CTFResourceProp **pItem, CRBPVDXmlSearch *pXml)
{

	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFSearch(pszTargetURI, pItem, pXml);

	m_pTransHandler->OnResultDAVMethod((const int)oResult, pszTargetURI, TF_DAV_SEARCH);

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::LOCK(const TFXMLCh* pszTargetURI, CDavLock* cLock, const char* pszIfHeader)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFLock(pszTargetURI, cLock, pszIfHeader);

	m_pTransHandler->OnResultDAVMethod((const int)oResult, pszTargetURI, TF_DAV_LOCK);

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::UNLOCK(const TFXMLCh* pszTargetURI, const char* pszLockToken)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFUnlock(pszTargetURI, pszLockToken);

	m_pTransHandler->OnResultDAVMethod(oResult, pszTargetURI, TF_DAV_UNLOCK);

	return oResult;
}

OI_RESULT 
CTFLibWorkSessionWorker::GET(const TFXMLCh* pszTargetURI, const char* pszLocalPath, OICRANGE* pRange)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFGet(pszTargetURI, pszLocalPath, pRange);

	m_pTransHandler->OnResultDAVMethod(oResult, pszTargetURI, TF_DAV_GET);

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::GET(const TFXMLCh* pszURI, int hFile, OICRANGE* pRange)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFGet(pszURI, hFile, pRange);

	m_pTransHandler->OnResultDAVMethod(oResult, pszURI, TF_DAV_GET);

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::PUT(const TFXMLCh* pszURI, int hFile, const char* pszIfHeader, const char* pszContentType)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFPut(pszURI, hFile, pszIfHeader, pszContentType);

	// TODO: Proxy のエラーは特別扱い
	if (m_pSession->m_nLastStatusCode == 407) oResult = OIAEFAILEDTOGETCRED;

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::DELETE(const TFXMLCh* pszTargetURI, const char* pszIfHeader, bool bForce)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFDelete(pszTargetURI, pszIfHeader);

	m_pTransHandler->OnResultDAVMethod(oResult, pszTargetURI, TF_DAV_DELETE);

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::PROPPATCH(const TFXMLCh* pszTargetURI, CRBPVDXmlProppatch* cXml, const char* pszIfHeader)
{
	setHandler();
	OI_RESULT oResult;

	oResult = m_pSession->DoTFProppatch(pszTargetURI, cXml, pszIfHeader);
	m_pTransHandler->OnResultDAVMethod(oResult, pszTargetURI, TF_DAV_PROPPATCH);

	return oResult;
}

OI_RESULT
CTFLibWorkSessionWorker::MOVE(const TFXMLCh* pszSrc, const TFXMLCh* pszDest, CRBPVDXmlMove *pXml, 
					   bool bOverwrite, const char *pszIfHeader)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFMove(pszSrc, pszDest, pXml, bOverwrite, pszIfHeader);

	m_pTransHandler->OnResultDAVMethod(oResult, pszDest, TF_DAV_MOVE);

	return oResult;
}

OI_RESULT 
CTFLibWorkSessionWorker::COPY(const TFXMLCh* pszSrc, const TFXMLCh* pszDest, CRBPVDXmlCopy *pXml, bool bOverwrite, const char *pszIfHeader)
{
	OI_RESULT oResult;

	setHandler();

	oResult = m_pSession->DoTFCopy(pszSrc, pszDest, pXml, bOverwrite, pszIfHeader);

	m_pTransHandler->OnResultDAVMethod(oResult, pszDest, TF_DAV_COPY);

	return oResult;
}

int	
CTFLibWorkSessionWorker::GetAllowMethod(void)
{
	return m_iMethod;
}

int
CTFLibWorkSessionWorker::GetAllowDASL(void)
{
	return m_iDASL;
}

/**
 *	今の接続状態を変更する
 *	@param	enuStat		TF_CONN_STATE
 *	@return void
 */
void 
CTFLibWorkSessionWorker::SetStatus(TF_CONN_STATE enuStat)
{
	// TODO: 同期化が必要
	m_enuState = enuStat;
}

void 
CTFLibWorkSessionWorker::rebuildSession(void)
{
	delete m_pSession;
	m_pSession = new CTFLibWorkSessionWrapper;
	CreateSession();
}

const TFXMLCh*
CTFLibWorkSessionWorker::GetServerID(void)
{
	return m_strServerID.c_str();
}

void
CTFLibWorkSessionWorker::SetProgressHandle(TFHANDLE hProgress)
{
	m_hProgress = hProgress;
	m_pSession->SetProgressHandle(hProgress);
}
