#include "TFUtils.h"
#include "TFServerManager.h"
#include "TFXMLServerSettings.h"
#include "TFServerResource.h"
#include "TFXMLTagServer.h"
#include "TFBroadCaster.h"
#include "TFLocalFileItem.h"

#ifdef _DEBUG
	static int gs_nTFServerMgr = 0;
	#define ADD_SVRMGR	gs_nTFServerMgr++; OI_DEBUG("CTFServerManager::gs_nTFServerMgr: %d\n", gs_nTFServerMgr)
	#define DEL_SVRMGR	gs_nTFServerMgr--; OI_DEBUG("CTFServerManager::gs_nTFServerMgr: %d\n", gs_nTFServerMgr)
#else
	#define ADD_SVRMGR
	#define DEL_SVRMGR
#endif

/**
 *	グローバル変数の初期化
 *
 */
CTFDavResourceList*				CTFServerManager::gs_pSvrItemList   = NULL;
CTFWsSystemInformationHandler	CTFServerManager::gs_cSystemHandler;
CTFWsSystemInformationHandler*	CTFServerManager::gs_pSystemHandler = &CTFServerManager::gs_cSystemHandler;
CTFCriticalSection				CTFServerManager::gs_SMCritSect;
CTFSSLCertificate*              CTFServerManager::gs_pSSLCert       = NULL;
CTFServerManager::TFSSLSYSSTORE CTFServerManager::gs_enuSSLStore    = TF_STORE_NONE;

CTFServerManager& GetServerManager() {
	static CTFServerManager manager;
	return manager;
}

CTFServerManager::CTFServerManager(void)
{
	ADD_SVRMGR;
}

CTFServerManager::~CTFServerManager(void)
{
	DEL_SVRMGR;
}

/**
 *	初期化
 */
void 
CTFServerManager::Initialize(void)
{
	gs_enuSSLStore = TF_STORE_NONE;
	if (gs_pSSLCert != NULL)
	{
		gs_pSSLCert = new CTFSSLCertificate;
	}
}

/**
 *	終了
 */
void
CTFServerManager::Terminate(void)
{
	if (gs_pSSLCert) delete gs_pSSLCert;
	if (gs_pSvrItemList != NULL) 
	{
		// EmptyListは内部のリストをdeleteしてくれる。
		gs_pSvrItemList->EmptyList();
//		delete gs_pSvrItemList;
	}

	gs_pSSLCert     = NULL;
	gs_pSvrItemList = NULL;
}

/**
 *	サーバアイテムを追加する
 *
 *	@param CTFServerResource* pSvrItem
 *	@return boolean
 */
bool 
CTFServerManager::AddServerItem(CTFServerResource* pSvrItem)
{
	if (pSvrItem == NULL) return false;
	CTFServerResource* pItem = (CTFServerResource*)pSvrItem->Clone();
	if (pItem == NULL) return false;

	if (gs_pSvrItemList == NULL)
		gs_pSvrItemList = new CTFDavResourceList;

	gs_pSvrItemList->AddItem(pItem);

	return true;
}

/**
 *	GetServerItem
 *	@param	pszServerID	const char*		サーバのID
 *	@return CTFServerResource*
 *
 *	(note)
 *	サーバのIDを用いてサーバ一件を返却します。
 *
 */
CTFServerResource* 
CTFServerManager::GetServerItem(const TFXMLCh* pszServerID)
{
	OI_DEBUG("TRACE: CTFServerManager::GetServerItem()\n");

	CTFCriticalSectionLocker	cLock(gs_SMCritSect);

	if (gs_pSvrItemList == NULL)
	{
		CTFServerManager::createServerList();
		if (gs_pSvrItemList == NULL) return NULL;
	}

	CTFDavResource *pDavResource = gs_pSvrItemList->GetItemWithServerID(pszServerID);
	CTFServerResource* pSvrItem = TFLIB_DYNAMIC_CAST(CTFServerResource, pDavResource);
	if (pSvrItem == NULL) return NULL;

//	return (CTFServerResource*)pSvrItem->Clone();
	return pSvrItem;
}

/**
 *	インデックスで直接サーバアイテムを取得する。
 *
 *	@param	int							nIndex		インデックス
 *	@return CTFServerResource*
 *
 */
CTFServerResource*
CTFServerManager::GetServerItem(int nIndex)
{
	OI_DEBUG("TRACE: CTFServerManager::GetServerItem()\n");

	CTFCriticalSectionLocker	cLock(gs_SMCritSect);

	if (gs_pSvrItemList == NULL)
	{
		CTFServerManager::createServerList();
		if (gs_pSvrItemList == NULL) return NULL;
	}

	CTFDavResource* pDavResource = gs_pSvrItemList->GetItemWithIndex(nIndex);
	CTFServerResource* pSvrItem = TFLIB_DYNAMIC_CAST(CTFServerResource, pDavResource);
	if (pSvrItem == NULL) return NULL;

	return pSvrItem;
}

/**
 *	サーバの一覧を取得する
 *	コピーが渡される為呼び出し側は必ず削除してください。
 *
 *	@param	void
 *	@return CTFDavResourceLise*
 */
CTFDavResourceList*
CTFServerManager::GetServerItemList(void)
{
	OI_DEBUG("TRACE: CTFServerManager::GetServerItemList()\n");

	CTFCriticalSectionLocker	cLock(gs_SMCritSect);

	if (gs_pSvrItemList == NULL)
	{
		CTFServerManager::createServerList();
		if (gs_pSvrItemList == NULL) return NULL;
	}

	return (CTFDavResourceList*)gs_pSvrItemList->Clone();
}

void
CTFServerManager::AddHandler(CTFWsSystemInformationHandler* pHandler)
{

	CTFCriticalSectionLocker	cLock(gs_SMCritSect);

	gs_pSystemHandler = pHandler;
}

CTFWsSystemInformationHandler* 
CTFServerManager::GetSysHandler(void)
{
	return gs_pSystemHandler;
}

/**
 *	証明書をクリアする
 */
void
CTFServerManager::RemoveCertStore(void)
{
	CTFCriticalSectionLocker	cLock(gs_SMCritSect);

	ClearCertificateStore();
	if (gs_pSSLCert != NULL) gs_pSSLCert->Clear();
}

/**
 *	証明書をシステムへストアする
 *	@param	pCerts		CDavSSLCertificateList*
 */
void
CTFServerManager::SaveSystemServerCert(CTFSSLCertificate* pCert)
{
	OI_ASSERT(pCert);
	CTFCriticalSectionLocker	cLock(gs_SMCritSect);

	unsigned int unCnt = pCert->GetCertCount();
	unsigned int unIndex = 0;

	TF_DEBUG("CTFServerManager::SaveSystemServerCerts() >> CertificateList Count : %d \n", unCnt);

	for (unIndex = 0; unIndex < unCnt; unIndex++)
	{
		void *pData			= NULL;
		unsigned int unLen	= 0;

		pCert->GetCertData(unIndex, &pData, &unLen);

		if (pData && unLen)
		{
			TFHANDLE pHandle;
			if (gs_pSystemHandler->OnOpenSystemStore(TF_SSL_CERT_STORE_NAME_CA, &pHandle))
			{
				gs_pSystemHandler->OnSaveCertStore(pHandle, pData, unLen);
				gs_pSystemHandler->OnCloseCertsStore(pHandle);
			}
		}
	}
}

// ----- Private ------

/**
 *	サーバ一覧を作成する
 *
 *	(note)
 *	グローバル変数(gs_pSvrItemList)が空の場合XMLファイルから
 *	サーバアイテムを作成します。
 *	作成が成功した場合システムのSSL証明書をストアする処理を行います。
 */
void 
CTFServerManager::createServerList(void)
{
	OI_DEBUG("TRACE: CTFServerManager::createServerList()\n");

	if (gs_pSvrItemList == NULL)
	{
		CTFXMLServerSettings cXMLSvrSetting;

		CTFXMLTagServer cXMLTagServer("location");
		TFSETTINGXMLTAG enuTAG = TFSETTINGXMLTAG_SERVER;

		cXMLSvrSetting.AddHandler(enuTAG, &cXMLTagServer);

		CTFLocalFileItem cItem;
		bool bGetPath = gs_pSystemHandler->OnSetConfigurationFilePath(cItem);
		if (bGetPath)
		{
			if (cXMLSvrSetting.LoadXML(enuTAG, cItem.GetPathW()))
			{
				gs_pSvrItemList = TFLIB_DYNAMIC_CAST(CTFDavResourceList, cXMLTagServer.GetServerItemList()->Clone());
			}
			else
			{
				// XMLから取得ができなかった場合は再度直接取得できるか試みる
				CTFResourceItemList* pItemList = 0;
				gs_pSystemHandler->OnRequestServerItemList(&pItemList);
				gs_pSvrItemList = TFLIB_DYNAMIC_CAST(CTFDavResourceList, pItemList->Clone());
			}
		}
	}
	
	// サーバリストが存在したらSSLの証明書をシステムから取得する
	// TODO: サーバリストが存在していなくてもSSLの証明書は取得する必要が
	//       有るでしょう。ここでこの処理をやるのは望ましくは無い。修正しなさい!!
	if (gs_pSvrItemList && gs_enuSSLStore == TF_STORE_NONE)
	{
		loadSystemCerts();
	}
}

/**
 *	システムから証明書をストアする。
 *
 *	(note)
 *	サーバリストを新規作成したときにのみ呼ばれます。
 *	gs_enuSSLStore == TF_STORE_NONEの時だけ
 *
 */
void
CTFServerManager::loadSystemCerts(void)
{
	OI_DEBUG("TRACE: CTFServerManager::loadSystemCerts()\n");

	RemoveCertStore();

	TF_SSL_CERT_HANDLE_TYPE enuHandleType = TF_SSL_CERT_UNKNOWN;
	gs_pSystemHandler->OnQueryHasSSLCertificate(enuHandleType);
	gs_enuSSLStore = (enuHandleType != TF_SSL_CERT_UNKNOWN) ? TF_STORE_TRUE : TF_STORE_FALSE;

	bool bResult = false;
	if (enuHandleType == TF_SSL_CERT_VOID)
	{
		TFHANDLE pHandle;

		// 中間証明書
		if (gs_pSystemHandler->OnOpenSystemStore(TF_SSL_CERT_STORE_NAME_CA, &pHandle))
		{
			if (gs_pSystemHandler->OnLoadCertsStore(pHandle, &gs_pSSLCert))
			{
				bResult = gs_pSystemHandler->OnCloseCertsStore(pHandle);
			}
		}

		if (bResult)
		{
			bResult = false;
			// ROOT証明書
			if (gs_pSystemHandler->OnOpenSystemStore(TF_SSL_CERT_STORE_NAME_ROOT, &pHandle))
			{
				if (gs_pSystemHandler->OnLoadCertsStore(pHandle, &gs_pSSLCert))
				{
					bResult = gs_pSystemHandler->OnCloseCertsStore(pHandle);
				}
			}
		}
	}
	else 
	{
		// TODO: ファイル要求を書きなさい
	}

	// SSL証明書をOnionへストアする
	if (bResult)
	{
		int nStoreCnt = gs_pSSLCert->SetCertificatesOnion();
		OI_DEBUG("CTFServerManager::loadSystemCerts >> Stored SSL Certificate Count :: %d\n", nStoreCnt);
	}

}

