#include "TFTransaction.h"
#include "TFLibWorkSessionManager.h"
#include "TFLibWorkSessionWorker.h"
#include "TFDavResource.h"
#include "TFServerManager.h"
#include "TFServerResource.h"
#include "TFResourcePropTFUser.h"
#include "TFResourceTFRootTree.h"
#include "TFIfHeader.h"
#include "TFXmlBodyPropPatch.h"
#include "TFXmlBodyMKCOL.h"
#include "TFXmlBodySearch.h"
#include "TFException.h"
#include "TFLocalFileItem.h"
#include "TFLocalFileItemList.h"
#include "TFLocalFileUtils.h"
#include "TFUtils.h"

CTFTransaction::CTFTransaction(void)
:m_bSeqParentchk(true),
 m_bConPending(false)
{
	m_pWsTransWorker = NULL;
}

CTFTransaction::~CTFTransaction(void)
{
}

void 
CTFTransaction::StartTrans(const TFXMLCh* pszServerID)
{
	OI_ASSERT(pszServerID);
	m_bConPending = true;
	
	// トランザクションの為に一度コネクションを作る
	// 作った後は再利用できるように一旦Managerへ返却する
	m_pWsTransWorker = CTFLibWorkSessionManager::GetSession(pszServerID, NULL);

	// ハンドラの登録
	m_pWsTransWorker->SetHandler(m_pSysHandler);
	m_pWsTransWorker->SetHandler(m_pTransHandler);
	m_pWsTransWorker->SetHandler(m_pNotifyHandler);

	if (m_pWsTransWorker) CTFLibWorkSessionManager::CloseSession(m_pWsTransWorker, m_bConPending);
}

void
CTFTransaction::EndTrans(void)
{
	m_bConPending = false;
	CTFLibWorkSessionManager::CloseSession(m_pWsTransWorker);
	m_pWsTransWorker = NULL;
}



/**
 * ROOT TREE SRARCHを実行する
 * @param	pszTargetURI		const char*
 * @return	ppItem				CTFResourceProp**
 */
bool
CTFTransaction::SearchRootTree(CTFResourceItem* pItem, CTFResourceProp** ppItem, TFSEARCH_RT enuType)
{
	OI_ASSERT(pItem);

	OI_RESULT	oResult;
	bool		bResult = false;
	TFHANDLE	hProgress;

	TFTRANSMODE	enuMode = TF_TRANS_ROOTTREESEARCH;

	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	TF_SEARCH_MODE enuSearchMode = TF_SEARCH_MODE_UNKNOWN;
	switch(enuType)
	{
		case TF_SEARCH_RT_ALL:
			enuSearchMode = TF_SEARCH_MODE_ALLLIST;
			break;

		case TF_SEARCH_RT_GROUP:
			enuSearchMode = TF_SEARCH_MODE_GROUPLIST;
			break;

		case TF_SEARCH_RT_MANAGEMENT:
			enuSearchMode = TF_SEARCH_MODE_MANAGEMENT;
			break;

		case TF_SEARCH_RT_EXECSQLSTMT:
			enuSearchMode = TF_SEARCH_MODE_EXECSQLSTMT;
			break;

		case TF_SEARCH_RT_REPOSDBSQLSTMT:
			enuSearchMode = TF_SEARCH_MODE_REPOSDBSQLSTMT;
			break;

		default:
			OI_ASSERT(false);
	}

	CTFDavResource *pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem != NULL)
	{
		CTFXmlBodySearch cXML;
		if (cXML.CreateBodyRootTree(enuSearchMode))
		{
			pWsWorker->SetHandler(m_pTransHandler);
			pWsWorker->SetHandler(m_pNotifyHandler);
			pWsWorker->SetHandler(m_pSysHandler);

			// ユーザ情報をもらう為に必要なURIの追加
			CTFServerResource* pSvrItem = GetServerManager().GetServerItem(pDavItem->GetServerID());
			if (pSvrItem)
			{
				TF_STRING_W strUserInfoURI;
				strUserInfoURI  = pSvrItem->GetURI();
				strUserInfoURI += X(TF_RTURI_USER);

				oResult = pWsWorker->SEARCH(strUserInfoURI.c_str(), ppItem, &cXML);
				bResult = (oResult == OI_OK);
			}
			else {
				// サーバが設定ファイルに存在しなかった
				// 戻りを調べる必要はない。
				m_pNotifyHandler->OnNotifyMessage(TF_RESULT_CONF_HOSTNOTFOUND, TF_DLG_NOTIFY_OK);
			}
		}
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

/**
 *	ステータス情報を返却する
 *	@param	pItem		CTFResourceItem*
 *	@param	ppItem		CTFResourceProp**
 *	@return boolean
 */
bool
CTFTransaction::SearchStatus(CTFResourceItem* pItem, CTFResourceProp** ppItem)
{
	OI_ASSERT(pItem);
	OI_RESULT	oResult;
	bool		bResult = false;

	TFTRANSMODE	enuMode = TF_TRANS_STATUSSEARCH;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem == NULL)
		return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	CTFXmlBodySearch cXML;
	if (cXML.CreateBodyStatus(TF_SEARCH_MODE_DETAILLIST))
	{
		pWsWorker->SetHandler(m_pTransHandler);
		pWsWorker->SetHandler(m_pNotifyHandler);
		pWsWorker->SetHandler(m_pSysHandler);

		oResult = pWsWorker->SEARCH(pDavItem->GetURI(), ppItem, &cXML);
		bResult = (oResult == OI_OK);
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool
CTFTransaction::SearchUser(const TFXMLCh* pszServerID, CTFResourcePropTFUser* pUsrItem, CTFResourceProp** ppItem, TF_SEARCH_UI enuType)
{
	OI_ASSERT(pszServerID);
	OI_RESULT	oResult;
	bool		bResult = false;
	TF_STRING_W	strUID, strURI;

	TFTRANSMODE	enuMode = TF_TRANS_USERSEARCH;

	if ( enuType & ( TF_SEARCH_UI_CONTENT | TF_SEARCH_UI_AVAILABLEGROUP) )
	{
		if (!pUsrItem) return false;
		strUID = pUsrItem->GetUID();
	}

	// UserInformationを投げるURIの作成
	strURI  = GetServerManager().GetServerItem(pszServerID)->GetRootURI();
	strURI += X(TF_URI_MNG_USER);

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, hProgress);
	if (pWsWorker == NULL) 
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	TF_SEARCH_MODE enuSeachMode;
	switch(enuType)
	{
		case TF_SEARCH_UI_TREELIST:
			enuSeachMode = TF_SEARCH_MODE_TREELIST;
			break;

		case TF_SEARCH_UI_DETAILLIST:
			enuSeachMode = TF_SEARCH_MODE_DETAILLIST;
			break;
			
		case TF_SEARCH_UI_CONTENT:
			enuSeachMode = TF_SEARCH_MODE_CONTENT;
			break;
		
		case TF_SEARCH_UI_AVAILABLEGROUP:
			enuSeachMode = TF_SEARCH_MODE_AVAILABLEGROUP;
			break;
		
		default:
			OI_ASSERT(false);
			break;			
	}
	
	CTFXmlBodySearch cXML;
	if (cXML.CreateBodyUser(enuSeachMode, strUID.c_str()))
	{
		pWsWorker->SetHandler(m_pTransHandler);
		pWsWorker->SetHandler(m_pNotifyHandler);
		pWsWorker->SetHandler(m_pSysHandler);

		oResult = pWsWorker->SEARCH(strURI.c_str(), ppItem, &cXML);
		bResult = (oResult == OI_OK);

		if (!bResult) (void)QueryDAVError(enuMode, TF_DAV_SEARCH, oResult, pUsrItem);
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool
CTFTransaction::SearchGroup(const TFXMLCh* pszServerID, CTFResourcePropTFGroup* pGrpItem, CTFResourceProp** ppItem, TF_SEARCH_GI enuType)
{
	OI_ASSERT(pszServerID);
	OI_RESULT	oResult;
	bool		bResult = false;
	TF_STRING_W	strURI;
	TFTRANSMODE	enuMode = TF_TRANS_GROUPSEARCH;

	if (IS_EMPTY(pszServerID)) return false;

	const TFXMLCh* pszURI = GetServerManager().GetServerItem(pszServerID)->GetRootURI();
	if (IS_EMPTY(pszURI)) return false;
	strURI  = pszURI;
	strURI += X(TF_URI_MNG_GROUP);

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, hProgress);
	if (pWsWorker == NULL) 
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);
	
	TF_SEARCH_MODE enuSearchMode = TF_SEARCH_MODE_UNKNOWN;
	switch(enuType)
	{
		case TF_SEARCH_GI_AVAILABLESQL:
			enuSearchMode = TF_SEARCH_MODE_AVAILABLESQL;
			break;

		case TF_SEARCH_GI_AVAILABLEUSER:
			enuSearchMode = TF_SEARCH_MODE_AVAILABLEUSER;
			break;

		case TF_SEARCH_GI_CONTENT:
			enuSearchMode = TF_SEARCH_MODE_CONTENT;
			break;

		case TF_SEARCH_GI_DETAILLIST:
			enuSearchMode = TF_SEARCH_MODE_DETAILLIST;
			break;

		case TF_SEARCH_GI_TREELIST:
			enuSearchMode = TF_SEARCH_MODE_TREELIST;
			break;

		default:
			OI_ASSERT(false);
	}

	CTFXmlBodySearch	cXML;
	if (cXML.CreateBodyGroup(enuSearchMode, pGrpItem))
	{
		oResult = pWsWorker->SEARCH(strURI.c_str(), ppItem, &cXML);
		bResult = (oResult == OI_OK);

		if (!bResult) (void)QueryDAVError(enuMode, TF_DAV_SEARCH, oResult, pGrpItem);
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool CTFTransaction::SearchSysMsg(const TFXMLCh*	pszServerID,
								  CTFResourceProp** ppItem,
								  TF_SEARCH_SMI		enuType,
								  const TFXMLCh*	pszMsgID)
{

	OI_RESULT		oResult;
	bool			bResult		= false;
	TF_STRING_W		strURI;
	TFTRANSMODE		enuMode		= TF_TRANS_SYSMEGSEARCH;

	TFHANDLE		hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, &hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	// URIの決定
	strURI = pWsWorker->GetServerItem()->GetRootURI();

	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	TF_SEARCH_MODE enuSearchMode = TF_SEARCH_MODE_UNKNOWN;
	switch(enuMode)
	{
		case TF_SEARCH_SMI_CONTENT:
			enuSearchMode = TF_SEARCH_MODE_CONTENT;
			break;

		case TF_SEARCH_SMI_CURRENTMSG:
			enuSearchMode = TF_SEARCH_MODE_CURRENTMSG;
			break;

		case TF_SEARCH_SMI_DETAILLIST:
			enuSearchMode = TF_SEARCH_MODE_DETAILLIST;
			break;
		default:
			OI_ASSERT(false);
	}

	CTFXmlBodySearch	cXML;
	if (cXML.CreateBodySysMsg(enuSearchMode, pszMsgID))
	{
		oResult = pWsWorker->SEARCH(strURI.c_str(), ppItem, &cXML);
		bResult = (oResult == OI_OK);
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;

}

bool
CTFTransaction::SearchUpdate(CTFResourceItem* pItem, CTFResourceProp** ppItem)
{
	OI_ASSERT(pItem);
	bool		bResult = false;

	TFTRANSMODE	enuMode = TF_TRANS_UPDSEARCH;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem == NULL)
		return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);
/*
	CTFXmlBodySearch	cXML;
	if (cXML.CreateBodyUpdate(NULL))
	{
		oResult = pWsWorker->SEARCH(pDavItem->GetURI(), ppItem, &cXML);
		bResult = (oResult == OI_OK);
	}
*/
	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool 
CTFTransaction::SearchDBMS(CTFResourceItem* pItem, CTFResourceProp** ppItem, TF_SEARCH_DI enuType, const char* pszDBMSID)
{
	OI_ASSERT(pItem);
	bool		bResult = false;

	TFTRANSMODE	enuMode = TF_TRANS_DBMSSEARCH;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem == NULL)
		return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

/*
	CTFLibSearchXML	cXML;
	if (cXML.PrepareDBMSInformationSearch(enuType, NULL)) 
	{
		oResult = pWsWorker->SEARCH(pDavItem->GetURI(), ppItem, &cXML);
		bResult = (oResult == OI_OK);
	}
*/
	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;

}

/**
 *	PROPFIND（depth 1)を実行する
 *	@param	pszTargetURI		const char*
 *	@param	ppItem				CTFResourceProp**
 */
bool
CTFTransaction::List(CTFResourceItem* pItem, CTFResourceProp** ppItem, bool bRecursive)
{
	OI_ASSERT(pItem);
	bool bResult = false;

	TFTRANSMODE	enuMode = TF_TRANS_LIST;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem == NULL)
		return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	OI_RESULT oResult;
	if (bRecursive)
	{
		bResult = GetResourceListRecursive(pDavItem->GetURI(), ppItem, pWsWorker);
	}
	else
	{
		oResult = pWsWorker->PROPFIND(pDavItem->GetURI(), ppItem, D_PFIND_ONE, LP_DETAIL);

		if (oResult == OI_OK)
		{
			bResult = true;
		}
		else 
		{
			// PROPFINDのエラーは表示に留める
//			QueryDAVError_Default(oResult);
			QueryDAVError(TF_TRANS_LIST, TF_DAV_PROPFIND, oResult, pDavItem);
			bResult = false;
		}
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool
CTFTransaction::MakeFolder(CTFResourceItem* pParentItem)
{
	OI_ASSERT(pParentItem);
	bool bResult = false;

	TFTRANSMODE	enuMode = TF_TRANS_MAKEFOLDER;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pParentItem);
	if (pDavItem == NULL)	return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pParentItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	OI_RESULT oResult;
	LOCKLIST vLock;
	if (!ExistResource(pDavItem->GetURI(), vLock, oResult, pWsWorker))
	{
		// 親フォルダでのエラーが発生した場合は継続処理は不可能
		// コネクションの開放
		m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
		CTFLibWorkSessionManager::CloseSession(pWsWorker);
		QueryDAVError_Default(oResult);
		return false;
	}

	// ここにきたなら親アイテムは存在したことになる
	TF_STRING_W strParentURI = pDavItem->GetURI();

	OI_STRING_A	strName, strOldName;
	TF_STRING_W strURI;

	// 新しい名前を作成する
	// 内部で新しいフォルダ名とかを問合せサーバに存在しないことを確認しながら
	// 処理を継続していきます。
	// ファイルが存在しないことが分かったときにこのループを抜けることができます。
	// 但し、PROPFINDで存在確認中のエラーは無条件に終了となります。
	int nRetry = 0;
	do {
		strOldName = strName;
		if ( (m_pTransHandler->OnQueryNewResourceName(TF_RES_COLLECTION, enuMode, strName, nRetry)) == false )
		{
			// キャンセルした場合はエラーとして終了する
			bResult = false;
			break;
		}
		else if ( strName.empty() || (strOldName.compare(strName) == 0) )
		{
			// 空白の場合や全開と同じ名前の場合再度問合せ
			nRetry++;
			continue;
		}

		// 新規作成するURIを作成する
		strURI  = pDavItem->GetURI();
		strURI += X(strName.c_str());

		LOCKLIST vTmpLock;
		if (ExistResource(strURI.c_str(), vTmpLock, oResult, pWsWorker))
		{
			// OI_OK: ファイルが存在した
			continue;
		}
		else
		{
			 if (oResult == OIDENOTFOUND)
			 {
				// OIDENOTFOUD: ファイルが存在しなかった
				bResult = true;
				break;
			 }
			 else
			 {
				// 予想できなかったエラー
				QueryDAVError(enuMode, TF_DAV_PROPFIND, oResult, pDavItem);
				bResult = false;
				break;
			 }
		}

	} while(true);

	// ロックトークンの作成
	OI_STRING_A strIfHeader;
	OI_STRING_A	strLockToken;
	bool bMyLocked = false;
	if (bResult)
	{
		if (vLock.empty())
		{
			oResult = GetLock(strURI.c_str(), pWsWorker, strLockToken);
			if (oResult == OI_OK)
			{
				CTFIfHeader::GetLockTokenIfHeader(strLockToken.c_str(), strURI.c_str(), strIfHeader);
				bMyLocked = true;
			}
			else {
				// TODO: ロックの失敗は今はエラーとしてだけ処理しています。
				// このままでよいはずがありません。
				bResult = false;
				QueryDAVError_Default(oResult);
			}
		}
		else 
		{
			CTFIfHeader::GetLockTokenIfHeader(vLock, strParentURI.c_str(), strIfHeader);
		}
	}

	// MKCOLメソッドの発行
	if (bResult)
	{
		oResult = pWsWorker->MKCOL(strURI.c_str(), strIfHeader.empty() ? NULL : strIfHeader.c_str(), NULL);
		if (QueryDAVError_Default(oResult) != TF_DLG_NOTIFY_NONE) 
		{
			bResult = false;
		}
	}

	// 自分がロックを行なった場合はロックの開放をします
	if (bMyLocked)
	{
		oResult = UnLock(strURI.c_str(), strLockToken.c_str() ,pWsWorker);
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool
CTFTransaction::MakeCol(const TFXMLCh* pszServerID, const TFXMLCh* pszParentURI, const TFXMLCh* pszName)
{
	OI_ASSERT(pszServerID);
	OI_ASSERT(pszParentURI);

	bool		bResult = false;
	OI_RESULT	oResult;

	TFTRANSMODE	enuMode = TF_TRANS_MAKEFOLDER;

	CTFLibWorkSessionWorker* pWsWorker;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	try {

		// コネクションの取得
		pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, hProgress);
		if (pWsWorker == NULL)
		{
			throw CTFException(TF_RESULT_UNEXPECTED);
		}

		// ハンドラの設定
		pWsWorker->SetHandler(m_pTransHandler);
		pWsWorker->SetHandler(m_pNotifyHandler);
		pWsWorker->SetHandler(m_pSysHandler);

		// 親フォルダのチェック
		LOCKLIST	vpLock;
		if (m_bSeqParentchk)
		{
			if (ExistResource(pszParentURI, vpLock, oResult, pWsWorker) == false)
			{
				QueryDAVError_Default(oResult);
				throw CTFException(m_enuLastResult);
			}
		}

		TF_STRING_W strNewCol;
		// 作成するコレクション名が存在しない場合新規作成フォルダとみなす
		if (IS_EMPTY(pszName))
		{
			// TODO: 新規作成フォルダの作成をしなさい
		}
		else
		{
			strNewCol = pszName;
		}

		// 新規フォルダのURIを作成
		TF_STRING_W strURI;
		strURI  = pszParentURI;
		strURI += strNewCol.c_str();
		strURI += X("/");

		LOCKLIST vLock;
		if (ExistResource(strURI.c_str(), vLock, oResult, pWsWorker))
		{
			if (IS_EMPTY(pszName))
			{
				// TODO: 新規作成の場合は名前が存在した為再度問い合わせるように
			}
			else 
			{
				// フォルダが既に存在していた場合はエラー
				m_enuLastResult = TF_RESULT_FOUNDFOLDER;
				throw CTFException(m_enuLastResult);
			}
		}

		oResult = pWsWorker->MKCOL(strURI.c_str(), NULL, NULL);
		bResult = (oResult == OI_OK);

		if (!bResult)
		{
			QueryDAVError(enuMode, TF_DAV_MKCOL, oResult, NULL);
			throw CTFException(m_enuLastResult);;
		}

	} catch(...) {
		bResult = false;
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;

}

bool 
CTFTransaction::MakeColEx(CTFResourceItem* ppItem, TFXMLBODYTYPE enuType, CTFResourceItem* pItem)
{
	OI_ASSERT(ppItem);
	bool bResult = false;
	OI_RESULT	oResult;
	TFTRANSMODE	enuMode = TF_TRANS_UNKNOWN;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, ppItem);
	if (pDavItem == NULL)	return false;

	// モードの設定
	switch(enuType)
	{
		case TF_XML_BODY_MSG:
			enuMode = TF_TRANS_MAKEMSG;
			break;

		case TF_XML_BODY_USR:
			enuMode = TF_TRANS_MAKEUSER;
			break;

		case TF_XML_BODY_GRP:
			enuMode = TF_TRANS_MAKEGROUP;
			break;

		case TF_XML_BODY_LINKDB:
			enuMode = TF_TRANS_MAKELINKDB;
			break;

		case TF_XML_BODY_SQL:
			enuMode = TF_TRANS_MAKESQL;
			break;

		default:
			enuMode	= TF_TRANS_UNKNOWN;
			break;
	}

	// プログレスの初期化
	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(ppItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	// 親フォルダのチェック
	LOCKLIST	pLock;
	if (m_bSeqParentchk)
	{
		if (ExistResource(pDavItem->GetURI(), pLock, oResult, pWsWorker) == false)
		{
			m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
			CTFLibWorkSessionManager::CloseSession(pWsWorker);
			QueryDAVError_Default(oResult);
			return false;
		}
	}

	if (TF_TRANS_MAKEUSER  == enuMode)
	{
		TF_STRING_W strURI;
		CTFResourcePropTFUser* pUsrItem;

		strURI  = pDavItem->GetURI();
		if (pItem != NULL)
		{
			pUsrItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFUser, pItem);
			if (pUsrItem != NULL)
			{
				CTFXmlBodyMKCOL	cXml;
				if (cXml.CreateBodyUser(*pUsrItem))
					oResult = pWsWorker->MKCOL(pUsrItem->GetURI(), NULL, &cXml);

				bResult = (oResult == OI_OK);

			}
		}
		else {
			TF_STRING_W strIdent;
			bResult = m_pTransHandler->OnQueryIdentifier(enuMode, strIdent, TF_LENGTH_UID, 0);
		}

		// MKCOL結果のチェック
		if (!bResult) QueryDAVError(enuMode, TF_DAV_MKCOL, oResult, pItem);

	}

	if (TF_TRANS_MAKEGROUP == enuMode)
	{
		TF_STRING_W strURI;
		CTFResourcePropTFGroup* pGrpItem;

		strURI = pDavItem->GetURI();
		if (pItem != NULL)
		{
			pGrpItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFGroup, pItem);
			if (pGrpItem)
			{
				CTFXmlBodyMKCOL cXML;
				if (cXML.CreateBodyGroup(*pGrpItem))
					oResult = pWsWorker->MKCOL(pGrpItem->GetURI(), NULL, &cXML);

				bResult = (oResult == OI_OK);
			}
		}
		else
		{
			// TODO: アイテムが存在しなったばあいの処理を記述しなさい
		}

		if (!bResult) (void)QueryDAVError(enuMode, TF_DAV_MKCOL, oResult, pItem);
	}

	if (TF_TRANS_MAKEMSG == enuMode)
	{
		TF_STRING_W	strIdent = (const TFXMLCh*)X("");
		TF_STRING_W	strURI;
		int nRetry = 0;
		LOCKLIST	theLock;
		do {
			bResult = m_pTransHandler->OnQueryIdentifier(enuMode, strIdent, TF_LENGTH_MESSAGE, nRetry);
			if (bResult == false)
			{
				// TODO: メッセージID作成をキャンセルした場合の処理をいれなさい。
				break;
			}

			// 識別子の長さが空か長すぎる場合は再度リクエスト
			if (strIdent.empty() || strIdent.length() > TF_LENGTH_MESSAGE)
			{
				nRetry++;
				continue;
			}

			strURI.erase();
			strURI  = pDavItem->GetURI();
			strURI += strIdent;
			strURI += X("/");
			if (ExistResource(strURI.c_str(), theLock, oResult, pWsWorker))
			{
				nRetry++;
				continue;
			}
			else if (oResult == OIDENOTFOUND)
			{
				bResult = true;
				break;
			}
			else
			{
				QueryDAVError_Default(oResult);
				bResult = false;
				break;
			}
		} while(true);

		if (bResult)
		{
			LOCKLIST	theLock;
			OI_STRING_A	strLockToken	= "";
			OI_STRING_A	strIfHeader		= "";

			// ロックをかける
			oResult = GetLock(strURI.c_str(), pWsWorker, strLockToken);
			if (oResult == OI_OK)
			{
				if (!strLockToken.empty())
				{
					CTFIfHeader cIf;
					cIf.GetLockTokenIfHeader(strLockToken.c_str(), strURI.c_str(), strIfHeader);
				}

				TF_STRING_W	strMessage;
				bool		bActive;
				m_pTransHandler->OnQuerySystemMessage(strIdent, strMessage, bActive);
				CTFResourcePropTFMsg	cMsgItem;
				cMsgItem.SetMsgID(strIdent.c_str());
				cMsgItem.SetMsg(strMessage.c_str());
				cMsgItem.SetActiveFlag(bActive);

				CTFXmlBodyMKCOL cXML;
				if (cXML.CreateBodyMessage(cMsgItem))
				{
					oResult = pWsWorker->MKCOL(strURI.c_str(), (strIfHeader.empty()) ? NULL : strIfHeader.c_str(), &cXML);
					QueryDAVError(enuMode, TF_DAV_MKCOL, oResult, pDavItem);
					bResult = (OI_OK == oResult);
				}
			}
		}

	}	// if (TF_TRANS_MAKEMSG == enuMode)

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool 
CTFTransaction::Download(CTFResourceItem* pItem, bool bRecursive)
{
	OI_ASSERT(pItem);

	bool bResult = false;

	TFTRANSMODE	enuMode = TF_TRANS_GETRESOURCE;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem == NULL)	return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
	if (pWsWorker == NULL) 
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	// ローカルユーティリティの作成
	CTFLocalFileUtils* pLocalUtils = m_pPFFactory->CreateLocalFileUtils();
	if (pLocalUtils == NULL) return false;
	// ハンドラの設定
	pLocalUtils->SetHandler(m_pTransHandler);

	// ダウンロードパス(ここはダウンロードするパスのTOPフォルダを取得）を取得する
	CTFLocalFileItem* pDownLoadItem;
	pDownLoadItem = m_pPFFactory->CreateLocalFileItem();
	m_pTransHandler->OnSetSystemPath(pDownLoadItem, CTFWsTransactionHandler::TF_SYSPATH_DOWNLOAD);

	CTFLocalFileItem* pLocalItem = m_pPFFactory->CreateLocalFileItem();

	// pItemを元にローカルのアイテムを生成させる
	m_pTransHandler->OnQueryDownloadItem(pLocalItem, pItem);

	// ダウンロードのパスを作成する
	TF_STRING_W strFullPath = pDownLoadItem->GetPathW();
	strFullPath += pLocalItem->GetPathW();

	pLocalItem->SetPath(0, (void*)strFullPath.c_str(), sizeof(TFXMLCh));

	CTFResourceProp* pProp = TFLIB_DYNAMIC_CAST(CTFResourceProp, pItem);
	if (pProp == NULL) {
		CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
		if (pDavItem == NULL) 
		{
			if (pLocalItem) delete pLocalItem;
			if (pLocalUtils) delete pLocalUtils;

			m_enuLastResult = TF_RESULT_UNEXPECTED;
			return false;
		}

		if (!GetOneResource(pDavItem->GetURI(), &pProp, pWsWorker))
		{
			if (pLocalItem) delete pLocalItem;
			if (pLocalUtils) delete pLocalUtils;

			m_enuLastResult = TF_RESULT_NOTFOUND;
			return false;		// ファイルがないかエラー
		}
	}

	// 処理するURIを通知する。
	if (hProgress) m_pNotifyHandler->OnPreSendProgressWindow(pDavItem->GetURI(), hProgress);

	TFFSTAT enuStat = e_NOERROR;
	if (pProp->IsCollection())
	{
		// フォルダ作成
		enuStat = pLocalUtils->Mkdir(*pLocalItem, true);
		if (enuStat == e_NOERROR)
			m_enuLastResult = TF_RESULT_OK;
		else
			m_enuLastResult = TF_RESULT_LSYS_UNEXPECTED;
	}
	else 
	{
		//　ダウンロード
		enuStat = pLocalUtils->Open(*pLocalItem, CTFLocalFileUtils::TF_F_RDWR | CTFLocalFileUtils::TF_F_NEW);
		if (enuStat != e_NOERROR)
		{
			// OPEN error
			if (enuStat == e_EINVAL)
			{
				// e_EINVALはファイル上位フォルダが無いケースもある。
				CTFLocalFileItem* pLocalParentItem = m_pPFFactory->CreateLocalFileItem();
				TF_STRING_W strParentPath = pLocalItem->GetPathW();
				CTFURL cURL(strParentPath.c_str());
				pLocalParentItem->SetPath(0, (void*)cURL.m_strDir.c_str(), sizeof(TFXMLCh)); 
				TFFSTAT r_enuStat = pLocalUtils->Mkdir(*pLocalParentItem, true);
				if (r_enuStat == e_NOERROR)
				{
					enuStat = pLocalUtils->Open(*pLocalItem, CTFLocalFileUtils::TF_F_RDWR | CTFLocalFileUtils::TF_F_NEW);
				}
				// 開放
				pLocalUtils->Close(*pLocalParentItem);
			}
		}

		// 二度のチェックを行うかもしれないのでここで再度enuStatの評価を行う
		if (enuStat == e_NOERROR)
		{
			OI_RESULT oResult = pWsWorker->GET(pDavItem->GetURI(), pLocalItem->GetFD(), NULL);

			// メソッド処理の結果調査
			(void)QueryDAVError(enuMode, TF_DAV_GET, oResult, pItem);
			if (oResult != OI_OK) enuStat = e_EINVAL;
		}
		else
		{
			// 完全なエラー（ローカルシステムに障害）
			m_enuLastResult = TF_RESULT_LSYS_UNEXPECTED;
		}
	}

	// ローカルファイルの結果
	m_pTransHandler->OnResultLocalResource(enuMode, TF_DAV_GET, pLocalItem, m_enuLastResult);

	if (enuStat == e_NOERROR)
	{
		// 開放
		pLocalUtils->Close(*pLocalItem);

		// 成功した場合は日付を再設定する
		pLocalUtils->SetTime(pLocalItem, pProp->GetCreationDate(), pProp->GetLastModified());

		// フォルダの場合は中身をPropFindするただしリカーシブ取得場合のみ
		if (pProp->IsCollection() && bRecursive)
		{
			CTFResourceProp* pSubProp = NULL;
			OI_RESULT oSubResult = pWsWorker->PROPFIND(pDavItem->GetURI(), &pSubProp, D_PFIND_ONE, LP_SIMPLECHECK, LP_EX_NONE, DP_NONE);
			if (oSubResult == OI_OK)
			{
				CTFDavResourceList* pList = TFLIB_DYNAMIC_CAST(CTFDavResourceList, pSubProp);
				if (pList)
				{
					CTFDavResource* pSubItem = NULL;
					for(;pSubItem = (CTFDavResource*)pList->GetNextItem();)
					{
						TF_STRING_W strURI = pSubItem->GetURI();
						RemoveSlash(strURI);
						if (strURI.compare(pDavItem->GetURI()) != 0)
						{
							// URIを入れ替える
							pSubItem->SetURI(strURI.c_str());
							CTFLibWorkSessionManager::CloseSession(pWsWorker, true);
							Download(pSubItem);
						}
					}
				}
			}
			if (pSubProp) delete pSubProp;
		}
	}

	bResult = (enuStat == e_NOERROR);

	// ローカルアイテムの消去
	if (pLocalItem) delete pLocalItem;

	// ローカルユーティリティの削除
	if (pLocalUtils) delete pLocalUtils;

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool 
CTFTransaction::Download(const TFXMLCh* pszServerID,
						 CTFResourceItemList* lpResource, bool bRecursive)
{
	// 引数のチェック
	if (IS_EMPTY(pszServerID)) return false;
	if (lpResource == NULL) return false;
	if (lpResource->IsEmpty()) return false;

	bool						bResult		= false;

	CTFResourceItem* pResource = NULL;
	CTFDavResource* pItem = NULL;
	while(true)
	{
		pResource = lpResource->GetNextItem();
		if (!pResource) break;
		bResult = Download(pResource, bRecursive);
		// エラー処理はDownLoad()に任せる
	}

	return bResult;
}

/**
 *	名称変更(RENAME)を行なう
 *	@param pItem		CTFResourceItem*	変更したいアイテム
 *	@return boolean
 *
 */
bool
CTFTransaction::ReName(CTFResourceItem* pItem)
{
	OI_ASSERT(pItem);
	bool	bResult = false;

	TFTRANSMODE	enuMode = TF_TRANS_RENAME;

	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
	if (pDavItem == NULL)	return false;

	// 変更したい名前の問合せ
	OI_STRING_A	strNewName;
	CTFURL	cURL(pDavItem->GetParentURL());
	strNewName = (const char*)X(cURL.m_strName.c_str());

	bResult = m_pTransHandler->OnQueryNewResourceName(TF_RES_UNKNOWN, enuMode, strNewName, 0);
	if (bResult == false || strNewName.empty())
		return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	// 名称を変えたい元リソースの存在確認(source チェック）
	OI_RESULT oResult;
	LOCKLIST vLock;
	if (!ExistResource(pDavItem->GetURI(), vLock, oResult, pWsWorker))
	{
		// コネクションの開放
		m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
		CTFLibWorkSessionManager::CloseSession(pWsWorker);
		QueryDAVError_Default(oResult);
		return false;
	}

	TF_STRING_W strURI;	
	strURI  = cURL.m_strDir.c_str();
	strURI += X(strNewName.c_str());

	// 変更したい名前の存在確認(Distination チェック）
	LOCKLIST vtheLock;
	if ((ExistResource(strURI.c_str(), vtheLock, oResult, pWsWorker)) == false)
	{
		if (oResult == OIDENOTFOUND)
		{
			bResult = true;
		}
		else
		{
			QueryDAVError_Default(oResult);
			bResult = false;
		}
	}
	else 
	{
		QueryDAVError(enuMode, TF_DAV_MOVE, oResult, pDavItem);
		bResult = false;
	}

	if (bResult)
	{
		CTFIfHeader cIF;
		OI_STRING_A	strLockTokenIfHeader;

		if (vLock.empty())
		{
			OI_STRING_A	strLockToken;
			oResult = GetLock(strURI.c_str(), pWsWorker, strLockToken);
			if (oResult == OI_OK)
				cIF.GetLockTokenIfHeader(strLockToken.c_str(), strURI.c_str(), strLockTokenIfHeader);
		}
		else
		{
			cIF.GetLockTokenIfHeader(vLock, pDavItem->GetURI(), strLockTokenIfHeader);
		}

		// 移動先(名前変更後）を作る
		TF_STRING_W strURL;
		strURL  = cURL.m_strDir.c_str();
		strURL += X(strNewName.c_str());
		oResult = pWsWorker->MOVE(pDavItem->GetURI(), strURL.c_str(), NULL, false, (strLockTokenIfHeader.empty()) ? NULL : strLockTokenIfHeader.c_str());
		bResult = (oResult == OI_OK);
		QueryDAVError_Default(oResult);
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool
CTFTransaction::ReName(	const TFXMLCh* pszServerID,
						const TFXMLCh* pszOldUri,
						const TFXMLCh* pszNewUri)
{
	OI_ASSERT(pszServerID);
	OI_ASSERT(pszOldUri);

	OI_RESULT					oResult;
	TFTRANSMODE					enuMode		= TF_TRANS_RENAME;
	CTFLibWorkSessionWorker*	pWsWorker	= NULL;
	bool						bResult		= false;
	bool						bDstLock	= false;
	bool						bSrcLock	= false;
	TF_STRING_W					strDstUri;
	OI_STRING_A					strLockToken;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	try {
		/*
		 *	変更後の名前の確認
		 *	もし新しい名前が設定されていない場合はハンドラで問い合わせる
		 */
		OI_STRING_A strNewName;
		if (IS_EMPTY(pszNewUri) || TF_STRING_W(pszNewUri).empty())
		{
			(void)m_pTransHandler->OnQueryNewResourceName(TF_RES_COLLECTION, enuMode, strNewName, 0);
		}
		else
		{
			strNewName = X(pszNewUri);
		}
		
		if (strNewName.empty())
			throw CTFException(TF_RESULT_USERCANCELD);
		
		// 名前変更を行うURIの存在確認
		LOCKLIST	vLock;
		if (!ExistResource(pszOldUri, vLock, oResult, pWsWorker))
		{
			throw CTFException(TF_RESULT_NOTFOUND);
		}
		else
		{
			CTFIfHeader cIf;
			OI_STRING_A	strLockTokenIfHeader;

			if (HAS_LOCK(vLock))
			{
				cIf.GetLockTokenIfHeader(vLock, pszOldUri, strLockTokenIfHeader);
			}
			else
			{
				TF_STRING_W	strDstURI;
				if (GetLock(pszOldUri, pWsWorker, strLockToken) == OI_OK)
				{
					cIf.GetLockTokenIfHeader(strLockToken.c_str(), pszOldUri, strLockTokenIfHeader);
					bSrcLock = true;
				}
			}

			strDstUri  = GetServerManager().GetServerItem(pWsWorker->GetServerID())->GetRootURI();
			strDstUri += pszNewUri;

			TF_STRING_W	strDstURL;
			strDstURL  = GetServerManager().GetServerItem(pWsWorker->GetServerID())->GetROOTURL();
			strDstURL += pszNewUri;

			if (ExistResource(strDstUri.c_str(), vLock, oResult, pWsWorker))
				throw CTFException(TF_RESULT_RESORCE_EXIST);

			if (GetLock(strDstUri.c_str(), pWsWorker, strLockToken) == OI_OK)
			{
                cIf.GetLockTokenIfHeader(strLockToken.c_str(), strDstUri.c_str(), strLockTokenIfHeader);
				bDstLock = true;
			}

			oResult = pWsWorker->MOVE(pszOldUri, strDstURL.c_str(), NULL, false, strLockTokenIfHeader.c_str());
			bResult = (oResult == OI_OK);
		}
		
	} catch (CTFException& e) {
		m_enuLastResult = e.GetCode();
		if (bSrcLock) UnLock(pszOldUri, strLockToken.c_str(), pWsWorker);
		bResult = false;
	} catch (...) {
		OI_ASSERT(false);
	}

	if (bDstLock)
		UnLock(strDstUri.c_str(), strLockToken.c_str(), pWsWorker);

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool
CTFTransaction::PatchProperty(const TFXMLCh* pszServerID, const TFXMLCh* pszParentURI, TFXMLBODYTYPE enuType, CTFResourceItem* pItem)
{
	OI_ASSERT(pszServerID);
	bool		bResult;
	OI_RESULT	oResult;

	TFTRANSMODE enuMode = TF_TRANS_UNKNOWN;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	// 親フォルダのチェック
	LOCKLIST	vpLock;
	if (m_bSeqParentchk)
	{
		if (ExistResource(pszParentURI, vpLock, oResult, pWsWorker) == false)
		{
			m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
			CTFLibWorkSessionManager::CloseSession(pWsWorker);
			QueryDAVError_Default(oResult);
			return false;
		}
	}

	// モードの設定
	switch(enuType)
	{
		case TF_XML_BODY_MSG:
			enuMode = TF_TRANS_PATCHMSG;
			break;

		case TF_XML_BODY_USR:
			enuMode = TF_TRANS_PATCHUSER;
			break;

		case TF_XML_BODY_GRP:
			enuMode = TF_TRANS_PATCHGROUP;
			break;

		case TF_XML_BODY_LINKDB:
			enuMode = TF_TRANS_PATCHLINKDB;
			break;

		case TF_XML_BODY_SQL:
			enuMode = TF_TRANS_PATCHSQL;
			break;

		case TF_XML_BODY_CHGLEADER_APPOINT:
		case TF_XML_BODY_CHGLEADER_DISMISS:
			enuMode = TF_TRANS_CHANGELEADER;
			break;

		default:
			enuMode	= TF_TRANS_UNKNOWN;
			break;
	}

	// ユーザ
	if (TF_TRANS_PATCHUSER == enuMode)
	{
		CTFResourcePropTFUser* pUsrItem;

		if (pItem != NULL)
		{
			pUsrItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFUser, pItem);
			if (pUsrItem != NULL)
			{
				CTFXmlBodyPropPatch cXml;
				if (cXml.CreateBodyUser(*pUsrItem))
					oResult = pWsWorker->PROPPATCH(pUsrItem->GetURI(), &cXml, NULL);

				bResult = (oResult == OI_OK);
			}
		}
	}

	// グループ
	if (TF_TRANS_PATCHGROUP == enuMode)
	{
		CTFResourcePropTFGroup* pGrpItem;
		pGrpItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFGroup, pItem);
		if (pGrpItem)
		{
			CTFXmlBodyPropPatch cXml;
			if (cXml.CreateBodyGroup(*pGrpItem))
				oResult = pWsWorker->PROPPATCH(pGrpItem->GetURI(), &cXml, NULL);

			bResult = (oResult == OI_OK);
			QueryDAVError(enuMode, TF_DAV_PROPPATCH, oResult, pGrpItem);
		}
	}

	// リーダー変更
	if (TF_TRANS_CHANGELEADER == enuMode)
	{
		CTFResourcePropTFGroup* pGrpItem;
		pGrpItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFGroup, pItem);

		CTFXmlBodyPropPatch cXml;
		TFCHANGELEADERTYPE nType;
		switch(enuType)
		{
			case TF_XML_BODY_CHGLEADER_APPOINT:
				nType = TF_CHG_LEADER_APPOINTED;
				break;
			case TF_XML_BODY_CHGLEADER_DISMISS:
				nType = TF_CHG_LEADER_DISMISS;
				break;
		}
		if (cXml.CreateBodyChangeLeader(*pGrpItem,  nType))
			oResult = pWsWorker->PROPPATCH(pGrpItem->GetURI(), &cXml, NULL);

		bResult = (oResult == OI_OK);
		QueryDAVError(enuMode, TF_DAV_PROPPATCH, oResult, pGrpItem);
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

/**
 *	リソースを削除する
 *	@param ppItem		CTFResourceItem*		親リソース
 *	@param	pItem		CTFResourceItem*		削除対象リソース
 *	@param	bforce		bool					強制削除（true: 強制 / false: 強制削除しない
 */
bool
CTFTransaction::Delete(CTFResourceItem* ppItem, CTFResourceItem* pItem, bool bforce)
{
	OI_ASSERT(ppItem);
	OI_RESULT	oResult;
	bool		bResult = true;
	TFTRANSMODE	enuMode = TF_TRANS_DELETE;

	CTFDavResource* ppDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, ppItem);
	if (ppDavItem == NULL)	return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(ppItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	// 親フォルダのチェック
	LOCKLIST	vLocks;
	if (m_bSeqParentchk)
	{
		if (ExistResource(ppDavItem->GetURI(), vLocks, oResult, pWsWorker) == false)
		{
			m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
			CTFLibWorkSessionManager::CloseSession(pWsWorker);
			QueryDAVError_Default(oResult);
			return false;
		}

		// TODO: ロックトークンを保持してないよ。
	}

	CTFDavResource* pDavItem;
	CTFDavResourceList cDavList;
	if (pItem == NULL)
	{
		CTFResourceItemList cItemList;
		m_pTransHandler->OnQuerySelectedItems(TF_DAV_DELETE, enuMode, cItemList);
		if (cItemList.IsEmpty())
		{
			m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
			CTFLibWorkSessionManager::CloseSession(pWsWorker);
			return true;
		}

		CTFDavResource* pTmpItem;
		for (pTmpItem = NULL; pTmpItem = TFLIB_DYNAMIC_CAST(CTFDavResource, cItemList.GetNextItem()); )
		{
			cDavList.AddItem((CTFDavResource*)pTmpItem->Clone());
		}
	}
	else {
		pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
		if (pDavItem)
		{
			cDavList.AddItem((CTFDavResource*)pDavItem->Clone());
		}
	}

	// 削除したいアイテムをループでまわす
	OI_STRING_A	strLockToken;
	OI_STRING_A	strLockTokenIfHeader;
	for (; pDavItem = (CTFDavResource*)cDavList.GetNextItem(); )
	{
		strLockToken.erase();
		strLockTokenIfHeader.erase();

		// deleteの前にPROPFINDでリソースの確認を行なうかをハンドラで問い合わせる
		if (m_pTransHandler->OnQueryPropfindBeforeMethod(enuMode, TF_DAV_DELETE))
		{
			if (!ExistResource(pDavItem->GetURI(), vLocks, oResult, pWsWorker))
			{
				// 削除したいのにアイテムが存在しなかったが結果的には正常エラーである
				// TODO: エラーアイテムをどうにかしなさい
				bResult = false;
				continue;
			}
		}
		else
		{
			// 事前に一覧を取得しないのであれば
			// ロックトークンを既にアイテムとしてもっているか調べる
			CTFResourcePropTF* pResTF = TFLIB_DYNAMIC_CAST(CTFResourcePropTF, pDavItem);
			if (pResTF)
			{
				vLocks = pResTF->GetLocklist();
			}
		}

		if (HAS_LOCK(vLocks))
		{
			CTFIfHeader cIf;
			cIf.GetLockTokenIfHeader(vLocks, pDavItem->GetURI(), strLockTokenIfHeader);
		}

		// 削除処理
		oResult = pWsWorker->DELETE(pDavItem->GetURI(), (strLockTokenIfHeader.empty()) ? NULL : strLockTokenIfHeader.c_str());
		if (oResult == OIDENOCONTENT || oResult == OI_OK)
			bResult = true;
		else
		{
			QueryDAVError(enuMode, TF_DAV_DELETE, oResult, pDavItem);
			bResult = false;

			// faild depend
			if (bforce && m_enuLastResult == TF_RESULT_FAILDDEPEND)
			{
				TFDLGNOTIFYTYPE nType = TF_DLG_NOTIFY_NONE;
				switch(pDavItem->GetType())
				{
					// ユーザとグループは強制削除が行えるので問い合わせる。
					case TFITEM_MNG_USR:
						nType = m_pNotifyHandler->OnNotifyMessage(TF_RESULT_USER_RESOURCE_EXIST,TF_DLG_NOTIFY_OK|TF_DLG_NOTIFY_NO);
						break;

					case TFITEM_MNG_GRP:
						nType = m_pNotifyHandler->OnNotifyMessage(TF_RESULT_USER_RESOURCE_EXIST,TF_DLG_NOTIFY_OK|TF_DLG_NOTIFY_NO);
						break;

					default:
						break;
				}

				if (nType & TF_DLG_NOTIFY_TRUE)
				{
					// 強制削除属性を追加する
					m_pTransHandler->SetFeature(CTFWsTransactionHandler::TF_TRANS_FEATURE_METHOD_DELETE_FORCE, true);

					// 強制削除ヘッダを追加して再チャレンジ
					oResult = pWsWorker->DELETE(pDavItem->GetURI(), (strLockTokenIfHeader.empty()) ? NULL : strLockTokenIfHeader.c_str());
					if (oResult == OIDENOCONTENT || oResult == OI_OK)
						bResult = true;
				}


			}
		}
			
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;
}

bool
CTFTransaction::Delete(const TFXMLCh* pszServerID, CTFResourceItem* ppItem, CTFResourceItem* pItem)
{
	OI_ASSERT(pszServerID);
	if (ppItem == NULL && pItem == NULL) OI_ASSERT(false);

	bool		bResult	= false;
	OI_RESULT	oResult;
	TFTRANSMODE	enuMode = TF_TRANS_DELETE;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	// 親フォルダのチェック
	LOCKLIST	vLocks;
	if (m_bSeqParentchk)
	{
		if (ppItem != NULL)
		{
			CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, ppItem);
			if (pDavItem)
			{
				if (ExistResource(pDavItem->GetURI(), vLocks, oResult, pWsWorker) == false)
				{
					m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
					CTFLibWorkSessionManager::CloseSession(pWsWorker);
					QueryDAVError_Default(oResult);
					return false;
				}
			}
		}
		// TODO: ロックトークンを保持してないよ。
	}

	// 削除するアイテムの一覧を作成する
	CTFResourceItem* pDelItem;
	CTFResourceItemList cItemList;
	// 親リソースが入ってる場合は中身を問い合わせる
	if (ppItem)
	{
		m_pTransHandler->OnQuerySelectedItems(TF_DAV_DELETE, enuMode, cItemList);
		if (cItemList.IsEmpty())
		{
			m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
			return true;
		}

	}
	else
	{
		cItemList.AddItem((CTFResourceItem*)pItem->Clone());
	}

	// 削除したいアイテムをループでまわす
	OI_STRING_A	strLockToken;
	OI_STRING_A	strLockTokenIfHeader;
	CTFDavResource* pDavItem;
	for (; pDelItem = cItemList.GetNextItem();)
	{
		pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pDelItem);
		if (pDavItem == NULL) continue;

		strLockToken.erase();
		strLockTokenIfHeader.erase();

		// deleteの前にPROPFINDでリソースの確認を行なうかをハンドラで問い合わせる
		if (m_pTransHandler->OnQueryPropfindBeforeMethod(enuMode, TF_DAV_DELETE))
		{
			if (!ExistResource(pDavItem->GetURI(), vLocks, oResult, pWsWorker))
			{
				// 削除したいのにアイテムが存在しなかったが結果的には正常エラーである
				// TODO: エラーアイテムをどうにかしなさい
				bResult = false;
				continue;
			}
		}

		if (HAS_LOCK(vLocks))
		{
			CTFIfHeader cIf;
			cIf.GetLockTokenIfHeader(vLocks, pDavItem->GetURI(), strLockTokenIfHeader);
		}

		// TODO: こんな安直な削除でよいわけないでしょ・・・
		oResult = pWsWorker->DELETE(pDavItem->GetURI(), (strLockTokenIfHeader.empty()) ? NULL : strLockTokenIfHeader.c_str());
		if (oResult == OIDENOCONTENT || oResult == OI_OK)
			bResult = true;
		else
		{
			QueryDAVError(enuMode, TF_DAV_DELETE, oResult, pDavItem);
			bResult = false;
		}
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return bResult;

}

bool 
CTFTransaction::MoveResource(const TFXMLCh* pszSrcServerID,
                             const TFXMLCh* pszDstUri,
                             const TFXMLCh* pszDstServerID,
                             CTFResourceItemList* pItemList)
{
	OI_ASSERT(pszSrcServerID);
	OI_ASSERT(pszDstUri);

	TFTRANSMODE					enuMode		=	TF_TRANS_MOVE;
	bool						bResult		=	false;
	CTFLibWorkSessionWorker*	pWsWorkerS	=	NULL;
	CTFLibWorkSessionWorker*	pWsWorkerD	=	NULL;
	CTFResourceItemList*		pMoveList	=	0;
	TF_STRING_W					strDstURL;							//< 移動先のURL
	bool						bOtherServer=	false;				//< 移動先が外部ならTRUE
	OI_RESULT					oResult;

	// 移動させるリソースが一件もない場合はハンドラで問い合わせる
	if (pItemList == NULL)
	{
		if (!m_pTransHandler->OnQuerySelectedItems(TF_DAV_MOVE, enuMode, *pMoveList))
		{
			m_enuLastResult = TF_RESULT_USERCANCELD;
			return true;
		}
	}
	else
	{
		// TODO: ポインタだけ渡していて平気ですか？
		pMoveList = pItemList;
	}

	// 移動するリソースが無かった場合はキャンセルとみなす
	if (pMoveList->GetItemCount() == 0)
	{
		m_enuLastResult = TF_RESULT_USERCANCELD;
		return true;
	}

	// 移動先のベースURLを作成する
	if (pszDstServerID == NULL)
	{
		strDstURL = GetServerManager().GetServerItem(pszSrcServerID)->GetROOTURL();
	}
	else
	{
		strDstURL = GetServerManager().GetServerItem(pszDstServerID)->GetROOTURL();
		bOtherServer = true;
	}

	// 不慮の事故(URLが取れないということはMOVEができない
	if (strDstURL.empty()){
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	try {

		pWsWorkerS = CTFLibWorkSessionManager::GetSession(pszSrcServerID, hProgress);
		if (pWsWorkerS  == NULL)
			throw CTFException(TF_RESULT_UNEXPECTED);

		if (bOtherServer)
		{
			pWsWorkerD = CTFLibWorkSessionManager::GetSession(pszDstServerID, hProgress);
			if (pWsWorkerD == NULL)
				throw CTFException(TF_RESULT_UNEXPECTED);
		}

		// 移動先の親フォルダの確認
		LOCKLIST		vpLock;
		CTFIfHeader		cIF;
		OI_STRING_A		strLockTokenIfHeader;
		OI_STRING_A		strLockToken;

		CTFLibWorkSessionWorker* pWsWorker = (pWsWorkerD) ? pWsWorkerD : pWsWorkerS;
		if (m_bSeqParentchk)
		{
			if (!ExistResource(pszDstUri, vpLock, oResult, pWsWorker))
				throw CTFException(TF_RESULT_NOTFOUND);

			if (HAS_LOCK(vpLock))
			{
				cIF.GetLockTokenIfHeader(vpLock, pszDstUri, strLockTokenIfHeader);
			}
		}

		CTFDavResource*		pDavItem;
		CTFURL				cURL;
		TF_STRING_W			strDstURI;
		LOCKLIST			vLock;
		CTFDavResourceList	cErrList;
		bool				bOverWrite			= false;
		TFDLGNOTIFYTYPE		enuOverWriteNotify	= TF_DLG_NOTIFY_NONE;

		for (; pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pMoveList->GetNextItem());)
		{
			cURL.parseURL(pDavItem->GetURI());
			strDstURI  = pszDstUri;
			strDstURI += cURL.m_strName;
			vLock.clear();

			// source propfind
			if (!ExistResource(pDavItem->GetURI(), vLock, oResult, pWsWorkerS))
			{
				cErrList.AddItem((CTFDavResource*)pDavItem->Clone());
				continue;
			}

			if (enuOverWriteNotify == TF_DLG_NOTIFY_NONE || enuOverWriteNotify & ~(TF_DLG_NOTIFY_ALL))
			{
				// destination propfind
				if (ExistResource(strDstURI.c_str(), vLock, oResult, pWsWorker))
				{
					enuOverWriteNotify = m_pNotifyHandler->OnNotifyMessage(TF_RESULT_RESORCE_EXIST,
																		   TF_DLG_NOTIFY_STD);

					if (enuOverWriteNotify & TF_DLG_NOTIFY_TRUE)
						bOverWrite = true;
				}
			}

			// source lock
			if (GetLock(pDavItem->GetURI(), pWsWorkerS, strLockToken) == OI_OK)
				cIF.GetLockTokenIfHeader(strLockToken.c_str(), pDavItem->GetURI(), strLockTokenIfHeader);
			
			// destination lock
			if (GetLock(strDstURI.c_str(), pWsWorker, strLockToken) == OI_OK)
				cIF.GetLockTokenIfHeader(strLockToken.c_str(), strDstURI.c_str(), strLockTokenIfHeader);

			if (bOtherServer)
			{
				// TODO: サーバ間移動の場合の処理を書きなさい。
			}
			else
			{
				oResult = pWsWorker->MOVE(pDavItem->GetURI(), strDstURI.c_str(), NULL, bOverWrite, strLockTokenIfHeader.c_str());
			}
		}

	} catch (CTFException& e) {
		m_enuLastResult = e.GetCode();
		bResult = false;
	}


	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorkerS, m_bConPending);
	if (pWsWorkerD) CTFLibWorkSessionManager::CloseSession(pWsWorkerD);

	return bResult;
}

bool 
CTFTransaction::MakeNewResource(const TFXMLCh* pszServerID)
{
	return false;
}

bool
CTFTransaction::Lock(CTFResourceItem* ppItem)
{
	OI_ASSERT(ppItem);
	
	TFTRANSMODE enuMode = TF_TRANS_LOCK;

	CTFResourceItemList* pItemList=0;
	m_pTransHandler->OnQuerySelectedItems(TF_DAV_LOCK, enuMode, *pItemList);
	if (pItemList->IsEmpty()) return true;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(ppItem, hProgress);
	if (pWsWorker == NULL)
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	CTFResourceItem* pItem = NULL;
	CTFResourceItemList cErrItem;
	OI_STRING_A	strLockToken;

	for (; pItem = pItemList->GetNextItem(); )
	{
		CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, pItem);
		if (pDavItem == NULL) continue;
		if (GetLock(pDavItem->GetURI(), pWsWorker, strLockToken) == OI_OK)
		{
			// TODO: LOCKの成功ケース完成させなさい
		}
		else {
			// TODO: LOCKの失敗ケース完成させなさい
			cErrItem.AddItem(pItem);
		}
	}

	if (!cErrItem.IsEmpty())
	{
		// TODO: エラーが存在した場合の処理
	}

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);
	return false;
}

bool
CTFTransaction::SetAvailableUser(const TFXMLCh* pszServerID, CTFResourceItemList* pGrpList, CTFResourceItemList* pItemList)
{
	OI_ASSERT(pszServerID);
	OI_ASSERT(pGrpList);
	bool bResult = false;
	OI_RESULT oResult;

	TFTRANSMODE	enuMode = TF_TRANS_AVAILABLEUSER;

	// 所属ユーザの確認
	CTFResourceItemList* pList=0;
	if (pItemList != NULL && pItemList->GetItemCount() > 0)
	{
		pList = pItemList;
	}
	else
	{
		// 所属させるユーザが一人もいない場合ハンドラで問い合わせる
		if (m_pTransHandler->OnQuerySelectedItems(TF_DAV_COPY, enuMode, *pList))
			return false;

		if (pList->GetItemCount() == 0)
			return false;
	}

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	CTFLibWorkSessionWorker* pWsWorker = CTFLibWorkSessionManager::GetSession(pszServerID, hProgress);
	if (pWsWorker == NULL) 
	{
		m_enuLastResult = TF_RESULT_UNEXPECTED;
		return false;
	}

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	CTFResourceItem* pItem;
	LOCKLIST		 vLocks;
	TF_STRING_W		strURI, strDstURL;
	TF_STRING_W		strUserURI;	// ユーザのURI
	for (; pItem = TFLIB_DYNAMIC_CAST(CTFResourceItem, pGrpList->GetNextItem()); )
	{
		strURI.erase();
		strDstURL.erase();
		strURI  = pItem->GetRootURI();
		strURI += X(TF_URI_MNG_GROUP);
		strURI += pItem->GetName();
		strURI += X("/");
		if (!ExistResource(strURI.c_str(), vLocks, oResult, pWsWorker))
		{
			// TODO: グループが存在しなかった場合どうするの？
			continue;
		}

		// ユーザがグループに存在するか確認
		pList->ResetItemPosition();
		CTFResourcePropTFUser* pUsrItem;
		TF_STRING_W strTmpURI = strURI;
		for (; pUsrItem = TFLIB_DYNAMIC_CAST(CTFResourcePropTFUser, pList->GetNextItem()); )
		{
			strURI  = strTmpURI;
			strURI += X(TF_AVAILABLE_USER_PREFIX);
			strURI += pUsrItem->GetUID();
			strURI += X("/");
			if (ExistResource(strURI.c_str(), vLocks, oResult, pWsWorker))
			{
				// TODO: 存在したらあきらめる処理を書きなさい
				continue;
			}

			// 存在しなかった場合ロックをかけてコピー処理（ユーザ所属）を行う
			strUserURI  = pItem->GetRootURI();
			strUserURI += X(TF_URI_MNG_USER);
			strUserURI += pUsrItem->GetUID();
			strUserURI += X("/");

			TF_STRING_A strLockToken;	// ロックトークン
			TF_STRING_A strIfHeader;	// ロックトークンヘッダ
			if (GetLock(strURI.c_str(), pWsWorker, strLockToken, 3) == OI_OK)
			{
				// distinationのURLを作成する
				strDstURL  = pItem->GetROOTURL();
				strDstURL += X(TF_URI_MNG_GROUP);
				strDstURL += pItem->GetName();
				strDstURL += X("/");
				strDstURL += X(TF_AVAILABLE_USER_PREFIX);
				strDstURL += pUsrItem->GetUID();
				strDstURL += X("/");

				CTFIfHeader cIf;
				cIf.GetLockTokenIfHeader(strLockToken.c_str(), strURI.c_str(), strIfHeader);
				oResult = pWsWorker->COPY(strUserURI.c_str(), strDstURL.c_str(), NULL, false, (strLockToken.empty()) ? NULL : strIfHeader.c_str());

				UnLock(strURI.c_str(), strLockToken.c_str(), pWsWorker);
				// TODO: Unlockのエラーはどうするの？
			}
			else
			{
				// TODO: ロックに失敗した
				continue;
			}
		}
	}

	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return true;
}

bool
CTFTransaction::Upload(const TFXMLCh* pszServerID,
					   CTFResourceItem* ppItem,
					   TF_STRVECTOR_W vFiles)
{

	if (IS_EMPTY(pszServerID)) return false;
	if (IS_EMPTY(ppItem) && vFiles.empty()) return false;

	bool						bResult		= false;
	CTFLibWorkSessionWorker*	pWsWorker	= NULL;
	CTFLocalFileItem*			pFile		= NULL;
	CTFLocalFileItemList		cFileList;

	// リザルトコードの処理化
	m_enuLastResult = TF_RESULT_UNEXPECTED;

	TFTRANSMODE enuMode = TF_TRANS_UPLOADRESOURCE;
	CTFDavResource* pDavItem = TFLIB_DYNAMIC_CAST(CTFDavResource, ppItem);
	if (pDavItem = NULL) return false;

	TFHANDLE hProgress;
	m_pNotifyHandler->OnInitializeProgressWindow(&hProgress, enuMode);

	// コネクションの取得
	pWsWorker = CTFLibWorkSessionManager::GetSession(ppItem, hProgress);
	if (pWsWorker == NULL) return false;

	// ハンドラの設定
	pWsWorker->SetHandler(m_pTransHandler);
	pWsWorker->SetHandler(m_pNotifyHandler);
	pWsWorker->SetHandler(m_pSysHandler);

	// サーバのURIを取得する
	const TFXMLCh* pszSvrURI = GetServerManager().GetServerItem(pszServerID)->GetRootURI();
	if (IS_EMPTY(pszSvrURI)) return false;

	// ファイルの一覧を取得する
	// Factoryでプラットフォーム依存のユーティリティを取得する
	CTFLocalFileUtils*	pFileUtiles = m_pPFFactory->CreateLocalFileUtils();
	if (pFileUtiles == NULL)
	{
		m_enuLastResult = TF_RESULT_LSYS_NOTFOUND;
		return false;
	}

	// ハンドラの登録を行う（このハンドラは渡してもそのまま放置でいい）
	pFileUtiles->SetHandler(m_pTransHandler);

	std::vector<TF_STRING_W>::iterator	it = vFiles.begin();
	TF_SIZE_T unTotalSize = 0;
	for (;it != vFiles.end();it++)
	{
		// ここで作成するpFileはリストに渡してしまう為
		// Clone不要。リストが全て消してくれます
		CTFLocalFileItemList cList;
		pFile = m_pPFFactory->CreateLocalFileItem();
		pFile->SetPath(0, (void*)(*it).c_str(), sizeof(TFXMLCh));
		cList.AddItem(pFile);

		// 中身を全て列挙する
		TFFSTAT f = pFileUtiles->EnumLocalFiles(&cList, pFileUtiles, unTotalSize);
		if (f != e_NOERROR) 
		{
			// ファイルがなかった。
			m_enuLastResult = TF_RESULT_LSYS_NOTFOUND;
			continue;
		}

		// cListはスタックの為中身を全てクローンで作り直します。
		// そうしないとcListの中身のItemが全て削除されてしまう
		for (CTFObject* pTmpItem; (pTmpItem = cList.GetNextItem()) != NULL; )
		{
			cFileList.AddItem((CTFLocalFileItem*)pTmpItem->Clone());
		}
	}

	for (CTFLocalFileItem* pItem; (pItem = cFileList.GetNextItem()) != NULL;)
	{
#if 0
		LPCTSTR pch;
		int nSize = 0;
		pItem->GetPath(0, (void**)&pch, &nSize);
#endif
		TF_STRING_W strURI = pszSvrURI;
		strURI += pItem->GetURI();

		// 開始メッセージを通知する。
		m_pNotifyHandler->OnPreSendProgressWindow(strURI.c_str(), hProgress);

		OI_RESULT oResult = OIGEINVALIDSTATE;
		TFDAVMETHOD enuMethod = TF_DAV_UNKNOWN;

		if (m_pTransHandler->OnQueryDetailLocalResource(pItem, enuMode) == false)
		{
			enuMethod = TF_DAV_PUT;
			m_enuLastResult = TF_RESULT_SKIP;
		}
		else
		{

			if (pItem->isFolder())
			{
				// MKCOL
				// 一時フォルダは無視
				if (pItem->isTemporary()) continue;
				enuMethod = TF_DAV_MKCOL;
				oResult = pWsWorker->MKCOL(strURI.c_str(), NULL, NULL);
			}
			else 
			{
				// PUT
				enuMethod = TF_DAV_PUT;
				TFFSTAT enuFstat;
				enuFstat = pFileUtiles->Open(*pItem, CTFLocalFileUtils::TF_F_RDONLY);
				if (enuFstat == e_NOERROR)
				{
					oResult = pWsWorker->PUT(strURI.c_str(), pItem->GetFD(), NULL, pItem->GetContentType());
					pFileUtiles->Close(*pItem);
				}
				else
				{
					oResult = OIGEFILEOPENFAILED;
				}
			}

			// メソッド処理の結果調査
			(void)QueryDAVError(enuMode, enuMethod, oResult, pItem);

		}

		m_pTransHandler->OnResultLocalResource(enuMode, enuMethod, pItem, m_enuLastResult);
	}


	// Factoryで作成したユーティリティ delete
	delete pFileUtiles;

	// コネクションの開放
	m_pNotifyHandler->OnTerminateProgressWindow(hProgress);
	CTFLibWorkSessionManager::CloseSession(pWsWorker, m_bConPending);

	return (m_enuLastResult == TF_RESULT_OK);

}

void CTFTransaction::SetKeepSession(bool bKeep)
{
	m_bConPending = bKeep;
}
