#include "TFXMLHolder.h"
#include "TFXMLParser.h"
#include "TFXMLErrorHandler.h"
#include <string.h>
#include <xercesc/framework/StdOutFormatTarget.hpp>
#include "TFPlatformsFactory.h"
#include "oimap.h"

CTFXMLHolder::CTFXMLHolder(void)
{
	m_pDOMParser = new CTFXMLParser;
	m_pDOMErrorHandler = new CTFXMLErrorHandler;

	m_pDOMParser->setDoNamespaces(true);
	m_pDOMParser->setIncludeIgnorableWhitespace(false);
	m_pDOMParser->setExitOnFirstFatalError(true);
    m_pDOMParser->setErrorHandler(m_pDOMErrorHandler);
}

CTFXMLHolder::~CTFXMLHolder(void)
{
	// DOMドキュメントを破棄する
	m_pDOMParser->reset();
	Destroy();

	// DOMパーサーを破棄する
	delete m_pDOMParser;
 	delete m_pDOMErrorHandler;
}

//***************************************************************************
/**
 *	XMLファイルを読み込み、DOMドキュメントに格納する
 *
 *	@param	szFilePath	LPCTSTR	ファイルパス
 *	@return	BOOL	成功:TRUE、失敗:FALSE
 *
 */
bool CTFXMLHolder::Load(const TFXMLCh* pszFilePath)
{
	if( !pszFilePath || TF_STRING_W(pszFilePath).empty() ) return false;

	// パーサーでxmlファイルをパースする
	// このときのDOMドキュメントは、
	// パーサーオブジェクトの持ち物であり、パーサーを破棄したときに消滅する
	// そのため、パーサーが消える前に、全ノードを別のドキュメントに移し替える必要がある
	try
	{
		m_pDOMParser->parse(pszFilePath);
	}
	catch(...)
	{
		// エラー時
		//OI_DEBUG(_T("CTFXMLHolder::Load() ... Read Failure!\n"));
		return false;
	}

	XNS(DOMDocument)* pSourceDoc = m_pDOMParser->getDocument();
	if( !pSourceDoc ) return false;

	// 現在のDOMドキュメントを消す
	m_pDOMParser->reset();
	Destroy();

	// 入力されたDOMドキュメントのルートノードを得る
	XNS(DOMElement)* pSourceRootNode = pSourceDoc->getDocumentElement();

	// 新しいDOMドキュメントを生成する
	// この時に同時に生成されるルートノードは、ダミーである
	XMLCh tempStr[100];
	XNS(XMLString)::transcode("LS", tempStr, 99);
	XNS(DOMImplementation) *impl = XNS(DOMImplementationRegistry)::getDOMImplementation(tempStr);
	XNS(DOMDocument)* pDOMDocument = impl->createDocument(	0,
															X("DummyRootNode"),	// ダミー
															0 );
	OI_ASSERT(pDOMDocument);

	// ドキュメントをCTFXMLUtilsのメンバー変数に設定する
	SetDocument(pDOMDocument,true);

	// パーサーが持っているルートノードをインポートする
	XNS(DOMNode)* pNewRootNode = pDOMDocument->importNode(pSourceRootNode, true);

	// ダミーのルートノード新しいDOMドキュメントからを消す
	XNS(DOMNode)* pDummyRootNode = pDOMDocument->getDocumentElement();
	pDOMDocument->removeChild(pDummyRootNode);

	// 新しいルートノードを新しいDOMドキュメントに差し込む
	pDOMDocument->appendChild(pNewRootNode);
	pDummyRootNode->release();


	return true;
}

//***************************************************************************
/**
 *	DOMドキュメントに格納されているデータをXMLファイルに出力する
 *
 *	@param		pszFilePath	const TFXMLCh*	ファイルパス
 *	@return	BOOL	成功:TRUE、失敗:FALSE
 */
bool CTFXMLHolder::Save(const TFXMLCh* pszFilePath)
{
	if( !pszFilePath || TF_STRING_W(pszFilePath).empty() ) return false;

	XNS(DOMDocument)* pDocument = GetDocument();
	if( !pDocument ) return false;

	// 書き込み時のコードを指定する
	XMLCh tempStr[100];
	XNS(XMLString)::transcode("LS", tempStr, 99);
	XNS(DOMImplementation) *impl = XNS(DOMImplementationRegistry)::getDOMImplementation(tempStr);
	// DOMWriterを生成する
//	XNS(DOMWriter)*	lpWriter = ((XNS(DOMImplementationLS)*)impl)->createDOMWriter();
	XNS(DOMLSSerializer)* lpSSerializer;
	XNS(DOMLSOutput)* lpOutput;
	lpSSerializer = ((XNS(DOMImplementationLS)*)impl)->createLSSerializer();
	lpOutput = ((XNS(DOMImplementationLS)*)(impl))->createLSOutput();

	lpSSerializer->setNewLine((const TFXMLCh*)L"\r\n");
//	lpWriter->setFeature(XNS(XMLUni)::fgDOMWRTFormatPrettyPrint, true);
	lpSSerializer->getDomConfig()->setParameter(XNS(XMLUni)::fgDOMWRTFormatPrettyPrint, true);
	XNS(LocalFileFormatTarget)* lpFormatTarget = 0;

	lpOutput->setByteStream(lpFormatTarget);
	bool bResult = false;
	try
	{
		lpFormatTarget = new XNS(LocalFileFormatTarget)(pszFilePath);
//		lpWriter->writeNode(lpFormatTarget, *pDocument);
		lpSSerializer->write(pDocument, lpOutput);
		bResult = true;	// ここまで処理が来たら、書き込み成功
	}
	catch(...)
	{
		// 書き込みエラー
		//OI_DEBUG(_T("*** CGDivyXMLHolder::Save() ... Write Failure!\n"));
	}

//	lpWriter->release();
	lpOutput->release();
	lpSSerializer->release();

	if(lpFormatTarget) delete lpFormatTarget;

	return bResult;
}

//***************************************************************************
/**
 *	DOMドキュメントに格納されているデータを標準出力に出力する
 *
 *	@return	BOOL	成功:TRUE、失敗:FALSE
 */
bool CTFXMLHolder::Print(XNS(DOMElement)* lpElement)
{
	XNS(DOMNode)* lpTargetNode = NULL;
	if (lpElement != NULL)
		lpTargetNode = lpElement;
	else
		lpTargetNode = GetDocument();

	if (lpTargetNode == NULL) return false;

	// 書き込み時のコードを指定する
	XMLCh tempStr[100];
	XNS(XMLString)::transcode("LS", tempStr, 99);
	XNS(DOMImplementation) *impl = XNS(DOMImplementationRegistry)::getDOMImplementation(tempStr);
	// DOMWriterを生成する
//	XNS(DOMWriter)*	lpWriter = ((XNS(DOMImplementationLS)*)impl)->createDOMWriter();
	XNS(DOMLSSerializer)* lpSSerializer = ((XNS(DOMImplementationLS)*)impl)->createLSSerializer();;
	XNS(DOMLSOutput)* lpOutput = ((XNS(DOMImplementationLS)*)(impl))->createLSOutput();

	lpSSerializer->setNewLine((const TFXMLCh*)L"\n");
	if (lpSSerializer->getDomConfig()->canSetParameter(XNS(XMLUni)::fgDOMWRTFormatPrettyPrint, true))
//	if (lpWriter->canSetFeature(XNS(XMLUni)::fgDOMWRTFormatPrettyPrint, true))
		lpSSerializer->getDomConfig()->setParameter(XNS(XMLUni)::fgDOMWRTFormatPrettyPrint, true);
//		lpWriter->setFeature(XNS(XMLUni)::fgDOMWRTFormatPrettyPrint, true);

	if (lpSSerializer->getDomConfig()->canSetParameter(XNS(XMLUni)::fgDOMWRTSplitCdataSections, true))
//	if (lpWriter->canSetFeature(XNS(XMLUni)::fgDOMWRTSplitCdataSections, false))
lpSSerializer->getDomConfig()->setParameter(XNS(XMLUni)::fgDOMWRTSplitCdataSections, true);
//		lpWriter->setFeature(XNS(XMLUni)::fgDOMWRTSplitCdataSections, false);

	XNS(StdOutFormatTarget)* lpFormatTarget = 0;
	bool bResult = false;
	try
	{
		lpFormatTarget = new XNS(StdOutFormatTarget)();	// 標準出力に出す
		lpOutput->setByteStream(lpFormatTarget);
		lpSSerializer->write(lpTargetNode, lpOutput);
//		lpWriter->writeNode(lpFormatTarget, *lpTargetNode);
		bResult = true;	// ここまで処理が来たら、書き込み成功
	}
	catch(...)
	{
		// 書き込みエラー
		//OI_DEBUG(_T("*** CGDivyXMLHolder::Save() ... Write Failure!\n"));
	}

//	lpWriter->release();
	lpOutput->release();
	lpSSerializer->release();

	if(lpFormatTarget) delete lpFormatTarget;

	return bResult;
}

