#include <stdio.h>
#include "TFXMLUtils.h"
#include "oimap.h"

CTFXMLUtils::CTFXMLUtils(void)
{
	m_pDOMDocument = 0;
	m_pRootElement = 0;
	m_bNeedToDestroy = false;
}

CTFXMLUtils::~CTFXMLUtils(void)
{
	Destroy();
}

/**
 *	DOMDocumentのルートノードに新しいNamespaceを追加する
 *
 *	@param	pszNamespaceURI	const char*	NamespaceのURL
 *	@param	pszPrefix		conat char*	Namespaceのプレフィックス(:不要)
 *	@return	bool			成功した場合はTRUE、それ以外はFALSE
 *
 */
bool 
CTFXMLUtils::AppendNamespace(const TFXMLCh* pszNSURL, const TFXMLCh* pszNSPrefix)
{

	if( m_pDOMDocument )
	{
		XNS(DOMElement)* pRootNode = m_pDOMDocument->getDocumentElement();

		if( pRootNode )
		{				
			TF_STRING_W pszXMLNS = (const TFXMLCh*)X("xmlns:"); 
			pszXMLNS.append(pszNSPrefix);

			XNS(DOMAttr)* pAttribute = m_pDOMDocument->createAttributeNS(
																X(OI_XML_NSURI),
																  pszXMLNS.c_str());
			pAttribute->setValue(pszNSURL);
			pRootNode->setAttributeNode(pAttribute);
			return true;
		}
	}

	return false;
}

bool
CTFXMLUtils::RemoveNamespace(const TFXMLCh*  pszNSURI, const TFXMLCh* pszNSPrefix)
{
	if ( m_pDOMDocument )
	{
		XNS(DOMElement)* pRootNode = m_pDOMDocument->getDocumentElement();
		
		if (pRootNode)
		{
		
			pRootNode->removeAttributeNS(pszNSURI, pszNSPrefix);
		}
	}
	
	return true;
}

/**
 *	デフォルトのURLを設定する
 *
 *	@param	szNSURL 	const TFXMLCh*		ネームスペースのURL
 *	@param	szNSPrefix	const TFXMLCh*		ネームスペースのプレフィックス
 *	@return	BOOL					TRUE:成功、FALSE:失敗
 */
bool 
CTFXMLUtils::SetDefaultNSURL(const TFXMLCh* pszNSURL, const TFXMLCh* pszNSPrefix)
{
	if( pszNSURL )
	{
		m_strDefaultNSURL = pszNSURL;
	}
	else
	{
		m_strDefaultNSURL.empty();
	}

	if( pszNSPrefix )
	{
		m_strDefaultNSPrefix = pszNSPrefix;
	}
	else
	{
		m_strDefaultNSPrefix.empty();
	}

	return true;
}


/**
 *	デフォルトのURLを得る
 *
 *	@param	pszNSURL	const TFXMLCh*		引数により指定されたネームスペースURL
 *	@return	const TFXMLCh*					ネームスペースURL（NULLの場合は未設定）
 */
const TFXMLCh* 
CTFXMLUtils::GetCurrentNSURL(const TFXMLCh* pszNSURL)
{
	if( pszNSURL )
	{
		return pszNSURL;
	}
	else
	{
		if( !m_strDefaultNSURL.empty() )
		{
			return m_strDefaultNSURL.c_str();
		}
		else
		{
			return 0;
		}
	}
}



/**
 *	新たにDOMドキュメントを生成する
 *
 *	@param	szRootNodeName	const char*	ルートノードの名前(NULLの場合、ルートノード無し)
 *	@param	pszNSURL		const char*	ネームスペースのURL(NULLの場合、ネームスペース無し)
 *	@param	szPrefix		const char*	ネームスペースのプレフィックス（:不要。NULLの場合、プレフィックス無し）
 *	@param	szHrefXSL		const char*	スタイルシートのURL（NULLの場合、スタイルシート無し）
 */
bool 
CTFXMLUtils::Create(const TFXMLCh* pszRootNodeName, const TFXMLCh* pszNSURL,
												  const TFXMLCh* pszNSPrefix, const TFXMLCh* pszXsltURL)
{
	if( m_pDOMDocument ) Destroy();

	try
	{
		m_bNeedToDestroy = true;

		// ImplementationRegistry
		XMLCh tempStr[100];
		XNS(XMLString)::transcode("LS", tempStr, 99);
		XNS(DOMImplementation) *impl = XNS(DOMImplementationRegistry)::getDOMImplementation(tempStr);


		// DOMドキュメントを生成
		if( pszRootNodeName && !TF_STRING_W(pszRootNodeName).empty() )
		{
			// ルートノード名を決める
			TF_STRING_W strRootNodeNameTmp;
			if( pszNSPrefix && !TF_STRING_W(pszNSPrefix).empty() )
			{
				strRootNodeNameTmp  = pszNSPrefix;
				strRootNodeNameTmp += (const TFXMLCh*)X(":");
			}
			strRootNodeNameTmp.append(pszRootNodeName);

			m_pDOMDocument = impl->createDocument(
											X(pszNSURL),
											strRootNodeNameTmp.c_str(),
											0
											);
		}
		else
		{
			// ルートノード無しのDOMドキュメントを作る
			m_pDOMDocument = impl->createDocument();
		}

		// プロセッシングインストラクションにスタイルシート(xsl)を設定
		if( pszXsltURL && !TF_STRING_W(pszXsltURL).empty() )
		{
			TF_STRING_W strPIData;
			strPIData  = (const TFXMLCh*)X("type=\"text/xsl\" href=\"");
			strPIData += pszXsltURL;
			strPIData += (const TFXMLCh*)X("\"");
			XNS(DOMProcessingInstruction)* pi = m_pDOMDocument->createProcessingInstruction(
															X("xml-stylesheet"),
															strPIData.c_str()
														);
			m_pDOMDocument->appendChild(pi);
		}
	}
	catch(...)
	{
		Destroy();
		return false;
	}

	return true;
}

/**
 *	DOMドキュメントを破棄する。
 *	DOMパーサーの状態もリセットされる
 *
 *	@return	bool	成功:true、失敗:false
 *
 */
bool
CTFXMLUtils::Destroy(void)
{
	if( m_pDOMDocument && m_bNeedToDestroy ) m_pDOMDocument->release();

	m_pDOMDocument = 0;
	m_bNeedToDestroy = false;

	return true;
}

/**
 *	DOMドキュメントを設定する
 *
 *	@param	pDOMDocument	XNS(DOMDocument)*	DOMドキュメント
 *	@param	bNeedToDestroy	bool				このクラスによってドキュメントの破棄が必要な場合はtrue
 *
 */
void CTFXMLUtils::SetDocument(XNS(DOMDocument)* pDOMDocument, bool bNeedToDestroy)
{
	if( m_pDOMDocument ) Destroy();

	m_pDOMDocument = pDOMDocument;
	m_bNeedToDestroy = bNeedToDestroy;
}

/**
 *	DOMドキュメントを得る
 *
 *	@return	XNS(DOMDocument)*	DOMドキュメント
 *
 */
XNS(DOMDocument)* 
CTFXMLUtils::GetDocument(void)
{
	return m_pDOMDocument;
}

/**
 *	ルートエレメントを設定する。
 *	このメソッドは、XMLパース結果を解析する際にDOMドキュメントを使わない場合に設定する
 *
 *	@param	pRootElement	XNS(DOMElement)*	ルートエレメント
 *
 */
void
CTFXMLUtils::SetRootElement(XNS(DOMElement)* pRootElement)
{
	if( !m_pDOMDocument )
	{
		m_pRootElement = pRootElement;
	}
	else
	{
		m_pRootElement = 0;
	}
}


/**
 *	DOMドキュメントのルートエレメントを得る。
 *	無ければメンバー変数のルートエレメントを返す。
 *
 *	@return	XNS(DOMElement)*	DOMエレメント
 *
 */
XNS(DOMElement)* 
CTFXMLUtils::GetRootElement(void)
{
	if( m_pDOMDocument )
	{
		return m_pDOMDocument->getDocumentElement();
	}
	else
	{
		return m_pRootElement;
	}
}


/**
 *	指定された親エレメントに指定された名前の子エレメントを追加する
 *
 *	@param	pParent		XNS(DOMElement)*	エレメント(NULL不可)
 *	@param	pszElement	const char*			エレメント名
 *	@return	bool							true:成功、false:失敗
 */
XNS(DOMElement)* 
CTFXMLUtils::AppendElement(XNS(DOMElement)* pParent, const TFXMLCh* pszElement,
													  const TFXMLCh* pszNSURL, const TFXMLCh* pszNSPrefix)
{

	// エレメント名が無い場合は、NULLを返す
	if( !pszElement || TF_STRING_W(pszElement).empty() ) return 0;

	// 親エレメントが無い場合も、NULLを返す
	if( !pParent || (pParent->getNodeType() != XNS(DOMNode)::ELEMENT_NODE) ) return 0;

	// DOMドキュメントが無い場合も、NULLを返す
	if( !m_pDOMDocument ) return 0;

	// ネームスペースURLを解決する
	// NULLが指定された場合は、デフォルトのURLを使用する
	const TFXMLCh* pszSetNSURL = GetCurrentNSURL(pszNSURL);

	// エレメントのQualifiedNameを生成する
	TF_STRING_W strQualifiedName;
	if( pszNSPrefix )
	{
		// 指定されたプレフィックスを付ける
		strQualifiedName = pszNSPrefix;
	}
	else
	{
		// デフォルトのプレフィックスを付ける
		strQualifiedName = m_strDefaultNSPrefix.c_str();
	}
	// プレフィックスがある場合は":"を付加する
	if( !strQualifiedName.empty() )
	{
		strQualifiedName.append(C2W(":"));
	}
	// エレメント名を追加する
	strQualifiedName.append(pszElement);

	// エレメントを生成する
	XNS(DOMElement)* pNewElement = m_pDOMDocument->createElementNS(	pszSetNSURL,
																	strQualifiedName.c_str()
																	);
	if( pNewElement )
	{
		// 親エレメントに新しいエレメントを追加する
		XNS(DOMElement)* pNewChild= (XNS(DOMElement)*)pParent->appendChild(pNewElement);
		if( pNewChild )
		{
			// 新しい子エレメントを返す
			return pNewChild;
		}
		else
		{
			// 追加に失敗した場合はNULLを返す（通常ありえない）
			OI_ASSERT(false);
		}
	}
	return NULL;
}

/**
 *	指定されたエレメントを消す。
 *	ルートエレメントの消去には使えない
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント(NULL不可)
 *	@return	bool							true:成功、false:失敗
 */
bool
CTFXMLUtils::RemoveElement(XNS(DOMElement)* pElement)
{
	if( !pElement || (pElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE) ) return false;

	if( pElement )
	{
		// 親ノードを得る
		XNS(DOMNode)* pParent = pElement->getParentNode();
		if( pParent )
		{
			// 親ノードから子ノードを除去する
			XNS(DOMNode)* pRemoved = pParent->removeChild(pElement);
			// 除去した子ノードを解放する
			if( pRemoved ) pRemoved->release();
			return true;
		}
	}
	return false;
}

/**
 *	指定されたエレメントを空エレメントにする。
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント(NULLの場合はルートエレメント)
 *	@return	bool							true:成功、false:失敗
 */
bool
CTFXMLUtils::EmptyElement(XNS(DOMElement)* pElement)
{
	XNS(DOMElement)* pTargetElement = NULL;
	if( pElement )
	{
		pTargetElement = pElement;
	}
	else
	{
		pTargetElement = GetRootElement();
	}
	
	if( !pTargetElement || (pTargetElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE) ) return false;

	XNS(DOMNode)* pNode = pTargetElement->getLastChild();
	while(pNode)
	{
		XNS(DOMNode)* pRemoved = pTargetElement->removeChild(pNode);
		if( pRemoved ) pRemoved->release();		// ノードを解放する
		pNode = pTargetElement->getLastChild();
	}

	return true;
}

/**
 *	指定された名前の子エレメントを返す
 *
 *	@param	pParent		XNS(DOMElement)*	親エレメント(0の場合はルートエレメント)
 *	@param	pszElement	const TFXMLCh*			エレメント名
 *	@param	iIndex		int					エレメントのインデックス番号（0以上）
 *	@param	pszNSURL	const TFXMLCh*			ネームスペースURL（0の場合はデフォルトのURLを使う）
 *	@return	XNS(DOMElement)*				!=0：エレメントが見つかった
 *											==0：該当するエレメントが無い
 */
XNS(DOMElement)* CTFXMLUtils::GetElement(XNS(DOMElement)* pParent, const TFXMLCh* pszElement, int iIndex, const TFXMLCh* pszNSURL)
{
	// エレメント名が無い場合は、NULLを返す
	if( !pszElement || TF_STRING_W(pszElement).empty() ) return 0;

	// ネームスペースURLを解決する
	// NULLが指定された場合は、デフォルトのURLを使用する
	const TFXMLCh* pszGetNSURL = GetCurrentNSURL(pszNSURL);

	if( pParent )
	{
		// 親エレメントが指定されている場合

		// エレメントノードであることを一応チェックする
		if( pParent->getNodeType() != XNS(DOMNode)::ELEMENT_NODE ) return 0;

		// 子エレメント（リスト）を探す
		XNS(DOMNodeList) *pNodeList = pParent->getElementsByTagNameNS(
																		pszGetNSURL,
																		pszElement);
		if( pNodeList )
		{
			// 子エレメントが見つかった場合、指定したインデックス番号のエレメントを得る
			XNS(DOMNode)* pTempNode = pNodeList->item(iIndex);
			if( pTempNode && (pTempNode->getNodeType() == XNS(DOMNode)::ELEMENT_NODE) )
			{
				// 子エレメントが見つかった場合は、そのエレメントを返す
				return (XNS(DOMElement)*)pTempNode;
			}
		}
	}
	else
	{
		// 親エレメントが指定されていない(NULL)の場合
		// ルートノードが指定された名前かどうかをチェックする

		// ルートエレメントを得る
		XNS(DOMElement)* pRootElement = GetRootElement();
		if( pRootElement )
		{
			// ネームスペースURLを得る
			const XMLCh* pszxRootElementNSURL = pRootElement->getNamespaceURI();

			if( (pszxRootElementNSURL && !TF_STRING_W(pszxRootElementNSURL).empty())
					&& (pszGetNSURL && !TF_STRING_W(pszGetNSURL).empty()) )
			{
				// ネームスペースURLが存在している場合は、NamespaceURLの一致をチェックする
				if( TF_STRING_W(pszxRootElementNSURL).compare(pszGetNSURL) != 0 )
				{
					return 0;
				}
			}
			else if( (pszxRootElementNSURL && !TF_STRING_W(X(pszxRootElementNSURL)).empty())
					|| (pszGetNSURL && !TF_STRING_W(pszGetNSURL).empty()) )
			{
				// ネームスペースURLが片方だけ存在してる場合は、不一致と見なす
				return 0;
			}

			// プレフィックスを含まないエレメント名を得る
			// （プレフィックス付きを得るには、getTagNameメソッドを使う）
			if( TF_STRING_W(pRootElement->getLocalName()).compare(pszElement) == 0 )
			{
				// エレメント名が指定された名前と同じなら、ルートエレメントを返す
				return pRootElement;
			}
		}
	}

	return 0;
}

/**
 *	指定された名前の子エレメントの数を返す
 *
 *	@param	pParent		XNS(DOMElement)*	親エレメント(NULL不可。ゼロを返す)
 *	@param	pszElement	const TFXMLCh*			エレメント名
 *	@param	pszNSURL	const TFXMLCh*			ネームスペースURL（NULLの場合はデフォルトのURLを使う）
 *	@return	int								エレメント数（無い場合はゼロを返す）
 */
int CTFXMLUtils::GetElementCount(XNS(DOMElement)* pParent, const TFXMLCh* pszElement, const TFXMLCh* pszNSURL)
{
	// エレメント名が無い場合は、NULLを返す
	if( !pszElement || TF_STRING_W(pszElement).empty() ) return 0;

	// ネームスペースURLを解決する
	// NULLが指定された場合は、デフォルトのURLを使用する
	const TFXMLCh* pszGetNSURL = GetCurrentNSURL(pszNSURL);

	if( pParent )
	{
		// 親エレメントが指定されている場合

		// エレメントノードであることを一応チェックする
		if( pParent->getNodeType() == XNS(DOMNode)::ELEMENT_NODE )
		{
			// 子エレメント（リスト）を探す
			XNS(DOMNodeList) *pNodeList = pParent->getElementsByTagNameNS(
																			pszGetNSURL,
																			pszElement );
			if( pNodeList )
			{
				// 子エレメントが見つかった場合、アイテム数を返す
				return pNodeList->getLength();
			}
		}
	}

	return 0;
}

/**
 *	指定されたエレメントの子エレメントを返す。
 *
 *	無い場合はゼロを返す。
 *	同じレベルに複数の子エレメントが存在しても、最初のエレメントしか返せない。
 *
 *	@param	pParent		XNS(DOMElement)*	親エレメント(ゼロの場合はルートエレメントを使用する)
 *	@return	XNS(DOMElement)*				子エレメント（無い場合はゼロを返す）
 */
XNS(DOMElement)* CTFXMLUtils::GetChildElement(XNS(DOMElement)* pParent)
{
	// 親エレメントが0の場合は、ルートエレメントを得る
	XNS(DOMElement)* pParentElement = (pParent != 0) ? pParent : GetRootElement();

	if( pParentElement )
	{
		// 親がエレメントノードであることを一応チェックする
		if( pParentElement->getNodeType() == XNS(DOMNode)::ELEMENT_NODE )
		{
			// 子ノードリストを得る
			XNS(DOMNodeList)* pChildNodeList = pParentElement->getChildNodes();
			if( pChildNodeList )
			{
				// 子ノードの中から最初に出てくるエレメントを探す
				int c = pChildNodeList->getLength();
				for( int i=0; i<c; i++ )
				{
					// 子ノードを1つ得る
					XNS(DOMNode)* pChildNode = pChildNodeList->item(i);
					if( pChildNode && (pChildNode->getNodeType() == XNS(DOMNode)::ELEMENT_NODE) )
					{
						// エレメントが見つかった場合は、そのエレメントを返す
						return (XNS(DOMElement)*)pChildNode;
					}
				}
			}
		}
	}

	return 0;
}

/**
 *	指定されたエレメントの名前を返す
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント(ゼロの場合はルートエレメントを使用する)
 *	@param	strName		TF_STRING_W&		エレメント名を返却してもらうための文字列オブジェクトの参照
 *	@param	strNSURL	TF_STRING_W&		Namespaceを返却してもらうための文字列オブジェクトの参照
 *	@return	bool		エレメント名が得られた場合はtrue
 */
bool CTFXMLUtils::GetElementName(XNS(DOMElement)* pElement, TF_STRING_W& strName, TF_STRING_W& strNSURL)
{
	// エレメントが0の場合は、ルートエレメントを得る
	XNS(DOMElement)* pThisElement = (pElement != 0) ? pElement : GetRootElement();

	strName.clear();
	strNSURL.clear();

	if( pThisElement )
	{
		// エレメントノードであることを一応チェックする
		if( pThisElement->getNodeType() == XNS(DOMNode)::ELEMENT_NODE )
		{
			// エレメント名とネームスペースURLを返す
			strName = pThisElement->getLocalName();
			const XMLCh* pszxNSURL = pThisElement->getNamespaceURI();
			if( pszxNSURL ) strNSURL = pszxNSURL;
			return true;
		}
	}
	return false;
}

/**
 *	エレメントにアトリビュートを付加する。
 *	既に同名のアトリビュートが存在する場合は、上書きされる
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント
 *	@param	pszAttribute	const TFXMLCh*		アトリビュート名
 *	@param	pszValue		const TFXMLCh*		値
 *	@return	bool							true:成功、false:失敗
 */
bool
CTFXMLUtils::SetAttribute(XNS(DOMElement)* pElement, const TFXMLCh* pszAttribute, const TFXMLCh* pszValue)
{

	if(	!pElement
		|| !pszAttribute
		|| TF_STRING_W(pszAttribute).empty()
		|| !pszValue ) return false;

	if( pElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE ) return false;

	// アトリビュートノードを生成する
	// NSURLは必ず"http://www.w3.org/2000/xmlns/"とすること
	XNS(DOMAttr)* pAttribute = m_pDOMDocument->createAttributeNS(	X(OI_XML_NSURI),
																	X(pszAttribute) );
	if( pAttribute )
	{
		// アトリビュートノードに値を設定する
		pAttribute->setNodeValue(X(pszValue));
		// エレメントにアトリビュートノードをセットする
		pElement->setAttributeNode(pAttribute);
		return true;
	}

	return false;
}

/**
 *	指定されたエレメントにCDATAセクションを設定する
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント(NULL不可)
 *	@param	pszValue		const TFXMLCh*			値
 *	@return	bool							true:成功、false:失敗
 */
bool
CTFXMLUtils::SetCDATASection(XNS(DOMElement)* pElement, const TFXMLCh* pszValue)
{
	if( !pElement || (pElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE) ) return false;

	if( pElement && pszValue )
	{

		XNS(DOMCDATASection)* pCDATA = m_pDOMDocument->createCDATASection(pszValue);
		if( pCDATA )
		{
			pElement->appendChild(pCDATA);
			return true;
		}
	}

	return false;
}


/**
 *	指定されたエレメントにTEXTノードを設定する
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント(NULL不可)
 *	@param	szValue		const TFXMLCh*			値
 *	@return	bool							true:成功、false:失敗
 */
bool 
CTFXMLUtils::SetText(XNS(DOMElement)* pElement, const TFXMLCh* pszValue)
{
	if( !pElement || (pElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE) ) return false;

	if( pElement && pszValue )
	{
		XNS(DOMText)* pText = m_pDOMDocument->createTextNode(pszValue);
		if( pText )
		{
			pElement->appendChild(pText);
			return true;
		}
	}

	return false;
}


/**
 *	指定されたエレメント内の指定された名前のアトリビュートを返す
 *
 *	@param	pElement		XNS(DOMElement)*	エレメント(NULL不可)
 *	@param	pszAttribute	const TFXMLCh*			アトリビュート名(NULL不可、空文字列不可)
 *	@param	strValue		TF_STRING_W&		値
 *	@return	bool								true:成功、false:失敗(当該アトリビュート無し、データ不変)
 */
bool 
CTFXMLUtils::GetAttribute(XNS(DOMElement)* pElement, const TFXMLCh* pszAttribute, TF_STRING_W& strValue)
{
	if( !pElement || !pszAttribute || TF_STRING_W(pszAttribute).empty() ) return false;
	if( pElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE ) return false;

	// アトリビュートが存在することを確認する
	// アトリビュートがあっても無くても、getAttributeは非NULLを返す
	if( pElement->hasAttribute(X(pszAttribute)) )
	{
		// アトリビュートが存在したら、中身を得る
		const XMLCh* pszxValue = pElement->getAttribute(pszAttribute);
		if( pszxValue )
		{
			// 非NULLなら、TRUEと値を返す
			strValue = pszxValue;
			return true;
		}
	}

	return false;
}

/**
 *	指定されたエレメント内のCDATAセクションを返す
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント(NULL不可)
 *	@param	strValue	TF_STRING_W&		値
 *	@return	BOOL						true:成功、false:失敗(当該データ無し、データ不変)
 */
bool 
CTFXMLUtils::GetCDATASection(XNS(DOMElement)* pElement, TF_STRING_W& strValue)
{
	if( !pElement || (pElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE) ) return false;

	strValue.erase();
	if( pElement )
	{
		XNS(DOMNode)* pNode = pElement->getFirstChild();
		if( pNode && ( pNode->getNodeType() == XNS(DOMNode)::CDATA_SECTION_NODE ) )
		{
			const XMLCh* pszxValue = pNode->getNodeValue();
			if( pszxValue )
			{
				strValue = pszxValue;
				return true;
			}
		}
	}
	return false;
}

/**
 *	指定されたエレメント内のTEXTを返す
 *
 *	@param	pElement	XNS(DOMElement)*	エレメント(NULL不可)
 *	@param	strValue	TF_STRING_W&		値
 *	@return	bool						true:成功、false:失敗(当該データ無し、データ不変)
 */
bool 
CTFXMLUtils::GetText(XNS(DOMElement)* pElement, TF_STRING_W& strValue)
{
	if( !pElement || (pElement->getNodeType() != XNS(DOMNode)::ELEMENT_NODE) ) return false;

	if( pElement )
	{
		XNS(DOMNode)* pNode = pElement->getFirstChild();
		if( pNode && ( pNode->getNodeType() == XNS(DOMNode)::TEXT_NODE ) )
		{
			const XMLCh* pszxValue = pNode->getNodeValue();
			if( pszxValue )
			{
				strValue = pszxValue;
				return true;
			}
		}
	}
	return false;
}
