
#include "TFXmlBodyPaser.h"
#include "TFResourcePropTF.h"
#include "TFResourcePropTFStatus.h"
#include "TFResourcePropTFUser.h"
#include "TFResourcePropTFMsg.h"
#include "TFResourcePropTFGroup.h"
#include "TFResourcePropTFUpdate.h"

#include <onion/DavResourceNode.h>

CTFXmlBodyPaser::CTFXmlBodyPaser(const TFXMLCh* pszServerID)
					:CTFXmlBodyPaserBase(pszServerID)
{
}

CTFXmlBodyPaser::~CTFXmlBodyPaser(void)
{
}

CTFDavResource* CTFXmlBodyPaser::Parse(CDavResourceNode* pNode)
{
	CTFResourcePropTF* pItem = new CTFResourcePropTF;

	pItem->SetServerID(m_strServerID.c_str());

	CTFURL cURL(pNode->URI());
	pItem->SetURI(cURL.m_strURI.c_str());

	unsigned long ulFlag = pNode->GetFlags();

	if (OI_RNP_CONTENTLANGUAGE & ulFlag)
	{
		pItem->SetContentLanguage(X(pNode->GetContentLanguage()));
	}

	if (OI_RNP_CONTENTLENGTH & ulFlag)
	{
		pItem->SetContentLength(pNode->GetContentLength());
	}

	if (OI_RNP_CREATIONDATE & ulFlag)
	{
		pItem->SetCreationDate(pNode->GetCreationDate());
	}

	// Creation Dateが含まれていないということはNULLリソースである
	if ((OI_RNP_CREATIONDATE & ulFlag) == 0)
	{
		pItem->m_bNullResource = true;
	}

	if (OI_RNP_ETAG & ulFlag)
	{
		pItem->SetEtag(X(pNode->GetETag()));
	}

	if (OI_RNP_RESOURCETYPE & ulFlag)
	{
		pItem->SetResourceType(pNode->GetResourceType());
	}

	if (OI_RNP_DISPLAYNAME & ulFlag)
	{
		pItem->SetDisplayName(pNode->DisplayName());
		pItem->SetName(pItem->GetDisplayName());
	}
	else 
	{
		pItem->SetDisplayName(cURL.m_strName.c_str());
		pItem->SetName(pItem->GetDisplayName());
	}

	if (OI_RNP_LASTMODIFIED & ulFlag)
	{
		pItem->SetLastModified(pNode->GetLastModified());
	}

	if (OI_RNP_LOCK & ulFlag)
	{
		int iLockCnt = pNode->GetLockCount();
		for (int i=0; i < iLockCnt; i++)
		{
			pItem->SetLock(pNode->GetLock(i));
		}
	}

	// TeamFile拡張プロパティの確認
	if (OI_RNP_EXTRAPROPERTIES & ulFlag)
	{
		TF_STRING_W strValue;

		XNS(DOMElement)* pExElement = (XNS(DOMElement)*)pNode->GetProp(200);
		OI_ASSERT(pExElement->getNodeType() == XNS(DOMNode)::ELEMENT_NODE);

		// TF:creator
		if( GetTFElementCDATASection(pExElement, TF_PROP_CREATOR, strValue) )
		{
			pItem->SetCreator(strValue.c_str());
		}

		// TF:lastmodifier
		if( GetTFElementCDATASection(pExElement, TF_PROP_LASTMODIFIER, strValue) )
		{
			pItem->SetLastModifire(strValue.c_str());
		}

		// TF:mailwatch
		XNS(DOMElement)* pMailWatch = GetTFChildElement(pExElement, TF_PROP_MAILWATCH);
		if( pMailWatch )
		{
			// TF:triggermethod
			if( GetTFElementText(pMailWatch, TF_PROP_TRIGGERMETHOD, strValue) )
			{
				pItem->SetMailWatchMethod(strValue.c_str());
			}

			XNS(DOMElement)* pMailWatchOwner = GetTFChildElement(pMailWatch, TF_PROP_MAILWATCHOWNER);
			if( pMailWatchOwner )
			{
				pItem->SetMailWatchOwner(true);

				// TODO: ここにもTF_PROP_TRIGGERMETHODエレメントのパースが必要
			}
		}

		// TF:trashinformation
		XNS(DOMElement)* pTrashInfo = GetTFChildElement(pExElement, TF_PROP_TRASHINFORMATION);
		if ( pTrashInfo )
		{

			// TF:deleter
			if (GetTFElementCDATASection(pTrashInfo, TF_PROP_DELETER, strValue))
			{
				pItem->SetDeleter(strValue.c_str());
			}

			// TF:deletion
			if (GetTFElementText(pTrashInfo, TF_PROP_DELETION, strValue))
			{
				pItem->SetDeletion(ParseTime(X(strValue.c_str())));
			}

		}

		// TF:resourcestate
		XNS(DOMElement)* pResState = GetTFChildElement(pExElement, TF_PROP_RESOURCESTATE);
		if (pResState)
		{
			XNS(DOMElement)* pState;
			pState = GetTFChildElement(pResState, TF_PROP_RESOURCESTATE_PUBLISHED);
			if (pState)
			{
				pItem->SetResourceState(TF_RES_STATE_PUBLISHED);
			}
		}
		
		// TF:user-privilege-grant-set
		XNS(DOMElement)* pPrivgrant = GetTFChildElement(pExElement, TF_PROP_USERPRIVILEGEGRANTSET);
		if (pPrivgrant)
		{
			XNS(DOMElement)* pPriv;
			TFUSERPRIV nPriv = TF_USER_PRIV_UNKNOWN;
			for (int i=0; TFPRIVILEGETABLE[i].pszName != NULL; i++)
			{
				pPriv = GetTFChildElement(pPrivgrant, TFPRIVILEGETABLE[i].pszName);
				if (pPriv)
				{
					nPriv |= TFPRIVILEGETABLE[i].nPriv;
				}
			}
			
			pItem->SetUserPrivilegeGrantSet(nPriv);
		}
		
	} // OI_RNP_EXTRAPROPERTIES & ulFlag

	return pItem;
}

CTFDavResource* 
CTFXmlBodyPaser::Parse(XNS(DOMNode)* pNode)
{
	if( pNode
		&& (pNode->getNodeType() == XNS(DOMNode)::ELEMENT_NODE) )
	{
		return ParseDiscovery((XNS(DOMElement)*)pNode);
	}

	return 0;
}

CTFDavResource*
CTFXmlBodyPaser::OnTFStatusDiscoveryFound(XNS(DOMElement)* pStatusDiscovery)
{
	TF_STRING_W strValue;

	// DAVリソース部分をパースする
	CDavResourceNode cDavNode;
	CTFDavResource* pItem = 0;
	if( cDavNode.Parse(pStatusDiscovery) == OI_OK )
	{
		// TFDavResourceオブジェクトに変換する
		pItem = Parse(&cDavNode);
	}

	CTFResourcePropTFStatus* pStatItem = new CTFResourcePropTFStatus;
	// DAVリソース部分(URI、DisplayName等)を取り込む
	if( pItem ) pStatItem->Import(*pItem);

	XNS(DOMElement)* pStatusInfo = GetTFChildElement(pStatusDiscovery, "statusinfo");
	if (pStatusInfo)
	{
		parseCommonUserProperty(pStatusInfo, *pStatItem);
	}

	if (pItem != NULL) delete pItem;
	return pStatItem;

}

CTFDavResource* 
CTFXmlBodyPaser::OnTFUserDiscoveryFound(XNS(DOMElement)* pUserDiscovery)
{
	TF_STRING_W strValue;

	// DAVリソース部分をパースする
	CDavResourceNode cDavNode;
	CTFDavResource* pItem = 0;
	if( cDavNode.Parse(pUserDiscovery) == OI_OK )
	{
		// TFDavResourceオブジェクトに変換する
		pItem = Parse(&cDavNode);
	}

	CTFResourcePropTFUser* pUsrItem = new CTFResourcePropTFUser;
	// DAVリソース部分(URI、DisplayName等)を取り込む
	if( pItem ) pUsrItem->Import(*pItem);

	XNS(DOMElement)* pUsrInfo = GetTFChildElement(pUserDiscovery, "userinfo");
	if (pUsrInfo)
	{
		parseCommonUserProperty(pUsrInfo, *pUsrItem);
	}

	if (pItem != NULL) delete pItem;
	return pUsrItem;
}

CTFDavResource* 
CTFXmlBodyPaser::OnTFGroupDiscoveryFound(XNS(DOMElement)* pGroupDiscovery)
{
	TF_STRING_W strValue;

	// DAVリソース部分をパースする
	CDavResourceNode cDavNode;
	CTFDavResource* pItem = 0;
	if( cDavNode.Parse(pGroupDiscovery) == OI_OK )
	{
		// TFDavResourceオブジェクトに変換する
		pItem = Parse(&cDavNode);
	}

	CTFResourcePropTFGroup* pGrpItem = new CTFResourcePropTFGroup;
	// DAVリソース部分(URI、DisplayName等)を取り込む
	if( pItem ) pGrpItem->Import(*pItem);

	XNS(DOMElement)* pGrpInfo = GetTFChildElement(pGroupDiscovery, "groupinfo");

	if (pGrpInfo)
	{
		// TF:gruopid
		if ( GetTFElementText(pGrpInfo, TF_PROP_GROUPID, strValue) )
		{
			pGrpItem->SetID(strValue.c_str());
		}

		// TF:name
		if ( GetTFElementCDATASection(pGrpInfo, TF_PROP_NAME, strValue) )
		{
			pGrpItem->SetName(strValue.c_str());
		}

		// TF: creationdt
		if ( GetTFElementText(pGrpInfo, TF_PROP_CREATEDT, strValue) )
		{
			pGrpItem->SetCreationDate(ParseTimeISO8601((const char*)X(strValue.c_str())));
		}

		// TF: updatedt
		if ( GetTFElementText(pGrpInfo, TF_PROP_UPDATEDT, strValue) )
		{
			pGrpItem->SetLastModified(ParseTimeISO8601((const char*)X(strValue.c_str())));
		}

		// TF: comment
		if ( GetTFElementCDATASection(pGrpInfo, TF_PROP_COMMENT, strValue) )
		{
			pGrpItem->SetComment(strValue.c_str());
		}

		// TF: groupmailwatch
		XNS(DOMElement)* pMail = GetTFChildElement(pGrpInfo, TF_PROP_GROUPMAILWATCH);
		if (pMail)
		{
			if (GetTFElementText(pGrpInfo, TF_PROP_TRIGGERMETHOD, strValue))
			{
				pGrpItem->SetMailWatch(strValue.c_str());
			}
		}

		// TF: groupcollectionuri
		if (GetTFElementText(pGrpInfo, TF_PROP_GROUPCOLLECTIONURI, strValue))
		{
			// See CTFXmlBodyPaserBase::GetDAVHref()
			TF_STRING_W strURI;
			strURI = X((XMLByte*)Unescape((const char*)X(strValue.c_str())).c_str());
			pGrpItem->SetGroupCollectionUri(strURI.c_str());
		}

		// TF: constraints
		XNS(DOMElement)* pConstraints = GetTFChildElement(pGrpInfo, TF_PROP_CONSTRAINTS);
		if (pConstraints)
		{
			XNS(DOMElement)* pChild=NULL;
			int nConstraints = TF_GROUPCONSTRAINTS_NONE;
			if (GetTFChildElement(pConstraints, TF_PROP_GROUP_CONSTRAINTS_WRITE))
				nConstraints |= TF_GROUPCONSTRAINTS_WRITE;

			if (GetTFChildElement(pConstraints, TF_PROP_GROUP_CONSTRAINTS_OPELOG))
				nConstraints |= TF_GROUPCONSTRAINTS_OPELOG;

			if (GetTFChildElement(pConstraints, TF_PROP_GROUP_CONSTRAINTS_SHOWPROPERTY))
				nConstraints |= TF_GROUPCONSTRAINTS_SHOWPROPERTY;
			
			pGrpItem->SetGroupConstraints(nConstraints);
		}

		// TF: groupstate
		XNS(DOMElement)* pGroupState = GetTFChildElement(pGrpInfo, TF_PROP_GROUPSTATE);
		{
			if (pGroupState)
			{
				int nGroupState = TF_GROUPSTATE_NONE;
				if (GetTFChildElement(pGroupState, TF_PROP_GROUP_STATE_ACTIVE))
					nGroupState |= TF_GROUPSTATE_ACTIVE;
				if (GetTFChildElement(pGroupState, TF_PROP_GROUP_STATE_INACTIVE))
					nGroupState |= TF_GROUPSTATE_INACTIVE;

				pGrpItem->SetGroupState(nGroupState);
			}
		}

		// TF: ownername
		if (GetTFElementCDATASection(pGrpInfo, TF_PROP_OWNER_NAME, strValue))
		{
			pGrpItem->SetGroupOwnerName(strValue.c_str());
		}

		// TF: ownerid
		if (GetTFElementCDATASection(pGrpInfo, TF_PROP_OWNER_ID, strValue))
		{
			pGrpItem->SetGroupOwnerID(strValue.c_str());
		}

		// TF: childcount
		if (GetTFElementText(pGrpInfo, TF_PROP_CHILDCOUND, strValue))
		{
			pGrpItem->SetChildCount(strValue.c_str());
		}
	}

	if (pItem != NULL) delete pItem;
	return pGrpItem;

}

CTFDavResource* 
CTFXmlBodyPaser::OnTFSysmsgDiscoveryFound(XNS(DOMElement)* pSysmsgDiscovery)
{
	TF_STRING_W strValue;

	// DAVリソース部分をパースする
	CDavResourceNode cDavNode;
	CTFDavResource* pItem = 0;
	if( cDavNode.Parse(pSysmsgDiscovery) == OI_OK )
	{
		// TFDavResourceオブジェクトに変換する
		pItem = Parse(&cDavNode);
	}

	CTFResourcePropTFMsg* pMsgItem = new CTFResourcePropTFMsg;
	// DAVリソース部分(URI、DisplayName等)を取り込む
	if( pItem ) pMsgItem->Import(*pItem);

	XNS(DOMElement)* pMsgInfo = GetTFChildElement(pSysmsgDiscovery, "sysmsginfo");

	if (pMsgInfo)
	{
		// TF:msgid
		if ( GetTFElementText(pMsgInfo, TF_PROP_MSGID, strValue) )
		{
			pMsgItem->SetMsgID(strValue.c_str());
			pMsgItem->SetName(pMsgItem->GetID());
		}

		// TF:msgtxt
		if ( GetTFElementCDATASection(pMsgInfo, TF_PROP_MSGTXT, strValue) )
		{
			pMsgItem->SetMsg(strValue.c_str());
		}

		// TF:active
		XNS(DOMElement)* pMsgActive = GetTFChildElement(pMsgInfo, TF_PROP_ACTIVE);
		if (pMsgActive)
		{
			pMsgItem->SetActiveFlag(true);
		}
		else {
			pMsgItem->SetActiveFlag(false);
		}

		// TF:createdt
		if ( GetTFElementText(pMsgInfo, TF_PROP_CREATEDT, strValue) )
		{
			pMsgItem->SetCreationDate(ParseTimeISO8601((const char*)X(strValue.c_str())));
		}

		// TF:updatedt
		if ( GetTFElementText(pMsgInfo, TF_PROP_UPDATEDT, strValue) )
		{
			pMsgItem->SetLastModified(ParseTimeISO8601((const char*)X(strValue.c_str())));
		}
	}

	if (pItem != NULL) delete pItem;
	return pMsgItem;

}

CTFDavResource* 
CTFXmlBodyPaser::OnTFUpdateDiscoveryFound(XNS(DOMElement)* pUpdateDiscovery)
{
	TF_STRING_W strValue;

	// DAVリソース部分をパースする
	CDavResourceNode cDavNode;
	CTFDavResource* pItem = 0;
	if( cDavNode.Parse(pUpdateDiscovery) == OI_OK )
	{
		// TFDavResourceオブジェクトに変換する
		pItem = Parse(&cDavNode);
	}

	CTFResourcePropTFUpdate* pUpItem = new CTFResourcePropTFUpdate;
	// DAVリソース部分(URI、DisplayName等)を取り込む
	if( pItem ) pUpItem->Import(*pItem);

	XNS(DOMElement)* pUpdInfo = GetTFChildElement(pUpdateDiscovery, "updateinfo");

	if (pUpdInfo)
	{
		// TF: name
		if ( GetTFElementCDATASection(pUpdInfo, TF_PROP_NAME, strValue) )
		{
			pUpItem->SetName(strValue.c_str());
		}

		// TF: version
		if ( GetTFElementCDATASection(pUpdInfo, TF_PROP_VERSION, strValue) )
		{
			pUpItem->SetVersion(strValue.c_str());
		}
		
		// TF: lineup
		if ( GetTFElementText(pUpdInfo, TF_PROP_LINEUP, strValue) )
		{
			pUpItem->SetLineup(strValue.c_str());
		}

		// TF: digest
		if ( GetTFElementText(pUpdInfo, TF_PROP_DIGEST, strValue) )
		{
			pUpItem->SetDigest(strValue.c_str());
		}

		// TF:createdt
		if ( GetTFElementText(pUpdInfo, TF_PROP_CREATEDT, strValue) )
		{
			pUpItem->SetCreationDate(ParseTimeISO8601((const char*)X(strValue.c_str())));
		}

		// TF:updatedt
		if ( GetTFElementText(pUpdInfo, TF_PROP_UPDATEDT, strValue) )
		{
			pUpItem->SetLastModified(ParseTimeISO8601((const char*)X(strValue.c_str())));
		}
	}

	if (pItem != NULL) delete pItem;
	return pUpItem;

}


CTFDavResource* CTFXmlBodyPaser::OnTFNoDiscoveryFound(XNS(DOMElement)* pResponse)
{
	TF_STRING_W strValue;

	// DAVリソース部分をパースする
	CDavResourceNode cDavNode;
	CTFDavResource* pItem = 0;
	if( cDavNode.Parse(pResponse) == OI_OK )
	{
		// TFDavResourceオブジェクトに変換する
		pItem = Parse(&cDavNode);
	}

	// RootTree検索かLinkDB系検索かを確認する
	// <TF:foldertype>タグがあればRootTree検索の結果と判断
	TF_STRING_W strFolderType;
	if( GetTFChildElementName(pResponse, C2W(TF_PROP_FOLDERTYPE), strFolderType) )
	{
		// For roottreesearch
		CTFResourceTFRootTree* pRTItem = new CTFResourceTFRootTree;

		// DAVリソース部分(URI、DisplayName等)を取り込む
		if( pItem ) pRTItem->Import(*pItem);

		// RootTreeSearch特有のエレメントを処理する
		// TF:foldertype
		pRTItem->SetFolderType(strFolderType.c_str());

		// TF:displayorder
		if( GetTFElementText(pResponse, TF_PROP_DISPLAYORDER, strValue) )
		{
			pRTItem->SetDisplayOrder(atoi(OI_STRING_A(X(strValue.c_str())).c_str()));
		}

		// TF:sqlno
		if ( GetTFElementText(pResponse, TF_PROP_SQLNO, strValue) )
		{
			pRTItem->SetSQLNo(strValue.c_str());
		}

		if( pItem ) delete pItem;
		return pRTItem;
	}
	else
	{
		// For linkdbsearch
		// TODO: Implement LinkDB response paser

		if( pItem ) delete pItem;
		return 0;
	}

}

void
CTFXmlBodyPaser::parseCommonUserProperty(XNS(DOMElement)* pElement, CTFResourcePropTFUser& pUserItem)
{
	TF_STRING_W strValue;
	
	// TF:userid
	if ( GetTFElementCDATASection(pElement, TF_PROP_USERID, strValue) )
	{
		pUserItem.SetUID(strValue.c_str());
	}

	// TF:name
	if ( GetTFElementCDATASection(pElement, TF_PROP_NAME, strValue) )
	{
		pUserItem.SetName(strValue.c_str());
	}

	// TF:usedstorage
	if ( GetTFElementText(pElement, TF_PROP_USEDSTORAGE, strValue) )
	{
		pUserItem.SetStorages(strValue.c_str(), TF_STAT_USED);
	}

	// TF:maxstorage
	if ( GetTFElementText(pElement, TF_PROP_MAXSTORAGE, strValue) )
	{
		pUserItem.SetStorages(strValue.c_str(), TF_STAT_MAX);
	}

	// TF:usedresource
	if ( GetTFElementText(pElement, TF_PROP_USEDRESOURCE, strValue) )
	{
		pUserItem.SetResources(strValue.c_str(), TF_STAT_USED);
	}

	// TF:maxresource
	if ( GetTFElementText(pElement, TF_PROP_MAXRESOURCE, strValue) )
	{
		pUserItem.SetResources(strValue.c_str(), TF_STAT_MAX);
	}

	// TF:admin
	XNS(DOMElement)* pAdmin = GetTFChildElement(pElement, TF_PROP_ADMIN);
	pUserItem.SetAdminFlag((pAdmin != NULL));

	// TF:creationdt
	if ( GetTFElementText(pElement, TF_PROP_CREATEDT, strValue) )
	{
		pUserItem.SetCreationDate(ParseTimeISO8601((const char*)X(strValue.c_str())));
	}

	// TF: updatedt
	if ( GetTFElementText(pElement, TF_PROP_UPDATEDT, strValue) )
	{
		pUserItem.SetLastModified(ParseTimeISO8601((const char*)X(strValue.c_str())));
	}

	// TF:lastaccessdt
	if ( GetTFElementText(pElement, TF_PROP_LASTACCESSDT, strValue) )
	{
		pUserItem.SetLastAccessDate(ParseTimeISO8601((const char*)X(strValue.c_str())));
	}

	// TF:lastaccessclient
	if ( GetTFElementCDATASection(pElement, TF_PROP_LASTACCESSCLIENT, strValue) )
	{
		pUserItem.SetLastAccessClient(strValue.c_str());
	}

	// TF:mailaddr
	if ( GetTFElementCDATASection(pElement, TF_PROP_MAILADDR, strValue) )
	{
		pUserItem.SetMailAddr(strValue.c_str());
	}

	// TF:password
	if ( GetTFElementCDATASection(pElement, TF_PROP_PASSWORD, strValue) )
	{
		pUserItem.SetPassWord(strValue.c_str());
	}
		
	// TF:comment
	if ( GetTFElementCDATASection(pElement, TF_PROP_COMMENT, strValue) )
	{
		pUserItem.SetComment(strValue.c_str());
	}

	// TF:accessdeny
	if ( GetTFElementText(pElement, TF_PROP_ACCESSDENY, strValue) )
	{
		pUserItem.SetAccessDeny(strValue.c_str());
	}

	// TF:state
	XNS(DOMElement)* pUserStatus = GetTFChildElement(pElement, TF_PROP_STATE);
	if (pUserStatus)
	{
		TFUSERSTATUS	nStatus = TF_USER_STATUS_UNKNOWN;
		XNS(DOMElement)* pStat;
		pStat = GetTFChildElement(pUserStatus, TF_PROP_USER_STATE_ACTIVE);
		if (pStat) nStatus |= TF_USER_STATUS_ACTIVE;

		pStat = GetTFChildElement(pUserStatus, TF_PROP_USER_STATE_INACTIVE);
		if (pStat) nStatus |= TF_USER_STATUS_INACTIVE;

		pStat = GetTFChildElement(pUserStatus, TF_PROP_USER_STATE_EXPIRED);
		if (pStat) nStatus |= TF_USER_STATUS_EXPIRD;

		pUserItem.SetStatus(nStatus);
	}

	// TF:expiration
	if ( GetTFElementText(pElement, TF_PROP_EXPIRATION, strValue) )
	{
		pUserItem.SetExpiration(ParseTimeISO8601((const char*)X(strValue.c_str())));
	}

	// TF: user-privilege
	XNS(DOMElement)* pUserPriv = GetTFChildElement(pElement, TF_PROP_USER_PRIVILEGE);
	if (pUserPriv)
	{
		TFUSERPRIV	nPriv = TF_USER_PRIV_UNKNOWN;
		XNS(DOMElement)* pPriv;
		pPriv = GetTFChildElement(pUserPriv, TF_PROP_USER_PRIV_READ);
		if (pPriv) nPriv |= TF_USER_PRIV_READ;

		pPriv = GetTFChildElement(pUserPriv, TF_PROP_USER_PRIV_UPLOAD);
		if (pPriv) nPriv |= TF_USER_PRIV_UPLOAD;

		pPriv = GetTFChildElement(pUserPriv, TF_PROP_USER_PRIV_READWRITE);
		if (pPriv) nPriv |= TF_USER_PRIV_READWRITE;

		pPriv = GetTFChildElement(pUserPriv, TF_PROP_USER_PRIV_SET_VIEW);
		if (pPriv) nPriv |= TF_USER_PRIV_SET_VIEW;

		pPriv = GetTFChildElement(pUserPriv, TF_PROP_USER_PRIV_GROUPCONSTRAINTIGNORE);
		if (pPriv) nPriv |= TF_USER_PRIV_GROUPCONSTRAINTSIGNORE;

		pPriv = GetTFChildElement(pUserPriv, TF_PROP_USER_PRIV_CONTROLOTHERUSER);
		if (pPriv) nPriv |= TF_USER_PRIV_CONTROLOTHERUSER;

		pPriv = GetTFChildElement(pUserPriv, TF_PROP_USER_PRIV_APPOINTEDGROUPLEADER);
		if (pPriv) nPriv |= TF_USER_PRIV_APPOINTEDGROUPLEADER;

		pUserItem.SetPrivilege(nPriv);
	}

	// TF: usertype
	if ( GetTFChildElementName(pElement, C2W(TF_PROP_USER_TYPE), strValue) )
	{
		pUserItem.SetUserType(strValue.c_str());
	}

	// TF: maxusercreation
	if ( GetTFElementText(pElement, TF_PROP_MAX_USERCREATION, strValue) )
	{
		pUserItem.SetMaxuserCreation(strValue.c_str());
	}

	// TF: ownername
	if ( GetTFElementCDATASection(pElement, TF_PROP_OWNER_NAME, strValue) )
	{
		pUserItem.SetOwnerName(strValue.c_str());
	}

	// TF: ownerid
	if ( GetTFElementCDATASection(pElement, TF_PROP_OWNER_ID, strValue) )
	{
		pUserItem.SetOwnerID(strValue.c_str());
	}

	// TF: otheruser
	XNS(DOMElement)* pOtherUser = GetTFChildElement(pElement, TF_PROP_OTHERUSER);
	pUserItem.SetOtherUser(pOtherUser!=NULL);

}


