
#include "TFLibDefinition.h"
#include "TFXmlBodyPaserBase.h"
#include "oimap.h"

CTFXmlBodyPaserBase::CTFXmlBodyPaserBase(const TFXMLCh* pszServerID)
{
	m_strServerID = pszServerID;
	m_strNSURL_TF = C2W(TF_XML_NSURI);
	m_strNSURL_DAV = C2W(OI_XML_DAVNAMESPACE);
}

CTFXmlBodyPaserBase::~CTFXmlBodyPaserBase(void)
{
}

/**
 * SEARCHメソッド結果のパースを行う。
 *
 * このメソッドが???discoveryタグを見つけた場合、
 * そのタグに応じたパーサーハンドラ(OnTF???DiscoveryFound)を呼び出す。
 * discoveryが見つからなかったら、OnTFNoDiscoveryFoundハンドラを呼ぶ
 *
 * @param	pResponse	XNS(DOMElement)*	CDavWorksession::OnSearchResponseで得られたDOMノード
 * @return	CTFDavResource*	パース後の結果が入ったオブジェクト（0の場合はパース失敗）
 *
 */
CTFDavResource* CTFXmlBodyPaserBase::ParseDiscovery(XNS(DOMElement)* pResponse)
{
	if( !pResponse ) return 0;	
	if( pResponse->getNodeType() != XNS(DOMNode)::ELEMENT_NODE ) return 0;

	/* <D:resopnse>
	 *    <D:href></D:href>
	 *    <D:propstat>
	 *      <D:prop>
	 *        <TF:???discovery>
	 *        ...
	 *
	 */

	CTFDavResource* pItem = 0;

	// <D:prop>を得る
	XNS(DOMElement)* pProp = GetElement(pResponse,C2W("prop"),0,m_strNSURL_DAV.c_str());
	if( pProp )
	{
		// <D:prop>の子エレメントを得る
		XNS(DOMElement)* pDiscovery = GetChildElement(pProp);
		if( pDiscovery )
		{
			TF_STRING_W strDiscoveryW;
			TF_STRING_W strDiscoveryNSURLW;
			// 子エレメント名とNamespace URLを得る
			// 子エレメントが<TF:???discovery>なら、各パーサーハンドラを呼ぶ
			if( GetElementName(pDiscovery,strDiscoveryW,strDiscoveryNSURLW) )
			{
				TF_STRING_W strDiscoveryNSURL	= C2W(strDiscoveryNSURLW.c_str());
				TF_STRING_W strDiscovery		= C2W(strDiscoveryW.c_str());

				if ( !strDiscoveryNSURL.compare(m_strNSURL_TF.c_str()) )
				{
					if( !strDiscovery.compare(C2W("userdiscovery")) )
					{
						pItem = OnTFUserDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("groupdiscovery")) )
					{
						pItem = OnTFGroupDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("updatediscovery")) )
					{
						pItem = OnTFUpdateDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("sysmsgdiscovery")) )
					{
						pItem = OnTFSysmsgDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("statusdiscovery")) )
					{
						pItem = OnTFStatusDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("dbmsdiscovery")) )
					{
						pItem = OnTFDbmsDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("dependsqldiscovery")) )
					{
						pItem = OnTFDependSqlDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("sqlpreferencediscovery")) )
					{
						pItem = OnTFSqlPreferenceDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("dbmserrdiscovery")) )
					{
						pItem = OnTFDbmsErrDiscoveryFound(pDiscovery);
					}
					else if( !strDiscovery.compare(C2W("analysiserrdiscovery")) )
					{
						pItem = OnTFAnalysisErrDiscoveryFound(pDiscovery);
					}	
					else
					{
						// <TF:???discovery>が無かった場合
						return OnTFNoDiscoveryFound(pResponse);
					}
				}
				else {
					// TF Namespace以外の場合
						return OnTFNoDiscoveryFound(pResponse);
				}
			}
			else
			{
				// <D:????>とかだった場合（TF Namespace以外、DAV）
				return OnTFNoDiscoveryFound(pResponse);
			}
		}
	}

	// URIを得てCTFDavResourceオブジェクトに格納する
	if( pItem )
	{
		TF_STRING_W strURI;
		if( GetDAVHref(pResponse, strURI) )
		{
			pItem->SetURI(strURI.c_str());
		}

		// サーバアイテムも設定する
		pItem->SetServerID(m_strServerID.c_str());
	}

	return pItem;
}

/**
 * D:hrefエレメント(URI)を得る
 *
 * @param	pParentElement	XNS(DOMElement)*	親エレメント
 * @param	strHref			TF_STRING_W&		URIを格納するための文字列オブジェクトの参照
 * @return	bool			結果が得られた場合はtrueを返す
 *
 */
bool CTFXmlBodyPaserBase::GetDAVHref(XNS(DOMElement)* pParentElement, TF_STRING_W& strHref)
{
	strHref.clear();
	TF_STRING_W strTemp;
	XNS(DOMElement)* pElement = GetElement(pParentElement, C2W(TF_PROP_HREF), 0, m_strNSURL_DAV.c_str());
	if( pElement )
	{
		if( GetText(pElement, strTemp) )
		{
			/* 
			 *	strTemp : TF_STRING_W
			 *		(XMLCh*)
			 *			X
			 *				(const char*)
			 *					Unescape
			 *						(XMLByte*)
			 *							X
			 *							strHref : TF_STRING_W
			 */ 
			strHref = X((XMLByte*)Unescape((const char*)X(strTemp.c_str())).c_str());
			return true;
		}
	}
	return false;
}

/**
 * 指定エレメントの子エレメントの名前を得る
 *
 * @param	pParentElement		XNS(DOMElement)*	親エレメント
 * @param	pszElementName		const char*			エレメント名
 * @param	strChildElementName	OI_STRING_A&	子エレメント名を格納するための文字列オブジェクトの参照
 * @return	bool									結果が得られた場合はtrueを返す
 *
 */
bool CTFXmlBodyPaserBase::GetTFChildElementName(	XNS(DOMElement)* pParentElement,
													const TFXMLCh* pszElementName,
													TF_STRING_W& strChildElementName)
{
	TF_STRING_W strURL;
	strChildElementName.clear();
	XNS(DOMElement)* pElement = GetElement(pParentElement, pszElementName, 0, m_strNSURL_TF.c_str());
	if( pElement )
	{
		// 子エレメントを得る。複数ある場合は最初に見つかった子エレメントを得る
		XNS(DOMElement)* pChildElement = GetChildElement(pElement);
		if( pChildElement )
		{
			// エレメント名とNSURLを得る
			bool bResult = GetElementName(pChildElement, strChildElementName, strURL);

			// Namespaceが一致したらtrueを返す
			return ( bResult && !strURL.compare(m_strNSURL_TF.c_str()) );
		}
	}
	return false;
}

/**
 * 指定エレメント内の文字列を得る。
 *
 * @param	pParentElement		XNS(DOMElement)*	親エレメント
 * @param	pszElementName		const char*		エレメント名
 * @param	strValue				TF_STRING_W&		テキストを格納するための文字列オブジェクトの参照
 * @return	bool										結果が得られた場合はtrueを返す
 *
 */
bool CTFXmlBodyPaserBase::GetTFElementText(XNS(DOMElement)* pParentElement,
										   const char* pszElementName,
										   TF_STRING_W& strValue )
{
	strValue.erase();
	XNS(DOMElement)* pElement = GetElement(pParentElement, C2W(pszElementName), 0, m_strNSURL_TF.c_str());
	if( pElement )
	{
		return GetText(pElement, strValue);
	}
	return false;
}

/**
 * 指定エレメント内のCDATAセクション内の文字列を得る。
 *
 * @param	pParentElement		XNS(DOMElement)*	親エレメント
 * @param	pszElementName		const char*		エレメント名
 * @param	strValue				TF_STRING_W&		テキストを格納するための文字列オブジェクトの参照
 * @return	bool										結果が得られた場合はtrueを返す
 *
 */
bool CTFXmlBodyPaserBase::GetTFElementCDATASection(XNS(DOMElement)* pParentElement,
										   const char* pszElementName,
										   TF_STRING_W& strValue )
{
	strValue.clear();
	XNS(DOMElement)* pElement = GetElement(pParentElement, C2W(pszElementName), 0, m_strNSURL_TF.c_str());
	if( pElement )
	{
		return GetCDATASection(pElement, strValue);
	}
	return false;
}

/**
 * 指定エレメント内のテキストかCDATAセクションの文字列を得る。
 * CDATAセクションの有無に関わらず文字列を取得できる。
 *
 * @param	pParentElement		XNS(DOMElement)*	親エレメント
 * @param	pszElementName		const char*		エレメント名
 * @param	strValue				TF_STRING_W&		テキストを格納するための文字列オブジェクトの参照
 * @return	bool										結果が得られた場合はtrueを返す
 *
 */
bool CTFXmlBodyPaserBase::GetTFElementAutoText(XNS(DOMElement)* pParentElement,
										   const char* pszElementName,
										   TF_STRING_W& strValue )
{
	strValue.clear();
	XNS(DOMElement)* pElement = GetElement(pParentElement, C2W(pszElementName), 0, m_strNSURL_TF.c_str());
	if( pElement )
	{
		switch(pElement->getNodeType())
		{
		case (XNS(DOMNode)::TEXT_NODE):
			return GetText(pElement, strValue);
			break;

		case (XNS(DOMNode)::CDATA_SECTION_NODE):
			return GetCDATASection(pElement, strValue);
			break;

		default:
			break;
		}
	}
	return false;
}

/**
 * 子エレメントを得る。
 *
 * @param	pParentElement		XNS(DOMElement)*	親エレメント
 * @param	pszElementName		const char*			子エレメント名
 * @return	XNS(DOMElement)*						結果が得られた場合は子エレメントノードを返す。失敗の場合は0
 *
 */
XNS(DOMElement)* CTFXmlBodyPaserBase::GetTFChildElement(XNS(DOMElement)* pParentElement,
														const char* pszElementName)
{
	return 	GetElement(pParentElement, C2W(pszElementName), 0, m_strNSURL_TF.c_str());
}

