/* 
   Http stream
   Copyright (C) 2003-2005, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavHttpStream.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <onion/DavHttpStream.h>
#include <onion/DavAuthManager.h>
#include <onion/DavMemBuffer.h>
#include <onion/DavWorkSession.h>
#include <onion/DavResponseBody.h>
#include <onion/DavServerPolicy.h>
#include <onion/RBPVDDummy.h>
#include <onion/PBCSMDummy.h>
#include <onion/PBCSMMemBuffer.h>
#include <onion/PBCSMXml.h>


CDavHttpStream::CDavHttpStream()
{
  m_pBody = NULL;
}

CDavHttpStream::~CDavHttpStream()
{
  if(m_pBody) {
    /*terminate unfinished transfer*/
    if(m_pBody->m_bEOF)
      end(m_pBody);
    else
      m_pSession->Disconnect();
    delete m_pBody;
    /*reset session's request pointer*/
    OI_ASSERT(m_pSession->m_pActiveRequest == this);
    m_pSession->m_pActiveRequest = NULL;
  }
}

OI_RESULT
CDavHttpStream::Read(void* pvData,
		     size_t unBufLen,
		     size_t* punReadLen)
{
  if(!m_pBody)
    return OIGEINVALIDSTATE;
  bool bContinue;
  size_t unReadLen;
  OI_RESULT enuRet = m_pBody->Read((char*)pvData, unBufLen, &unReadLen);

  if(OI_RETRY == enuRet) {
    bContinue = 
      m_pSession->OnReceiveProgress(this,
				    m_pBody->m_unReadLen,
				    m_pBody->m_unBodyLen,
				    (m_pBody->m_enuMode == M_RSP_CLENGTH));
    if(!bContinue) {
      enuRet = OI_USERCANCELED;
      m_pSession->Disconnect();
      unReadLen = 0;
    }
  } else {
    if(OI_OK != enuRet) {
      OI_DEBUG("Error occured during stream read: %s\n",
	       GetErrorDescription(enuRet));
    }
    unReadLen = 0;
  }
  if(punReadLen)
    *punReadLen = unReadLen;
  return enuRet;
}

OI_RESULT
CDavHttpStream::Write(void* pvData,
		      size_t unBufLen,
		      size_t* punWrittenLen)
{
  return OIGENOTIMPLEMENTED;
}

OI_RESULT
CDavHttpStream::createStream(CDavWorkSession* pSession,
			     const X& strURI,
			     OI_STREAM_MODE enuMode)
{
  OI_RESULT enuRet;

  if(!pSession)
    return OIGEINVALIDARG;

  /*only one request can be processed at a time*/
  if(pSession->m_pActiveRequest)
    return OIGEINVALIDSTATE;

  /*only read is implemented so far*/
  if(enuMode == M_STM_WRITE)
    return OIGENOTIMPLEMENTED;

  /*hard-coded get method*/
  enuRet = Create(pSession, T_REQ_GET, OI_METHOD_GET, strURI);
  if(OI_OK != enuRet)
    return enuRet;

  m_pSession->m_nLastStatusCode = -1;
  m_pSession->m_strErrorDescription.erase();

  CRBPVDDummy pvdDummy;
  CPBCSMDummy csmDummy(this);
  m_pBody = new CDavResponseBody(this, m_pSession->getSocket());

  for(enuRet = OI_RETRY; enuRet == OI_RETRY;){
    /*if this is true, body is not read, we should disconnect*/
    bool bSimpleError = false;

    m_pBody->Reset();
    CDavXmlBuffer* pXmlBuffer = m_pSession->GetXmlBuffer();
    CDavMemBuffer* pMemBuffer = m_pSession->GetMemBuffer();
    m_pRBProvider = &pvdDummy;
    m_pPBConsumer = &csmDummy;

    pXmlBuffer->Clear();
    pMemBuffer->Clear();

    enuRet = begin(m_pBody);
    if(enuRet != OI_OK)
      break;
    /* headers are processed after begin() */

    /* check content type
     * notice Content-Type header is not always required here because
     * this is a generic function and we have server policy mechanism.
     * we don't check Content-Type every time.
     */
    OI_STRING_A strCType;
    OI_STRING_A strHeaderName = OI_RSPHDR_CTYPE;	//Content-Type
    MakeLowerA(strHeaderName);
    OI_HEADERMAP::iterator itHeader = m_mapRespHeaders.find(strHeaderName);
    /* do we have a Content-Type header? */
    if(itHeader != m_mapRespHeaders.end()) {
      /* this is guarenteed by read_resp_header() */
      OI_ASSERT(itHeader->second.size() > 0);
      strCType = itHeader->second[0];
    }

    m_pBody->m_strContentType = strCType;

    //chop off tail after semicolon
    OI_STRING_A::size_type nSemiColon = strCType.find(';');
    if(nSemiColon != OI_STRING_A::npos)
      strCType.erase(nSemiColon);

    OI_STRING_A strServer = m_pSession->m_strServerString;
    int nStatus = m_HttpStatus.m_nCode;
    CDavServerPolicy* pPolicy = m_pSession->m_pServerPolicy;
    OIOPERATION opr;

    enuRet = pPolicy->CheckResponse(m_pSession,
				    strServer.c_str(),
				    m_strMethod.c_str(), 
				    nStatus, strCType.c_str(),
				    &opr);
    if(enuRet != OI_OK)
      break;

    m_pSession->m_enuLastOperation = opr.enuOpCode;
    enuRet = opr.enuErrCode;

    OI_RESULT enuPullResult = OI_OK;
    switch(opr.enuOpCode){
    case OP_CACHE_HTML:
      {
	/*this should not be supported in stream mode*/
	enuRet = OIYEOPNOTSUPPORTED;
	bSimpleError = true;
	break;
      }
    case OP_PARSE_ERR_XML:
      {
	if(opr.szCutElement[0] == '\0')
	  strcpy(opr.szCutElement, OI_XML_CUTELEMENT); //"response"
	CPBCSMXml errorConsumer(this, opr.szCutElement, pXmlBuffer);
	enuPullResult = errorConsumer.PullResponseBody(m_pBody);
	break;
      }
    case OP_SIMPLE_ERR:
      {
	enuRet = opr.enuErrCode;
	bSimpleError = true;
	break;
      }
    case OP_USE_DEFAULT:
      {
	/*expected behaviour*/
	m_pSession->m_pActiveRequest = this;
	return OI_OK;
      }
    case OP_AUTHENTICATE:
      {
	/* for the HEAD mess */
	if(M_RSP_NO_BODY == m_pBody->m_enuMode){
	  enuPullResult = OI_OK;
	} else {
	  CPBCSMMemBuffer bufWriter(this);
	  //call OnPreRecv() ?
	  enuPullResult = bufWriter.PullResponseBody(m_pBody);
	  //call OnPostRecv() ?
	  if(OI_OK != enuPullResult)
	    break;
	}

	enuRet = m_pSession->m_pAuthManager->QueryEndRequest(this);
	break;
      }
    default:
      bSimpleError = true;
      /* this is a fatal error, abort */
      enuPullResult = enuRet = OIYEILLEGALOPCODE;
      break;
    }

    if(bSimpleError) {
      pSession->Disconnect();
      break;
    } else if(OI_OK != enuPullResult){
      /*something's wrong, disconnect & return an error*/
      enuRet = enuPullResult;
      break;
    }  else {
      OI_RESULT enuEndResult = end(m_pBody);
      if(OI_OK != enuEndResult){
	enuRet = enuEndResult;
	break;
      }
      softDisconnect();
    }
  }

  /*will QueryEndRequest return OI_OK?*/
  OI_ASSERT(OI_OK != enuRet);

  delete m_pBody;
  m_pBody = NULL;

  pSession->Disconnect();
  return enuRet;
}
