/* 
   Variable length buffer
   Copyright (C) 2003-2004, Lei Jiang <sledge10@hotmail.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA

   $Id: DavMemBuffer.cpp 132 2005-06-24 09:09:43Z komat $
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <stdlib.h>
#include <onion/DavMemBuffer.h>


CDavMemBuffer::CDavMemBuffer(size_t unInitSize, 
			     size_t unSizeLimit, size_t unIncrSize)
{
  if(unSizeLimit > OI_MEMBUF_SIZELIMIT ||
     unSizeLimit == 0) unSizeLimit = OI_MEMBUF_SIZELIMIT;
  if(unInitSize > unSizeLimit) unInitSize = unSizeLimit;

  m_unSizeLimit	= unSizeLimit;
  m_unIncrSize	= unIncrSize;

  m_pyBufHead = malloc(unInitSize + 4);
  m_unBufLen = unInitSize;
  Clear();
}

CDavMemBuffer::~CDavMemBuffer()
{
  if(m_pyBufHead)
    free(m_pyBufHead);
}

size_t
CDavMemBuffer::grow(size_t unNewSize)
{
  if(unNewSize <= m_unBufLen || unNewSize > m_unSizeLimit)
    return m_unBufLen;

  void* pNewBuf = realloc(m_pyBufHead, unNewSize + 4);

  if(pNewBuf){
    m_pyBufHead = pNewBuf;
    m_unBufLen = unNewSize;
  }
  return m_unBufLen;
}

bool
CDavMemBuffer::ensureCapacity(size_t unWriteSize)
{
  if(m_unDataLen + unWriteSize < m_unBufLen)
    return true;

  size_t unNewSize;
  if(m_unIncrSize > 0)
    unNewSize = ((m_unDataLen + unWriteSize)/m_unIncrSize + 1) * m_unIncrSize;
  else
    unNewSize = (m_unDataLen + unWriteSize) * 2;

  size_t unNewBufLen = grow(unNewSize);

  if(unNewBufLen < unNewSize) 
    return false;
  else
    return true;
}

size_t
CDavMemBuffer::Write(const void* pyData, size_t unDataLen)
{
  if(!ensureCapacity(unDataLen) || unDataLen <= 0 || pyData == NULL)
    return 0;

  memcpy((unsigned char*)m_pyBufHead + m_unDataLen, pyData, unDataLen);
  m_unDataLen += unDataLen;
  return unDataLen;
}

const void*
CDavMemBuffer::GetData()
{
  unsigned char* pyTail = (unsigned char*)m_pyBufHead + m_unDataLen;

  *(pyTail + 0) = 0;
  *(pyTail + 1) = 0;
  *(pyTail + 2) = 0;
  *(pyTail + 3) = 0;

  return m_pyBufHead;
}

void
CDavMemBuffer::Clear()
{
  unsigned char* pHead = (unsigned char*)m_pyBufHead;
  m_unDataLen = 0;
  *(pHead + 0) = 0;
  *(pHead + 1) = 0;
  *(pHead + 2) = 0;
  *(pHead + 3) = 0;
  m_strContentType.erase();
}

size_t
CDavMemBuffer::GetBufferSize()
{
  return m_unBufLen;
}

size_t
CDavMemBuffer::GetDataSize()
{
  return m_unDataLen;
}

size_t
CDavMemBuffer::GetIncrementSize()
{
  return m_unIncrSize;
}

size_t
CDavMemBuffer::SetIncrementSize(size_t unNewSize)
{
  size_t unOldValue = m_unIncrSize;
	
  m_unIncrSize = (unNewSize > OI_MEMBUF_SIZELIMIT)?
    OI_MEMBUF_SIZELIMIT:unNewSize;
  return unOldValue;
}
